# -*- coding: utf-8 -*-
"""
Configuration Manager
Handles saving and loading plugin configuration
"""

import json
import os
from qgis.core import QgsMessageLog, Qgis
from qgis.PyQt.QtCore import QSettings

class ConfigManager:
    """Manages plugin configuration persistence"""
    
    def __init__(self):
        self.settings = QSettings()
        self.config_key = "excel_auto_exporter"
        
    def save_config(self, config):
        """
        Save configuration to QSettings
        
        :param config: Configuration dictionary
        :type config: dict
        """
        try:
            config_json = json.dumps(config)
            self.settings.setValue(f"{self.config_key}/config", config_json)
            QgsMessageLog.logMessage(
                "Configuration saved successfully", 
                "Excel Auto Exporter", 
                Qgis.Info
            )
            return True
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error saving configuration: {str(e)}", 
                "Excel Auto Exporter", 
                Qgis.Critical
            )
            return False
    
    def load_config(self):
        """
        Load configuration from QSettings
        
        :return: Configuration dictionary
        :rtype: dict
        """
        try:
            config_json = self.settings.value(f"{self.config_key}/config", "{}")
            config = json.loads(config_json)
            
            # Ensure config has required structure
            if not config:
                config = self.get_default_config()
            
            return config
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error loading configuration: {str(e)}", 
                "Excel Auto Exporter", 
                Qgis.Warning
            )
            return self.get_default_config()
    
    def get_default_config(self):
        """
        Get default configuration structure
        
        :return: Default configuration
        :rtype: dict
        """
        return {
            "monitored_layers": {},
            "auto_export_enabled": True,
            "default_output_folder": os.path.expanduser("~/Documents/QGIS_Excel_Exports")
        }
    
    def add_monitored_layer(self, layer_name, fields, output_path):
        """
        Add a layer to monitored layers configuration
        
        :param layer_name: Name of the layer
        :type layer_name: str
        :param fields: List of field names to export
        :type fields: list
        :param output_path: Path where Excel will be saved
        :type output_path: str
        """
        config = self.load_config()
        
        config["monitored_layers"][layer_name] = {
            "fields": fields,
            "output_path": output_path
        }
        
        self.save_config(config)
    
    def remove_monitored_layer(self, layer_name):
        """
        Remove a layer from monitored layers
        
        :param layer_name: Name of the layer to remove
        :type layer_name: str
        """
        config = self.load_config()
        
        if layer_name in config["monitored_layers"]:
            del config["monitored_layers"][layer_name]
            self.save_config(config)
    
    def get_monitored_layers(self):
        """
        Get list of monitored layers
        
        :return: Dictionary of monitored layers
        :rtype: dict
        """
        config = self.load_config()
        return config.get("monitored_layers", {})
    
    def is_auto_export_enabled(self):
        """
        Check if auto-export is enabled
        
        :return: True if auto-export is enabled
        :rtype: bool
        """
        config = self.load_config()
        return config.get("auto_export_enabled", True)
    
    def set_auto_export(self, enabled):
        """
        Enable or disable auto-export
        
        :param enabled: Enable auto-export
        :type enabled: bool
        """
        config = self.load_config()
        config["auto_export_enabled"] = enabled
        self.save_config(config)
    
    def get_default_output_folder(self):
        """
        Get default output folder
        
        :return: Path to default output folder
        :rtype: str
        """
        config = self.load_config()
        return config.get("default_output_folder", os.path.expanduser("~/Documents/QGIS_Excel_Exports"))
    
    def set_default_output_folder(self, folder_path):
        """
        Set default output folder
        
        :param folder_path: Path to folder
        :type folder_path: str
        """
        config = self.load_config()
        config["default_output_folder"] = folder_path
        self.save_config(config)
