# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AddressSearch
                                 A QGIS plugin
 Plugin enables to search Estonian official addresses and cadastral units.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-12-14
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Estonian Land Board
        email                : maaamet@maaamet.ee
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path

from qgis.core import QgsApplication, Qgis
from PyQt5.QtGui import QCursor, QPixmap
from PyQt5.QtWidgets import QApplication
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QModelIndex
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import QDockWidget
from .modules.apiaddress import ApiAddress
from .modules.coordinates import Coordinates
from .modules.capture_tool import CaptureTool
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the DockWidget
from .address_search_dockwidget import AddressSearchDockWidget
from .address_search_widget_settings import AddressSearchWidgetSettings


class AddressSearch:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.canvas = self.iface.mapCanvas()

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'AddressSearch_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Estonian Address Search')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'AddressSearch')
        self.toolbar.setObjectName(u'AddressSearch')

        #print "** INITIALIZING AddressSearch"

        self.pluginIsActive = False
        self.dockwidget = None
        self.widgetSettings = None

        self.data_from_api = ""
        self.capt_tool = None
        self.capture_tool_activate = False
        self.settings_icon = QgsApplication.iconPath("mActionOptions.svg")
        self.capture_tool_icon = QgsApplication.iconPath("cursors/mCapturePoint.svg")

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('AddressSearch', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/address_search/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Estonian Address Search'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING AddressSearch"
        #set settings to remeber them
        self.set_settings()

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        #disconnects
        self.dockwidget.tb_capture.setChecked(False)
        self.canvas.unsetMapTool(self.capt_tool)
        self.capture_tool_activate = False
        self.capt_tool = None

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD AddressSearch"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Estonian Address Search'),
                action)
            self.iface.removeToolBarIcon(action)
     
        if self.capt_tool:
            self.canvas.unsetMapTool(self.capt_tool)
            self.dockwidget.tb_capture.setChecked(False)
            self.capture_tool_activate = False

        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------
    def clear_search(self):
        self.dockwidget.le_input_address.clear()
        self.api_address.set_results_model()
        self.api_address.initialize_table_widget()

    def enable_disable_capture_tool(self):
        """Activate or deactivate the geocoding tool to get the address on click on the map"""
        self.dockwidget.le_input_address.clear()
        self.api_address.set_results_model()
        self.api_address.initialize_table_widget()
        if not self.capture_tool_activate:
            self.capt_tool = CaptureTool(self.iface, self.dockwidget, self)
            self.canvas.setMapTool(self.capt_tool)
            self.canvas.setCursor(QCursor(QPixmap(self.capture_tool_icon)))
            self.capture_tool_activate = True
            self.dockwidget.tb_capture.setChecked(True)
        else:
            QApplication.restoreOverrideCursor()
            self.canvas.unsetMapTool(self.capt_tool)
            self.capt_tool = None
            self.capture_tool_activate = False
            self.dockwidget.tb_capture.setChecked(False)

    def zoom_to_address(self, index):
        """Zoom to selected address object"""
        bbox = self.api_address.get_search_response_bbox(index)
        if bbox is None:
            message = 'Address without location info'
            self.iface.messageBar().pushMessage('Info',
                                                    message,
                                                    level=Qgis.Info,
                                                    )
            return
        try:
            address_geometry = self.api_address.get_search_response_geometry(index)
            self.coord.set_x_transform_reverse()
            self.coord.set_bbox_crs(bbox)
            self.coord.zoom_to_canvas(address_geometry)
        except Exception:
            return

    def pan_to_address(self, index):
        """Pan to selected address"""
        point_xy = self.api_address.get_search_response_coordinates(index)
        if point_xy is None:
            message = 'Address without location info'
            self.iface.messageBar().pushMessage('Info',
                                                    message,
                                                    level=Qgis.Info,
                                                    )
            return
        try:
            address_geometry = self.api_address.get_search_response_geometry(index)
            self.coord.set_x_transform_reverse()
            self.coord.set_latitude_longitude_crs(point_xy)
            self.coord.pan_to_canvas(address_geometry)
        except Exception:
            return

    def on_results_view_clicked(self, index):
        """when User selects an address from results
        this address is shown on map and address info
        is shown in detailed view"""
        try:
            row = index.row()
            self.coord.set_canvas_project()
            self.coord.set_destination_crs()
            self.coord.take_crs_from_project(self.iface)
            if self.widgetSettings.rb_zoom.isChecked():
                self.zoom_to_address(row)
            else: #pan is checked
                self.pan_to_address(row)
            response_prop = self.api_address.get_selected_response_properties(row)
            self.api_address.populate_table_widget(response_prop)
        except Exception as ex:
            self.iface.messageBar().pushMessage('Warning',
                                                str(ex),
                                                level=Qgis.Warning,
                                                )

    def search_type_changed(self):
        """disble/ enable --Select address object type---
        group box when search type radio buttons are clicked"""
        if self.widgetSettings.rb_general.isChecked():
            self.widgetSettings.gb_filters.setEnabled(False)
        else:
            self.widgetSettings.gb_filters.setEnabled(True)
    
    def set_filters(self):
        """Set address search filters / address object types to search from"""
        fltr_lst = []
        if self.widgetSettings.chb_building.isChecked():
            fltr_lst.append("EHITISHOONE")
        if self.widgetSettings.chb_cadastral.isChecked():
            fltr_lst.append("KATASTRIYKSUS")
        if self.widgetSettings.chb_traffic.isChecked():
            fltr_lst.append("TANAV")
        if self.widgetSettings.chb_ehak.isChecked():
            fltr_lst.append("EHAK")
        if self.widgetSettings.chb_small_place.isChecked():
            fltr_lst.append("VAIKEKOHT")
        if  len(fltr_lst) > 0:
            fltr_join = ",".join(fltr_lst)
            fltr = "&features=" + fltr_join
            return fltr
        else:
            return None

    def set_search_params(self):
        """Reads user settings and sets address search parameters"""
        # how many results to show:
        results_txt = self.widgetSettings.tb_restults_nr.toPlainText()
        if results_txt.isdecimal():
            if 0 < int(results_txt) <= 100:
                results = int(results_txt)
        else:
            results = 10
        self.api_address.results = str(results)

        # if unique addresses only:
        if self.widgetSettings.chb_unique.isChecked():
            self.api_address.unique = '1' 
        else:
            self.api_address.unique = '0' 

        # if general address search (without filters) is checked:
        if self.widgetSettings.rb_general.isChecked():
            self.api_address.url_params = "{base_url}results={results}&unik={unique}&geometry=1&address=".format(base_url = self.api_address.BASE_URL,
                                results = self.api_address.results, unique = self.api_address.unique)
        # else search with filters and what filters / data stores to use:
        else:
            self.api_address.fltr = self.set_filters()
            self.api_address.url_params = "{base_url}results={results}&unik={unique}&geometry=1{fltr}&address=".format(base_url = self.api_address.BASE_URL,
                                results = self.api_address.results, unique = self.api_address.unique, fltr = self.api_address.fltr)

    def address_processing(self):
        """Launch the address processing"""
        address_entry = self.dockwidget.le_input_address.text()
        if len(address_entry) == 0:
            self.clear_search()
        if len(address_entry) < 3:  # api gives error when address string is less than 3:
            return

        self.api_address.set_search_url(address_entry)
        self.api_address.set_request()
        if self.api_address.check_if_any_addresses_found():
            self.api_address.populate_results_view()
            self.api_address.initialize_table_widget()
            response_properties = self.api_address.get_selected_response_properties(0)
            self.api_address.populate_table_widget(response_properties)
        else:
            self.api_address.set_results_model()
            self.api_address.initialize_table_widget()
            message = 'Address not found'
            self.iface.messageBar().pushMessage('Info',
                                                    message,
                                                    level=Qgis.Info,
                                                    )

    def set_visible_properties(self, state):
        """Shows selected address detail view"""
        if state == Qt.Checked:
            self.dockwidget.tw_details.setVisible(True)
        else:
            self.dockwidget.tw_details.setVisible(False)

    def set_connections(self):
        self.api_address.dialog.lv_results.clicked[QModelIndex].connect(
            self.on_results_view_clicked
            )
        self.dockwidget.chb_view_details.stateChanged.connect(
            self.set_visible_properties
            )
        self.dockwidget.tb_settings.clicked.connect(
            self.show_result_setting
            )
        self.dockwidget.tb_capture.clicked.connect(
            self.enable_disable_capture_tool
            )
        #auto search when typing:
        self.dockwidget.le_input_address.textEdited.connect(
            self.address_processing
        )
        self.widgetSettings.pb_ok.clicked.connect(
            self.set_search_params
        )
        self.widgetSettings.pb_ok.clicked.connect(
            self.set_settings
        )
        self.widgetSettings.pb_ok.clicked.connect(
            self.widgetSettings.close
        )
        self.widgetSettings.pb_cancel.clicked.connect(
            self.read_settings
        )
        self.widgetSettings.pb_cancel.clicked.connect(
            self.widgetSettings.close
        )
        # closing settings from X button / spontaneously (not from Cancel or OK buttons i.e. unspontaneously): 
        self.widgetSettings.closing_spontaneous.connect(
            self.read_settings
        )
        self.widgetSettings.rb_general.toggled.connect(
            self.search_type_changed
        )

    def show_result_setting(self):

        if self.widgetSettings is None:
            self.widgetSettings = AddressSearchWidgetSettings()

        self.widgetSettings.show()

    def set_settings(self):
        """Sets the user settings to remeber them"""
        self.settings.setValue("results", self.api_address.results)
        self.settings.setValue("unique", self.api_address.unique)
        self.settings.setValue("fltr", self.api_address.fltr)
        self.settings.setValue("zoom", self.widgetSettings.rb_zoom.isChecked())
        self.settings.setValue("search_general", self.widgetSettings.rb_general.isChecked())

    def read_settings(self):
        """Reads user settings from previous session"""
        if self.settings.value("search_general"):
            self.widgetSettings.rb_general.setChecked(True)
            self.widgetSettings.gb_filters.setEnabled(False)
        else:
            self.widgetSettings.rb_object.setChecked(True)
            self.widgetSettings.gb_filters.setEnabled(True)

        self.api_address.results = self.settings.value("results")
        self.widgetSettings.tb_restults_nr.setPlainText(self.settings.value("results"))

        self.api_address.unique = self.settings.value("unique")
        if self.api_address.unique == '1': 
            self.widgetSettings.chb_unique.setChecked(True)
        else:
            self.widgetSettings.chb_unique.setChecked(False)

        if self.settings.value("zoom"):
            self.widgetSettings.rb_zoom.setChecked(True)
        else:
            self.widgetSettings.rb_pan.setChecked(True)

        self.api_address.fltr = self.settings.value("fltr")
        if self.api_address.fltr is None:
            self.widgetSettings.rb_general.setChecked(True)
            self.widgetSettings.gb_filters.setEnabled(False)
            return
        if 'EHITISHOONE' in self.api_address.fltr:
            self.widgetSettings.chb_building.setChecked(True)
        else:
            self.widgetSettings.chb_building.setChecked(False)
        if 'KATASTRIYKSUS' in self.api_address.fltr:
            self.widgetSettings.chb_cadastral.setChecked(True)
        else:
            self.widgetSettings.chb_cadastral.setChecked(False)
        if 'TANAV' in self.api_address.fltr:
            self.widgetSettings.chb_traffic.setChecked(True)
        else:
            self.widgetSettings.chb_traffic.setChecked(False)
        if 'EHAK' in self.api_address.fltr:
            self.widgetSettings.chb_ehak.setChecked(True)
        else:
            self.widgetSettings.chb_ehak.setChecked(False)
        if 'VAIKEKOHT' in self.api_address.fltr:
            self.widgetSettings.chb_small_place.setChecked(True)
        else:
            self.widgetSettings.chb_small_place.setChecked(False)

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING AddressSearch"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = AddressSearchDockWidget()
                self.widgetSettings = AddressSearchWidgetSettings()
                self.settings = QtCore.QSettings(QtCore.QSettings.IniFormat,
                               QtCore.QSettings.UserScope,
                               'Estonian Land Board', 'Estonian Address Search')
                self.dockwidget.tw_details.setVisible(False)
                self.api_address = ApiAddress(self.dockwidget)
                self.coord = Coordinates(self.dockwidget, self.canvas)
                self.read_settings()
                self.set_search_params()
                self.dockwidget.tb_settings.setIcon(QIcon(self.settings_icon))
                self.dockwidget.tb_capture.setIcon(QIcon(self.capture_tool_icon))
                self.set_connections()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
