from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsRectangle,
    QgsWkbTypes,
    QgsPoint,
    QgsPointXY,
    QgsGeometry,
    QgsMessageLog,
    Qgis
)
from qgis.gui import QgsRubberBand
from PyQt5.QtCore import Qt, QTimer


class Coordinates:

    def __init__(self, dialog, canvas):
        self.dialog = dialog
        self.longitude = 0.0
        self.latitude = 0.0
        self.long_min = 0.0
        self.lat_min = 0.0
        self.long_max = 0.0
        self.lat_max = 0.0
        self.scale = 50
        self.project_canvas = ""
        self.destination_crs = ""
        self.x_transform = ""
        self.x_transform_reverse = ""
        self.epsg_project = ""
        self.point_project = []
        self.bbox_l_est = []
        self.canvas = canvas
        self.address_geom = QgsRubberBand(canvas, QgsWkbTypes.PolygonGeometry)
        self.address_geom.setStrokeColor(Qt.red)
        self.address_geom.setWidth(3)


    def set_canvas_project(self):
        """Return the canvas of the project"""
        self.project_canvas = self.canvas.mapSettings().destinationCrs()
        return self.project_canvas

    def set_destination_crs(self):
        """Return the L-EST crs"""
        self.destination_crs = QgsCoordinateReferenceSystem.fromEpsgId(3301)
        return self.destination_crs

    def set_x_transform(self):
        """Return the x transform from project crs to destination crs"""
        self.x_transform = QgsCoordinateTransform(
            self.project_canvas,
            self.destination_crs,
            QgsProject.instance())
        return self.x_transform

    def set_x_transform_reverse(self):
        """Return the coordinate transform reverse - to project crs"""
        try:
            self.x_transform_reverse = QgsCoordinateTransform(
                QgsCoordinateReferenceSystem.fromEpsgId(3301),
                QgsCoordinateReferenceSystem.fromEpsgId(
                    int(self.epsg_project.lstrip("EPSG:"))),
                QgsProject.instance())
        except Exception as e:
            self.message_log('Could not transform coordinates to project CRS')
            raise RuntimeError('Could not transform coordinates to project CRS') from e
        return self.x_transform_reverse


    def set_latitude_longitude_wgs84(self):
        """Set the latitude and the longitude coordinates for project crs"""
        self.longitude = self.point_project[0]
        self.latitude = self.point_project[1]

    def set_bbox_wgs84(self):
        """Set the latitude and the longitude bbox coordinates"""
        self.long_min = self.bbox_l_est[0]
        self.lat_min = self.bbox_l_est[1]
        self.long_max = self.bbox_l_est[2]
        self.lat_max = self.bbox_l_est[3]

    def set_latitude_longitude_crs(self, point):
        """Set the latitude and the longitude coordinates for destination crs"""
        self.longitude = point[0]
        self.latitude = point[1]

    def set_bbox_crs(self, bbox):
        """Set the latitude and the longitude bbox 
        coordinates for destination crs"""
        self.long_min = float(bbox[0].split(",")[0])
        self.lat_min = float(bbox[0].split(",")[1])
        self.long_max = float(bbox[2].split(",")[0])
        self.lat_max = float(bbox[2].split(",")[1])

    def set_point_to_wgs84(self, point):
        """Transform the destination crs point 
        to the project crs"""
        self.point_project = self.x_transform.transform(point)
        return self.point_project

    def set_bbox_to_wgs84(self, bbox):
        """Transform the crs points 
        to the project crs"""
        long_min_l_est = float(bbox[0].split(",")[0])
        lat_min_l_est = float(bbox[0].split(",")[1])
        long_max_l_est = float(bbox[2].split(",")[0])
        lat_max_l_est = float(bbox[2].split(",")[1])
        self.bbox_l_est = self.x_transform.transform(QgsRectangle(
            long_min_l_est,
            lat_min_l_est,
            long_max_l_est,
            lat_max_l_est,
        ))
        return self.bbox_l_est

    def zoom_to_canvas(self, footprint_geom):
        """Zoom to the bbox"""
        rectangle = self.x_transform_reverse.transform(QgsRectangle(
            self.long_min,
            self.lat_min,
            self.long_max,
            self.lat_max
        ))
        self.canvas.setExtent(rectangle)
        if footprint_geom is not None:
            self.draw_footprint(footprint_geom)
        self.canvas.refresh()

    def pan_to_canvas(self, footprint_geom):
        """pan to the point"""
        rectangle = self.x_transform_reverse.transform(QgsRectangle(
            self.longitude,
            self.latitude,
            self.longitude,
            self.latitude
        ))
        self.canvas.setExtent(rectangle)
        if footprint_geom is not None:
            self.draw_footprint(footprint_geom)
        self.canvas.refresh()

    def reset_rubberbands(self):
        self.address_geom.reset()

    def draw_footprint(self, geom):
        self.address_geom.reset(QgsWkbTypes.PolygonGeometry)
        crs = QgsCoordinateReferenceSystem.fromEpsgId(4180)

        if geom['type'] != 'Polygon':
            return

        for p in geom['coordinates']:
            coords = []
            for coord_pair in p:
                coords.append(coord_pair)
            plgn = QgsGeometry.fromPolygonXY( 
                [[ QgsPointXY( pair[0], pair[1] ) for pair in coords ]] )
            self.address_geom.addGeometry(plgn, crs)

        QTimer.singleShot(1700, self.reset_rubberbands)


    def take_crs_from_project(self, iface):
        """Take the EPSG from the project
        Returns:
            [object]
        """
        self.epsg_project = iface \
            .mapCanvas() \
            .mapSettings() \
            .destinationCrs() \
            .authid()
        if self.epsg_project == '':
            raise ValueError('Project CRS must be defined to show the address object on the map!')
        return self.epsg_project

    
    def message_log(self, msg=""):
        QgsMessageLog.logMessage('{} {}'.format(self.__class__.__name__, msg), 'Estonian Address Search', Qgis.Warning)
