"""Manage the tool"""
from qgis.gui import QgsMapTool
from qgis.core import Qgis, QgsMessageLog
from .coordinates import Coordinates
from .apiaddress import ApiAddress


class CaptureTool(QgsMapTool):
    def __init__(self, iface, dialog, address_instance):
        QgsMapTool.__init__(self, iface.mapCanvas())
        self.canvas = iface.mapCanvas()
        self.iface = iface
        self.dialog = dialog
        self.address_instance = address_instance
        self.coord = Coordinates(self.dialog, self.canvas)
        self.api_address = ApiAddress(self.dialog)

    def canvasReleaseEvent(self, event):
        x = event.pos().x()
        y = event.pos().y()
        self.coord.set_canvas_project()
        self.coord.set_destination_crs()
        self.coord.set_x_transform()
        point = self.canvas.getCoordinateTransform().toMapCoordinates(x, y)
        self.coord.set_point_to_wgs84(point)
        self.coord.set_latitude_longitude_wgs84()
        self.api_address.set_search_url_xy(self.coord.longitude, self.coord.latitude)
        self.dialog.le_input_address.clear()

        if self.api_address.set_request():
            self.address_instance.data_from_api = self.api_address.content
            self.api_address.initialize_table_widget()
            if self.api_address.check_if_any_addresses_found():
                response_label = self.api_address.get_reverse_response_label()
                response_properties = self.api_address.get_selected_response_properties(0)
                self.dialog.le_input_address.setText(response_label)
                self.api_address.set_results_model()
                self.api_address.populate_table_widget(response_properties)
            else:
                message = " no address was not found with given coordinates "
                message_error = message + f'EPSG:3301 lon,lat = {self.coord.longitude},{self.coord.latitude}'
                self.message_log(message_error)
                self.iface.messageBar().pushMessage('Warning',
                                                    message_error,
                                                    level=Qgis.Warning,
                                                    )
        else:
            message = " An error was returned for the request:  "
            message_error = message +  self.api_address.url
            self.message_log(message_error)
            self.iface.messageBar().pushMessage('Warning',
                                                message_error,
                                                level=Qgis.Warning,
                                                )

    def activate(self):
        message = " click on map to get the address..."
        self.iface.messageBar().pushMessage('Info',
                                            message,
                                            level=Qgis.Info,
                                            )
        self.dialog.tb_capture.setChecked(True)
        self.address_instance.capture_tool_activate = True

    def deactivate(self):
        QgsMapTool.deactivate(self)
        self.dialog.tb_capture.setChecked(False)
        self.address_instance.capture_tool_activate = False
        self.deactivated.emit()
   
    def message_log(self, msg=""):
        QgsMessageLog.logMessage('{} {}'.format(self.__class__.__name__, msg), 'Estonian Address Search', Qgis.Info)