import urllib.parse
import json
import ssl
import requests
from urllib.request import urlopen
from qgis.core import Qgis, QgsMessageLog
from qgis.PyQt.QtWidgets import QTableWidgetItem
from PyQt5.Qt import QApplication, QUrl, QDesktopServices
from qgis.PyQt.QtCore import QTranslator, QCoreApplication, QModelIndex
from PyQt5.QtGui import QStandardItem, QStandardItemModel

class ApiAddress:
    def __init__(self, dialog=None):
        self.dialog = dialog
        self.BASE_URL = "https://inaadress.maaamet.ee/inaadress/gazetteer?"
        self.url = ""
        self.url_params = ""
        self.unique = "false"
        self.results = "10"
        self.fltr = ""
        self.json_data = ""
        self.content = {}
        self.model = QStandardItemModel()

        self.error_message_no_address_locate = "Address not found with given coordinates"
        self.error_message_no_address_found = "Address not found"
        self.error_message_connection = "Connection failed"
        self.success_message_connection = "Connection established"


    def set_search_url(self, address_string):
        """Sets the search url with address from input 
        and user defined parameters"""
            
        adr_str_parse = urllib.parse.quote(address_string, safe='\'')

        self.url = self.url_params + adr_str_parse

        return self.url

    def set_search_url_xy(self, x, y):
        """Sets the search url with coordinates from capture tool
        This url is used for the reverse geocoding"""

        self.url = self.BASE_URL \
                                    + 'x=' + str(x) + '&' + 'y=' + str(y)
        return self.url


    def set_request(self):
        """Open the url"""
        try:
            r = requests.get(url=self.url, timeout=10)
            self.content = r.json()
            return self.content
        except Exception:
            self.message_log(
                f'{self.error_message_no_address_found}: {self.content}')
            return False


    def check_if_any_addresses_found(self):
        """Checks if any addresses were returned from the response"""
        try:
            if "addresses" in self.content:
                if len(self.content['addresses']) > 0:
                    return True
                return False
            return False
        except Exception:
            self.message_log('Could not get the addresses from address api')
            return False

    def get_selected_response_properties(self, index):
        """Returns selected address object properties"""
        try:
            properties = \
            self.content['addresses'][index]
            return properties
        except Exception:
            self.message_log('Could not get selected address properties')
            return False

    def recode_type(self, in_type):
        """Sets abbrevation to the response address type"""
        if in_type == "KATASTRIYKSUS":
            return "[KÜ] "
        elif in_type == "EHITISHOONE":
            return "[H] "
        elif in_type == "EHAK":
            return "[HÜ] "
        elif in_type == "VAIKEKOHT":
            return "[VK] "
        elif in_type == "TANAV":
            return "[LP] "
        else:
            return "-- "

    def populate_results_view(self):
        """Shows list of found addresses and their types 
        (e.g. building, cadastral unit, street..)"""
        try:
            self.set_results_model()
            for v in self.content['addresses']:
                in_type = v['liikVal']
                adr = self.recode_type(in_type) + v['ipikkaadress']
                itm = QStandardItem(adr)
                self.model.appendRow(itm)
        except Exception:
            self.message_log('Could not get the addresses from address api')

    def set_results_model(self):
        """Refresh the results view"""
        self.dialog.lv_results.setModel(self.model)
        self.model.clear()

    def get_reverse_response_label(self):
        """Retruns the found address when searching with coordinates"""
        try:
            data = self.content['addresses'][0]
            found_address = data['ipikkaadress']
            return found_address
        except Exception:
            self.message_log('Could not get the address from address api')
            return None

    def get_search_response_coordinates(self, index):
        """Returns the coordinates of the response at specified index"""
        try:
            address = self.content['addresses'][index]
            x = float(address['viitepunkt_x'])
            y = float(address['viitepunkt_y'])
            search_coordinates = (x, y)
        except Exception:
            self.message_log('Could not get the selected address coordinates')
            return None
        return search_coordinates

    def get_search_response_bbox(self, index):
        """Returns the bounding box of the response at specified index"""
        try:
            address = self.content['addresses'][index]
            bbox_str = address['boundingbox']
            bbox = bbox_str.split(" ")
        except Exception:
            self.message_log('Could not get the selected address bounding box')
            return None
        return bbox

    def get_search_response_geometry(self, index):
        """Returns the geometry in geographical 
        coordinates of the response at specified index"""
        try:
            address = self.content['addresses'][index]
            address_geometry_str = address['geometry']
            address_geometry = json.loads(address_geometry_str) #to json
        except Exception:
            self.message_log('Could not get the selected address geometry')
            return None
        return address_geometry

    def initialize_table_widget(self):
        """Initializes address info/ details GUI table"""
        self.dialog.tw_details.clear()
        self.dialog.tw_details.setRowCount(0)
        self.dialog.tw_details.setColumnCount(2)
        head_attribute = "attribute"
        head_value = "value"
        self.dialog.tw_details.setHorizontalHeaderItem(
            0, QTableWidgetItem(head_attribute))
        self.dialog.tw_details.setHorizontalHeaderItem(
            1, QTableWidgetItem(head_value))

    def populate_table_widget(self, data):
        """Shows all data from api of selected address"""
        self.dialog.tw_details.setRowCount(len(data))

        i = 0
        for attribute, value in data.items():
            self.dialog.tw_details.setItem(
                i, 0, QTableWidgetItem(str(attribute)))
            self.dialog.tw_details.setItem(
                i, 1, QTableWidgetItem(str(value)))
            i += 1

    def message_log(self, msg=""):
        QgsMessageLog.logMessage('{} {}'.format(self.__class__.__name__, msg), 'Estonian Address Search', Qgis.Info)