import os
import codecs

from qgis.PyQt import QtCore, uic, Qt
from qgis.PyQt.QtWidgets import QDialog, QDialogButtonBox
from qgis.core import (
    QgsMessageLog,
    Qgis,
    QgsVectorLayer,
    QgsFeature,
    QgsProject,
    QgsGeometry,
    QgsPointXY,
)

from .polyline import decode as polyline_decode


def get_ui_class(ui_file):
    """Get UI Python class from .ui file.
       Can be filename.ui or subdirectory/filename.ui
    :param ui_file: The file of the ui
    :type ui_file: str
    """
    os.path.sep.join(ui_file.split("/"))
    ui_file_path = os.path.abspath(
        os.path.join(os.path.dirname(__file__), ui_file))
    return uic.loadUiType(ui_file_path)[0]


EXAMPLES = [
    {
        "layer_name": "Ring Road Yogya (precision = 5)",
        "precision": 5,
        "encoded_polyline": r"norn@i{j`TzC{RwHcHy[aBgs@gBmy@sCw_AcUcz@gRkq@wPy\kQiB}Bg@cE|@eHbKye@fSww@tJwv@zCaYEgLl@yFdCuI~JyOpN_OpLmMtA}H_Hud@u@aTXyG~CiR~Nut@bUk}A|CwS~AuAtCeAxIRryAhHQrwB`~A~Ang@zBpL|@xH~@pPy@",
    },
    {
        "layer_name": "Routing",
        "precision": 6,
        "encoded_polyline": r"ezeqxAi~kfMBjBHvBJrAZvCZrCnP~xA`DpYPdBJz@b@dEPdCPnB`@vFz@rKB`@`AjMdC|[v@nJvAnQ\\vE`BjU_@Fa@HaARui@rLLbBzAzR`@vF^|ElFpr@HfA",
    },
    {
        "layer_name": "Euler Route",
        "precision": 6,
        "encoded_polyline": r"g{|pxA_nkeMlFmClCvBdAd@l@Vk@nD_@|Ao@vAa@n@[z@K`@Kj@Er@?d@Dj@Df@Jd@LXHRTZXXb@TXDxMr@\\F\\HRFZJZTZ\\bD`FVh@Tf@Rb@Rj@Ph@Pv@Lv@Jv@Hx@BjA?dAAdAGjAKtAeCl[M~AIhA?nA@pALvAZbA`@z@`@b@b@^p@Zn@Lv@B|@Ev@M|@QtHyC`DOpFmC|DyDvAiBvBjEbQyJxC{AxCaCrElBdEz@L\\mClDuA`FeAzFZlApGkFzBw@`F{GzHqLjf@uu@dF{Ix@cBf@iAvBwAbDuDrKwPp@gA`GkJjJmQ|GuO`F_MxDqLzFcRzAiG~AkG~AgHpHe]vEmSnHq]`FgU^kBR_Ad@yBh@sBpBwKhBaL`B_ML}@tAwLnAsNbAaOr@{Nb@wNXiODgHAuGGwTIiIUuGg@{Ko@eMc@iIwGidAkFs{@o@cDa@}IWaBw@aBLsDFeHIkLK{NIaRKwJq@aOq@sOe@wHSkDY_IAeBDgBFsAPoB~BkSz@eFnAoFxA{Fn@yBt@{BlA{C~@kBp@mAr@iAx@mAv@aAnAwAvBqBtG}DZUrFsFu@sBsAuDH]D_@Aa@Ea@Ke@I[wB_GOy@Iu@Gq@Eq@CcA]k@UUi@i@QuACYwi@tQgGzBaDbBkFvDgE|CSJODMAKCEIDHJBL@NERKfE}CjFwD`DcBfG{Bvi@uQBXCYwi@tQgGzBaDbBkFvDgE|CSJODMAKCEIa@u@~LmRzRiWzBkCnOsNnB}BXo@^mATw@Uv@_@lAYn@oB|BoOrN{BjC{RhW_MlR`@t@jMeRjSyWvAaBpQ}PzA_CRg@Nk@Oj@Sf@{A~BqQ|PwA`BkSxWkMdRwO`VgArBo@xAm@dBc@fB_@tB[xBoAzMIrAErB?hDHhDTlDXzCb@dDh@pCd@~Ah@dBp@tBxA~Ci@p@h@q@T[ISHRUZFP|KpSv@hBh@bBJd@Jb@Hj@Db@B~@Ar@MrDAxA\\rR\\lOcArDy@aa@AaBDuCDgBNqD@i@Ai@Gk@[sAe@}Aw@oBkD{GuEiIk@x@j@y@_@u@mAmCcAmCiA}Dk@wCi@sDa@oDWeDI_DAaDFeCRwCdAkLZaCd@iCn@aCl@iBn@wAfAmBnAiBVc@vLwQa@s@ECEAC?E@EBcKbHmC~AoVjLyFvCyF_H_C{@AnAw@CyCfAuA@oE_@eISsBGo@?n@?rBFdIRnE^tAAxCgAv@B@oABaDoFJqCEcHa@gGqBgF}C}H{EjGtJ|C|Gf@fAXf@^r@nBrCvBbDGvBFwBwBcDoBsC_@s@Yg@g@gA}C}GkGuJgVaMo@Wg@Eo@Bn@Cf@DzBqHvA{EEWrBUT?TDZH[IUEU?sBTCc@?a@Bs@Hi@Je@JY|AqDvVed@v\\qm@w\\pm@wVdd@}ApDKXKd@Ih@Cr@cD{BkFuD_@@a@I}A~FiBlHeb@cZKICICKCMAMFaB?KCKEEEEgT}Nz@gD{@fD}AhGmAzElA{E|AiGfT|NDDDDBJ?JG`B@LBLBJBHJHdb@bZu@fDk@rCuD`Ro@nCn@oCtDaRj@sCt@gDhBmH|A_G}^oWmA{@yA|FxA}Fq@e@O@k@h@ODQAmh@c^lh@b^P@NEj@i@NAp@d@lAz@|^nW`@H^AVM`@]b@m@h@_Ab@u@hW_e@EECCEGg@u@{G~LzG_MnCcFoCbFf@t@DFBBDDFBHDF@H@H@L?LAJANEJGHIJMdOiXxLqTyLpTyBeDpK_Sb@w@|BjD}BkDfFwJp@kAq@jAgFvJc@v@w@kA]i@Ou@{NiTGKEMAQBODMdBaDDMBM@SA}@?SBQDOjAyBlTn\\mTo\\KMyJoNr@cAs@bASYOMSKRJNLRXoEvFgKrOqCrE}AnBqBtDaCdFqA|CiAlC_AbCa@jA`@kA~@cChAmCpA}C`CeFpBuD|AoBpCsEfKsOnEwFxJnNr@y@h@m@tCcDuCbDi@l@vTj\\l@gAn@gAdHmMeHlMo@fAm@fAwTk\\yJyN`JiKN\\O]nLuM|@eAt@cAxTbi@BFZr@l@kALNf@j@g@k@MOm@jAp@tA{EzIzE{IXZ^T^Nj@Nc@rCb@sCXiBYhBNBD?bFh@dAjERb@lA|B`ItH~DsEpFoFhCgCiCfCVt@^`AVp@Wq@_@aAWu@qFnF_ErEaIuHmA}BSc@v@eACIAKAG?I?G@GBI@GFIjD_FbH}JxCbGp@aAZe@fA{AgAzA[d@q@`A^v@FLBVETGJyDjFxDkFFKDUCWGM_@w@yCcGkAaCjA`CvGmJmAaClA`Cf@bAFNBTETGLeAxAdAyAFMDUCUGOg@cAwGlJcH|JgAiCqA{CSVKHMHKDOBO?Q?SGOGoBzBwBxCvByCnB{BOQMUaF}L_AmCg@aB[iAUw@m@mC]aBqBjApBkAd@Y|@k@}@j@Pt@`Am@aAl@Qu@e@X\\`Bl@lCf@_@|BnIT[kB{IjBzIUZ}BoIg@^Tv@ZhAf@`B~@lC`F|LLTNPNFRFP?N?NCJELIJIRWfRqW`I}Kt@_AfAhCgAiCn@w@zF{HnAyAjCgDM]GOQa@kDgJKJJKjDfJP`@FNL\\hBoBDCbEsDlEiDr@nCs@oCmEhDcErDEBM]aBiEQaBi@{Ah@zAP`B`BhEL\\iBnBkCfDoAxAcBqEbBpEbAhCcAiC{FzHo@v@gHiS~@aAp@s@bFkFcFjFq@r@_A`AfHhSu@~@oLe]WToAlAIFIFKBIAMEs@k@KSmAgDEKIESEK?GHCJCR?PHpAi@tAh@uAIqA?QBSBKFIJ?RDHDDJlAfDJRr@j@LDH@JCHGHGnAmAVUnLd]aI|KnAjCoAkC]q@[[_@QcBe@g@C_CN_A?cAKkAWi@Q[QW]Sa@Mc@I_@Cc@@c@F_@J[JSdBoCFUBUAUEQGO}DgEc@wCb@vC|DfEFNDP@TCTGTeBnCKRKZG^Ab@Bb@H^Lb@R`@V\\ZPh@PjAVbAJ~@?~BOf@BbBd@^PZZ\\p@gRpWpAzCfAhCkD~EGHAFCHAF?F?H@F@JBHw@dA}@rAGEICGAE?GBGBGFEDo@`AoYlh@uAuBtAtBnYmh@`@p@cAlBNVfAbB\\h@dBrCfGxJgGyJjByDnGlJl@oAtCbEHJIKuCcEm@nAsBdErBeEoGmJkBxDeBsCkErH`CxD|DnGyCfFS^R_@xCgFdC~DeC_E}DoGyCfFU^_G`Kg@z@zAbC`BjChH}LiH|LaBkC{AcCf@{@aCyD}CnF_C~Dr@fAs@gA~B_E|EzHz@wA`@s@Ve@Wd@a@r@{@vA}B~DaAbB`AcBkDsFsQb[rQc[jDrFhDrFiDsF|B_EnGdKzAbC|Wmd@}Wld@{AcC}B~DsQ`[W_@c@s@i@{@wAbCvAcCaA}AsEmH_A{AwA`CvAaCmAoBOIOF{DzGk@bAATaHjN`HkN@UGSq@aAQWuAuBtAtBPVp@`AFRj@cAzD{GNGNHlAnB~@zAdAiBzL}ShBvCPWQVhBvC{L|SzL}SiBwCiBwC{L|SeAhBrElHdAgBsEoHrEnHeAfBuAbCsEmHtAcCuAbCaA}A`A|ArElH~@|A_A}AtAcC`A|ArQc[dBpCeBqCsQb[h@z@b@r@V^iOfWnAxCoAyCy@gBaH}HoEiFnEhF`H|Hx@fBhOgWrQa[|B_ElQwZjEsH_CuD~BtD~A`Cl@cAb@u@c@t@m@bApEpHfIeN^m@_@l@gIdNfL|QbC~KrEiB@FfAcApMyJqMxJgAbAAGsEhBeZzY}`@l`@{FnDmFrAa@s@a@u@cGcKm@|@[l@aHnLoW{H^gBHOE{AdFkJb@sBbDwGcDvGc@rBeFjJDzAIN_@fBQEPDnWzHpCf@bEXhAFnDAnAKvGcBdGuDxaA_aAyaA~`AeGtDwGbBoAJoD@rCdA~Bn@`C@~Fg@lDmAxEsCdPaMfKsIfM{JzKyIrFqErGaFdG{C~EyBlEaBdGaBzHcBrF{@jD_AbJeBzC_ArE}@`Bc@|EgBHg@e@A_RE}QMkIGcGj@Iq@a@aDMoA_AX{@NoF`@gFCeGS}D]_BWwAa@mAi@mBcAi@a@{@o@gCuBe@c@{@y@pBO@?vKy@Se@Uo@Uw@WgA[cBIaACo@@q@Dg@Fc@Ry@fD_M`@cA^k@l@s@xDiDCOgEgWGc@?i@JsCFe@N]nIuKoItKO\\Gd@KrC?h@Fb@fEfWBN|CqCDPTp@Uq@EQ}CpCyDhDm@r@_@j@a@bAgD~LSx@Gb@Ef@Ap@Bn@H`AZbBVfATv@Tn@Rd@vFa@ZUX_@jLuWkLtWY^[TZUX_@jLuWgC}HfC|HkLtWY^[TwF`@wKx@A?qBNoQwPnQvPz@x@d@b@fCtBz@n@vDhNwDiNh@`@lBbAlAh@vA`@~AV|D\\dGRfFBnFa@z@O~@YLnA`@`DHp@gNhDmG~Bo@d@I]c@Zb@[H\\n@e@lG_CfNiDbGk@jIF|QL~QDd@@If@}EfBaBb@sE|@{C~@cJdBkD~@sFz@{HbBeG`BmE`B_FxBeGzCsG`FsFpE{KxIgMzJgKrIeP`MyErCmDlA_Gf@aCA_Co@sCeAiAGcEYqCg@`HoLZm@l@}@bGbK`@t@`@r@lFsAzFoD|`@m`@uT}]yd@`x@xd@ax@tT|]dZ{YcC_LgL}QqEqH_BaCkErHg@y@f@x@mQvZoGeK}E{HwBkDQYPXvBjD|CoF`CxD~FaKT_@xCgFaCyDjEsH]i@gAcBOWbAmBa@q@sAwBcBmCqO|XaCqD`CpDkIdO\\JpAlBqAmB]KjIeOpO}XbBlCrAvBn@aADEFGFCFCD?F@HBFD|@sAeAkEcFi@E?OCk@O_@O_@UY[q@uA[s@CGyTci@u@bA}@dAoLtMaJhKiA}Aa@lBYCSBQFWNYXmDbF_E`GiC|DyBnDkBfDqBvDAB@CpBwDjBgDxBoDhC}D~DaGlDcFXYVOPGRCXB`@mBDg@B]?YGWsA_COOMG}TuIeA_@iA]qRyF{@SaFi@{@?k@Ai@nBgCrIiQzl@GREXEb@YHWTWTW`@O\\KZSj@cC`ImDdLwDpK}IvZcEgCeByAaIqFy}@kn@x}@jn@`IpFK`@k@xBg@pBI^cn@ac@bn@`c@LJLVNXHTBNBH?DzB~A{B_BcBhFeGiEiOoKs@pCr@qChOnKs@pCr@qCdGhE|BbB}BcBk@pCwFvOiDbLO`Ae@lDMl@mBnHQl@kD~Kq@jBOd@c@nAMVe@nA[v@e@~@cArBs@pAg@~@GKyA}B_CbE~BcE|DuO}DtOxA|BFJz@pAf@_At@sAbAuBp@sAq@rAcAtBu@rAg@~@gAnBqCvEiChEcClDm@t@e@l@_CdC~BeCd@m@l@u@bCmDhCiEbA~A^`@fI|EhEG`OhIaOiIiEFgI}E_@a@kCnEgCpDq@x@e@l@yA~AxA_Bd@m@LT~@|Ahx@`k@pHuWqHtWix@ak@_A}AMUp@y@fCqDjCoEXe@vBqDhAoBl@gAn@mAdAuBv@eBtIfILFP@LCHKlE}MmE|MIJMBQAMGhFbGRq@Sp@tM`OuMaOiFcGuIgImAkAlAjAPa@h@uAp@kBdAyC~DqM|IoYwAeA~IgXHUIT_JfX_JfYyDrM_AvCq@hBg@pASh@Ri@f@qAp@iB~@wCxDsM~IgYvAdA|P`M~NrKjP`LrSjOtJhHuJiHYbANb@HZ?PCJMNQLMJKPGZEp@GLUbAIdAAp@Id@c@|Ab@}AHe@@q@HeATcAFMRWNIRCL@JBXLvC~BJLHPPl@HXLRDFVTt@kCh@}BpDpCqDqCi@|Bu@jCaFhQnAbAuTbx@tTcx@h@`@nAdAoAeAi@a@~D_NvAeCjC_DkC~CwAdC_E~MoAcAaPjj@mHcEWWS_@G_@Ag@F]JYPYjAqALOHSV_AW~@IRMNkApAQXKXG\\@f@F^R^VVlHbEcDhLeD|Kw@jC~AhAzBzAt@aCdUux@jCgJkCfJeUtx@sB}A~C}K_D|KeBqA}_@iX|_@hXdBpArB|AbNvK~@n@n@RpLA@dDAeDCcDzDI{DHBbDqL@o@S_Ao@cNwKu@`C}@vCcCgBe]hgAd]igAuAeA{m@ac@n@uCkDcCjDbChM`Jv@oCw@nCiMaJo@tCzm@`c@z@oC}_@eX|_@dX{@nCiRzl@hR{l@tAdAdAkC|@cC}@bCeAjCbCfBUv@kc@fsAoB}@fE_NgE~MnB|@eA~CcA~CiBgAiBgA_YiN~XhN~@qCdA{C{g@uV|@_F}@~Ei@xCyZwNxZvNh@yCkfA{h@vDaTyDyCfFcF_@iA^hAlCiCmChCgFbFxDxCwD`Tw@_@w@_@v@^bBkJR_AjAwCkAPq@dAYzAcBnJbBoJX{Ap@eAjAQkAvCS~@cBjJv@^jfAzh@zg@tVeAzCjB~@aAxChBfAbA_DdA_Djc@gsATw@hN|Kr@f@s@g@iN}K|@wC~MtK?fEd@ZdBh@rMAsM@eBi@e@[?bI?vIGtISfI]dIg@zGy@fJYxBqC`TiDgBme@}Tle@|ThDfBpCaTXyBx@gJf@{G\\eIRgIFuI?wI?cI?gEn@f@n@d@l@Tv@JvKGfD]gD\\@bD?nB?|AG~OKzIWnI_@vHq@zJ{A~NwCjUuP}HtP|HdClAeCmAvCkUzA_Op@{J^wHVoIJ{IF_P?}AxC@vC??k@Pw@j@[dLOd@?Lt[Mu[e@?eLNAgEnDGpA@`BFfBF@lDAmD~SxAp@FIpCHqCq@G_TyAgBGaBGqAAoDFiAB}BN|BOhACC_E?k@KaACe@A]Mg[Cs@c@_EIsAC_BEcIoC?{CCAo@WwEA{@C{JoB{AIXHYnBzABzJ@z@VvE@n@@p@R~s@S_t@Aq@zCBnC?IoPAY?sC@eEDgDJuDP}CZ}Dd@uDl@wD\\eBn@cChAeDn@oAh@k@eEcEqDqDpDpDdEbEi@j@o@nAiAdDo@bC]dBm@vDe@tD[|DQ|CKtDEfDAdE?rC|CVVmItNlAuNmAWlI}CW@XHnPDbIB~AHrAb@~DBr@Lf[@\\Bd@iDJhDKJ`A?j@hDBhD?|BJrWjB@e@`@uPa@tPAd@KtDrAJJsDsAMrALZ{M[zMKrD`Jj@|@R^X^d@p@eBLm@V}ALy@GuGMmA{@oF[iCKoDFsCxBit@yBht@GrCJnDZhCz@nFLlAFtGMx@W|AWW{@k@k@Wo@QcJq@bJp@n@Pj@Vz@j@VVMl@q@dBVr@FXDZFz@?n@Av@KlD`CLxA_@Ru@NiEQi@o@_@g@I}Ca@|C`@f@Hn@^Ph@OhESt@yA^aCMi@`ToFfkBKvDJwDcAc@rG}fC?SCSEQEOGOGKIIUKQE{Ly@}@Gs@xWyBSKvDkE[iAGkDUY~Jy@~PObCNcCx@_QX_K?W{@[z@Z?VjDThAFjEZgAra@}MiAQdCPeC|MhAIjCeNoAUfDq@jHcAjJqAx@pAy@bAkJp@kHTgDdNnAHkCfAsa@JwDxBRr@yWcAGaMu@}CGa@?`@?|CF`Mt@bAF|@FzLx@PDTJHHFJFNDNDPBR?RsG|fCbAb@nFgkBzCVW|GeBxt@vGl@b@A^Of@o@Vg@TVh@r@t@tBNdADdCe@hVd@iVEeCOeAu@uBi@s@UWVeAPu@DkCEjCQt@WdAWf@g@n@_@Nc@@wGm@m@|Gs@|Vi@hSv@H|Jx@xC\\PJb@ZVfAJdA?lAStA[j@s@b@q@Bc@CoCQnCPb@Bp@Cr@c@Zk@RuA?mAKeAWgAc@[`@cPHsDhAPZJrAh@ZF`@?^Kj@k@bAaCJMJE`AGFCHMH_AXcEYbEI~@ILdAbEW`IVaIeAcEGBaAFKDKLcA`Ck@j@_@Ja@?[GsAi@[KiAQIrDa@bPQKyC]}Jy@w@I_@`Ng@vQa@CWCVB`@BdEZzDXOnFQxGAh@@VBVDRFNJNJNJDLDLBtGj@b@HZJVLPPPVLXNf@Jp@Bj@?zFFn@F^k@h@yAp@g@NcAJyAD{I_@k@mC]kA\\jAj@lCzI^xAEbAKf@OxAq@j@i@FPJPNLNFPBb@?d@M`@a@Nk@Fk@b@hHIdHoAjZB`B^fAn@rAFRU|FGh@mFz\\TNNRJXF^Bb@?l@Cp@wIjz@a@nDbADn@Mj@[pVqUl@kDaXnX{B}@vIkz@Bq@?m@dB_A|GzBtBwAl@iFm@hFuBvA}G{BeB~@Cc@G_@KYOSUOlF{\\Fi@T}FGSo@sA_@gACaBnAkZHeHc@iHAo@Iy@O{@e@{CA}@Hk@Zg@\\Yn@a@rDsBPKV|C\\V|CoAhBmArByBpD}FlKkQ^WZAZHHHVuDPgMXeJBcFJsBKrBCbFYdJQfMWtDNNVxAXrCR|D?xBUrF|B@|Q{~@}Qz~@}BAmArZIpCIEYG_@A_@D_@J[R[ZW^Uf@Of@Kp@En@Ap@HhUA|E]lD\\mD@}EIiU@q@Do@Jq@Ng@Tg@V_@Z[ZS^K^E^@XFHDZRTRRVPZP^Np@Hr@Fz@~@p_@GpCNhGg@LcSzA^m@zAiEv@qDo@mBG}K]kMi@oIOgBNfBh@nI\\jMF|Kn@lBw@pD{AhE_@l@[n@Mn@Eh@?j@Lt@Tr@hGtK^x@Rp@Lz@@z@CbAQbBMbAYfCXgCLcAPcBBcAA{@M{@Sq@_@y@iGuKUs@Mu@?k@Di@Lo@Zo@bS{Af@MREbNaANAAmDFoBVwAWyf@B?|LUbLS[a\\Z`\\h@xk@~@C_ABZ`]~@C_AB`AldABzFEhKg@zc@p@Hb@kUVg[eAikAVAtWs@tBGhAE^A_@@iADuBFuWr@W@dAhkAWf[c@jUsBbr@YbEXcErBcr@q@ImBro@WxDa@vEv@RRgCSfCw@S}@U|@TaArI}DbWr@^bByJcBxJs@_@kEpWg@vBaBnE`BoEf@wBjEqW|DcW`AsI`@wEVyDlBso@g@E[Eo@WuCkAeAq@]c@a@gAm@cCO_Bo@aOg@gHFiED{DRkEQyB@aD?SASG[IWy@{Ac@m@oIqAUfDTgDJaAZgBl@wCLmL_B{LaCkDc@o@S?@r@DfF?\\GZQJi@DAe@WiHCgQBfQVhH@d@DrD?hFO|LO`CeGbu@QdBYx@s@l@}@FoOuAhKi~@|@}HBq@Ei@G[O]UYQKMC{Ag@zAf@LBPJTXN\\FZDh@Cp@}@|HyEuAxEtAiKh~@mCzUdNfEnF|@RSH]n@qTo@pTI\\SRoF}@eNgEqDt[iQeF}Ai@_@D^E|Ah@hQdFeGli@mSgFWQQ]KUEWCW?a@Hy@dEk^V}BL]M\\W|B{B{@zBz@eEj^Ix@?`@BVsAd@yOaExO`ErAe@DVJTP\\VPlSfFdGmi@jQvEpHpAVBXGTQRYTc@Lc@Fw@|@_XHa@JWNWxAsARi@Fk@TcApAsRf@aHg@`HqArRUbAGj@Sh@yArAOVKVI`@}@~WGv@Mb@Ub@SXUPYFWCqHqAeQ|yA[dAa@h@g@X\\rBJTIpAbGhBeDf[dDg[cGiBHqAHTPHlGhBmGiBQIIUKU]sBk@FkMkDmAvKyAk@kAc@uA[w@Ev@DtAZjAb@xAj@eE`_@[tCQ|AdA\\~CbAFb@YnFFfAv@bBfAh@gAi@w@cBGgAXoFGc@rDcGx@kHJUvMfEgA|JfA}JwMgEKTy@jHsDbG_DcAeA]P}AZuCmBg@`@qEs@OsH_BZcEcOgAgBCAmB[wFuH_t@KmB@qBHgBNeBTcB`G}T~DaP|AyGn@_HsFRrFSv@gFw@fFiIcDJq@\\}BJq@bA\\fG~BvAoG~Iu`@`@kCRsA`@oDzB|@`XoXrL{L|@|@j@k@NUVg@Pc@Le@Jo@ViBX{B^sC_@rCYzBWhBKn@Md@Qb@Wf@OTk@j@_PhP~OiP}@}@sLzLVaB~AgJdBePeBdP_BfJW`Bm@jDqVpUk@Zo@LcAESrAa@jC_Jt`@bCa@XYvHyRhCyGPa@RONCb@BrDnATNNPFRBP?NqF`l@p@pDTUP_DzGmc@{Glc@Q~CUTq@qDpFal@?OCQGSOQUOsDoAc@COBc@c@{BqAcBiAbBhAzBpAb@b@SNQ`@iCxGZ\\HTAXq@nEp@oE@YIU[]wHxRYXcC`@wAnG|DxArAf@d@PdA`@TJEXe@xDG\\RHtBv@j@T~B|@_C}@k@UuBw@SIF]d@yDDYUKeAa@e@QsAg@}DyAgG_CcA]Kp@]|BKp@Ih@]zBuJoDtJnDz@T`ACi@pDQh@i@d@c@Na@Ba@EgFkBc@]Wc@O_ACu@BoAL}AM|ACnABt@N~@Vb@b@\\fFjB`@D`@Cb@Oh@e@Pi@h@qDaAB{@U\\{BHi@hIbDjCfAkCgAo@~G}AxG_E`PaG|TUbBOdBdBZbJdBrA}FfBeOgBdOsA|FzDt@{Du@cJeBeB[IfBApBJlBtH~s@ZvF@lB?FCzBKbCOxB_@dCi@zBcAjCgAzByAzB}AjB{A`AmBb@sCj@gAAu@Gm@MuAYaC}@_IoE{E}AWlDa@~Be@nAuAtBaXpV}ArA|AsA`XqVtAuBd@oA`@_CVmD~BmCBc@CqACUOyAKq@S{@wH}V{CcLo@wC]uBW{AI}AH|AVzA\\tBn@vCzCbLvH|VRz@Jp@NxAlBeAJCFBFBDBFHpElHNTNNTNVNVJp@Jp@@`@Ef@QZOf@[nCiB^Wd@a@b@e@f@m@\\m@Xu@d@mCPiBCYoFbBOyAm@Ns@Lo@@a@?a@Mc@a@_@c@gFiJoGkK]w@U}@Ig@oBkPq@eCsFgKrFfKp@dCnBjPHf@T|@\\v@nGjKfFhJ^b@b@`@`@L`@?n@Ar@Ml@OhC{@dA[kDgZjDfZeAZiCz@NxAnFcBBXQhBe@lCYt@]l@g@l@c@d@e@`@_@VoChBg@Z[Ng@Pa@Dq@Aq@KWKWOUOOOOUqEmHGIECGCGCKBmBdABTBpACb@_ClC_FgBqj@}YYeNcCTbCUXdNgCmAwBaAQsHrBKsBJPrHvB`AfClArBzbAp@dCq@eCq@A}AAqCC}ASrAgBTgAJ_BK_Fi@cXs@{]r@z]Q@uFTIAYI[UaEuDIWE[Eg@Is@M[WW{SqJzSpJVVLZHr@Df@DZHV`EtDZTXHH@tFUPAtBKuBJh@bXJ~EK~AUfAsAfBaAc@w@q@iI}I}DoDaCuBaPeLaKqFaCiAoBo@rGud@nAgJ_UiI_EuAyOcF}FaBeKwCvAaA{@WuD_AwA]WMWOQQQSWc@i@fBPRRR^RhA\\dAXeAYiA]_@SSSQSOUQ_@g@aBc@aBb@`Bf@`BP^NTkBdHsCiLw@sDv@rDpCuBvQgOwQfOqCtBrChLjBeHh@gBd@gA^u@v@y@hb@e[^S\\Ot@NxBv@rg@xRsg@yRyBw@u@O]N_@Rib@d[w@x@_@t@e@fAVb@PRPPVNVLvA\\tD~@z@VbPaL|@_@rEc@d@B|Cf@fDbAnDpA`JhDvM~EmDnVAH@IlDoV^iCJs@bBoLcBnLKr@fDpAgDqA_@hCwM_FaJiDoDqAgDcA}Cg@e@CsEb@}@^cP`LwA`AqDcApDbAdKvC|F`BxObF~DtA~ThIxIjDxNzFlVpJmVqJv@uFdAqHeApHw@tFyN{FyIkDT_BCA_UgIkEyAiOaFmDaAsKwCX_AZSl@e@Ok@EQDPNj@m@d@[RY~@rKvClD`AhO`FjExA~TfIB@U~AoAfJsGtd@qD_AmA_@k[aKWEOAWEg@Gy@Ou@Us@]y@i@o@g@i@e@h@d@n@f@x@h@r@\\t@Tx@Nf@FWzHV{HVDN@VDj[`KlA^pD~@}@tG|@uGnBn@`ChA`KpF`PdL`CtB|DnDhI|Iv@p@`Ab@m@~A}B`GmChHk@vAkAcAyA_@xA^jAbAbE|GfBdEh@Vr@\\hFdC`AlBaAmBiFeCs@]]tBa@nAwAbCmBtBrAFlAGdDk@zEy@{Ex@eDj@mAFsAGlBuBrCm@pCm@l@E`@T~@zA|A|CnAo@x@JnFbKoFcKy@KoAn@j@lA`Op]t@|@bAZjA@kAAcA[JdE~KlXkG}AsEiAoPuE~EiH_FhHnPtErEhAjG|AzEbLjBvCpBvBbAl@`Bj@lAVr@@^Gx@Yn@_@b@e@f@q@g@p@c@d@o@^y@X_@Fs@AIpB}CfNl@xKrElBdEz@L\\mClDuA`FeAzFZlAo\\|XkBz@jB{@n\\}XpGkFzBw@gBhE_GlHmIfJPd@\\z@]{@Qe@uHlIoP`O}MtJaOxI{FdCqBjA}HrCqPtF{I`BoGlAmCcUwChAvCiAlCbUcBXuGx@uHb@eHFeIMsCO]A?f@AR?f@?g@@S?g@wBMsLiAsCm@_HcBAh@E`ADaA@i@~GbBrCl@rLhAvBL\\@JeIKdIrCNdILdHGtHc@tGy@bBYHl@JfAKgAIm@nGmAzIaBpPuF|HsCpBkAMs@@yAhAsGd@aCt@yArUgM~MaH_N`HsUfMu@xAe@`CiArGAxALr@zFeC`OyI|MuJnPaOtHmIlIgJ~FmHfBiE`F{GzHqLVf@j@jAk@kAWg@jf@uu@dF{Ix@cBi@u@h@t@f@iAxCkGrKyU\\s@rAwCbEiJlDqHk@{A^}EbBiPcBhP_@|Ej@zArCiFvGuPdF_PhHqUlCmKlBoH~AkG_BjGmBnHmClKiHpUeF~OwGtPsChFmDpHcEhJsAvC]r@sKxUyCjGvBwAbDuDrKwPNVd@z@e@{@OWp@gA`GkJjJmQ|GuO`F_MPPn@l@o@m@QQxDqLzFcRzAiGTNv@l@w@m@UO~AkG~AgHpHe]vEmSnHq]oHp]wElSqHd]iAw@mD_Cz@iDZqA~DoPJ{@Dk@@k@Ai@Am@CS{HxQzHyQEWKa@S[cIxQMaBL`Bs@hBu@pBiGfUn@h@vCxBb@Zc@[wCyBo@i@i@`Bq@~Ao@jALn@Dj@Eh@Ov@wH`PmAjCcCjGeAdEg@nDf@oDdAeEbCkGlAkCvHaPNw@Di@Ek@Mo@sArB}Xl_@i@j@i@\\m@VDr@SzAe@`BiB|EiE~Iq@zA]c@uBbEsCkDyAtCUYuAjCtAkCTXxAuCrCjDtBcE\\b@eGrMcChFmFtKmArCZb@aAnB`AoBLR|AaDrFgL\\^aA~BW~AV_B`A_C]_@sFfL}A`DMS[c@lAsClFuKe@m@oDrHg@q@{CnGq@vALPMQp@wAzCoGf@p@nDsHd@l@bCiFdGsMp@{AhE_JhB}Ed@aBR{AEs@cA`@qIdBu^hHuNvBaFp@gw@hJgBNeA?_AEQjAPkA{Aa@q@_@i@i@gAiAeAyAw@eBq@qB_@qBUoBS_D[{DcELkEPjEQbEM|CaB}C`BZzDR~CTnB^pBp@pBv@dBdAxAfAhAh@h@p@^zA`@~@DdA?fBOfw@iJ`Fq@B~@?`@CVhC~@x@j@TvDFfDXIr@DbBDZBVLXTb@h@VXLRz@N\\DJPKQ]E{@OMS[n@oAlCQZP[nAmCZo@WYc@i@YUWM[CcBEs@EYHQD]Lu@p@wA`AeBz@c@n@pBhBjBfBkBgBqBiBcCxEm@zA_@xAIpACd@AVMxAGr@]~AqAzCGgALyAG{AAgB@fBFzAMxAFfAeG|NC`AJdA^fB{A`@o@D]E_@Al@hEm@iEK?e@@ByBH_@Z]f@GBgB\\yBNc@dFiNVeBMyCw@lAQlBaBvAw@tC_@zAS`@aA`ACh@q@|Am@|AB`CZ[P}@^w@x@_@^k@`@IOb@]xBCfBg@F[\\I^CxB}BHw@s@U}BT|Bv@r@|BId@AJ?^@\\DAoA[uBZtB@nAn@EzAa@PAjBKvCzEwC{EbK{OV_ADc@Dg@CwDRuBz@cCpD}I`@T`@P\\FXB`@GVEpBcDqBbDJAb@AR@NFRHLJh@r@vApBH^Bh@Cr@Mt@}ArDu@pBc@jBy@tHe@~Ad@_Bd@Cl@]^a@lFoMmFnM_@`@m@\\e@Bx@uHb@kBt@qB|AsDLu@Bs@Ci@I_@jAqCLgADaBDg@\\_ARWTQVGV@nAhANFLBLEJIxEyJyExJKHMDMCOGoAiAWAWFUPSV]~@Ef@E`BMfAkApCwAqB|HcQb@_A\\S\\LjA`B`@R^O_@Na@SkAaB]M]Rc@~@KQmBbElBcEJP}HbQi@s@MKSIOGSAc@@K@WDa@FYC]Ga@Qa@UqD|I{@bCStBBvDEf@Eb@W~@pCbF}JrO|JsO`Ax@r@NjBJ~@Qc@bBs@rCyCtGiAx@oBf@}@J_@?QAOKWY{AcCzAbCVXNJP@CzABzAHvAZ|@b@Zx@AyArFkBdEwFbGvFcGjBeExAsFrAi@bCeBvGcGvAmBrAoDhAyDfCwHvA_DHJ|@fADH|@hA}@iAEI}@gAIKP[bAmAx@qBvAoGn@kA}BnVi@jBPtAvB{GdBgGtMiXl@k@~@kBV}ApKmTqKlTW|A_AjBm@j@uMhXeBfGwBzGuCfGlBbC_CxEoY~b@y@tCx@uCnY_c@~ByEmBcCwDjGaDhDaD~CuGzHsDxCqDnAmDqIsDaFwByEm@qB@q@Ap@l@pBvBxErD`FlDpIpDoArDyCtG{H`D_D`DiDvDkGtCgGxAjBf@n@DFWf@EEi@u@EGHOHQHJd@n@TXUYe@o@IKIPINDFh@t@DDVg@EGg@o@yAkBQuAh@kBMOLN|BoVo@jAwAnGy@pBcAlAGIoAaBGIGIc@k@Y]U[TZX\\b@j@FHFHnA`BFHQZwA~CgCvHiAxDsAnDwAlBwGbGcCdBsAh@y@@c@[[}@IwAC{AB{A^?|@KnBg@hAy@xCuGr@sCb@cB_APkBKs@OaAy@qCcFcKzOkBJQ@_@gBKeABaAdG}NpA{C\\_BFs@{Bu@AiAVg@TYjAfARFSGkAgAUXWf@@hAzBt@LyA@WBe@HqA^yAl@{AbCyEb@o@dB{@vAaAt@q@\\MPEGgDUwDy@k@iC_AKXe@f@}AjAcCtAoKfEb@pCm@nCl@oCzBLjDLbC`@|@Zn@f@o@g@}@[cCa@kDMk@vCMnAIlDHlBHxBLxCWdBeFhNa@H_@j@y@^_@v@Q|@[ZCaCl@}AWe@Ru@E{BfA_FdAw@fBqHf@eC~Ae@_Bd@g@dCgBpHeAv@gA~EDzBSt@Vd@p@}ABi@`AaARa@^{Av@uC`BwAPmBv@mAIyBImBHmDLoAj@wC{BMc@qCcGFsFm@o@Bg@L_@^s@nAe@VaFz@K^}BzJ|B{JJ_@`F{@d@Wr@oA^_@f@Mn@CrFl@GfBwDjPwD~OvD_PvDkP`GpDeCnKs@|CqAy@@K@I?I?IAICGCICGCEEEECGAEAE?D?D@F@DBDDBDBFBHBF@H?H?HAHAJCFCHEFEDGBGBG?G?GCECGGEGCICIAI?K?K@K@KBGBEDGDEDCDAFAG@E@EBEDEFCDCFwA{@vAz@AJAJ?J?J@HBHBHDFFFDBFBF?F?FCFCDEDGBIBGpAx@wDbPy@g@c@Ec@EYSS]Qe@}@k@YpAPzANr@T`@|CvB}CwBUa@Os@Q{AUeAeCuB_@q@yBgEiBkGhBjGxBfE^p@dCtBTdAXqA|@j@Pd@R\\XRb@Db@Dx@f@cAbE[zA}@nEvD|NVx@V`@HRr@jAJPp@lALPFLJR]h@Ua@{GmL{e@k}@_@i@aCnDkChEjCiE`CoDYq@{@oDhBgAdC_@I`@AHUpAvCa@xCa@hAEd@XTLFDbCxAjBfAkBgAcCyAGEUMe@YiADyC`@wC`@TqA@IHa@eC^iBfAi@aBeBeDuGsSWmDYaCSmBRlBX`CVlDtGrSdBdDh@`Bz@nDXp@^h@ze@j}@zGlLT`@R\\jbAtkBT^bA}AcA|AZh@TXPTZTd@Td@Nb@Fx@JhQa@`BQjBc@rA_@~@a@hB_A`Ai@~AaAx@s@l@q@`AkA`B}Bl@mAmH{POi@C]D[Pg@xMuXyMtXQf@EZB\\Nh@lHzPtAjBfD|CgD}CuAkBm@lAaB|BaAjAm@p@y@r@_B`AaAh@iB~@_A`@sA^kBb@aBPiQ`@^l@tAbCdAxBLhAnGtL\\|@t@zCZjB^rAr@`BlVpc@]l@\\m@mVqc@s@aB_@sA[kBu@{C]}@oGuL[h@eDfFk@z@]h@_Bd@u@dAra@hu@sa@iu@}BmEe@kAe@cCYsA[}@q@s@i@k@OOB[@e@C_@AUI[Ic@S_@m_AkgBS_@qY}h@yCcF}CuDi@a@DeCIcD]}D\\|DHbDEdCy@m@_JwQ~IvQx@l@IlAUfAjBnDjXdi@kXei@kBoDTgAHmAh@`@|CtDxCbFpY|h@R^l_AjgBR^Hb@HZ@TB^Ad@CZK`@IZK\\QX]h@iGnHoCjDq[je@SGiC{@MAM?MDKDaGzEQJQDcCb@bCc@PEPK`G{EJELEL?L@hCz@RF{P`Wu@xAc@fAe@|AYvAQrAIvAClB@|AF~AfCdR`BzLrBo@nAa@Vl@XP`@@^QbLsLdO{SdGyH~D}C_E|CMi@S}@Ki@IMGKGMnCkF~@gBg@Wl@iDhByJdAiDeAhDiBxJm@hD}EoCoC|DnC}D|EnCf@Vv@d@nAp@oAq@w@e@_AfBT`@FHLPXf@`CuBnEdBoEeBaCtBNr@Os@{DlDzDmDYg@MQGIUa@oCjFeE}Hg@r@mBrCQR\\n@]o@wE~G\\n@]o@gCpDEFgA~AjA|BqDdFsAnBEFEFDGDG}@eBEGQ]SVRWYk@Xj@P\\DF|@dB^t@DFEG_@u@rAoBeBcDEGiAuBsB{DY`@Xa@eAoBqBsDuAgCtAfCpBrDdAnBrBzDW`@Va@hAtBDFdBbDpDeF^r@DHEI_@s@kA}BfA_BDGfCqDvE_HPSlBsCf@s@gFqJs@bAr@cAaAgB~A|@_B}@wAmCqBsDpBrDvAlC`AfB\\i@xI|EyI}EXa@J[^iBfB_JlBkDpHnEbIsLcIrLqHoEhLiPyAoCgAgBfAfBxAnCjEdIkEeIdEaG\\QTIV?TNNPrTz`@eGpIuA|AmMeHqBkAgEcCfEbCpBjAlMdHtA}A_IsNyEwIxEvIcFjHbFkH~HrN|ElFrFbGeHRyEFqEl@sFrA_HpDI]WoAVnAH\\~GqDrFsApEm@xEGdHSpFQvAe@fB_Bb@|@r@tAdAhBr@zAz@sAxA_ChCaEiC`EyA~B{@rAn@tAb@tBt@jDF\\oAFnAGpJrBvFnArFLxCHh@Af@?nXbA^?r@Id@C?OL{FMzF?NrBFsBGC|AEnB?N@NDLDFDFHBHBICICEGEGEMAO?ODoBB}Ae@Bs@H_@?oXcAg@?Hm@b@}EHa@FOlDqEV[m@oCUITHl@nCTVVL`@Dbd@dBXBz@D~@F_AGMrFLsFr@kEh@eDFi@jAf@CjAAh@@i@BkAr@ZEpCCpA]CATWhTaGnAT|A`@dHuArHaDdNb@^^xAvCtBnElDjEfEb@aATVZ`@bFxG`@r@a@s@cFyG[a@UWgCoCmDeCfTeHt@i@Tw@Uv@u@h@gTdHlDdCfCnCc@`AkEgEoEmDwCuB_@yAc@_@Gv@a@xBm@xBiAfEELuAgABKdA_EFSbBhAGPiAbEM`@{AqA{GdV_@jAgApDfAqD^kAzGeVzApALa@hAcEFQcBiAGReA~DCJtAfADMhAgEl@yB`@yBFw@`DeNtAsHa@eHU}AwEuBvEtB`GoAViT@UK?DeC@WV@bAF?XE~B?ViBK?UDaCE`C?ThBJ?WD_C?YcAGWAAVEdCJ?\\BBqADqCs@[kAg@Gh@i@dDs@jE{@EYCfI_i@D{@?c@Qo@a]kW`]jWPn@?b@RDtATrATsAUuCbQeBhKdBiKtCcQn@sDkAe@jAd@vBr@xB|@o@`B}CI|CHn@aByB}@\\sBd@wCjCiQ|AeM`@kEz@eK{@dKa@jE}AdMkChQe@vCkCmAc@Wo^sXq@fCwB{AoArFnAsFaCaBiAjEQqAIQGKKIKKuG_FqB{AcVkQYM_DVWN{@fAiA_CcA{BbAzBhA~Bz@gAVO~CWXLbVjQpBzAtG~EJJJHFJHPPpAFf@BR?\\CZEZgR~x@\\N]OOp@O^U^[T_@Ne@Bw@KyM}Dw@c@i@e@a@m@cAmBbAlB`@l@h@d@v@b@xM|Dv@Jd@C^OZUT_@N_@Nq@fR_y@D[B[?]CSGg@hAkE`C`BvBzAq@hCIGGAIDGLUbAgH`[fHa[{A_AiH~Z|A`AwHd\\C\\@d@Ae@B]vHe\\XPxDbCpCdBqCeByDcCYQ}AaAoIv^|Ah@}Ai@iA_@hA^{CvLzCwLnIw^hH_[oBmAnBlAzA~@TcAFMHEF@HFp@iCp@gCQMEKAMBQFWGVCP@LDJPLn^rXb@VuAfBo^kXn^jXiAnAUXTYhAoAtAgBPYfDuUp@uFvAgMp@qH^}Dj@kGtAoN_FoB}R_H{UjcAzUkcA}AaAi@xBh@yB|A`AvSs}@l@qCgBy@`CsPPwAJ}CFuAGtAK|CQvAaCrPi@rCoE|RsG~XaBaA]UoLkHnLjH\\T`B`AmDjOaBcAlDiOtGaYuG`YmDhOg@vB}GrYgIb^g@vBGVKXMPQLE@xBrAaA~D`A_ErK_e@sK~d@xA`AyAaAyBsAMFOBU?QGOGUMo@g@mT}OoAeA_B}AgDcEEIEM?EAG?G?F@F?DDLDHfDbE~A|AnAdAlT|On@f@TLNFPFT?NC{@hDoJuGqOgL{HkHxAeDyAdDmAxClAyCzHjHpOfLk@zBYfAeLbc@Sn@Yh@Yd@]`@[\\i@b@m@Xo@Pg@Ba@@e@G]KqX}c@pX|c@k@Se@W]Wa@_@]c@kTy]?AOYMYG[Gc@E]?_@?c@Fk@NcAAh@A`@B\\BXBNDTHTl@jAm@kAIUEUCOCYC]@a@@i@BaA?m@Ca@Ea@I_@Sg@W_@UUOMKIJHNLTTV^Rf@H^D`@B`@?l@C`AObAGj@?b@?^D\\Fb@FZLXNX?@jTx]\\b@`@^\\Vd@Vj@R\\Jd@F`@Af@Cn@Ql@Yh@c@Z]\\a@Xe@Xi@Ro@dLcc@XgAj@{BnJtGz@iDLGDAPMLQJYFWf@wBfIc^|GsY]SqG_EpG~D\\R~AdAyGpYcBcAWQuEmDkHoFcByAqBwBkEsF]e@aHtN\\b@hCfDhCxBbJxGFFDJ?LAJYnAEHGDG@IAuJiGaD}BkCiD]c@{FrLCHCJ?DaCyCuAiBpHsOq@{@MQLPfA_CrDuF\\`@rJeSHSd@{ADWJm@f@iDJm@No@L]|MsXv@cBTa@ZYXIZEhD@`JIHAHCJGFEDGDKFOBO`BuLdB_MhBl@f@Ng@OiBm@eB~LaBtLCNGNEJEFGDKFIBI@aJHm@lEl@mE^yCxB{OnAgJoAfJxHvCn@To@UyHwCMESIRHLDyBzO_@xCiDA[DYH[XU`@w@bBpDvEqDwEm@w@uCwDtCvDl@v@}MrXM\\On@Kl@g@hDoAaB}@mA_@e@oAaB`@d@]c@q@}@p@|@\\b@a@e@nA`B^d@|@lAnA`BKl@EVe@zAIRsJdSgFnKfFoK]a@sDtFgA~Bp@z@qHrOuAjBg@bAf@cAtAkBtAhB`CxC?EBKBIzFsL\\b@jChD`D|BtJhGH@FAFEDIXoA@K?MEKGGcJyGiCyBiCgD]c@yAvCgB}BfB|BxAwCcBwBbBvB`HuN~`@{y@_a@zy@cBwBbBvB\\d@jErFpBvBbBxAjHnFtElDVPbBbAxGqY_BeAf@wB`BbAlDkOrG_YnE}Rh@sCfBx@m@pCwSr}@|R~G~EnBuAnNk@jG_@|DaMwE`MvEq@pHwAfMq@tFgDtUQXjClAlCnAq@`Bp@aBTs@Ur@mCoA]rBwBs@o@rDuAUSEEz@gI~h@cd@eBa@EWMUWWZmDpEGNI`@c@|EIl@i@@yCIsFMwFoAqJsBG]u@kDc@uBo@uAs@{AeAiBs@uAc@}@gB~AwAd@qFPzBbC{BcCsFcG}EmFdGqIsT{`@OQUOW?UH]PeE`GiLhPy@q@]i@]i@_AcB~@bB\\h@\\h@x@p@mBjDgB~I_@hBKZY`@]h@fFpJdE|HFLFJHLaAz@S[]\\\\]TYX_@JMKLY^UXRZ`A{@Jh@R|@Lh@eGxHeOzScLrL_@Pa@AYQWm@oA`@sBn@aB{LWJSJR`BY\\YZX[X]SaBRKVKgCeRG_BA}AUA[?S?UHSNoBvBwRbT{E|B{FfGIPGT]jBIj@Gd@Cn@M`O{e@cB@i@Ah@ze@bBLaOBo@Fe@Hk@d@jB?LAFKxL?D?D@BBBB@B@H?At@zNf@@}@tCHFwGg@C@eAhBDX@YAiBEAdAf@BGvGuCIA|@{Ng@qAECrBef@cBaXiA{PeAyRuAy`@kCiAKwA_@u@[t@ZvA^hAJx`@jCDmBd_@hCpMn@fOd@gOe@qMo@e_@iC^gQH{@PMLKjb@@AoD@nDpLTG}QI{@[iDg@iEMyBDa@lA_BtC}BbEkBzToIj@c@hAgA~@uCj@_DMi@hQ{KjG}Eb^}\\|FoGdRkVhNmStF{IlDoFpAyA|@_AZWh@UrA]RILGJKJUL]Ng@Nm@Ng@LJfBlAn@d@Qr@ELGVABCLHF`@Xh@Zj@^ZR|@l@LHzA`ALH`An@JFbAp@PL~AbAHFvBtAHFGX{ASSFmAt@oBxB]\\}Av@cCrBeH|JFLtCtFh@`ADId@s@d@z@e@{@e@r@EHi@aAuCuFGMdH}JbCsB|Aw@\\]nByBlAu@RGzARFYIGwBuAIG_BcAQMcAq@KGaAo@MI{AaAMI}@m@[Sk@_@i@[a@YIGBM@CFWDMPs@o@e@gBmAMKzBiHt@wAlDcFfRiZ~DaGS~A{@pHTh@\\g@]f@Ui@z@qHR_BvJ}NjNaVfGcMbL{WhGgPfNyd@pCaLhDwQtGoZ~@gD~@{EdHi\\pEeSH]Ji@@a@Gq@?SHc@WOk]gPMGM|@L}@cAe@YdBXeByFkCUKMELDTJxFjCbAd@Lm@T_@XUl@Se@xBS~@_@jBaFfUa@UWM]Qt@yDmD}Am@lDgAg@nBeNn@cGvGzBwG{Bo@bGoBdNfAf@l@mDlD|AbBmKn@wDo@vDcBlKu@xD\\PVL`@T`FgUM[Kg@Ek@Dj@Jf@LZ^kBR_Ad@yBh@sBpBwKhBaL`B_MVHt@Vu@WWIL}@tAwLnAsNbAaOr@{Nb@wNXiOP@dA?eA?QADgHAuGPAhAAiA@Q@GwTIiIUuGg@{Ko@eMc@iIwGidAvGhdAg\\x@NjOrJYsJXZ`][a]OkO_@ab@a@oc@CuCB_Bx@cR{@Mz@LdEj@~ArX_BsXeEk@zCam@JqA`B}ELsDFeHIkLK{NIaRKwJq@aOq@sOe@wHSkDY_IAeBDgBFsAPoB~BkSz@eFnAoFxA{Fn@yBt@{BlA{C~@kBp@mAr@iAx@mAv@aAnAwAvBqBtG}DZUrFsFdDaDdHwHLIHCHAF?G?I@IBMHeHvHeD`Du@sBSRYZ{NvLiBjBi@l@uKgR}H`AeEhDdEiD|HaAtKfRe@j@{@hAsAfBeBtCgAtBc@|@oIs@cFw@e@Gw@Bm@Hq@Ri@b@k@z@_@z@W~@Ox@Q~AG|@?jBFx@Jn@Vp@Tb@\\\\d@TdATjLzAy@vFs@bGaCm@i@IwFi@kT`DqC`@QDIDELkAvIGTMP_@^uCvCc@f@wBxBWVHPh@xAi@yAIQVWvByBb@g@tCwC^_@LQFUjAwIDMHEPEpCa@jTaDvFh@h@H`Cl@mAvKWpCI`C?~BBlAh@fMdCvh@R~JJ|Q@hPCbHEdBMxCi@|GQpCiAOqC]o@b@iCYQzEP{EhCXn@c@pC\\hANGx@kCxh@_AKsAFrAG~@JWpFw@bOIpBAfBB~BLnM_@@mABEqCj@CBnB@v@gCH[{j@BkAJcAXm@pBKA_CoEHCyAuCH?mEHaJ\\ySJ@\\BRBAV`AlFLBB[b@Fc@GCZMCaAmF@WB_@FSZaA`@kAPg@La@VBhBJR@SAiBKWCM`@Qf@a@jA[`AGRC^SC]CKAMCq@G]Cj@uCFOHILAJBDJDN?TEjCDkC?UEOEKKCM@IHGNk@tCkG|ZjG}ZWC{@IqCYeRgBUAv@ePw@dPCp@kGw@Do@QCE~@uAMuI]uBBUvAo@v@y@ZiGACsCYmBWaBAyHiA?SpBg@Zo@JyAo@eBIdBHxAn@n@Kf@[RqBhA?@xHV`BXlBBrChG@x@[n@w@TwAZoDXoBd@mAp@kAb@iARmB@_B\\D]Ec@[Uk@Ig@Is@AoCEsAGs@Mo@O_@OYOSWQWKSESC_@?aAH_CTiDh@i@Rm@`@aA~@s@p@u@AYfF_@@[r@e@pKQxIF~@HV}@F]CKAOe@Es@Qu@uAiC_BcB~AbBtAhCPt@Dr@Nd@J@\\B|@G|C`E~AvCThAEv@Dw@UiA_BwC}CaEIWG_APyId@qKZs@^AXgFt@@r@q@`A_Al@a@h@ShDi@~BU`AI^?DyDD}@HiAFkA@u@@qABuA@o@Fq@Fa@^gCJ}@HgAT_GrBAOmFTCfJg@k@mMrEyFaBoD`BnDsExFj@lMgJf@UBY@i@oAKs@_@}B]mBw@uCgBgFsEsK{A_FeA}D_AgFk@sDaAgIc@yEYwESoEQuFKwEGeJBsE@k@Aj@CrEFdJJvEPtFRnEXvEb@xE`AfIj@rD~@fFdA|DzA~ErErKfBfFv@tC\\lB^|BJr@h@nAXANlFsB@U~FIfAK|@_@fCG`@Gp@An@CtAApAAt@GjAIhAE|@ExDRBRDVJVPNRNXN^Ln@Fr@DrA@nCHr@Hf@Tj@b@ZA~ASlBc@hAq@jAe@lAYnB[nDtBCtI\\tALD_APBEn@jGv@e@`KUAq@pNh@DxRbBb@@c@AyRcBi@Ep@qNT@d@aKBq@T@dRfBpCXz@HVB\\Bp@FLB]xSI`J?lEtCIBxAnEI@~BqBJYl@KbACjAZzj@fCIAw@CoBk@BDpClAC^AMoMC_C@gBHqBv@cOVqFjCyh@Fy@PqCh@}GLyCDeBBcHAiPK}QS_KeCwh@_Fn@Y@YGuBqAWWw@{AUUmDkAlDjATTv@zAVVtBpAXFXA~Eo@i@gMCmA?_CHaCVqClAwKr@cGx@wFkL{AeAUe@U]]Uc@Wq@Ko@Gy@?kBF}@P_BNy@V_A^{@j@{@h@c@p@Sl@Iv@Cd@FbFv@nIr@}ApEw@vC{ChMzCiMv@wC|AqEb@}@fAuBdBuCrAgBz@iAd@k@h@m@hBkBzNwLX[RSLONQP[L[Lg@Jq@Di@F}B`@C\\I^UrDgC`@UTCVA`@B\\FZHPJLDShBEZL@LDHDFFFFFJhAvDiAwDGKGGGGIEMEMAD[RiBPNVXZf@xAjDL\\P|@Q}@M]yAkD[g@WYQOMEQK[I]Ga@CNyGfBE?n@Bd@Hp@Nl@E^SpBRqBD_@Om@Iq@Ce@?o@gBDOxGW@UBa@TsDfC_@T]Ha@BG|BEh@Kp@Mf@MZQZOPMNsAuDH]D_@Aa@Ea@Ke@I[wB_GOy@Iu@Gq@Eq@CcA]k@UUi@i@QuAPtAh@h@TT\\j@BbADp@Fp@Ht@Nx@vB~FHZJd@D`@@`@E^I\\rAtDt@rBsFrF[TuG|DwBpBoAvAw@`Ay@lAs@hAq@lA_AjBmAzCu@zBo@xByAzFoAnF{@dF_CjSQnBGrAEfB@dBX~HRjDd@vHp@rOp@`OJvJH`RJzNHjLGdHMrDv@`BV`B`@|In@bDjFr{@TCx@My@LUBkFs{@o@cDa@}IWaBd@GzAO{ANe@Fw@aBaB|EKpA{C`m@y@bRC~ABtC`@nc@R?dCEm@uK`G{Io@qKn@pKaGzIl@tKeCDS?^`b@f\\y@b@hIPC~@O_ANQBn@dMf@zKTtGHhIFvT@tGEfHYhOc@vNs@zNcA`OoArNuAvLM|@aB~LiB`LqBvKi@rBm@RYTU^Ml@LFj]fPVNIb@?RFp@A`@Kh@I\\qEdSeHh\\_AzEz@`@p@gC~DyKpDqJhB{Dj@c@~@RqBbNwArKSvAwCmAVmA`CcJhAyDpCuGpCcRp@T@WAVq@UDg@@iA@u@FeEV}D\\cEb@cDVeBf@_Cg@~BWdBc@bD]bEW|DGdE}@a@[OKEeCoASMFYFIz@]TSL]jFcUxAsASnALpCGjBuA`TAnAZDNeAxFcVxBgMLDMEyBfMyFbVOdA_AxXmAdZSdEKjDCtIF`IEzGmAb`@uA|[w@vN{H`^Gr@@t@Df@Jf@J\\NXTXpApAt@`@fBh@nAd@oAe@gBi@u@a@qAqAUYOYK]Kg@Eg@Au@Fs@zHa^v@wNtA}[lAc`@D{GGaIBuIJkDReElAeZ~@yX[E_@CAVCrBBsB@W^B@oAtAaTFkBMqCRoAyArAkFbUM\\UR{@\\GHGXRLdCnAJDZN|@`@At@AhAEf@qCbRqCtGiAxDaCbJWlA[vBWbDi@nJ_BrY_@tFy@`FmAnD_HhPaCnG`CoG~GiPlAoDx@aF^uF~AsYh@oJVcDZwBvClAQrCUhh@SvMYlVMlEy@vOa@bGaAbIaAbHiA~LIzADrARx@Xl@\\d@oEeBqEcBOEOCS@MDMHIFIJINeCxEdCyEHOHKHGLILERANBNDpEbBnEdBt@t@\\Zf@\\~IhD_JiDg@]][u@u@]e@Ym@Sy@EsA_G_AgBe@oDViCnBqCpAkD`AmC|EaHjAoCfF]hJwAhAiItJ{Dn@sFpIuCfEaDdC]u@\\t@`DeCtCgErFqIzDo@hIuJvAiA\\iJnCgF`HkAlC}EjDaApCqAhCoBnDWfBd@~F~@H{AhA_M`AcH`AcI`@cGx@wOLmEXmVRwMTih@PsCRwAvAsKpBcN_ASk@b@iBzDqDpJ_ExKq@fC{@a@_AfDuGnZiDvQqC`LgNxd@iGfPcLzWgGbMkN`VwJ|N_E`GgRhZmDbFu@vA{BhHOf@Ol@Of@M\\KTKJMFSHsA\\i@T[V}@~@qAxAmDnFuFzIiNlSeRjV}FnGc^|\\kG|EiQzKLh@k@~C_AtCiAfAk@b@{TnIcEjBuC|BmA~AE`@LxBf@hEZhDHz@aDfBiCNs@Ik@GWLGl@?tF?uFFm@VMj@Fr@HhCO`DgBvA_@nHmBvCyAwCxAoHlBwA^F|QpKTh@?vDQ|AaAdAiAeAhA}A`AwDPi@?WdXVeXqKUqLUkb@AMJ_@eAgA?e@c@i@eB]oAkAu@cBLsHbBrHcBbBMjAt@\\nAh@dBd@b@fA?^dAQLIz@_@fQs`@kCkAOqA_@{@a@q@a@aA_AMPLQ{@mAu@uA_@}@Uq@NYfAs@rE_@vFPwFQsE^gAr@OXw@vAx@pBjBvCbAhA~A`A{@bDoBxAeA`@k@NuGB{@B_FSgCH{Cf@{Cv@kCH_C]~B\\jCIzCw@zCg@fCI~ERz@CtGCj@OdAa@nByAz@cD_BaAcAiAm@|@MPLQl@}@j@{@k@z@kBwCy@qBv@wATp@^|@t@tAz@lA`A~@p@`@z@`@pA^jANr`@jCElBxRtAEdB{AJqAFs@PYNiBxA_DtB}@ROPAPBL~LjGL\\DhCEiCM]_MkGCM@QNQ|@S~CuBhByAXOr@QpAGzAKDeBzPdA`XhAdf@bBfe@xAf@kAN]bA@XAY@cAAcAC@y@Hs@Jc@L_@PWxFmGJIJGJCH?LALBNBJDBDXZuEnKa@d@[LZM`@e@tEoKPo@No@Fs@?w@E_AYeCXdCD~@?v@Gr@On@Qn@Y[CEKEOCMCM@I?KBKFKHyFlGQVM^Kb@Ir@Ax@iRq@hRp@bABO\\g@jAq@xAYp@Kb@aA`EOn@GX}CjBs@`@oF_@_Tp@qAXkAHqCHkATaYn@gDXuDh@cB^_D|@qEnBeDdBwDnCq[zTp[{TvDoCdDeBpEoB~C}@OgAu@TaC\\qDDkDSaD[qAC}GNsABIcFHbFrAC|GOpAB`DZjDRpDE`C]t@UNfAbB_@tDi@fDY`Yo@jAUpCIjAIpAY~Sq@nF^@t@BvBDjE?`@}BBcM\\eEJqA\\{AdAzAeApA]dEKbM]|BCjFo@D?B@BBBD@DFz@BTPjC@RPlCBXPdC@VFp@Fb@H^FPR^f@j@VRt@Z~FBlQ{BJhAJvA`@fFa@gFKwAKiAhW{CdDi@vAc@^Qt@_@bYqP\\GRAj@Ll@i@dB}AHEJ?TBtOfTuOgTLAJAPKVOpOmMzEsD^d@_@e@{ErDqOlMWNQJK@M@UCK?IDeB|Am@h@uz@|v@tz@}v@TbDPvBQwBl@K|@Ht@b@nLhOxI~LyI_MlBqDzInL{IoLfA}ArAuA|AcBrAeBh@}@d@u@jEwIkEvIe@t@i@|@sAdB}AbBsAtAgA|AmBpD]`CyLePa@Ua@Ec@Jb@K`@D`@TxLdPzElGnBdCfAzAnKvNPLXHXDYEYIQMoKwNgA{Aj@uArAdB`@T`@B`@?h@@r@Nl@f@bB|B\\p@X`APhAFfA?z@IlAHmA?{@GgAQiAYaA]q@cB}Bm@g@s@Oi@Aa@?a@Ca@UsAeB\\u@pAkC~JbNHLnAxC~FrIuBbEpAzBdJtLvDpBwDqBeJuLqA{BaFyGyOb\\xOc\\`FxGtBcEbMdPDJFf@f@lGg@mGGg@EKcMePlAeCxGeNtG{MxDbFyDcFxEmJyElJuGzMyGdN|NbSP_@l@qAlAcCdBqDh@mAi@lAwHiKvHhKeBpDwHiKvHhKmAbCm@pAQ^qAhJzCqE~Ac@McAm@gBm@y@l@x@l@fBLbAvBk@xCy@bEgAa@kDaAwCaAyCaBwBzCkG`@h@a@i@{CjGyAxCaOsR`OrRxAyC`BvB`AxCdB_BeB~A`AvC`@jDdF}AK}@J|@eF|AcEfApBxQqByQyCx@wBj@_Bb@{CpEDHEIpAiJ}NcSmAdC_GsIoAyCIM_KcNqAjC]t@k@tAoBeC{EmG\\aCoLiOu@c@}@Im@JK@w@f@_c@d`@fAjDPl@Qm@jCwBrBrG`I{GfIhTgIiTaIzGsBsGkCvBgAkDqRbQq@h@SNMLgBhAwAn@mAb@iAZi@JwANuCPgCCoMCoBKnBJnMBfCBtCQvAOh@KhA[lAc@vAo@fBiALMKqACa@Go@[iE[_EOoBNnBZ~DZhEFn@{@x@g@ZcI|AwKx@kBLcB@yDB{BCmOy@q@KyB{@y@Ug@IwFa@_NaAc@IsFeEk@q@_@q@kD{IjDzI^p@j@p@rFdE@tB?lCPURKVA|El@AXIzAH{A@Y|KtAfFXfPz@|@F`CHdM^PJLTHXIYMUQKeM_@aCI}@G?b@?c@gP{@gFY}KuA}Em@W@SJQT?mCAuBb@H~M`AFsBHkB`@_LHoBRoFj@}PzASnB_AoB~@{ARgDZ{YnE{QrCyHn@oCh@I`@e@hC~H}@`Eg@t@Bt@Lv@Zr@f@h@h@b@n@d@bAnKnWzFb@dHh@hL|@jHfAnMv@AzA@{AxAHyAIoMw@kHgAiL}@IlBHmBeHi@{Fc@oKoWe@cAc@o@i@i@s@g@w@[u@Mu@CaEf@_I|@aEb@y@Rs@ZmC~Ad@jBf@]VQ?RBRnCdKPn@T|@U}@Qo@oCeKCS?SDSFOd@QnRqCoRpCe@PGNERWPg@\\mAz@e@Zd@[lA{@zDvN~@KRr@Ss@_AJ{DwNe@kBaVlNuKnGa@l@Sz@Ef@}CiAcAOW@WHVIVAbANsA}ErA|E|ChAjA`@kAa@Dg@R{@`@m@tKoGd@rBcAl@gI~EOTKZCNBOJ[NUfI_FbAm@dBeAfOwIgOvIeBdAe@sB`VmNlC_Br@[x@S`Ec@d@iCHa@nCi@xHo@zQsC[pDm@|F_@nD^oDl@}FZqDzYoEfD[k@|PSnFInBa@~KIjBGrBvF`@f@Hx@TxBz@p@JlOx@zBBxDCbBAjBMvKy@bI}Af@[z@y@B`@JpAJdABZsJxIrJyIz@y@ZY[X{@x@C[KeAROp@i@`@fAa@gApRcQ~b@e`@v@g@JAUcDk@MS@]FcYpPu@^_@PwAb@eDh@iWzCmQzB_GCu@[WSg@k@S_@GQI_@Gc@Gq@AWQeCCYQmCASQkCCUG{@AECECCCAE?kFn@?a@EkECwBAu@r@a@|CkBFYNo@`AaEJc@Xq@p@yAvJVwJWge@yABsBpAD@u@L@Z?hLZZ@J?K?[AiL[[?MAI?CACACCAC?E?EJyL`@@a@A@G?Me@kBsA_@rA^zGiH{GhH\\kBFUHQzFgGzE}BvRcTnBwBROTIR?Z?T@BmBHwAPsAXwAd@}Ab@gAS[g@JwBd@vBe@f@KGu@IiAHhAFt@RZt@yAzPaWjBl@kBm@p[ke@nCkDhGoH\\i@PYJ]H[Ja@NNh@j@p@r@Z|@XrAd@bCd@jA|BlEt@eA~Ae@\\i@j@{@dDgFZi@MiAeAyBuAcC_@m@y@Kc@Ge@Oe@U[UQUUY[i@U_@kbAukBS]\\i@KSGMMQq@mAKQs@kAISWa@Wy@wD}N|@oEXRGZEZ?dA?eAD[F[YSZ{AbAcEvDcPr@}CdCoKaGqDFgBbGGnKgEbCuA|AkAd@g@JYBW?a@C_AtNwBzFoNhBcFbAoGrBcOrAuLsAtLsBbOcAnGiBbF{FnNt^iHpIeBbAa@l@Wh@]h@k@|Xm_@rAsBn@kAp@_Bh@aBhGgUt@qB`BSaBRr@iBbIyQ_AiAlBiF^kAd@oBfCqNzGef@b@kFwCm@mBa@yCm@sGqAiDs@iE}@QE}FkAk@w@j@v@|FjAPDhE|@hDr@rGpAxCl@lB`@vCl@c@jF{Gdf@gCpNe@nB_@jAmBhFqG`QpGaQ~@hARZJ`@DVBR@l@@h@vD~BdAp@eAq@wD_CAj@Ej@Kz@_EnP[pA{@hDlD~BhAv@_BfH_BjG{AhG{FbRyDpLaF~L}GtOkJlQaGjJuCkCtCjCq@fAsKvPcDtDwBvAg@hAy@bBeFzIkf@tu@{HpLaFzG{Bv@qGjF[mAdA{FtAaFlCmDM]eE{@sEmByC`CyCzAcQxJwBkEwAhB}DxDqFlCaDNuHxC}@Pw@L}@Dw@Co@Mq@[c@_@a@c@a@{@[cAMwAAqA?oAHiAL_BX@vA@\\EtEsBb@GZBXLZh@[i@YM[Cc@FuErB]DwAAYAdCm[JuAFkA@eA?eACkAIy@Kw@Mw@Qw@Qi@dW{^eWz^Sk@Sc@Ug@Wi@cDaF[][U[KSGuAfu@tAgu@]I]GyMs@YEElAmAjp@HtEOhINiIIuElAkp@DmAc@UYYU[ISMYKe@Eg@Ek@?e@Ds@Jk@Ja@Z{@[z@K`@Kj@Er@}JhBSBRCQsEbGoD`A[r@F`@Zl@rA`@o@n@wA^}Aj@oDbF}]n@qEo@pEcF|]m@WeAe@mCwB^cN}OlH|OmHWoCVnC_@bNaLxF`LyFlCvBdAd@l@Vk@nD_@|Ao@vAa@n@m@sAa@[s@GaAZcGnDyC`Ai@PT~DnDa@oD`@oCZqh@rGq@Jp@Kph@sGnC[U_Ea@iCIk@tBkA|@wFq@gFaDpA{@\\zAzL{A{Lz@]`DqAi@gEw@kGs@eGfQyCh@lFi@mFvPwCbEu@{Ebj@v@NnA[x@WzVs^H}@dAuMeAtMI|@{Vr^y@VoAZw@OSlAe@bCmAhDk@fAi@Xo@LI@iQnBhQoBHAn@Mh@Yj@gAlAiDd@cCRmAeC_Aq@[cAa@bA`@p@ZdC~@zEcj@`LcCfAWgAVaLbCcEt@wPvCgQxCyEnAs@P{@NyBPc@mDeDsTuGgb@gNzAfN{AtGfb@dDrTb@lDWFeALu_@tE{Ed@gJbAxCdq@yCeq@yDd@mA{YlAzYxDe@fJcA_AsUlIy@f@pAg@qAmIx@}Ai_@]yH}QvB|QwBo@wPzXyCz@W{@V{XxCcCik@lFRzERtCLhGf@kBwKyC_KkCeJjCdJxC~JjBvK^hFTjCLfA`D`T~AbKn@zDo@{D_BcKsAXrAYaDaTMgAUkC_@iFbE`@Z_K[~JlF|@dGbBdLxDzIhExJnGo@xCYb@[RqCp@aS~D`S_EpCq@ZSXc@n@yCrC`CfFrE}ChJUd@Q\\a@^c@NkFhAaSdEQBQOEg@e@sGk@wHsRfErRgEj@vHd@rGDf@PNPC`SeEjFiAb@O`@_@P]Te@|CiJ`P|OdBdCR`ADl@A|@St@[\\[Re@PqaAlSpaAmSd@QZSZ]Ru@@}@Em@SaAeBeCaP}OgFsEsCaCyJoG{IiEeLyDeGcBmF}@cEa@iGg@uCMJwJHuGmA_AgA{@HYHa@De@Be@?g@?u@Cc@Gm@Mi@uBaHqAbAkLjJkNqc@jNpc@jLkJpAcAtB`HLh@Fl@Bb@iBtAmA@{AjBg@jDf@kDzAkBlAAhBuA?t@?f@Cd@Ed@I`@IXO`@Yb@u@h@_@J]F_@B_B@{DByDJeEXsGn@iBXiAmMu@sAt@rAhAlMhBYrGo@dEYxDKzDC~AA^C\\G^Kt@i@Xc@Na@fAz@lA~@bCsA|CeAY[Y[_@m@c@s@Ui@Ym@Ws@a@aAcL}^gAp@_CwIcBqG}@mG|@lG_DnEWZiLhJ}EvDa@VYLWHUHq@Rp@STIVIXM`@W|EwDhLiJrBnG^TLHVRLJRVNZN\\BDBBB?ZED@J\\K]EA[DC?CCCEO]O[SWMKWSMI_@UsBoGo@wBLc@Nu@Da@Ds@@s@?q@Eq@MiA]_C{Lku@zLju@\\~BLhAbCy@iN}{@aC`AgB{KfBzK`CaAhN|{@fB{@qNu{@iAsGhArG^SP?RDZJr\\fMzPrGfBlC_AbDuJdHhAhDzKkJpIkJlOxFd@eDhG~BnOrFzDgB|@`Dr@fC[pCc@tDWtBVuBb@uDZqCs@gC}@aD{DfBoOsFiG_CNiAOhAe@dDxTpInDBbC_@jAm@tAoBhAQiAPuAnBkAl@cC^oDCvKnEtCr@~@gA_AfAuCs@wKoEyTqImOyFeI}CdI|CqIjJ{KjJiAiDwUrRvUsRo@oBsAmBgAi@uNyFtNxFfAh@rAlBn@nBtJeH~@cDgBmC{PsGs\\gM[KSEQ?_@R_Br@aB}JvBt@aAuFbK~Dz@Zne@vQxGke@yGje@`HrCvS`IpDvAqDwAwSaIaHsCg@rDqe@qQpe@pQxGfCn@gDo@fD{@`Gz@aGyGgCf@sDoe@wQ{@[cK_E`AtFrI~Ch@yDhFi_@D[J]NYfAmBT{@PaAdD_VbOvFcOwFeD~UQ`AUz@tc@vQlE}ZlPckAjU|Iz@\\r@aF\\kClAeDmAdD]jCs@`F{@]g@nDf@oDp@aFq@`FkU}ImPbkAqS_Iz@}Fx@aGlEg\\UeAfBaM`BiLL}@mRoHwDhXvDiXr@cFs@bFuEiBr@aFv@cG_Au@gB{AfBzA~@t@hVnIiVoIw@bGs@`F}DcBkAlB{`@po@mBdLVRZPbRlHjDgVTgA\\mA`JsP~@aAUs@Gs@Fr@Tr@h@h@lCfAmCgAi@i@_A`AaJrP]lAUfAkDfVfM~ElCbAmCcApJar@qJ`r@gM_FcRmH[QWSlBeLz`@qo@eBsC`DiGpCcFmBiBlBhBrTa]sRej@gVsq@mEgMOe@wEoMvEnMNd@lEfMfVrq@u@t@qMrMWTYFW@[QY_@_My^~Lx^X^ZPVAXGVUpMsMeVwq@iIfIhIgIoEcM[Zw@x@uCrCtCsCv@y@Z[Oe@[Zy@x@kIlIXt@Yu@jImIx@y@Z[Qc@cB|AMHOD}@JKDKFwDtDIFK?GEIMaAwC`AvCHLFDJ?HGvDuDJGJE|@KNELIbB}AcEmLe@qAAE[Z}@x@|@y@Z[@Dd@pAr@s@s@r@QNPObElLPb@Nd@nEbMdVvq@~Ndb@J`@D\\?ZCZERGT_RhZIJMHLIHK~QiZFUDSB[?[E]Ka@_Oeb@t@u@rRdj@|NwT`B|CaB}CVc@rCgCtCgApF`PqFaPgCyHfCxHjDqAkDpAuCfAsDgKsOkc@sJkXrJjXtAgF^wAJq@@y@Am@Ik@Oe@wBwGSYQGUASBgBLfBMRCT@PFRXvBvGNd@Hj@@l@Ax@Kp@_@vAuAfFrOjc@dC_Cz@dCt@aBjDkKnY_~@}BwBwOjg@u@d@gAPaEXa@M]m@_@cAaD}IMm@Be@Ty@zAiGRe@`B_@P[~@oEBo@Os@{BmG?m@RiAShA?l@zBlGNr@Cn@_AnEQZaB^Sd@{AhGUx@Cd@Ll@`D|I^bA\\l@`@L`EYfAQt@e@vOkg@|BvBvTmq@wTlq@oY~}@kDjKu@`B{@eCcCgH_Mk]~Lj]bCfHeC~B}BzB|AjEAr@c@x@lA|BmA}Bb@y@@s@}AkE|B{BrDfKnGpQvBwBj_@beAp@tBq@uBk_@ceArFqFf@oDf@{Dg@zD_FgBaCiBg@{@c@mBEkBPaDl@{B^gApo@_qB`AyC`A{CgB_AtEsNxArAyAsA~I_YmD_CcGgEuEeAKISMwDoCcH_FwF_E`@cBbDgOg\\aSf\\`SfAcFq[cRp[bRvAqGhAgFxA}Gua@uYkCnLjCoLyMkJxMjJn@sCpCnBrMdJbIxFcIyFsMeJqCoBmA}@yDmCwCuB_BgA~AfAe@f@g@d@[ZcJtI_@gCkTpSuKbKu@{BuDcLtDbLt@zBcFzEbF{EtKcKjTqSdIyHd@c@{AyEmA{@_EsCcVwP{BhCzBiCpW{cAnRqu@r@qCt@mIyXwRv@mCr@cCcVwQbVvQhCyJtEbDdCdB`h@n^v@Xz@F~@I|@[n@m@d@q@Zu@\\gAt@cCiCwBw@uDmAkDqBiC~EaAjI~FkI_G_F`AoATyNcKxNbKnAUpBhClAjDv@tDhCvBhDeL`GsRoYiSmEzAlE{AoAkEgEwCfEvCnAjEnYhShBgGN_@PYfHgJ|AmF}AlFgHfJQXO^iBfGaGrRiDdLu@bCh@zD@b@yCrLaAxD`AyDxCsLAc@i@{D]fA[t@e@p@o@l@}@Z_AH{@Gw@Yah@o^eCeBuEcDi@_@{DoCcCeBm@g@]]a@o@]w@U_AMeACkAFiAFu@Jg@bDsMcDrMKf@Gt@GhABjALdAT~@\\v@`@n@\\\\l@f@bCdBgSm@ZmA[lAeg@y]ZmAbDkMrDwNzA_GRi@XWVM[yBIYQQeIyFs@g@cDhMrLlIsLmICJBKbDiMr@f@bAyDcAxDdIxFPPHXZxB\\EXF\\NvEbDxQhMyQiMwEcD]OYG]DWLYVSh@{A~FsDvNhb@zYpEmQqElQkCnKzCtBWbAVcA{CuBjCoKib@{YcDjM[lAw@~Cv@_Ddg@x]fSl@zDnCh@^jDcNkDbNiCxJs@bCw@lCxXvRXiDYhDu@lIbVvPcVwPs@pCoRpu@cD_C{t@tt@p]nVyCzLqEnQsBdIiAhEoGsEnGrEjAlBL`@Ma@kAmBhAiErBeInAz@nLzHoL{HoA{@pEoQxC{Lq]oV{AeAoA_eAg@sd@iC`K_BiA~AhAhCaKl@_C`Lsc@dCyJtAmFfh@t]gh@u]z@gD{@fDuAlFeCxJaLrc@_BgAyCsBCTETDUBUxCrB~AfAm@~Bf@rd@nA~dAzAdAzt@ut@bD~BqWzcAbVvPdAeEfM{f@gMzf@eAdE~DrClAz@zAxE`@a@a@`@e@b@eIxH^fCl@jEuA|FtA}Fm@kEwS|Rs@sBr@rBvA`EpL|GqL}GwAaEyR~Qs@sBiYhXc@d@[b@Yb@Yf@Sj@Qp@m@`Cl@aCPq@Rk@Xg@Xc@Zc@b@e@hYiXr@rBiYhX]^UZS\\S\\KXzWpMo@jDx@^l@kDm@jDy@_@gXyMwAs@vAr@s@rCzgAri@f@{Cml@qYll@pYg@zCwDdTQz@Ml@M^KPa@d@jEzLkE{LuLlLgP|OoBtBnBuB|K`Z|F{FvGSfAr@rMi@^eBxAIyAH_@dBsMh@gAs@wGRlZsT|Ao@|Cs@lCSzAZpAv@f@z@JpBIrBs@|Dg@rAe@?gI^a@pBu@gBt@fB`@qBwAHvAIfI_@zAtEX`@z@PnCOj@HdE`EeEaEk@IoCN{@QYa@{AuEd@?f@sAr@}DHsBKqBg@{@qAw@{A[mCR}Cr@}An@mZrT}FzFlHbSfBMRCT@PFRXvBvGNd@Hj@@l@Ax@Kp@_@vA^wAJq@@y@Am@Ik@Oe@wBwGSYQGUASBgBLkDh@oSyk@nSxk@jDi@mHcS}KaZqAkDmPme@lPle@pAjDfP}Ot@rBhA`D`BtEkE`EjEaE~GsG_HrGaBuEiAaDu@sBtLmL`@e@JQL_@Lm@P{@eDcBwBeAvBdAdDbBvDeT`h@zVGZkChO~BjA_CkAjCiOF[ah@{V{gAsi@r@sCfXxMn@kD{WqMCDMf@i@tBh@uBLg@BEJYR]R]T[\\_@hYiXVr@fCnHgCoHWs@xR_RvS}RbJuIZ[f@e@d@g@vCtBxDlClA|@o@rCta@tYyA|GiAfFwApGgAbFcDfOa@bBaApEoA|FeA|EeAvEk@xCmArGlAsGj@yCzJ`HzFtC{FuC{JaHdAwEdA}EnA}FcAo@m[qRl[pRs@R{\\sPb@aC{XcNo@Gm@T}OjO|OkOl@Un@FzXbNc@`Cz\\rPr@SbAn@`AqEvF~DbH~EvDnCYlAXmARL~@sD~AeG_BdG_ArDJHtEdAbGfElD~B_J~XMK}FqCcCkAbCjA|FpCLJuErNfB~@aAzCjB`Axk@xWb@sCc@rCyk@yWkBaAkB_A_g@wV~f@vV_ApChBfAqo@~pB~A|Azo@uqB{o@tqBa@tAO|A@rAZlAZv@t@x@|EjB}EkBu@y@[w@[mAAsAN}A`@uA_B}A_ByA~AxA_@fAm@zBQ`DDjBb@lBf@z@`ChB~EfBg@nDsFpFwBvBfEtLqBnBrWzt@L`@Fb@@d@A`@Kp@zBfA`GxC^c@?@x@}@uCaJdAgAXQTOTKPGXGX?`@?ZH[Ia@?Y?YFQFUJUNYPeAfAtC`Jy@|@?A_@b@aGyCXaC@gAE{@SeAiXgv@hXfv@RdAp@AN@LDNJHNdAcAeAbA`DxIaDyIIOOKMEOAq@@Dz@AfAY`CgAtHfAuH{BgAJq@@a@Ae@Gc@Ma@sW{t@pBoBgEuLqBpBaDcJeAUgBfCk^nk@WtAVuAj^ok@fBgCdAT`DbJfErLiR|Qg@yAE[A_@@a@FWHWNOPO~CcBXSJQDY?YASEUEOISHRDNDT@R?XEXKPYR_DbBQNONIVGVA`@@^DZf@xAw@v@ONININGREPG`@uA`K}@zG|@{GtAaKFa@DQFSHOHONOv@w@hR}QgEsLpBqBoGqQsCfCWb@}NvTsT`]qCbFaDhGdBrCjAmB|DbBtEhBlRnHr@cFs@bFxTrIp@aFq@`FyTsIM|@aBhLgB`MTdAmEf\\y@`G{@|FpS~HmE|Ztd@fQgAz@KJILITGXiBjMpO`GqOaGw@rFaA`H`AaHv@sFhBkMFYHUHMJKfA{@JIJMFOFSDSnDmWee@gRde@fRxHzCb@cDq@qFlBeN|Ew]|AyKrA{AsAzA}AxKdC~@RFSGeC_A}Ev]`Cz@ZJ[KaC{@mBdNp@pFc@bDyH{CoDlWERGRGNKLKHud@gQuc@wQgAlBOXK\\EZiFh_@i@xDsI_DwBu@gCb@fCc@}@sF_CbAt@rEu@sE~BcAvAo@ZbB[cBwAn@|@rF`B|J~As@pNt{@XN^?_@?YOgBz@cCx@Dp@?p@Ar@Er@E`@Ot@Mb@Yt@a@v@i@l@kMnK}@v@c@\\e@V]Lo@RDn@LC|BzX}B{XMBEo@n@S\\Md@Wb@]|@w@jMoKh@m@`@w@Xu@n@vBV[~CoEzCeC{CdCbBpG~BvIeBfAPb@vF`RwFaRQc@dBgAjIrXvA~EwA_FkIsXfAq@Su@Ga@Ce@Bm@Hi@T}@U|@Ih@Cl@Bd@F`@Rt@bL|^`@`AVr@Xl@Th@b@r@^l@XZXZ}CdAcCrAItGKvJ{ESmFSsDAoEJwE\\}G`A_G|@tCz~@xPiBaBu`@`Bt`@yPhB~D~_A~OoBu@uQt@tQpAOqAN_PnB_E_`AuC{~@~F}@|GaAvE]nEKrD@bChk@n@vP\\xH_Ed@~De@|Ah_@~@rUzEe@xCfq@fAfWb@vLmAP_AHyz@`Ha@KY_@Yu@MaAC}@B|@L`AXt@X^`@Jxz@aHKpUQnAWr@dAf@xQr@bLdAcLeAyQs@eAg@Vs@PoAJqU~@IlAQD~@DVJNJNNFN?N?d]eENENCPOHQDQDUBSh@mIToDpA}RpNn@qNo@?uD?tDqA|RUnD|FbAToD}FcA|FbAjGjAUlDTmDkGkAUnD}FcAi@lICRETEPIPQNOBODe]dEO?O?OGKOKOEWE_Ac@wLcQzBbQ{BgAgWcQfBiPvBc^`Ev@`S~PkB_QjBTvFUwFw@aSb^aEhPwBbQgByCgq@t_@uEVvFWwFdAMbDrQnFhBoFiBS_@UUSM[K[Ga@Ca@BOBUFSHQLMJKJKNKNGLINKXIZGZEXE\\Ad@?\\?XX`GeLtAdLuAYaG?Y?]@e@D]DYF[H[JYKXIZGZEXE\\Ad@?\\?XX`GeLtAdLuAYaG?Y?]@e@D]DYF[H[JYHOFMJOJOJKLKPMRITGNCQyDE{@Dz@PxD`@C`@BZFZJRLTTR^cDsQVGxBQz@Or@QxEoAr@dGv@jGh@fEp@fF}@vFuBjAHj@`@hCh@QxCaAPrE|JiB?d@Dj@Df@Jd@LXHRTZXXb@TXDxMr@\\F\\HRFZJZTZ\\bD`FVh@Tf@Rb@Rj@Ph@Pv@Lv@Jv@Hx@BjA?dAAdAGjAKtAeCl[M~AIhA?nA@pALvAZbA`@z@`@b@b@^p@Zn@Lv@B|@Ev@M|@QtHyC`DOpFmC|DyDvAiBHKvDeFzFyI{FxIwDdFIJvBjEbQyJxC{AxCaCm@yK|CgNHqBmAWaBk@cAm@qBwBkBwC{EcL_LmXKeEu@}@aOq]k@mA}A}C|AU[q@Zp@b@z@j@fAk@gAc@{@vFkAwFjA}AT_A{Aa@Um@DqCl@sCl@vAcC`@oA\\uBdCyNeCxNi@WgBeEcE}Gj@wAlCiH|BaGl@_B|ARpCB|A@p@@sB{bApj@|Y~EfBzE|A~HnE`C|@tAXl@Lt@FKnDiAdFzHzO{H{OhAeFJoDfA@rCk@lBc@zAaA|AkBxA{BfA{BbAkCh@{B^eCNyBJcCB{B?GfBBbOfA[bErH~Ar@Na@pElBf@dEa_@lAwKhBgPfFed@gFdd@oDiAaCo@iCk@y@I{@Cz@Bx@HhCj@`Cn@nDhAiBfPjMjDj@Gf@Y`@i@ZeAdQ}yAkQwEpDu[lC{UnOtA|@Gr@m@Xy@PeBdGcu@NaCN}L?iFEsDh@EPKF[?]EgFAs@R?b@n@`CjD~AzLMlLm@vC[fBK`AnIpAb@l@x@zAHVFZ@R?RA`DPxBSjEEzDGhEf@fHn@`ON~Al@bC`@fA\\b@dAp@tCjAn@VZDf@Df@{c@DiKC{FaAmdA[a]i@yk@cLR}LTC?Vxf@WvAGnB@lDO@cN`ASDOiGFqC_Aq_@G{@Is@Oq@Q_@Q[SWUS[SHqClAsZTsF?yBS}DYsCWyAOOII[I[@_@VmKjQqD|FsBxBiBlA}CnA]WW}Cx@s@fAoAx@oAtAcChAaCfAwCp@wB`@wAP}@Hu@FcABsAA_AG{AaAwCiAy@}@c@i@aAMaADmAXu@`AS`AHb@f@n@xJHdBaAwCiAy@}@c@i@aAMaADmAXu@`AS`AHb@f@n@xJHdBFzA@~@CrAGbAIt@Q|@a@vAq@vBgAvCiA`CuAbCy@nAgAnAy@r@QJsDrBo@`@]X[f@Ij@@|@d@zCNz@Hx@@n@Gj@Oj@a@`@e@Lc@?QCOGOMKQGQG_@Go@?{FCk@Kq@Og@MYQWQQWM[Kc@IuGk@MCMEKEKOKOGOESCWAW@i@PyGNoFtF`@X?TILUHOFc@d@aNk@sBj@rBe@`NGb@INMTUHY?uFa@{DYeE[f@wQ^aNw@IyAMB}@C|@]`N\\aNxALa@`Ng@xQOnFNoFwAIh@}Qi@|QeA|_@I|CtAH`Ak]~Fb@NoFOnFA^@_@zDX{DY_Gc@aAj]zGb@{Gc@uAIEzBWnDyB~OeAjHzO|GnEZoE[{O}GdAkH`DtAhF`@vAe@~@iAjBaCXw@JgEKfEYv@kB`C_AhAwAd@iFa@aDuAxB_PVoDD{BH}CdA}_@vAHf@yQuAMiAMhALtALx@Hy@I`@aNv@Hh@iSr@}Vl@}GdByt@V}G{CWh@aTJmD@w@?o@G{@E[GYWs@_@e@_@Y}@SaJk@sAKw@Ev@DJuDsWkB}BKiD?iDCB~D@fEk@ZQv@?j@wC?yCA?oBAcDwKFw@Km@Uo@e@o@g@_NuK{B{A_BiAv@kCdD}KwS}NvS|N~ApA_BqAbDiL`Pkj@`FiQWUEGMSIYQm@IQKMwC_CYMKCMASBOHSVDq@F[JQLKPMLOBK?QI[Oc@XcAsSkOo[|fAn[}fAkPaLkTnu@mAfEs@~BI`@GZCd@?^@XD\\DXJZJVLPNPPPhKfIiKgIQQOQMQKWK[EYE]AY?_@Be@F[Ha@r@_ClAgEoAcAzD_NMqDlIqYmIpYLpD{D~MsDwCrDvCnAbAjTou@_OsKySxs@a@|A`@}AxSys@}PaMdJcYsA_AvAwEnA{DoAzDwAvErA~@bEvCxCvBfI~F`@Xa@YgI_GyCwBcEwCvAuEsAaArA`ABGDCDAD?F@HDJFjBrAlP~KmP_LkBsAKGIEGAE?E@EBCFwAtEeJbY}InY_EpMeAxCq@jBi@tAQ`@w@dBeAtBo@lAm@fAiAnBwBpDYd@cA_BpCwEfAoB{@qAf@_Ar@qAbAsBd@_AZw@d@oALWb@oANe@p@kBjD_LPm@lBoHLm@d@mDNaAhDcLvFwOj@qCbBiF?ECICOIUOYMWMKH_@f@qBj@yBJa@dBxAy@pCx@qCbEfC_A|C~@}Ct@f@u@g@|IwZvDqKlDeLbCaIRk@J[N]Va@VUeEmDjFkQkFjQoNle@cAfBqJ~ZpJ_[bAgBnNme@dElDVUXIRAXDTHd@\\pXlRdItFeIuFYfAuDjOtDkOyWyQ[M[EQA[BWLUPSTS\\Ob@_CnHfFnD~DpC|F|DXiAYhA}F}DViAWhA_EqCVgAWfAgFoD~BoHNc@R]RUTQVMZCP@ZDZLxWxQxMdJcBxEgCzHkFsDjFrDfC{HbByEdAlA_BnEgCzHa@lAeBbGcBlGcA|DwExQn\\rU`@TXHX@ZEd@Sl@a@^a@\\g@bL{SaNkS`NjScLzS]f@_@`@m@`@e@R[DYAYIa@Uo\\sUiArEhAsEkAy@kApEjAqEdFcSsj@i`@bIsWcIrW{Jd\\zJe\\rj@h`@v@uC`BoGfBeG`@qAa@pAgBdGaBnGw@tCeFbSjAx@vEyQbA}DbBmGdBcG`@mAfC{H~AoEeAmA^cA`AgCjAqCrAaD`CcFy@_BwBeEvBdEx@~AaCbFsA`DkApCaAfCyCsBCi@Ec@E_@K]IWKUOUMOOMKK_G_EMIKMIOEKESAQBWCV@PDRDJHNJLLH~F~DJJNLLNNTJTHVJ\\D^Db@Bh@xCrB_@bAyMeJXgAqXmRe@]UIYES@Dc@DYFSn@Ch@Bh@FZNZPzGtEJFJ@L@NALEJGPQQPKFMDO@MAKAKG{GuE[Q[Oi@Gi@Co@BhQ{l@|@fBd@|@xBfEyBgEe@}@}@gBfCsIh@oBj@@z@?`Fh@z@RpRxFhA\\dA^|TtILFNNrA~BFV?XC\\Ef@hA|AxJxNs@x@JLkAxBENCP?R@|@ARCLELeB`DELCN@PDLFJzNhTNt@\\h@v@jAqK~RMTwAyBcCwCqDcG_BeC~AdCpDbGbCvCvAxBmLfTmOqUlOpUlLgTLUxBdDeOhXKLIHKFODK@M@M?IAIAGAIEGCiW~d@c@t@i@~@c@l@a@\\WLjFtDbDzB?`@Bb@DVwAzE{BpHn@VfV`M|HzEfF|CfGpBbH`@pCDnFKpEcBzGiDvFoD|o@og@hZuShHcDjAk@kAj@iHbDiZtS}o@ng@wFnD{GhDqEbBpEcBzGiDvFoD|o@og@hZuShHcDjAk@rImCNaAFi@BiAAYIoAHnA@XChAGh@O`AsIlCkAj@iHbDiZtS}o@ng@wFnD{GhDqEbBBeICdIC`D~Bz@xF~GxFwCnVkLlC_BbKcHDCDAB?D@DB`@r@wLvQcA_@cAz@aAn@}@`@y@ZcCl@bCm@x@[|@a@`Ao@bA{@bA^Wb@oAhBgAlBo@vAm@hBo@`Ce@hC[`CeAjLSvCGdC@`DH~CVdD`@nDh@rDj@vChA|DbAlClAlC^t@tEhIjDzGv@nBd@|AZrAFj@@h@Ah@OpDEfBEtC@`Bx@`a@R|IS}IbAsD]mO]sR@yALsD@s@C_AEc@Ik@Kc@Ke@i@cBw@iB}KqSGQyA_Dq@uBi@eBe@_BES",
    },
]


FORM_CLASS = get_ui_class("main_dialog.ui")


class MainDialog(QDialog, FORM_CLASS):
    def __init__(self, parent=None, iface=None):
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.iface = iface
        self.buttonBox.accepted.connect(self.load_polyline)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        self.buttonBox.button(
            QDialogButtonBox.Reset).clicked.connect(self.clear)
        self.randomSamplePushButton.clicked.connect(self.load_random_sample)
        self.current_sample = 0

    def load_random_sample(self):
        example_idx = self.current_sample % len(EXAMPLES)
        example = EXAMPLES[example_idx]
        self.current_sample += 1
        self.layerNameLineEdit.setText(example["layer_name"])
        self.polylineTextEdit.setPlainText(example["encoded_polyline"])
        self.precisionSpinBox.setValue(example["precision"])

    def load_polyline(self):
        polyline_string = self.polylineTextEdit.toPlainText()
        polyline_string = codecs.decode(polyline_string, "unicode_escape")
        coordinates = polyline_decode(
            polyline_string, precision=self.precisionSpinBox.value(), geojson=True
        )
        qgis_coords = [QgsPointXY(x, y) for x, y in coordinates]
        geom = QgsGeometry.fromPolylineXY(qgis_coords)

        # Create memory layer
        layer = QgsVectorLayer(
            "LineString?crs=epsg:4326&field=id:integer",
            self.layerNameLineEdit.text(),
            "memory",
        )
        pr = layer.dataProvider()
        feature = QgsFeature()
        feature.setAttributes([1])
        feature.setGeometry(geom)
        pr.addFeatures([feature])
        layer.updateExtents()
        style_path = os.path.join(
            os.path.dirname(os.path.realpath(__file__)), "redLineStyle.qml"
        )
        layer.loadNamedStyle(style_path)
        QgsProject.instance().addMapLayer(layer)

        # Zoom to the layer
        if self.zoomToLayerCheckBox.isChecked():
            self.iface.mapCanvas().setExtent(layer.extent())

    def clear(self):
        self.polylineTextEdit.clear()
