# -*- coding: utf-8 -*-

"""
/***************************************************************************
 emiTools
                                 A QGIS plugin
 This plugin compiles tools used by EMI-PB

                              -------------------
        begin                : 2024-10-10
        copyright            : (C) 2024 by Alexandre Parente Lima
        email                : alexandre.parente@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandre Parente Lima'
__date__ = '2024-10-10'
__copyright__ = '(C) 2024 by Alexandre Parente Lima'

# This will get replaced with a git SHA1 when you do a git archive
__revision__ = '$Format:%H$'

import os
import re
import string
from datetime import datetime, date
from osgeo import gdal
from qgis.PyQt.QtCore import QDate
from qgis.core import QgsProject, QgsMessageLog

from .emi_tools_util import tr


def validate_cpf_logic(cpf_number) -> bool:
    s = ''.join(filter(str.isdigit, str(cpf_number)))
    if len(s) != 11:
        return False
    # rejeita CPFs com todos dígitos iguais
    if s == s[0] * 11:
        return False
    nums = list(map(int, s))
    total = sum(a * b for a, b in zip(nums[:9], range(10, 1, -1)))
    dv = (total * 10) % 11
    if dv == 10:
        dv = 0
    if dv != nums[9]:
        return False
    total = sum(a * b for a, b in zip(nums[:10], range(11, 1, -1)))
    dv = (total * 10) % 11
    if dv == 10:
        dv = 0
    return dv == nums[10]


def validate_cnpj_logic(cnpj_number) -> bool:
    s = ''.join(filter(str.isdigit, str(cnpj_number)))
    if len(s) != 14:
        return False
    if s == s[0] * 14:
        return False
    nums = list(map(int, s))
    w1 = [5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2]
    w2 = [6, 5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2]
    dv1 = 11 - (sum(n * w for n, w in zip(nums[:12], w1)) % 11)
    dv1 = 0 if dv1 > 9 else dv1
    dv2 = 11 - (sum(n * w for n, w in zip(nums[:13], w2)) % 11)
    dv2 = 0 if dv2 > 9 else dv2
    return nums[12] == dv1 and nums[13] == dv2


def format_cpf_logic(cpf_string) -> str:
    cleaned_string = ''.join(filter(str.isdigit, str(cpf_string)))
    if len(cleaned_string) != 11:
        raise ValueError(tr("Invalid number. Please provide an 11-digit numeric string."))
    return f"{cleaned_string[:3]}.{cleaned_string[3:6]}.{cleaned_string[6:9]}-{cleaned_string[9:]}"


def format_cnpj_logic(cnpj_string) -> str:
    cleaned_string = ''.join(filter(str.isdigit, str(cnpj_string)))
    if len(cleaned_string) != 14:
        raise ValueError(tr("Invalid number. Pass a numeric string as the input parameter."))
    return f"{cleaned_string[:2]}.{cleaned_string[2:5]}.{cleaned_string[5:8]}/{cleaned_string[8:12]}-{cleaned_string[12:]}"


def format_cpf_cnpj_logic(cpf_cnpj_string) -> str:
    cleaned_string = ''.join(filter(str.isdigit, str(cpf_cnpj_string)))
    if len(cleaned_string) == 11:
        return format_cpf_logic(cleaned_string)
    if len(cleaned_string) == 14:
        return format_cnpj_logic(cleaned_string)
    raise ValueError(tr("Invalid number. Pass a numeric string as the input parameter.."))


def mask_cpf_logic(cpf_number) -> str:
    s = ''.join(filter(str.isdigit, str(cpf_number)))
    if len(s) != 11:
        return "Invalid CPF"
    return f"***.{s[3:6]}.{s[6:9]}-**"


def mask_name_logic(full_name) -> str:
    parts = str(full_name).split()
    if len(parts) < 3:
        return str(full_name)
    middle = ['*' * len(p) for p in parts[1:-1]]
    return parts[0] + ' ' + ' '.join(middle) + ' ' + parts[-1]


# List of words that must remain in lowercase in names/titles
# according to Portuguese language conventions and ABNT standards.

PT_BR_LOWERCASE_WORDS = {
    # Artigos definidos e indefinidos
    'a', 'o', 'as', 'os',
    'um', 'uma', 'uns', 'umas',

    # Preposições simples
    'de', 'em', 'por', 'para', 'com', 'sem', 'sob', 'sobre',
    'até', 'após', 'ante', 'contra', 'desde', 'entre', 'trás',

    # Conjunções coordenativas
    'e', 'ou', 'mas', 'nem',

    # Conjunções subordinativas comuns
    'se', 'que', 'porque', 'como', 'quando', 'conforme',
    'embora', 'caso', 'enquanto', 'logo', 'pois', 'porquanto', 'salvo',

    # Contrações com artigos
    'da', 'do', 'das', 'dos',
    'na', 'no', 'nas', 'nos',
    'à', 'às', 'ao', 'aos',
    'pela', 'pelo', 'pelas', 'pelos',

    # Contrações com pronomes
    'dela', 'dele', 'delas', 'deles',
    'nela', 'nele', 'nelas', 'neles',

    # Contrações demonstrativas
    'deste', 'desta', 'destes', 'destas',
    'neste', 'nesta', 'nestes', 'nestas',
    'daquele', 'daquela', 'daqueles', 'daquelas',
    'naquele', 'naquela', 'naqueles', 'naquelas',

    # Outras contrações usuais
    'doutro', 'doutros', 'doutra', 'doutras',
    'noutro', 'noutros', 'noutra', 'noutras',

    # Palavras de locuções
    'depois', 'antes', 'além', 'aquém'
}

_PUNCT = set('"' + "'«»“”‘’" + string.punctuation)  # pontuação que pode estar colada
_STRONG_PUNCT = {'.', ':', '!', '?', ';'}  # pontuação que reinicia capitalização em títulos


def _split_affixes(token: str):
    """Separates leading and trailing punctuation from the core of the word."""
    if not token:
        return "", "", ""
    i, j = 0, len(token) - 1
    while i <= j and token[i] in _PUNCT:
        i += 1
    while j >= i and token[j] in _PUNCT:
        j -= 1
    return token[:i], token[i:j + 1], token[j + 1:]


def _capitalize_core(core: str) -> str:
    """Capitalizes a regular word."""
    if not core:
        return core
    return core[0].upper() + core[1:].lower()


def _process_hyphenated(core: str, force_capitalize: bool, lowercase_words: set) -> str:
    """Processes hyphenated words part by part."""
    parts = core.split('-')
    out = []
    for part in parts:
        if not part:
            out.append(part)
            continue
        lw = part.lower()
        if force_capitalize or lw not in lowercase_words:
            out.append(_capitalize_core(part))
        else:
            out.append(lw)
    return '-'.join(out)


def format_capitalization_logic(
        text: str,
        force_after_strong_punct: bool = False
) -> str:
    """
    Capitalizes names/titles according to PT-BR/ABNT rules:
      - the first word is always capitalized;
      - articles/prepositions/conjunctions remain lowercase (list);
      - after strong punctuation (.:;!?) the next word is capitalized;
      - handles hyphenated words.
    """
    if not text:
        return ""

    lower = PT_BR_LOWERCASE_WORDS

    tokens = str(text).split()
    result = []
    next_force = True

    for tok in tokens:
        lead, core, trail = _split_affixes(tok)

        if core:
            lw = core.lower()
            force_cap = next_force
            if '-' in core:
                new_core = _process_hyphenated(core, force_cap, lower)
            else:
                if force_cap or lw not in lower:
                    new_core = _capitalize_core(core)
                else:
                    new_core = lw
        else:
            new_core = core

        result.append(f"{lead}{new_core}{trail}")

        # ABNT: restarts capitalization after strong punctuation (in titles)
        if force_after_strong_punct and any(ch in _STRONG_PUNCT for ch in tok):
            next_force = True
        else:
            next_force = False

    return ' '.join(result)


#   This dictionary for all satellites.
SATELLITE_PROPERTIES = {
    # Regex Pattern: { 'name': ..., 'date_format': ..., 'source': ... }

    # Landsat Family
    r'^LC09': {'name': 'LandSat 9', 'date_format': 'YYYYMMDD', 'source': 'United States Geological Survey (USGS).'},
    r'^LC08': {'name': 'LandSat 8', 'date_format': 'YYYYMMDD', 'source': 'United States Geological Survey (USGS).'},
    r'^LE07': {'name': 'LandSat 7', 'date_format': 'YYYYMMDD', 'source': 'United States Geological Survey (USGS).'},
    r'^LT05': {'name': 'LandSat 5', 'date_format': 'YYYYMMDD', 'source': 'United States Geological Survey (USGS).'},
    r'^LT04': {'name': 'LandSat 4', 'date_format': 'YYYYMMDD', 'source': 'United States Geological Survey (USGS).'},
    r'^LM0[1-3]': {'name': 'LandSat MSS (1–3)', 'date_format': 'YYYYMMDD',
                   'source': 'United States Geological Survey (USGS).'},

    # Sentinel Family (Copernicus)
    r'^S1[AB]': {'name': 'Sentinel 1', 'date_format': 'YYYYMMDD',
                 'source': "European Union's Earth Observation Programme (COPERNICUS)."},
    r'^S2[A-C]': {'name': 'Sentinel 2', 'date_format': 'YYYYMMDD',
                  'source': "European Union's Earth Observation Programme (COPERNICUS)."},
    r'^S3[AB]': {'name': 'Sentinel 3', 'date_format': 'YYYYMMDD',
                 'source': "European Union's Earth Observation Programme (COPERNICUS)."},
    r'^S5P': {'name': 'Sentinel 5P', 'date_format': 'YYYYMMDD',
              'source': "European Union's Earth Observation Programme (COPERNICUS)."},

    # Pattern for individual bands
    r'^T\d{2}[A-Z]{3}': {'name': 'Sentinel 2', 'date_format': 'YYYYMMDD',
                         'source': "European Union's Earth Observation Programme (COPERNICUS)."},

    # NASA Satellites (EOS)
    r'^MOD|MYD': {'name': 'MODIS', 'date_format': 'JULIAN_y_ddd',
                  'source': 'National Aeronautics and Space Administration (NASA).'},
    r'^A\d{7}\b': {'name': 'MODIS', 'date_format': 'JULIAN_y_ddd',
                   'source': 'National Aeronautics and Space Administration (NASA).'},
    r'^(VNP|VJ\d{2})': {'name': 'VIIRS', 'date_format': 'JULIAN_y_ddd',
                        'source': 'National Aeronautics and Space Administration (NASA).'},
    r'^AST_': {'name': 'ASTER', 'date_format': 'MMDDYYYY', 'source': 'NASA/METI.'},

    # Indian Remote Sensing Satellites (IRS)
    r'^L[34]_RS[12]|^AW_RS[12]': {'name': 'Resourcesat', 'date_format': 'YYYYMMDD',
                                  'source': 'Indian Space Research Organisation (ISRO).'},
    r'^C[123]_': {'name': 'Cartosat', 'date_format': 'YYYYMMDD',
                  'source': 'Indian Space Research Organisation (ISRO).'},

    # Sino-Brazilian Satellite
    r'^CBERS': {'name': 'CBERS', 'date_format': 'YYYYMMDD',
                'source': 'Instituto Nacional de Pesquisas Espaciais (INPE) / China Academy of Space Technology (CAST).'},
    r'^CBERS[_-]?4A': {'name': 'CBERS-4A', 'date_format': 'YYYYMMDD',
                       'source': 'Instituto Nacional de Pesquisas Espaciais (INPE) / China Academy of Space Technology (CAST).'},

    # High-Resolution Commercial Satellites
    r'WV0[1-4]|GE01': {'name': 'Maxar/DigitalGlobe', 'date_format': 'DDMONYY', 'source': 'Maxar Technologies.'},
    r'^IK01': {'name': 'IKONOS', 'date_format': 'YYYYMMDD', 'source': 'Maxar Technologies.'},
    r'^QB02': {'name': 'QuickBird', 'date_format': 'YYYYMMDD', 'source': 'Maxar Technologies.'},

    # Planet (variações)
    r'PSScene': {'name': 'PlanetScope', 'date_format': 'YYYYMMDD',
                 'source': 'Includes material © (2025) Planet Labs Inc. All rights reserved.'},
    r'^SkySat': {'name': 'SkySat', 'date_format': 'YYYYMMDD',
                 'source': 'Includes material © (2025) Planet Labs Inc. All rights reserved.'},
    r'_psb_|_pss_': {'name': 'PlanetScope', 'date_format': 'YYYYMMDD',
                     'source': 'Includes material © (2025) Planet Labs Inc. All rights reserved.'}
}


def get_satellite_logic(filename) -> str:
    """
    Identifies the satellite from the filename and returns a dictionary of its properties.
    """
    for pattern, properties in SATELLITE_PROPERTIES.items():
        if re.search(pattern, filename, re.IGNORECASE):
            # Return the properties dictionary
            return properties
    return None


def get_image_date_logic(filename) -> date:
    """
    Extracts the acquisition date from the filename, returning a datetime.date.
    """
    info = get_satellite_logic(filename)
    if not info:
        raise ValueError("Could not identify satellite to determine date format.")
    date_format = info['date_format']

    if date_format == 'YYYYMMDD':
        possible_dates_str = re.findall(r'\d{8}', filename)
        vals = []
        for s in possible_dates_str:
            try:
                vals.append(datetime.strptime(s, "%Y%m%d").date())
            except ValueError:
                pass
        if vals:
            return min(vals)

    elif date_format == 'JULIAN_y_ddd':
        match = re.search(r'A?(\d{4})(\d{3})', filename, re.IGNORECASE)
        if match:
            year, day_of_year = match.groups()
            return datetime.strptime(f'{year}{day_of_year}', '%Y%j').date()

    elif date_format == 'MMDDYYYY':
        possible_dates_str = re.findall(r'\d{8}', filename)
        vals = []
        for s in possible_dates_str:
            for fmt in ("%Y%m%d", "%m%d%Y"):
                try:
                    vals.append(datetime.strptime(s, fmt).date())
                except ValueError:
                    pass
        if vals:
            return min(vals)

    elif date_format == 'DDMONYY':
        match = re.search(r'(\d{2}[A-Za-z]{3}\d{2})', filename)
        if match:
            return datetime.strptime(match.group(1).upper(), '%d%b%y').date()

    raise ValueError(f"Could not parse date from filename with expected format '{date_format}'.")


def get_layer_custom_property_logic(layer_name: str, property_key: str):
    """
    Returns the value of a 'Custom Property' from a layer in the project.
    """

    layers = QgsProject.instance().mapLayersByName(layer_name)

    if not layers:
        QgsMessageLog.logMessage(f"Function 'get_layer_custom_property' could not find layer: {layer_name}",
                                 'Python Functions')
        return None

    # Get the first layer found with this name
    layer = layers[0]

    property_value = layer.customProperty(property_key)
    return property_value
