# -*- coding: utf-8 -*-

"""
/***************************************************************************
 emiTools
                                 A QGIS plugin
 This plugin compiles tools used by EMI-PB

                              -------------------
        begin                : 2024-10-10
        copyright            : (C) 2024 by Alexandre Parente Lima
        email                : alexandre.parente@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandre Parente Lima'
__date__ = '2024-10-10'
__copyright__ = '(C) 2024 by Alexandre Parente Lima'

__revision__ = '$Format:%H$'

import os
import tempfile
import zipfile
from datetime import datetime
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing, QgsVectorFileWriter,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterFolderDestination,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingException,
                       QgsProject, QgsVectorLayer,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingFeatureSource,
                       QgsWkbTypes)

from .emi_tools_util import (tr,
                             get_validated_folder,
                             create_memory_layer,
                             save_as_vector,
                             compress_to_zip,
                             get_associated_files)


class emiToolsExportTerms(QgsProcessingAlgorithm):
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'
    FORMAT_INFO = {
        0: {'driver': 'ESRI Shapefile', 'ext': 'shp', 'zip_exts': ['shp', 'shx', 'dbf', 'prj', 'cpg']},
        1: {'driver': 'GPKG', 'ext': 'gpkg', 'zip_exts': ['gpkg']},
        2: {'driver': 'KML', 'ext': 'kml', 'zip_exts': ['kml']}
    }

    def initAlgorithm(self, config=None):

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                'layer',
                tr('Input layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # Parameter to select the field for the embargo term number
        self.addParameter(
            QgsProcessingParameterField(
                'num_tei_field',
                tr('Embargo term field'),
                parentLayerParameterName='layer',
                type=QgsProcessingParameterField.String,
                defaultValue='numero_tad'
            )
        )

        # Parameter to select the field for the embargo term series
        self.addParameter(
            QgsProcessingParameterField(
                'serie_tei_field',
                tr('Embargo term series field'),
                parentLayerParameterName='layer',
                type=QgsProcessingParameterField.String,
                defaultValue='serie_tad'
            )
        )

        # Setting the default output folder to the user's home directory
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_FOLDER,
                tr('Output folder')
            )
        )

        # Parameter to choose the output format
        self.addParameter(
            QgsProcessingParameterEnum(
                'output_format',
                tr('Output file format:'),
                options=[info['driver'] for info in self.FORMAT_INFO.values()], defaultValue=0
            )
        )

        # Add parameter to export all features to a single file
        self.addParameter(
            QgsProcessingParameterBoolean(
                'export_all_to_single',
                tr('Export all features to a single file'),
                defaultValue=False
            )
        )

        # Parameter to compress the output file
        self.addParameter(
            QgsProcessingParameterBoolean(
                'compress_output',
                tr('Compress output file copy (.zip)'),
                defaultValue=False
            )
        )

        # Parameter to load the output file
        self.addParameter(
            QgsProcessingParameterBoolean(
                'load_output',
                tr('Open output files after executing the algorithm'),
                defaultValue=True
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        feedback = QgsProcessingMultiStepFeedback(1, feedback)
        layer = self.parameterAsSource(parameters, 'layer', context)
        if layer is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, 'layer'))

        extracted_features = list(layer.getFeatures())
        num_tei_field = parameters['num_tei_field']
        serie_tei_field = parameters['serie_tei_field']
        export_all_to_single = self.parameterAsBoolean(parameters, 'export_all_to_single', context)
        output_format = self.parameterAsEnum(parameters, 'output_format', context)
        compress_output = self.parameterAsBoolean(parameters, 'compress_output', context)

        # Get and validate output folder
        output_folder_param = self.parameterAsString(parameters, self.OUTPUT_FOLDER, context)
        output_folder = get_validated_folder(output_folder_param)

        # Check for duplicate features
        self.check_duplicates(extracted_features, num_tei_field)

        # Create a temporary layer
        temp_layer = self.temp_layer(layer, extracted_features, context)

        # Remove unnecessary fields and rename others
        self.remover_fields(temp_layer, num_tei_field, serie_tei_field)

        # Rename the fields
        self.rename_fields(temp_layer, num_tei_field, serie_tei_field)

        output_files = []
        today_str = "lote" + datetime.today().strftime('%Y%m%d')

        # Export all to a single file or individual files
        if export_all_to_single:
            output_files.extend(self.export_single_file(temp_layer, output_folder, today_str, output_format, feedback))
        else:
            output_files.extend(self.export_individual_files(temp_layer, output_folder, output_format, feedback))

        # Compress files if option is checked
        if compress_output:
            self.compress_files(output_files, output_format, output_folder, feedback)

        # Load output files into the project
        if self.parameterAsBoolean(parameters, 'load_output', context):
            self.load_output_files(output_files)

        return {self.OUTPUT_FOLDER: output_folder}

    def temp_layer(self, layer, extracted_features, context):
        # Checks if the layer is a QgsProcessingFeatureSource
        if isinstance(layer, QgsProcessingFeatureSource):
            # Gets the source name and the associated layer
            source_name = layer.sourceName()
            source_layer = QgsProject.instance().mapLayersByName(source_name)

            if not source_layer:
                raise QgsProcessingException(self.invalidSourceError({'layer': layer}, 'layer'))

            source_layer = source_layer[0]

        # Use the utility function to create the memory layer
        temp_layer = create_memory_layer(
            'Temporary Layer',
            QgsWkbTypes.displayString(source_layer.wkbType()),
            source_layer.crs().authid(),
            source_layer.fields()
        )

        # Adds the extracted features to the temporary layer
        temp_layer.dataProvider().addFeatures(extracted_features)

        return temp_layer

    def check_duplicates(self, extracted_features, num_tei_field):
        tad_numbers = [f[num_tei_field] for f in extracted_features]
        duplicated_tads = {tad_number: tad_numbers.count(tad_number) for tad_number in set(tad_numbers) if
                           tad_numbers.count(tad_number) > 1}
        if duplicated_tads:
            raise QgsProcessingException(tr(f"Duplicate embargo term numbers found: {list(duplicated_tads.keys())}"))

    def remover_fields(self, temp_layer, num_tei_field, serie_tei_field):
        # Fields that should be kept
        fields_to_keep = [num_tei_field, serie_tei_field]
        provider = temp_layer.dataProvider()
        fields_to_remove = [temp_layer.fields().indexOf(field.name()) for field in temp_layer.fields() if
                            field.name() not in fields_to_keep]
        if fields_to_remove:
            provider.deleteAttributes(fields_to_remove)

        # Update the layer's fields
        temp_layer.updateFields()

    def rename_fields(self, temp_layer, num_tei_field, serie_tei_field):

        # Renaming dictionary
        field_renames = {
            num_tei_field: 'NUM_TEI',
            serie_tei_field: 'SERIE_TEI'
        }
        provider = temp_layer.dataProvider()

        # Rename the fields
        for old_name, new_name in field_renames.items():
            if temp_layer.fields().indexOf(old_name) != -1:
                provider.renameAttributes({temp_layer.fields().indexOf(old_name): new_name})

        # Update the layer's fields
        temp_layer.updateFields()

    def export_single_file(self, layer, output_folder, today_str, output_format, feedback):
        extension = self.get_extension(output_format)
        output_file = os.path.join(output_folder, f"TEI_{today_str}_sicafi.{extension}")

        try:
            # O utilitário agora gerencia feedback.pushInfo e feedback.reportError internamente
            save_as_vector(layer, output_file, feedback)
            return [output_file]
        except Exception:
            return []

    def export_individual_files(self, layer, output_folder, output_format, feedback):
        output_files = []
        driver_name = self.get_output_format_string(output_format)
        extension = self.get_extension(output_format)
        num_tei_index = layer.fields().indexOf('NUM_TEI')
        if num_tei_index == -1:
            raise QgsProcessingException(tr("Field 'NUM_TEI' not found in the layer after renaming."))

        for feature in layer.getFeatures():
            if feedback.isCanceled(): break
            tad_number = feature.attribute(num_tei_index)
            if not tad_number:
                feedback.pushWarning(tr(f"Skipping feature {feature.id()} due to empty 'NUM_TEI' value."))
                continue

            output_file = os.path.join(output_folder, f"TEI_{tad_number}_sicafi.{extension}")

            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = driver_name
            options.fileEncoding = "UTF-8"
            options.filterFids = [feature.id()]

            # Aqui mantemos a lógica manual pois save_as_vector não suporta filterFids nativamente sem modificação
            error = QgsVectorFileWriter.writeAsVectorFormatV3(layer, output_file,
                                                              QgsProject.instance().transformContext(), options)

            if error[0] == QgsVectorFileWriter.NoError:
                output_files.append(output_file)
                feedback.pushInfo(tr(f"Saved file: {output_file}"))
            else:
                feedback.reportError(tr(f"Error saving file for TEI {tad_number}: {error[1]}"))

        feedback.pushInfo(tr(f"Total number of saved files: {len(output_files)}"))
        return output_files

    def load_output_files(self, list_output_files):

        layers_to_add = []
        for output_file in list_output_files:
            layer_name = os.path.basename(output_file)
            layer = QgsVectorLayer(output_file, layer_name, "ogr")
            if not layer.isValid():
                raise QgsProcessingException(f"Failed to load layer: {layer_name}")
            layers_to_add.append(layer)
        QgsProject.instance().addMapLayers(layers_to_add)

    def compress_files(self, output_files, output_format, output_folder, feedback):
        format_info = self.FORMAT_INFO.get(output_format)
        if not format_info: return

        file_extension = format_info['ext']

        for output_file in output_files:
            base_name = os.path.splitext(os.path.basename(output_file))[0]
            zip_output_file = os.path.join(output_folder, f"{base_name}_{file_extension}.zip")

            # O utilitário agora gerencia feedback.pushInfo e feedback.reportError internamente
            associated_files = get_associated_files(output_file)

            if associated_files:
                compress_to_zip(associated_files, zip_output_file, feedback)

    def get_extension(self, output_format):
        return self.FORMAT_INFO.get(output_format, {}).get('ext', 'tmp')

    def get_output_format_string(self, output_format):
        return self.FORMAT_INFO.get(output_format, {}).get('driver', 'unknown')

    def name(self):
        return "emiToolsExportTerms"

    def displayName(self):
        return tr("Export polygons to SICAFI")

    def group(self):
        return tr("Emi Tools")

    def groupId(self):
        return ""

    def shortHelpString(self):
        return tr(
            "This algorithm exports features from a polygon layer to individual or single vector files, compatible with the upload of embargoed areas to the Cadastro, Arrecadação e Fiscalização System (Sicafi). "
            "The algorithm removes unnecessary fields, renames mandatory fields to 'NUM_TEI' and 'SERIE_TEI', and offers the option to compress the output files into a ZIP archive."
        )

    def createInstance(self):
        return emiToolsExportTerms()