# -*- coding: utf-8 -*-

"""
/***************************************************************************
 emiTools
                                 A QGIS plugin
 This plugin compiles tools used by EMI-PB
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-10
        copyright            : (C) 2024 by Alexandre Parente Lima
        email                : alexandre.parente@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandre Parente Lima'
__date__ = '2024-10-10'
__copyright__ = '(C) 2024 by Alexandre Parente Lima'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtCore import QDate
from qgis.core import QgsApplication
from qgis.core import QgsExpression
from qgis.utils import qgsfunction


def tr(string):
    return QCoreApplication.translate('@default', string)


@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def validate_cpf(cpf_number, feature, parent):
    """
        Returns True if the provided string is a valid Brazilian Individual Taxpayer Identification Number (CPF). Otherwise, returns False.

        <h4>Syntax</h4>
        <p><b style="color:#0a6099;">validate_cpf</b> (<i style="color:#bf0c0c;">string</i>)</p>

        <h4>Arguments</h4>
        <p><i style="color:#bf0c0c;">string</i>: A string containing 11 numeric characters representing a CPF number.</p>

        <h4>Example:</h4>
        <ul>
          <li> validate_cpf('000.000.000-00') -> True</li>
          <li> validate_cpf('000.000.000-01') -> False</li>
        </ul>
    """

    # Remove non-numeric characters
    cpf_number = ''.join(filter(str.isdigit, cpf_number))
    
    # Checks if the CPF number has 11 digits
    if len(cpf_number) != 11:
        return False

    # Validates CPF algorithm
    cpf_digits = list(map(int, cpf_number))
    cpf_sum = sum(a * b for a, b in zip(cpf_digits[:9], range(10, 1, -1)))
    cpf_remainder = (cpf_sum * 10) % 11
    if cpf_remainder == 10:
        cpf_remainder = 0
    if cpf_remainder != cpf_digits[9]:
        return False

    cpf_sum = sum(a * b for a, b in zip(cpf_digits[:10], range(11, 1, -1)))
    cpf_remainder = (cpf_sum * 10) % 11
    if cpf_remainder == 10:
        cpf_remainder = 0
    if cpf_remainder != cpf_digits[10]:
        return False

    return True



@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def validate_cnpj(cnpj_number, feature, parent):
    """
    Returns True if the provided string is a valid Brazilian Corporate Taxpayer Identification Number (CNPJ). Otherwise, returns False.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">validate_cnpj</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: A 14-character text string representing a CNPJ number.</p>
  
    <h4>Example:</h4>
    <ul>
      <li> validate_cnpj('00.000.000/0000-00') -> True</li>
      <li> validate_cnpj('00.000.000/0000-01') -> False</li>
    </ul>
    """
    # Remove non-numeric characters
    cnpj_number = ''.join(filter(str.isdigit, cnpj_number))
    
    # Check if the CNPJ number has 14 digits
    if len(cnpj_number) != 14:
        return False
    else:
        sum = 0
        weight = [5,4,3,2,9,8,7,6,5,4,3,2]
        
        # Calculating the first CNPJ check digit
        for n in range(12):
            value = int(cnpj_number[n]) * weight[n]
            sum += value
        verifying_digit = sum % 11
        first_verifying_digit = 0 if verifying_digit < 2 else 11 - verifying_digit
        
        # Calculating the second CNPJ check digit
        sum = 0
        weight = [6,5,4,3,2,9,8,7,6,5,4,3,2]
        for n in range(13):
            sum += int(cnpj_number[n]) * weight[n]
        verifying_digit = sum % 11
        second_verifying_digit = 0 if verifying_digit < 2 else 11 - verifying_digit
        
        # Check if the provided CNPJ is valid
        if cnpj_number[-2:] == f"{first_verifying_digit}{second_verifying_digit}":
            return True
        return False

@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def format_cpf(cpf_number, feature, parent):
    """
    Returns a formatted string for CPF (Brazilian individual taxpayer ID).
    <br><br>Note: This function does not verify the validity of the provided number.
    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">format_cpf</b> (<i style="color:#bf0c0c;">string</i>)</p>
    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: A numeric string containing 11 digits.</p>
    <h4>Example:</h4>
    <ul>
      <li> format_cpf('00000000000') -> 000.000.000-00</li>
      <li> format_cpf('BR000,000,000-00') -> 000.000.000-00</li>
    </ul>
    """

    # Filter out non-digit characters and keep only numeric digits
    cpf_number = ''.join(filter(str.isdigit, cpf_number))

    if len(cpf_number) != 11:
        raise ValueError(tr("Invalid number. Please provide an 11-digit numeric string."))

    # Format the CPF
    formatted = f"{cpf_number[:3]}.{cpf_number[3:6]}.{cpf_number[6:9]}-{cpf_number[9:]}"
    
    return formatted
    
    
@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def format_cnpj(number, feature, parent):
    """
        Returns a formatted string for CNPJ (Brazilian corporate taxpayer ID).
        <br><br>Note: This function does not verify the validity of the provided number.
        <h4>Syntax</h4>
        <p><b style="color:#0a6099;">format_cpf_cnpj</b> (<i style="color:#bf0c0c;">string</i>)</p>
        <h4>Arguments</h4>
        <p><i style="color:#bf0c0c;">string</i>: 11 or 14-numeric character string.</p>
        <h4>Example:</h4>
        <ul>
          <li> format_cnpj('00000000000000') -> 00.000.000/0000-00</li>
          <li> format_cnpj('BR00,000,000/0000-00') -> 00.000.000/0000-00</li>
        </ul>
        """
    
    # format CPF or CNPJ
    number = ''.join(filter(str.isdigit, number))

    if len(number) == 11:
        formatted = f"{number[:3]}.{number[3:6]}.{number[6:9]}-{number[9:]}"
    elif len(number) == 14:
        formatted = f"{number[:2]}.{number[2:5]}.{number[5:8]}/{number[8:12]}-{number[12:]}"
    else:
        raise Exception(tr("Invalid number. Pass a numeric string as the input parameter."))
    
    return formatted


@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def format_cpf_cnpj(number, feature, parent):
    """
        Returns a formatted string for CPF (Brazilian individual taxpayer ID) or CNPJ (Brazilian corporate taxpayer ID).
        <br><br>Note: This function does not verify the validity of the provided number.
        <h4>Syntax</h4>
        <p><b style="color:#0a6099;">format_cpf_cnpj</b> (<i style="color:#bf0c0c;">string</i>)</p>
        <h4>Arguments</h4>
        <p><i style="color:#bf0c0c;">string</i>: 11 or 14-numeric character string.</p>
        <h4>Example:</h4>
        <ul>
          <li> format_cpf_cnpj('00000000000') -> 000.000.000-00</li>
          <li> format_cpf_cnpj('00000000000000') -> 00.000.000/0000-00</li>
          <li> format_cpf_cnpj('000,000,000-00') -> 000.000.000-00</li>
          <li> format_cpf_cnpj('BR00,000,000/0000-00') -> 00.000.000/0000-00</li>
        </ul>
        """
    
    # format CPF or CNPJ
    number = ''.join(filter(str.isdigit, number))

    if len(number) == 11:
        formatted = f"{number[:3]}.{number[3:6]}.{number[6:9]}-{number[9:]}"
    elif len(number) == 14:
        formatted = f"{number[:2]}.{number[2:5]}.{number[5:8]}/{number[8:12]}-{number[12:]}"
    else:
        raise Exception("Invalid number. Pass a numeric string as the input parameter..")
    
    return formatted

@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def mask_cpf(cpf_number, feature, parent):
    """
    Masks a CPF number by hiding the first three and last two digits.
    Non-numeric characters are removed before applying the mask.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">mask_cpf</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: A string containing 11 numeric characters representing a CPF number.</p>
    
    <h4>Example</h4>
    <ul>
      <li>mask_cpf('000.000.000-00') → '***.000.000-**'.</li>
      <li>mask_cpf('000.000.000') → "Invalid CPF" if the provided number does not have exactly 11 digits.</li>
    </ul>
    """
     
    #Remove non-numeric characters
    cpf_number = ''.join(filter(str.isdigit, cpf_number))
    
    #Check if CPF number has 11 digits
    if len(cpf_number) != 11:
        return "Invalid CPF"
    
    #Mask the first three and last two digits
    tarjado = '***.' + cpf_number[3:6] + '.' + cpf_number[6:9] + '-**'
    
    #Display documentation when requested
    return tarjado

@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def mask_name(full_name, feature, parent):
    """
    Masks the middle part of a name by hiding the internal parts, leaving the first and last names visible.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">mask_name</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: A string containing the full name to be masked.</p>
    
    <h4>Returns</h4>
    <ul>
      <li>A string with the masked middle part of the name, e.g. 'Alexandre ******* Lima'.</li>
    </ul>
    """

    # Split the full name into parts
    name_parts = full_name.split()

    # Check if the name has at least 3 parts to mask the middle
    if len(name_parts) < 3:
        return full_name  # Return the name as is if it cannot be masked

    # Mask the middle part (all parts except the first and last)
    masked_middle = ['*' * len(part) for part in name_parts[1:-1]]

    # Construct the masked name
    masked_name = name_parts[0] + ' ' + ' '.join(masked_middle) + ' ' + name_parts[-1]

    return masked_name

@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def format_proper_name(text, feature, parent):
    """
    Formats a proper name following the conventions of Portuguese in Brazil. It capitalizes the first letter of each name, except articles, prepositions, and conjunctions, which remain lowercase unless they are the first word of the name.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">format_proper_name</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: The input proper name to be formatted.</p>

    <h4>Returns</h4>
    <p><i>str</i>: The formatted proper name following the capitalization rules of Portuguese in Brazil.</p>
    
    <h4>Example:</h4>
    <pre>
    format_proper_name('joaquim maria machado de assis') -> 'Joaquim Maria Machado de Assis'
    </pre>
    """
    # Split text into words
    words = text.split()

    # List of words to keep in lowercase (articles, prepositions, and conjunctions)
    lowercase_words = {'da', 'de', 'do', 'das', 'dos', 'e', 'em', 'no', 'na', 'nos', 'nas', 'a', 'o', 'as', 'os', 'por', 'com'}

    # Capitalize the first word
    formatted_text = words[0].capitalize()

    # Format remaining words
    for word in words[1:]:
        if word.lower() not in lowercase_words:
            formatted_text += ' ' + word.capitalize()
        else:
            formatted_text += ' ' + word.lower()

    return formatted_text

@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def format_title_abnt(text, feature, parent):
    """
    Formats the input text as a title according to the guidelines set by the ABNT (Brazilian Association of Technical Standards). It capitalizes all major words, excluding articles, prepositions, and conjunctions, unless they are the first word of the title.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">format_title_abnt</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Arguments</h4>
    <p><i style="color:#bf0c0c;">string</i>: The input text to be formatted.</p>
  
    <h4>Returns</h4>
    <p><i>str</i>: The formatted text following ABNT title formatting.</p>
    
    <h4>Example:</h4>
    <pre>
    format_title_abnt('Qgis: Um Sistema de Informação Geográfica livre e aberto.') -> 'QGIS: Um Sistema de Informação Geográfica Livre e Aberto.'
    </pre>
    """
    # Split text into words
    words = text.split()
    
    # List of words to keep in lowercase (articles, prepositions, and conjunctions)
    lowercase_words = {'a', 'à', 'ao', 'às', 'o', 'os', 'as', 'de', 'do', 'dos', 'das', 'e', 'em', 'no', 'na', 'nos', 'nas', 'com', 'por', 'para'}
    
    # Format first word as uppercase
    formatted_text = words[0].capitalize()
    
    # Format remaining words according to ABNT guidelines
    for word in words[1:]:
        if word.lower() not in lowercase_words:
            formatted_text += ' ' + word.capitalize()
        else:
            formatted_text += ' ' + word.lower()
    
    return formatted_text


@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def get_image_date(name, feature, parent):
    """
    Returns the date of the satellite image based on the provided file name, following the compact naming convention of the filename.
    
    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">get_date_image</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Argument</h4>
    <p><i style="color:#bf0c0c;">String</i>: The file name of the Landsat or Sentinel image.</p>
  
    <h4>Example</h4>
    <p>get_image_date('LC08_L1TP_216065_20210206_20210305_01_T1') -> QDate('2021-02-06')</p>
    """
    if name[0:4]=="LC08":  
        return QDate.fromString( name[17:25], "yyyyMMdd" )
    elif name[0:4]=="LE07":
        return QDate.fromString( name[17:25], "yyyyMMdd" )
    elif name[0:4]=="LT05":
        return QDate.fromString( name[17:25], "yyyyMMdd" )    
    elif name[0:3]=="S2A":
        return QDate.fromString( name[11:19], "yyyyMMdd" )
    elif name[0:3]=="S2B":
        return QDate.fromString( name[11:19], "yyyyMMdd" )
    elif name[0:3]=="T25":
        return QDate.fromString( name[7:15], "yyyyMMdd" )
    elif name[0:2]=="PS":
        return QDate.fromString( name[3:11], "yyyyMMdd" )
    else:
        raise Exception("Invalid string, please enter a valid Sentinel or Landsat name (e.g., 'S2A_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443', 'LC08_L1TP_216065_20210206_20210305_01_T1')")
        
   
@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def get_satellite_name(filename, feature, parent):
    """
    Returns the satellite name based on the provided file name, following the compact naming convention.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">get_satellite_name</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Argument</h4>
    <p><i style="color:#bf0c0c;">string</i>: The file name of the satellite image.</p>
  
    <h4>Example</h4>
    <p>get_satellite_name('LC08_L1TP_216065_20210206_20210305_01_T1') -> 'LandSat 8'</p>
    <p>get_satellite_name('S2A_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443') -> 'Sentinel 2A'</p>
    <p>get_satellite_name('S2B_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443') -> 'Sentinel 2B'</p>
    """
    if filename[0:4]=="LC08":
        return "LandSat 8"
    elif filename[0:4]=="LE07":
        return "LandSat 7"
    elif filename[0:4]=="LT05":
        return "LandSat 5"
    elif filename[0:3]=="S2A":
        return "Sentinel 2A"
    elif filename[0:3]=="S2B":
        return "Sentinel 2B"
    elif filename[0:3]=="S2c":
        return "Sentinel 2C"        
    elif filename[0:3]=="T25":
        return "Sentinel 2"
    elif filename[0:2]=="PS":
        return "PlanetScope"
    else:
        raise Exception("Invalid filename. Please provide a valid filename according to the compact naming convention for Sentinel2 or Landsat8 images (e.g., 'S2A_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443', 'LC08_L1TP_216065_20210206_20210305_01_T1')")
        
@qgsfunction(args='auto', group='EMI Tools', register=True, usesgeometry=False, referenced_columns=[])
def get_image_source(filename, feature, parent):
    """
    Returns the satellite image source based on the file name.

    <h4>Syntax</h4>
    <p><b style="color:#0a6099;">get_source_imagen</b> (<i style="color:#bf0c0c;">string</i>)</p>

    <h4>Argument</h4>
    <p><i style="color:#bf0c0c;">string</i>: The file name of the satellite image.</p>
  
    <h4>Example</h4>
    <p>get_source_imagen('LC08_L1TP_216065_20210206_20210305_01_T1') -> 'United States Geological Survey (USGS).'</p>
    <p>get_source_imagen('S2A_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443') -> 'European Union\'s Earth Observation Programme (COPERNICUS).'</p>
        """
    if filename[0:4]=="LC09":
        return "United States Geological Survey (USGS)."
    elif filename[0:4]=="LE08":
        return "United States Geological Survey (USGS)."
    elif filename[0:4]=="LE07":
        return "United States Geological Survey (USGS)."
    elif filename[0:4]=="LT05":
        return "United States Geological Survey (USGS)."
    elif filename[0:3]=="S2A":
        return "European Union\'s Earth Observation Programme (COPERNICUS)."
    elif filename[0:3]=="S2B":
        return "European Union\'s Earth Observation Programme (COPERNICUS)."
    elif filename[0:3]=="S2C":
        return "European Union\'s Earth Observation Programme (COPERNICUS)."
    elif filename[0:3]=="T25":
        return "European Union\'s Earth Observation Programme (COPERNICUS)."
    elif filename[0:2]=="PS":
        return "Rede MAIS/MJSP, inclui material © (2023) Planet Labs Inc. Todos os direitos reservados."
    else:
        raise Exception("Invalid filename. Please provide a valid filename according to the compact naming convention for Sentinel2 or Landsat8 images (e.g., 'S2A_MSIL1C_20170105T013442_N0204_R031_T53NMJ_20170105T013443', 'LC08_L1TP_216065_20210206_20210305_01_T1')")






