# -*- coding: utf-8 -*-

"""
/***************************************************************************
 emiTools
                                 A QGIS plugin
 This plugin compiles tools used by EMI-PB
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-10
        copyright            : (C) 2024 by Alexandre Parente Lima
        email                : alexandre.parente@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandre Parente Lima'
__date__ = '2024-10-10'
__copyright__ = '(C) 2024 by Alexandre Parente Lima'

from qgis.core import (QgsApplication,
                       QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterVectorDestination,
                       QgsProcessingParameterBoolean,
                       QgsProcessingException,
                       QgsExifTools,
                       QgsCoordinateFormatter,
                       QgsFields,
                       QgsField,
                       QgsFeature,
                       QgsVectorLayer,
                       QgsPointXY,
                       QgsVectorFileWriter,
                       QgsPoint,
                       QgsGeometry,
                       QgsProject,
                       QgsProcessingParameterFile)


from qgis.PyQt.QtCore import (QCoreApplication,
                              QDateTime,
                              QVariant)

import os
from .emi_tools_util import tr


class emiToolsImportGeotaggedPhotos(QgsProcessingAlgorithm):
    INPUT_FOLDER = 'INPUT_FOLDER'
    OUTPUT_FILE = 'OUTPUT_FILE'
    NO_EXTRACT_DJI_XMP = 'NO_EXTRACT_DJI_XMP'


    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_FOLDER,
                tr('Input folder'),
                behavior=QgsProcessingParameterFile.Folder
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT_FILE,
                tr('Output file')
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NO_EXTRACT_DJI_XMP,
                tr('Do not import XMP tags used by DJI'),
                defaultValue=False
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        input_folder = self.parameterAsString(parameters, self.INPUT_FOLDER, context)
        output_file = self.parameterAsOutputLayer(parameters, self.OUTPUT_FILE, context)
        no_extract_dji_xmp = self.parameterAsBool(parameters, self.NO_EXTRACT_DJI_XMP, context)

        feature_exif_dict = {}
        erro_feature_exif_dict = {}

        # Get all image files in the folder
        image_extensions = ('.jpg', '.jpeg', '.tif', '.tiff')
        for root, dirs, files in os.walk(input_folder):
            for file in files:
                if file.lower().endswith(image_extensions):
                    image_path = os.path.join(root, file)

                    try:
                        exif_dict = {
                            'photo': image_path,
                            'filename': os.path.basename(image_path),
                            'directory': os.path.dirname(image_path),
                            **self.get_exif_data(image_path, no_extract_dji_xmp, feedback)
                        }

                        # Adds the dictionary to feature_exif_dict
                        if exif_dict.get('latitude') and exif_dict.get('longitude'):
                            feature_exif_dict[file] = exif_dict
                        else:
                            erro_feature_exif_dict[file] = exif_dict

                    except Exception as e:
                        feedback.reportError(tr(f"Error processing {file}: {str(e)}"))
                        erro_feature_exif_dict[file] = {'error': str(e)}

        #create the point layer
        point_layer = self.create_points_layer(feature_exif_dict)

        #exports the layer to a file
        self.export_output_file(point_layer, output_file)


        feedback.pushInfo(tr(
            f"A total of {len(feature_exif_dict)} images with geotags and {len(erro_feature_exif_dict)} images without geotags were identified."))

        return {self.OUTPUT_FILE: output_file}

    def get_exif_data(self, temp_file_path, no_extract_dji_xmp, feedback):
        """
        Extracts EXIF and XMP metadata from an image file using a single readTags() call.
        """
        exif_tools = QgsExifTools()

        if not exif_tools.hasGeoTag(temp_file_path):
            return {}

        # Uses exif_tools.getGeoTag to abstract EXIF complexity,
        # ensuring safe and standardized geographic coordinate extraction
        geo_tag_result = exif_tools.getGeoTag(temp_file_path)
        exif_geo_tag = geo_tag_result[0]

        exif_latitude = exif_geo_tag.y() if isinstance(exif_geo_tag, QgsPoint) else None
        exif_longitude = exif_geo_tag.x() if isinstance(exif_geo_tag, QgsPoint) else None

        latitude_dms = QgsCoordinateFormatter.formatY(exif_latitude, QgsCoordinateFormatter.FormatDegreesMinutesSeconds,
                                                      2) if exif_latitude else None
        longitude_dms = QgsCoordinateFormatter.formatX(exif_longitude,
                                                       QgsCoordinateFormatter.FormatDegreesMinutesSeconds,
                                                       2) if exif_longitude else None
        exif_coordinates_str = f"{latitude_dms}, {longitude_dms}" if latitude_dms and longitude_dms else None

        # Uses readTags for DJI XMP/other tags
        tags = exif_tools.readTags(temp_file_path)

        def get_float(tag_name):
            val = tags.get(tag_name)
            try:
                return float(val) if val is not None else None
            except (TypeError, ValueError):
                return None

        exif_dict = {
            'altitude': get_float('Exif.GPSInfo.GPSAltitude'),
            'direction': get_float('Exif.GPSInfo.GPSImgDirection'),
            'rotation': None,
            'longitude': exif_longitude,
            'latitude': exif_latitude,
            'timestamp': tags.get('Exif.Photo.DateTimeOriginal') if isinstance(tags.get('Exif.Photo.DateTimeOriginal'),
                                                                               QDateTime) else None,
            'coordinates': exif_coordinates_str,
            'model': tags.get('Exif.Image.Model') or None
        }

        # DJI tags
        xmp_tags = {}
        if not no_extract_dji_xmp:
            dji_fields = [
                'Xmp.drone-dji.CamReverse',
                'Xmp.drone-dji.FlightPitchDegree',
                'Xmp.drone-dji.FlightRollDegree',
                'Xmp.drone-dji.FlightYawDegree',
                'Xmp.drone-dji.GimbalPitchDegree',
                'Xmp.drone-dji.GimbalReverse',
                'Xmp.drone-dji.GimbalRollDegree',
                'Xmp.drone-dji.GimbalYawDegree',
                'Xmp.drone-dji.RelativeAltitude'
            ]
            for tag in dji_fields:
                key = tag.split('.')[-1]
                xmp_tags[key] = get_float(tag)

            exif_dict.update(xmp_tags)

        #Set rotation field
        exif_dict['rotation'] = exif_dict['direction'] or xmp_tags.get(
            'FlightYawDegree') if not no_extract_dji_xmp else None

        return exif_dict


    def create_points_layer(self, feature_exif_dict):
        # Uses the first item as a reference for the field structure
        first_exif = next(iter(feature_exif_dict.values()))
        keys = list(first_exif.keys())

        # Defines the fields based on the value types from the first EXIF entry
        fields = QgsFields()

        #Ensure that 'direction' and 'rotation' are created as float fields.
        for sub_key in keys:
            if sub_key in ['altitude',
                           'direction',
                           'rotation',
                           'CamReverse',
                           'FlightPitchDegree',
                           'FlightRollDegree',
                           'FlightYawDegree',
                           'GimbalPitchDegree',
                           'GimbalReverse',
                           'GimbalRollDegree',
                           'GimbalYawDegree',
                           'RelativeAltitude',
                           ]:
                fields.append(QgsField(sub_key, QVariant.Double))
                continue

        #Dynamic assignment, in case the algorithm supports other tags in the future.
            value = first_exif.get(sub_key)
            if isinstance(value, int):
                fields.append(QgsField(sub_key, QVariant.Int))
            elif isinstance(value, float):
                fields.append(QgsField(sub_key, QVariant.Double))
            elif isinstance(value, QDateTime):
                fields.append(QgsField(sub_key, QVariant.DateTime))
            else:
                fields.append(QgsField(sub_key, QVariant.String))

        #Creates the point layer
        point_layer = QgsVectorLayer("Point?crs=EPSG:4326", "Pontos Imagens", "memory")
        provider = point_layer.dataProvider()
        provider.addAttributes(fields)
        point_layer.updateFields()

        # Adds features based on the values in the same order as the fields
        for exif_data in feature_exif_dict.values():
            exif_latitude = exif_data.get('latitude')
            exif_longitude = exif_data.get('longitude')

            if exif_latitude is not None and exif_longitude is not None:
                point = QgsPointXY(exif_longitude, exif_latitude)
                feature = QgsFeature()
                feature.setGeometry(QgsGeometry.fromPointXY(point))
                feature.setAttributes([exif_data.get(k) for k in keys])
                provider.addFeature(feature)

        point_layer.updateExtents()
        return point_layer

    def export_output_file(self, point_layer, output_file):
        # Extracts the driverName based on the file format in the output_file input
        extension = os.path.splitext(output_file)[1].lstrip('.')
        driver_name = QgsVectorFileWriter.driverForExtension(extension)

        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = driver_name
        options.fileEncoding = 'UTF-8'
        transform_context = QgsProject.instance().transformContext()

        # Saves the file in the specified format
        error = QgsVectorFileWriter.writeAsVectorFormatV3(point_layer, output_file, transform_context, options)
        if error[0] != QgsVectorFileWriter.NoError:
            raise QgsProcessingException(tr(f"Error saving the file: {error[1]}"))
        return

    def name(self):
        return 'emiToolsImportGeotaggedPhotos'

    def displayName(self):
        return tr('Import geotagged photos from DJI drones')

    def group(self):
        return tr("Emi Tools")

    def groupId(self):
        return ""

    def shortHelpString(self):
        return tr(
            "This algorithm generates a point layer based on georeferenced locations (geotags) extracted from JPEG images in a source folder."
            "It supports both standard EXIF metadata and specific tags used by DJI drones."
        )

    def createInstance(self):
        return emiToolsImportGeotaggedPhotos()