# -*- coding: utf-8 -*-

"""
/***************************************************************************
 emiTools
                                 A QGIS plugin
 This plugin compiles tools used by EMI-PB
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-10
        copyright            : (C) 2024 by Alexandre Parente Lima
        email                : alexandre.parente@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandre Parente Lima'
__date__ = '2024-10-10'
__copyright__ = '(C) 2024 by Alexandre Parente Lima'

from qgis.core import (QgsApplication,
                       QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterVectorDestination,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterField,
                       QgsProcessingException,
                       QgsVectorFileWriter,
                       QgsSvgMarkerSymbolLayer,
                       QgsMarkerSymbol,
                       QgsWkbTypes,
                       QgsRuleBasedRenderer,
                       QgsSymbolLayer,
                       QgsProperty,
                       QgsEditorWidgetSetup,
                       QgsLayerDefinition,
                       QgsProject,
                       QgsVectorLayer)

from qgis.PyQt.QtCore import QCoreApplication, QVariant
import os
from .emi_tools_util import tr

class emiToolsApplyStyleGeotaggedPhotos(QgsProcessingAlgorithm):
    INPUT_FILE = 'INPUT_FILE'
    OUTPUT_FILE = 'OUTPUT_FILE'
    CONFIG_MAP_TIPS = 'CONFIG_MAP_TIPS'
    CONFIG_PHOTO_FIELD = 'CONFIG_PHOTO_FIELD'
    PHOTO_FIELD = 'PHOTO_FIELD'
    ROTATION_FIELD = 'ROTATION_FIELD'
    EXPORT_STYLE = 'EXPORT_STYLE'

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterVectorLayer(
            self.INPUT_FILE, tr('Input layer'), [QgsProcessing.TypeVectorPoint]))

        self.addParameter(QgsProcessingParameterField(
            self.PHOTO_FIELD, tr('Field containing photo path'), parentLayerParameterName=self.INPUT_FILE,
            type=QgsProcessingParameterField.String, defaultValue='photo', optional=False))

        self.addParameter(QgsProcessingParameterField(
            self.ROTATION_FIELD, tr('Field containing the camera direction'), parentLayerParameterName=self.INPUT_FILE,
            type=QgsProcessingParameterField.Numeric, defaultValue='rotation', optional=False))

        self.addParameter(QgsProcessingParameterBoolean(
            self.CONFIG_MAP_TIPS, tr('Configure map tips'), defaultValue=False))

        self.addParameter(QgsProcessingParameterBoolean(
            self.CONFIG_PHOTO_FIELD, tr('Configure the form attributes for the photo field'), defaultValue=False))

        self.addParameter(QgsProcessingParameterBoolean(
            self.EXPORT_STYLE, tr('Export the Layer Definition file (QLR)'), defaultValue=False))

        self.addParameter(QgsProcessingParameterVectorDestination(
            self.OUTPUT_FILE, tr('Output file')))

    def processAlgorithm(self, parameters, context, feedback):
        input_layer = self.parameterAsVectorLayer(parameters, self.INPUT_FILE, context)
        photo_field = self.parameterAsString(parameters, self.PHOTO_FIELD, context)
        rotation_field = self.parameterAsString(parameters, self.ROTATION_FIELD, context)
        config_map_tips = self.parameterAsBool(parameters, self.CONFIG_MAP_TIPS, context)
        config_photo_field = self.parameterAsBool(parameters, self.CONFIG_PHOTO_FIELD, context)
        export_style = self.parameterAsBool(parameters, self.EXPORT_STYLE, context)
        output_file = self.parameterAsOutputLayer(parameters, self.OUTPUT_FILE, context)

        if not input_layer:
            raise QgsProcessingException(tr("Input layer is not valid."))

        exported_layer = self.export_output_file(input_layer, output_file)

        self.apply_symbology(exported_layer, rotation_field)

        if config_map_tips:
            self.configure_map_tips(exported_layer, photo_field)

        if config_photo_field:
            self.configure_photo_field(exported_layer, photo_field)

        if export_style:
            qlr_path = self.export_definition_file(output_file, exported_layer)
            feedback.pushInfo(tr(f"Layer definition file exported to: {qlr_path}"))

        return {self.OUTPUT_FILE: exported_layer}

    def export_output_file(self, input_layer, output_file):

        # Define the driver based on the file extension.
        driver_name = QgsVectorFileWriter.driverForExtension(os.path.splitext(output_file)[1][1:])
        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = driver_name
        options.fileEncoding = 'UTF-8'

        transform_context = QgsProject.instance().transformContext()

        # Export file
        error = QgsVectorFileWriter.writeAsVectorFormatV3(input_layer, output_file, transform_context, options)
        if error[0] != QgsVectorFileWriter.NoError:
            raise QgsProcessingException(tr(f"Error saving the file: {error[1]}"))

        # Reloads the layer from the exported file.
        exported_layer = QgsVectorLayer(output_file, os.path.basename(output_file), 'ogr')
        if not exported_layer.isValid():
            raise QgsProcessingException(tr("Error loading the exported layer."))

        # Adds to the project without displaying it automatically.
        QgsProject.instance().addMapLayer(exported_layer, False)
        QgsProject.instance().layerTreeRoot().addLayer(exported_layer)

        return exported_layer

    def apply_symbology(self, exported_layer, rotation_field):
        svg_path_arrow = os.path.join(QgsApplication.svgPaths()[0], 'arrows', 'Arrow_01.svg')
        svg_path_point = os.path.join(QgsApplication.svgPaths()[0], 'gpsicons', 'city_medium.svg')

        symbol_arrow = QgsMarkerSymbol.defaultSymbol(QgsWkbTypes.PointGeometry)
        svg_marker_arrow = QgsSvgMarkerSymbolLayer(svg_path_arrow)
        svg_marker_arrow.setDataDefinedProperty(QgsSymbolLayer.PropertyAngle, QgsProperty.fromField(rotation_field))
        symbol_arrow.changeSymbolLayer(0, svg_marker_arrow)

        symbol_point = QgsMarkerSymbol.defaultSymbol(QgsWkbTypes.PointGeometry)
        svg_marker_point = QgsSvgMarkerSymbolLayer(svg_path_point)
        symbol_point.changeSymbolLayer(0, svg_marker_point)

        root_rule = QgsRuleBasedRenderer.Rule(None)

        rule1 = QgsRuleBasedRenderer.Rule(symbol_point)
        rule1.setLabel(tr("Photos without direction"))
        rule1.setFilterExpression('"rotation" IS NULL OR "rotation" = 0')
        root_rule.appendChild(rule1)

        rule2 = QgsRuleBasedRenderer.Rule(symbol_arrow)
        rule2.setLabel(tr("Photos with direction"))
        rule2.setIsElse(True)
        root_rule.appendChild(rule2)

        renderer = QgsRuleBasedRenderer(root_rule)
        exported_layer.setRenderer(renderer)

        return exported_layer

    def configure_map_tips(self, exported_layer, photo_field):
        expression = f"""
            <table>
                <tr>
                    <th>[%"filename"%]</th>
                </tr> 
                <tr>
                <th><img src="file:///[%"{photo_field}"%]" width="350" height="250"></th>
                </tr>
            </table>
        """
        exported_layer.setMapTipTemplate(expression)

    def configure_photo_field(self, exported_layer, photo_field):
        fields = exported_layer.fields()
        if photo_field not in [f.name() for f in fields]:
            raise QgsProcessingException(tr(f"Photo field '{photo_field}' not found in layer."))

        field_idx = fields.indexOf(photo_field)
        config = {
            'DocumentViewer': 1,
            'FileWidget': True,
            'FullUrl': True,
            'UseLink': True
        }
        widget_type = 'ExternalResource'
        widget_setup = QgsEditorWidgetSetup(widget_type, config)
        exported_layer.setEditorWidgetSetup(field_idx, widget_setup)

    def export_definition_file(self, output_file, exported_layer):
        base_path, _ = os.path.splitext(output_file)
        qlr_file_path = f"{base_path}.qlr"

        layer_tree_node = QgsProject.instance().layerTreeRoot().findLayer(exported_layer.id())
        if not layer_tree_node:
            raise QgsProcessingException(tr(f"Layer node not found in the layer tree (ID: {exported_layer.id()})."))

        try:
            QgsLayerDefinition.exportLayerDefinition(qlr_file_path, [layer_tree_node])
        except Exception as e:
            raise QgsProcessingException(tr(f"Error exporting the QLR file: {str(e)}"))

        return qlr_file_path

    def name(self):
        return 'emiToolsApplyStyleGeotaggedPhotos'

    def displayName(self):
        return tr('Apply style to geotagged photo layer')

    def group(self):
        return tr("Emi Tools")

    def groupId(self):
        return ""

    def shortHelpString(self):
        return tr(
            "This algorithm applies custom symbology to a point layer with geotagged photos, distinguishing photos with or without recorded direction. "
            "Also allows configuring map tips with image previews, adjusting the photo field to display as an external resource, "
            "and exporting a Layer Definition file (QLR)."
        )

    def createInstance(self):
        return emiToolsApplyStyleGeotaggedPhotos()