# -*- coding: utf-8 -*-
"""
/***************************************************************************
 EDEInterpolationDialog
								 A QGIS plugin
 Evidence Density Estimation (EDE) interpolation of archaeological settlement data.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
							 -------------------
		begin				 : 2019-04-10
		git sha				 : $Format:%H$
		copyright			 : (C) 2019 by Peter Demján
		email				 : peter.demjan@gmail.com
 ***************************************************************************/

/***************************************************************************
 *																		   *
 *	 This program is free software; you can redistribute it and/or modify  *
 *	 it under the terms of the GNU General Public License as published by  *
 *	 the Free Software Foundation; either version 2 of the License, or	   *
 *	 (at your option) any later version.								   *
 *																		   *
 ***************************************************************************/
"""

import os
import tempfile

from PyQt5 import uic
from PyQt5 import QtWidgets

from qgis.gui import QgsFileWidget
from qgis.core import QgsFieldProxyModel, QgsMapLayerProxyModel
from qgis.utils import iface

from .ede_interpolation_process import EDEInterpolationProcess

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'ede_interpolation_dialog_base.ui'))

class EDEInterpolationDialog(QtWidgets.QDialog, FORM_CLASS):
	
	def __init__(self, parent=None):
		
		super(EDEInterpolationDialog, self).__init__(parent)
				
		self.setupUi(self)
		
		self.combo_input_layer.setFilters(QgsMapLayerProxyModel.PointLayer)
		
		self.file_summed.setStorageMode(QgsFileWidget.SaveFile)
		self.path_layers.setStorageMode(QgsFileWidget.GetDirectory)
		self.path_layers.setFilePath(self.get_temp_dir())
		self.path_layers.fileChanged.connect(self.on_path_layers_changed)
		
		for combo in [self.combo_sp_accuracy, self.combo_dating_mean, self.combo_dating_uncert]:
			combo.setFilters(QgsFieldProxyModel.Numeric)
			combo.setFilters(QgsFieldProxyModel.Numeric)
			combo.setFilters(QgsFieldProxyModel.Numeric)
		self.combo_dating_type.setFilters(QgsFieldProxyModel.String)
		for combo in [self.combo_sp_accuracy, self.combo_dating_mean, self.combo_dating_uncert, self.combo_dating_type]:
			combo.fieldChanged.connect(self.on_field_changed)
		for edit in [self.edit_duration, self.edit_diameter, self.edit_time_step, self.edit_time_from, self.edit_time_to, self.edit_cell_size]:
			edit.textEdited.connect(self.on_text_edited)
		self.combo_dating_fixed.currentIndexChanged.connect(self.on_field_changed)
		
		self.combo_input_layer.layerChanged.connect(self.on_input_layer_changed)
		self.button_box.accepted.connect(self.on_accept)
		
		self.update_fields()
		
		self.validate()
	
	def get_temp_dir(self):
		
		tempdir = os.path.normpath(os.path.abspath(os.path.join(tempfile.gettempdir(), "ede_interpolation")))
		if not os.path.exists(tempdir):
			os.makedirs(tempdir)
		return tempdir
	
	def update_fields(self):
		
		layer = self.combo_input_layer.currentLayer()		
		if layer is not None:
			self.combo_sp_accuracy.setLayer(layer)
			self.combo_dating_mean.setLayer(layer)
			self.combo_dating_uncert.setLayer(layer)
			self.combo_dating_type.setLayer(layer)
	
	def validate(self):
		
		def check_number(value):
			
			try:
				float(value)
				return True
			except:
				pass
			return False
		
		def does_validate():
			
			if not self.path_layers.filePath():
				return False
			for combo in [self.combo_sp_accuracy, self.combo_dating_mean, self.combo_dating_uncert]:
				if not combo.currentField():
					return False
			if not ((self.combo_dating_fixed.currentText() != "not fixed") or self.combo_dating_type.currentField()):
				return False
			for edit in [self.edit_duration, self.edit_diameter, self.edit_time_step, self.edit_cell_size]:
				if not check_number(edit.text()):
					return False
			for edit in [self.edit_time_from, self.edit_time_to]:
				value = edit.text()
				if value and not check_number(value):
					return False
			return True
		
		self.button_box.button(QtWidgets.QDialogButtonBox.Ok).setEnabled(does_validate())
	
	def submit(self):
		
		site_duration = float(self.edit_duration.text())
		site_diameter = float(self.edit_diameter.text())
		time_step = float(self.edit_time_step.text())
		cell_size = float(self.edit_cell_size.text())
		
		path_layers = self.path_layers.filePath()
		path_summed = self.file_summed.filePath()
		
		field_spatial_accur = self.combo_sp_accuracy.currentField()
		field_dating = self.combo_dating_mean.currentField()
		field_dating_uncert = self.combo_dating_uncert.currentField()
		field_dating_type = self.combo_dating_type.currentField()
		
		dating_type_fixed = self.combo_dating_fixed.currentText()
		if dating_type_fixed == "not fixed":
			dating_type_fixed = None
		
		time_from = self.edit_time_from.text()
		if time_from:
			time_from = float(time_from)
		else:
			time_from = None
		time_to = self.edit_time_to.text()
		if time_to:
			time_to = float(time_to)
		else:
			time_to = None
		
		data = dict(
			upd = dict(
				dating = [],
				uncert = [],
				coords = [],
				accur = [],
			),
			npd = dict(
				dating = [],
				uncert = [],
				coords = [],
				accur = [],
			)
		)
		found_data = False
		layer = self.combo_input_layer.currentLayer()
		for feature in layer.getFeatures():
			if dating_type_fixed:
				typ = dating_type_fixed.lower()
			else:
				typ = feature[field_dating_type].lower()
				if typ not in ["upd", "npd"]:
					continue
			if not feature.hasGeometry():
				continue
			dating = feature[field_dating]
			uncert = feature[field_dating_uncert]
			accur = feature[field_spatial_accur]
			coords = feature.geometry().asPoint()
			if (not dating) or (not uncert) or (not accur):
				continue
			data[typ]["dating"].append(dating)
			data[typ]["uncert"].append(uncert)
			data[typ]["accur"].append(accur)
			data[typ]["coords"].append([coords.x(), coords.y()])
			found_data = True
		
		if found_data:
			
			colors = [button.color() for button in [self.color_button_1, self.color_button_2, self.color_button_3]]
			
			EDEInterpolationProcess(data, site_duration, site_diameter, time_step, time_from, time_to, cell_size, self.check_approximate.isChecked(), path_layers, path_summed, layer.crs(), colors)
	
	def on_input_layer_changed(self, layer):
		
		self.update_fields()
	
	def on_field_changed(self, field):
		
		self.validate()
		self.combo_dating_type.setEnabled(self.combo_dating_fixed.currentText() == "not fixed")
	
	def on_text_edited(self, text):
		
		self.validate()
	
	def on_path_layers_changed(self, path):
		
		self.validate()
	
	def on_accept(self):
		
		self.submit()

