# -*- coding: utf-8 -*-
"""
/***************************************************************************
Ecosystem Respiration Tool - a QGIS plugin 
                               
 This plugin calculates ecosystem respiration from thermal images.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-02-06
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Florian Ellsäßer
        email                : el-flori@gmx.de
        
More information: https://github.com/FloEll/Ecosystem_Respiration_Tool

This is still an experimental version and I don't claim that it is perfect 
yet. If you spot bugs or problems with the method, please contact me! I'm 
always happy to improve this software. 

This software is based on two scientific publications: 
    
    Heskel et al. (2016) - Convergence in temperature response of leaf 
    respiration across biomes and plant functional types, Proceedings of the 
    National Academy of Sciences, 13 (14) 3832-3837; DOI: 10.1073/pnas.1520282113 
    
If you want to cite this Plugin, please refer to: 
    
    F. Ellsäßer (2021) - Ecosystem Respiration Tool [...]

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import libraries
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, Qgis, QgsRasterLayer

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .ecores_dialog import EcoResDialog
import os.path
# Import Libraries for the Ecosystem Respiration Tool 
import gdal
import numpy as np

class EcoRes:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'EcoRes_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&EcoRes')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('EcoRes', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/ecores/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'This plugin calculates Ecosystem respiration from thermal images.'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&EcoRes'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def select_input_file(self):
        '''Opens a file browser and populates the output_name lineEdit widget
        with the file path and name the user chose'''
        file_name, _filter = QFileDialog.getOpenFileName(
                self.dlg, 'Select input raster name','','*.tif')
        base_name = file_name
        self.dlg.input_name.setText(file_name)
        in_raster = QgsRasterLayer(file_name, base_name)
        QgsProject.instance().addMapLayer(in_raster)
        
    def select_output_raster_file(self):
        '''Opens a file browser and populates the output_name lineEdit widget
        with the file path and name the user chose'''
        filename, _filter = QFileDialog.getSaveFileName(
                self.dlg, 'Select output raster name','','*.tif')
        self.dlg.output_raster_name.setText(filename)
            
    def select_output_file(self):
        '''Opens a file browser and populates the output_name lineEdit widget
        with the file path and name the user chose'''
        filename, _filter = QFileDialog.getSaveFileName(
                self.dlg, 'Select output file name','','*.csv')
        self.dlg.output_name.setText(filename) 
        
    def read_lst_img(self,in_file, na_val=None):
        '''This function reads the thermal image and extracts land surface 
        temperature (lst) projection (prj) and georeference data (geo).
        :in_file : path and file name
        :na_val : float that specifies NaN values
        '''     
        # open the new ratser
        new_raster = gdal.Open(in_file,gdal.GA_ReadOnly)
        # get the (only) band as input
        self.dlg.lst = new_raster.GetRasterBand(1).ReadAsArray()
        self.dlg.na_val = None
        
        # set all zeros to NaN
        self.get_nan_value()
        self.dlg.lst[self.dlg.lst == 0.0] = None
        self.dlg.lst[self.dlg.lst == self.dlg.na_val] = None
        # set all zeros to NaN
        self.dlg.lst[self.dlg.lst == 0.0] = self.dlg.na_val
        # if input is in Kelvin, convert to °C
        if np.mean(self.dlg.lst[~np.isnan(self.dlg.lst)]) > 180.0:
            self.dlg.lst = self.dlg.lst - 273.15
        else:
            pass
        
        # define the projection prj and the georeference data (geo)
        self.dlg.prj = new_raster.GetProjection()
        self.dlg.geo = new_raster.GetGeoTransform()
        # get longitude and latitude of geo
        self.dlg.lon = float(self.dlg.geo[0])
        self.dlg.lat = float(self.dlg.geo[3])
        
        # create an empty variable to receive the results
        self.dlg.ecores = None 
        
    # get NaN values from input field
    def get_nan_value(self):
        if self.dlg.na_val == 'None':
            self.dlg.na_val = None
        else:
            try:
                self.dlg.na_val = float(self.dlg.nan_input.text())
            except:
                self.dlg.na_val = None
                
    def get_model_parameters(self):
        '''This function reads the input of the gui and assigns it to the parameters'''
        # https://gis.stackexchange.com/questions/310011/qgis-qcombobox-list-controls-form-input
        
        #input
        self.read_lst_img(self.in_file) # read the lst image
        # fill the a, b and c parameters
        self.dlg.a = float(self.dlg.a_input.text())
        self.dlg.b = float(self.dlg.b_input.text())
        self.dlg.c = float(self.dlg.c_input.text())
        
        # check if the output values will be inverted
        if self.dlg.invert_output_checkBox.isChecked():
            self.invert_output = True
        else:
            self.invert_output = False
            
        # check if the output values will be the exponential of ln R
        if self.dlg.create_exponential_output_checkBox.isChecked():
            self.create_exponential_output = True
        else:
            self.create_exponential_output = False
                   
    def calculate_ecoRes(self):
        '''This function calculates the ecosystem respiration after 
        Heskel et al. (2016) eq. 1
        '''
        # set all Nones to zero so they are avoided in the calcualtion 
        #self.dlg.lst[self.dlg.lst == None] = 0.0
                # ecores = a + b*lst + c*lst² -> a is multiplied and divided through 
            # lst to set all the None values to None. Without that all None 
            # values would be set to a 
        self.dlg.ecores = np.log(self.dlg.a + 
                                     (self.dlg.lst * self.dlg.b) + 
                                     ((self.dlg.lst)**2 * self.dlg.c))
            # now reset the zeros to None again 
            #self.dlg.lst[self.dlg.lst == 0.0] = self.dlg.na_val
            #self.dlg.ecores[self.dlg.lst == 0.0] = self.dlg.na_val
            
        
        
        # check if we should invert/reverse the output 
        if self.invert_output == True:
            # get the mean first 
            mean_res = np.mean(self.dlg.ecores[~np.isnan(self.dlg.ecores)])
            # subtract the mean
            mean_subtracted = self.dlg.ecores - mean_res  
            # now subtract the subtractions again from the mean again
            self.dlg.ecores = mean_res - mean_subtracted
        else:
            pass
        
        # check if we should invert/reverse the output 
        if self.create_exponential_output == True:
            # just calculate the exponential of the output
            self.dlg.ecores_exp = np.exp(self.dlg.ecores)
        else:
            pass
        
        
        
        
    def write_output_images(self):
        '''This function writes the output data into a GeoTIFF'''
        
        # get the shape (rows and columns) of the lst data
        rows,cols=np.shape(self.dlg.lst)
        driver = gdal.GetDriverByName('GTiff')
        # define the number of output bands
        nbands=1
        # create an empty output raster
        out_raster = driver.Create(self.dlg.output_raster_name.text(), cols, 
                                   rows, nbands, gdal.GDT_Float32)
        # set the parameters of the output raster (geo and prj)
        out_raster.SetGeoTransform(self.dlg.geo)
        out_raster.SetProjection(self.dlg.prj)
        # Write ecores to band 1
        band_1=out_raster.GetRasterBand(1)
        band_1.SetNoDataValue(0)
        # check if we should make the output exponential
        if self.create_exponential_output == True:
            # just calculate the exponential of the output
            band_1.WriteArray(self.dlg.ecores_exp)
        else:
            band_1.WriteArray(self.dlg.ecores)
        band_1.FlushCache()
                                
        # Flush Cache
        out_raster.FlushCache()
        del out_raster
        
        # load layer into qgis
        qgis_raster = QgsRasterLayer(self.dlg.output_raster_name.text(), 'Ecosystem Respiration Output')
        QgsProject.instance().addMapLayer(qgis_raster)
        
    def write_stats(self,file,flux_name,flux):
        '''This function writes the output stats min, max and mean of the input
        and output raster'''
        file.write(flux_name + ',' + str(np.mean(flux[~np.isnan(flux)])) + ',' +
                              str(np.min(flux[~np.isnan(flux)])) + ',' +
                              str(np.max(flux[~np.isnan(flux)])) + '\n'
                              )
        
    def write_output_stats(self):
        '''This function opens an output .csv file and fills it with stats'''
        # write the output data in a .csv file
        with open(self.dlg.output_name.text(), 'w') as output_file:
            # write an out file with the most important stats
            # model parameters
            output_file.write('Model parameters:' + '\n')
            output_file.write('coefficient a: ' + ',' + str(self.dlg.a) + '\n')
            output_file.write('coefficient b: ' + ',' + str(self.dlg.b) + '\n')
            output_file.write('coefficient c: ' + ',' + str(self.dlg.c) + '\n')
            output_file.write('temp mean [°C]: ' + ',' + str(np.mean(self.dlg.lst[~np.isnan(self.dlg.lst)])) + '\n')
            #output_file.write('5% quantile: ' + str(np.quantile(self.dlg.lst[~np.isnan(self.dlg.lst)],0.05)) + '\n')
            #output_file.write('95% quantile: ' + str(np.quantile(self.dlg.lst[~np.isnan(self.dlg.lst)],0.95)) + '\n')
            output_file.write('Ecosystem respiration mean: ' + str(np.mean(self.dlg.ecores[~np.isnan(self.dlg.ecores)])) + '\n')
            #output_file.write('5% quantile: ' + str(np.quantile(self.dlg.ecores[~np.isnan(self.dlg.ecores)],0.05)) + '\n')
            #output_file.write('95% quantile: ' + str(np.quantile(self.dlg.ecores[~np.isnan(self.dlg.ecores)],0.95)) + '\n')
            output_file.write('type ' +  ',' + 'mean ' +  ',' + 'min ' +  ',' + 'max ' + '\n')
            self.write_stats(output_file,'Temperature [°C]',self.dlg.lst)
            self.write_stats(output_file,'Ecosystem respiration [µmol/m²/s]',self.dlg.ecores)
    
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            # start the dialog
            self.dlg = EcoResDialog()
            # check the input files
            self.dlg.search_input_button.clicked.connect(self.select_input_file)
            self.dlg.search_output_raster_button.clicked.connect(self.select_output_raster_file)
            self.dlg.search_output_button.clicked.connect(self.select_output_file)
                   
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # define in raster
            self.in_file = self.dlg.input_name.text()
            # get the model parameters from the GUI
            self.get_model_parameters()
            # calculate the Ecosystem respiration
            self.calculate_ecoRes()
            # write results to a .tif file and load it into qgis
            self.write_output_images()
            # write output stats file 
            self.write_output_stats()
            # Display a push message that EcoRes was successful
            self.iface.messageBar().pushMessage(
                    'Success', 'The Ecosystem Respiration was calculated successfully!',
                    level=Qgis.Success, duration=3) 
            
            
