# -*- coding: utf-8 -*-

"""
EconoMe - Connecting your QGIS-Project with EconoMe.
Version 1.3.1

Copyright (C) 2025 Federal Office for the Environment FOEN, Switzerland

Authors
  - Kevin Helzel (WSL Institute for Snow and Avalanche Research SLF, Davos, Switzerland)
  - Tobia Lezuo (Geotest AG, Davos, Switzerland)

Email
  - Plugin & Code: kevin.helzel@slf.ch
  - EconoMe project leader: linda.ettlin@slf.ch

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.

Generated by QGIS Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
"""

# QGIS Main Imports
from qgis.PyQt.QtCore import Qt, QCoreApplication, QEvent, QObject, QSettings, QSize, QSizeF, QThread, QTranslator, QVariant, pyqtSignal
from qgis.PyQt.QtGui import QColor, QCursor, QFont, QIcon, QMovie, QPalette, QPixmap
from qgis.PyQt.QtWidgets import QAction, QVBoxLayout, QComboBox, QCompleter, QDialog, QDialogButtonBox, QFileDialog, QFormLayout, QFrame, QHBoxLayout, QLabel, QLineEdit, QListView, QListWidget, QListWidgetItem, QPushButton, QScrollArea, QSizePolicy, QSpacerItem, QWidget
from qgis.core import QgsApplication, QgsCoordinateTransformContext, QgsExpressionContextUtils, QgsFeatureRequest, QgsField, QgsFields, QgsLineSymbol, QgsPalLayerSettings, QgsSingleSymbolRenderer, QgsVectorLayerSimpleLabeling, QgsTextBackgroundSettings, Qgis, QgsGraduatedSymbolRenderer, QgsMapLayerProxyModel, QgsProject, QgsRendererRange, QgsSymbol, QgsVectorFileWriter, QgsVectorLayer, NULL, QgsAuthMethodConfig
from qgis.gui import QgsMapLayerComboBox
# QGIS Processing Imports
import processing
from processing.tools import dataobjects

# Custom imports
import os.path
import sys
import time
from datetime import datetime
import json
import requests
from requests.auth import HTTPBasicAuth
from concurrent.futures import ThreadPoolExecutor
from collections import Counter
import copy
import sip
import textwrap

# Import dialogs
from .econo_me_dialog import EconoMeDialog
from .econo_me_dialog_settings_menu import EconoMeDialogSettingsMenu
from .econo_me_dialog_status import StatusDialog
from .econo_me_dialog_asset_handling import DamagePotentialMismatchResolveDialog
from .logger import log


class WorkerThread(QThread):
    """Executes a function in a separate thread."""

    setup_UI = pyqtSignal(str)
    show_status = pyqtSignal(str, str, str)  # Signal emitted when status dialog should be opened
    revalidate_econome_user_info = pyqtSignal()  # Signal emitted when user info should be revalidated
    finished = pyqtSignal()  # Signal emitted when task is done

    def __init__(self, parent_dialog, task_function, should_revalidate=False):
        """Initialize worker thread."""

        super().__init__()
        self.parent_dialog = parent_dialog
        self.task_function = task_function
        self.should_revalidate = should_revalidate

    def run(self):
        """Execute the function in a safe thread."""

        # Execute task
        self.task_function(self)
        # Revalidate user info as needed
        if self.should_revalidate:
            self.revalidate_econome_user_info.emit()
        # Task is done
        self.finished.emit()


class AreYouSure(QDialog):
    """Custom dialog for user yes/no input."""

    def __init__(self, message, pipeline=None, parent=None):

        # Initialize class
        super().__init__()
        self.message = message
        self.pipeline = pipeline
        self.parent = parent

        self.button_action = None

        # UI tweaks
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                    QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                    QLabel { color: black; }
                    QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                    QPushButton:hover { background-color: #D0D0D0; }
                    QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                """)  # Force light mode when dark mode is detected

        # Dialog header
        title = self.tr("EconoMe") + " - " + self.tr("Plugin Status")
        self.setWindowTitle(title)
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))

        # Main layout
        main_layout = QVBoxLayout()
        # Add a label
        label_layout = QHBoxLayout()
        label = QLabel(message)
        label.setAlignment(Qt.AlignHCenter)
        label.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        label_layout.addSpacerItem(QSpacerItem(10, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        label_layout.addWidget(label)
        label_layout.addSpacerItem(QSpacerItem(10, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))

        # Add "YES" and "NO" or customized buttons
        button_layout = QHBoxLayout()
        if self.pipeline is "upload_consequence_analysis":
            save_button = QPushButton(self.tr("SPEICHERN / IN DIE KARTE LADEN"))
            upload_only_button = QPushButton(self.tr("NUR HOCHLADEN"))
            save_button.setCursor(QCursor(Qt.PointingHandCursor))
            upload_only_button.setCursor(QCursor(Qt.PointingHandCursor))
            save_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
            upload_only_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
            button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
            button_layout.addWidget(save_button)
            button_layout.addWidget(upload_only_button)
            button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
            # Activate buttons
            save_button.clicked.connect(self.first_action)
            upload_only_button.clicked.connect(self.second_action)
        else:
            delete_button = QPushButton(self.tr("LÖSCHEN"))
            cancel_button = QPushButton(self.tr("ABBRECHEN"))
            delete_button.setCursor(QCursor(Qt.PointingHandCursor))
            cancel_button.setCursor(QCursor(Qt.PointingHandCursor))
            delete_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
            cancel_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
            button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
            button_layout.addWidget(delete_button)
            button_layout.addWidget(cancel_button)
            button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
            # Activate buttons
            delete_button.clicked.connect(self.accept)
            cancel_button.clicked.connect(self.reject)

        # Add horizontal layouts to main layout
        main_layout.addLayout(label_layout)
        main_layout.addLayout(button_layout)

        # Set main layout
        self.setLayout(main_layout)

    def first_action(self):
        self.button_action = "First"
        self.accept()

    def second_action(self):
        self.button_action = "Second"
        self.accept()


class ParameterProfileDialog(QDialog):
    """Custom dialog for setting the parameter profiles for the damage potential."""

    def __init__(self, fields, mandatory_fields, language, glossary, parent=None):
        super().__init__(parent)
        # Instance variables
        self.fields = fields
        self.mandatory_fields = mandatory_fields
        self.language = language
        self.glossary = glossary
        # UI tweaks
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        self.resize(450, 550)
        self.init_ui()

    def init_ui(self):

        # Create layout
        main_layout = QVBoxLayout()
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_widget = QWidget()
        form_layout = QFormLayout(scroll_widget)

        # Get current plugin language as an English word (default to English)
        if self.language == "DE":
            language = "German"
        elif self.language == "EN":
            language = "English"
        elif self.language == "FR":
            language = "French"
        elif self.language == "IT":
            language = "Italian"
        else:
            language = "English"

        # Add label + empty combobox pairs
        for field_name in self.fields:
            # LABEL
            label_name = next((term[language] for term in self.glossary if term["APITerm"] == field_name), None)  # Retrieve label name from glossary
            tooltip = next((term[language + '-Text'] for term in self.glossary if term["APITerm"] == field_name), None)   # Retrieve tooltip from glossary
            if len(tooltip) > 50:
                tooltip = textwrap.fill(tooltip, width=50, break_long_words=False, replace_whitespace=False)
            label = None
            if field_name in self.mandatory_fields:
                label = QLabel(label_name + " <span style='color: red'>*</span>")  # Create the label
            else:
                label = QLabel(label_name)  # Create the label
            label.setToolTip(tooltip)
            label.setObjectName(f"LABEL_{field_name}")  # Set label object name
            # COMBO BOX
            combo = QComboBox()  # Create combo box
            combo.setObjectName(f"COMBO_BOX_{field_name}")  # Set combo box object name
            # Add LABEL and COMBO BOX to layout
            form_layout.addRow(label, combo)

        scroll_area.setWidget(scroll_widget)
        main_layout.addWidget(scroll_area)

        # OK/Cancel buttons
        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        ok_button = buttons.button(QDialogButtonBox.Ok)
        cancel_button = buttons.button(QDialogButtonBox.Cancel)

        if self.language == "DE":
            ok_button.setText("OK")
            cancel_button.setText("ABBRECHEN")
        elif self.language == "EN":
            ok_button.setText("OK")
            cancel_button.setText("CANCEL")
        elif self.language == "FR":
            ok_button.setText("OK")
            cancel_button.setText("ANNULER")
        elif self.language == "IT":
            ok_button.setText("OK")
            cancel_button.setText("ANNULA")

        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)

        # Add buttons to main layout
        main_layout.addWidget(buttons)

        # Custom resizing based on number of fields
        height = 22 * len(self.fields) + 145
        self.resize(450, height)

        # Apply layout
        self.setLayout(main_layout)


class SaveLoadDialog(QDialog):
    """Custom dialog for saving and loading layers into the map."""

    def __init__(self, layers, pipeline):

        # Initialize class
        super().__init__()
        self.layers = layers
        self.pipeline = pipeline
        self.selected_dam_pot_layers = [layer_tuple[2] for layer_tuple in layers]

        # UI tweaks
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                            QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                            QLabel { color: black; }
                            QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                            QPushButton:hover { background-color: #D0D0D0; }
                            QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                        """)  # Force light mode when dark mode is detected

        # General window settings
        title = self.tr("EconoMe") + " - " + self.tr("Konsequenzanalyse: Output-Layer") if self.pipeline == "upload_consequence_analysis" else self.tr("EconoMe") + " - " + self.tr("Risiko: Output-Layer")
        self.setWindowTitle(title)
        height_adjust_variable = len(set(self.selected_dam_pot_layers))  # Workaround; improve next time
        if pipeline == "upload_consequence_analysis":
            if len(set(self.selected_dam_pot_layers)) <= 1:
                window_height = 350 - (35 * (5 - height_adjust_variable)) + 35
                if sys.platform == "darwin":
                    window_height += 5
            else:
                window_height = 350 - (35 * (5 - height_adjust_variable))
                if sys.platform == "darwin":
                    window_height += (15 * height_adjust_variable) + 5
            self.setFixedSize(400, int(window_height))
        else:
            self.setFixedSize(400, 180)
        layout = QVBoxLayout()

        # Create checkmark list for intersection layers
        if pipeline == "upload_consequence_analysis":
            # Create info label
            self.label_select_dam_pot = QLabel(self.tr("Schadenpotential wählen: "))
            self.label_select_dam_pot.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
            layout.addWidget(self.label_select_dam_pot, alignment=Qt.AlignHCenter)
            # Create list widget
            self.listWidget = QListWidget()
            # Create and style list widget
            self.listWidget.setFixedWidth(220)
            self.listWidget.setSpacing(2)
            self.listWidget.setStyleSheet("""
                        QListWidget {
                            background-color: #f8f8f8;
                            border: 1px solid #ccc;
                            border-radius: 6px;
                            padding: 4px;
                        }
                        QListWidget::item {
                            padding: 4px 8px;
                        }
                        QListWidget::item:hover {
                            background-color: #eaeaea;
                        }
                    """)
            self.listWidget.setFocusPolicy(Qt.NoFocus)
            # Define list options
            options = [self.tr("Gebäude"), self.tr("Strassen"), self.tr("Schienen"), self.tr("Mech. Aufstiegshilfen"), self.tr("Leitungen"), self.tr("Personen im Freien"), self.tr("Flächen")]
            for opt in options:
                if opt in self.selected_dam_pot_layers:
                    item = QListWidgetItem(opt)
                    item.setCheckState(Qt.Checked)
                    item.setFlags(item.flags() | 2)  # Make sure it's enabled
                    self.listWidget.addItem(item)

            self.listWidget.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            self.listWidget.itemClicked.connect(SaveLoadDialog.on_item_clicked)
            if len(set(self.selected_dam_pot_layers)) <= 1:
                maximum_height = (35 * height_adjust_variable)
                if sys.platform == "darwin":
                    maximum_height += 5
            else:
                maximum_height = (35 * height_adjust_variable) - (5 * height_adjust_variable)
                if sys.platform == "darwin":
                    maximum_height = 40 + (height_adjust_variable*20) - 5
            self.listWidget.setMaximumHeight(int(maximum_height))
            layout.addWidget(self.listWidget, alignment=Qt.AlignHCenter)  # Add list widget to layout

            # Line for visual separation of dialog elements
            line = QFrame()
            line.setFrameShape(QFrame.HLine)
            line.setFrameShadow(QFrame.Sunken)
            layout.addWidget(line)

        # Label & Input Field for Save Path
        self.label = QLabel(self.tr("Speicherort (optional):"))
        self.label.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self.path_input = QLineEdit(self)
        self.path_input.setStyleSheet("color: black;")
        self.browse_button = QPushButton(self.tr("Speicherort auswählen ..."))
        self.browse_button.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self.browse_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.browse_button.clicked.connect(self.select_directory)

        # Save & Load Buttons
        self.save_button = QPushButton(self.tr("Layer speichern und in die Karte laden"))
        self.save_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.save_button.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self.save_button.clicked.connect(lambda: self.handle_layers(True))
        self.load_button = QPushButton(self.tr("Nur in Karte laden (als Temporär-Layer)"))
        self.load_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.load_button.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self.load_button.clicked.connect(lambda: self.handle_layers(False))

        layout.addWidget(self.label, alignment=Qt.AlignHCenter)
        layout.addWidget(self.path_input)
        layout.addWidget(self.browse_button)

        # Line for visual separation of dialog elements
        line = QFrame()
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        layout.addWidget(line)

        # Add buttons to layout
        layout.addWidget(self.save_button)
        layout.addWidget(self.load_button)

        self.setLayout(layout)

    def get_selected_options(self):
        """Retrieves user selected options from a list widget."""

        checked = []
        for i in range(self.listWidget.count()):
            item = self.listWidget.item(i)
            if item.checkState():
                checked.append(item.text())
        return checked

    def handle_layers(self, save):
        """Processes layers in terms of saving and loading them into the map."""

        # Save or add layers to the map
        if save is True:  # Save layers if user has selected a valid folder
            save_path = self.path_input.text().strip()
            if not save_path:
                self.path_input.setText(self.tr("Bitte Ordner ausgewählen ... (!)"))
                self.path_input.setStyleSheet("color: red;")
                return
            # Loop over layer list
            for layer_tuple in self.layers:
                # Skip intersection layer if user has not selected the corresponding damage potential type
                if self.pipeline == "upload_consequence_analysis":
                    if layer_tuple[2] not in self.get_selected_options():
                        continue
                # Get save path
                save_path = self.path_input.text().strip()
                # Get layer name and define save path
                layer_name = layer_tuple[0].name()
                save_path = os.path.join(save_path, f"{layer_name}.gpkg")
                # Save Vector Options
                options = QgsVectorFileWriter.SaveVectorOptions()
                options.driverName = "GPKG"
                options.fileEncoding = "UTF-8"
                # Ensure proper coordinate transformation context
                context = QgsCoordinateTransformContext()
                # Get QGIS version numbers
                qgis_major_version, qgis_minor_version = map(int, Qgis.QGIS_VERSION.split('.')[:2])
                # Avoid deprecation warnings while saving
                if qgis_major_version > 3 or (qgis_major_version == 3 and qgis_minor_version >= 40):  # QGIS 3.40+
                    QgsVectorFileWriter.writeAsVectorFormatV3(
                        layer_tuple[0], save_path, context, options
                    )
                elif qgis_major_version == 3 and qgis_minor_version >= 32:  # QGIS 3.32 to 3.39
                    error, new_file = QgsVectorFileWriter.writeAsVectorFormatV2(
                        layer_tuple[0], save_path, context, options
                    )
                saved_layer = QgsVectorLayer(f"{save_path}|layername={layer_name}", layer_name, "ogr")
                # Visualize risk layers only
                if self.pipeline == "visualize_risks":
                    SaveLoadDialog.visualize_layer(saved_layer, layer_tuple[1], "risk")
                # Add risk layer to map
                QgsProject.instance().addMapLayer(saved_layer)  # type: ignore
            # Close the dialog
            self.accept()
        else:  # Just add layers to the map
            for layer_tuple in self.layers:
                damage_potential_type = layer_tuple[1]
                if self.pipeline == "upload_consequence_analysis":  # Only add intersection layers if user has selected the corresponding damage potential type
                    if layer_tuple[2] in self.get_selected_options():

                        layer = layer_tuple[0]
                        if not layer.isValid():
                            raise Exception("Layer failed to load!")

                        # econome point objects: affected by exactly 1 intensity
                        if damage_potential_type.upper() == "BUILDING":

                            SaveLoadDialog.visualize_layer(layer, layer_tuple[1], "intensity")
                            QgsProject.instance().addMapLayer(layer) 
                            
                        # line and polygon objects: affected by multiple intensities along their geometry
                        elif layer.geometryType() in [1, 2]: 
                            
                            # create symbol and placement
                            if layer.geometryType() == 2:  # Polygon geometries
                                symbol = QgsSymbol.defaultSymbol(layer.geometryType())
                                symbol.setColor(QColor('lightgrey'))
                                placing = QgsPalLayerSettings.Horizontal
                            else:
                                symbol = QgsLineSymbol.createSimple({'color': 'lightgrey', 'width': '1'})
                                placing = QgsPalLayerSettings.Line
                                
                            renderer = QgsSingleSymbolRenderer(symbol)
                            layer.setRenderer(renderer)

                            # Define labeling settings
                            label_settings = QgsPalLayerSettings()
                            expr_str = (
                                "'Weak: ' || to_string(round(\"intensity_weak\", 2)) || ' ' || \"unit\" || "
                                "'\nModerate: ' || to_string(round(\"intensity_moderate\", 2)) || ' ' ||  \"unit\" || "
                                "'\nStrong: ' || to_string(round(\"intensity_strong\", 2)) || ' ' || \"unit\" "
                            )
                            weak_translated = self.tr("Weak")
                            moderate_translated = self.tr("Moderate")
                            strong_translated = self.tr("Strong")
                            expr_str = expr_str.replace("Weak", weak_translated)
                            expr_str = expr_str.replace("Moderate", moderate_translated)
                            expr_str = expr_str.replace("Strong", strong_translated)
                            
                            label_settings.placement = placing  # placement for line layers
                            label_settings.fieldName = expr_str  
                            label_settings.isExpression = True

                            # Optional: text formatting
                            text_format = label_settings.format()
                            text_format.setSize(10)
                            text_format.setFont(QFont("Arial"))
                            text_format.setColor(QColor("black"))

                            # --- Background settings ---
                            bg_settings = QgsTextBackgroundSettings()
                            bg_settings.setEnabled(True)
                            bg_settings.setFillColor(QColor("white"))  # White background
                            bg_settings.setStrokeColor(QColor("black"))  # Black border
                            bg_settings.setStrokeWidth(0.5)  # Border width
                            bg_settings.setRadii(QSizeF(3.0, 3.0))  # Rounded corners (px)
                            bg_settings.setSize(QSizeF(2.0, 2.0))  # Padding (horizontal, vertical)
                            bg_settings.setOpacity(1.0)

                            text_format.setBackground(bg_settings)
                            label_settings.setFormat(text_format)

                            # Enable labeling
                            labeling = QgsVectorLayerSimpleLabeling(label_settings)
                            layer.setLabelsEnabled(True)
                            layer.setLabeling(labeling)

                            QgsProject.instance().addMapLayer(layer) 
                            layer.triggerRepaint()

                elif self.pipeline == "visualize_risks":  # Add all risk layers
                    QgsProject.instance().addMapLayer(layer_tuple[0])  # type: ignore
            # Close the dialog
            self.accept()

    @staticmethod
    def on_item_clicked(item):

        # Toggle check state
        if item.checkState() == Qt.Checked:
            item.setCheckState(Qt.Unchecked)
        else:
            item.setCheckState(Qt.Checked)

    def select_directory(self):
        """Opens file dialog to select a directory."""

        folder = QFileDialog.getExistingDirectory(self, "Select Directory")
        if folder:
            self.path_input.setText(folder)
            self.path_input.setStyleSheet("color: black;")

    @staticmethod
    def visualize_layer(layer, dam_pot_type, vis_type):
        """Visualizes an input vector layer."""

        # Get layer fields
        layer_fields = [field.name() for field in layer.fields()]

        # Convert "r_indiv" to a float value if necessary
        if "r_indiv" in layer_fields:
            layer.startEditing()
            if "r_indiv" in layer_fields:
                # Prepare values for rendering
                for feature in layer.getFeatures():
                    try:
                        if feature["r_indiv"]:
                            feature["r_indiv"] = float(feature["r_indiv"])
                            layer.updateFeature(feature)
                    except ValueError as e:
                        log.info("Couldn't convert column 'r_indiv' to float values!")
                        log.error(e)
            layer.commitChanges()

        ranges = []  # Initialize emtpy ranges list

        # Define first range
        symbol1 = QgsSymbol.defaultSymbol(layer.geometryType())
        if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
            symbol1.setWidth(1.5)
        range1 = None
        if vis_type == "risk":
            symbol1.setColor(QColor("lightblue"))
            range1 = QgsRendererRange(0, 0.000001, symbol1, "< 1e-6")
        elif vis_type == "intensity":
            symbol1.setColor(QColor("#e1eebf"))
            range1 = QgsRendererRange(0.5, 1.5, symbol1, "weak")
        ranges.append(range1)

        # Define second range
        symbol2 = QgsSymbol.defaultSymbol(layer.geometryType())
        if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
            symbol2.setWidth(1.5)
        range2 = None
        if vis_type == "risk":
            symbol2.setColor(QColor("orange"))
            range2 = QgsRendererRange(0.000001, 0.00001, symbol2, "1e-6 - 1e-5")
        elif vis_type == "intensity":
            symbol2.setColor(QColor("#a2d8a2"))
            range2 = QgsRendererRange(1.5, 2.5, symbol2, "moderate")
        ranges.append(range2)

        # Define third range
        symbol3 = QgsSymbol.defaultSymbol(layer.geometryType())
        if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
            symbol3.setWidth(1.5)
        range3 = None
        if vis_type == "risk":
            symbol3.setColor(QColor("red"))
            range3 = QgsRendererRange(0.00001, 1, symbol3, "> 1e-5")
        elif vis_type == "intensity":
            symbol3.setColor(QColor("#88bb87"))
            range3 = QgsRendererRange(2.5, 3.5, symbol3, "strong")
        ranges.append(range3)

        renderer = None
        if vis_type == "risk":
            renderer = QgsGraduatedSymbolRenderer("r_indiv", ranges)  # Create a graduated symbol renderer
        elif vis_type == "intensity":
            attribute_name = layer.fields()[-1].name()
            renderer = QgsGraduatedSymbolRenderer(attribute_name, ranges)  # Create a graduated symbol renderer

        # Apply renderer to the layer
        layer.setRenderer(renderer)


class RightClickFilter(QObject):
    """Right-click event filter."""

    def __init__(self, callback):
        super().__init__()
        self.callback = callback  # Function to execute when right-click was detected

    def eventFilter(self, obj, event):
        # Filter only right-clicks and on the label
        if event.type() == QEvent.MouseButtonPress and event.button() == Qt.RightButton:
            damage_potential_type = "_".join(obj.objectName().split("_")[4:]).lower()  # Get damage potential type
            damage_potential_name = obj.text()
            self.callback(damage_potential_type, damage_potential_name)  # Execute function
            return True  # stop event propagation
        return False


class WheelFilter(QObject):
    """This filter disables mouse wheel scrolling through combo box options."""

    def eventFilter(self, obj, event):
        if event.type() == QEvent.Wheel:
            return True  # block the wheel event
        return False


class EconoMe:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):

        # INSTANCE ATTRIBUTES
        # ---------------------

        # QGIS project & interface
        self.iface = iface
        self.qgis_project = None
        # Plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # Main dialog
        self.main_dialog = None
        # Login & API & User Projects
        self.ECONOME_USER_INFO = None
        self.ECONOME_USER_INFO_FILE_PATH = None
        self.ECONOME_API_GLOSSARY = None
        self.ECONOME_API_NAME = "Explore"
        self.set_econome_api_version()
        self.ECONOME_PROJECT_NUMBER = None
        self.ECONOME_PROJECT_LIST = None
        self.ECONOME_USER_NAME = None
        self.ECONOME_PASSWORD = None
        self.LOGIN_ERROR_MESSAGE = None
        self.USER_INFO_VALIDATED = False
        self.CHANGING_INTENSITY_LAYER = False
        # Status dialog icons
        self.scaled_pixmap_checkmark = None
        self.scaled_pixmap_warning = None
        # Animations
        self.PROGRESS_INFO = None
        self.PROGRESS_INFO_TEXT = None
        self.PROGRESS_GIF = None
        self.PROGRESS_OVERLAY = None
        self.PROGRESS_ANIMATION = None
        self.ACTIVE_PROGRESS_ANIMATION = None
        # Thread pipelines
        self.NEW_THREAD = None
        self.THREAD_PIPELINE = None
        self.OPEN_SAVE_LAYERS_DIALOG = None
        # Consequence analysis
        self.SAVE_INTERSECTION_LAYERS_FROM_CONSEQUENCE_ANALYSIS = False
        self.INTERSECTION_LAYERS_CONSEQUENCE_ANALYSIS = []
        # Scenario Status
        subkeys = ["intensity_column", "general"]
        self.SCENARIO_STATUS = {key: {subkey: {"status": -1, "info": ""} for subkey in subkeys} for key in ["FREE", "30", "100", "300", "EXTREME"]}
        # Damage potential
        self.DAMAGE_POTENTIAL_LAYER_LIST = None
        self.SELECTED_DAMAGE_POTENTIAL_LAYERS = None
        self.DAMAGEPOTENTIAL_LABELS = None
        self.DAMAGEPOTENTIAL_LABEL_FILTERS = None
        self.DAMAGE_POTENTIAL_GEOMETRY_TYPES = None
        self.DAMAGE_POTENTIAL_ASSETS_HANDLING_OPTION = ""
        self.DAMAGE_POTENTIAL_LAYER_STATUS = {
            "BUILDING": -1,
            "ROAD": -1,
            "RAILWAY": -1,
            "CABLE_TRANSPORT": -1,
            "LINE_PIPE": -1,
            "PEOPLE_OUTDOOR": -1,
            "OTHER_ASSET": -1
        }
        self.DAMAGEPOTENTIAL_FIELDS = {
            "building": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Occupancy", "OccupancyComment", "Presence", "PresenceComment"],
            "road": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Occupancy", "OccupancyComment", "Transits", "ConnectedRouteID"],
            "railway": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "TrainLength", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "DailyFreightTraffic", "Occupancy", "OccupancyComment", "DangerousTerrain", "Transits", "ValueEngine", "ConnectedRouteID"],
            "cable_transport": ["AssetIdentID", "AssetTypeID", "PassengerVolume", "Speed", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Transits"],
            "line_pipe": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment"],
            "people_outdoor": ["AssetIdentID", "AssetTypeID", "Description", "Occupancy", "Presence", "PresenceComment"],
            "other_asset": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment"]
        }
        self.DAMAGEPOTENTIAL_REQUIRED_FIELDS = {
            "building": ["AssetIdentID", "AssetTypeID"],
            "road": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed"],
            "railway": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "TrainLength"],
            "cable_transport": ["AssetIdentID", "AssetTypeID", "PassengerVolume", "Speed"],
            "line_pipe": ["AssetIdentID", "AssetTypeID"],
            "people_outdoor": ["AssetIdentID", "AssetTypeID", "Occupancy"],
            "other_asset": ["AssetIdentID", "AssetTypeID"]
        }
        self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE = {
            "BUILDING": ['Gebäude', 'Sonderobjekte'], 
            "ROAD": ['Strassenverkehr'],
            "RAILWAY": ['Schienenverkehr'],
            "CABLE_TRANSPORT": ['Mechanische Aufstiegshilfe'],
            "LINE_PIPE": ['Leitungen'],
            "PEOPLE_OUTDOOR": ['Personen im Freien'],
            "OTHER_ASSET": ['Landwirtschaft, Wald und Grünanlagen', 'Infrastruktur']
        }
        self.ASSET_TYPEID_NAME_BY_CATEGORYNAME = {
            'Gebäude': {
                'Bahnhof': 14,
                'Einkaufszentrum': 9,
                'Frei wählbares Gebäude': 64,
                'Garage (Parkeinheit)': 4,
                'Hotel': 5,
                'Industrie-/ Gewerbegebäude': 6,
                'Keller (nur für Wasserprozesse)': 91,
                'Kirche': 12,
                'Öffentliches Gebäude': 18,
                'Öffentlicher Parkplatz': 88,
                'Schule / Kindergarten': 10,
                'Schuppen / Remise': 3,
                'Spital': 11,
                'Sportanlage (Gebäude)': 15,
                'Stationsgebäude mechanische Aufstiegshilfe': 63,
                'Viehstall': 2,
                'Wohneinheit Einfamilienhaus': 1,
                'Wohneinheit Mehrfamilienhaus': 87
            },
            'Sonderobjekte': {
                'ARA': 19,
                'Campingplatz': 21,
                'Zeltplatz': 89,
                'Deponie': 20,
                'Frei wählbares Sonderobjekt': 66,
                'Kraftwerk': 8,
                'Sendeanlage': 65,
                'Wasserreservoir': 22
            },
            'Strassenverkehr': {
                'Brücke Autobahn': 34,
                'Brücke Einzelfahrzeuge': 31,
                'Brücke Kantonsstrasse': 33,
                'Brücke Gemeindestrasse': 32,
                'Asphaltweg': 56,
                'Kiesweg': 55,
                'Frei wählbares Strassenverkehrsobjekt': 67,
                'Gemeindestrasse': 26,
                'Kantonsstrasse': 25,
                'Nationalstrasse': 24
            },
            'Leitungen': {
                'Abwasser unter Terrain': 60,
                'Frei wählbares Leitungsobjekt': 82,
                'Freileitung': 38,
                'Gas auf Terrain': 61,
                'Gas unter Terrain': 80,
                'Hochspannungsleitung <= 150 kV': 57,
                'Hochspannungsleitung > 150 kV': 58,
                'Strom unter Terrain': 37,
                'Telekommunikation auf Terrain': 62,
                'Telekommunikation unter Terrain': 81,
                'Wasser auf Terrain': 59,
                'Wasser unter Terrain': 79
            },
            'Mechanische Aufstiegshilfe': {
                'Frei wählbares Objekt (mech. Aufstiegshilfe)': 78,
                'Gondelbahn': 75,
                'Luftseilbahn': 76,
                'Sessellift': 74,
                'Skilift': 73,
                'Standseilbahn': 77
            },
            'Landwirtschaft, Wald und Grünanlagen': {
                'extensive Flächen': 43,
                'Frei wählbares Objekt (Landwirtschaft, Grünflächen, Wald)': 86,
                'Friedhof': 84,
                'Gemüsekultur': 47,
                'Golfanlage': 83,
                'intensive Flächen': 41,
                'Nutzwald': 45,
                'Obstplantage': 48,
                'Parkanlage': 42,
                'Rebberg': 46,
                'Schutzwald': 44,
                'Sportaussenanlage': 85
            },
            'Schienenverkehr': {
                'Brücke Einspur': 68,
                'Brücke Mehrspur': 69,
                'Einspur': 28,
                'Mehrspur': 29,
                # 'Sonderobjekt Brücke Schienenverkehr': 90  # not existing anymore based on Peter's info
            },
            'Infrastruktur': {
                # 'Durchtrennung Kabelanlage (pro Schadenstelle)': 71, # not existing anymore based on Peter's info
                'Bahninfrastruktur': 72,
                # 'Schaltposten': 70, # not existing anymore based on Peter's info
                'Schutzbauwerk': 54
            },
            'Personen im Freien': {
                'Personen im Freien': 2002
            }
        }

        self.PROCESS_TYPES = None
        self.ASSET_UNIT_BY_OBJECT_TYPEID = None
        
        self.QVARIANT_TYPES_DICT = {
            "integer": [QVariant.Int, QVariant.LongLong], 
            "float": [QVariant.Double], 
            "string": [QVariant.String], 
            "boolean": [QVariant.Bool]
        } 
        self.AFFECTED_ASSET_SPECIAL_PARAMETERS = {
            "BUILDING": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string"), ("PresenceValue", "numZeroToOne"), ("PresenceComment", "string"), ("ProtectionValue", "numZeroToOne"), ("ProtectionComment", "string")],
            "ROAD": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string"), ("p_vSp", "numZeroToOne"), ("p_vSp_Comment", "string"), ("AlarmSystem", "boolean")],
            "RAILWAY": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string"), ("p_vSp", "numZeroToOne"), ("p_vSp_Comment", "string"), ("AlarmSystem", "boolean")],
            "CABLE_TRANSPORT": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string")],
            "LINE_PIPE": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string")],
            "PEOPLE_OUTDOOR": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string")],
            "OTHER_ASSET": [("SOL_ID", "integer"), ("p_rA", "real"), ("p_rA_Comment", "string")]
        }
        # Intensity layers
        self.SELECTED_INTENSITY_MAP_LAYERS = None
        self.SELECTED_INTENSITY_COLUMNS = None
        # Risk layers
        self.RISK_OUTPUT_VARIABLES = {
            "BUILDING": ["R_Sw", "R_P(CHF)", "R", "r_indiv"],
            "ROAD": ["R_Sw", "R_P(CHF)", "R", "r_indiv", "R_Dir"],
            "RAILWAY": ["R_Sw", "R_P(CHF)", "R", "r_indiv", "R_Dir", "R_Anp"],
            "CABLE_TRANSPORT": ["R_Sw", "R_P(CHF)", "R", "r_indiv"],
            "LINE_PIPE": ["R_Sw"],
            "PEOPLE_OUTDOOR": ["R_P(CHF)", "R", "r_indiv", "R_Dir", "R_Anp"],
            "OTHER_ASSET": ["R_Sw"]
        }
        self.RISK_OUTPUT_TEMP_LAYERS = []

        # Plugin language
        self.ECONOME_PLUGIN_LANGUAGE = QgsExpressionContextUtils.globalScope().variable("econome_language")  # type: ignore
        if self.ECONOME_PLUGIN_LANGUAGE is None:  # Use qgis locale instead
            self.ECONOME_PLUGIN_LANGUAGE = QSettings().value('locale/userLocale')[0:2]
            if self.ECONOME_PLUGIN_LANGUAGE not in ["DE", "EN", "FR", "IT"]:
                self.ECONOME_PLUGIN_LANGUAGE = "EN"  # Default to English if no valid language is set
        self.ECONOME_PLUGIN_LANGUAGE_NAMES = {
            "DE": "German",
            "EN": "English",
            "FR": "French",
            "IT": "Italian"
        }

        # Initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{locale}.qm'.format(locale=locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&EconoMe')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Translate plugin
        self.switch_language(self.ECONOME_PLUGIN_LANGUAGE)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)  # type: ignore
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        action.setToolTip("EconoMe")
        action.setWhatsThis("EconoMe QGIS Plugin")

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def animate_progress(self, animate, progress_name):
        """UX feature: Animates time during which plugin is working."""

        # Create animation
        if animate is True:
            
            if not self.PROGRESS_ANIMATION:
                # Create an overlay image to blur the background
                self.PROGRESS_OVERLAY = QLabel(self.main_dialog)
                self.PROGRESS_OVERLAY.setStyleSheet("background-color: rgba(255, 255, 255, 200);")
                # Add info text (optional)
                if self.THREAD_PIPELINE == "econome_api_switch":
                    self.PROGRESS_INFO = QLabel(self.main_dialog)
                    self.PROGRESS_INFO.setText(self.tr("Changing to ") + self.ECONOME_API_NAME + " API")
                    self.PROGRESS_INFO.setFixedSize(250, 30)
                    self.PROGRESS_INFO.setAlignment(Qt.AlignCenter)
                    self.PROGRESS_INFO.setWordWrap(False)
                    self.PROGRESS_INFO.setStyleSheet("background-color: transparent; color: black; font-weight: bold;")
                    x_pos = (self.main_dialog.width() - self.PROGRESS_INFO.width()) // 2
                    y_pos = (self.main_dialog.height() - self.PROGRESS_INFO.height()) - 700 // 2
                    self.PROGRESS_INFO.move(x_pos, y_pos)
                    self.PROGRESS_INFO.setVisible(True)
                # Get turning cogs GIF and define QMovie
                self.PROGRESS_GIF = QLabel(self.main_dialog)
                self.PROGRESS_GIF.setAttribute(Qt.WA_TranslucentBackground, True)
                self.PROGRESS_GIF.setStyleSheet("background: transparent;")
                self.PROGRESS_GIF.setAutoFillBackground(False)
                self.PROGRESS_GIF.setVisible(False)
                self.PROGRESS_ANIMATION = QMovie(os.path.join(self.plugin_dir, "icons", "silver-gear-cogs.gif"))
                self.ACTIVE_PROGRESS_ANIMATION = progress_name
                self.PROGRESS_GIF.setMovie(self.PROGRESS_ANIMATION)
                self.PROGRESS_GIF.setAlignment(Qt.AlignCenter)
                # Sizing animation and overlay
                self.PROGRESS_GIF.setFixedSize(281, 320)  # Fixed GIF size
                self.PROGRESS_OVERLAY.setFixedSize(self.main_dialog.width(), self.main_dialog.height())  # Overlay size
                self.PROGRESS_ANIMATION.setScaledSize(QSize(80, 80))  # Scale it
                # Absolute positioning in dialog
                x_pos = (self.main_dialog.width() - self.PROGRESS_GIF.width()) // 2
                y_pos = (self.main_dialog.height() - self.PROGRESS_GIF.height()) // 2
                self.PROGRESS_GIF.move(x_pos, y_pos)
                # Set GIF and overlay visible
                self.PROGRESS_OVERLAY.setVisible(True)
                self.PROGRESS_GIF.setVisible(True)
                self.PROGRESS_ANIMATION.start()

        else:  # Stop animation and remove from UI

            if self.PROGRESS_INFO:
                self.PROGRESS_INFO.setParent(None)  # type: ignore
                self.PROGRESS_INFO.deleteLater()
                self.PROGRESS_INFO = None

            if self.PROGRESS_ANIMATION:

                # Clean up
                self.PROGRESS_ANIMATION.stop()
                self.PROGRESS_ANIMATION.deleteLater()
                self.PROGRESS_ANIMATION = None
                self.PROGRESS_GIF.setParent(None)  # type: ignore
                self.PROGRESS_GIF.deleteLater()
                self.PROGRESS_GIF = None
                self.PROGRESS_OVERLAY.setParent(None)  # type: ignore
                self.PROGRESS_OVERLAY.deleteLater()
                self.PROGRESS_OVERLAY = None

                # Custom actions for different thread pipelines
                # --

                if self.THREAD_PIPELINE == "upload_consequence_analysis" and self.SAVE_INTERSECTION_LAYERS_FROM_CONSEQUENCE_ANALYSIS and self.OPEN_SAVE_LAYERS_DIALOG:
                    self.show_save_or_load_dialog()

                if self.THREAD_PIPELINE == "visualize_risks" and len(self.RISK_OUTPUT_TEMP_LAYERS) != 0 and self.OPEN_SAVE_LAYERS_DIALOG:
                    self.show_save_or_load_dialog()

                # Reset thread pipeline
                self.THREAD_PIPELINE = None
            
    def change_settings(self):
        """Opens the settings menu and applies changes to plugin."""

        # Initiate settings dialog
        dialog = EconoMeDialogSettingsMenu(
            econome_user_name=self.ECONOME_USER_NAME,
            econome_password=self.ECONOME_PASSWORD,
            econome_api_base_path=self.ECONOME_API_PROJECT_BASE_PATH
        )
        title = self.tr("EconoMe") + " - " + self.tr("Einstellungen")
        dialog.setWindowTitle(title)
        dialog.COMBO_BOX_API_SELECTION.setCursor(QCursor(Qt.PointingHandCursor))
        dialog.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Ok).setCursor(QCursor(Qt.PointingHandCursor))
        dialog.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Cancel).setCursor(QCursor(Qt.PointingHandCursor))

        # Modify setting options
        # API selection
        API_list = ["EconoMe", "Explore"]
        # add dev api for us developers
        user_full_name = QgsExpressionContextUtils.globalScope().variable("user_full_name")  # type: ignore
        if user_full_name in ["Tobia Lezuo", "Kevin Patrick Helzel"]:
            API_list.extend(["Road-Risk", "Dev-EconoMe", "Dev-Explore", "Dev-Road-Risk"])
        for API in API_list:
            if API not in [dialog.COMBO_BOX_API_SELECTION.itemText(i) for i in range(dialog.COMBO_BOX_API_SELECTION.count())]:
                dialog.COMBO_BOX_API_SELECTION.addItem(API)
        dialog.COMBO_BOX_API_SELECTION.setCurrentText(self.ECONOME_API_NAME)
        # Language selection
        LANGUAGES_list = ["DE", "FR", "IT", "EN"]
        for LANGUAGE in LANGUAGES_list:
            if LANGUAGE not in [dialog.COMBO_BOX_LANGUAGE_SELECTION.itemText(i) for i in
                                range(dialog.COMBO_BOX_LANGUAGE_SELECTION.count())]:
                dialog.COMBO_BOX_LANGUAGE_SELECTION.addItem(LANGUAGE)
        dialog.COMBO_BOX_LANGUAGE_SELECTION.setCurrentText(self.ECONOME_PLUGIN_LANGUAGE)

        #  Define OK | CANCEL workflow
        result = dialog.exec_()
        # User applies (new) settings
        if result == QDialog.Accepted:

            old_api = ""
            selected_api = dialog.COMBO_BOX_API_SELECTION.currentText()
            new_language = dialog.COMBO_BOX_LANGUAGE_SELECTION.currentText()

            apply_new_settings = selected_api != self.ECONOME_API_NAME or self.ECONOME_PLUGIN_LANGUAGE != new_language
            econome_api_changed = selected_api != self.ECONOME_API_NAME

            # Change API name
            if selected_api != self.ECONOME_API_NAME:
                old_api = self.ECONOME_API_NAME
                self.ECONOME_API_NAME = selected_api
                self.OPEN_SAVE_LAYERS_DIALOG = False
                if old_api.startswith("Dev") or selected_api.startswith("Dev"):
                    self.main_dialog.LABEL_ECONOME_API_USER_NAME.setText("")
                    self.main_dialog.LABEL_ECONOME_API_PASSWORD.setText("")

            self.set_econome_api_version()

            # Only perform translation if language was changed
            if self.ECONOME_PLUGIN_LANGUAGE != new_language:
                self.ECONOME_PLUGIN_LANGUAGE = dialog.COMBO_BOX_LANGUAGE_SELECTION.currentText()
                # translate the plugin
                self.switch_language(self.ECONOME_PLUGIN_LANGUAGE)

                # Update the variable econome_language in the qgis settings
                QgsExpressionContextUtils.setGlobalVariable('econome_language', self.ECONOME_PLUGIN_LANGUAGE)  # type: ignore

            # Restart the plugin to make sure the ui is translated
            if apply_new_settings:
                if econome_api_changed:
                    self.THREAD_PIPELINE = "econome_api_switch"
                    QgsExpressionContextUtils.setProjectVariable(self.qgis_project, f"econome_project_id_{old_api}", self.ECONOME_PROJECT_NUMBER)
                self.main_dialog.close()
                self.run()

    def clean_up(self, file_path):
        """This method removes a file from the disk."""

        # Delete file
        if file_path is not None:
            if os.path.exists(file_path):
                os.remove(file_path)

    def compare_damage_potential(self, layer_assets, uploaded_damage_potential, damage_potential_type, valid_parameter_pairs):
        """This method compares the damage potential from a GIS layer with stored data in EconoMe."""

        # Check for existing uploaded damage potential
        if len(uploaded_damage_potential) != 0:
            # Create two "AssetIdentID" lists
            uploaded_AssetIdentIDs = [asset["AssetIdentID"] for asset in uploaded_damage_potential]
            layer_AssetIdentIDs = [str(asset["AssetIdentID"]) for asset in layer_assets["AssetList"]]
            # Create empty attribute mismatch "AssetIdentID" list
            mismatch_dict = {}
            # Get layer AssetTypeIDs
            layer_AssetTypeIDs = [asset["AssetTypeID"] for asset in layer_assets["AssetList"]]
            # Check if AssetIdentIDs match the damage potential type
            OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]] for item in sublist]
            FILTERED_LAYER_ASSET_TYPE_IDS = [AssetTypeID for AssetTypeID in layer_AssetTypeIDs if AssetTypeID not in OBJECT_CATEGORIES]
            # Check if damage potential type was changed to another, not fitting category
            if len(FILTERED_LAYER_ASSET_TYPE_IDS) == 0:
                # CASE: All uploaded assets are present in the damage potential layer
                if set(uploaded_AssetIdentIDs).issubset(layer_AssetIdentIDs):
                    # Loop over all layer assets
                    for layer_asset in layer_assets["AssetList"]:
                        asset_missmatch_dict = {}  # Mismatch dictionary
                        corresponding_uploaded_asset = [asset for asset in uploaded_damage_potential if asset["AssetIdentID"] == str(layer_asset["AssetIdentID"])]  # Get corresponding uploaded asset
                        if len(corresponding_uploaded_asset) > 0:
                            # Loop over all fields
                            for pp in valid_parameter_pairs:
                                if pp["API"] in layer_asset.keys():
                                    # If asset value is a float, make sure to compare truncated numbers
                                    if isinstance(layer_asset[pp["API"]], float):
                                        layer_asset[pp["API"]] = EconoMe.truncFloat(layer_asset[pp["API"]])
                                    if isinstance(corresponding_uploaded_asset[0][pp["API"]], float):
                                        corresponding_uploaded_asset[0][pp["API"]] = EconoMe.truncFloat(corresponding_uploaded_asset[0][pp["API"]])
                                    # Check for difference between layer asset and uploaded asset
                                    if layer_asset[pp["API"]] != corresponding_uploaded_asset[0][pp["API"]]:
                                        asset_missmatch_dict.update({pp["API"]: {"Layer": layer_asset[pp["API"]], "API": corresponding_uploaded_asset[0][pp["API"]]}})  # Update mismatch dictionary
                        else:  # Layer asset is not present in EconoMe
                            asset_missmatch_dict = {"Missing": self.tr("Fehlt in EconoMe.")}
                        if asset_missmatch_dict != {}:
                            mismatch_dict.update({layer_asset["AssetIdentID"]: asset_missmatch_dict})
                    # CHECK for entries in mismatch dictionary
                    if len(mismatch_dict.keys()) == 0:
                        return "COMPLETE MATCH", mismatch_dict
                    else:
                        handling_dialog = DamagePotentialMismatchResolveDialog(mode='MISMATCH', mismatch=mismatch_dict, main_dialog=self.main_dialog)
                        if handling_dialog.exec_() == QDialog.Accepted:
                            return handling_dialog.selected_option, mismatch_dict
                        else:
                            return "CANCEL", {}
                # CASE: Not all uploaded assets are present in the damage potential layer -> Handling dialog
                elif not set(layer_AssetIdentIDs).issubset(uploaded_AssetIdentIDs) or len(set(uploaded_AssetIdentIDs)) > len(layer_AssetIdentIDs):
                    mismatch_dict = {AssetIdentID: {"Missing": "Fehlt im Layer."} for AssetIdentID in list(set(uploaded_AssetIdentIDs) - set(layer_AssetIdentIDs))}
                    handling_dialog = DamagePotentialMismatchResolveDialog(mode='MISSING', mismatch=mismatch_dict, main_dialog=self.main_dialog)
                    if handling_dialog.exec_() == QDialog.Accepted:
                        return handling_dialog.selected_option, mismatch_dict
                    else:
                        return "CANCEL", {}
            else:
                # CASE: AssetIdentID doesn't match object category -> PREVENT sync
                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials!"), message_text=self.tr("\nEine oder mehrere Objektart-ID wurde/n im Layer zu einer nicht der Kategorie des Schadenpotentials entprechenden ID geändert.\nAus diesem Grund wird der Layer durch EconoMe zurückgesetzt.\n")).exec_()
                return "KEEP ECONOME ASSET TYPE IDS", {}
        else:
            # CASE: No uploaded damage potential existing -> Direct POST of layer assets
            # Get layer AssetTypeIDs
            layer_AssetTypeIDs = [asset["AssetTypeID"] for asset in layer_assets["AssetList"]]
            # CHECK if AssetIdentIDs match the damage potential type
            OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]] for item in sublist]
            FILTERED_LAYER_ASSET_IDENT_IDS = [AssetIdentID for AssetIdentID in layer_AssetTypeIDs if AssetIdentID not in OBJECT_CATEGORIES]
            # CHECK if damage potential type was changed to another category
            if len(FILTERED_LAYER_ASSET_IDENT_IDS) == 0:
                return "POST LAYER ASSETS", {}
            else:
                # CASE: AssetIdentID doesn't match object category -> PREVENT upload
                return "INVALID OBJECT CATEGORY", {}

    def create_template_layer(self, dam_pot_type, dam_pot_name):
        """This method creates a template layer for a certain damage potential type with all necessary fields."""

        # CHECK: User logged in?
        if self.USER_INFO_VALIDATED:

            # Set template layer name
            template_layer_name = self.tr("Schadenobjekte: {DAMAGE_POTENTIAL_TYPE}").format(DAMAGE_POTENTIAL_TYPE=dam_pot_name)

            # Create a new vector layer of a specific geometry
            if dam_pot_type in ["building", "other_asset"]:  # Polygon geometry
                geom = "Polygon?crs=EPSG:2056"
            elif dam_pot_type in ["road", "railway", "cable_transport", "line_pipe"]:  # Line geometry
                geom = "LineString?crs=EPSG:2056"

            try:
                template_layer = QgsVectorLayer(geom, template_layer_name, "memory")
            except Exception as e:
                raise RuntimeError(f"Layer creation failed {e}")

            # Fetch API infos
            API_GET_responses = self.multiple_API_GET_requests([
                self.ECONOME_API_PROFILE_BASE_PATH,
                self.ECONOME_API_GLOSSARY_BASE_PATH + f"/{self.ECONOME_PLUGIN_LANGUAGE.lower()}"
            ])

            DEFAULT_PARAMETER_PROFILE = API_GET_responses[0].result()[1]  # Get default parameter profile
            GLOSSARY = API_GET_responses[1].result()[1]  # Get glossary

            TYPE_DICT = {item['API']: item['Type'] for item in DEFAULT_PARAMETER_PROFILE['ParameterList'] if item["API"] in self.DAMAGEPOTENTIAL_FIELDS[dam_pot_type]}  # Create a parameter/type dictionary
            POSSIBLE_PARAMETERS = self.DAMAGEPOTENTIAL_FIELDS[dam_pot_type]  # List of possible parameters

            GLOSSARY_DICT = {item["APITerm"]: item[self.ECONOME_PLUGIN_LANGUAGE_NAMES[self.ECONOME_PLUGIN_LANGUAGE]] for item in GLOSSARY}  # Glossary dictionary

            # Harvest QgsField list
            fields = template_layer.fields()

            for parameter in POSSIBLE_PARAMETERS:

                # Find correct attribute type
                parameter_type = TYPE_DICT[parameter]
                if parameter_type == "integer":
                    field_type = QVariant.Int
                elif parameter_type == "float":
                    field_type = QVariant.Double
                elif parameter_type == "string":
                    field_type = QVariant.String
                elif parameter_type == "boolean":
                    field_type = QVariant.Bool
                else:
                    raise ValueError(f"Value type of attribute {parameter} not recognized.")
                # Find correct attribute type
                parameter_type = TYPE_DICT[parameter]
                if parameter_type == "integer":
                    field_type = QVariant.Int
                elif parameter_type == "float":
                    field_type = QVariant.Double
                elif parameter_type == "string":
                    field_type = QVariant.String
                elif parameter_type == "boolean":
                    field_type = QVariant.Bool
                else:
                    raise ValueError(f"Value type of attribute {parameter} not recognized.")

                # Append to fields list
                field_name = GLOSSARY_DICT[parameter]
                fields.append(QgsField(field_name, field_type))
                # Append to fields list
                field_name = GLOSSARY_DICT[parameter]
                fields.append(QgsField(field_name, field_type))

            # Apply the changes to the layer
            template_layer.startEditing()
            data_provider = template_layer.dataProvider()
            data_provider.addAttributes(fields)
            template_layer.updateFields()
            template_layer.commitChanges()
            # Apply the changes to the layer
            template_layer.startEditing()
            data_provider = template_layer.dataProvider()
            data_provider.addAttributes(fields)
            template_layer.updateFields()
            template_layer.commitChanges()

            # Add layer to qgis project table of contents
            try:
                if template_layer.fields().count() != 0:
                    added_successfully = self.qgis_project.addMapLayer(template_layer)
                    if added_successfully:
                        # Message user for successful template layer creation
                        StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=self.tr("ERFOLG"),message_text=self.tr("\nGIS Layer-Vorlage für den Schadenpotentialtyp {DAMAGE_POTENTIAL_NAME} wurde erfolgreich erstellt.\n").format(DAMAGE_POTENTIAL_NAME=dam_pot_name)).exec_()
                else:
                    raise RuntimeError("No columns found in template layer!")
            except Exception as e:
                # Message user for unsuccessful template layer creation
                log.error(f"Could not create GIS template layer: {e}")
                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("FEHLER"), message_text=self.tr("\nGIS Layer-Vorlage für den Schadenpotentialtyp {DAMAGE_POTENTIAL_NAME} konnte nicht erstellt werden.\n").format(DAMAGE_POTENTIAL_NAME=dam_pot_name)).exec_()

    def delete_damage_potential_affected_assets(self, annuality):
        """This method deletes affected damage potential assets by type."""

        # Filter scenes and get corresponding Scene ID
        with requests.get(
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/scene',
                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
        ) as get_scenes:
            SCENES = get_scenes.json()
            process_ID = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            variant_ID = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]
            scene_ID = [scene["SceneID"] for scene in SCENES if
                        int(scene["Annuality"]) == int(annuality) and int(scene["ProcessID"]) == int(
                            process_ID) and int(scene["VariantID"]) == int(variant_ID)][0]
            # Get existing affected assets
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_affected_assets:
                UPLOADED_AFFECTED_ASSETS = get_affected_assets.json()
                # CHECK: Existing affected assets for specified scene ?
                if len(UPLOADED_AFFECTED_ASSETS) != 0:
                    are_you_sure = AreYouSure(self.tr(
                        "\nSind sie sicher, dass sie die Betroffenheit der Schadenpotentialobjekte für das {annuality}-jährliche Szenario löschen möchten ?\n").format(
                        annuality=annuality), parent=self.main_dialog)
                    user_choice = are_you_sure.exec_()
                    if user_choice == QDialog.Accepted:
                        # Delete filtered damage potential assets
                        with requests.delete(
                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as delete_affected_assets:
                            # SUCCESS
                            if delete_affected_assets.status_code == 200:
                                StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg"),
                                             message_text=self.tr(
                                                 "\nBetroffenheit der Schadenpotentialobjekte erfolgreich gelöscht!\n")).exec_()
                            # ERROR
                            else:
                                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler"),
                                             message_text=f"\n{delete_affected_assets.json()}\n").exec_()
                else:
                    # USER INFO: Keine AffectedAssets vorhanden
                    StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler"),
                                 message_text=self.tr(
                                     "\nKeine betroffenen Schadenpotentialobjekte definiert!\n")).exec_()

    def delete_damage_potential_assets(self, damage_potential_type=None):
        """This method deletes damage potential assets by object category."""
        if not damage_potential_type:
            # Get damage potential type from sender
            damage_potential_type = self.main_dialog.sender().objectName().split("BUTTON_DAM_POT_DEL_")[-1]

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # Get current damage potential
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_damage_potential:
                UPLOADED_DAMAGE_POTENTIAL = get_damage_potential.json()
                # CHECK: Existing damage potential in EconoMe
                if len(UPLOADED_DAMAGE_POTENTIAL) != 0:
                    # Filter damage potential by object category
                    OBJECT_CATEGORIES = [item for sublist in
                                         [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in
                                          self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]]
                                         for item in sublist]
                    FILTERED_UPLOADED_DAMAGE_POTENTIAL = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if
                                                          asset["AssetTypeID"] in OBJECT_CATEGORIES]
                    # CHECK: Are there assets of the corresponding damage potential type ?
                    if len(FILTERED_UPLOADED_DAMAGE_POTENTIAL) != 0:
                        DAMAGE_POTENTIAL_LABEL_TEXT = getattr(self.main_dialog,
                                                              f"LABEL_DAM_POT_LAYER_{damage_potential_type.upper()}").text()
                        are_you_sure = AreYouSure(self.tr(
                            "\nSind Sie sicher, dass Sie alle Schadenobjekte der Objektkategorie '{DAMAGE_POTENTIAL_LABEL_TEXT}' in EconoMe löschen möchten ?\n").format(
                            DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), parent=self.main_dialog)
                        user_choice = are_you_sure.exec_()
                        if user_choice == QDialog.Accepted:
                            # Delete filtered damage potential assets
                            patch_asset_ids = {
                                "AssetIDList": [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL]}
                            with requests.patch(
                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                    json=patch_asset_ids,
                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                            ) as patch_damage_potential:
                                # SUCCESS
                                if patch_damage_potential.status_code == 200:
                                    text = self.tr(
                                        "\nAlle Schadenobjekte der Objektkategorie '{DAMAGE_POTENTIAL_LABEL_TEXT}' in EconoMe gelöscht!\n").format(
                                        DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT)
                                    if getattr(self.main_dialog,
                                               f"LAYER_DAM_POT_{damage_potential_type.upper()}").currentText():
                                        self.update_dpl_status(damage_potential_type, 0, statustip_text=text)
                                    else:
                                        self.update_dpl_status(damage_potential_type, -1)
                                    StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=self.tr(
                                        "Schadenpotential {DAMAGE_POTENTIAL_LABEL_TEXT} erfolgreich gelöscht").format(
                                        DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT),
                                                 message_text=text).exec_()
                                # ERROR
                                else:
                                    StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr(
                                        "Fehler beim Löschen des Schadenpotentials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(
                                        DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT),
                                                 message_text=self.tr("\nAPI Fehlermeldung: {response}\n").format(
                                                     response=patch_damage_potential.json())).exec_()
                    else:
                        # USER INFO: Kein Schadenpotential vorhanden
                        StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr(
                            "Fehler beim Löschen des Schadenpotentials {category}").format(
                            category=damage_potential_type.upper()), message_text=self.tr(
                            "\nIn EconoMe ist kein Schadenpotential Objektkategorie '{category}' definiert!\n").format(
                            category=damage_potential_type.upper())).exec_()
                else:
                    # USER INFO: Kein Schadenpotential vorhanden
                    StatusDialog(status_icon=self.scaled_pixmap_warning,
                                 message_title=self.tr("Fehler beim Löschen des Schadenpotentials {category}").format(
                                     category=damage_potential_type.upper()), message_text=self.tr(
                            "\nIn EconoMe ist kein Schadenpotential Objektkategorie '{category}' definiert!\n").format(
                            category=damage_potential_type.upper())).exec_()

    def disable_buttons_and_dropdowns(self, disable):
        """Disable UI elements until user is logged in."""
        widgets = self.main_dialog.findChildren((QPushButton, QComboBox))
        widget_names_to_exclude = [
            "LINE_EDIT_ECONOME_USER_NAME", "LINE_EDIT_ECONOME_PASSWORD", "BUTTON_ECONOME_PROJECT_CONNECT",
            "BUTTON_PLUGIN_SETTINGS", "COMBO_BOX_ECONOME_PROJECT_LIST",
        ]
        widget_names_to_exclude_enable = [
            "BUTTON_PP_BUILDING", "BUTTON_PP_ROAD", "BUTTON_PP_RAILWAY", "BUTTON_PP_CABLE_TRANSPORT",
            "BUTTON_PP_LINE_PIPE", "BUTTON_PP_PEOPLE_OUTDOOR", "BUTTON_PP_OTHER_ASSET",
            "BUTTON_DAM_POT_SYNC_BUILDING", "BUTTON_DAM_POT_SYNC_ROAD", "BUTTON_DAM_POT_SYNC_RAILWAY",
            "BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT", "BUTTON_DAM_POT_SYNC_LINE_PIPE", "BUTTON_DAM_POT_SYNC_PEOPLE_OUTDOOR", "BUTTON_DAM_POT_SYNC_OTHER_ASSET",
        ]
        for widget in widgets:
            # CHECK: exclude the login area
            if widget.objectName() not in widget_names_to_exclude:
                if not disable and widget.objectName() not in widget_names_to_exclude_enable:
                    widget.setEnabled(True)
                elif disable:
                    widget.setEnabled(False)
                    # CHECK: normal and layer combo boxes
                    if isinstance(widget, QComboBox):
                        # CHECK: intensity map combo boxesempty selection of layer combo boxes - do not empty options!
                        if isinstance(widget, QgsMapLayerComboBox):
                            widget.setLayer(QgsVectorLayer(""))  # type: ignore
                        # CHECK: empty options of other combo boxes - automatically empties options as well
                        else:
                            widget.clear()

    def empty_widgets(self):
        # Clear labels
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_PEOPLE_OUTDOOR.clear()
        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET.clear()
        # Reset drop-downs
        self.main_dialog.LAYER_DAM_POT_BUILDING.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_ROAD.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_RAILWAY.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setCurrentIndex(0)
        self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
        self.main_dialog.COMBO_BOX_MEASURE_VARIANT.clear()
        self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setCurrentIndex(0)
        # Reset drop-down texts
        self.main_dialog.LABEL_HAZARD_PROCESSES.setText(self.tr("GEFAHRENPROZESS(E)"))
        self.main_dialog.LABEL_MEASURE_VARIANTS.setText(self.tr("MASSNAHMENVARIANTE(N)"))
        # Reset custom scenario labels
        self.main_dialog.LABEL_SCENARIO_FREE.setText(self.tr(' Frei'))
        self.main_dialog.LABEL_SCENARIO_EXTREME.setText(self.tr(' Extrem'))

    def establish_onEdit_damage_potential_layer_slot(self, damage_potential_type, current_layer, old_layer_name):
        """Establishes connection between layer and onEdit_damage_potential_layer method and disconnects the old connections."""

        # 1) backup the old connection to be able to disconnect
        if hasattr(self, f"onEdit_damage_potential_layer_slot_{damage_potential_type}"):
            old_connection = getattr(self, f"onEdit_damage_potential_layer_slot_{damage_potential_type}")

            # 2) try to disconnect the old connection from the current layer
            if current_layer:
                if not current_layer.signalsBlocked():
                    try:
                        current_layer.editingStarted.disconnect(old_connection)
                        delattr(current_layer, f"econome_connection")
                        delattr(current_layer, "econome_status_widget")
                    except:
                        pass

            # 3) try to disconnect the old connection from the old layer
            if old_layer_name != "":
                search_results = self.qgis_project.mapLayersByName(old_layer_name)
                old_layer = search_results[0] if len(search_results) > 0 else None
                if old_layer:
                    if not old_layer.signalsBlocked():
                        try:
                            old_layer.editingStarted.disconnect(old_connection)
                        except TypeError:
                            pass
                        if hasattr(old_layer, "econome_connection") and hasattr(old_layer, "econome_status_widget"):
                            delattr(old_layer, "econome_connection")
                            delattr(old_layer, "econome_status_widget")

        # 4) create the new connection, save it to the class and connect it to the current layer
        if current_layer:
            new_connection = lambda: self.onEdit_damage_potential_layer(damage_potential_type, current_layer)
            current_layer.editingStarted.connect(new_connection)
            setattr(self, f"onEdit_damage_potential_layer_slot_{damage_potential_type}", new_connection)
            setattr(current_layer, f"econome_connection", new_connection)
            setattr(current_layer, "econome_status_widget",
                    getattr(self.main_dialog, f"IMAGE_LABEL_DAM_POT_STATUS_{damage_potential_type.upper()}"))

    def execute_in_separate_thread(self, task_function, thread_pipeline):
        """Uses a worker thread to execute a function in parallel."""

        self.THREAD_PIPELINE = thread_pipeline

        # CHECK: Save intersection results from consequence analysis ?
        if thread_pipeline == "upload_consequence_analysis":
            save_intersection_results_prompt = AreYouSure(self.tr("\nWollen Sie die Verschneidungsresultate der Konsequenzanalyse lokal speichern\n und/oder in die Karte laden ?\n"), pipeline=thread_pipeline, parent=self.main_dialog)
            user_choice = save_intersection_results_prompt.exec_()
            if user_choice == QDialog.Accepted:
                if save_intersection_results_prompt.button_action == "First":
                    self.SAVE_INTERSECTION_LAYERS_FROM_CONSEQUENCE_ANALYSIS = True
                    self.INTERSECTION_LAYERS_CONSEQUENCE_ANALYSIS = []
                    self.NEW_THREAD = WorkerThread(self, task_function, False)
                elif save_intersection_results_prompt.button_action == "Second":
                    self.SAVE_INTERSECTION_LAYERS_FROM_CONSEQUENCE_ANALYSIS = False
                    self.OPEN_SAVE_LAYERS_DIALOG = False
            else:
                return
        
        # Working animation
        self.animate_progress(True, thread_pipeline)

        # Run the "real" task in a QThread
        if self.THREAD_PIPELINE == "econome_api_switch":
            self.NEW_THREAD = WorkerThread(self, task_function, False)
            self.NEW_THREAD.revalidate_econome_user_info.connect(self.validate_econome_user_info)
        else:
            self.NEW_THREAD = WorkerThread(self, task_function, False)

        # Connect signals and start worker thread
        self.NEW_THREAD.setup_UI.connect(self.launch_plugin_UI_setup)
        self.NEW_THREAD.show_status.connect(self.status_in_separate_thread)
        self.NEW_THREAD.finished.connect(lambda: self.animate_progress(False, thread_pipeline))
        self.NEW_THREAD.start()

    def get_affected_assets(self, worker, damage_potential_input_dict, DAMAGE_POTENTIAL, SCENARIO_INPUT_DICT, PARAMETER_PROFILE, AFFECTED_ASSETS_DICT):
        
        # DO: get derived variables
        damage_potential_layer = damage_potential_input_dict["layer"]
        damage_potential_type = damage_potential_input_dict["type"]
        annualities_return_period = [self.main_dialog.LABEL_SCENARIO_FREE.text().strip(), "30", "100", "300",
                                     self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()]

        # DO: initialize storage variables

        SPECIAL_PARAMETER_DICT = {}

        # DO: Filter damage potential by object category
        OBJECT_CATEGORIES = [item for sublist in
                             [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for
                              obj_cat in
                              self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type]]
                             for item in sublist]
        AssetIdentID_user_name = [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if p["API"] == "AssetIdentID"][0]
        AssetTypeID_user_name = [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if p["API"] == "AssetTypeID"][0]
        FILTERED_DAMAGE_POTENTIAL = [asset for asset in DAMAGE_POTENTIAL if asset["AssetTypeID"] in OBJECT_CATEGORIES]

        # LOOP over scenarios of the current hazard process and for the current damage potential type
        for ann_rp, scenario_input_dict in SCENARIO_INPUT_DICT.items():
            # CHECK: Is the scenario active?
            if scenario_input_dict["status"] == -1:
                continue

            # get local variables once
            intensity_layer = scenario_input_dict["intensity_layer"]
            intensity_column = scenario_input_dict["intensity_column"]
            intensity_layer_name = intensity_layer.name()
            intensity_layer_fields = intensity_layer.fields()

            # %%# SPECIAL PARAMETERS ####
            special_parameters_in_layer = []
            for param in self.AFFECTED_ASSET_SPECIAL_PARAMETERS[damage_potential_type]:
                # CHECK: Is the special parameter in the layer
                if param[0] in intensity_layer_fields.names():
                    field = intensity_layer_fields.field(param[0])
                    # CHECK: Is the special parameter of the correct type?
                    if field.typeName().lower() in param[1]:
                        special_parameters_in_layer.append(param[0])
                    else:
                        # USER INFO: Special parameter fields with wrong type
                        user_info_title = self.tr("Fehler")
                        user_info_text = self.tr(
                            "\nDie Attributspalte '{special_parameter}' im Intensitäts-Layer '{intens_layer_name}' hat den falschen Datentyp. Typ muss {musthave_type} haben.\n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                            special_parameter=special_parameter, intens_layer_name=intensity_layer_name,
                            musthave_type=param[1])
                        worker.show_status.emit(user_info_title, user_info_text, 'error')
                        return None, None

            # Comments must be present
            for special_parameter in special_parameters_in_layer:

                if not special_parameter.endswith("_Comment"):
                    # CHECK: no parameter without comment
                    if f"{special_parameter}_Comment" not in special_parameters_in_layer:
                        # USER INFO: parameter found but no comment
                        user_info_title = self.tr("Fehler")
                        user_info_text = self.tr(
                            "\nDie Attributspalte '{special_parameter}_Comment' fehlt im Intensitäts-Layer '{intens_layer_name}'. Wenn '{special_parameter}' angepasst wird muss dies auch begründet werden.\n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                            special_parameter=special_parameter, intens_layer_name=intensity_layer_name)
                        worker.show_status.emit(user_info_title, user_info_text, 'error')
                        return None, None
                    comment_values = [feature[f"{special_parameter}_Comment"] for feature in
                                      intensity_layer.getFeatures()]
                    # CHECK: no empty comments
                    if any(value == NULL or value == "" for value in comment_values):
                        # USER INFO: Special parameter comment field empty
                        user_info_title = self.tr("Fehler")
                        user_info_text = self.tr(
                            "\nDie Attributspalte '{special_parameter}_Comment' im Intensitäts-Layer '{intens_layer_name}' enthält leere Begründungen. Wenn '{special_parameter}' angepasst wird muss dies auch begründet werden.\n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                            special_parameter=special_parameter, intens_layer_name=intensity_layer_name)
                        worker.show_status.emit(user_info_title, user_info_text, 'error')
                        return None, None
                # CHECK: no comment without parameter
                elif special_parameter.endswith("_Comment") and special_parameter.strip(
                        "_Comment") not in special_parameters_in_layer:
                    # USER INFO: comment found but no parameter
                    user_info_title = self.tr("Fehler")
                    user_info_text = self.tr(
                        "\nDie Attributspalte '{comment}' im Intensitäts-Layer '{intens_layer_name}' kann nicht übertragen werden, wenn nicht auch eine Attributsspalte {value} mitgegeben wird.\n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                        comment=special_parameter, intens_layer_name=intensity_layer_name,
                        value=special_parameter.strip("_Comment"))
                    worker.show_status.emit(user_info_title, user_info_text, 'error')
                    return None, None

            if special_parameters_in_layer:
                SPECIAL_PARAMETER_DICT[ann_rp] = special_parameters_in_layer

            # %%# INTERSECTIONS ####
            damage_potential_with_intensity = self.get_hazard_intensity(
                damage_potential_layer=damage_potential_layer,
                damage_potential_econome=FILTERED_DAMAGE_POTENTIAL,
                intensity_layer=intensity_layer,
                intensity_layer_column=intensity_column,
                intensity_layer_annuality=ann_rp,
                damage_potential_type=damage_potential_type,
                special_parameters_columns=special_parameters_in_layer,
                asset_ident_id_column=AssetIdentID_user_name,
                object_type_id_column=AssetTypeID_user_name,
            )

            if type(damage_potential_with_intensity) == QgsVectorLayer:
                SCENARIO_INPUT_DICT[ann_rp]["intensity_results"] = damage_potential_with_intensity
            elif isinstance(damage_potential_with_intensity, dict):
                # USER INFO: Special parameter comment field missing
                user_info_title = self.tr("Fehler beim Übertragen der IK-Spezialparameter")
                user_info_text = self.tr(
                    "\nFür folgende Schadenobjekte können die Spezialparameter {special_parameters_in_layer} im Szenario {annuality} nicht eindeutig zugeordnet werden. Diese Objekte durchschneiden IK-Layer Geometrien mit mit unterschiedlichen Werten in den Spezialparameter-Attributen: \n{duplicate_features}\n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                    special_parameters_in_layer=special_parameters_in_layer,
                    annuality=ann_rp,
                    duplicate_features=", ".join(damage_potential_with_intensity))
                worker.show_status.emit(user_info_title, user_info_text, 'error')
                return None, None
            elif isinstance(damage_potential_with_intensity, list):
                # USER INFO: Special parameter comment field missing
                user_info_title = self.tr("Fehler beim Übertragen der IK-Spezialparameter")
                user_info_text = self.tr(
                    "\nDie Spezialparameter {special_parameters_in_layer} im Szenario {annuality} können nicht übertragen werden, da im Schadenpotential-Layer {damage_potential_type} folgende Spalten existieren, und diese mit Spezialparametern im Intensitäts-Layer in Konflikt stehen: \n{conflicting_fields}\n\nLösche diese spalten im Schadenpotential-Layer, um fortzufahren. \n\nKonsequenzenanalyse wurde nicht hochgeladen.").format(
                    special_parameters_in_layer=special_parameters_in_layer,
                    annuality=ann_rp,
                    damage_potential_type=damage_potential_type,
                    conflicting_fields="\n - ".join(str(field) for field in damage_potential_with_intensity))
                worker.show_status.emit(user_info_title, user_info_text, 'error')
                return None, None

            # %%# PREPARE AFFECTED ASSETS DICTIONARY ####
            # Translate the intersection layer information into an api-readable dict
            AFFECTED_ASSETS = {"AffectedAssetList": []}

            if SCENARIO_INPUT_DICT[ann_rp]["intensity_results"]:
                
                # Get the features that are affected
                if damage_potential_type == "BUILDING":
                    intensity_max_column = intensity_column + "_max"
                    affected_features = [feature for feature in SCENARIO_INPUT_DICT[ann_rp][
                        "intensity_results"].getFeatures() if feature[intensity_max_column]]
                else:
                    affected_features = [feature for feature in SCENARIO_INPUT_DICT[ann_rp][
                        "intensity_results"].getFeatures() if
                                            feature['intensity_weak'] > 0 or feature[
                                                'intensity_moderate'] > 0 or feature[
                                                'intensity_strong'] > 0]
                            
                for feature in affected_features:
                    new_affected_asset = None
                    corresponding_asset = [asset for asset in FILTERED_DAMAGE_POTENTIAL if
                                            asset["AssetIdentID"] == str(
                                                feature[AssetIdentID_user_name])][0]
                    corresponding_asset_type_id = corresponding_asset["AssetTypeID"]

                    # _____________________ #
                    # ECONOME POINT OBJECTS #
                    # Kategorien Gebäude, Sonderobjekte, hier zusammengefasst in "BUILDING"
                    # affected is binary -> 1 intensity class as text
                    if damage_potential_type == "BUILDING":
                        intensity_max = int(feature[intensity_max_column])
                        intensity_class = "weak" if intensity_max == 1 else "moderate" if intensity_max == 2 else "strong" if intensity_max == 3 else None
                        new_affected_asset = {
                            "AssetID": corresponding_asset['AssetID'],
                            "IntensityClass": intensity_class
                        }
                    
                    # ____________________________ #
                    # ECONOME LINE OR AREA OBJECTS #
                    # Kategorien Strassenverkehr (line), Scheinverkehr (line), Leitungen (line), Mechanische Aufstiegshilfe (line), Landwirtschaft-Wald-Grünanalagen (area), Infrasturktur (area), Personen im Freien (area)
                    # affected is in asset unit -> 1 value for all 3 intensity classes [unit]
                    # unit can be [m, a, count, people] -> is adapted in the method get_hazard_intensity
                    else:
                        intensity_weak = float(feature['intensity_weak']) if feature['intensity_weak'] else float(0)
                        intensity_moderate = float(feature['intensity_moderate']) if feature['intensity_moderate'] else float(0)
                        intensity_strong = float(feature['intensity_strong']) if feature['intensity_strong'] else float(0)
                        
                        # only add affected asset if at least one intensity class is affected
                        if (intensity_weak + intensity_moderate + intensity_strong) > 0:
                            new_affected_asset = {
                                "AssetID": corresponding_asset['AssetID'],
                                "WeakIntensity": intensity_weak,
                                "ModerateIntensity": intensity_moderate,
                                "StrongIntensity": intensity_strong
                            }
                        
                    # __________________ #
                    # SPECIAL PARAMETERS #
                    # Add special parameters if they exist in the layer
                    for special_parameter in special_parameters_in_layer:
                        if feature[special_parameter] == NULL:
                            new_affected_asset[special_parameter] = feature[special_parameter]

                    if new_affected_asset:
                        AFFECTED_ASSETS['AffectedAssetList'].append(new_affected_asset)
                            
                if AFFECTED_ASSETS['AffectedAssetList']:
                    AFFECTED_ASSETS_DICT[ann_rp].extend(
                        AFFECTED_ASSETS['AffectedAssetList'])

        return AFFECTED_ASSETS_DICT, SPECIAL_PARAMETER_DICT

    def get_econome_mode(self):
        """Reads the EconoMe API name from the QGIS project scope."""

        try:  # Try to read EconoMe mode
            mode = QgsExpressionContextUtils.projectScope(self.qgis_project).variable('econome_mode')  # type: ignore
            if mode is not None:
                # Save in instance variable
                self.ECONOME_API_NAME = str(mode)
                # Adjust project and profile base path of the API
                self.set_econome_api_version()
        except Exception as e:
            log.warning(f"Couldn't get econome mode: {e}")
            pass

    def get_econome_project_number(self):
        """Reads the EconoMe project number from the QGIS project scope."""

        try:  # Try to read the project number and set the widget
            project_id = QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(f"econome_project_id_{self.ECONOME_API_NAME}")  # type: ignore
            if project_id is not None:
                selected_project_name = [info[1] for info in self.ECONOME_PROJECT_LIST if info[0] == project_id]
                if len(selected_project_name) != 0:
                    index = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.findText(
                        f"{selected_project_name[0]}, {project_id}")
                    if index != -1:
                        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentIndex(index)
                        return
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentIndex(0)
        except Exception as e:
            log.info(f"No previously used project number found: {e}")
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentIndex(0)

    def get_econome_user_info(self):
        """Writes and exchanges EconoMe user info to/with the QGIS Auth Manager."""

        try:  # Retrieve stored credentials

            auth_manager = QgsApplication.authManager()  # type: ignore
            found_authentication = False

            if not auth_manager.masterPasswordIsSet():
                auth_manager.setMasterPassword()

            # Check for available login methods
            for saved_config_id, config in auth_manager.availableAuthMethodConfigs().items():
                if config.name() == self.ECONOME_API_NAME or (config.name() == "Explore" and self.ECONOME_API_NAME == "EconoMe") or (config.name() == "EconoMe" and self.ECONOME_API_NAME == "Explore") or (self.ECONOME_API_NAME.startswith("Dev") and config.name().startswith("Dev")):
                    credentials_successfully_loaded = auth_manager.loadAuthenticationConfig(saved_config_id, config, True)
                    if credentials_successfully_loaded:
                        found_authentication = True
                        info_KeyA = credentials_successfully_loaded[1].config("api_key_a")
                        info_KeyB = credentials_successfully_loaded[1].config("api_key_b")
                        break

            if not found_authentication:
                log.info("No saved user info found!")
                info_KeyA = ""
                info_KeyB = ""

        except Exception as e:
            log.error(f"Couldn't retrieve stored user info: {e}")
            info_KeyA = ""
            info_KeyB = ""
        
        # Apply changes to UI
        self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.setText(info_KeyA)
        self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.setText(info_KeyB)

    def get_hazard_intensity(self, damage_potential_layer, damage_potential_econome, intensity_layer, intensity_layer_column, intensity_layer_annuality, damage_potential_type, special_parameters_columns=None, asset_ident_id_column=None, object_type_id_column = None):

        # Get infos for layer naming
        TIMESTAMP = datetime.now().strftime('%Y%m%d_%H%M%S')
        SELECTED_HAZARD_PROCESS = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText()
        SELECTED_MEASURE_VARIANT = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText()

        # Initialize intersection layer
        intersection_layer = None

        # _____________________ #
        # ECONOME POINT OBJECTS #
        # Kategorien Gebäude, Sonderobjekte, hier zusammengefasst in "BUILDING"
        # affected is binary -> 1 intensity class as text
        if damage_potential_type == "BUILDING":

            # Define the parameters for processing algorithm
            params = {
                'INPUT': damage_potential_layer,
                'JOIN': intensity_layer,
                'PREDICATE': [0],  # Spatial predicate: [0] means "intersects"
                'JOIN_FIELDS': [intensity_layer_column],
                'DISCARD_NONMATCHING': True,
                'PREFIX': '',
                'SUMMARIES': [3],  # Summary type: Maximum Value
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }

            context = dataobjects.createContext()  # Get context
            context.setInvalidGeometryCheck(QgsFeatureRequest.GeometryNoCheck)  # Don't check geometries for validity

            # Run the processing tool "Join by location (summary)"
            intersection_layer = processing.run("qgis:joinbylocationsummary", params, context=context)['OUTPUT']

        # ____________________ #
        # ECONOME LINE OBJECTS #
        # Kategorien Strassenverkehr, Scheinverkehr, Leitungen, Mechanische Aufstiegshilfe
        # affected is in asset unit -> 1 value for all 3 intensity classes [unit]
        elif damage_potential_layer.geometryType() == 1:

            # Create a copy of the input damage potential layer
            type_and_projection = "LineString?crs={}".format(damage_potential_layer.crs().authid())
            intersection_layer = QgsVectorLayer(type_and_projection, f"{damage_potential_layer.name()}_temp", "memory")
            damage_potential_layer_fields = damage_potential_layer.fields()

            intersection_layer.startEditing()
            intersection_layer.dataProvider().addAttributes(damage_potential_layer_fields)

            # Copy all features from the original layer to the new layer
            for feature in damage_potential_layer.getFeatures():
                intersection_layer.dataProvider().addFeature(feature)

            # Add new fields to the line layer to store overlay lengths
            intersection_layer.dataProvider().addAttributes([
                QgsField('intensity_weak', QVariant.Double),
                QgsField('intensity_moderate', QVariant.Double),
                QgsField('intensity_strong', QVariant.Double),
                QgsField('unit', QVariant.String)
            ])
            intersection_layer.updateFields()

            # Create a dictionary to store lengths for each line feature
            line_lengths = {f.id(): {'low': 0, 'moderate': 0, 'high': 0} for f in intersection_layer.getFeatures()}

            # Perform intersection, calculate line lengths and write line lengths to dictionary
            for line_feat in intersection_layer.getFeatures():
                line_geom = line_feat.geometry()
                for poly_feat in intensity_layer.getFeatures():
                    poly_geom = poly_feat.geometry()
                    # intensity = poly_feat[intensity_layer_column]
                    intersection = line_geom.intersection(poly_geom)  # Intersect line with polygon
                    if not intersection.isEmpty():
                        intensity_class = poly_feat[intensity_layer_column]
                        length = intersection.length()
                        if intensity_class == 1:
                            line_lengths[line_feat.id()]["low"] += length
                        elif intensity_class == 2:
                            line_lengths[line_feat.id()]["moderate"] += length
                        elif intensity_class == 3:
                            line_lengths[line_feat.id()]["high"] += length

            # Update line layer with the calculated lengths
            for line_feat in intersection_layer.getFeatures():
                line_feat['intensity_weak'] = line_lengths[line_feat.id()]['low']
                line_feat['intensity_moderate'] = line_lengths[line_feat.id()]['moderate']
                line_feat['intensity_strong'] = line_lengths[line_feat.id()]['high']
                # add unit
                if not self.ASSET_UNIT_BY_OBJECT_TYPEID:
                    # Create a dict of damage object units
                    get_assets_response = requests.get(
                        self.ECONOME_API_OPEN_BASE_PATH + '/asset',
                        auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD))
                    )
                    if get_assets_response.status_code == 200:
                        self.ASSET_UNIT_BY_OBJECT_TYPEID = {int(item['AssetTypeID']): item['Unit'] for item in get_assets_response.json()}
                    else:
                        log.warning("Couldn't get asset units per type ID!")
                unit = self.ASSET_UNIT_BY_OBJECT_TYPEID.get(int(line_feat[object_type_id_column]))
                line_feat['unit'] = unit
                # special units
                if unit != "m":
                    corresponding_asset = [asset for asset in damage_potential_econome if
                                            asset["AssetIdentID"] == str(
                                                line_feat[asset_ident_id_column])][0]
                    num = float(corresponding_asset['Count'])
                    total_size = float(feature.geometry().length()) 
                    # convert
                    line_feat['intensity_weak'] = line_feat['intensity_weak'] / total_size * num
                    line_feat['intensity_moderate'] = line_feat['intensity_moderate'] / total_size * num
                    line_feat['intensity_strong'] = line_feat['intensity_strong'] / total_size * num
                intersection_layer.updateFeature(line_feat)

            # Remove features with zero intensity in all classes
            features_to_delete = [feat.id() for feat in intersection_layer.getFeatures() if
                                    feat['intensity_weak'] == 0 and feat['intensity_moderate'] == 0 and
                                    feat['intensity_strong'] == 0]
            intersection_layer.dataProvider().deleteFeatures(features_to_delete)

            intersection_layer.commitChanges()  # Apply changes to layer

        # ____________________ #
        # ECONOME AREA OBJECTS #
        # Kategorien Landwirtschaft-Wald-Grünanalagen, Infrasturktur, Personen im Freien
        # affected is in asset unit -> 1 value for all 3 intensity classes [unit]
        elif damage_potential_layer.geometryType() == 2:

            # Create a copy of the input damage potential layer
            type_and_projection = f"Polygon?crs={damage_potential_layer.crs().postgisSrid()}"
            intersection_layer = QgsVectorLayer(type_and_projection, "Temporary Layer", "memory")
            damage_potential_layer_fields = damage_potential_layer.fields()

            intersection_layer.startEditing()
            intersection_layer.dataProvider().addAttributes(damage_potential_layer_fields)

            # Copy all features from the original layer to the new layer
            for feature in damage_potential_layer.getFeatures():
                intersection_layer.dataProvider().addFeature(feature)

            # Add new fields to the line layer to store overlay lengths
            intersection_layer.dataProvider().addAttributes([
                QgsField('intensity_weak', QVariant.Double),
                QgsField('intensity_moderate', QVariant.Double),
                QgsField('intensity_strong', QVariant.Double),
                QgsField('unit', QVariant.String)
            ])
            intersection_layer.updateFields()

            # Create a dictionary to store lengths for each line feature
            polygon_areas = {f.id(): {'low': 0, 'moderate': 0, 'high': 0} for f in intersection_layer.getFeatures()}

            # Perform intersection, calculate area and write line lengths to dictionary
            for poly_feat in intersection_layer.getFeatures():
                poly_geom = poly_feat.geometry()
                for intensity_feature in intensity_layer.getFeatures():
                    intensity_geom = intensity_feature.geometry()
                    # intensity = intensity_feature[intensity_layer_column]
                    intersection = poly_geom.intersection(intensity_geom)  # Intersect polygon with polygon
                    if not intersection.isEmpty():
                        intensity_class = intensity_feature[intensity_layer_column]
                        area = intersection.area() / 100 # m2 to a
                        if intensity_class == 1:
                            polygon_areas[poly_feat.id()]["low"] += area
                        elif intensity_class == 2:
                            polygon_areas[poly_feat.id()]["moderate"] += area
                        elif intensity_class == 3:
                            polygon_areas[poly_feat.id()]["high"] += area

            # Update the polygon layer with the calculated lengths
            for poly_feat in intersection_layer.getFeatures():
                
                poly_feat['intensity_weak'] = polygon_areas[poly_feat.id()]['low']
                poly_feat['intensity_moderate'] = polygon_areas[poly_feat.id()]['moderate']
                poly_feat['intensity_strong'] = polygon_areas[poly_feat.id()]['high']
                # add unit
                if not self.ASSET_UNIT_BY_OBJECT_TYPEID:
                    # Create a dict of damage object units
                    get_assets_response = requests.get(
                        self.ECONOME_API_OPEN_BASE_PATH + '/asset',
                        auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD))
                    )
                    if get_assets_response.status_code == 200:
                        self.ASSET_UNIT_BY_OBJECT_TYPEID = {int(item['AssetTypeID']): item['Unit'] for item in get_assets_response.json()}
                    else:
                        log.warning("Couldn't get asset units per object type ID!")
                unit = self.ASSET_UNIT_BY_OBJECT_TYPEID.get(int(poly_feat[object_type_id_column]))
                poly_feat['unit'] = unit
                # special units
                if unit != "a":
                    corresponding_asset = [asset for asset in damage_potential_econome if
                                            asset["AssetIdentID"] == str(
                                                poly_feat[asset_ident_id_column])][0]
                    num = float(corresponding_asset['Count'])
                    total_size = float(poly_feat.geometry().area())/100 # m2 to a
                    # convert
                    poly_feat['intensity_weak'] = poly_feat['intensity_weak'] / total_size * num
                    poly_feat['intensity_moderate'] = poly_feat['intensity_moderate'] / total_size * num
                    poly_feat['intensity_strong'] = poly_feat['intensity_strong'] / total_size * num
                intersection_layer.updateFeature(poly_feat)

            # Remove features with zero intensity in all classes
            features_to_delete = [feat.id() for feat in intersection_layer.getFeatures() if
                                  feat['intensity_weak'] == 0 and feat['intensity_moderate'] == 0 and
                                  feat['intensity_strong'] == 0]
            intersection_layer.dataProvider().deleteFeatures(features_to_delete)

            intersection_layer.commitChanges()
        
        # ___________________#
        # SPECIAL PARAMETERS #
        if special_parameters_columns:
            
            # CHECK: does the damage potential layer have a column that is also in the special parameters columns? 
            # this leads to a conflict e.g. the added column being p_rA_2 and the wrong/NULL values being inherited
            conflict_columns = [field_name for field_name in damage_potential_layer.fields().names() if field_name in special_parameters_columns]
            if conflict_columns:
                return conflict_columns
                
            params = {
                    'INPUT': damage_potential_layer,
                    'JOIN': intensity_layer,
                    'JOIN_FIELDS': special_parameters_columns,
                    'PREDICATE': [0],  # Spatial predicate: [0] means "intersects"
                    'METHOD': 0,  # Create separate feature for each matching feature (one-to-many join)
                    'PREFIX': '',
                    'OUTPUT': 'TEMPORARY_OUTPUT'
                }

            context = dataobjects.createContext()
            context.setInvalidGeometryCheck(QgsFeatureRequest.GeometryNoCheck)

            # Run the processing tools
            special_params_layer = processing.run("qgis:joinattributesbylocation", params, context=context)['OUTPUT']
            
            # test if duplicate special parameters exist: not allowed
            ambiguous_special_params = {}
            if special_params_layer.featureCount() > intersection_layer.featureCount():
                names = [f[asset_ident_id_column] for f in special_params_layer.getFeatures()]
                duplicate_names = [x for x in set(names) if names.count(x) > 1]
                if duplicate_names:
                    duplicate_assets_special_params = {}
                    for f in special_params_layer.getFeatures():
                        if f[asset_ident_id_column] in duplicate_names:
                            current_special_param_values = [f[field] for field in special_parameters_columns]
                            if f[asset_ident_id_column] not in duplicate_assets_special_params:
                                duplicate_assets_special_params[f[asset_ident_id_column]] = current_special_param_values
                            elif f[asset_ident_id_column] != current_special_param_values:
                                for i in range(len(current_special_param_values)):
                                    value_1 = current_special_param_values[i]
                                    value_2 = duplicate_assets_special_params[f[asset_ident_id_column]][i]
                                    field = special_parameters_columns[i]
                                    if value_1 != value_2:
                                        ambiguous_special_params[f[asset_ident_id_column]] = self.tr(f"Betroffen von unterschiedlichen Werten für {field}: {value_1} und {value_2}")
                                
                    if ambiguous_special_params:
                        return ambiguous_special_params
                
            # join the special parameters layer with the temp_layer
            intersection_layer.startEditing()
            
            params = {
                'INPUT': intersection_layer,
                'INPUT_2': special_params_layer,
                'FIELD': asset_ident_id_column,  # Field to match on
                'FIELD_2': asset_ident_id_column,  # Field to match on
                'FIELDS_TO_COPY': [],
                'METHOD': 1,
                'PREFIX': '',
                'OUTPUT': 'TEMPORARY_OUTPUT',
                'DISCARD_NONMATCHING': False
            }
            
            context = dataobjects.createContext()
            context.setInvalidGeometryCheck(QgsFeatureRequest.GeometryNoCheck)
            # Run the processing tool
            intersection_layer = processing.run("qgis:joinattributestable", params, context=context)['OUTPUT']
            
            intersection_layer.commitChanges()
        
        # Name layer
        intersection_layer.setName("Intersection_" + intensity_layer_annuality + "_" + damage_potential_layer.name() + "_" + "_".join(SELECTED_MEASURE_VARIANT.split(', ')[1][:25].split(" ")) + "_" + "_".join(SELECTED_HAZARD_PROCESS.split(', ')[2][:25].split(" ")) + "_" + TIMESTAMP)
        # Add layer to intersection layers list if correct pipeline
        if self.THREAD_PIPELINE == "upload_consequence_analysis" and self.SAVE_INTERSECTION_LAYERS_FROM_CONSEQUENCE_ANALYSIS is True:
            label_dam_pot_type_translated = self.main_dialog.findChild(QLabel, f"LABEL_DAM_POT_LAYER_{damage_potential_type}").text()
            self.INTERSECTION_LAYERS_CONSEQUENCE_ANALYSIS.append((intersection_layer, damage_potential_type, label_dam_pot_type_translated))
            
        return intersection_layer

    def get_project_status(self, worker):
        """Retrieves EconoMe project status information and fills plugin UI."""

        # Create project status dictionary
        PROJECT_STATUS = {
            "HazardProcesses": {
                "Count": 0
            },
            "Variants": {
                "Count": 0
            },
            "Scenarios": {
                "Active": []
            }
        }

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            self.ECONOME_PROJECT_NUMBER = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(", ")[-1]
            if self.ECONOME_PROJECT_NUMBER != "0":
                
                # Save the project selection

                API_GET_responses = self.multiple_API_GET_requests([
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/process',
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/variant'
                ])

                HAZARD_PROCESSES = API_GET_responses[0].result()[1]
                MEASURE_VARIANTS = API_GET_responses[1].result()[1]
                
                # CHECK: Response is a list
                if isinstance(HAZARD_PROCESSES, list):
                    PROJECT_STATUS["HazardProcesses"]["Count"] = len(HAZARD_PROCESSES)
                    self.main_dialog.LABEL_HAZARD_PROCESSES.setText(f"{self.tr('GEFAHRENPROZESS(E)')} ({len(HAZARD_PROCESSES)})")
                    if not self.PROCESS_TYPES:
                        # Get process types translations from EconoMe API
                        process_types_translations = requests.get(
                            self.ECONOME_API_GLOSSARY_BASE_PATH + '/processes',
                            auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD))
                        )
                        if process_types_translations.status_code == 200:
                            lang = self.ECONOME_PLUGIN_LANGUAGE_NAMES[self.ECONOME_PLUGIN_LANGUAGE]
                            self.PROCESS_TYPES = {pt['ProcessID']: pt[lang] for pt in process_types_translations.json()}
                        else:
                            log.warning(f"Couldn't get process types translations!")
                    hazard_processes_list = [(process["ProcessID"], self.PROCESS_TYPES[str(process["ProcessTypeID"])], process["ProcessName"]) for process in HAZARD_PROCESSES]
                    self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
                    # Fill the drop-down with all hazard processes
                    for hazard_process in hazard_processes_list:
                        if f"{hazard_process[0]}, {hazard_process[1]}, {hazard_process[2]}" not in [self.main_dialog.COMBO_BOX_HAZARD_PROCESS.itemText(i) for i in range(self.main_dialog.COMBO_BOX_HAZARD_PROCESS.count())]:
                            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.addItem(f"{hazard_process[0]}, {hazard_process[1]}, {hazard_process[2]}")
                    # Restore process selection from the last session
                    try:
                        selected_process = QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable('econome_process')
                        if selected_process is not None:
                            index_process = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.findText(selected_process)
                            if index_process != -1:
                                self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setCurrentIndex(index_process)
                            else:
                                self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setCurrentIndex(0)
                        else:
                            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setCurrentIndex(0)
                    except Exception as e:
                        log.info("Couldn't restore hazard process selection!")
                        log.warning(e)
                        self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setCurrentIndex(0)
                    # Update the scenarios and info
                    selected_hazard_process = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText()
                    selected_hazard_process_info = [hazard_process for hazard_process in HAZARD_PROCESSES if hazard_process["ProcessID"] == int(selected_hazard_process.split(",")[0])]
                    if len(selected_hazard_process_info) != 0:
                        # Set annualities for the two free scenarios
                        self.main_dialog.LABEL_SCENARIO_FREE.setText(str(selected_hazard_process_info[0]["AnnualityFree"]["Annuality"]))
                        self.main_dialog.LABEL_SCENARIO_EXTREME.setText(str(selected_hazard_process_info[0]["AnnualityExtreme"]["Annuality"]))

                else: # If not clear drop-down with hazard processes
                    HAZARD_PROCESSES = []
                    self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
                    self.main_dialog.LABEL_HAZARD_PROCESSES.setText(f"{self.tr('GEFAHRENPROZESS(E)')} ({len(HAZARD_PROCESSES)})")

                # CHECK: Response is a list
                if isinstance(MEASURE_VARIANTS, list):
                    PROJECT_STATUS["Variants"]["Count"] = len(MEASURE_VARIANTS)
                    self.main_dialog.LABEL_MEASURE_VARIANTS.setText(f"{self.tr('MASSNAHMENVARIANTE(N)')} ({PROJECT_STATUS['Variants']['Count']})")
                    self.main_dialog.COMBO_BOX_MEASURE_VARIANT.clear()
                    # Fill the drop-down with all variants
                    measure_variants_list = [f"{variant['VariantID']}, {variant['VariantName']}" for variant in MEASURE_VARIANTS]
                    before_variant_option = "0, " + self.tr("Vor Massnahme")
                    measure_variants_list.insert(0, before_variant_option)
                    self.main_dialog.COMBO_BOX_MEASURE_VARIANT.addItems(measure_variants_list)
                    
                    # Restore variant selection from the last session
                    try:
                        selected_variant = QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable('econome_variant')
                        if selected_variant is not None:
                            index_variant = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.findText(selected_variant)
                            if index_variant != -1:
                                self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setCurrentIndex(index_variant)
                            else:
                                self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setCurrentIndex(0)
                        else:
                            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setCurrentIndex(0)
                    except Exception as e:
                        log.info("Could not restore variant selection!")
                        log.error(e)
                        self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setCurrentIndex(0)
                    
                return PROJECT_STATUS
        
        # EconoMe user info not validated or project == 0
        self.empty_widgets()
        return {}

    def get_situation_info(self):
        # CHECK: User info validated
        if self.USER_INFO_VALIDATED:
            # get situation info
            self.ECONOME_PROJECT_NUMBER = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(", ")[-1]
            current_process_id = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            current_variant_id = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]

            # get process info
            get_process = requests.get(
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/process" + f"/{current_process_id}",
                auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD)))

            return current_process_id, current_variant_id, get_process

    def has_internet_connection(self, timeout=5):
        """Return True if there is internet connectivity."""

        try:
            response = requests.get("https://www.google.com", timeout=timeout)
            return True
        except requests.RequestException as e:
            log.info("No internet connection!")
            log.error(e)
            return False
        
    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        self.add_action(
            icon_path,
            text=self.tr(u'EconoMe'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def launch_plugin(self, worker):
        """Starts up the plugin."""

        # ## CHECK INTERNET CONNECTIVITY
        # -------------------------------
        if not self.has_internet_connection():
            worker.show_status.emit(self.tr("Fehler"), f"\nIt seems that you're not connected to the Internet 🛜🚫.\nThe plugin requires a functioning internet connection!\n", "warning")

        # Get the EconoMe mode from the QGIS project scope
        self.get_econome_mode()

        # Get EconoMe user info from the QGIS Auth Manager
        self.get_econome_user_info()
        self.validate_econome_user_info()

        # Setup plugin in sized junks
        time.sleep(0.5)

        if self.main_dialog is not None:  # CHECK: Does plugin window exist?

            worker.setup_UI.emit('General setup & project restoration')
            time.sleep(1.5)
            worker.setup_UI.emit('Damage potential')
            time.sleep(0.5)
            worker.setup_UI.emit('Situation dropdowns & Intensity layers')
            time.sleep(0.5)
            worker.setup_UI.emit('Consequence Analysis & Risk calculation')

    def launch_plugin_UI_setup(self, task):
        """Prepares the plugin UI."""

        if task == 'General setup & project restoration':

            # ## ECONOME PROJECT LOGIN & SETTINGS MENU
            # ------------------------------------------

            # UI tweaks
            self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setCursor(QCursor(Qt.PointingHandCursor))  # Pointing cursor
            self.main_dialog.BUTTON_PLUGIN_SETTINGS.setCursor(QCursor(Qt.PointingHandCursor))  # Pointing cursor

            # CONNECTIONS
            try:  # Disconnect (if existing connections/signals)
                self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.clicked.disconnect()
                self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.textChanged.disconnect()
                self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.textChanged.disconnect()
                self.main_dialog.BUTTON_PLUGIN_SETTINGS.clicked.disconnect()
            except TypeError:
                pass

            self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.clicked.connect(self.validate_econome_user_info)  # CONNECT: Validate user info
            self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.textChanged.connect(self.onChange_econome_user_info)  # CONNECT: Re-validate when user info has changed
            self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.textChanged.connect(self.onChange_econome_user_info)  # CONNECT: Re-validate when user info has changed
            self.main_dialog.BUTTON_PLUGIN_SETTINGS.clicked.connect(self.change_settings)  # CONNECT: Change settings

            # ## PROJECT STATUS & SELECTION
            # ------------------------------

            # UI modifications
            self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.setCursor(QCursor(Qt.PointingHandCursor))  # Pointing cursor
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setMaximumWidth(525)  # Set max width

            # Create auto-complete helper for project list
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setEditable(True)  # Make combo box editable
            project_list_completer = QCompleter(self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.model(), self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST)  # Initialize QCompleter
            project_list_completer.setCompletionMode(QCompleter.PopupCompletion)  # PopupCompletion
            project_list_completer.setFilterMode(Qt.MatchContains)  # Partial matching

            # Styling auto-complete popup
            new_list_view = QListView()
            new_list_view.setMouseTracking(True)
            new_list_view.setFocusPolicy(Qt.StrongFocus)
            new_list_view.setAttribute(Qt.WA_Hover, True)
            new_list_view.setStyleSheet("""
                QListView::item:hover {
                    background-color: #d0f0ff;
                }
                QListView::item:selected {
                    background-color: #3399ff;
                    color: white;
                }
            """)
            project_list_completer.setPopup(new_list_view)
            project_list_completer.popup().setStyleSheet("")

            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCompleter(project_list_completer)  # Set completer on combo box

            # Set background of QCompleter LineEdit (important for dark mode)
            line_edit = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.lineEdit()
            line_edit.setStyleSheet("""
                background: white;
                color: black;
                border: none;
            """)

            # CONNECTIONS
            try:  # Disconnect (if existing connections/signals)
                self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.clicked.disconnect()
                project_list_completer.activated[str].disconnect()
                self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.activated[str].disconnect()
            except TypeError:
                pass

            self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.clicked.connect(lambda: self.execute_in_separate_thread(self.get_project_status,"get_project_status"))  # CONNECT: Reload project details
            project_list_completer.activated[str].connect(self.onChange_project_selection)  # CONNECT: Update details on project selection change
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.activated[str].connect(self.onChange_project_selection)  # CONNECT: Update details on project selection change

            # Get the project selection from the QGIS project scope else set it to no selection
            self.get_econome_project_number()

            # Get process types translations from EconoMe API
            process_types_translations = requests.get(
                self.ECONOME_API_GLOSSARY_BASE_PATH + '/processes',
                auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD))
            )
            if process_types_translations.status_code == 200:
                lang = self.ECONOME_PLUGIN_LANGUAGE_NAMES[self.ECONOME_PLUGIN_LANGUAGE]
                self.PROCESS_TYPES = {pt['ProcessID']:pt[lang] for pt in process_types_translations.json()}
            else:
                log.warning(f"Couldn't get process types translations!")

            # manually trigger project update
            self.onChange_project_selection(startup=True)  # manually call here because the connection over activated[str].connect() doesnt seem to take the setCurrentIndex signal

            # Try to restore the project layer selection from the last session (stored in QGIS project variables)
            try:
                econome_dpl = QgsExpressionContextUtils.projectScope(self.qgis_project).variable('econome_dpl').split("|")  # type: ignore
                self.SELECTED_DAMAGE_POTENTIAL_LAYERS = {i.split(":")[0]: i.split(":")[1] for i in econome_dpl}
            except (AttributeError, TypeError, IndexError, ValueError) as e:
                log.info("Couldn't restore last project layer selection!")
                log.error(e)
                self.SELECTED_DAMAGE_POTENTIAL_LAYERS = {}
            try:
                econome_iml = QgsExpressionContextUtils.projectScope(self.qgis_project).variable('econome_iml').split("|")  # type: ignore
                self.SELECTED_INTENSITY_MAP_LAYERS = {i.split(":")[0]: i.split(":")[1] for i in econome_iml}
            except (AttributeError, TypeError, IndexError, ValueError) as e:
                log.info("Couldn't restore last project layer selection!")
                log.error(e)
                self.SELECTED_INTENSITY_MAP_LAYERS = {}
            try:
                econome_ic = QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable('econome_ic').split("|")  # type: ignore
                self.SELECTED_INTENSITY_COLUMNS = {i.split(":")[0]: i.split(":")[1] for i in econome_ic}
            except (AttributeError, TypeError, IndexError, ValueError) as e:
                log.info("Couldn't restore last project layer selection!")
                log.error(e)
                self.SELECTED_INTENSITY_COLUMNS = {}

        elif task == "Damage potential":

            # Create a dict of damage object units
            get_assets_response = requests.get(
                self.ECONOME_API_OPEN_BASE_PATH + '/asset',
                auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD))
            )

            if get_assets_response.status_code == 200:
                self.ASSET_UNIT_BY_OBJECT_TYPEID = {int(item['AssetTypeID']): item['Unit'] for item in get_assets_response.json()}
            else:
                log.warning("Couldn't get asset units per object type ID!")

            # Create a list of damage potential labels
            self.DAMAGEPOTENTIAL_LABELS = [
                self.main_dialog.LABEL_DAM_POT_LAYER_BUILDING,
                self.main_dialog.LABEL_DAM_POT_LAYER_ROAD,
                self.main_dialog.LABEL_DAM_POT_LAYER_RAILWAY,
                self.main_dialog.LABEL_DAM_POT_LAYER_CABLE_TRANSPORT,
                self.main_dialog.LABEL_DAM_POT_LAYER_LINE_PIPE,
                self.main_dialog.LABEL_DAM_POT_LAYER_PEOPLE_OUTDOOR,
                self.main_dialog.LABEL_DAM_POT_LAYER_OTHER_ASSET
            ]

            self.DAMAGEPOTENTIAL_LABEL_FILTERS = []  # Store filters to prevent GC

            # UI tweaks
            for label in self.DAMAGEPOTENTIAL_LABELS:
                label.setCursor(QCursor(Qt.PointingHandCursor))
                label.setToolTip(self.tr("Mit Rechtsklick eine GIS-Layer-Vorlage herunterladen!"))
                new_right_click_filter = RightClickFilter(self.create_template_layer)
                label.installEventFilter(new_right_click_filter)
                self.DAMAGEPOTENTIAL_LABEL_FILTERS.append(new_right_click_filter)

            try:  # Disconnect (if existing connections/signals)
                self.main_dialog.LAYER_DAM_POT_BUILDING.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_ROAD.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_RAILWAY.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_LINE_PIPE.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.layerChanged.disconnect()
                self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.layerChanged.disconnect()
                self.main_dialog.BUTTON_PP_BUILDING.clicked.disconnect()
                self.main_dialog.BUTTON_PP_ROAD.clicked.disconnect()
                self.main_dialog.BUTTON_PP_RAILWAY.clicked.disconnect()
                self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.clicked.disconnect()
                self.main_dialog.BUTTON_PP_LINE_PIPE.clicked.disconnect()
                self.main_dialog.BUTTON_PP_PEOPLE_OUTDOOR.clicked.disconnect()
                self.main_dialog.BUTTON_PP_OTHER_ASSET.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_PEOPLE_OUTDOOR.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_PEOPLE_OUTDOOR.clicked.disconnect()
                self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.clicked.disconnect()
            except TypeError:
                pass

            # Establish layer change events on damage potential layer dropdowns
            self.main_dialog.LAYER_DAM_POT_BUILDING.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_ROAD.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_RAILWAY.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.layerChanged.connect(self.onChange_damage_potential_layer)
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.layerChanged.connect(self.onChange_damage_potential_layer)

            # Establish damage potential attribute profiles button connections & pointing cursor
            self.main_dialog.BUTTON_PP_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_BUILDING.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_BUILDING.setEnabled(False)
            self.main_dialog.BUTTON_PP_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_ROAD.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_ROAD.setEnabled(False)
            self.main_dialog.BUTTON_PP_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_RAILWAY.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_RAILWAY.setEnabled(False)
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.BUTTON_PP_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_LINE_PIPE.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_LINE_PIPE.setEnabled(False)
            self.main_dialog.BUTTON_PP_PEOPLE_OUTDOOR.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_PEOPLE_OUTDOOR.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_PEOPLE_OUTDOOR.setEnabled(False)
            self.main_dialog.BUTTON_PP_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_PP_OTHER_ASSET.clicked.connect(self.set_parameter_profiles)
            self.main_dialog.BUTTON_PP_OTHER_ASSET.setEnabled(False)

            # Establish damage potential synchronization button connections & pointing cursor
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer(),
                                                          "building",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING))
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer(), "road",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD))
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer(),
                                                          "railway",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY))
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer(),
                                                          "cable_transport",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT))
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer(),
                                                          "line_pipe",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE))
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_PEOPLE_OUTDOOR.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_PEOPLE_OUTDOOR.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.currentLayer(),
                                                          "people_outdoor",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_PEOPLE_OUTDOOR))
            self.main_dialog.BUTTON_DAM_POT_SYNC_PEOPLE_OUTDOOR.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.clicked.connect(
                lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer(),
                                                          "other_asset",
                                                          self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET))
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.setEnabled(False)

            # Establish click connections on damage potential delete button connections & pointing cursor
            self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_PEOPLE_OUTDOOR.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_PEOPLE_OUTDOOR.clicked.connect(self.delete_damage_potential_assets)
            self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.clicked.connect(self.delete_damage_potential_assets)

            # Only allow vector layers and add an empty item as default to each map layer combo box
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES = {}
            self.main_dialog.LAYER_DAM_POT_BUILDING.setFilters(QgsMapLayerProxyModel.PolygonLayer | QgsMapLayerProxyModel.PointLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["building"] = "Vector Polygon Layer, Vector Point Layer"
            self.main_dialog.LAYER_DAM_POT_ROAD.setFilters(QgsMapLayerProxyModel.LineLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["road"] = "Vector Line Layer"
            self.main_dialog.LAYER_DAM_POT_RAILWAY.setFilters(QgsMapLayerProxyModel.LineLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["railway"] = "Vector Line Layer"
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setFilters(QgsMapLayerProxyModel.LineLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["cable_transport"] = "Vector Line Layer"
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setFilters(QgsMapLayerProxyModel.LineLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["line_pipe"] = "Vector Line Layer"
            self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["people_outdoor"] = "Vector Polygon Layer"
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.DAMAGE_POTENTIAL_GEOMETRY_TYPES["other_asset"] = "Vector Polygon Layer"

            # Set saved layer names for damage potential
            if self.USER_INFO_VALIDATED:
                for damage_potential_type, saved_layer_name in self.SELECTED_DAMAGE_POTENTIAL_LAYERS.items():
                    combo_box_layer_widget = getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type}")
                    try:
                        found_layers = self.qgis_project.mapLayersByName(saved_layer_name)
                        if len(found_layers) == 1:
                            combo_box_layer_widget.setLayer(found_layers[0])
                            self.establish_onEdit_damage_potential_layer_slot(damage_potential_type, found_layers[0], "")
                    except (AttributeError, TypeError, RuntimeError) as e:
                        log.info("Couldn't set saved layer names for damage potential!")
                        log.error(e)

                self.update_group_box_damage_potential()

                # Get glossary
                with requests.get(
                        self.ECONOME_API_GLOSSARY_BASE_PATH,
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as get_glossary:
                    self.ECONOME_API_GLOSSARY = get_glossary.json()

                # Set tooltips for damage potential layer combo boxes
                epl = self.ECONOME_PLUGIN_LANGUAGE
                language = "German" if epl == "DE" else "French" if epl == "FR" else "Italian" if epl == "IT" else "English"
                for damage_potential_type in self.DAMAGE_POTENTIAL_LAYER_STATUS.keys():
                    damage_potential_type_translated = getattr(self.main_dialog, f"LABEL_DAM_POT_LAYER_{damage_potential_type}").text()
                    must_have_fields = [gl[language] for gl in self.ECONOME_API_GLOSSARY if gl["APITerm"] in self.DAMAGEPOTENTIAL_REQUIRED_FIELDS[damage_potential_type.lower()]]
                    must_have_fields_str = ", ".join(must_have_fields)
                    geom_type = self.DAMAGE_POTENTIAL_GEOMETRY_TYPES[damage_potential_type.lower()]
                    getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type.upper()}").setToolTip(self.tr(f"Schadenpotential {damage_potential_type_translated} \n{geom_type}\nPflichtattribute ({len(must_have_fields)}): {must_have_fields_str}\nOptionale Attribute: siehe Parameter-Profil"))

        elif task == "Situation dropdowns & Intensity layers":

            # SITUATION DROPDOWNS
            # ---------------------

            try:  # Disconnect any existing dropdown events (to avoid errors when plugin is re-opened)
                self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentIndexChanged.disconnect()
                self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentIndexChanged.disconnect()
            except TypeError:
                pass

            # Establish dropdown events on situation dropdowns
            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentIndexChanged.connect(self.update_situation_status)
            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentIndexChanged.connect(self.update_situation_status)

            # ## INTENSITY LAYERS
            # ---------------------

            try:  # Disconnect any existing layer change events (to avoid errors when plugin is re-opened)
                self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.layerChanged.disconnect()
                self.main_dialog.LAYER_INTENSITY_SCENARIO_30.layerChanged.disconnect()
                self.main_dialog.LAYER_INTENSITY_SCENARIO_100.layerChanged.disconnect()
                self.main_dialog.LAYER_INTENSITY_SCENARIO_300.layerChanged.disconnect()
                self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.layerChanged.disconnect()
            except TypeError:
                pass

            # Only allow polygon layers for intensity maps
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setFilters(QgsMapLayerProxyModel.PolygonLayer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setFilters(QgsMapLayerProxyModel.PolygonLayer)

            # Establish layer change events on intensity layers dropdowns
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.layerChanged.connect(self.onChange_intensity_layer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.layerChanged.connect(self.onChange_intensity_layer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.layerChanged.connect(self.onChange_intensity_layer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.layerChanged.connect(self.onChange_intensity_layer)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.layerChanged.connect(self.onChange_intensity_layer)

            # Connect the intensity map layer column dropdowns to the update function
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.currentIndexChanged.connect(
                self.onChange_intensity_column)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.currentIndexChanged.connect(
                self.onChange_intensity_column)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.currentIndexChanged.connect(
                self.onChange_intensity_column)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.currentIndexChanged.connect(
                self.onChange_intensity_column)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.currentIndexChanged.connect(
                self.onChange_intensity_column)

            # Deactivate the intensity map layer column dropdowns until the intensity layers are selected
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.setEnabled(False)

            # See which scenarios are active and activate/deactivate the widgets accordingly and set saved layer names for intensity maps
            self.update_situation_status()

            # Establish connections on intensity layers delete buttons
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.clicked.connect(
                lambda: self.delete_damage_potential_affected_assets(
                    self.main_dialog.LABEL_SCENARIO_FREE.text().strip()))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.clicked.connect(
                lambda: self.delete_damage_potential_affected_assets("30"))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.clicked.connect(
                lambda: self.delete_damage_potential_affected_assets("100"))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.clicked.connect(
                lambda: self.delete_damage_potential_affected_assets("300"))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.clicked.connect(
                lambda: self.delete_damage_potential_affected_assets(
                    self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()))

        elif task == "Consequence Analysis & Risk calculation":

            # ## RISK CALCULATION AND VISUALIZATION
            # --------------------------------------

            try:  # Disconnect any existing button connections (to avoid errors when plugin is re-opened)
                self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.clicked.disconnect()
                self.main_dialog.BUTTON_VISUALIZE_RISKS.clicked.disconnect()
            except TypeError:
                pass

            # Establish risk calculation and visualization button connections & pointing cursor
            self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.clicked.connect(
                lambda: self.execute_in_separate_thread(self.upload_consequence_analysis,
                                                        "upload_consequence_analysis"))
            self.main_dialog.BUTTON_VISUALIZE_RISKS.setCursor(QCursor(Qt.PointingHandCursor))
            self.main_dialog.BUTTON_VISUALIZE_RISKS.clicked.connect(
                lambda: self.execute_in_separate_thread(self.visualize_risks, "visualize_risks")
            )

    def multiple_API_GET_requests(self, URL_GET_REQUESTS):
        """Execute multiple API requests at once."""

        def fetch(url):
            
            try:
                response = requests.get(
                    url,
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                )
                return url, response.json()
            except Exception as e:
                log.info("Couldn't retrieve multiple API requests!")
                log.error(e)
                return url, f"Error: {e}"

        with ThreadPoolExecutor(max_workers=5) as executor:
            futures = [executor.submit(fetch, url) for url in URL_GET_REQUESTS]
            return futures

    def onChange_damage_potential_layer(self):
        # Get sender info
        sender_widget = self.main_dialog.sender()
        current_layer = sender_widget.currentLayer()
        current_layer_name = sender_widget.currentText()
        damage_potential_type = sender_widget.objectName().split("DAM_POT_")[-1]

        # get the old layer info
        if damage_potential_type in self.SELECTED_DAMAGE_POTENTIAL_LAYERS:
            old_layer_name = self.SELECTED_DAMAGE_POTENTIAL_LAYERS[damage_potential_type]
        else:
            old_layer_name = ""

        # update the status label
        if current_layer_name == "" or self.ECONOME_PROJECT_NUMBER == "0":
            self.update_dpl_status(damage_potential_type, -1, statustip_text=self.tr("Kein Layer ausgewählt."))
            getattr(self.main_dialog, f"BUTTON_PP_{damage_potential_type.upper()}").setEnabled(False)
            getattr(self.main_dialog, f"BUTTON_DAM_POT_SYNC_{damage_potential_type.upper()}").setEnabled(False)
        else:
            self.update_dpl_status(damage_potential_type, 0,
                                   statustip_text=self.tr("Layer neu ausgewählt. Nicht synchronisiert."))
            self.establish_onEdit_damage_potential_layer_slot(damage_potential_type, current_layer, old_layer_name)
            getattr(self.main_dialog, f"BUTTON_PP_{damage_potential_type.upper()}").setEnabled(True)
            getattr(self.main_dialog, f"BUTTON_DAM_POT_SYNC_{damage_potential_type.upper()}").setEnabled(True)

        # save selected layer name
        current_layer_name = self.main_dialog.sender().currentText()
        self.SELECTED_DAMAGE_POTENTIAL_LAYERS.update({damage_potential_type: current_layer_name})

    def update_group_box_damage_potential(self):
        """Collapse or expand the damage potential group box based on selected layers."""

        special_categories_selected = any(
            getattr(self.main_dialog, f"LAYER_DAM_POT_{dp.upper()}").currentText() != ""
            for dp in ["cable_transport", "line_pipe", "people_outdoor", "other_asset"]
        )
        # if no special categories are selected, the group box can be collapsed
        self.main_dialog.GROUP_BOX_DAMAGE_POTENTIAL.setCollapsed(not special_categories_selected)
        
    def onChange_econome_user_info(self):
        """Resets plugin UI when EconoMe username or password is modified."""

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # EconoMe user info -> not validated
            self.USER_INFO_VALIDATED = False
            # Clear project combo box
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
            # Disable buttons and dropdowns
            self.disable_buttons_and_dropdowns(True)
            # Display messsage -> not connected
            # self.main_dialog.LABEL_ECONOME_PROJECT_NUMBER.setText(self.tr("Projektstatus "))
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(self.tr("Nicht verbunden"))
            # Remove green color from login button
            self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet("background-color: none;")
            # Get project status
            self.execute_in_separate_thread(self.get_project_status, "get_project_status")

    def onChange_intensity_column(self):

        sender_widget = self.main_dialog.sender()
        intensity_column = sender_widget.currentText()
        annuality = sender_widget.objectName().split("COMBO_BOX_INTENSITY_COLUMN_NAME_")[-1]
        if intensity_column != "":
            intensity_layer = getattr(self.main_dialog, f"LAYER_INTENSITY_SCENARIO_{annuality}").currentLayer()
            intensity_field = intensity_layer.fields().field(intensity_column)
            if intensity_field.type() in self.QVARIANT_TYPES_DICT["integer"] + self.QVARIANT_TYPES_DICT["float"]:
                intensity_values = [f[intensity_column] for f in intensity_layer.getFeatures()]
                intensity_unique_values = list(set(intensity_values))
                if all(value in [0, 1, 2, 3] for value in intensity_unique_values):
                    self.update_scenario_status(annuality, 1, sender="intensity_column",
                                            info_text=self.tr("Die Intensitäts-Spalte ist gültig."))
                else:
                    self.update_scenario_status(annuality, 0, sender="intensity_column",
                                            message_title=self.tr("Fehler"), info_text=self.tr(
                            f"\nUngültige Identitäts-Spalte.\nIntensitäts-Klassen müssen der BAFU-Skala entsprechen: [(0), 1, 2, 3].\nDie ausgewählte Spalte {intensity_column} hat den Wertebereich {intensity_unique_values}.\n"))
            else:
                self.update_scenario_status(annuality, 0, sender="intensity_column", message_title=self.tr("Fehler"),
                                        info_text=self.tr(
                                            f"\nUngültige Identitäts-Spalte.\nIntensitäts-Spalte muss eine Zahl sein!\nDie ausgewählte Spalte {intensity_column} hat den Typ {intensity_field.typeName()}.\n"))

            # save selected intensity column name
            current_process_id = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            current_variant_id = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]
            if current_variant_id != "" and current_process_id != "":
                key = f"{current_process_id}_{current_variant_id}_{annuality}"
                self.SELECTED_INTENSITY_COLUMNS.update({key: intensity_column})
        else:
            self.update_scenario_status(annuality, 0, sender="intensity_column", info_text=self.tr("Keine Intensitäts-Spalte ausgewählt."))

    def onChange_intensity_layer(self):

        self.CHANGING_INTENSITY_LAYER = True
        sender_widget = self.main_dialog.sender()
        annuality = sender_widget.objectName().split("INTENSITY_SCENARIO_")[-1]

        # fill dropdown with layer fields
        current_layer = sender_widget.currentLayer()
        self.update_intensity_layer_dependent_widgets(annuality, current_layer)

        # save selected layer name
        current_layer_name = sender_widget.currentText()
        current_process_id = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
        current_variant_id = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]
        if current_variant_id != "" and current_process_id != "":
            key = f"{current_process_id}_{current_variant_id}_{annuality}"
            self.SELECTED_INTENSITY_MAP_LAYERS.update({key: current_layer_name})
        self.CHANGING_INTENSITY_LAYER = False

    def onChange_project_selection(self, startup=False):
        """Switches project number and updates EconoMe user info."""

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # CHECK: Project selected
            if self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText():
                # CHECK: Project selection changed
                if self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(', ')[-1] != self.ECONOME_PROJECT_NUMBER or startup:
                    self.ECONOME_PROJECT_NUMBER = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(', ')[-1]
                    # CHECK: No selection
                    if not self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(', ')[-1] == "0":
                        # Valid project selected
                        self.disable_buttons_and_dropdowns(False)
                        self.execute_in_separate_thread(self.get_project_status, "get_project_status")
                        self.update_dpl_status_all()
                    else:
                        # No project selected
                        self.empty_widgets()
                        self.disable_buttons_and_dropdowns(True)

    def on_dialog_closed(self):
        """Closes main dialog properly."""

        # Save the selections of this session (except project number)
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_mode", self.ECONOME_API_NAME)
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_dpl", "|".join(f"{key}:{value}" for key, value in self.SELECTED_DAMAGE_POTENTIAL_LAYERS.items()))
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_iml", "|".join(f"{key}:{value}" for key, value in self.SELECTED_INTENSITY_MAP_LAYERS.items()))
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_ic", "|".join(f"{key}:{value}" for key, value in self.SELECTED_INTENSITY_COLUMNS.items()))
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_process", self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText())
        QgsExpressionContextUtils.setProjectVariable(self.qgis_project, "econome_variant", self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText())
        
        # 
        if not self.THREAD_PIPELINE == "econome_api_switch":
            QgsExpressionContextUtils.setProjectVariable(self.qgis_project, f"econome_project_id_{self.ECONOME_API_NAME}", self.ECONOME_PROJECT_NUMBER)
            
        # Disconnect all layer change signals
        for damage_potential_type in self.DAMAGE_POTENTIAL_LAYER_STATUS.keys():
            if damage_potential_type in self.SELECTED_DAMAGE_POTENTIAL_LAYERS:
                layer = getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type.upper()}").currentLayer()
                if layer:
                    if not layer.signalsBlocked():
                        try:
                            layer.editingStarted.disconnect(getattr(self, f"onEdit_damage_potential_layer_slot_{damage_potential_type}"))
                        except TypeError:
                            continue

        # Delete the dialog
        self.main_dialog.deleteLater()
        self.main_dialog = None
        self.first_start = True

    def onEdit_damage_potential_layer(self, damage_potential_type, layer):
        """Changes the damage potential status state and symbol when the layer is starting to be edited. """

        econome_status_widget = getattr(layer, "econome_status_widget")
        # case all exists: change the status
        if self is not None and econome_status_widget is not None:
            if not sip.isdeleted(econome_status_widget):
                if not self.main_dialog is None:
                    self.update_dpl_status(damage_potential_type, 0, statustip_text=self.tr("Layer in Bearbeitungsmodus."))
                    return

        # case either dialog or widget doesn't exist: disconnect
        layer.editingStarted.disconnect(getattr(layer, "econome_connection"))
        delattr(layer, "econome_connection")
        delattr(layer, "econome_status_widget")

    def open_main_dialog(self):
        """This method creates the main plugin dialog."""

        # Initialize main plugin class
        self.main_dialog = EconoMeDialog(self.iface.mainWindow())  # IMPORTANT: Parent it to the main window
        
    def run(self):
        """This method is the first method called when the plugin opens."""

        # ## PLUGIN LAUNCH
        # ------------------

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started

        if self.first_start:

            self.first_start = False  # Don't open plugin two times
            self.open_main_dialog()  # Open dialog
            self.main_dialog.setWindowFlags(Qt.Window | Qt.WindowMinimizeButtonHint | Qt.WindowCloseButtonHint)  # Support minimizing the dialog window
            self.main_dialog.finished.connect(self.on_dialog_closed)  # Proper clean-up when dialog is closed
            self.disable_buttons_and_dropdowns(True)  # Disable dialog elements until user is logged in

            log.info("Plugin loaded!")
            log.info(f"Display language: {self.ECONOME_PLUGIN_LANGUAGE}")
            log.info(f"QGIS Version: {Qgis.QGIS_VERSION}")  # type: ignore

            # Reference to GIS project
            self.qgis_project = QgsProject.instance()  # type: ignore

            self.main_dialog.show()  # Show main dialog
            self.empty_widgets()  # Clear all widgets in the dialog

            # Set dialog size based on system OS
            if sys.platform == 'darwin':  # For Mac OS
                self.main_dialog.setFixedSize(580, 620)
            else:  # For Windows and others
                self.main_dialog.setFixedSize(570, 610)

            # Transform main dialog into scrollable window
            # -----

            # 1. Create a content wrapper and reparent UI
            self.main_dialog.content = QWidget()
            content_layout = QVBoxLayout(self.main_dialog.content)

            # Move ALL the dialog's existing widgets into this wrapper
            while self.main_dialog.layout().count():
                item = self.main_dialog.layout().takeAt(0)
                if item.widget():
                    content_layout.addWidget(item.widget())
                elif item.layout():
                    content_layout.addLayout(item.layout())

            content_layout.addStretch()

            # 2. Create a scroll area
            self.main_dialog.scroll = QScrollArea()
            self.main_dialog.scroll.setStyleSheet("""
                QScrollBar:vertical {
                    width: 10px;               /* Width of the vertical scrollbar */
                    margin: 0px 0px 0px 0px;   /* Optional: margins around the scrollbar */
                }
            """)
            self.main_dialog.scroll.setWidgetResizable(True)
            self.main_dialog.scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            self.main_dialog.scroll.setFixedWidth(self.main_dialog.width() - 20)
            self.main_dialog.content.setFixedWidth(self.main_dialog.scroll.width())
            self.main_dialog.scroll.setWidget(self.main_dialog.content)

            # 3. Place scroll area back into the dialog
            self.main_dialog.layout().addWidget(self.main_dialog.scroll)

            # ---------------------------------------------

            # Make pixmap for warning and success icons
            path_to_checkmark = os.path.join(self.plugin_dir, "icons", "checkmark.png")
            pixmap_checkmark = QPixmap(path_to_checkmark)
            self.scaled_pixmap_checkmark = pixmap_checkmark.scaled(
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
                Qt.KeepAspectRatio,
                Qt.SmoothTransformation)
            path_to_warning = os.path.join(self.plugin_dir, "icons", "warning.png")
            pixmap_warning = QPixmap(path_to_warning)
            self.scaled_pixmap_warning = pixmap_warning.scaled(
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
                Qt.KeepAspectRatio,
                Qt.SmoothTransformation)

        else:  # Plugin was reopened
            log.info("Prevented reload with plugin already open!")
            self.main_dialog.raise_()
            self.main_dialog.activateWindow()
            return

        # Call 'launch_plugin' function in a separate thread
        if self.THREAD_PIPELINE == "econome_api_switch":
            self.execute_in_separate_thread(self.launch_plugin, "econome_api_switch")
            log.info("API changed by user!")
        else:
            self.execute_in_separate_thread(self.launch_plugin, "launch_plugin")

    def set_econome_api_version(self):
        """Applies user-defined settings."""

        self.ECONOME_API_OPEN_BASE_PATH = "https://api.econome.ch/VOP2"

        # Set new API references
        if self.ECONOME_API_NAME == "EconoMe":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/V2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/V2/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api.econome.ch/V2/glossary"

        elif self.ECONOME_API_NAME == "Explore":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/VEX2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/VEX2/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api.econome.ch/VEX2/glossary"

        elif self.ECONOME_API_NAME == "Road-Risk":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/RR1/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/RR1/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api.econome.ch/RR1/glossary"

        elif self.ECONOME_API_NAME == "Dev-EconoMe":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api-dev.econome.ch/V2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api-dev.econome.ch/V2/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api-dev.econome.ch/V2/glossary"


        elif self.ECONOME_API_NAME == "Dev-Explore":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api-dev.econome.ch/VEX2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api-dev.econome.ch/VEX2/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api-dev.econome.ch/VEX2/glossary"

        elif self.ECONOME_API_NAME == "Dev-Road-Risk":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api-dev.econome.ch/RR1/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api-dev.econome.ch/RR1/profile"
            self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api-dev.econome.ch/RR1/glossary"

    def set_parameter_profiles(self):
        """Handles parameter profile dialogs."""
        pp_type = self.main_dialog.sender().objectName().split('PP_')[-1].lower()

        DAMAGE_POTENTIAL_LAYER = None
        dialog = None

        # Get glossary
        with requests.get(
                self.ECONOME_API_GLOSSARY_BASE_PATH,
                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
        ) as get_glossary:
            GLOSSARY = get_glossary.json()

        title = self.tr("EconoMe") + " - "
        if pp_type == "building":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['building'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['building'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Gebäude")

        elif pp_type == "road":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['road'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['road'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Strassen")

        elif pp_type == "railway":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['railway'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['railway'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Schiene")

        elif pp_type == "cable_transport":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['cable_transport'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['cable_transport'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Mech. Aufstiegshilfe")

        elif pp_type == "line_pipe":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['line_pipe'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['line_pipe'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Leitungen")

        elif pp_type == "people_outdoor":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['people_outdoor'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['people_outdoor'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Personen im Freien")

        elif pp_type == "other_asset":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['other_asset'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['other_asset'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Andere Objekte")
        
        if dialog:

            dialog.setWindowTitle(title)
            labels = dialog.findChildren(QLabel)

            # Disable scrolling through combo box options
            for comboBox in dialog.findChildren(QComboBox):
                disable_scroll_filter = WheelFilter(comboBox)
                comboBox.installEventFilter(disable_scroll_filter)

            # find all fields in the layer
            fields = DAMAGE_POTENTIAL_LAYER.fields()
            integer_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["integer"]]
            number_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["float"]]
            string_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["string"]]
            bool_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["boolean"]]

            # Retrieve parameter profile
            with requests.get(
                    self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_parameter_profile:
                PARAMETER_PROFILE = get_parameter_profile.json()
                PARAMETER_PROFILE['Empty'] = False

            # if there is no parameter profile, create an empty one to have the type information
            if 'Code' in PARAMETER_PROFILE:
                empty_parameter_profile = {
                    "ParameterList": [{"API": label.objectName().split('_')[1], "User": ""} for label in labels]}
                with requests.post(
                        self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                        json=empty_parameter_profile,
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as post_empty_parameter_profile:
                    PARAMETER_PROFILE = post_empty_parameter_profile.json()
                    PARAMETER_PROFILE['Empty'] = True

            # Fill damage potential layer field names in combo boxes and select a parameter if set in the parameter profile
            for label in labels:
                
                attribute = label.objectName().split('_')[1]
                matching_parameter = [item for item in PARAMETER_PROFILE['ParameterList'] if item["API"] == attribute][0]
                parameter_type = matching_parameter['Type']
                label.setText(f"{label.text()} [{parameter_type}]")
                allowed_fields = integer_fields if parameter_type == "integer" else number_fields if parameter_type == "float" else string_fields if parameter_type == "string" else bool_fields if parameter_type == "boolean" else []
                matching_comboBox = dialog.findChild(QComboBox, f"COMBO_BOX_{attribute}")
                matching_comboBox.addItem("")
                matching_comboBox.addItems([field.name() for field in allowed_fields])
                if not PARAMETER_PROFILE['Empty']:
                    if f"{matching_parameter['User'].strip()}" in [field.name() for field in fields]:
                        index = matching_comboBox.findText(f"{matching_parameter['User'].strip()}")
                        matching_comboBox.setCurrentIndex(index)

            result = dialog.exec_()
            if result:
                # CREATE: Updated parameter profile dictionary
                updated_parameter_profile = {"ParameterList": []}
                # Loop over all user input combo boxes
                for comboBox in dialog.findChildren(QComboBox):
                    if not PARAMETER_PROFILE['Empty']:
                        existing_parameter_pair = [item for item in PARAMETER_PROFILE['ParameterList'] if
                                                   item["API"] == comboBox.objectName().split("_")[-1]][0]
                        if existing_parameter_pair["User"] != comboBox.currentText():
                            updated_parameter_profile["ParameterList"].append(
                                {"API": existing_parameter_pair["API"], "User": comboBox.currentText()})
                    else:
                        parameter_name = comboBox.objectName().split("_")[-1]
                        updated_parameter_profile["ParameterList"].append(
                            {"API": parameter_name, "User": comboBox.currentText()})

                if len(updated_parameter_profile["ParameterList"]) != 0:

                    if not PARAMETER_PROFILE['Empty']:
                        with requests.put(
                                self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                                json=updated_parameter_profile,
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as put_new_parameter_profile:
                            # USER INFO
                            if put_new_parameter_profile.status_code == 200:
                                self.update_dpl_status(pp_type, 0, statustip_text=self.tr("Attributspalten im Parameterprofil wurden aktualisiert. Bitte neu synchronisieren."))
                                StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg"), message_text=self.tr("\nAttributspalten für {category} erfolgreich aktualisiert!\n").format(category=dialog.windowTitle().split(': ')[-1])).exec_()

                            else:
                                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=f"\n{put_new_parameter_profile.json()}\n").exec_()
                    else:
                        with requests.put(
                                self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                                json=updated_parameter_profile,
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as put_new_parameter_profile:
                            # USER INFO
                            if put_new_parameter_profile.status_code == 200:
                                StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg"), message_text=self.tr("\nAttributspalten für {category} erfolgreich angelegt!\n").format(category=dialog.windowTitle().split(': ')[-1])).exec_()
                            else:
                                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=f"\n{put_new_parameter_profile.json()}\n").exec_()

    def show_save_or_load_dialog(self):
        """Shows save & load dialog."""

        if self.THREAD_PIPELINE == "upload_consequence_analysis":
            SaveLoadDialog(layers=self.INTERSECTION_LAYERS_CONSEQUENCE_ANALYSIS, pipeline=self.THREAD_PIPELINE).exec_()

        if self.THREAD_PIPELINE == "visualize_risks":
            SaveLoadDialog(layers=self.RISK_OUTPUT_TEMP_LAYERS, pipeline=self.THREAD_PIPELINE).exec_()

    def status_in_separate_thread(self, title, text, pixmap_type):
        """Shows status dialog inside worker thread."""

        if pixmap_type == 'success':
            StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=title, message_text=text).exec_()
        else:
            StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=title, message_text=text).exec_()

    def switch_language(self, locale_code):
        # Uninstall previous translator if any
        if self.translator:
            QCoreApplication.removeTranslator(self.translator)

        # Attempt to load the new translation
        plugin_dir = os.path.dirname(__file__)
        qm_path = os.path.join(plugin_dir, 'i18n', f"{locale_code.lower()}.qm")

        if os.path.exists(qm_path):
            self.translator = QTranslator()
            self.translator.load(qm_path)
            QCoreApplication.installTranslator(self.translator)

    def synchronize_damage_potential(self, damage_potential_layer, damage_potential_type, status_label):
        """This method synchronizes the damage potential with the EconoMe API."""

        # Check for valid user info
        if self.USER_INFO_VALIDATED:
            # Get damage potential layer
            DAMAGE_POTENTIAL_LAYER = damage_potential_layer
            # Check if damage potential layer is valid
            if DAMAGE_POTENTIAL_LAYER:
                # Check if damage potential layer is not in toggle edit mode
                if not DAMAGE_POTENTIAL_LAYER.isEditable():
                    # Damage potential type
                    DAMAGE_POTENTIAL_TYPE = damage_potential_type
                    DAMAGE_POTENTIAL_LABEL_TEXT = getattr(self.main_dialog, f"LABEL_DAM_POT_LAYER_{DAMAGE_POTENTIAL_TYPE.upper()}").text()
                    # Layer Asset list
                    LAYER_ASSETS = {'AssetList': []}
                    # Get current damage potential
                    with requests.get(
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                    ) as get_damage_potential:
                        UPLOADED_DAMAGE_POTENTIAL = get_damage_potential.json()
                    # Filter damage potential by object category
                    OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[DAMAGE_POTENTIAL_TYPE.upper()]] for item in sublist]
                    FILTERED_UPLOADED_DAMAGE_POTENTIAL = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetTypeID"] in OBJECT_CATEGORIES]
                    # Retrieve parameter profile and get parameter associations
                    with (requests.get(
                            self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{DAMAGE_POTENTIAL_TYPE}",
                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                    ) as get_parameter_profile):
                        PARAMETER_PROFILE = get_parameter_profile.json()
                        # Check for parameter profile
                        if get_parameter_profile.status_code == 200:
                            # Get valid parameter pairs
                            damage_potential_valid_parameter_pairs = [pp for pp in PARAMETER_PROFILE['ParameterList'] if pp["User"].strip()]
                            damage_potential_valid_parameter_keys = [pp["API"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["User"].strip()]
                            # Check if required fields were filled out
                            if set(self.DAMAGEPOTENTIAL_REQUIRED_FIELDS[damage_potential_type]).issubset(damage_potential_valid_parameter_keys):
                                AssetIdentID_user_name = [pp["User"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["API"].strip() == "AssetIdentID"][0]
                                AssetTypeID_user_name = [pp["User"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["API"].strip() == "AssetTypeID"][0]
                                layer_fields_names = [field.name() for field in DAMAGE_POTENTIAL_LAYER.fields()]
                                user_field_names = [pp["User"] for pp in damage_potential_valid_parameter_pairs]
                                # Check if the pp defined fields are present in the damage potential layer -> without this check, the user will not notice that the layer is not synchronizing correctly
                                if set(user_field_names).issubset(set(layer_fields_names)):
                                    typechecked_user_field_names = [pp["User"] for pp in damage_potential_valid_parameter_pairs if DAMAGE_POTENTIAL_LAYER.fields().field(DAMAGE_POTENTIAL_LAYER.fields().indexFromName(pp["User"])).type() in self.QVARIANT_TYPES_DICT[pp["Type"]]]
                                    # Check field types
                                    if set(user_field_names).issubset(typechecked_user_field_names):
                                        connected_columns_string = self.tr("Verknüpfte Spalten ({num_connected_cols}):\n - ").format(num_connected_cols=len(typechecked_user_field_names))
                                        connected_columns_string += '\n - '.join(typechecked_user_field_names)
                                        # Check that all features have an AssetIdentID
                                        AssetIdentID_list = [feature[AssetIdentID_user_name] for feature in DAMAGE_POTENTIAL_LAYER.getFeatures()]
                                        if all(AssetIdentID_list):
                                            # Create asset list from layer
                                            for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                                new_asset = {}
                                                for parameter_pair in damage_potential_valid_parameter_pairs:
                                                    if parameter_pair["Type"] == "string":
                                                        if feature[parameter_pair["User"]] == NULL:
                                                            new_asset[parameter_pair["API"]] = None
                                                        elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                            new_asset[parameter_pair["API"]] = str(feature[parameter_pair["User"]].value())
                                                        else:
                                                            new_asset[parameter_pair["API"]] = str(feature[parameter_pair["User"]])
                                                    elif parameter_pair["Type"] == "integer":
                                                        if feature[parameter_pair["User"]] == NULL:
                                                            new_asset[parameter_pair["API"]] = None
                                                        elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                            new_asset[parameter_pair["API"]] = int(feature[parameter_pair["User"]].value())
                                                        else:
                                                            new_asset[parameter_pair["API"]] = int(feature[parameter_pair["User"]])
                                                    elif parameter_pair["Type"] == "float":
                                                        if feature[parameter_pair["User"]] == NULL:
                                                            new_asset[parameter_pair["API"]] = None
                                                        elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                            new_asset[parameter_pair["API"]] = EconoMe.truncFloat(float(feature[parameter_pair["User"]].value()))
                                                        else:
                                                            new_asset[parameter_pair["API"]] = EconoMe.truncFloat(float(feature[parameter_pair["User"]]))
                                                    elif parameter_pair["Type"] == "boolean":
                                                        if feature[parameter_pair["User"]] == NULL:
                                                            new_asset[parameter_pair["API"]] = None
                                                        elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                            new_asset[parameter_pair["API"]] = bool(feature[parameter_pair["User"]].value())
                                                        else:
                                                            new_asset[parameter_pair["API"]] = bool(feature[parameter_pair["User"]])
                                                LAYER_ASSETS["AssetList"].append(new_asset)
                                            # Create list of asset ident ids
                                            layer_asset_ident_ids = [asset["AssetIdentID"] for asset in LAYER_ASSETS["AssetList"]]
                                            set_layer_asset_ident_ids = set(layer_asset_ident_ids)
                                            if len(layer_asset_ident_ids) == len(set_layer_asset_ident_ids):
                                                # Compare layer assets and uploaded damage potential
                                                comparison_result, mismatch_dict = self.compare_damage_potential(LAYER_ASSETS, FILTERED_UPLOADED_DAMAGE_POTENTIAL, damage_potential_type, damage_potential_valid_parameter_pairs)
                                                if comparison_result == "CANCEL":
                                                    self.update_dpl_status(damage_potential_type, 0, message_text=self.tr("\nKonflikte zwischen Schadenobjekten im Layer und Schadenobjekten in EconoMe: \nKeine Option ausgewählt.\n"))
                                                elif comparison_result == "INVALID OBJECT CATEGORY":
                                                    text = self.tr("\nEine oder mehrere Objektart-ID im Layer passt nicht zur Kategorie des Schadenpotentials!\n")
                                                    text+= "\n".join([f"{key}: {value}" for key, value in mismatch_dict.items()])
                                                    self.update_dpl_status(damage_potential_type, 0, message_text=text)
                                                elif comparison_result == "POST LAYER ASSETS":  # when no econome assets are present
                                                    # Post new damage potential from layer
                                                    
                                                    with requests.post(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                            json=LAYER_ASSETS,
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as post_damage_potential:
                                                        if post_damage_potential.status_code == 200:
                                                            created_assets = json.loads(post_damage_potential.text)["AssetList"]
                                                            text = connected_columns_string
                                                            text += self.tr("\n\n{new_assets_count} neue Schadenobjekte wurden erfolgreich vom Layer auf EconoMe hochgeladen:\n\n").format(new_assets_count=len(created_assets))
                                                            for i, posted_asset_dict in enumerate(created_assets):
                                                                text += f"{i + 1}) "
                                                                text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=posted_asset_dict['AssetIdentID'])
                                                            self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                        else:
                                                            self.update_dpl_status(damage_potential_type, 0, message_text=self.tr("\nAPI-Fehlermeldung beim Hochladen des Schadenpotentials:\n{response}\n").format(response=json.loads(post_damage_potential.text)))
                                                # Layer assets match completely with assets in EconoMe
                                                elif comparison_result == "COMPLETE MATCH":  # when all attributes match
                                                    text = connected_columns_string
                                                    text += self.tr("\n\nLayer-Schadenobjekte stimmen mit EconoMe-Schadenobjekten überein.\n")
                                                    self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                
                                                # Update EconoMe assets
                                                elif comparison_result == "KEEP LAYER ASSETS":  # when user decides to overwrite econome assets with layer assets
                                                    """
                                                    ATTENTION: 
                                                    Only valid parameter pairs should be overwritten as the user might have changed attributes in EconoMe that are not present in the layer
                                                    Therefore we put the layer attributes (instead of deleting and re-posting) into the EconoMe API.
                                                    New layer assets are posted to EconoMe.
                                                    """
                                                    # 1) collect assets to patch/put/post in lists of dicts with correct API keys
                                                    # 1.1) Assets where the type was changed in layer: we need to patch and then post them, thereby we need to preserve all EconoMe-sided stored attributes
                                                    layer_assets_to_change_type = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and "AssetTypeID" in mismatch_dict[layer_asset["AssetIdentID"]].keys() ]
                                                    uploaded_assets_to_change_type = [asset for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and "AssetTypeID" in mismatch_dict[asset["AssetIdentID"]].keys()]
                                                    patch_asset_ids = [asset["AssetID"] for asset in uploaded_assets_to_change_type]
                                                    assets_to_change_type = []
                                                    for uploaded_asset in copy.deepcopy(uploaded_assets_to_change_type):
                                                        asset_to_change_type = uploaded_asset
                                                        for key, value in mismatch_dict[uploaded_asset["AssetIdentID"]].items():
                                                            asset_to_change_type[key] = value["Layer"]
                                                        asset_to_change_type.pop("AssetID") # not supported by API post
                                                        assets_to_change_type.append(asset_to_change_type)
                                                    # 1.2) Assets that exist in EconoMe: we need to update/put them
                                                    layer_assets_to_put = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[layer_asset["AssetIdentID"]] != {"Missing":self.tr("Fehlt in EconoMe.")} and "AssetTypeID" not in mismatch_dict[layer_asset["AssetIdentID"]].keys()]
                                                    assets_to_put = []
                                                    assets_to_put_status_code = 0
                                                    for asset_to_put in copy.deepcopy(layer_assets_to_put):
                                                        asset_to_put.pop("AssetTypeID") # not supported by API put (see above assets_to_change_type)
                                                        asset_to_put.update({"AssetID": [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == asset_to_put["AssetIdentID"]][0]})
                                                        asset_to_put.pop("AssetIdentID") # not supported by API put
                                                        assets_to_put.append(asset_to_put)
                                                    # 1.3) Assets that were newly created in the layer: we need to post them
                                                    assets_to_post = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[layer_asset["AssetIdentID"]] == {"Missing":self.tr("Fehlt in EconoMe.")}]
                                                    assets_to_post_status_code = 0
                                                    # 2) patch/put/post the assets
                                                    # 2.1) Patch assets where the type was changed in layer
                                                    if len(assets_to_change_type) != 0:
                                                        with requests.patch(
                                                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                                                json={"AssetIDList":patch_asset_ids},
                                                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                        ) as patch_damage_potential:
                                                            with requests.post(
                                                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                                    json={"AssetList":assets_to_change_type},
                                                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                            ) as reput_damage_potential:
                                                                # Feedback to user about the syc errors or success
                                                                if patch_damage_potential.status_code == 200 and reput_damage_potential.status_code == 200:
                                                                    text = connected_columns_string
                                                                    text += self.tr("Der Objekttyp folgender {category} Schadenobjekte wurde auf EconoMe mit Objekttyp aus Layer überschrieben.\n").format(category=len(assets_to_change_type))
                                                                    text += self.tr("Dafür musste jedes Schadenobjekt auf EconoMe gelöscht werden und neu erstellt werden. Alle Attribute wurden erhalten.\n\n")
                                                                    for i, reput_asset_dict in enumerate(json.loads(reput_damage_potential.text)["AssetList"]):
                                                                        text += f" {i + 1}) "
                                                                        text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=reput_asset_dict['AssetIdentID'])
                                                                    self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                                else:
                                                                    text = ""
                                                                    if patch_damage_potential.status_code != 200:
                                                                        text = self.tr("\nAPI-Fehlermeldung beim Updaten des Objekttyps (Schritt 1 patch):\n{response}\n").format(response=json.loads(patch_damage_potential.text))
                                                                    elif reput_damage_potential.status_code != 200:
                                                                        text += self.tr("\nAPI-Fehlermeldung beim Updaten des Objekttyps (Schritt 2 post):\n{response}\n").format(response=json.loads(reput_damage_potential.text))
                                                                    self.update_dpl_status(damage_potential_type, 0, message_text=text)
                                                    # 2.2) Put updated damage potential from layer
                                                    if len(assets_to_put) != 0:
                                                        with requests.put(
                                                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                                json={"AssetList": assets_to_put},
                                                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                        ) as put_damage_potential:
                                                            assets_to_put_status_code = put_damage_potential.status_code
                                                            # Feedback to user about the syc errors or success
                                                            if put_damage_potential.status_code == 200:
                                                                text = connected_columns_string
                                                                text += self.tr("\n\nDie Attribute folgender {new_assets_length} Schadenobjekte wurden auf EconoMe mit Attributen aus dem Layer überschrieben:\n\n").format(new_assets_length=len(assets_to_put))
                                                                text += "\n"
                                                                for i, put_asset_dict in enumerate(layer_assets_to_put):
                                                                    text += f" {i + 1}) "
                                                                    text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=put_asset_dict['AssetIdentID'])
                                                                self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                            else:
                                                                self.update_dpl_status(damage_potential_type, 0, message_text=self.tr("\nAPI-Fehlermeldung beim Updaten des vorhandenen Schadenpotentials (put):\n{response}\n").format(response=json.loads(put_damage_potential.text)))
                                                    # 2.3) Post new damage potential from layer
                                                    if len(assets_to_post) != 0:
                                                        with requests.post(
                                                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                                json={"AssetList":assets_to_post},
                                                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                        ) as post_damage_potential:
                                                            assets_to_post_status_code = post_damage_potential.status_code
                                                            # Feedback to user about the syc errors or success
                                                            if post_damage_potential.status_code == 200:
                                                                # for newly created assets, we want to save the standard-created attributes from EconoMe to the layer
                                                                DAMAGE_POTENTIAL_LAYER.startEditing()
                                                                for posted_asset_dict in json.loads(post_damage_potential.text)["AssetList"]:
                                                                    expression = f'"{AssetIdentID_user_name}" = \'{posted_asset_dict["AssetIdentID"]}\''
                                                                    request = QgsFeatureRequest().setFilterExpression(expression)
                                                                    feature = next(DAMAGE_POTENTIAL_LAYER.getFeatures(request), None)
                                                                    for parameter_pair in damage_potential_valid_parameter_pairs:
                                                                        feature.setAttribute(parameter_pair["User"], posted_asset_dict[parameter_pair["API"]])
                                                                        DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                                DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                                # user feedback
                                                                text = connected_columns_string
                                                                text += self.tr("\n\n{new_assets_length} neue Schadenobjekte wurden vom Layer auf EconoMe hochgeladen:\n\n").format(new_assets_length=len(assets_to_post))
                                                                for i, posted_asset_dict in enumerate(json.loads(post_damage_potential.text)["AssetList"]):
                                                                    text += f" {i + 1}) "
                                                                    text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=posted_asset_dict['AssetIdentID'])
                                                                self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                            else:
                                                                self.update_dpl_status(damage_potential_type, 0, message_text=self.tr("\nAPI-Fehlermeldung beim Hochladen des neuen Schadenpotentials (post):\n{response}\n").format(response=json.loads(post_damage_potential.text)))
                                                    # Checkmark should only be changed if both requests were successful
                                                    if len(assets_to_put) != 0 and len(assets_to_post) != 0:
                                                        if assets_to_post_status_code != 200 or assets_to_put_status_code != 200:
                                                            self.update_dpl_status(damage_potential_type, 0, statustip_text=self.tr("Es gab Fehler beim Synchronisieren des Schadenpotentials. Bitte erneut synchronisieren."))
                                                # Update Layer assets
                                                elif comparison_result == "KEEP ECONOME ASSETS":
                                                    # Change layer assets based on EconoMe
                                                    DAMAGE_POTENTIAL_LAYER.startEditing()
                                                    updated_assetIdentIDs = []
                                                    deleted_assetIdentIDs = []
                                                    for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                                        AssetIdentID = feature[AssetIdentID_user_name]
                                                        if AssetIdentID in mismatch_dict.keys():
                                                            corresponding_uploaded_asset = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[AssetIdentID_user_name])]
                                                            if len(corresponding_uploaded_asset) != 0:
                                                                for parameter_pair in damage_potential_valid_parameter_pairs:
                                                                    feature.setAttribute(parameter_pair["User"], corresponding_uploaded_asset[0][parameter_pair["API"]])
                                                                    DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                                updated_assetIdentIDs.append(AssetIdentID)
                                                            else:
                                                                DAMAGE_POTENTIAL_LAYER.dataProvider().deleteFeatures([feature.id()])
                                                                deleted_assetIdentIDs.append(AssetIdentID)
                                                    DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                    text = connected_columns_string
                                                    if len(updated_assetIdentIDs) != 0:
                                                        text += self.tr("\n\nDie Attribute folgender Schadenobjekte wurden im Layer mit Attributen aus EconoMe überschrieben:\n\n")
                                                        for i, AssetIdentID in enumerate(updated_assetIdentIDs):
                                                            text += f" {i + 1}) "
                                                            text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=AssetIdentID)
                                                    if len(deleted_assetIdentIDs) != 0:
                                                        text += self.tr("\n\nFolgende {len_objects} Schadenobjekte wurden im Layer gelöscht:\n\n").format(len_objects=len(deleted_assetIdentIDs))
                                                        for i, AssetIdentID in enumerate(deleted_assetIdentIDs):
                                                            text += f" {i + 1}) "
                                                            text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=AssetIdentID)
                                                    self.update_dpl_status(damage_potential_type, 1, message_text=text)
                                                elif comparison_result == "KEEP ECONOME ASSET TYPE IDS":
                                                    # Change layer assets based on EconoMe
                                                    DAMAGE_POTENTIAL_LAYER.startEditing()
                                                    for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                                        corresponding_uploaded_asset = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[AssetIdentID_user_name])]
                                                        if len(corresponding_uploaded_asset) != 0:
                                                            feature.setAttribute(AssetTypeID_user_name, corresponding_uploaded_asset[0]["AssetTypeID"])
                                                            DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                    DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                    StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Layer-Objektart-IDs {DAMAGE_POTENTIAL_LABEL_TEXT} zurückgesetzt!").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("Bitte erneut synchronisieren!")).exec_()
                                                    self.update_dpl_status(damage_potential_type, 0)

                                                elif comparison_result == "DELETE MISSING ASSETS":
                                                    # Delete category-filtered damage potential assets from EconoMe
                                                    patch_asset_ids = [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[asset["AssetIdentID"]] == {"Missing":"Fehlt im Layer."}]
                                                    patch_assetIdentIDs = [asset["AssetIdentID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[asset["AssetIdentID"]] == {"Missing":"Fehlt im Layer."}]
                                                    with requests.patch(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                                            json={"AssetIDList": patch_asset_ids},
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as patch_damage_potential:
                                                        if patch_damage_potential.status_code == 200:
                                                            text = connected_columns_string
                                                            text += self.tr("\nFolgende {len_objects} Schadenobjekte wurden erfolgreich von EconoMe gelöscht:\n").format(len_objects=len(patch_asset_ids))
                                                            for i, deleted_assetIdentID in enumerate(patch_assetIdentIDs):
                                                                text += f" {i + 1}) "
                                                                text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=deleted_assetIdentID)
                                                            text += self.tr("\n\nEs wird noch nach Attribut-Konflikten gesucht...")
                                                            StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                            self.synchronize_damage_potential(DAMAGE_POTENTIAL_LAYER, DAMAGE_POTENTIAL_TYPE, status_label)
                                                elif comparison_result == "NO CHANGE":
                                                    # USER INFO: No changes made
                                                    self.update_dpl_status(damage_potential_type, 0, message_title=self.tr("Hinweis"), message_text=self.tr("\nBitte fehlende Schadenobjekte im Layer hinzufügen!\n"))
                                                else:
                                                    # USER INFO: internal issue unknown option
                                                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Interner Fehler"), message_text=self.tr("\nUnbekannte Option ausgewählt.\n")).exec_()
                                            else:
                                                # USER INFO: Some of the layer features have the same AssetIdentID
                                                duplicate_AssetIdentIDs = [item for item, count in Counter(layer_asset_ident_ids).items() if count > 1]
                                                text = self.tr("\nFolgende AssetIdentIDs kommen im Layer mehrmals vor:\n")
                                                for i, duplicate_AssetIdentID in enumerate(duplicate_AssetIdentIDs):
                                                    text += f" {i + 1}) "
                                                    text += QCoreApplication.translate("EconoMe","Objekt {asset_ident_id} \n").format(asset_ident_id=duplicate_AssetIdentID)
                                                text += self.tr("\n\nBitte nur eindeutige AssetIdentIDs verwenden!\n")
                                                self.update_dpl_status(damage_potential_type, 0, message_text=text)
                                        else:
                                            # USER INFO: Some of the layer features have no AssetIdentID / Null values for AssetIdentID
                                            assets_missing_AssetIdentID = [feature for feature in DAMAGE_POTENTIAL_LAYER.getFeatures() if not feature[AssetIdentID_user_name]]
                                            text = self.tr("{len_objects} Schadenobjekte im Layer haben keine AssetIdentID! \nBitte AssetIdentID hinzufügen!\n").format(len_objects=len(assets_missing_AssetIdentID))
                                            self.update_dpl_status(damage_potential_type, 0, message_text=text)
                                    else:
                                        # USER INFO: Some of the pp defined fields have a wrong data type
                                        wrong_datatype_fields = set(user_field_names) - set(typechecked_user_field_names)
                                        text = self.tr("Folgende {len_wrong_datatype_fields} Layer-Spalten haben einen falschen Datentyp:\n").format(len_wrong_datatype_fields=len(wrong_datatype_fields))
                                        for field_name in wrong_datatype_fields:
                                            text += f"\n{field_name}"
                                        text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                                        self.update_dpl_status(damage_potential_type, 0, message_text=text)
                                else:
                                    # USER INFO: Some of the pp defined fields are not present in the damage potential layer
                                    missing_fields = set(user_field_names) - set(layer_fields_names)
                                    text = self.tr("Folgende Layer-Spalten wurden im Parameter-Profil einem EconoMe-Parameter zugewiesen, aber konnten im Layer nicht gefunden werden:\n").format(missing_fields_length=len(missing_fields))
                                    for missing_field in missing_fields:
                                        text += f"\n{missing_field}"
                                    text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                                    self.update_dpl_status(damage_potential_type, 0, message_text=text)
                            else:
                                # USER INFO: Not all required fields are defined in the parameter profile
                                missing_required_fields = set(self.DAMAGEPOTENTIAL_REQUIRED_FIELDS[damage_potential_type]) - set(damage_potential_valid_parameter_keys)
                                text = self.tr("Folgenden {fields_length} Pflicht-Attributen wurden im Parameter-Profil keine Spalte des Layers zugewiesen:\n").format(fields_length=len(missing_required_fields))
                                for required_field in missing_required_fields:
                                    text += f"\n{required_field}"
                                text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                                self.update_dpl_status(damage_potential_type, 0, message_text=text)
                        else:
                            # USER INFO: No parameter profile found
                            self.update_dpl_status(damage_potential_type, 0, message_text=self.tr("\nBitte Parameter-Profil definieren: Jedem Attribut eine Attributspalte des Layer zuweisen!\n"))
                else:
                    # USER INFO: No damage potential layer selected
                    self.update_dpl_status(damage_potential_type, 0, message_title=self.tr("Fehler"), message_text=self.tr("\nSchadenpotenzial-Layer im Bearbeitungsmodus!\nBitte schliesse den Bearbeitungsmodus ab, um fortzufahren.!\n"))
            else:
                # USER INFO: Invalid damage potential layer
                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=self.tr("\nBitte Schadenpotential-Layer auswählen!\n")).exec_()
        else:
            # USER INFO: Invalid login info
            StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=self.tr("\nUngültige Anmeldedaten!\n")).exec_()

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('EconoMe', message)

    @staticmethod
    def truncFloat(number, decimals=5):
        """Truncates a float after a given number of decimals."""

        s = f"{number:.10f}"  # Convert to string with enough decimals

        if '.' in s:  # If string contains a float, cut it off after decimals
            whole, frac = s.split('.')
            return float(f"{whole}.{frac[:decimals]}")

        return float(s)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&EconoMe'),
                action)
            self.iface.removeToolBarIcon(action)

        try:
            if hasattr(self, 'main_dialog'):
                if self.main_dialog:
                    self.main_dialog.close()
                    self.main_dialog.setParent(None)
                    del self.main_dialog
        except Exception as e:
            log.info("Unload error when closing QGIS!")
            log.error(e)

    def upload_consequence_analysis(self, worker):
        """This method defines the affected assets and uploads the consequence analysis to EconoMe."""

        self.OPEN_SAVE_LAYERS_DIALOG = True

        # CHECK USER INFO VALIDATION ###
        if not self.USER_INFO_VALIDATED:
            # USER INFO: No valid user info
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nUngültige Anmeldedaten: Projektnummer oder API-Keys!\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        
        # CHECK DAMAGE POTENTIAL LAYERS ###
        # CHECK: No damage potential layer selected? (status == -1)
        if all(status == -1 for status in self.DAMAGE_POTENTIAL_LAYER_STATUS.values()):
            # USER INFO: No damage potential layer selected
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nKein Schadenpotential-Layer ausgewählt!\nBitte wähle mindestens 1 Schadenpotential-Layer aus.\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        # CHECK: Any damage potential layer not selected or invalid? (status == 0)
        if any(status == 0 for status in self.DAMAGE_POTENTIAL_LAYER_STATUS.values()):
            # USER INFO: Damage potential layers not valid
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nNicht alle Schadenpotential-Layers sind bereit für den Verschnitt!\nBitte synchronisiere die ausgewählten Schadenpotential-Layer oder wähle sie ab, um fortzufahren!\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        # DO: create the dict of the damage potential layers
        DAMAGE_POTENTIAL_INPUT_DICT = {}
        for damage_potential_type in self.DAMAGE_POTENTIAL_LAYER_STATUS.keys():
            DAMAGE_POTENTIAL_INPUT_DICT[damage_potential_type] = {
                "type": damage_potential_type,
                "layer": getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type}").currentLayer(),
                "status": self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type]
            }
        # CHECK: Any damage potential layer in edit mode? # TODO: is this check necessary? - should be included in status!
        if any(damage_potential_input_dict["layer"].isEditable() for damage_potential_input_dict in DAMAGE_POTENTIAL_INPUT_DICT.values() if damage_potential_input_dict["layer"] is not None):
            # USER INFO: Damage potential layer in toggle edit mode
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nSchadenpotential-Layer im Bearbeitungsmodus!\nBitte schliesse den Bearbeitungsmodus für alle Schadenpotential-Layer ab, um fortzufahren.\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        
        # CHECK INTENSITY MAP LAYERS ###
        scenario_general_status = [status_dict["general"]["status"] for status_dict in self.SCENARIO_STATUS.values()]
        # CHECK: No intensity map layer selected? (status == -1)
        if all(status == -1 for status in scenario_general_status):
            # USER INFO: No intensity layer selected
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nKein Intensitäts-Layer ausgewählt!\nBitte wähle mindestens 1 Intensitäts-Layer aus.\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        # CHECK: Any intensity column not selected or invalid? (status == 0)
        if any(status == 0 for status in scenario_general_status):
            # USER INFO: Intensity Values not valid
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr(f"\nIntensitäts-Layer nicht bereit.\n\n")
            invalid_scen_dict_info = {ann_name: status_dict['general']['info'] for ann_name, status_dict in self.SCENARIO_STATUS.items() if status_dict['general']['status'] == 0}
            user_info_text += "\n".join([f"{ann_name}: {info}" for ann_name, info in invalid_scen_dict_info.items()])
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return
        
        # DO: create the dict of the scenarios
        # return period as string
        annualities_return_period = [self.main_dialog.LABEL_SCENARIO_FREE.text().strip(), "30", "100", "300", self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()]
        annualities_name = list(self.SCENARIO_STATUS.keys())
        # fill in the layer
        SCENARIO_INPUT_DICT = {}
        for ann_rp, ann_name in zip(annualities_return_period, annualities_name):
            # CHECK: Only take the active and selected scenarios
            if self.SCENARIO_STATUS[ann_name]["general"]["status"] == -1:
                continue
            SCENARIO_INPUT_DICT[ann_rp] = {
                "intensity_layer": getattr(self.main_dialog, f"LAYER_INTENSITY_SCENARIO_{ann_name}").currentLayer(),
                "intensity_column": getattr(self.main_dialog, f"COMBO_BOX_INTENSITY_COLUMN_NAME_{ann_name}").currentText(),
                "intensity_results": None,
                "status": self.SCENARIO_STATUS[ann_name]["general"]["status"],
                "ann_name": ann_name,
                "process_name": "".join(self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[2:]).strip(" "),
                "variant_name": "".join(self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[1:]).strip(" ")
            }
        AFFECTED_ASSETS_DICT = {ann_rp : [] for ann_rp in SCENARIO_INPUT_DICT.keys()}
        # CHECK: Any intensity layer in edit mode? # TODO: is this check necessary? - should be included in status!
        if any(scenario_input_dict["intensity_layer"].isEditable() for scenario_input_dict in SCENARIO_INPUT_DICT.values() if scenario_input_dict["intensity_layer"] is not None):
            # USER INFO: Intensity layer in toggle edit mode
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nIntensitäts-Layer im Bearbeitungsmodus!\nBitte schliesse den Bearbeitungsmodus für alle Intensitäts-Layer ab, um fortzufahren.\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return

        # CHECK CRS MATCH ###
        # sample all crs epsgs from damage potential layers
        damage_potential_layers_crs_epsg = set()
        for damage_potential_input_dict in DAMAGE_POTENTIAL_INPUT_DICT.values():
            damage_potential_layer = damage_potential_input_dict["layer"]
            if damage_potential_layer is not None:
                damage_potential_layers_crs_epsg.add(damage_potential_layer.crs().postgisSrid())
        # sample all crs epsgs from intensity layers
        intensity_layers_crs_epsg = set()
        for scenario_input_dict in SCENARIO_INPUT_DICT.values():
            intensity_layer = scenario_input_dict["intensity_layer"]
            if intensity_layer is not None:
                intensity_layers_crs_epsg.add(intensity_layer.crs().postgisSrid())

        # CHECK: Do all damage potential layers and intensity layers have the same crs epsg?
        if not intensity_layers_crs_epsg == damage_potential_layers_crs_epsg or len(intensity_layers_crs_epsg) != 1:
            # USER INFO: Damage potential layer and intensity maps with different coordinate systems
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nNicht alle ausgewählten Layers haben das selbe Koordinatensystem.\n")
            user_info_text += self.tr("Für einen korrekten Verschnitt müssen alle Schadenpotential-Layer und alle Intensitätskarten dasselbe Koordinatensystem verwenden.\n\n")
            user_info_text += self.tr("Aktuelle Koordinatensysteme der Schadenpotential-Layer: EPSG {dam_pot_epsgs}\n").format(dam_pot_epsgs=", ".join([str(epsg) for epsg in damage_potential_layers_crs_epsg]))
            user_info_text += self.tr("Aktuelle Koordinatensysteme der Intensitäts-Layer: EPSG {intensity_epsgs}\n").format(intensity_epsgs=", ".join([str(epsg) for epsg in intensity_layers_crs_epsg]))
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return

        # LOOP over the selected damage potential layers
        UPDATED_DAMAGE_POTENTIAL_LAYERS_DICT = {}
        for damage_potential_type, damage_potential_input_dict in DAMAGE_POTENTIAL_INPUT_DICT.items():
            # CHECK: Is the damage potential layer selected?
            if damage_potential_input_dict["status"] == -1:
                continue

            # Fetch API infos
            API_GET_responses = self.multiple_API_GET_requests([
                self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{damage_potential_type.lower()}",
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/scene',
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset'
            ])

            PARAMETER_PROFILE = API_GET_responses[0].result()[1]  # Get corresponding parameter profile
            SCENES = API_GET_responses[1].result()[1]  # Get scenes
            DAMAGE_POTENTIAL = API_GET_responses[2].result()[1]  # Get damage potential

            # -> CORE: GET AFFECTED ASSETS
            AFFECTED_ASSETS_DICT, SPECIAL_PARAMETER_DICT = self.get_affected_assets(
                worker,
                damage_potential_input_dict,
                DAMAGE_POTENTIAL,
                SCENARIO_INPUT_DICT,
                PARAMETER_PROFILE,
                AFFECTED_ASSETS_DICT
            )

            # CHECK: Error in finding affected assets - was raised inside get_affected_assets method - no need to check here
            if AFFECTED_ASSETS_DICT is None:
                self.OPEN_SAVE_LAYERS_DIALOG = False
                return

            # keep track of the updated damage potential layers
            UPDATED_DAMAGE_POTENTIAL_LAYERS_DICT[damage_potential_type] = damage_potential_input_dict["layer"].name()

        API_ERRORS_DICT = {}
        # LOOP over annualities of the current hazard process and for all damage potential types
        for ann_rp, scenario_input_dict in SCENARIO_INPUT_DICT.items():
            
            # CHECK: Is the scenario active?
            if scenario_input_dict["status"] == -1:
                continue

            # Get corresponding scene ID based on annuality, selected process IDs and variant IDs
            process_ID = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            variant_ID = 0
            if self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]:
                variant_ID = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(",")[0]
            scenes_match = [scene["SceneID"] for scene in SCENES if
                            int(scene["Annuality"]) == int(ann_rp) and int(scene["ProcessID"]) == int(
                                process_ID) and int(scene["VariantID"]) == int(variant_ID)]
            if scenes_match:
                scene_ID = scenes_match[0]

                # API: delete existing affected assets for the scene
                with requests.delete(
                        self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as delete_before:

                    # CHECK: Only continue if AffectedAssetList is not empty
                    if AFFECTED_ASSETS_DICT[ann_rp]:
                        # API: post affected assets for the scene
                        with requests.post(
                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                json={'AffectedAssetList': AFFECTED_ASSETS_DICT[ann_rp]},
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as post_affected_assets:
                            if post_affected_assets.status_code != 200:
                                API_ERRORS_DICT[ann_rp] = json.loads(post_affected_assets.text)

        if API_ERRORS_DICT:
            # USER INFO: Error
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nFehler beim Hochladen der Konsequenzanalyse!\n")
            user_info_text += " ".join(
                [f"{key}: {value['Message']}" for key, value in API_ERRORS_DICT.items()])
            user_info_text += " ".join(
                [f"{key}: {value['Details']}" for key, value in API_ERRORS_DICT.items()])
            worker.show_status.emit(user_info_title, user_info_text, 'error')
            self.OPEN_SAVE_LAYERS_DIALOG = False
            return

        else:
            # USER INFO: Success
            user_info_title = self.tr("Erfolg")
            user_info_text = self.tr("\nKonsequenzanalyse erfolgreich hochgeladen!\n\nBetroffenheit aktualisiert für den Prozess {process} \n\nFür die Szenarien: \n - {scenarios} \n\nFür Schadenpotential-Layer: \n - {damage_potential_layers}\n\nGefundene Spezialparameter: \n - {special_parameters_str}\n").format(
                process = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText(),
                scenarios= "\n - ".join([f"{key} : {len(value)} betroffene Objekte" for key, value in AFFECTED_ASSETS_DICT.items()]),
                damage_potential_layers="\n - ".join([f"{self.tr(key.capitalize())}: {value}" for key, value in UPDATED_DAMAGE_POTENTIAL_LAYERS_DICT.items()]),
                special_parameters_str = "\n - ".join([f"{ann_rp}: {special_parameter_info}" for ann_rp, special_parameter_info in SPECIAL_PARAMETER_DICT.items()]) if SPECIAL_PARAMETER_DICT else self.tr("Keine Spezialparameter."),
            )
            worker.show_status.emit(user_info_title, user_info_text, 'success')

    def update_dpl_status(self, damage_potential_type, status, message_text=None, message_title=None, statustip_text=None):
        damage_potential_type = damage_potential_type.upper()
        # set the status
        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type] = status

        # update the image label
        icon = self.scaled_pixmap_warning if status == 0 else self.scaled_pixmap_checkmark if status == 1 else QPixmap()
        image_label_widget = getattr(self.main_dialog, f"IMAGE_LABEL_DAM_POT_STATUS_{damage_potential_type}")
        image_label_widget.setPixmap(icon)

        if message_text:
            # inform the user
            if message_title is None:
                damage_potential_type_translated = getattr(self.main_dialog, f"LABEL_DAM_POT_LAYER_{damage_potential_type}").text()
                message_title = self.tr("Erfolg: Schadenpotenzial {DAM_POT_TYPE_TRANSLATED} synchronisiert").format(DAM_POT_TYPE_TRANSLATED=damage_potential_type_translated) if status == 1 else self.tr("Fehler beim Synchronisieren des Schadenpotenzials {DAM_POT_TYPE_TRANSLATED}").format(DAM_POT_TYPE_TRANSLATED=damage_potential_type_translated) if status == 0 else self.tr("Hinweis")
            StatusDialog(status_icon=icon, message_title=message_title, message_text="\n"+message_text).exec_()
            statustip_text = message_text

        if statustip_text:
            # set the tool tip of the widget
            image_label_widget.setToolTip(statustip_text)
        else:
            # reset the tool tip of the widget
            image_label_widget.setToolTip("")

    def update_dpl_status_all(self):
        """Updates the status of all damage potential layers in the UI."""

        for damage_potential_type, status in self.DAMAGE_POTENTIAL_LAYER_STATUS.items():
            layer_selected = getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type}").currentLayer() is not None
            status = -1 if not layer_selected else 0
            self.update_dpl_status(damage_potential_type, status, statustip_text="Project opened. Please synchronize to check status.")

    def update_econome_user_info(self, info):
        """Updates EconoMe user info in the QGIS Auth Manager."""

        auth_manager = QgsApplication.authManager()  # type: ignore
        found_authentication = False

        # Log into database
        if not auth_manager.masterPasswordIsSet():
            auth_manager.setMasterPassword()

        # Check for available login methods
        for saved_config_id, config in auth_manager.availableAuthMethodConfigs().items():
            if config.name() == self.ECONOME_API_NAME:
                credentials_successfully_loaded = auth_manager.loadAuthenticationConfig(saved_config_id, config, True)
                if credentials_successfully_loaded:
                    found_authentication = True
                    break

        if not found_authentication:
            new_cfg = QgsAuthMethodConfig()
            new_cfg.setName(self.ECONOME_API_NAME)
            new_cfg.setMethod("Basic")
            new_cfg.setConfig("api_key_a", info["ECONOME_USER_NAME"])
            new_cfg.setConfig("api_key_b", info["ECONOME_PASSWORD"])
            auth_manager.storeAuthenticationConfig(new_cfg)
            QgsExpressionContextUtils.setGlobalVariable('econome_cfg_id', new_cfg.id())  # type: ignore

    def update_econome_user_info_in_ui(self, validated, message):
        """Updates the UI after validating the EconoMe user info."""

        # save validation status
        self.USER_INFO_VALIDATED = validated
        
        # set background color of connect button
        color = '#27ae60' if validated else '#c0392b'
        self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet(f"background-color: {color};")
        
        # set font color of projects combo box
        if validated:
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: 'black'; }")
        else:
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: 'red'; }")

        # set label of projects/project status combo box
        labeltext = self.tr("Projekt(e)") if validated else self.tr("Projektstatus")
        self.main_dialog.LABEL_ECONOME_PROJECT_NUMBER.setText(labeltext)
        
        # set content of projects combo box
        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
        if not validated and message:
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(message)
        elif validated:
            for project_info in self.ECONOME_PROJECT_LIST:
                if project_info[0] not in [self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.itemText(i).split(', ') for i in range(self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.count())]:
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(f"{project_info[1]}, {project_info[0]}")

    def update_intensity_layer_dependent_widgets(self, annuality, current_layer):

        intensity_column_widget = getattr(self.main_dialog, f"COMBO_BOX_INTENSITY_COLUMN_NAME_{annuality}")

        if current_layer:
            # refill the dropdown with field names and enable
            intensity_column_widget.clear()
            intensity_column_widget.addItem("")
            for field in current_layer.fields():
                intensity_column_widget.addItem(field.name())
            intensity_column_widget.setCurrentIndex(0)
            intensity_column_widget.setEnabled(True)
        else:
            # reset the dropdown and disable and reset status
            intensity_column_widget.clear()
            intensity_column_widget.setEnabled(False)
            self.update_scenario_status(annuality, -1, "intensity_column")

        # reset selection of intensity column
        intensity_column_widget.setCurrentIndex(0)

    def update_scenario_status(self, annuality, sender_status, sender, info_text="", message_title=None):

        # set the sender status and info text
        self.SCENARIO_STATUS[annuality][sender]["status"] = sender_status
        self.SCENARIO_STATUS[annuality][sender]["info"] = info_text

        # set the general status and info text
        all_sender_status = [value["status"] for key, value in self.SCENARIO_STATUS[annuality].items() if
                             key != "general"]
        general_status = 1 if all(status == 1 for status in all_sender_status) else 0 if any(
            status == 0 for status in all_sender_status) else -1
        all_sender_info = "\n".join(
            [f"{key} : {value['info']}" for key, value in self.SCENARIO_STATUS[annuality].items() if
             key != "general" and value["info"] != ""])
        self.SCENARIO_STATUS[annuality]["general"]["status"] = general_status
        self.SCENARIO_STATUS[annuality]["general"]["info"] = all_sender_info

        # update the image label
        icon = self.scaled_pixmap_warning if general_status == 0 else self.scaled_pixmap_checkmark if general_status == 1 else QPixmap()
        image_label_widget = getattr(self.main_dialog, f"IMAGE_LABEL_INT_MAP_STATUS_{annuality}")
        image_label_widget.setPixmap(icon)

        # update the statustip text
        image_label_widget.setToolTip(all_sender_info)

        # inform the user
        if message_title:
            message_title = message_title + self.tr(" Szenario") + f" {annuality}."
            StatusDialog(status_icon=icon, message_title=message_title, message_text=info_text).exec_()

    def update_situation_status(self):

        if self.USER_INFO_VALIDATED:
            current_process_id, current_variant_id, get_process = self.get_situation_info()

            if len(json.loads(get_process.text)) != 0:
                SCENARIOS = {key: value["Active"] for key, value in json.loads(get_process.text).items() if
                             key.startswith('Annuality') and value is not None}
                for annuality_key, active in SCENARIOS.items():
                    ann = annuality_key.strip("Annuality")
                    ann_upper = ann.upper() if ann in ["Extreme", "Free"] else str(ann)

                    # Activate or deactivate widgets based on the scenario active status
                    for key, widget_name in self.main_dialog.SCENARIO_WIDGETS[ann_upper].items():
                        getattr(self.main_dialog, widget_name).setEnabled(active)

                    # Set annuality string labels for free and extreme scenario
                    if ann_upper == "FREE":
                        if active:
                            free_annuality = \
                            [value["Annuality"] for key, value in json.loads(get_process.text).items() if
                             key == annuality_key][0]
                        else:
                            free_annuality = self.tr("Frei")
                        self.main_dialog.LABEL_SCENARIO_FREE.setText(str(free_annuality))
                    elif ann_upper == "EXTREME":
                        if active:
                            extreme_annuality = \
                            [value["Annuality"] for key, value in json.loads(get_process.text).items() if
                             key == annuality_key][0]
                        else:
                            extreme_annuality = self.tr("Extrem")
                        self.main_dialog.LABEL_SCENARIO_EXTREME.setText(str(extreme_annuality))

                    # fill widgets based on selected layers dict or empty the widgets
                    layer_combobox = getattr(self.main_dialog,
                                             self.main_dialog.SCENARIO_WIDGETS[ann_upper]["LAYER_INTENSITY"])
                    column_combobox = getattr(self.main_dialog, self.main_dialog.SCENARIO_WIDGETS[ann_upper][
                        "COMBO_BOX_INTENSITY_COLUMN_NAME"])
                    if active:
                        key = f"{current_process_id}_{current_variant_id}_{ann_upper}"
                        saved_layer_name = self.SELECTED_INTENSITY_MAP_LAYERS.get(key)
                        if saved_layer_name is not None:
                            found_layers = self.qgis_project.mapLayersByName(saved_layer_name)
                            if len(found_layers) == 1:
                                # SUCCESS: found the layer
                                layer_combobox.setLayer(found_layers[0])
                                saved_intensity_column_name = self.SELECTED_INTENSITY_COLUMNS.get(key)
                                if saved_intensity_column_name in found_layers[0].fields().names():
                                    # SUCCESS: found the layer and the column, setting it to the combo box
                                    column_combobox.setCurrentText(saved_intensity_column_name)
                                else:
                                    # FAIL: column not found, clearing the combo box
                                    column_combobox.setCurrentIndex(0)
                                continue
                    # FAIL: 1) scenario not active or 2) layer not saved or 3) layer not found or 4) multiple layers found
                    layer_combobox.setLayer(QgsVectorLayer(""))
                    column_combobox.clear()
                    getattr(self.main_dialog,
                            self.main_dialog.SCENARIO_WIDGETS[ann_upper]["IMAGE_LABEL_INT_MAP_STATUS"]).clear()

    def validate_econome_user_info(self):
        """Checks if API keys were given and are valid."""

        # Get username and password from lineEdit
        ECONOME_USER_NAME = self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.text().strip()
        ECONOME_PASSWORD = self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.text().strip()

        # CHECK: Username and password filled
        if ECONOME_USER_NAME and ECONOME_PASSWORD:

            # Make test API request
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH,
                    auth=HTTPBasicAuth(ECONOME_USER_NAME, ECONOME_PASSWORD),
                    timeout=10
            ) as make_test_request:
                # Check if API is responding
                try:
                    TEST_REQUEST = make_test_request.json()
                except ValueError as e:
                    if isinstance(e, requests.exceptions.JSONDecodeError) or isinstance(e, json.JSONDecodeError):
                        TEST_REQUEST = "JSONDecodeError"
                        log.error(f"API call failed due to JSON Decode Error: {e}")
                    else:
                        TEST_REQUEST = []
                        log.error(e)

            # Test API request was successful
            if 'ProjectList' in TEST_REQUEST:
                # Test if user has at least 1 project
                if len(TEST_REQUEST['ProjectList']) > 0:  # type: ignore
                    # Set EconoMe user info
                    self.ECONOME_USER_NAME = ECONOME_USER_NAME
                    self.ECONOME_PASSWORD = ECONOME_PASSWORD
                    self.update_econome_user_info({"ECONOME_USER_NAME": ECONOME_USER_NAME, "ECONOME_PASSWORD": ECONOME_PASSWORD})
                    self.ECONOME_PROJECT_LIST = [(0, self.tr("Kein Projekt ausgewählt"))] + [(proj['ProjectIdentID'], proj['ProjectTitle']) for proj in TEST_REQUEST['ProjectList']]  # type: ignore
                    validated, message = True, None
                else:  # no projects found
                    validated, message = False, self.tr("No projects found ... valid credentials?")
            else:  # Test API request was not successful -> Receive error messages and display them in the combo box
                if "Errorcode" in TEST_REQUEST:
                    validated, message = False, f"{TEST_REQUEST['Errormessage']}"  # type: ignore
                elif TEST_REQUEST == "JSONDecodeError":
                    validated, message = False, self.tr("API returns an error ... valid credentials?")
                elif TEST_REQUEST == []:
                    validated, message = False, self.tr("API doesn't respond ...")
                else:
                    validated, message = False, f"{TEST_REQUEST['Message']}"  # type: ignore

        else:  # User name and/or password were not filled
            validated, message = False, self.tr("Please enter user name and password ...")

        # Update UI
        self.update_econome_user_info_in_ui(validated, message)
        if validated:
            self.execute_in_separate_thread(self.get_project_status, "get_project_status")

    def visualize_risks(self, worker):
        """Fetches risk from EconoMe and creates a spatial risk layer based on damage potential."""

        self.RISK_OUTPUT_TEMP_LAYERS = []  # Create empty result layers dict

        # CHECK: Valid user info
        if self.USER_INFO_VALIDATED:

            # Get all currently selected damage potential layers and store them in a dict
            DAMAGE_POTENTIAL_LAYER_DICT = {
                "BUILDING": self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer(),
                "ROAD": self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer(),
                "RAILWAY": self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer(),
                "CABLE_TRANSPORT": self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer(),
                "LINE_PIPE": self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer(),
                "PEOPLE_OUTDOOR": self.main_dialog.LAYER_DAM_POT_PEOPLE_OUTDOOR.currentLayer(),
                "OTHER_ASSET": self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer()
            }

            # CHECK: At least one damage potential layer must be selected
            if any(layer is not None for layer in DAMAGE_POTENTIAL_LAYER_DICT.values()):

                # Create a dict of all input damage potential layers
                current_damage_potential_layers = {k: v for k, v in DAMAGE_POTENTIAL_LAYER_DICT.items() if v is not None}
                current_damage_potential_layer_status = [value for key, value in self.DAMAGE_POTENTIAL_LAYER_STATUS.items() if key in current_damage_potential_layers.keys()]
                
                # CHECK: Are all selected damage potential layers synced with EconoMe?
                if all(status == 1 for status in current_damage_potential_layer_status):

                    # Loop over damage potential layers
                    for damage_potential_type, current_damage_potential_layer in current_damage_potential_layers.items():

                        # Fetch API infos
                        API_GET_responses = self.multiple_API_GET_requests([
                            self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{damage_potential_type.lower()}",
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/damagepotential/asset",
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/scene"
                        ])

                        PARAMETER_PROFILE = API_GET_responses[0].result()[1]  # Get corresponding parameter profile
                        UPLOADED_DAMAGE_POTENTIAL = API_GET_responses[1].result()[1]  # Get damage potential
                        SCENES = API_GET_responses[2].result()[1]  # Get scenes

                        # Read selected hazard process
                        SELECTED_HAZARD_PROCESS = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText()
                        # Read selected measure variant
                        SELECTED_MEASURE_VARIANT = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText()

                        # CHECK: Existing scenes
                        if len(SCENES) != 0:

                            # Free and extreme annualities
                            free_annuality = "-1"
                            extreme_annuality = "-1"

                            # Check for free and extreme annuality
                            if self.main_dialog.LABEL_SCENARIO_FREE.text().isdigit():
                                free_annuality = self.main_dialog.LABEL_SCENARIO_FREE.text().strip()
                            if self.main_dialog.LABEL_SCENARIO_EXTREME.text().isdigit():
                                extreme_annuality = self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()

                            # Store risk data output
                            RISK_OUTPUT_DICT = {
                                free_annuality: {
                                    "risk": None
                                },
                                "30": {
                                    "risk": None
                                },
                                "100": {
                                    "risk": None
                                },
                                "300": {
                                    "risk": None
                                },
                                extreme_annuality: {
                                    "risk": None
                                }
                            }

                            # Get risk from all scenes matching user selection

                            # Create scene ID dictionary
                            SCENE_ID_DICT = {
                                free_annuality: [scene["SceneID"] for scene in SCENES if scene["Annuality"] == int(free_annuality) and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "30": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 30 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "100": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 100 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "300": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 300 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                extreme_annuality: [scene["SceneID"] for scene in SCENES if scene["Annuality"] == int(extreme_annuality) and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])]
                            }

                            FILTERED_GET_RISK_REQUESTS = {}
                            for annuality, scene_id in SCENE_ID_DICT.items():
                                if len(scene_id) != 0:
                                    FILTERED_GET_RISK_REQUESTS[annuality] = self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/risk/scene/{scene_id[0]}"

                            # Fetch API Risk values
                            API_GET_responses = self.multiple_API_GET_requests(list(FILTERED_GET_RISK_REQUESTS.values()))

                            index = 0
                            for response in API_GET_responses:
                                RISK_OUTPUT_DICT[list(FILTERED_GET_RISK_REQUESTS.keys())[index]]["risk"] = response.result()[1]
                                index += 1

                            has_risk = any(val["risk"] not in (None, []) for val in RISK_OUTPUT_DICT.values())

                            # CHECK: Are there risks in EconoMe
                            if has_risk:
                                self.OPEN_SAVE_LAYERS_DIALOG = True
                                # Create a copy of the damage potential layer as a memory layer with the prefix of the hazard process
                                type_and_projection = ""
                                if current_damage_potential_layer.geometryType() == 0:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"Point?crs={current_damage_potential_layer.crs().postgisSrid()}"
                                elif current_damage_potential_layer.geometryType() == 1:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"LineString?crs={current_damage_potential_layer.crs().postgisSrid()}"
                                elif current_damage_potential_layer.geometryType() == 2:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"Polygon?crs={current_damage_potential_layer.crs().postgisSrid()}"

                                current_damage_potential_layer_AssetIdentID = [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if p["API"] == "AssetIdentID"][0]
                                current_damage_potential_layer_fields = current_damage_potential_layer.fields()
                                timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
                                risks_translation_str = self.tr("Risiken")
                                new_risk_layer = QgsVectorLayer(type_and_projection, f"{risks_translation_str}_{SELECTED_MEASURE_VARIANT.split(', ')[1][:25]}_{SELECTED_HAZARD_PROCESS.split(', ')[2][:25]}_{damage_potential_type}_{timestamp}", "memory")
                                new_risk_layer.dataProvider().addAttributes(current_damage_potential_layer_fields)

                                # Switch to edit mode
                                new_risk_layer.startEditing()

                                # Copy all features from the original layer to the new layer
                                for feature in current_damage_potential_layer.getFeatures():
                                    new_risk_layer.dataProvider().addFeature(feature)

                                # Loop over all risks and define new fields
                                for annuality in RISK_OUTPUT_DICT.keys():
                                    new_fields = QgsFields()
                                    # CHECK: Is there risk data ?
                                    if RISK_OUTPUT_DICT[annuality]["risk"]:
                                        # Define new fields if not existing
                                        for output_variable in self.RISK_OUTPUT_VARIABLES[damage_potential_type]:
                                            if output_variable not in current_damage_potential_layer_fields:
                                                new_fields.append(QgsField(f"{output_variable}_{annuality}", QVariant.Double))
                                                new_risk_layer.dataProvider().addAttributes(new_fields)
                                        # Update fields
                                        new_risk_layer.updateFields()

                                # Loop over all risks and fill fields
                                for annuality in RISK_OUTPUT_DICT.keys():
                                    # CHECK: Is there risk data ?
                                    if RISK_OUTPUT_DICT[annuality]["risk"]:
                                        # LOOP over all features
                                        for feature in new_risk_layer.getFeatures():
                                            # Get corresponding Asset ID
                                            corresponding_asset_id = [asset["AssetID"] for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[current_damage_potential_layer_AssetIdentID])][0]
                                            result = [elem for elem in RISK_OUTPUT_DICT[annuality]["risk"] if int(elem["AssetID"]) == int(corresponding_asset_id)]  # type: ignore
                                            if len(result) != 0:
                                                # Loop over all output variables
                                                for output_variable in self.RISK_OUTPUT_VARIABLES[damage_potential_type]:
                                                    # Enter attributes
                                                    feature.setAttribute(f"{output_variable}_{annuality}", result[0][output_variable])
                                            # Update feature
                                            new_risk_layer.updateFeature(feature)

                                # Save edits
                                new_risk_layer.commitChanges()
                                # Set color renderer

                                new_risk_layer_fields = [field.name() for field in new_risk_layer.fields()]

                                # Add summary risk fields
                                if f"R_{free_annuality}" in new_risk_layer_fields or "R_30" in new_risk_layer_fields or "R_100" in new_risk_layer_fields or "R_300" in new_risk_layer_fields or f"R_{extreme_annuality}" in new_risk_layer_fields:
                                    if damage_potential_type not in ["OTHER_ASSET", "LINE_PIPE"]:
                                        new_fields = QgsFields()
                                        new_fields.append(QgsField("R", QVariant.Double))
                                        new_fields.append(QgsField("r_indiv", QVariant.Double))
                                        new_risk_layer.dataProvider().addAttributes(new_fields)
                                        new_risk_layer.updateFields()

                                new_risk_layer_fields = [field.name() for field in new_risk_layer.fields()]
                                new_risk_layer.startEditing()

                                # Add summary risk fields
                                for feature in new_risk_layer.getFeatures():

                                    if "R" in new_risk_layer_fields:
                                        r_sum = 0
                                        if f"R_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"R_{free_annuality}"] is not None:
                                                r_sum += feature[f"R_{free_annuality}"]
                                        if "R_30" in new_risk_layer_fields:
                                            if feature["R_30"] is not None:
                                                r_sum += feature["R_30"]
                                        if "R_100" in new_risk_layer_fields:
                                            if feature["R_100"] is not None:
                                                r_sum += feature["R_100"]
                                        if "R_300" in new_risk_layer_fields:
                                            if feature["R_300"] is not None:
                                                r_sum += feature["R_300"]
                                        if f"R_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"R_{extreme_annuality}"] is not None:
                                                r_sum += feature[f"R_{extreme_annuality}"]
                                        feature.setAttribute("R", r_sum)

                                    if "r_indiv" in new_risk_layer_fields:
                                        r_indiv_sum = 0
                                        if f"r_indiv_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"r_indiv_{free_annuality}"] is not None:
                                                r_indiv_sum += feature[f"r_indiv_{free_annuality}"]
                                        if f"r_indiv_30" in new_risk_layer_fields:
                                            if feature["r_indiv_30"] is not None:
                                                r_indiv_sum += feature["r_indiv_30"]
                                        if f"r_indiv_100" in new_risk_layer_fields:
                                            if feature["r_indiv_100"] is not None:
                                                r_indiv_sum += feature["r_indiv_100"]
                                        if f"r_indiv_300" in new_risk_layer_fields:
                                            if feature["r_indiv_300"] is not None:
                                                r_indiv_sum += feature["r_indiv_300"]
                                        if f"r_indiv_{extreme_annuality}" in new_risk_layer_fields:
                                            if feature[f"r_indiv_{extreme_annuality}"] is not None:
                                                r_indiv_sum += feature[f"r_indiv_{extreme_annuality}"]
                                        feature.setAttribute("r_indiv", r_indiv_sum)

                                    new_risk_layer.updateFeature(feature)

                                new_risk_layer.commitChanges()

                                # Visualize r_indiv
                                if "r_indiv" in new_risk_layer_fields:

                                    new_risk_layer.startEditing()

                                    for feature in new_risk_layer.getFeatures():
                                        try:
                                            if feature["r_indiv"]:
                                                feature["r_indiv"] = float(feature["r_indiv"])
                                                new_risk_layer.updateFeature(feature)
                                        except ValueError as e:
                                            log.info("Could not convert column 'r_indiv' into float values!")
                                            log.error(e)

                                    new_risk_layer.commitChanges()

                                    ranges = []

                                    # Define first range
                                    symbol1 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())  # type: ignore
                                    symbol1.setColor(QColor("lightblue"))
                                    if damage_potential_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol1.setWidth(1.5)  # type: ignore
                                    range1 = QgsRendererRange(0, 0.000001, symbol1, "< 1e-6")
                                    ranges.append(range1)

                                    # Define second range
                                    symbol2 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())  # type: ignore
                                    if damage_potential_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol2.setWidth(1.5)  # type: ignore
                                    symbol2.setColor(QColor("orange"))
                                    range2 = QgsRendererRange(0.000001, 0.00001, symbol2, "1e-6 - 1e-5")
                                    ranges.append(range2)

                                    # Define third range
                                    symbol3 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())  # type: ignore
                                    if damage_potential_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol3.setWidth(1.5)  # type: ignore
                                    symbol3.setColor(QColor("red"))
                                    range3 = QgsRendererRange(0.00001, 1, symbol3, "> 1e-5")
                                    ranges.append(range3)

                                    renderer = QgsGraduatedSymbolRenderer("r_indiv", ranges)

                                    # Set the renderer to the layer
                                    new_risk_layer.setRenderer(renderer)

                                # Collect risk layers
                                if new_risk_layer:
                                    label_dam_pot_type_translated = self.main_dialog.findChild(QLabel, f"LABEL_DAM_POT_LAYER_{damage_potential_type}").text()
                                    self.RISK_OUTPUT_TEMP_LAYERS.append((new_risk_layer, damage_potential_type, label_dam_pot_type_translated))
                            else:
                                # USER INFO: No risk found to visualize!
                                user_info_title = self.tr("Fehler")
                                user_info_text = self.tr("\nKein Risiko zum Visualisieren vorhanden!\n")
                                worker.show_status.emit(user_info_title, user_info_text, 'error')
                        else:
                            # USER INFO: Keine Szenen definiert
                            user_info_title = self.tr("Fehler")
                            user_info_text = self.tr("\nKeine Szenen vorhanden!\n")
                            worker.show_status.emit(user_info_title, user_info_text, 'error')
                else:
                    # USER INFO: Not synced damage potential layers
                    user_info_title = self.tr("Fehler")
                    user_info_text = self.tr("\nBitte ausgewählte Schadenpotential-Layer mit EconoMe synchronisieren, um Risiken zu visualisieren!\n")
                    worker.show_status.emit(user_info_title, user_info_text, 'error')
            else:
                # USER INFO: Kein Schadenpotential ausgewählt
                user_info_title = self.tr("Fehler")
                user_info_text = self.tr("\nBitte Schadenpotential-Layer auswählen, um Risiken zu visualisieren!\n")
                worker.show_status.emit(user_info_title, user_info_text, 'error')
        else:
            # USER INFO: Ungültige Anmeldedaten
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nUngültige Anmeldedaten! (Projektnummer oder API-Keys)\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
