# -*- coding: utf-8 -*-

"""
EconoMe - Connecting your QGIS-Project with EconoMe.
Version 1.3

Copyright (C) 2025 Federal Office for the Environment FOEN, Switzerland

Authors
  - Kevin Helzel (WSL Institute for Snow and Avalanche Research SLF, Davos, Switzerland)
  - Tobia Lezuo (Geotest AG, Davos, Switzerland)

Email
  - Plugin & Code: kevin.helzel@slf.ch
  - EconoMe project leader: linda.ettlin@slf.ch

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.
"""

import os
import sys
import logging
import warnings
import traceback
from logging.handlers import RotatingFileHandler
from qgis.core import Qgis, QgsMessageLog, QgsApplication, QgsProject
from qgis.PyQt.QtWidgets import QMessageBox


PLUGIN_NAME = "EconoMe"


def get_log_file_path():
    """Return the path to the plugin's log file inside the current QGIS project directory."""

    project_path = QgsProject.instance().fileName()

    if project_path:  # Project is saved somewhere on disk
        project_dir = os.path.dirname(project_path)
        log_dir = os.path.join(project_dir, "debug_logs", "plugins", PLUGIN_NAME)
    else:  # Fallback if no project is loaded yet -> Save in QGIS settings directory
        log_dir = os.path.join(QgsApplication.qgisSettingsDirPath(), "debug_logs", "plugins", PLUGIN_NAME)

    os.makedirs(log_dir, exist_ok=True)
    return os.path.join(log_dir, "plugin.log")


def setup_logger():
    """Create and configure a logger with file + QGIS log handlers."""

    log_file = get_log_file_path()

    logger = logging.getLogger(PLUGIN_NAME)
    logger.setLevel(logging.DEBUG)

    # Remove old handlers to prevent writing to old files
    if logger.hasHandlers():
        for handler in logger.handlers[:]:
            try:
                handler.close()
            except Exception as e:
                print("Error closing old handler:", e)
            logger.removeHandler(handler)

    # Clear previous log session (truncate the file)
    with open(log_file, "w", encoding="utf-8") as f:
        f.write("")  # Clears old logs

    # File handler with rotation (max 500 KB, 3 backups)
    fh = RotatingFileHandler(log_file, maxBytes=500 * 1024, backupCount=3, encoding="utf-8")
    fh.setLevel(logging.DEBUG)
    formatter = logging.Formatter("%(asctime)s - %(levelname)s - %(name)s - %(message)s")
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    # QGIS log handler (warnings and above)
    class QgisLogHandler(logging.Handler):
        """Custom logging handler that forwards log records to the QGIS Log Messages panel."""

        def emit(self, record):
            try:
                msg = self.format(record)

                # Map Python logging levels to QGIS log levels
                level_map = {
                    logging.DEBUG: Qgis.Info,
                    logging.INFO: Qgis.Info,
                    logging.WARNING: Qgis.Warning,
                    logging.ERROR: Qgis.Critical,
                    logging.CRITICAL: Qgis.Critical,
                }

                qgis_level = level_map.get(record.levelno, Qgis.Info)

                QgsMessageLog.logMessage(msg, PLUGIN_NAME, level=qgis_level)
            except Exception as e:
                # Prevent recursive logging if something goes wrong in emit()
                print(f"[{PLUGIN_NAME} QgisLogHandler error]: {e}")

    qgis_handler = QgisLogHandler()
    qgis_handler.setFormatter(formatter)
    qgis_handler.setLevel(logging.WARNING)
    logger.addHandler(qgis_handler)

    logger.info("----------")
    logger.info("NEW PLUGIN SESSION started!")
    logger.info("----------")
    logger.info("Logger initialized!")
    logger.info("----------")
    return logger


def global_exception_hook(exctype, value, tb):
    """Catch all uncaught exceptions, log them, and show user message."""

    msg = "".join(traceback.format_exception(exctype, value, tb))
    log.critical("Unhandled exception:\n%s", msg)

    # Optional: show friendly popup (non-blocking)
    try:
        from qgis.utils import iface
        iface.messageBar().pushCritical(
            PLUGIN_NAME,
            "An unexpected error occurred. Check the plugin log for details.",
        )
    except Exception:
        # If iface not ready (early init), show a plain Qt dialog
        QMessageBox.critical(None, PLUGIN_NAME, "Unexpected error. Check log file.")


# -------------------------------------------------------------------
# Create logger and set global hook
# -------------------------------------------------------------------

log = setup_logger()
warnings.simplefilter("default")
logging.captureWarnings(True)
sys.excepthook = global_exception_hook
