# -*- coding: utf-8 -*-

"""
EconoMe - Connecting your QGIS-Project with EconoMe.
Version 1.3

Copyright (C) 2025 Federal Office for the Environment FOEN, Switzerland

Authors
  - Kevin Helzel (WSL Institute for Snow and Avalanche Research SLF, Davos, Switzerland)
  - Tobia Lezuo (Geotest AG, Davos, Switzerland)

Email
  - Plugin & Code: kevin.helzel@slf.ch
  - EconoMe project leader: linda.ettlin@slf.ch

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.

Generated by QGIS Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
"""

import os
import requests
from requests.auth import HTTPBasicAuth
from .econo_me_dialog_status import StatusDialog
from .logger import get_log_file_path

from qgis.PyQt.QtCore import Qt, QSize
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtGui import QPalette, QDesktopServices, QIcon, QCursor, QPixmap
from qgis.PyQt.QtWidgets import QDialog, QHBoxLayout, QVBoxLayout, QPushButton, QSizePolicy, QSpacerItem, QDialogButtonBox
from PyQt5.QtCore import QUrl


FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'econo_me_dialog_settings_menu.ui'))


class BugReport(QDialog):
    """Custom dialog for bug report."""

    def __init__(self):

        # Initialize class
        super().__init__()

        # UI tweaks
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                    QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                    QLabel { color: black; }
                    QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                    QPushButton:hover { background-color: #D0D0D0; }
                    QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                """)  # Force light mode when dark mode is detected

        # Dialog header
        title = self.tr("EconoMe") + " - " + self.tr("Bug Report")
        self.setWindowTitle(title)
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))

        # Main layout
        main_layout = QVBoxLayout()

        # Add subject input
        self.subject_line_edit = QtWidgets.QLineEdit()
        self.subject_line_edit.setPlaceholderText(self.tr("Subject"))

        # Add description input
        self.description_line_edit = QtWidgets.QTextEdit()
        self.description_line_edit.setPlaceholderText(self.tr("Please provide a short description of the problem!"))

        # Add "YES" and "NO" or customized buttons
        button_layout = QHBoxLayout()
        send_button = QPushButton(self.tr("SEND"))
        cancel_button = QPushButton(self.tr("CANCEL"))
        send_button.setCursor(QCursor(Qt.PointingHandCursor))
        cancel_button.setCursor(QCursor(Qt.PointingHandCursor))
        send_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
        cancel_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
        button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        button_layout.addWidget(send_button)
        button_layout.addWidget(cancel_button)
        button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        # Activate buttons
        send_button.clicked.connect(self.accept)
        cancel_button.clicked.connect(self.reject)

        # Add horizontal layouts to main layout
        main_layout.addWidget(self.subject_line_edit)
        main_layout.addWidget(self.description_line_edit)
        main_layout.addLayout(button_layout)

        # Set main layout
        self.setLayout(main_layout)


class EconoMeDialogSettingsMenu(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, econome_user_name=None, econome_password=None, econome_api_base_path=None, parent=None):
        """Constructor."""

        super(EconoMeDialogSettingsMenu, self).__init__(parent)
        self.setupUi(self)

        # Force light mode when dark mode is detected
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                            QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                            QLabel { color: black; }
                            QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                            QPushButton:hover { background-color: #D0D0D0; }
                            QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                        """)

        # Define instance and other variables
        self.econome_user_name = econome_user_name
        self.econome_password = econome_password
        self.econome_api_base_path = econome_api_base_path
        self.plugin_dir = os.path.dirname(__file__)

        # Make pixmap for warning and success icons
        path_to_checkmark = os.path.join(self.plugin_dir, "icons", "checkmark.png")
        pixmap_checkmark = QPixmap(path_to_checkmark)
        self.scaled_pixmap_checkmark = pixmap_checkmark.scaled(QSize(30, 30), Qt.KeepAspectRatio, Qt.SmoothTransformation)
        path_to_warning = os.path.join(self.plugin_dir, "icons", "warning.png")
        pixmap_warning = QPixmap(path_to_warning)
        self.scaled_pixmap_warning = pixmap_warning.scaled(QSize(30, 30),Qt.KeepAspectRatio, Qt.SmoothTransformation)

        # Connect buttons
        self.BUTTON_LINK_ECONOME.clicked.connect(self.on_econome_link_clicked)
        self.BUTTON_LINK_MANUAL.clicked.connect(self.on_manual_link_clicked)
        self.BUTTON_REPORT_BUG.clicked.connect(self.send_bug_report)
        self.BUTTON_LINK_ECONOME.setCursor(QCursor(Qt.PointingHandCursor))
        self.BUTTON_LINK_MANUAL.setCursor(QCursor(Qt.PointingHandCursor))
        self.BUTTON_REPORT_BUG.setCursor(QCursor(Qt.PointingHandCursor))

        # Override OK|CANCEL buttons from button box for translation
        self.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Ok).setText(self.tr("OK"))
        self.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Cancel).setText(self.tr("ABBRECHEN"))
      
    def on_econome_link_clicked(self):

        language = self.COMBO_BOX_LANGUAGE_SELECTION.currentText()
        SUPPORTED_LANGUAGES = {"DE", "FR", "IT"}

        if language in SUPPORTED_LANGUAGES:
            url = QUrl(f"https://econome.ch/eco_work/eco_wiki_main.php?lang={language.lower()}")
            QDesktopServices.openUrl(url)
        else:
            QDesktopServices.openUrl(QUrl("https://econome.ch/eco_work/eco_wiki_main.php"))
        
    def on_manual_link_clicked(self):
        """This function opens a plugin manual in a new browser window."""

        language = self.COMBO_BOX_LANGUAGE_SELECTION.currentText()
        SUPPORTED_LANGUAGES = {"DE", "EN", "FR", "IT"}

        if language in SUPPORTED_LANGUAGES:
            url = QUrl(f"https://econome.ch/econome_doc/wiki/manual_plugin_{language[0]}.pdf")
            QDesktopServices.openUrl(url)

    def send_bug_report(self):
        """This function sends a plugin bug report via email through the EconoMe API."""

        new_bug_report = BugReport()  # Open bug report dialog
        user_choice = new_bug_report.exec_()  # Wait for user input

        if user_choice == QDialog.Accepted:  # CASE: SEND BUG REPORT

            subject = new_bug_report.subject_line_edit.text()  # Get subject line input
            message = new_bug_report.description_line_edit.toPlainText()  # Get description line input
            
            # Get contents of log file
            log_file_path = get_log_file_path()
            try:
                with open(log_file_path, "r", encoding="utf-8") as f:
                    lines = f.readlines()
                log_contents = "\n".join(lines)
            except FileNotFoundError:
                log_contents = "Log file not found: " + log_file_path
            except Exception as e:
                log_contents = f"Error reading log file:\n{e}"

            if subject and message:  # CASE: Ready to send

                AUTH = HTTPBasicAuth(self.econome_user_name, self.econome_password)
                BASE_PATH = self.econome_api_base_path.replace("/project", "")

                bug_report_dict = {
                    'Subject': subject,
                    'Message': message,
                    'Log': log_contents
                }

                with requests.post(
                        BASE_PATH + f"/bugreport",
                        auth=AUTH,
                        json=bug_report_dict
                ) as response:

                    # Check server response
                    if response.status_code == 200:  # CASE: Bug report submitted
    
                        StatusDialog(
                            status_icon=self.scaled_pixmap_checkmark,
                            message_title=self.tr("BUG REPORT SUCCESSFULLY SUBMITTED"),
                            message_text=self.tr("\nThank you for submitting a bug report!\nYou will be contacted via email asap by a member of the EconoMe developer team.\n")
                        ).exec_()

                    else:  # CASE: Something went wrong ...

                        StatusDialog(
                            status_icon=self.scaled_pixmap_warning,
                            message_title=self.tr("ERROR WHILE SENDING BUG REPORT"),
                            message_text=self.tr("\nSorry that didn't work!\nPlease send your bug description alternatively per email to econome@slf.ch!\n")
                        ).exec_()
                        
            else:  # CASE: User didn't fill subject and description

                StatusDialog(
                    status_icon=self.scaled_pixmap_warning,
                    message_title=self.tr("BUG REPORT NOT SUBMITTED"),
                    message_text=self.tr("\nPlease fill in a subject and a short description of the problem!\n")
                ).exec_()
