# -*- coding: utf-8 -*-

"""
EconoMe - Connecting your QGIS-Project with EconoMe.
Version 1.2.3

Copyright (C) 2025 Federal Office for the Environment FOEN, Switzerland

Authors
  - Kevin Helzel (WSL Institute for Snow and Avalanche Research SLF, Davos, Switzerland)
  - Tobia Lezuo (Geotest AG, Davos, Switzerland)

Email
  - Plugin & Code: kevin.helzel@slf.ch
  - EconoMe project leader: linda.ettlin@slf.ch

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.

Generated by QGIS Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
"""

# QGIS Main Imports
from qgis.PyQt.QtCore import Qt, QCoreApplication, QEvent, QObject, QSettings, QSize, QThread, QTranslator, QVariant, pyqtSignal
from qgis.PyQt.QtGui import QColor, QCursor, QIcon, QMovie, QPalette, QPixmap
from qgis.PyQt.QtWidgets import QAction, QApplication, QVBoxLayout, QComboBox, QCompleter, QDialog, QDialogButtonBox, QFileDialog, QFormLayout, QHBoxLayout, QLabel, QLineEdit, QListView, QPushButton, QScrollArea, QSizePolicy, QSpacerItem, QWidget
from qgis.core import QgsApplication, QgsCoordinateTransformContext, QgsExpressionContextUtils, QgsFeatureRequest, QgsField, QgsFields, Qgis, QgsGraduatedSymbolRenderer, QgsMapLayerProxyModel, QgsProject, QgsRendererRange, QgsSymbol, QgsVectorFileWriter, QgsVectorLayer, NULL
# QGIS Processing Imports
import processing
from processing.tools import dataobjects

# Custom imports
import os.path
from datetime import datetime
import tempfile
import json
import requests
from requests.auth import HTTPBasicAuth
from concurrent.futures import ThreadPoolExecutor, as_completed
from collections import Counter
import copy

# Initialize Qt resources from file resources.py
from .resources import *

# Import dialogs
from .econo_me_dialog import EconoMeDialog
from .econo_me_dialog_settings_menu import EconoMeDialogSettingsMenu
from .econo_me_dialog_status import StatusDialog
from .econo_me_dialog_asset_handling import DamagePotentialMismatchResolveDialog


class WorkerThread(QThread):
    """Executes a function in a separate thread."""

    show_status = pyqtSignal(str, str, str)  # Signal emitted when status dialog should be opened
    revalidate_econome_user_info = pyqtSignal()  # Signal emitted when user info should be revalidated
    finished = pyqtSignal()  # Signal emitted when task is done

    def __init__(self, parent_dialog, task_function, should_revalidate=False):
        """Initialize worker thread."""

        super().__init__()
        self.parent_dialog = parent_dialog
        self.task_function = task_function
        self.should_revalidate = should_revalidate

    def run(self):
        """Execute the function in a safe thread."""

        # Execute task
        self.task_function(self)
        # Revalidate user info as needed
        if self.should_revalidate:
            self.revalidate_econome_user_info.emit()
        # Task is done
        self.finished.emit()


class AreYouSure(QDialog):
    """Custom dialog for user yes/no input."""

    def __init__(self, message, parent=None):

        # Initialize class
        super(AreYouSure, self).__init__(parent)

        # Force light mode when dark mode is detected
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                    QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                    QLabel { color: black; }
                    QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                    QPushButton:hover { background-color: #D0D0D0; }
                    QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                """)

        # Dialog header
        title = self.tr("EconoMe") +  " - " + self.tr("Plugin Status")
        self.setWindowTitle(title)
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))

        # Main layout
        main_layout = QVBoxLayout()
        # Add a label
        label_layout = QHBoxLayout()
        label = QLabel(message)
        label.setAlignment(Qt.AlignHCenter)
        label.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        label_layout.addSpacerItem(QSpacerItem(10, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        label_layout.addWidget(label)
        label_layout.addSpacerItem(QSpacerItem(10, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        # Add an "DELETE"  and "CANCEL" button
        button_layout = QHBoxLayout()
        delete_button = QPushButton(self.tr("LÖSCHEN"))
        cancel_button = QPushButton(self.tr("ABBRECHEN"))
        delete_button.setCursor(QCursor(Qt.PointingHandCursor))
        cancel_button.setCursor(QCursor(Qt.PointingHandCursor))
        delete_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
        cancel_button.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed)
        button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        button_layout.addWidget(delete_button)
        button_layout.addWidget(cancel_button)
        button_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        
        delete_button.clicked.connect(self.accept)
        cancel_button.clicked.connect(self.reject)
        
        # Add horizontal layouts to main layout
        main_layout.addLayout(label_layout)
        main_layout.addLayout(button_layout)

        # Set main layout
        self.setLayout(main_layout)


class ParameterProfileDialog(QDialog):
    """Custom dialog for setting the parameter profiles for the damage potential."""

    def __init__(self, fields, mandatory_fields, language, glossary, parent=None):
        super().__init__(parent)
        # Instance variables
        self.fields = fields
        self.mandatory_fields = mandatory_fields
        self.language = language
        self.glossary = glossary
        # UI tweaks
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        self.resize(450, 550)
        self.init_ui()

    def init_ui(self):

        # Create layout
        main_layout = QVBoxLayout()
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_widget = QWidget()
        form_layout = QFormLayout(scroll_widget)

        # Get current plugin language as an English word (default to English)
        if self.language == "DE":
            language = "German"
        elif self.language == "EN":
            language = "English"
        elif self.language == "FR":
            language = "French"
        elif self.language == "IT":
            language = "Italian"
        else:
            language = "English"

        # Add label + empty combobox pairs
        for field_name in self.fields:
            # LABEL
            label_name = next((term[language] for term in self.glossary if term["APITerm"] == field_name), None)  # Retrieve label name from glossary
            tooltip = next((term[language + '-Text'] for term in self.glossary if term["APITerm"] == field_name), None)   # Retrieve tooltip from glossary
            label = None
            if field_name in self.mandatory_fields:
                label = QLabel(label_name + " <span style='color: red'>*</span>")  # Create the label
            else:
                label = QLabel(label_name)  # Create the label
            label.setToolTip(tooltip)
            label.setObjectName(f"LABEL_{field_name}")  # Set label object name
            # COMBO BOX
            combo = QComboBox()  # Create combo box
            combo.setObjectName(f"COMBO_BOX_{field_name}")  # Set combo box object name
            # Add LABEL and COMBO BOX to layout
            form_layout.addRow(label, combo)

        scroll_area.setWidget(scroll_widget)
        main_layout.addWidget(scroll_area)

        # OK/Cancel buttons
        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        ok_button = buttons.button(QDialogButtonBox.Ok)
        cancel_button = buttons.button(QDialogButtonBox.Cancel)

        if self.language == "DE":
            ok_button.setText("OK")
            cancel_button.setText("ABBRECHEN")
        elif self.language == "EN":
            ok_button.setText("OK")
            cancel_button.setText("CANCEL")
        elif self.language == "FR":
            ok_button.setText("OK")
            cancel_button.setText("ANNULER")
        elif self.language == "IT":
            ok_button.setText("OK")
            cancel_button.setText("ANNULA")

        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)

        # Add buttons to main layout
        main_layout.addWidget(buttons)

        # Custom resizing based on number of fields
        height = 22 * len(self.fields) + 145
        self.resize(450, height)

        # Apply layout
        self.setLayout(main_layout)


class SaveLoadDialog(QDialog):
    """Custom dialog for saving and loading layers into the map."""
    def __init__(self, temp_layers):
        # Initialize class
        super().__init__()
        self.setWindowIcon(QIcon(os.path.join(os.path.dirname(__file__), "icon.png")))
        self.temp_layers = temp_layers
        # Force light mode when dark mode is detected
        if self.palette().color(QPalette.Window).value() < 100:
            self.setStyleSheet("""
                            QWidget { background-color: #F0F0F0; color: black; }  /* Light gray */
                            QLabel { color: black; }
                            QPushButton { background-color: #E0E0E0; border: 1px solid #C0C0C0; padding: 5px; }
                            QPushButton:hover { background-color: #D0D0D0; }
                            QLineEdit, QTextEdit { background-color: white; color: black; border: 1px solid #B0B0B0; }
                        """)
        # General window settings
        title = self.tr("EconoMe") +  " - " + self.tr("Risiko: Output-Layer")
        self.setWindowTitle(title)
        self.setFixedSize(400, 250)
        layout = QVBoxLayout()
        # Label & Input Field for Save Path
        self.label = QLabel(self.tr("Speicherort (optional):"))
        self.path_input = QLineEdit(self)
        self.path_input.setStyleSheet("color: black;")
        self.browse_button = QPushButton(self.tr("Speicherort auswählen ..."))
        self.browse_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.browse_button.clicked.connect(self.select_directory)
        # Buttons
        self.save_button = QPushButton(self.tr("Layer speichern und in die Karte laden"))
        self.save_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.save_button.clicked.connect(lambda: self.handle_layers(True))
        self.load_button = QPushButton(self.tr("Nur in Karte laden (als Temporär-Layer)"))
        self.load_button.setCursor(QCursor(Qt.PointingHandCursor))
        self.load_button.clicked.connect(lambda: self.handle_layers(False))

        layout.addWidget(self.label)
        layout.addWidget(self.path_input)
        layout.addWidget(self.browse_button)
        layout.addWidget(self.save_button)
        layout.addWidget(self.load_button)

        self.setLayout(layout)

    def select_directory(self):
        """Open file dialog to select a directory."""
        folder = QFileDialog.getExistingDirectory(self, "Select Directory")
        if folder:
            self.path_input.setText(folder)
            self.path_input.setStyleSheet("color: black;")

    def handle_layers(self, save):
        """Handle layers in terms of saving and loading them into the map."""

        # Save layers
        if save == True:
            save_path = self.path_input.text().strip()
            if not save_path:
                self.path_input.setText("Bitte Ordner ausgewählen ... (!)")
                self.path_input.setStyleSheet("color: red;")
                return
            # Loop over risk result layers
            for layer_tuple in self.temp_layers:
                # Get save path
                save_path = self.path_input.text().strip()
                # Get layer name and define save path
                layer_name = layer_tuple[0].name()
                save_path = os.path.join(save_path, f"{layer_name}.gpkg")
                # Save Vector Options
                options = QgsVectorFileWriter.SaveVectorOptions()
                options.driverName = "GPKG"
                options.fileEncoding = "UTF-8"
                # Ensure proper coordinate transformation context
                context = QgsCoordinateTransformContext()
                # Get QGIS version numbers
                qgis_major_version, qgis_minor_version = map(int, Qgis.QGIS_VERSION.split('.')[:2])
                # Avoid deprecation warnings while saving
                if qgis_major_version > 3 or (qgis_major_version == 3 and qgis_minor_version >= 40):  # QGIS 3.40+
                    QgsVectorFileWriter.writeAsVectorFormatV3(
                        layer_tuple[0], save_path, context, options
                    )
                elif qgis_major_version == 3 and qgis_minor_version >= 32:  # QGIS 3.32 to 3.39
                    error, new_file = QgsVectorFileWriter.writeAsVectorFormatV2(
                        layer_tuple[0], save_path, context, options
                    )
                saved_layer = QgsVectorLayer(f"{save_path}|layername={layer_name}", layer_name, "ogr")
                # Visualize layer
                self.visualize_layer(saved_layer, layer_tuple[1])
                # Add risk layer to map
                QgsProject.instance().addMapLayer(saved_layer)
            # Close the dialog
            self.accept()
        # Just load into map
        else:
            for layer_tuple in self.temp_layers:
                # Add risk layer to map
                QgsProject.instance().addMapLayer(layer_tuple[0])
            # Close the dialog
            self.accept()

    def visualize_layer(self, layer, dam_pot_type):

        # Get layer fields
        layer_fields = [field.name() for field in layer.fields()]
        # Visualize r_indiv
        if "r_indiv" in layer_fields:

            layer.startEditing()

            for feature in layer.getFeatures():
                try:
                    if feature["r_indiv"]:
                        feature["r_indiv"] = float(feature["r_indiv"])
                        layer.updateFeature(feature)
                except ValueError:
                    print(f"Invalid value: {feature['r_indiv']}")
            layer.commitChanges()

            ranges = []

            # Define first range
            symbol1 = QgsSymbol.defaultSymbol(layer.geometryType())
            symbol1.setColor(QColor("lightblue"))
            if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                symbol1.setWidth(1.5)
            range1 = QgsRendererRange(0, 0.000001, symbol1, "< 1e-6")
            ranges.append(range1)

            # Define second range
            symbol2 = QgsSymbol.defaultSymbol(layer.geometryType())
            if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                symbol2.setWidth(1.5)
            symbol2.setColor(QColor("orange"))
            range2 = QgsRendererRange(0.000001, 0.00001, symbol2, "1e-6 - 1e-5")
            ranges.append(range2)

            # Define third range
            symbol3 = QgsSymbol.defaultSymbol(layer.geometryType())
            if dam_pot_type in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                symbol3.setWidth(1.5)
            symbol3.setColor(QColor("red"))
            range3 = QgsRendererRange(0.00001, 1, symbol3, "> 1e-5")
            ranges.append(range3)

            renderer = QgsGraduatedSymbolRenderer("r_indiv", ranges)

            # Set the renderer to the layer
            layer.setRenderer(renderer)


class RightClickFilter(QObject):
    """Right-click event filter."""

    def __init__(self, callback):
        super().__init__()
        self.callback = callback  # Function to execute when right-click was detected

    def eventFilter(self, obj, event):
        # Filter only right-clicks and on the label
        if event.type() == QEvent.MouseButtonPress and event.button() == Qt.RightButton:
            damage_potential_type = "_".join(obj.objectName().split("_")[3:]).lower()  # Get damage potential type
            damage_potential_name = obj.text()
            self.callback(damage_potential_type, damage_potential_name)  # Execute function
            return True  # stop event propagation
        return False


class EconoMe:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):

        # Save reference to the QGIS interface
        self.iface = iface
        # Initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # --- Own variables
        # Login & API & User Projects
        self.ECONOME_USER_INFO = None
        self.ECONOME_USER_INFO_FILE_PATH = None
        self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/V2/project"
        self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/V2/profile"
        self.ECONOME_API_GLOSSARY_BASE_PATH = "https://api.econome.ch/V2/glossary"
        self.ECONOME_API_NAME = "EconoMe"
        # If the user already used the plugin >> use that language
        self.ECONOME_PLUGIN_LANGUAGE = QgsExpressionContextUtils.globalScope().variable("econome_language")
        if self.ECONOME_PLUGIN_LANGUAGE is None:  # Use qgis locale instead
            self.ECONOME_PLUGIN_LANGUAGE = QSettings().value('locale/userLocale')[0:2]
            if self.ECONOME_PLUGIN_LANGUAGE not in ["DE", "EN", "FR", "IT"]:
                self.ECONOME_PLUGIN_LANGUAGE = "EN"  # Default to English if no valid language is set
        self.ECONOME_PLUGIN_LANGUAGE_NAMES = {
            "DE": "German",
            "EN": "English",
            "FR": "French",
            "IT": "Italian"
        }
        self.ECONOME_PROJECT_NUMBER = None
        self.ECONOME_PROJECT_LIST = None
        self.ECONOME_USER_NAME = None
        self.ECONOME_PASSWORD = None
        self.LOGIN_ERROR_MESSAGE = None
        self.USER_INFO_VALIDATED = False
        # Animations
        self.PROGRESS_INFO_TEXT = None
        self.PROGRESS_GIF = None
        self.PROGRESS_OVERLAY = None
        self.PROGRESS_ANIMATION = None
        # Thread pipelines
        self.NEW_THREAD = None
        self.THREAD_PIPELINE = None
        self.OPEN_SAVE_LAYERS_DIALOG = None
        # Intensity Maps
        self.INTENSITY_LAYER_COLUMN_STATUS = {
            "FREE": None,
            "30": None,
            "100": None,
            "300": None,
            "EXTREME": None
        }
        # Damage potential
        self.DAMAGE_POTENTIAL_LAYER_LIST = None
        self.DAMAGE_POTENTIAL_ASSETS_HANDLING_OPTION = ""
        self.DAMAGE_POTENTIAL_LAYER_STATUS = {
            "BUILDING": 0,
            "ROAD": 0,
            "RAILWAY": 0,
            "CABLE_TRANSPORT": 0,
            "LINE_PIPE": 0,
            "OTHER_ASSET": 0
        }
        self.DAMAGEPOTENTIAL_FIELDS = {
            "building": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Occupancy", "OccupancyComment", "Presence", "PresenceComment"],
            "road": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Occupancy", "OccupancyComment", "Transits", "ConnectedRouteID"],
            "railway": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "TrainLength", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "DailyFreightTraffic", "Occupancy", "OccupancyComment", "DangerousTerrain", "Transits", "ValueEngine", "ConnectedRouteID"],
            "cable_transport": ["AssetIdentID", "AssetTypeID", "PassengerVolume", "Speed", "BeneficiaryID", "Description", "Count", "Value", "ValueComment", "Transits"],
            "line_pipe": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment"],
            "other_asset": ["AssetIdentID", "AssetTypeID", "BeneficiaryID", "Description", "Count", "Value", "ValueComment"]
        }
        self.DAMAGEPOTENTIAL_REQUIRED_FIELDS = {
            "building": ["AssetIdentID", "AssetTypeID"],
            "road": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed"],
            "railway": ["AssetIdentID", "AssetTypeID", "DailyTraffic", "Speed", "TrainLength"],
            "cable_transport": ["AssetIdentID", "AssetTypeID", "PassengerVolume", "Speed"],
            "line_pipe": ["AssetIdentID", "AssetTypeID"],
            "other_asset": ["AssetIdentID", "AssetTypeID"]
        }
        self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE = {
            "BUILDING": ['Gebäude', 'Sonderobjekte'],
            "ROAD": ['Strassenverkehr'],
            "RAILWAY": ['Schienenverkehr'],
            "CABLE_TRANSPORT": ['Mechanische Aufstiegshilfe'],
            "LINE_PIPE": ['Leitungen'],
            "OTHER_ASSET": ['Landwirtschaft, Wald und Grünanlagen', 'Infrastruktur']
        }
        self.ASSET_TYPEID_NAME_BY_CATEGORYNAME = {
            'Gebäude': {
                'Bahnhof': 14,
                'Einkaufszentrum': 9,
                'Frei wählbares Gebäude': 64,
                'Garage (Parkeinheit)': 4,
                'Hotel': 5,
                'Industrie-/ Gewerbegebäude': 6,
                'Keller (nur für Wasserprozesse)': 91,
                'Kirche': 12,
                'Öffentliches Gebäude': 18,
                'Öffentlicher Parkplatz': 88,
                'Schule / Kindergarten': 10,
                'Schuppen / Remise': 3,
                'Spital': 11,
                'Sportanlage (Gebäude)': 15,
                'Stationsgebäude mechanische Aufstiegshilfe': 63,
                'Viehstall': 2,
                'Wohneinheit Einfamilienhaus': 1,
                'Wohneinheit Mehrfamilienhaus': 87
            },
            'Sonderobjekte': {
                'ARA': 19,
                'Campingplatz': 21,
                'Zeltplatz': 89,
                'Deponie': 20,
                'Frei wählbares Sonderobjekt': 66,
                'Kraftwerk': 8,
                'Sendeanlage': 65,
                'Wasserreservoir': 22
            },
            'Strassenverkehr': {
                'Brücke Autobahn': 34,
                'Brücke Einzelfahrzeuge': 31,
                'Brücke Kantonsstrasse': 33,
                'Brücke Gemeindestrasse': 32,
                'Asphaltweg': 56,
                'Kiesweg': 55,
                'Frei wählbares Strassenverkehrsobjekt': 67,
                'Gemeindestrasse': 26,
                'Kantonsstrasse': 25,
                'Nationalstrasse': 24
            },
            'Leitungen': {
                'Abwasser unter Terrain': 60,
                'Frei wählbares Leitungsobjekt': 82,
                'Freileitung': 38,
                'Gas auf Terrain': 61,
                'Gas unter Terrain': 80,
                'Hochspannungsleitung <= 150 kV': 57,
                'Hochspannungsleitung > 150 kV': 58,
                'Strom unter Terrain': 37,
                'Telekommunikation auf Terrain': 62,
                'Telekommunikation unter Terrain': 81,
                'Wasser auf Terrain': 59,
                'Wasser unter Terrain': 79
            },
            'Mechanische Aufstiegshilfe': {
                'Frei wählbares Objekt (mech. Aufstiegshilfe)': 78,
                'Gondelbahn': 75,
                'Luftseilbahn': 76,
                'Sessellift': 74,
                'Skilift': 73,
                'Standseilbahn': 77
            },
            'Landwirtschaft, Wald und Grünanlagen': {
                'extensive Flächen': 43,
                'Frei wählbares Objekt (Landwirtschaft, Grünflächen, Wald)': 86,
                'Friedhof': 84,
                'Gemüsekultur': 47,
                'Golfanlage': 83,
                'intensive Flächen': 41,
                'Nutzwald': 45,
                'Obstplantage': 48,
                'Parkanlage': 42,
                'Rebberg': 46,
                'Schutzwald': 44,
                'Sportaussenanlage': 85
            },
            'Schienenverkehr': {
                'Brücke Einspur': 68,
                'Brücke Mehrspur': 69,
                'Einspur': 28,
                'Mehrspur': 29,
                'Sonderobjekt Brücke Schienenverkehr': 90
            },
            'Infrastruktur': {
                'Durchtrennung Kabelanlage (pro Schadenstelle)': 71,
                'Bahninfrastruktur': 72,
                'Schaltposten': 70,
                'Schutzbauwerk': 54
            }
        }
        self.QVARIANT_TYPES_DICT = {
            "integer": [QVariant.Int, QVariant.LongLong], 
            "float": [QVariant.Double], 
            "string": [QVariant.String], 
            "boolean": [QVariant.Bool]
        } 
        self.AFFECTED_ASSET_SPECIAL_PARAMETERS = {
            "BUILDING": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string"), ("PresenceValue", "numZeroToOne"), ("PresenceComment", "string"), ("ProtectionValue", "numZeroToOne"), ("ProtectionComment", "string")],
            "ROAD": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string"), ("p_vSp", "numZeroToOne"), ("p_vSp_Comment", "string"), ("AlarmSystem", "boolean")],
            "RAILWAY": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string"), ("p_vSp", "numZeroToOne"), ("p_vSp_Comment", "string"), ("AlarmSystem", "boolean")],
            "CABLE_TRANSPORT": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string")],
            "LINE_PIPE": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string")],
            "OTHER_ASSET": [("SOL_ID", "integer"), ("p_rA", "number"), ("p_rA_Comment", "string")]
        }
        self.RISK_OUTPUT_VARIABLES = {
            "BUILDING": ["R_Sw", "R_P(CHF)", "R", "r_indiv"],
            "ROAD": ["R_Sw", "R_P(CHF)", "R", "r_indiv", "R_Dir"],
            "RAILWAY": ["R_Sw", "R_P(CHF)", "R", "r_indiv", "R_Dir", "R_Anp"],
            "CABLE_TRANSPORT": ["R_Sw", "R_P(CHF)", "R", "r_indiv"],
            "LINE_PIPE": ["R_Sw"],
            "OTHER_ASSET": ["R_Sw"]
        }
        self.RISK_OUTPUT_TEMP_LAYERS = []

        # Initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{locale}.qm'.format(locale=locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&EconoMe')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Translate plugin
        self.switch_language(self.ECONOME_PLUGIN_LANGUAGE)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        action.setToolTip("EconoMe")
        action.setWhatsThis("EconoMe QGIS Plugin")

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def animate_progress(self, animate):
        """UX feature: Animates time during which plugin is working."""

        # Create animation
        if animate is True:
            # Create an overlay image to blur the background
            self.PROGRESS_OVERLAY = QLabel(self.main_dialog)
            self.PROGRESS_OVERLAY.setStyleSheet("background-color: rgba(255, 255, 255, 200);")
            # Add info text (optional)
            if self.THREAD_PIPELINE == "update_settings":
                self.PROGRESS_INFO = QLabel(self.main_dialog)
                if self.ECONOME_API_NAME == "EconoMe":
                    self.PROGRESS_INFO.setText("Explore API >>> EconoMe API")
                elif self.ECONOME_API_NAME == "Explore":
                    self.PROGRESS_INFO.setText("EconoMe API >>> Explore API")
                self.PROGRESS_INFO.setFixedSize(250, 30)
                self.PROGRESS_INFO.setAlignment(Qt.AlignCenter)
                self.PROGRESS_INFO.setWordWrap(False)
                self.PROGRESS_INFO.setStyleSheet("background-color: transparent; color: black; font-weight: bold;")
                x_pos = (self.main_dialog.width() - self.PROGRESS_INFO.width()) // 2
                y_pos = (self.main_dialog.height() - self.PROGRESS_INFO.height()) - 520 // 2
                self.PROGRESS_INFO.move(x_pos, y_pos)
                self.PROGRESS_INFO.setVisible(True)
            # Get turning cogs GIF and define QMovie
            self.PROGRESS_GIF = QLabel(self.main_dialog)
            self.PROGRESS_GIF.setVisible(False)
            self.PROGRESS_ANIMATION = QMovie(os.path.join(self.plugin_dir, "general-icons", "silver-gear-cogs.gif"))
            self.PROGRESS_GIF.setMovie(self.PROGRESS_ANIMATION)
            self.PROGRESS_GIF.setAlignment(Qt.AlignCenter)
            # Sizing animation and overlay
            self.PROGRESS_GIF.setFixedSize(281, 320)  # Fixed GIF size
            self.PROGRESS_OVERLAY.setFixedSize(self.main_dialog.width(), self.main_dialog.height()) # Overlay size
            self.PROGRESS_ANIMATION.setScaledSize(QSize(80, 80))  # Scale it
            # Absolute positioning in dialog
            x_pos = (self.main_dialog.width() - self.PROGRESS_GIF.width()) // 2
            y_pos = (self.main_dialog.height() - self.PROGRESS_GIF.height()) // 2
            self.PROGRESS_GIF.move(x_pos, y_pos)
            # Set GIF and overlay visible
            self.PROGRESS_OVERLAY.setVisible(True)
            self.PROGRESS_GIF.setVisible(True)
            self.PROGRESS_ANIMATION.start()
        # Stop animation and remove from UI
        else:
            # Clean up
            self.PROGRESS_ANIMATION.stop()
            self.PROGRESS_ANIMATION.deleteLater()
            self.PROGRESS_ANIMATION = None
            self.PROGRESS_GIF.setParent(None)
            self.PROGRESS_GIF.deleteLater()
            self.PROGRESS_GIF = None
            self.PROGRESS_OVERLAY.setParent(None)
            self.PROGRESS_OVERLAY.deleteLater()
            self.PROGRESS_OVERLAY = None
            if self.THREAD_PIPELINE == "update_settings":
                self.PROGRESS_INFO.setParent(None)
                self.PROGRESS_INFO.deleteLater()
                self.PROGRESS_INFO = None
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.clear()
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD.clear()
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY.clear()
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT.clear()
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE.clear()
                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET.clear()
            # Custom stuff
            if self.OPEN_SAVE_LAYERS_DIALOG == True and len(self.RISK_OUTPUT_TEMP_LAYERS) != 0:
                self.show_save_or_load_dialog()
            else:
                if self.THREAD_PIPELINE == "visualize_risks":
                    # USER INFO: Kein Risiko zum Visualisieren vorhanden!
                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler"), message_text = self.tr("\nKein Risiko zum Visualisieren vorhanden!\n")).exec_()
            # Reset thread pipeline
            self.THREAD_PIPELINE = ""

    def apply_settings(self, worker):
        """Applies user-defined settings."""

        # Set new API references
        if self.ECONOME_API_NAME == "EconoMe":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/V2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/V2/profile"

        elif self.ECONOME_API_NAME == "Explore":
            # API URLs
            self.ECONOME_API_PROJECT_BASE_PATH = "https://api.econome.ch/VEX2/project"
            self.ECONOME_API_PROFILE_BASE_PATH = "https://api.econome.ch/VEX2/profile"

    def change_settings(self):
        """Opens the settings menu and applies changes to plugin."""

        ## Initiate settings dialog
        dialog = EconoMeDialogSettingsMenu()
        title = self.tr("EconoMe") + " - " + self.tr("Einstellungen")
        dialog.setWindowTitle(title)
        dialog.COMBO_BOX_API_SELECTION.setCursor(QCursor(Qt.PointingHandCursor))
        dialog.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Ok).setCursor(QCursor(Qt.PointingHandCursor))
        dialog.BUTTON_BOX_ExecuteAlgorithm.button(QDialogButtonBox.Cancel).setCursor(QCursor(Qt.PointingHandCursor))

        ## Modify setting options
        # API selection
        API_list = ["EconoMe", "Explore"]
        for API in API_list:
            if API not in [dialog.COMBO_BOX_API_SELECTION.itemText(i) for i in range(dialog.COMBO_BOX_API_SELECTION.count())]:
                dialog.COMBO_BOX_API_SELECTION.addItem(API)
        dialog.COMBO_BOX_API_SELECTION.setCurrentText(self.ECONOME_API_NAME)
        # Language selection
        LANGUAGES_list = ["DE", "FR", "IT", "EN"]
        for LANGUAGE in LANGUAGES_list:
            if LANGUAGE not in [dialog.COMBO_BOX_LANGUAGE_SELECTION.itemText(i) for i in
                                range(dialog.COMBO_BOX_LANGUAGE_SELECTION.count())]:
                dialog.COMBO_BOX_LANGUAGE_SELECTION.addItem(LANGUAGE)
        dialog.COMBO_BOX_LANGUAGE_SELECTION.setCurrentText(self.ECONOME_PLUGIN_LANGUAGE)

        ## Define OK | CANCEL workflow
        result = dialog.exec_()
        # User applies (new) settings
        if result == QDialog.Accepted:

            # Change API name
            if dialog.COMBO_BOX_API_SELECTION.currentText() == "EconoMe" and self.ECONOME_API_NAME != "EconoMe":
                self.ECONOME_API_NAME = "EconoMe"
                self.OPEN_SAVE_LAYERS_DIALOG = False
                self.THREAD_PIPELINE = "update_settings"
                self.execute_in_separate_thread(self.apply_settings)
            elif dialog.COMBO_BOX_API_SELECTION.currentText() == "Explore" and self.ECONOME_API_NAME != "Explore":
                self.ECONOME_API_NAME = "Explore"
                self.OPEN_SAVE_LAYERS_DIALOG = False
                self.THREAD_PIPELINE = "update_settings"
                self.execute_in_separate_thread(self.apply_settings)

            ### Language switch ###
            new_language = dialog.COMBO_BOX_LANGUAGE_SELECTION.currentText()
            # Only perform translation if language was changed
            if self.ECONOME_PLUGIN_LANGUAGE != new_language:
                self.ECONOME_PLUGIN_LANGUAGE = dialog.COMBO_BOX_LANGUAGE_SELECTION.currentText()
                # translate the plugin
                self.switch_language(self.ECONOME_PLUGIN_LANGUAGE)

                # update the variable econome_language in the qgis settings
                QgsExpressionContextUtils.setGlobalVariable('econome_language', self.ECONOME_PLUGIN_LANGUAGE)

                # restart the plugin to make sure the ui is translated
                self.main_dialog.close()
                self.run()

    def clean_up(self, file_path):
        """This method removes a file from the disk."""

        # Delete file
        if file_path is not None:
            if os.path.exists(file_path):
                os.remove(file_path)

    def compare_damage_potential(self, layer_assets, uploaded_damage_potential, damage_potential_type, valid_parameter_pairs):
        """This method compares the damage potential from a GIS layer with stored data in EconoMe."""

        # Check for existing uploaded damage potential
        if len(uploaded_damage_potential) != 0:
            # Create two "AssetIdentID" lists
            uploaded_AssetIdentIDs = [asset["AssetIdentID"] for asset in uploaded_damage_potential]
            layer_AssetIdentIDs = [str(asset["AssetIdentID"]) for asset in layer_assets["AssetList"]]
            # Create empty attribute mismatch "AssetIdentID" list
            mismatch_dict = {}
            # Get layer AssetTypeIDs
            layer_AssetTypeIDs = [asset["AssetTypeID"] for asset in layer_assets["AssetList"]]
            # Check if AssetIdentIDs match the damage potential type
            OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]] for item in sublist]
            FILTERED_LAYER_ASSET_TYPE_IDS = [AssetTypeID for AssetTypeID in layer_AssetTypeIDs if AssetTypeID not in OBJECT_CATEGORIES]
            # Check if damage potential type was changed to another category
            if len(FILTERED_LAYER_ASSET_TYPE_IDS) == 0:
                # CASE: All uploaded assets are present in the damage potential layer
                if set(uploaded_AssetIdentIDs).issubset(layer_AssetIdentIDs):
                    # Check for attribute differences
                    for layer_asset in layer_assets["AssetList"]:
                        asset_missmatch_dict = {}
                        corresponding_uploaded_asset = [asset for asset in uploaded_damage_potential if asset["AssetIdentID"] == str(layer_asset["AssetIdentID"])]
                        if len(corresponding_uploaded_asset) > 0:
                            for pp in valid_parameter_pairs:
                                if pp["API"] in layer_asset.keys():
                                    if layer_asset[pp["API"]] != corresponding_uploaded_asset[0][pp["API"]]:
                                        asset_missmatch_dict.update({pp["API"]: {"Layer": layer_asset[pp["API"]], "API": corresponding_uploaded_asset[0][pp["API"]]}})
                        else:
                            asset_missmatch_dict = {"Missing":self.tr("Fehlt in EconoMe.")}
                        if asset_missmatch_dict != {}:
                            mismatch_dict.update({layer_asset["AssetIdentID"]: asset_missmatch_dict})
                    if len(mismatch_dict.keys()) == 0:
                        return "COMPLETE MATCH", mismatch_dict
                    else:
                        handling_dialog = DamagePotentialMismatchResolveDialog(mode = 'MISMATCH', mismatch = mismatch_dict, main_dialog = self.main_dialog)
                        if handling_dialog.exec_() == QDialog.Accepted:
                            return handling_dialog.selected_option, mismatch_dict
                        else:
                            return "CANCEL", {}
                # CASE: not all uploaded assets are present in the damage potential layer
                elif not set(layer_AssetIdentIDs).issubset(uploaded_AssetIdentIDs) or len(set(uploaded_AssetIdentIDs)) > len(layer_AssetIdentIDs):
                    mismatch_dict = {AssetIdentID : {"Missing":"Fehlt im Layer."} for AssetIdentID in list(set(uploaded_AssetIdentIDs) - set(layer_AssetIdentIDs))}
                    handling_dialog = DamagePotentialMismatchResolveDialog(mode = 'MISSING', mismatch = mismatch_dict, main_dialog = self.main_dialog)
                    if handling_dialog.exec_() == QDialog.Accepted:
                        return handling_dialog.selected_option, mismatch_dict
                    else:
                        return "CANCEL", {}
            else:
                # CASE PREVENT SYNC: AssetIdentID doesn't match object category
                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials!"), message_text=self.tr("\nEine oder mehrere Objektart-ID wurde/n im Layer zu einer nicht der Kategorie des Schadenpotentials entprechenden ID geändert.\nAus diesem Grund wird der Layer durch EconoMe zurückgesetzt.\n")).exec_()
                return "KEEP ECONOME ASSET TYPE IDS", {}
        else:
            # CASE ONLY POST: No uploaded damage potential existing
            # Get layer AssetTypeIDs
            layer_AssetTypeIDs = [asset["AssetTypeID"] for asset in layer_assets["AssetList"]]
            # Check if AssetIdentIDs match the damage potential type
            OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]] for item in sublist]
            FILTERED_LAYER_ASSET_IDENT_IDS = [AssetIdentID for AssetIdentID in layer_AssetTypeIDs if AssetIdentID not in OBJECT_CATEGORIES]
            # Check if damage potential type was changed to another category
            if len(FILTERED_LAYER_ASSET_IDENT_IDS) == 0:
                return "POST LAYER ASSETS", {}
            else:
                # CASE PREVENT UPLOAD: AssetIdentID doesn't match object category
                return "INVALID OBJECT CATEGORY", {}

    def create_template_layer(self, dam_pot_type, dam_pot_name):
        """This method creates a template layer for a certain damage potential type with all necessary fields."""

        # CHECK: User logged in?
        if self.USER_INFO_VALIDATED:

            # Set template layer name
            template_layer_name = self.tr("Schadenobjekte: {DAMAGE_POTENTIAL_TYPE}").format(DAMAGE_POTENTIAL_TYPE=dam_pot_name)

            # Create a new vector layer of a specific geometry
            if dam_pot_type in ["building", "other_asset"]:  # Polygon geometry
                template_layer = QgsVectorLayer("Polygon?crs=EPSG:2056", template_layer_name, "memory")
            elif dam_pot_type in ["road", "railway", "cable_transport", "line_pipe"]:  # Line geometry
                template_layer = QgsVectorLayer("LineString?crs=EPSG:2056", template_layer_name, "memory")
            else:  # Print a warning message to the user
                StatusDialog(
                    status_icon = self.scaled_pixmap_warning,
                    message_title=self.tr("FEHLER"),
                    message_text=self.tr("\nGIS Layer-Vorlage für den Schadenpotentialtyp {DAMAGE_POTENTIAL_TYPE} konnte nicht erstellt werden.\nUnkompatible QGIS-Version!\n").format(DAMAGE_POTENTIAL_TYPE=dam_pot_type.capitalize())
                ).exec_()
                template_layer = None

            # CHECK: Template layer ?
            if template_layer:

                # Get default parameter profile
                with requests.get(
                    self.ECONOME_API_PROFILE_BASE_PATH,
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as get_default_parameter_profile:
                    DEFAULT_PARAMETER_PROFILE = get_default_parameter_profile.json()

                TYPE_DICT = {item['API']: item['Type'] for item in DEFAULT_PARAMETER_PROFILE['ParameterList'] if item["API"] in self.DAMAGEPOTENTIAL_FIELDS[dam_pot_type]}  # Create a parameter/type dictionary
                POSSIBLE_PARAMETERS = self.DAMAGEPOTENTIAL_FIELDS[dam_pot_type]  # List of possible parameters

                # Get translated parameter names in glossary
                with requests.get(
                    self.ECONOME_API_GLOSSARY_BASE_PATH + f"/{self.ECONOME_PLUGIN_LANGUAGE.lower()}",
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as get_glossary_de:
                    GLOSSARY = get_glossary_de.json()

                GLOSSARY_DICT = {item["APITerm"]: item[self.ECONOME_PLUGIN_LANGUAGE_NAMES[self.ECONOME_PLUGIN_LANGUAGE]] for item in GLOSSARY}  # Glossary dictionary

                # Harvest QgsField list
                fields = template_layer.fields()
                for parameter in POSSIBLE_PARAMETERS:

                    # Find correct attribute type
                    parameter_type = TYPE_DICT[parameter]
                    if parameter_type == "integer":
                        field_type = QVariant.Int
                    elif parameter_type == "float":
                        field_type = QVariant.Double
                    elif parameter_type == "string":
                        field_type = QVariant.String
                    elif parameter_type == "boolean":
                        field_type = QVariant.Bool
                    else:
                        raise ValueError(f"Value type of attribute {parameter} not recognized.")

                    # Append to fields list
                    field_name = GLOSSARY_DICT[parameter]
                    fields.append(QgsField(field_name, field_type))

                # Apply the changes to the layer
                template_layer.startEditing()
                data_provider = template_layer.dataProvider()
                data_provider.addAttributes(fields)
                template_layer.updateFields()
                template_layer.commitChanges()

                try:  # Add template layer to QGIS project table of contents
                    QgsProject.instance().addMapLayer(template_layer)
                    # Message user for successful template layer creation
                    StatusDialog(
                        status_icon=self.scaled_pixmap_checkmark,
                        message_title=self.tr("ERFOLG"),
                        message_text=self.tr("\nGIS Layer-Vorlage für den Schadenpotentialtyp {DAMAGE_POTENTIAL_NAME} wurde erfolgreich erstellt.\n").format(DAMAGE_POTENTIAL_NAME=dam_pot_name)
                    ).exec_()
                except:
                    # Message user for unsuccessful template layer creation
                    StatusDialog(
                        status_icon=self.scaled_pixmap_warning,
                        message_title=self.tr("FEHLER"),
                        message_text=self.tr("\nGIS Layer-Vorlage für den Schadenpotentialtyp {DAMAGE_POTENTIAL_NAME} konnte nicht erstellt werden.\n").format(DAMAGE_POTENTIAL_NAME=dam_pot_name)
                    ).exec_()

    def delete_damage_potential_affected_assets(self, annuality):
        """This method deletes affected damage potential assets by type."""

        # Filter scenes and get corresponding Scene ID
        with requests.get(
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/scene',
                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
        ) as get_scenes:
            SCENES = get_scenes.json()
            process_ID = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            variant_ID = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]
            scene_ID = [scene["SceneID"] for scene in SCENES if
                        int(scene["Annuality"]) == int(annuality) and int(scene["ProcessID"]) == int(
                            process_ID) and int(scene["VariantID"]) == int(variant_ID)][0]
            # Get existing affected assets
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_affected_assets:
                UPLOADED_AFFECTED_ASSETS = get_affected_assets.json()
                # CHECK: Existing affected assets for specified scene ?
                if len(UPLOADED_AFFECTED_ASSETS) != 0:
                    are_you_sure = AreYouSure(self.tr("\nSind sie sicher, dass sie die Betroffenheit der Schadenpotentialobjekte für das {annuality}-jährliche Szenario löschen möchten ?\n").format(annuality=annuality))
                    user_choice = are_you_sure.exec_()
                    if user_choice == QDialog.Accepted:
                        # Delete filtered damage potential assets
                        with requests.delete(
                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as delete_affected_assets:
                            # SUCCESS
                            if delete_affected_assets.status_code == 200:
                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title = self.tr("Erfolg"), message_text = self.tr("\nBetroffenheit der Schadenpotentialobjekte erfolgreich gelöscht!\n")).exec_()
                            # ERROR
                            else:
                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler"), message_text = f"\n{delete_affected_assets.json()}\n").exec_()
                else:
                    # USER INFO: Keine AffectedAssets vorhanden
                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler"), message_text = self.tr("\nKeine betroffenen Schadenpotentialobjekte definiert!\n")).exec_()

    def delete_damage_potential_assets(self, damage_potential_type, status_label):
        """This method deletes damage potential assets by object category."""

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # Get current damage potential
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_damage_potential:
                UPLOADED_DAMAGE_POTENTIAL = get_damage_potential.json()
                # CHECK: Existing damage potential in EconoMe
                if len(UPLOADED_DAMAGE_POTENTIAL) != 0:
                    # Filter damage potential by object category
                    OBJECT_CATEGORIES = [item for sublist in
                                         [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in
                                          self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type.upper()]]
                                         for item in sublist]
                    FILTERED_UPLOADED_DAMAGE_POTENTIAL = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if
                                                          asset["AssetTypeID"] in OBJECT_CATEGORIES]
                    # CHECK: Are there assets of the corresponding damage potential type ?
                    if len(FILTERED_UPLOADED_DAMAGE_POTENTIAL) != 0:
                        DAMAGE_POTENTIAL_LABEL_TEXT = getattr(self.main_dialog, f"LABEL_DAM_POT_{damage_potential_type.upper()}").text()
                        are_you_sure = AreYouSure(self.tr("\nSind Sie sicher, dass Sie alle Schadenobjekte der Objektkategorie '{DAMAGE_POTENTIAL_LABEL_TEXT}' in EconoMe löschen möchten ?\n").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT))
                        user_choice = are_you_sure.exec_()
                        if user_choice == QDialog.Accepted:
                            # Delete filtered damage potential assets
                            patch_asset_ids = {
                                "AssetIDList": [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL]}
                            with requests.patch(
                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                    json=patch_asset_ids,
                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                            ) as patch_damage_potential:
                                # SUCCESS
                                if patch_damage_potential.status_code == 200:
                                    StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title = self.tr("Erfolg: Schadenpotential {DAMAGE_POTENTIAL_LABEL_TEXT} erfolgreich gelöscht").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text = self.tr("\nAlle Schadenobjekte der Objektkategorie '{DAMAGE_POTENTIAL_LABEL_TEXT}' in EconoMe gelöscht!\n").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT)).exec_()
                                    status_label.clear()
                                # ERROR
                                else:
                                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler beim Löschen des Schadenpotentials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text = self.tr("\nAPI Fehlermeldung: {response}\n").format(response=patch_damage_potential.json())).exec_()
                    else:
                        # USER INFO: Kein Schadenpotential vorhanden
                        StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler beim Löschen des Schadenpotentials {category}").format(category=damage_potential_type.upper()), message_text = self.tr("\nIn EconoMe ist kein Schadenpotential Objektkategorie '{category}' definiert!\n").format(category=damage_potential_type.upper())).exec_()
                else:
                    # USER INFO: Kein Schadenpotential vorhanden
                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler beim Löschen des Schadenpotentials {category}").format(category=damage_potential_type.upper()), message_text = self.tr("\nIn EconoMe ist kein Schadenpotential Objektkategorie '{category}' definiert!\n").format(category=damage_potential_type.upper())).exec_()

    def disable_buttons_and_dropdowns(self, disable):
        """Disable UI elements until user is logged in."""

        if disable is True:
            # Damage potential layer dropdowns
            self.main_dialog.LAYER_DAM_POT_BUILDING.setEnabled(False)
            self.main_dialog.LAYER_DAM_POT_ROAD.setEnabled(False)
            self.main_dialog.LAYER_DAM_POT_RAILWAY.setEnabled(False)
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setEnabled(False)
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setEnabled(False)
            # Parameter profile buttons
            self.main_dialog.BUTTON_PP_BUILDING.setEnabled(False)
            self.main_dialog.BUTTON_PP_ROAD.setEnabled(False)
            self.main_dialog.BUTTON_PP_RAILWAY.setEnabled(False)
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.BUTTON_PP_LINE_PIPE.setEnabled(False)
            self.main_dialog.BUTTON_PP_OTHER_ASSET.setEnabled(False)
            # Damage potential sync buttons
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.setEnabled(False)
            # Damage potential delete buttons
            self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.setEnabled(False)
            self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.setEnabled(False)
            # Affected assets delete buttons
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.setEnabled(False)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.setEnabled(False)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.setEnabled(False)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.setEnabled(False)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.setEnabled(False)
            # Process and Measures dropdowns
            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setEnabled(False)
            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setEnabled(False)
            # Intensity layer and column dropdowns
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setEnabled(False)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setEnabled(False)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setEnabled(False)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setEnabled(False)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.setEnabled(False)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.setEnabled(False)
            # Consequence analysis, Visualize risk buttons
            self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.setEnabled(False)
            self.main_dialog.BUTTON_VISUALIZE_RISKS.setEnabled(False)
        else:
            # Damage potential layer dropdowns
            self.main_dialog.LAYER_DAM_POT_BUILDING.setEnabled(True)
            self.main_dialog.LAYER_DAM_POT_ROAD.setEnabled(True)
            self.main_dialog.LAYER_DAM_POT_RAILWAY.setEnabled(True)
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setEnabled(True)
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setEnabled(True)
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setEnabled(True)
            # Parameter profile buttons
            self.main_dialog.BUTTON_PP_BUILDING.setEnabled(True)
            self.main_dialog.BUTTON_PP_ROAD.setEnabled(True)
            self.main_dialog.BUTTON_PP_RAILWAY.setEnabled(True)
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.setEnabled(True)
            self.main_dialog.BUTTON_PP_LINE_PIPE.setEnabled(True)
            self.main_dialog.BUTTON_PP_OTHER_ASSET.setEnabled(True)
            # Damage potential sync buttons
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.setEnabled(True)
            # Damage potential delete buttons
            self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.setEnabled(True)
            self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.setEnabled(True)
            # Affected assets delete buttons
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.setEnabled(True)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.setEnabled(True)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.setEnabled(True)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.setEnabled(True)
            self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.setEnabled(True)
            # Process and Measures dropdowns
            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setEnabled(True)
            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setEnabled(True)
            # Intensity layer and column dropdowns
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setEnabled(True)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setEnabled(True)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setEnabled(True)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setEnabled(True)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setEnabled(True)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.setEnabled(True)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.setEnabled(True)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.setEnabled(True)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.setEnabled(True)
            self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.setEnabled(True)
            # Consequence analysis, Visualize risk buttons
            self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.setEnabled(True)
            self.main_dialog.BUTTON_VISUALIZE_RISKS.setEnabled(True)

    def execute_in_separate_thread(self, task_function):
        """Uses a worker thread to execute a function in parallel."""

        # Working animation
        self.animate_progress(True)

        # Run the "real" task in a QThread
        if self.THREAD_PIPELINE == "update_settings":
            self.NEW_THREAD = WorkerThread(self, task_function, True)
            self.NEW_THREAD.revalidate_econome_user_info.connect(self.validate_econome_user_info)
        else:
            self.NEW_THREAD = WorkerThread(self, task_function, False)

        # Connect signals and start worker thread
        self.NEW_THREAD.show_status.connect(self.status_in_separate_thread)
        self.NEW_THREAD.finished.connect(lambda: self.animate_progress(False))
        self.NEW_THREAD.start()

    def get_econome_user_info(self):
        """Reads .txt file containing EconoMe user info (project numbers and API keys)"""

        # CHECK: Existing user info file
        prefix = "econome_user_info_"
        files_in_directory = os.listdir(self.plugin_dir)
        matching_files = [file for file in files_in_directory if file.startswith(prefix) and file.endswith(".txt")]
        # If there is a file, read and return EconoMe user infos
        if matching_files:
            # Get file path
            self.ECONOME_USER_INFO_FILE_PATH = os.path.join(self.plugin_dir, matching_files[0])
            # Read out data
            file = open(self.ECONOME_USER_INFO_FILE_PATH, "r")
            info_ID = file.readline().strip()
            info_ID_list = file.readline().strip()
            info_KeyA = file.readline().strip()
            info_KeyB = file.readline().strip()
            file.close()
            # Return EconoMe user infos
            return {"ECONOME_PROJECT_NUMBER": info_ID, "ECONOME_PROJECT_LIST": info_ID_list.split(', '), "ECONOME_USER_NAME": info_KeyA, "ECONOME_PASSWORD": info_KeyB}
        else:
            # Return an empty dictionary (if no file is found)
            self.ECONOME_USER_INFO_FILE_PATH = None
            return None

    def get_hazard_intensity(self, damage_potential_layer, intensity_layer, intensity_layer_column, damage_potential_type):
        # For point features or buildings
        if damage_potential_layer.geometryType() == 0 or damage_potential_type == "BUILDING":
            # Define the parameters for the processing algorithm
            params = {
                'INPUT': damage_potential_layer,
                'JOIN': intensity_layer,
                'PREDICATE': [0],  # Spatial predicate: [0] means "intersects"
                'JOIN_FIELDS': [intensity_layer_column],
                'DISCARD_NONMATCHING': False,
                'PREFIX': '',
                'SUMMARIES': [3],  # Summary type: Maximum Value
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }

            context = dataobjects.createContext()
            context.setInvalidGeometryCheck(QgsFeatureRequest.GeometryNoCheck)

            # Run the processing tool
            result = processing.run("qgis:joinbylocationsummary", params, context=context)

            return result['OUTPUT']

        # For line features
        elif damage_potential_layer.geometryType() == 1:
            # Create a copy of the input damage potential layer
            type_and_projection = f"LineString?crs={damage_potential_layer.crs().postgisSrid()}"
            temp_layer = QgsVectorLayer(type_and_projection, "Temporary Layer", "memory")
            damage_potential_layer_fields = damage_potential_layer.fields()

            temp_layer.startEditing()

            temp_layer.dataProvider().addAttributes(damage_potential_layer_fields)

            # Copy all features from the original layer to the new layer
            for feature in damage_potential_layer.getFeatures():
                temp_layer.dataProvider().addFeature(feature)

            # Add new fields to the line layer to store overlay lengths
            temp_layer.dataProvider().addAttributes([
                QgsField('intensity_weak', QVariant.Double),
                QgsField('intensity_moderate', QVariant.Double),
                QgsField('intensity_strong', QVariant.Double)
            ])
            temp_layer.updateFields()

            # Create a dictionary to store lengths for each line feature
            line_lengths = {f.id(): {'low': 0, 'moderate': 0, 'high': 0} for f in temp_layer.getFeatures()}

            # Perform intersection and calculate lengths
            for line_feat in temp_layer.getFeatures():
                line_geom = line_feat.geometry()

                for poly_feat in intensity_layer.getFeatures():
                    poly_geom = poly_feat.geometry()
                    intensity = poly_feat[intensity_layer_column]

                    # Intersect line with polygon
                    intersection = line_geom.intersection(poly_geom)

                    if not intersection.isEmpty():
                        intensity_class = poly_feat[intensity_layer_column]
                        length = intersection.length()
                        if intensity_class == 1:
                            line_lengths[line_feat.id()]["low"] += length
                        elif intensity_class == 2:
                            line_lengths[line_feat.id()]["moderate"] += length
                        elif intensity_class == 3:
                            line_lengths[line_feat.id()]["high"] += length

            # Update the line layer with the calculated lengths
            for line_feat in temp_layer.getFeatures():
                line_feat['intensity_weak'] = line_lengths[line_feat.id()]['low']
                line_feat['intensity_moderate'] = line_lengths[line_feat.id()]['moderate']
                line_feat['intensity_strong'] = line_lengths[line_feat.id()]['high']
                temp_layer.updateFeature(line_feat)

            temp_layer.commitChanges()

            return temp_layer

        # For polygon features
        elif damage_potential_layer.geometryType() == 2:
            # Create a copy of the input damage potential layer
            type_and_projection = f"Polygon?crs={damage_potential_layer.crs().postgisSrid()}"
            temp_layer = QgsVectorLayer(type_and_projection, "Temporary Layer", "memory")
            damage_potential_layer_fields = damage_potential_layer.fields()

            temp_layer.startEditing()

            temp_layer.dataProvider().addAttributes(damage_potential_layer_fields)

            # Copy all features from the original layer to the new layer
            for feature in damage_potential_layer.getFeatures():
                temp_layer.dataProvider().addFeature(feature)

            # Add new fields to the line layer to store overlay lengths
            temp_layer.dataProvider().addAttributes([
                QgsField('intensity_weak', QVariant.Double),
                QgsField('intensity_moderate', QVariant.Double),
                QgsField('intensity_strong', QVariant.Double)
            ])
            temp_layer.updateFields()

            # Create a dictionary to store lengths for each line feature
            polygon_areas = {f.id(): {'low': 0, 'moderate': 0, 'high': 0} for f in temp_layer.getFeatures()}

            # Perform intersection and calculate lengths
            for poly_feat in temp_layer.getFeatures():
                poly_geom = poly_feat.geometry()
                for intensity_feature in intensity_layer.getFeatures():
                    intensity_geom = intensity_feature.geometry()
                    intensity = intensity_feature[intensity_layer_column]

                    # Intersect polygon with polygon
                    intersection = poly_geom.intersection(intensity_geom)

                    if not intersection.isEmpty():
                        intensity_class = intensity_feature[intensity_layer_column]
                        area = intersection.area() / 100
                        if intensity_class == 1:
                            polygon_areas[poly_feat.id()]["low"] += area
                        elif intensity_class == 2:
                            polygon_areas[poly_feat.id()]["moderate"] += area
                        elif intensity_class == 3:
                            polygon_areas[poly_feat.id()]["high"] += area

            # Update the line layer with the calculated lengths
            for poly_feat in temp_layer.getFeatures():
                poly_feat['intensity_weak'] = polygon_areas[poly_feat.id()]['low']
                poly_feat['intensity_moderate'] = polygon_areas[poly_feat.id()]['moderate']
                poly_feat['intensity_strong'] = polygon_areas[poly_feat.id()]['high']
                temp_layer.updateFeature(poly_feat)

            temp_layer.commitChanges()

            return temp_layer

    def get_project_status(self):
        """Retrieves EconoMe project status information and fills plugin UI."""

        # Create project status dictionary
        PROJECT_STATUS = {
            "HazardProcesses": {
                "Count": 0
            },
            "Variants": {
                "Count": 0
            },
            "Scenarios": {
                "Active": []
            }
        }

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:

            API_GET_responses = self.multiple_API_GET_requests([
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/process',
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/variant'
            ])

            HAZARD_PROCESSES = API_GET_responses[0].result()[1]
            MEASURE_VARIANTS = API_GET_responses[1].result()[1]

            # CHECK: Response is a list
            if isinstance(HAZARD_PROCESSES, list):
                PROJECT_STATUS["HazardProcesses"]["Count"] = len(HAZARD_PROCESSES)
                self.main_dialog.LABEL_HAZARD_PROCESSES.setText(f"{self.tr('GEFAHRENPROZESS(E)')} ({len(HAZARD_PROCESSES)})")
                hazard_processes_list = [(process["ProcessID"], process["ProcessType"], process["ProcessName"]) for process in HAZARD_PROCESSES]
                self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
                # Fill the drop-down with all hazard processes
                for hazard_process in hazard_processes_list:
                    if f"{hazard_process[0]}, {hazard_process[1]}, {hazard_process[2]}" not in [self.main_dialog.COMBO_BOX_HAZARD_PROCESS.itemText(i) for i in range(self.main_dialog.COMBO_BOX_HAZARD_PROCESS.count())]:
                        self.main_dialog.COMBO_BOX_HAZARD_PROCESS.addItem(f"{hazard_process[0]}, {hazard_process[1]}, {hazard_process[2]}")
                        self.main_dialog.COMBO_BOX_HAZARD_PROCESS.setCurrentText(f"{hazard_process[0]}, {hazard_process[1]}, {hazard_process[2]}")
                # Get selected hazard process
                selected_hazard_process = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText()
                selected_hazard_process_info = [hazard_process for hazard_process in HAZARD_PROCESSES if hazard_process["ProcessID"] == int(selected_hazard_process.split(",")[0])]
                if len(selected_hazard_process_info) != 0:
                    # Set annualities for the two free scenarios
                    self.main_dialog.LABEL_SCENARIO_FREE.setText(str(selected_hazard_process_info[0]["AnnualityFree"]["Annuality"]))
                    self.main_dialog.LABEL_SCENARIO_EXTREME.setText(str(selected_hazard_process_info[0]["AnnualityExtreme"]["Annuality"]))

            else: # If not clear drop-down with hazard processes
                HAZARD_PROCESSES = []
                self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
                self.main_dialog.LABEL_HAZARD_PROCESSES.setText(f"{self.tr('GEFAHRENPROZESS(E)')} ({len(HAZARD_PROCESSES)})")

            # CHECK: Response is a list
            if isinstance(MEASURE_VARIANTS, list):
                PROJECT_STATUS["Variants"]["Count"] = len(MEASURE_VARIANTS)
                self.main_dialog.LABEL_MEASURE_VARIANTS.setText(f"{self.tr('MASSNAHMENVARIANTE(N)')} ({PROJECT_STATUS['Variants']['Count']})")
                measure_variants_list = [(variant["VariantID"], variant["VariantName"]) for variant in MEASURE_VARIANTS]
                self.main_dialog.COMBO_BOX_MEASURE_VARIANT.clear()
                measure_variants_list.insert(0, (0, self.tr("Vor Massnahme")))
                for measure_variant in measure_variants_list:
                    if f"{measure_variant[0]}, {measure_variant[1]}" not in [self.main_dialog.COMBO_BOX_MEASURE_VARIANT.itemText(i) for i in range(self.main_dialog.COMBO_BOX_MEASURE_VARIANT.count())]:
                        self.main_dialog.COMBO_BOX_MEASURE_VARIANT.addItem(f"{measure_variant[0]}, {measure_variant[1]}")
                        self.main_dialog.COMBO_BOX_MEASURE_VARIANT.setCurrentText(f"{measure_variant[0]}, {measure_variant[1]}")

            # Clear labels
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET.clear()
            
            return PROJECT_STATUS

        # EconoMe user info not validated
        else:
            # Clear labels
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE.clear()
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET.clear()
            # Reset drop-downs
            self.main_dialog.LAYER_DAM_POT_BUILDING.setCurrentIndex(0)
            self.main_dialog.LAYER_DAM_POT_ROAD.setCurrentIndex(0)
            self.main_dialog.LAYER_DAM_POT_RAILWAY.setCurrentIndex(0)
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setCurrentIndex(0)
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setCurrentIndex(0)
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setCurrentIndex(0)
            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.clear()
            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.clear()
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setCurrentIndex(0)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setCurrentIndex(0)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setCurrentIndex(0)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setCurrentIndex(0)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setCurrentIndex(0)
            # Reset drop-down texts
            self.main_dialog.LABEL_HAZARD_PROCESSES.setText(self.tr("GEFAHRENPROZESS(E)"))
            self.main_dialog.LABEL_MEASURE_VARIANTS.setText(self.tr("MASSNAHMENVARIANTE(N)"))
            # Reset custom scenario labels
            self.main_dialog.LABEL_SCENARIO_FREE.setText(self.tr(' Frei'))
            self.main_dialog.LABEL_SCENARIO_EXTREME.setText(self.tr(' Extrem'))
            # Return empty dictionary
            return {}

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        self.add_action(
            icon_path,
            text=self.tr(u'EconoMe'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def multiple_API_GET_requests(self, URL_GET_REQUESTS):
        """Execute multiple API requests at once."""

        def fetch(url):
            try:
                response = requests.get(
                    url,
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                )
                return url, response.json()
            except Exception as e:
                return url, f"Error: {e}"

        with ThreadPoolExecutor(max_workers=5) as executor:
            futures = [executor.submit(fetch, url) for url in URL_GET_REQUESTS]
            return futures

    def onChange_econome_user_info(self):
        """Resets plugin UI when EconoMe user name or password is modified."""

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # EconoMe user info -> not validated
            self.USER_INFO_VALIDATED = False
            # Clear project combo box
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
            # Disable buttons and dropdowns
            self.disable_buttons_and_dropdowns(True)
            # Display messsage -> not connected
            self.main_dialog.LABEL_ECONOME_PROJECT_NUMBER.setText(self.tr("Projektstatus "))
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(self.tr("Nicht verbunden"))
            # Remove green color from login button
            self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet("background-color: none;")
            # Get project status
            self.get_project_status()

    def onChange_project_selection(self):
        """Switches project number and updates EconoMe user info."""

        # CHECK: EconoMe user info validated
        if self.USER_INFO_VALIDATED:
            # CHECK: Project selected
            if self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText():
                # CHECK: Project selection changed
                if self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(', ')[-1] != self.ECONOME_PROJECT_NUMBER:
                    # Set new project number
                    self.ECONOME_PROJECT_NUMBER = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.currentText().split(', ')[-1]
                    # Overwrite project infos in the temporary text file which persists until QGIS is closed
                    if isinstance(self.ECONOME_USER_INFO_FILE_PATH, str):
                        with open(self.ECONOME_USER_INFO_FILE_PATH, "w") as file:
                            file.write(f"{self.ECONOME_PROJECT_NUMBER}\n")
                            file.write(f"{', '.join([item[0] for item in self.ECONOME_PROJECT_LIST])}\n")
                            file.write(f"{self.ECONOME_USER_NAME}\n")
                            file.write(f"{self.ECONOME_PASSWORD}\n")
                            file.flush()
                    else:
                        with open(self.ECONOME_USER_INFO_FILE_PATH.name, "w") as file:
                            file.write(f"{self.ECONOME_PROJECT_NUMBER}\n")
                            file.write(f"{', '.join([item[0] for item in self.ECONOME_PROJECT_LIST])}\n")
                            file.write(f"{self.ECONOME_USER_NAME}\n")
                            file.write(f"{self.ECONOME_PASSWORD}\n")
                            file.flush()
                    # Get project status
                    self.get_project_status()

    def on_dialog_closed(self):
        """Prevent that main dialog is opened more than 1x time."""
        # Update the project variable with the selected layers
        project = QgsProject.instance()
        QgsExpressionContextUtils.setProjectVariable(project, "econome_selected_layers_dict", self.SELECTED_LAYERS_DICT)
        self.main_dialog.deleteLater()
        self.main_dialog = None
        self.first_start = True

    def run(self):
        """Run method that performs all the real work."""

        # ## PLUGIN LAUNCH
        # ------------------

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.main_dialog = EconoMeDialog() # Initialize main plugin class
            self.main_dialog.setWindowFlags(Qt.Window | Qt.WindowMinimizeButtonHint | Qt.WindowCloseButtonHint) # Support minimizing the dialog window
            self.main_dialog.finished.connect(self.on_dialog_closed) # Proper clean-up when dialog is closed
            self.disable_buttons_and_dropdowns(True) # Disable dialog elements until user is logged in

        self.main_dialog.show() # Show main dialog
        self.main_dialog.setMaximumWidth(600) # Set max width
        self.main_dialog.setMaximumHeight(800)  # Set max height

        # try to restore the project layer selection from the last session (stored in QGIS project variables)
        try:
            project = QgsProject.instance()
            self.SELECTED_LAYERS_DICT = dict(QgsExpressionContextUtils.projectScope(project).variable('econome_selected_layers_dict'))
            if self.SELECTED_LAYERS_DICT is None:
                self.SELECTED_LAYERS_DICT = {"DAMAGE_POTENTIAL": {}, "INTENSITY_MAPS": {}} 
        except:
            self.SELECTED_LAYERS_DICT = {"DAMAGE_POTENTIAL": {}, "INTENSITY_MAPS": {}}

        # Make pixmap for warning and success icons
        path_to_checkmark = os.path.join(self.plugin_dir, "general-icons", "checkmark.png")
        pixmap_checkmark = QPixmap(path_to_checkmark)
        self.scaled_pixmap_checkmark = pixmap_checkmark.scaled(
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
            Qt.KeepAspectRatio,
            Qt.SmoothTransformation)
        path_to_warning = os.path.join(self.plugin_dir, "general-icons", "warning.png")
        pixmap_warning = QPixmap(path_to_warning)
        self.scaled_pixmap_warning = pixmap_warning.scaled(
            self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
            Qt.KeepAspectRatio,
            Qt.SmoothTransformation)

        # Clear risk output temporary layer list
        self.RISK_OUTPUT_TEMP_LAYERS = []

        # ## ECONOME PROJECT LOGIN & SETTINGS MENU
        # ------------------------------------------

        # UI tweaks
        self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setCursor(QCursor(Qt.PointingHandCursor)) # Pointing cursor
        self.main_dialog.BUTTON_PLUGIN_SETTINGS.setCursor(QCursor(Qt.PointingHandCursor)) # Pointing cursor

        # Get EconoMe user info (if existing)
        existing_user_info = self.get_econome_user_info()
        if existing_user_info: # Set user infos
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentText(existing_user_info["ECONOME_PROJECT_NUMBER"])
            self.ECONOME_PROJECT_NUMBER = existing_user_info["ECONOME_PROJECT_NUMBER"]
            self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.setText(existing_user_info["ECONOME_USER_NAME"])
            self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.setText(existing_user_info["ECONOME_PASSWORD"])
            self.validate_econome_user_info()
        else: # Read/validate user input otherwise
            self.validate_econome_user_info()

        # CONNECTIONS

        try: # Disconnect (if existing connections/signals)
            self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.clicked.disconnect()
            self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.textChanged.disconnect()
            self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.textChanged.disconnect()
            self.main_dialog.BUTTON_PLUGIN_SETTINGS.clicked.disconnect()
        except TypeError:
            pass

        self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.clicked.connect(self.validate_econome_user_info) # CONNECT: Validate user info
        self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.textChanged.connect(self.onChange_econome_user_info) # CONNECT: Re-validate when user info has changed
        self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.textChanged.connect(self.onChange_econome_user_info) # CONNECT: Re-validate when user info has changed
        self.main_dialog.BUTTON_PLUGIN_SETTINGS.clicked.connect(self.change_settings) # CONNECT: Change settings

        # ## PROJECT STATUS & SELECTION
        # ------------------------------

        # UI modifications
        self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.setCursor(QCursor(Qt.PointingHandCursor))  # Pointing cursor
        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setMaximumWidth(525) # Set max width
        # self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(self.tr("Nicht verbunden"))  # Set "not connected" project status

        # Create auto-complete helper for project list
        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setEditable(True) # Make combo box editable
        project_list_completer = QCompleter(self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.model(), self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST) # Initialize QCompleter
        project_list_completer.setCompletionMode(QCompleter.PopupCompletion) # PopupCompletion
        project_list_completer.setFilterMode(Qt.MatchContains)  # Partial matching

        # Styling auto-complete popup
        new_list_view = QListView()
        new_list_view.setMouseTracking(True)
        new_list_view.setFocusPolicy(Qt.StrongFocus)
        new_list_view.setAttribute(Qt.WA_Hover, True)
        new_list_view.setStyleSheet("""
            QListView::item:hover {
                background-color: #d0f0ff;
            }
            QListView::item:selected {
                background-color: #3399ff;
                color: white;
            }
        """)
        project_list_completer.setPopup(new_list_view)
        project_list_completer.popup().setStyleSheet("")

        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCompleter(project_list_completer) # Set completer on combo box

        # CONNECTIONS

        try: # Disconnect (if existing connections/signals)
            self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.clicked.disconnect()
            project_list_completer.activated[str].disconnect()
            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.activated[str].disconnect()
        except TypeError:
            pass

        self.main_dialog.BUTTON_RELOAD_PROJECT_DETAILS.clicked.connect(self.get_project_status) # CONNECT: Reload project details
        project_list_completer.activated[str].connect(self.onChange_project_selection) # CONNECT: Update details on project selection change
        self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.activated[str].connect(self.onChange_project_selection) # CONNECT: Update details on project selection change

        # ## TEMPLATE LAYERS
        # --------------------

        # Create a list of damage potential labels
        self.DAMAGEPOTENTIAL_LABELS = [
            self.main_dialog.LABEL_DAM_POT_BUILDING,
            self.main_dialog.LABEL_DAM_POT_ROAD,
            self.main_dialog.LABEL_DAM_POT_RAILWAY,
            self.main_dialog.LABEL_DAM_POT_CABLE_TRANSPORT,
            self.main_dialog.LABEL_DAM_POT_LINE_PIPE,
            self.main_dialog.LABEL_DAM_POT_OTHER_ASSET
        ]

        self.DAMAGEPOTENTIAL_LABEL_FILTERS = [] # Store filters to prevent GC

        # UI tweaks
        for label in self.DAMAGEPOTENTIAL_LABELS:
            label.setCursor(QCursor(Qt.PointingHandCursor))
            label.setToolTip(self.tr("Mit Rechtsklick eine GIS-Layer-Vorlage herunterladen!"))
            filter = RightClickFilter(self.create_template_layer)
            label.installEventFilter(filter)
            self.DAMAGEPOTENTIAL_LABEL_FILTERS.append(filter)

        # ## MAP LAYER COMBO BOXES
        # -------------------------

        # Create list of all map layer combo boxes
        map_layer_combo_boxes = [
            self.main_dialog.LAYER_DAM_POT_BUILDING,
            self.main_dialog.LAYER_DAM_POT_ROAD,
            self.main_dialog.LAYER_DAM_POT_RAILWAY,
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT,
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE,
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET,
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE,
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30,
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100,
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300,
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME,
        ]

        # ## DAMAGE POTENTIAL
        # ---------------------

        try: # Disconnect (if existing connections/signals)
            self.main_dialog.LAYER_DAM_POT_BUILDING.layerChanged.disconnect()
            self.main_dialog.LAYER_DAM_POT_ROAD.layerChanged.disconnect()
            self.main_dialog.LAYER_DAM_POT_RAILWAY.layerChanged.disconnect()
            self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.layerChanged.disconnect()
            self.main_dialog.LAYER_DAM_POT_LINE_PIPE.layerChanged.disconnect()
            self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.layerChanged.disconnect()
            self.main_dialog.BUTTON_PP_BUILDING.clicked.disconnect()
            self.main_dialog.BUTTON_PP_ROAD.clicked.disconnect()
            self.main_dialog.BUTTON_PP_RAILWAY.clicked.disconnect()
            self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.clicked.disconnect()
            self.main_dialog.BUTTON_PP_LINE_PIPE.clicked.disconnect()
            self.main_dialog.BUTTON_PP_OTHER_ASSET.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.clicked.disconnect()
            self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.clicked.disconnect()
        except TypeError:
            pass

        # Establish layer change events on damage potential layer dropdowns
        self.main_dialog.LAYER_DAM_POT_BUILDING.layerChanged.connect(self.update_asset_layer_status)
        self.main_dialog.LAYER_DAM_POT_ROAD.layerChanged.connect(self.update_asset_layer_status)
        self.main_dialog.LAYER_DAM_POT_RAILWAY.layerChanged.connect(self.update_asset_layer_status)
        self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.layerChanged.connect(self.update_asset_layer_status)
        self.main_dialog.LAYER_DAM_POT_LINE_PIPE.layerChanged.connect(self.update_asset_layer_status)
        self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.layerChanged.connect(self.update_asset_layer_status)

        # Establish damage potential attribute profiles button connections & pointing cursor
        self.main_dialog.BUTTON_PP_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_BUILDING.clicked.connect(
            lambda: self.set_parameter_profiles("building", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING))
        self.main_dialog.BUTTON_PP_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_ROAD.clicked.connect(
            lambda: self.set_parameter_profiles("road", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD))
        self.main_dialog.BUTTON_PP_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_RAILWAY.clicked.connect(
            lambda: self.set_parameter_profiles("railway", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY))
        self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_CABLE_TRANSPORT.clicked.connect(
            lambda: self.set_parameter_profiles("cable_transport",
                                                self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT))
        self.main_dialog.BUTTON_PP_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_LINE_PIPE.clicked.connect(
            lambda: self.set_parameter_profiles("line_pipe", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE))
        self.main_dialog.BUTTON_PP_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_PP_OTHER_ASSET.clicked.connect(
            lambda: self.set_parameter_profiles("other_asset", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET))

        # Establish damage potential synchronization button connections & pointing cursor
        self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_BUILDING.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer(),
                                                      "building", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING))
        self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_ROAD.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer(), "road",
                                                      self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD))
        self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_RAILWAY.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer(), "railway",
                                                      self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY))
        self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_CABLE_TRANSPORT.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer(),
                                                      "cable_transport",
                                                      self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT))
        self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_LINE_PIPE.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer(),
                                                      "line_pipe",
                                                      self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE))
        self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_SYNC_OTHER_ASSET.clicked.connect(
            lambda: self.synchronize_damage_potential(self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer(),
                                                      "other_asset",
                                                      self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET))

        # Establish click connections on damage potential delete button connections & pointing cursor
        self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_BUILDING.clicked.connect(
            lambda: self.delete_damage_potential_assets("building",
                                                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING))
        self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_ROAD.clicked.connect(
            lambda: self.delete_damage_potential_assets("road", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_ROAD))
        self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_RAILWAY.clicked.connect(
            lambda: self.delete_damage_potential_assets("railway", self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_RAILWAY))
        self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_CABLE_TRANSPORT.clicked.connect(
            lambda: self.delete_damage_potential_assets("cable_transport",
                                                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_CABLE_TRANSPORT))
        self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_LINE_PIPE.clicked.connect(
            lambda: self.delete_damage_potential_assets("line_pipe",
                                                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_LINE_PIPE))
        self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DAM_POT_DEL_OTHER_ASSET.clicked.connect(
            lambda: self.delete_damage_potential_assets("other_asset",
                                                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_OTHER_ASSET))

        # Only allow vector layers and add an empty item as default to each map layer combo box
        self.main_dialog.LAYER_DAM_POT_BUILDING.setFilters(QgsMapLayerProxyModel.PolygonLayer | QgsMapLayerProxyModel.PointLayer)
        self.main_dialog.LAYER_DAM_POT_BUILDING.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_BUILDING.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_ROAD.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.main_dialog.LAYER_DAM_POT_ROAD.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_ROAD.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_RAILWAY.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.main_dialog.LAYER_DAM_POT_RAILWAY.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_RAILWAY.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_LINE_PIPE.setCurrentIndex(0)
        self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.setCurrentIndex(0)

        # set saved layer names for damage potential
        for damage_potential_type, saved_layer_name in self.SELECTED_LAYERS_DICT["DAMAGE_POTENTIAL"].items():
            combo_box_layer_widget = getattr(self.main_dialog, f"LAYER_DAM_POT_{damage_potential_type}")
            try:
                found_layers = QgsProject.instance().mapLayersByName(saved_layer_name)
                if len(found_layers) == 1:
                    combo_box_layer_widget.setLayer(found_layers[0])
                else:
                    issue = f"Could not set layer: None or multiple layers found with the name {saved_layer_name}"
                    # print(issue)
                    # TODO: error message popup to warn the user that he has multiple layers with the same name
            except:
                pass

        # ## SITUATION DROPDOWNS
        # ---------------------

        try: # Disconnect any existing dropdown events (to avoid errors when plugin is re-opened)
            self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentIndexChanged.disconnect()
            self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentIndexChanged.disconnect()
        except TypeError:
            pass

        # Establish dropdown events on situation dropdowns
        self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentIndexChanged.connect(self.update_situation_status)
        self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentIndexChanged.connect(self.update_situation_status)

        # ## INTENSITY LAYERS
        # ---------------------

        try: # Disconnect any existing layer change events (to avoid errors when plugin is re-opened)
            self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.layerChanged.disconnect()
            self.main_dialog.LAYER_INTENSITY_SCENARIO_30.layerChanged.disconnect()
            self.main_dialog.LAYER_INTENSITY_SCENARIO_100.layerChanged.disconnect()
            self.main_dialog.LAYER_INTENSITY_SCENARIO_300.layerChanged.disconnect()
            self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.layerChanged.disconnect()
        except TypeError:
            pass

        # Clear the intensity layer combo boxes
        self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_30.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_100.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_300.setCurrentIndex(0)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setAllowEmptyLayer(True)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.setCurrentIndex(0)

        # Establish layer change events on intensity layers dropdowns
        self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.layerChanged.connect(self.update_intensity_layer_status)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_30.layerChanged.connect(self.update_intensity_layer_status)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_100.layerChanged.connect(self.update_intensity_layer_status)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_300.layerChanged.connect(self.update_intensity_layer_status)
        self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.layerChanged.connect(self.update_intensity_layer_status)
        
        # Connect the intensity map layer column dropdowns to the update function
        self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.currentIndexChanged.connect(self.update_intensity_column_status)
        self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.currentIndexChanged.connect(self.update_intensity_column_status)
        self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.currentIndexChanged.connect(self.update_intensity_column_status)
        self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.currentIndexChanged.connect(self.update_intensity_column_status)
        self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.currentIndexChanged.connect(self.update_intensity_column_status)

        # See which scenarios are active and activate/deactivate the widgets accordingly and set saved layer names for intensity maps
        self.update_situation_status()

        # Establish connections on intensity layers delete buttons
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_FREE.clicked.connect(
            lambda: self.delete_damage_potential_affected_assets(
                self.main_dialog.LABEL_SCENARIO_FREE.text().strip()))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_30.clicked.connect(
            lambda: self.delete_damage_potential_affected_assets("30"))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_100.clicked.connect(
            lambda: self.delete_damage_potential_affected_assets("100"))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_300.clicked.connect(
            lambda: self.delete_damage_potential_affected_assets("300"))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_DEL_AFFECTED_ASSETS_EXTREME.clicked.connect(
            lambda: self.delete_damage_potential_affected_assets(
                self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()))

        # ## RISK CALCULATION AND VISUALIZATION
        # --------------------------------------

        try: # Disconnect any existing button connections (to avoid errors when plugin is re-opened)
            self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.clicked.disconnect()
            self.main_dialog.BUTTON_VISUALIZE_RISKS.clicked.disconnect()
        except TypeError:
            pass

        # Establish risk calculation and visualization button connections & pointing cursor
        self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_UPLOAD_CONSEQUENCE_ANALYSIS.clicked.connect(
            lambda: self.execute_in_separate_thread(self.upload_consequence_analysis))
        self.main_dialog.BUTTON_VISUALIZE_RISKS.setCursor(QCursor(Qt.PointingHandCursor))
        self.main_dialog.BUTTON_VISUALIZE_RISKS.clicked.connect(
            lambda: self.execute_in_separate_thread(self.visualize_risks))

        # ## QGIS EXIT
        # --------------

        try: # Disconnect any existing signals (to avoid errors when plugin is re-opened)
            QCoreApplication.instance().aboutToQuit.disconnect()
        except TypeError:
            pass

        QCoreApplication.instance().aboutToQuit.connect(lambda: self.clean_up(self.ECONOME_USER_INFO_FILE_PATH))

    def set_parameter_profiles(self, pp_type, status_label):
        """Handles parameter profile dialogs."""

        DAMAGE_POTENTIAL_LAYER = None
        dialog = None

        # Get glossary
        with requests.get(
                self.ECONOME_API_GLOSSARY_BASE_PATH,
                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
        ) as get_glossary:
            GLOSSARY = get_glossary.json()

        title = self.tr("EconoMe") + " - "
        if pp_type == "building":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['building'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['building'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Gebäude")

        elif pp_type == "road":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['road'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['road'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Strassen")

        elif pp_type == "railway":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['railway'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['railway'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Schiene")

        elif pp_type == "cable_transport":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['cable_transport'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['cable_transport'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Mech. Aufstiegshilfe")

        elif pp_type == "line_pipe":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['line_pipe'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['line_pipe'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Leitungen")

        elif pp_type == "other_asset":

            DAMAGE_POTENTIAL_LAYER = self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer()
            if DAMAGE_POTENTIAL_LAYER:
                dialog = ParameterProfileDialog(self.DAMAGEPOTENTIAL_FIELDS['other_asset'], self.DAMAGEPOTENTIAL_REQUIRED_FIELDS['other_asset'], self.ECONOME_PLUGIN_LANGUAGE, GLOSSARY)
                title += self.tr("Attributspalten: Andere Objekte")
        
        if dialog:

            dialog.setWindowTitle(title)
            labels = dialog.findChildren(QLabel)

            # find all fields in the layer
            fields = DAMAGE_POTENTIAL_LAYER.fields()
            integer_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["integer"]]
            number_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["float"]]
            string_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["string"]]
            bool_fields = [field for field in fields if field.type() in self.QVARIANT_TYPES_DICT["boolean"]]

            # Retrieve parameter profile
            with requests.get(
                    self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
            ) as get_parameter_profile:
                PARAMETER_PROFILE = get_parameter_profile.json()
                PARAMETER_PROFILE['Empty'] = False

            # if there is no parameter profile, create an empty one to have the type information
            if 'Code' in PARAMETER_PROFILE:
                empty_parameter_profile = {
                    "ParameterList": [{"API": label.objectName().split('_')[1], "User": ""} for label in labels]}
                with requests.post(
                        self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                        json=empty_parameter_profile,
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as post_empty_parameter_profile:
                    PARAMETER_PROFILE = post_empty_parameter_profile.json()
                    PARAMETER_PROFILE['Empty'] = True

            # Fill damage potential layer field names in combo boxes and select a parameter if set in the parameter profile
            for label in labels:
                attribute = label.objectName().split('_')[1]
                matching_parameter = [item for item in PARAMETER_PROFILE['ParameterList'] if item["API"] == attribute][
                    0]
                parameter_type = matching_parameter['Type']
                label.setText(f"{label.text()} [{parameter_type}]")
                allowed_fields = integer_fields if parameter_type == "integer" else number_fields if parameter_type == "float" else string_fields if parameter_type == "string" else bool_fields if parameter_type == "boolean" else []
                matching_comboBox = dialog.findChild(QComboBox, f"COMBO_BOX_{attribute}")
                matching_comboBox.addItem("")
                matching_comboBox.addItems([field.name() for field in allowed_fields])
                if not PARAMETER_PROFILE['Empty']:
                    if f"{matching_parameter['User'].strip()}" in [field.name() for field in fields]:
                        index = matching_comboBox.findText(f"{matching_parameter['User'].strip()}")
                        matching_comboBox.setCurrentIndex(index)

            result = dialog.exec_()
            if result:
                # CREATE: Updated parameter profile dictionary
                updated_parameter_profile = {"ParameterList": []}
                # Loop over all user input combo boxes
                for comboBox in dialog.findChildren(QComboBox):
                    if not PARAMETER_PROFILE['Empty']:
                        existing_parameter_pair = [item for item in PARAMETER_PROFILE['ParameterList'] if
                                                   item["API"] == comboBox.objectName().split("_")[-1]][0]
                        if existing_parameter_pair["User"] != comboBox.currentText():
                            updated_parameter_profile["ParameterList"].append(
                                {"API": existing_parameter_pair["API"], "User": comboBox.currentText()})
                    else:
                        parameter_name = comboBox.objectName().split("_")[-1]
                        updated_parameter_profile["ParameterList"].append(
                            {"API": parameter_name, "User": comboBox.currentText()})

                if len(updated_parameter_profile["ParameterList"]) != 0:
                    # create the pixmap
                    path_to_checkmark = os.path.join(self.plugin_dir, "general-icons", "checkmark.png")
                    pixmap_checkmark = QPixmap(path_to_checkmark)
                    self.scaled_pixmap_checkmark = pixmap_checkmark.scaled(
                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
                        Qt.KeepAspectRatio,
                        Qt.SmoothTransformation)
                    path_to_warning = os.path.join(self.plugin_dir, "general-icons", "warning.png")
                    pixmap_warning = QPixmap(path_to_warning)
                    self.scaled_pixmap_warning = pixmap_warning.scaled(
                        self.main_dialog.IMAGE_LABEL_DAM_POT_STATUS_BUILDING.size(),
                        Qt.KeepAspectRatio,
                        Qt.SmoothTransformation)

                    if not PARAMETER_PROFILE['Empty']:
                        with requests.put(
                                self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                                json=updated_parameter_profile,
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as put_new_parameter_profile:
                            # USER INFO
                            if put_new_parameter_profile.status_code == 200:
                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg"), message_text=self.tr("\nAttributspalten für {category} erfolgreich aktualisiert!\n").format(category=dialog.windowTitle().split(': ')[-1])).exec_()
                                self.DAMAGE_POTENTIAL_LAYER_STATUS[pp_type.upper()] = 0
                                status_label.clear()
                            else:
                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=f"\n{put_new_parameter_profile.json()}\n").exec_()
                    else:
                        with requests.put(
                                self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{pp_type}",
                                json=updated_parameter_profile,
                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                        ) as put_new_parameter_profile:
                            # USER INFO
                            if put_new_parameter_profile.status_code == 200:
                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg"), message_text=self.tr("\nAttributspalten für {category} erfolgreich angelegt!\n").format(category=dialog.windowTitle().split(': ')[-1])).exec_()
                            else:
                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=f"\n{put_new_parameter_profile.json()}\n").exec_()

    def show_save_or_load_dialog(self):
        """Shows save & load dialog."""
        SaveLoadDialog(self.RISK_OUTPUT_TEMP_LAYERS).exec_()

    def status_in_separate_thread(self, title, text, pixmap_type):
        """Shows status dialog inside worker thread."""

        if pixmap_type == 'success':
            StatusDialog(status_icon=self.scaled_pixmap_checkmark, message_title=title, message_text=text).exec_()
        else:
            StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=title, message_text=text).exec_()

    def switch_language(self, locale_code):
        # Uninstall previous translator if any
        if self.translator:
            QCoreApplication.removeTranslator(self.translator)

        # Attempt to load the new translation
        plugin_dir = os.path.dirname(__file__)
        qm_path = os.path.join(plugin_dir, 'i18n', f"{locale_code.lower()}.qm")

        if os.path.exists(qm_path):
            self.translator = QTranslator()
            self.translator.load(qm_path)
            QCoreApplication.installTranslator(self.translator)

    def synchronize_damage_potential(self, damage_potential_layer, damage_potential_type, status_label):
        """This method synchronizes the damage potential with the EconoMe API."""

        # Check for valid user info
        if self.USER_INFO_VALIDATED:
            # Get damage potential layer
            DAMAGE_POTENTIAL_LAYER = damage_potential_layer
            # Check if damage potential layer is valid
            if DAMAGE_POTENTIAL_LAYER:
                # Damage potential type
                DAMAGE_POTENTIAL_TYPE = damage_potential_type
                DAMAGE_POTENTIAL_LABEL_TEXT = getattr(self.main_dialog, f"LABEL_DAM_POT_{DAMAGE_POTENTIAL_TYPE.upper()}").text()
                # Layer Asset list
                LAYER_ASSETS = {'AssetList': []}
                # Get current damage potential
                with requests.get(
                        self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as get_damage_potential:
                    UPLOADED_DAMAGE_POTENTIAL = get_damage_potential.json()
                # Filter damage potential by object category
                OBJECT_CATEGORIES = [item for sublist in [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for obj_cat in self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[DAMAGE_POTENTIAL_TYPE.upper()]] for item in sublist]
                FILTERED_UPLOADED_DAMAGE_POTENTIAL = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetTypeID"] in OBJECT_CATEGORIES]
                # Retrieve parameter profile and get parameter associations
                with (requests.get(
                        self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{DAMAGE_POTENTIAL_TYPE}",
                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                ) as get_parameter_profile):
                    PARAMETER_PROFILE = get_parameter_profile.json()
                    # Check for parameter profile
                    if get_parameter_profile.status_code == 200:
                        # Get valid parameter pairs
                        damage_potential_valid_parameter_pairs = [pp for pp in PARAMETER_PROFILE['ParameterList'] if pp["User"].strip()]
                        damage_potential_valid_parameter_keys = [pp["API"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["User"].strip()]
                        # Check if required fields were filled out
                        if set(self.DAMAGEPOTENTIAL_REQUIRED_FIELDS[damage_potential_type]).issubset(damage_potential_valid_parameter_keys):
                            AssetIdentID_user_name = [pp["User"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["API"].strip() == "AssetIdentID"][0]
                            AssetTypeID_user_name = [pp["User"] for pp in PARAMETER_PROFILE['ParameterList'] if pp["API"].strip() == "AssetTypeID"][0]
                            layer_fields_names = [field.name() for field in DAMAGE_POTENTIAL_LAYER.fields()]
                            user_field_names = [pp["User"] for pp in damage_potential_valid_parameter_pairs]
                            # Check if the pp defined fields are present in the damage potential layer -> without this check, the user will not notice that the layer is not synchronizing correctly
                            if set(user_field_names).issubset(set(layer_fields_names)):
                                typechecked_user_field_names = [pp["User"] for pp in damage_potential_valid_parameter_pairs if DAMAGE_POTENTIAL_LAYER.fields().field(DAMAGE_POTENTIAL_LAYER.fields().indexFromName(pp["User"])).type() in self.QVARIANT_TYPES_DICT[pp["Type"]]]
                                # Check field types
                                if set(user_field_names).issubset(typechecked_user_field_names):
                                    connected_columns_title = self.tr("Verknüpfte Spalten: {num_connected_cols}: ").format(num_connected_cols=len(typechecked_user_field_names))
                                    connected_columns_list = ', '.join(typechecked_user_field_names)
                                    connected_columns_string =  f"{connected_columns_title}{connected_columns_list}"
                                    # Check that all features have an AssetIdentID
                                    AssetIdentID_list = [feature[AssetIdentID_user_name] for feature in DAMAGE_POTENTIAL_LAYER.getFeatures()]
                                    if all(AssetIdentID_list):
                                        # Create asset list from layer
                                        for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                            new_asset = {}
                                            for parameter_pair in damage_potential_valid_parameter_pairs:
                                                if parameter_pair["Type"] == "string":
                                                    if feature[parameter_pair["User"]] == NULL:
                                                        new_asset[parameter_pair["API"]] = None
                                                    elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                        new_asset[parameter_pair["API"]] = str(feature[parameter_pair["User"]].value())
                                                    else:
                                                        new_asset[parameter_pair["API"]] = str(feature[parameter_pair["User"]])
                                                elif parameter_pair["Type"] == "integer":
                                                    if feature[parameter_pair["User"]] == NULL:
                                                        new_asset[parameter_pair["API"]] = None
                                                    elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                        new_asset[parameter_pair["API"]] = int(feature[parameter_pair["User"]].value())
                                                    else:
                                                        new_asset[parameter_pair["API"]] = int(feature[parameter_pair["User"]])
                                                elif parameter_pair["Type"] == "float":
                                                    if feature[parameter_pair["User"]] == NULL:
                                                        new_asset[parameter_pair["API"]] = None
                                                    elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                        new_asset[parameter_pair["API"]] = float(feature[parameter_pair["User"]].value())
                                                    else:
                                                        new_asset[parameter_pair["API"]] = float(feature[parameter_pair["User"]])
                                                elif parameter_pair["Type"] == "boolean":
                                                    if feature[parameter_pair["User"]] == NULL:
                                                        new_asset[parameter_pair["API"]] = None
                                                    elif isinstance(feature[parameter_pair["User"]], QVariant):
                                                        new_asset[parameter_pair["API"]] = bool(feature[parameter_pair["User"]].value())
                                                    else:
                                                        new_asset[parameter_pair["API"]] = bool(feature[parameter_pair["User"]])
                                            LAYER_ASSETS["AssetList"].append(new_asset)
                                        # Create list of asset ident ids
                                        layer_asset_ident_ids = [asset["AssetIdentID"] for asset in LAYER_ASSETS["AssetList"]]
                                        set_layer_asset_ident_ids = set(layer_asset_ident_ids)
                                        if len(layer_asset_ident_ids) == len(set_layer_asset_ident_ids):
                                            # Compare layer assets and uploaded damage potential
                                            comparison_result, mismatch_dict = self.compare_damage_potential(LAYER_ASSETS, FILTERED_UPLOADED_DAMAGE_POTENTIAL, damage_potential_type, damage_potential_valid_parameter_pairs)
                                            if comparison_result == "CANCEL":
                                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {category}").format(category=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nKeine Option ausgewählt.\n")).exec_()
                                                status_label.setPixmap(self.scaled_pixmap_warning)
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            elif comparison_result == "INVALID OBJECT CATEGORY":
                                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {category}").format(category=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nEine oder mehrere Objektart-ID im Layer passt nicht zur Kategorie des Schadenpotentials!\n")).exec_()
                                                status_label.setPixmap(self.scaled_pixmap_warning)
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            elif comparison_result == "POST LAYER ASSETS": # when no econome assets are present
                                                # Post new damage potential from layer
                                                with requests.post(
                                                        self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                        json=LAYER_ASSETS,
                                                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                ) as post_damage_potential:
                                                    if post_damage_potential.status_code == 200:
                                                        created_assets = json.loads(post_damage_potential.text)["AssetList"]
                                                        text = connected_columns_string
                                                        text += self.tr("\n{new_assets_count} neue Schadenobjekte wurden erfolgreich vom Layer auf EconoMe hochgeladen:\n").format(new_assets_count=len(created_assets))
                                                        for posted_asset_dict in created_assets:
                                                            text += f"{posted_asset_dict['AssetIdentID']}\n"
                                                        StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                        status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                                    else:
                                                        StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nAPI-Fehlermeldung beim Hochladen des Schadenpotentials:\n{response}\n").format(response=json.loads(post_damage_potential.text))).exec_()
                                                        status_label.setPixmap(self.scaled_pixmap_warning)
                                                        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            # Layer assets match completely with assets in EconoMe
                                            elif comparison_result == "COMPLETE MATCH": # when all attributes match
                                                status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                                text = connected_columns_string
                                                text += self.tr("\nLayer-Schadenobjekte stimmen mit EconoMe-Schadenobjekten überein.\n")
                                                StatusDialog(status_icon=self.scaled_pixmap_checkmark,message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT),message_text=text).exec_()
                                            # Update EconoMe assets
                                            elif comparison_result == "KEEP LAYER ASSETS": # when user decides to overwrite econome assets with layer assets
                                                """
                                                ATTENTION: 
                                                Only valid parameter pairs should be overwritten as the user might have changed attributes in EconoMe that are not present in the layer
                                                Therefore we put the layer attributes (instead of deleting and re-posting) into the EconoMe API.
                                                New layer assets are posted to EconoMe.
                                                """
                                                # 1) collect assets to patch/put/post in lists of dicts with correct API keys
                                                # 1.1) Assets where the type was changed in layer: we need to patch and then post them, thereby we need to preserve all EconoMe-sided stored attributes
                                                layer_assets_to_change_type = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and "AssetTypeID" in mismatch_dict[layer_asset["AssetIdentID"]].keys() ]
                                                uploaded_assets_to_change_type = [asset for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and "AssetTypeID" in mismatch_dict[asset["AssetIdentID"]].keys()]
                                                patch_asset_ids = [asset["AssetID"] for asset in uploaded_assets_to_change_type]
                                                assets_to_change_type = []
                                                for uploaded_asset in copy.deepcopy(uploaded_assets_to_change_type):
                                                    asset_to_change_type = uploaded_asset
                                                    for key, value in mismatch_dict[uploaded_asset["AssetIdentID"]].items():
                                                        asset_to_change_type[key] = value["Layer"]
                                                    asset_to_change_type.pop("AssetID") # not supported by API post
                                                    assets_to_change_type.append(asset_to_change_type)
                                                # 1.2) Assets that exist in EconoMe: we need to update/put them
                                                layer_assets_to_put = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[layer_asset["AssetIdentID"]] != {"Missing":self.tr("Fehlt in EconoMe.")} and "AssetTypeID" not in mismatch_dict[layer_asset["AssetIdentID"]].keys()]
                                                assets_to_put = []
                                                assets_to_put_status_code = 0
                                                for asset_to_put in copy.deepcopy(layer_assets_to_put):
                                                    asset_to_put.pop("AssetTypeID") # not supported by API put (see above assets_to_change_type)
                                                    asset_to_put.update({"AssetID": [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == asset_to_put["AssetIdentID"]][0]})
                                                    asset_to_put.pop("AssetIdentID") # not supported by API put
                                                    assets_to_put.append(asset_to_put)
                                                # 1.3) Assets that were newly created in the layer: we need to post them
                                                assets_to_post = [layer_asset for layer_asset in LAYER_ASSETS["AssetList"] if layer_asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[layer_asset["AssetIdentID"]] == {"Missing":self.tr("Fehlt in EconoMe.")}]
                                                assets_to_post_status_code = 0
                                                # 2) patch/put/post the assets
                                                # 2.1) Patch assets where the type was changed in layer
                                                if len(assets_to_change_type) != 0:
                                                    with requests.patch(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                                            json={"AssetIDList":patch_asset_ids},
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as patch_damage_potential:
                                                        with requests.post(
                                                                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                                json={"AssetList":assets_to_change_type},
                                                                auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                        ) as reput_damage_potential:
                                                            # Feedback to user about the syc errors or success
                                                            if patch_damage_potential.status_code == 200 and reput_damage_potential.status_code == 200:
                                                                text = connected_columns_string
                                                                text += self.tr("Der Objekttyp folgender {category} Schadenobjekte wurde auf EconoMe mit Objekttyp aus Layer überschrieben.\n").format(category=len(assets_to_change_type))
                                                                text += self.tr("Dafür musste jedes Schadenobjekt auf EconoMe gelöscht werden und neu erstellt werden. Alle Attribute wurden erhalten.\n")
                                                                for reput_asset_dict in json.loads(reput_damage_potential.text)["AssetList"]:
                                                                    text += f"\n{reput_asset_dict['AssetIdentID']}"
                                                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                                status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                                            else:
                                                                text = ""
                                                                if patch_damage_potential.status_code != 200:
                                                                    text = self.tr("\nAPI-Fehlermeldung beim Updaten des Objekttyps (Schritt 1 patch):\n{response}\n").format(response=json.loads(patch_damage_potential.text))
                                                                elif reput_damage_potential.status_code != 200:
                                                                    text += self.tr("\nAPI-Fehlermeldung beim Updaten des Objekttyps (Schritt 2 post):\n{response}\n").format(response=json.loads(reput_damage_potential.text))
                                                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                                status_label.setPixmap(self.scaled_pixmap_warning)
                                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                                # 2.2) Put updated damage potential from layer
                                                if len(assets_to_put) != 0:
                                                    with requests.put(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                            json={"AssetList": assets_to_put},
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as put_damage_potential:
                                                        assets_to_put_status_code = put_damage_potential.status_code
                                                        # Feedback to user about the syc errors or success
                                                        if put_damage_potential.status_code == 200:
                                                            text = connected_columns_string
                                                            text += self.tr("\nDie Attribute folgender {new_assets_length} Schadenobjekte wurden auf EconoMe mit Attributen aus dem Layer überschrieben:\n").format(new_assets_length=len(assets_to_put))
                                                            #for put_asset_dict in json.loads(put_damage_potential.text)["AssetList"]: # TODO report bug an Peter: API does not return all updated assets
                                                            for put_asset_dict in layer_assets_to_put:
                                                                text += f"\n{put_asset_dict['AssetIdentID']}"
                                                            StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                            status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                                        else:
                                                            StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nAPI-Fehlermeldung beim Updaten des vorhandenen Schadenpotentials (put):\n{response}\n").format(response=json.loads(put_damage_potential.text))).exec_()
                                                            status_label.setPixmap(self.scaled_pixmap_warning)
                                                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                                # 2.3) Post new damage potential from layer
                                                if len(assets_to_post) != 0:
                                                    with requests.post(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                                            json={"AssetList":assets_to_post},
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as post_damage_potential:
                                                        assets_to_post_status_code = post_damage_potential.status_code
                                                        # Feedback to user about the syc errors or success
                                                        if post_damage_potential.status_code == 200:
                                                            # for newly created assets, we want to save the standard-created attributes from EconoMe to the layer
                                                            DAMAGE_POTENTIAL_LAYER.startEditing()
                                                            for posted_asset_dict in json.loads(post_damage_potential.text)["AssetList"]:
                                                                expression = f'"{AssetIdentID_user_name}" = \'{posted_asset_dict["AssetIdentID"]}\''
                                                                request = QgsFeatureRequest().setFilterExpression(expression)
                                                                feature = next(DAMAGE_POTENTIAL_LAYER.getFeatures(request), None)
                                                                for parameter_pair in damage_potential_valid_parameter_pairs:
                                                                    feature.setAttribute(parameter_pair["User"], posted_asset_dict[parameter_pair["API"]])
                                                                    DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                            DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                            # user feedback
                                                            text = connected_columns_string
                                                            text += self.tr("{new_assets_length} neue Schadenobjekte wurden vom Layer auf EconoMe hochgeladen:\n").format(new_assets_length=len(assets_to_post))
                                                            for posted_asset_dict in json.loads(post_damage_potential.text)["AssetList"]:
                                                                text += f"\n{posted_asset_dict['AssetIdentID']}"
                                                            StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                            status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                                        else:
                                                            StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nAPI-Fehlermeldung beim Hochladen des neuen Schadenpotentials (post):\n{response}\n").format(response=json.loads(post_damage_potential.text))).exec_()
                                                            status_label.setPixmap(self.scaled_pixmap_warning)
                                                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                                # Checkmark should only be changed if both requests were successful
                                                if len(assets_to_put) != 0 and len(assets_to_post) != 0:
                                                    if assets_to_post_status_code != 200 or assets_to_put_status_code != 200:
                                                        status_label.setPixmap(self.scaled_pixmap_warning)
                                                        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            # Update Layer assets
                                            elif comparison_result == "KEEP ECONOME ASSETS":
                                                # Change layer assets based on EconoMe
                                                DAMAGE_POTENTIAL_LAYER.startEditing()
                                                for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                                    AssetIdentID = feature[AssetIdentID_user_name]
                                                    if AssetIdentID in mismatch_dict.keys():
                                                        updated_assetIdentIDs = []
                                                        deleted_assetIdentIDs = []
                                                        corresponding_uploaded_asset = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[AssetIdentID_user_name])]
                                                        if len(corresponding_uploaded_asset) != 0:
                                                            for parameter_pair in damage_potential_valid_parameter_pairs:
                                                                feature.setAttribute(parameter_pair["User"], corresponding_uploaded_asset[0][parameter_pair["API"]])
                                                                DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                            updated_assetIdentIDs.append(AssetIdentID)
                                                        else:
                                                            DAMAGE_POTENTIAL_LAYER.dataProvider().deleteFeatures([feature.id()])
                                                            deleted_assetIdentIDs.append(AssetIdentID)
                                                DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                text = connected_columns_string
                                                if len(updated_assetIdentIDs) != 0:
                                                    text += self.tr("\nDie Attribute folgender Schadenobjekte wurden im Layer mit Attributen aus EconoMe überschrieben:\n")
                                                    for AssetIdentID in updated_assetIdentIDs:
                                                        text += f"\n{AssetIdentID}"
                                                if len(deleted_assetIdentIDs) != 0:
                                                    text += self.tr("\n\nFolgende {len_objects} Schadenobjekte wurden im Layer gelöscht:\n").format(len_objects=len(deleted_assetIdentIDs))
                                                    for AssetIdentID in deleted_assetIdentIDs:
                                                        text += f"\n{AssetIdentID}"
                                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                status_label.setPixmap(self.scaled_pixmap_checkmark)
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 1
                                            elif comparison_result == "KEEP ECONOME ASSET TYPE IDS":
                                                # Change layer assets based on EconoMe
                                                DAMAGE_POTENTIAL_LAYER.startEditing()
                                                for feature in DAMAGE_POTENTIAL_LAYER.getFeatures():
                                                    corresponding_uploaded_asset = [asset for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[AssetIdentID_user_name])]
                                                    if len(corresponding_uploaded_asset) != 0:
                                                        feature.setAttribute(AssetTypeID_user_name, corresponding_uploaded_asset[0]["AssetTypeID"])
                                                        DAMAGE_POTENTIAL_LAYER.updateFeature(feature)
                                                DAMAGE_POTENTIAL_LAYER.commitChanges()
                                                StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Layer-Objektart-IDs {DAMAGE_POTENTIAL_LABEL_TEXT} zurückgesetzt!").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("Bitte erneut synchronisieren!")).exec_()
                                                status_label.clear()
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            elif comparison_result == "DELETE MISSING ASSETS":
                                                # Delete category-filtered damage potential assets from EconoMe
                                                patch_asset_ids = [asset["AssetID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[asset["AssetIdentID"]] == {"Missing":"Fehlt im Layer."}]
                                                patch_assetIdentIDs = [asset["AssetIdentID"] for asset in FILTERED_UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] in mismatch_dict.keys() and mismatch_dict[asset["AssetIdentID"]] == {"Missing":"Fehlt im Layer."}]
                                                with requests.patch(
                                                        self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/delete',
                                                        json={"AssetIDList": patch_asset_ids},
                                                        auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                ) as patch_damage_potential:
                                                    if patch_damage_potential.status_code == 200:
                                                        text = connected_columns_string
                                                        text += self.tr("\nFolgende {len_objects} Schadenobjekte wurden erfolgreich von EconoMe gelöscht:\n").format(len_objects=len(patch_asset_ids))
                                                        for deleted_assetIdentID in patch_assetIdentIDs:
                                                            text += f"\n{deleted_assetIdentID}"
                                                        text += self.tr("\n\nEs wird noch nach Attribut-Konflikten gesucht...")
                                                        StatusDialog(status_icon = self.scaled_pixmap_checkmark, message_title=self.tr("Erfolg: Schadenpotenzial {DAMAGE_POTENTIAL_LABEL_TEXT} synchronisiert").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                                        self.synchronize_damage_potential(DAMAGE_POTENTIAL_LAYER, DAMAGE_POTENTIAL_TYPE, status_label)
                                            elif comparison_result == "NO CHANGE":
                                                # USER INFO: No changes made
                                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Hinweis"), message_text=self.tr("\nBitte fehlende Schadenobjekte im Layer hinzufügen!\n")).exec_()
                                                status_label.setPixmap(self.scaled_pixmap_warning)
                                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                            else:
                                                # USER INFO: internal issue unknown option
                                                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Interner Fehler"), message_text=self.tr("\nUnbekannte Option ausgewählt.\n")).exec_()
                                        else:
                                            # USER INFO: Some of the layer features have the same AssetIdentID
                                            duplicate_AssetIdentIDs = [item for item, count in Counter(layer_asset_ident_ids).items() if count > 1]
                                            text = self.tr("\nFolgende AssetIdentIDs kommen im Layer mehrmals vor:\n")
                                            for duplicate_AssetIdentID in duplicate_AssetIdentIDs:
                                                text += f"\n{duplicate_AssetIdentID}"
                                            text += self.tr("\n\nBitte nur eindeutige AssetIdentIDs verwenden!\n")
                                            StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT),message_text=text).exec_()
                                            status_label.setPixmap(self.scaled_pixmap_warning)
                                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                    else:
                                        # USER INFO: Some of the layer features have no AssetIdentID / Null values for AssetIdentID
                                        assets_missing_AssetIdentID = [feature for feature in DAMAGE_POTENTIAL_LAYER.getFeatures() if not feature[AssetIdentID_user_name]]
                                        text = self.tr("{len_objects} Schadenobjekte im Layer haben keine AssetIdentID! \nBitte AssetIdentID hinzufügen!\n").format(len_objects=len(assets_missing_AssetIdentID))
                                        StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT),message_text=text).exec_()
                                        status_label.setPixmap(self.scaled_pixmap_warning)
                                        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                                else:
                                    # USER INFO: Some of the pp defined fields have a wrong data type
                                    wrong_datatype_fields = set(user_field_names) - set(typechecked_user_field_names)
                                    text = self.tr("Folgende {len_wrong_datatype_fields} Layer-Spalten haben einen falschen Datentyp:\n").format(len_wrong_datatype_fields=len(wrong_datatype_fields))
                                    for field_name in wrong_datatype_fields:
                                        text += f"\n{field_name}"
                                    text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                                    StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                    status_label.setPixmap(self.scaled_pixmap_warning)
                                    self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                            else:
                                # USER INFO: Some of the pp defined fields are not present in the damage potential layer
                                missing_fields = set(user_field_names) - set(layer_fields_names)
                                text = self.tr("Folgende Layer-Spalten wurden im Parameter-Profil einem EconoMe-Parameter zugewiesen, aber konnten im Layer nicht gefunden werden:\n").format(missing_fields_length=len(missing_fields))
                                for missing_field in missing_fields:
                                    text += f"\n{missing_field}"
                                text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                                StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                                status_label.setPixmap(self.scaled_pixmap_warning)
                                self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                        else:
                            # USER INFO: Not all required fields are defined in the parameter profile
                            missing_required_fields = set(self.DAMAGEPOTENTIAL_REQUIRED_FIELDS[damage_potential_type]) - set(damage_potential_valid_parameter_keys)
                            text = self.tr("Folgenden {fields_length} Pflicht-Attributen wurden im Parameter-Profil keine Spalte des Layers zugewiesen:\n").format(fields_length=len(missing_required_fields))
                            for required_field in missing_required_fields:
                                text += f"\n{required_field}"
                            text += self.tr("\n\nBitte Parameter-Profil erneuern.\n")
                            StatusDialog(status_icon=self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=text).exec_()
                            status_label.setPixmap(self.scaled_pixmap_warning)
                            self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
                    else:
                        # USER INFO: No parameter profile found
                        StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler beim synchronisieren des Schadenpotenzials {DAMAGE_POTENTIAL_LABEL_TEXT}").format(DAMAGE_POTENTIAL_LABEL_TEXT=DAMAGE_POTENTIAL_LABEL_TEXT), message_text=self.tr("\nBitte Parameter-Profil definieren: Jedem Attribut eine Attributspalte des Layer zuweisen!\n")).exec_()
                        status_label.setPixmap(self.scaled_pixmap_warning)
                        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type.upper()] = 0
            else:
                # USER INFO: Invalid damage potential layer
                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=self.tr("\nBitte Schadenpotential-Layer auswählen!\n")).exec_()
        else:
            # USER INFO: Invalid login info
            StatusDialog(status_icon = self.scaled_pixmap_warning, message_title=self.tr("Fehler"), message_text=self.tr("\nUngültige Anmeldedaten!\n")).exec_()

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('EconoMe', message)

    def update_asset_layer_status(self):
        # Get sender info
        sender_widget = self.main_dialog.sender()
        damage_potential_type = sender_widget.objectName().split("DAM_POT_")[-1]

        # clear image label
        image_label_widget = getattr(self.main_dialog, f"IMAGE_LABEL_DAM_POT_STATUS_{damage_potential_type}")
        image_label_widget.clear()
        self.DAMAGE_POTENTIAL_LAYER_STATUS[damage_potential_type] = 0

        # save selected layer name
        current_layer_name = self.main_dialog.sender().currentText()
        self.SELECTED_LAYERS_DICT['DAMAGE_POTENTIAL'].update({damage_potential_type: current_layer_name})

    def update_intensity_layer_status(self):
        # Get sender info
        sender_widget = self.main_dialog.sender()
        annuality = sender_widget.objectName().split("INTENSITY_SCENARIO_")[-1]

        # fill dropdown with layer fields
        current_layer = self.main_dialog.sender().currentLayer()
        combo_box_name = f"COMBO_BOX_INTENSITY_COLUMN_NAME_{annuality}"
        combo_box_widget = getattr(self.main_dialog, combo_box_name)
        if current_layer:
            combo_box_widget.clear()
            combo_box_widget.addItem("")
            # Add field names to dropdown
            for field in current_layer.fields():
                combo_box_widget.addItem(field.name())
            combo_box_widget.setCurrentIndex(0)
        else:
            combo_box_widget.clear()
            self.INTENSITY_LAYER_COLUMN_STATUS[annuality] = None

        # save selected layer name
        current_layer_name = self.main_dialog.sender().currentText()
        current_process_id = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
        current_variant_id = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]
        if current_variant_id != "" and current_process_id != "":
            key = f"{current_process_id}_{current_variant_id}_{annuality}"
            self.SELECTED_LAYERS_DICT['INTENSITY_MAPS'].update({key: current_layer_name})

    def update_intensity_column_status(self):
        sender_widget = self.main_dialog.sender()
        intensity_column = sender_widget.currentText()
        annuality = sender_widget.objectName().split("COMBO_BOX_INTENSITY_COLUMN_NAME_")[-1]
        image_label_widget = getattr(self.main_dialog, f"IMAGE_LABEL_INT_MAP_STATUS_{annuality}")
        if intensity_column != "":
            intensity_layer = getattr(self.main_dialog, f"LAYER_INTENSITY_SCENARIO_{annuality}").currentLayer()
            intensity_field = intensity_layer.fields().field(intensity_column)
            if intensity_field.type() in self.QVARIANT_TYPES_DICT["integer"] + self.QVARIANT_TYPES_DICT["float"]:
                intensity_values = [f[intensity_column] for f in intensity_layer.getFeatures()]
                intensity_unique_values = list(set(intensity_values))
                if all(value in [0, 1, 2, 3] for value in intensity_unique_values):
                    image_label_widget.setPixmap(self.scaled_pixmap_checkmark)
                    self.INTENSITY_LAYER_COLUMN_STATUS[annuality] = 1
                else:
                    self.INTENSITY_LAYER_COLUMN_STATUS[annuality] = 0
                    image_label_widget.setPixmap(self.scaled_pixmap_warning)
                    StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler"), message_text = self.tr(f"\nUngültige Identitäts-Spalte.\nIntensitäts-Klassen müssen der BAFU-Skala entsprechen: [(0), 1, 2, 3].\nDie ausgewählte Spalte {intensity_column} hat den Wertebereich {intensity_unique_values}.\n")).exec_()
            else:
                self.INTENSITY_LAYER_COLUMN_STATUS[annuality] = 0
                image_label_widget.setPixmap(self.scaled_pixmap_warning)
                StatusDialog(status_icon = self.scaled_pixmap_warning, message_title = self.tr("Fehler"), message_text = self.tr(f"\nUngültige Identitäts-Spalte.\nIntensitäts-Spalte muss eine Zahl sein!\nDie ausgewählte Spalte {intensity_column} hat den Typ {intensity_field.typeName()}.\n")).exec_()
        else:
            self.INTENSITY_LAYER_COLUMN_STATUS[annuality] = -1
            image_label_widget.clear()
    
    def update_situation_status(self):

        # CHECK: User info validated
        if self.USER_INFO_VALIDATED:

            # get situation info
            current_process_id = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
            current_variant_id = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]

            # get process info
            get_process = requests.get(
                self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/process" + f"/{current_process_id}",
                auth=HTTPBasicAuth(str(self.ECONOME_USER_NAME), str(self.ECONOME_PASSWORD)))
            if len(json.loads(get_process.text)) != 0:
                SCENARIOS = {key: value["Active"] for key, value in json.loads(get_process.text).items() if
                             key.startswith('Annuality') and value is not None}
                for annuality_key, active in SCENARIOS.items():
                    annuality_str = annuality_key.strip("Annuality")
                    annuality = annuality_str.upper() if annuality_str in ["Extreme", "Free"] else int(annuality_str)
                    # get widgets
                    layer_combo_box_widget = getattr(self.main_dialog, f"LAYER_INTENSITY_SCENARIO_{annuality}")
                    column_name_combo_box_widget = getattr(self.main_dialog, f"COMBO_BOX_INTENSITY_COLUMN_NAME_{annuality}")
                    delete_button_widget = getattr(self.main_dialog, f"BUTTON_DEL_AFFECTED_ASSETS_{annuality}")

                    # if scenario is active, we activate the widgets, add the checkmark and fill the dropdown with the saved layer
                    if active:
                        layer_combo_box_widget.setEnabled(True)
                        column_name_combo_box_widget.setEnabled(True)
                        delete_button_widget.setEnabled(True)

                        # Set annuality string labels for free and extreme scenario
                        if annuality_str == "Free":
                            free_annuality = [value["Annuality"] for key, value in json.loads(get_process.text).items() if
                                              key == annuality_key][0]
                            self.main_dialog.LABEL_SCENARIO_FREE.setText(str(free_annuality))
                        elif annuality_str == "Extreme":
                            extreme_annuality = \
                            [value["Annuality"] for key, value in json.loads(get_process.text).items() if
                             key == annuality_key][0]
                            self.main_dialog.LABEL_SCENARIO_EXTREME.setText(str(extreme_annuality))

                        # auto-select the saved layer
                        key = f"{current_process_id}_{current_variant_id}_{annuality}"
                        saved_layer_name = self.SELECTED_LAYERS_DICT['INTENSITY_MAPS'].get(key)
                        try:
                            found_layers = QgsProject.instance().mapLayersByName(saved_layer_name)
                            if len(found_layers) == 1:
                                layer_combo_box_widget.setLayer(found_layers[0])
                            else:
                                issue = f"Could not set layer: None or multiple layers found with the name {saved_layer_name}"
                                # print(issue)
                                # TODO: error message popup to warn the user that he has multiple layers with the same name
                        except:
                            pass

                    # if scenario is inactive, we deactivate all widgets of this scenario, remove the checkmark and reset label strings (free/extreme)
                    else:
                        layer_combo_box_widget.setEnabled(False)
                        layer_combo_box_widget.setCurrentIndex(0)
                        column_name_combo_box_widget.setEnabled(False)
                        column_name_combo_box_widget.clear()
                        delete_button_widget.setEnabled(False)
                        if annuality_str == "Free":
                            self.main_dialog.LABEL_SCENARIO_FREE.setText(self.tr("Frei"))
                        elif annuality_str == "Extreme":
                            self.main_dialog.LABEL_SCENARIO_EXTREME.setText(self.tr("Extrem"))

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&EconoMe'),
                action)
            self.iface.removeToolBarIcon(action)

    def upload_consequence_analysis(self, worker):
        """This method defines the affected assets and uploads the consequence analysis to EconoMe."""

        self.OPEN_SAVE_LAYERS_DIALOG = False
        self.THREAD_PIPELINE = "consequence_analysis"

        # CHECK: Valid user info
        if self.USER_INFO_VALIDATED:
            # DO: Create damage potential layer dictionary
            DAMAGE_POTENTIAL_LAYER_DICT = {
                "BUILDING": self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer(),
                "ROAD": self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer(),
                "RAILWAY": self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer(),
                "CABLE_TRANSPORT": self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer(),
                "LINE_PIPE": self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer(),
                "OTHER_ASSET": self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer()
            }
            # CHECK: One damage potential layer must be selected
            if any(layer is not None for layer in DAMAGE_POTENTIAL_LAYER_DICT.values()):
                # DO: Create a dictionary of all currently selected damage potential layers
                current_damage_potential_layers = {k: v for k, v in DAMAGE_POTENTIAL_LAYER_DICT.items() if v is not None}
                not_synced_damage_potential_layer_keys = [key for key in current_damage_potential_layers.keys() if self.DAMAGE_POTENTIAL_LAYER_STATUS[key] == 0]
                # CHECK: Selected damage potential layers are synced with EconoMe
                if len(not_synced_damage_potential_layer_keys) == 0:
                    # CHECK: At least one intensity map layer must be selected
                    if not all(layer is None for layer in self.INTENSITY_LAYER_COLUMN_STATUS.values()):
                        # CHECK: All sekected intensity map layers need to have a column (status -1)
                        if not any(status == -1 for status in self.INTENSITY_LAYER_COLUMN_STATUS.values()):
                            # CHECK: All selected intensity map layer columns must be valid (status 0)
                            if not any(status == 0 for status in self.INTENSITY_LAYER_COLUMN_STATUS.values()):
                                # DO: Create a dictionary of all intensity layers, intensity columns and a placeholder for the intensity results
                                INTENSITY_LAYER_DICT = {
                                    self.main_dialog.LABEL_SCENARIO_FREE.text().strip(): {
                                        "intensity_layer": self.main_dialog.LAYER_INTENSITY_SCENARIO_FREE.currentLayer(),
                                        "intensity_column": self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_FREE.currentText(),
                                        "intensity_results": None
                                    },
                                    "30": {
                                        "intensity_layer": self.main_dialog.LAYER_INTENSITY_SCENARIO_30.currentLayer(),
                                        "intensity_column": self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_30.currentText(),
                                        "intensity_results": None
                                    },
                                    "100": {
                                        "intensity_layer": self.main_dialog.LAYER_INTENSITY_SCENARIO_100.currentLayer(),
                                        "intensity_column": self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_100.currentText(),
                                        "intensity_results": None
                                    },
                                    "300": {
                                        "intensity_layer": self.main_dialog.LAYER_INTENSITY_SCENARIO_300.currentLayer(),
                                        "intensity_column": self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_300.currentText(),
                                        "intensity_results": None
                                    },
                                    self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip(): {
                                        "intensity_layer": self.main_dialog.LAYER_INTENSITY_SCENARIO_EXTREME.currentLayer(),
                                        "intensity_column": self.main_dialog.COMBO_BOX_INTENSITY_COLUMN_NAME_EXTREME.currentText(),
                                        "intensity_results": None
                                    }
                                }
                                # DO: Get selection info
                                current_intensity_layer_info = {k: v for k, v in INTENSITY_LAYER_DICT.items() if v["intensity_layer"] is not None and v["intensity_column"] != ""}
                                non_selected_intensity_layer_columns = {k: v for k, v in INTENSITY_LAYER_DICT.items() if v["intensity_layer"] is not None and v["intensity_column"] == ""}
                                AFFECTED_ASSETS_SCENARIO_COLLECTION = {
                                    self.main_dialog.LABEL_SCENARIO_FREE.text().strip(): [],
                                    "30": [],
                                    "100": [],
                                    "300": [],
                                    self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip(): []
                                }
                                # Create a storage dictionary for errors
                                UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT = {}
                                # Layer error
                                intensityLayerIsValid = True
                                # DO: Loop over selected damage potential layers
                                for damage_potential_type, current_damage_potential_layer in current_damage_potential_layers.items():
                                    # DO: Get hazard intensities
                                    if INTENSITY_LAYER_DICT[self.main_dialog.LABEL_SCENARIO_FREE.text().strip()][
                                        "intensity_layer"] and INTENSITY_LAYER_DICT[
                                        self.main_dialog.LABEL_SCENARIO_FREE.text().strip()]["intensity_column"]:
                                        damage_potential_withIntensity_Free = self.get_hazard_intensity(
                                            current_damage_potential_layer,
                                            INTENSITY_LAYER_DICT[
                                                self.main_dialog.LABEL_SCENARIO_FREE.text().strip()][
                                                "intensity_layer"],
                                            INTENSITY_LAYER_DICT[
                                                self.main_dialog.LABEL_SCENARIO_FREE.text().strip()][
                                                "intensity_column"],
                                            damage_potential_type
                                        )
                                        INTENSITY_LAYER_DICT[self.main_dialog.LABEL_SCENARIO_FREE.text().strip()][
                                            "intensity_results"] = damage_potential_withIntensity_Free
                                    if INTENSITY_LAYER_DICT["30"]["intensity_layer"] and INTENSITY_LAYER_DICT["30"][
                                        "intensity_column"]:
                                        damage_potential_withIntensity_30 = self.get_hazard_intensity(
                                            current_damage_potential_layer,
                                            INTENSITY_LAYER_DICT["30"]["intensity_layer"],
                                            INTENSITY_LAYER_DICT["30"]["intensity_column"],
                                            damage_potential_type
                                        )
                                        INTENSITY_LAYER_DICT["30"]["intensity_results"] = damage_potential_withIntensity_30
                                    if INTENSITY_LAYER_DICT["100"]["intensity_layer"] and INTENSITY_LAYER_DICT["100"][
                                        "intensity_column"]:
                                        damage_potential_withIntensity_100 = self.get_hazard_intensity(
                                            current_damage_potential_layer,
                                            INTENSITY_LAYER_DICT["100"]["intensity_layer"],
                                            INTENSITY_LAYER_DICT["100"]["intensity_column"],
                                            damage_potential_type
                                        )
                                        INTENSITY_LAYER_DICT["100"]["intensity_results"] = damage_potential_withIntensity_100
                                    if INTENSITY_LAYER_DICT["300"]["intensity_layer"] and INTENSITY_LAYER_DICT["300"][
                                        "intensity_column"]:
                                        damage_potential_withIntensity_300 = self.get_hazard_intensity(
                                            current_damage_potential_layer,
                                            INTENSITY_LAYER_DICT["300"]["intensity_layer"],
                                            INTENSITY_LAYER_DICT["300"]["intensity_column"],
                                            damage_potential_type
                                        )
                                        INTENSITY_LAYER_DICT["300"]["intensity_results"] = damage_potential_withIntensity_300
                                    if INTENSITY_LAYER_DICT[self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()][
                                        "intensity_layer"] and INTENSITY_LAYER_DICT[
                                        self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()]["intensity_column"]:
                                        damage_potential_withIntensity_Extreme = self.get_hazard_intensity(
                                            current_damage_potential_layer,
                                            INTENSITY_LAYER_DICT[
                                                self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()][
                                                "intensity_layer"],
                                            INTENSITY_LAYER_DICT[
                                                self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()][
                                                "intensity_column"],
                                            damage_potential_type
                                        )
                                        INTENSITY_LAYER_DICT[self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()][
                                            "intensity_results"] = damage_potential_withIntensity_Extreme

                                    # DO: Retrieve parameter profile and get parameter associations
                                    with requests.get(
                                            self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{damage_potential_type.lower()}",
                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                    ) as get_parameter_profile:
                                        PARAMETER_PROFILE = get_parameter_profile.json()

                                    # DO: Get scenes
                                    with requests.get(
                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/scene',
                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                    ) as get_scenes:
                                        SCENES = get_scenes.json()

                                    # DO: Get damage potential
                                    with requests.get(
                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + '/damagepotential/asset',
                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                    ) as get_damage_potential:
                                        DAMAGE_POTENTIAL = get_damage_potential.json()
                                    # DO: Filter damage potential by object category
                                    OBJECT_CATEGORIES = [item for sublist in
                                                        [list(self.ASSET_TYPEID_NAME_BY_CATEGORYNAME[obj_cat].values()) for
                                                        obj_cat in
                                                        self.ASSET_CATEGORIES_BY_DAMAGEPOTENTIAL_TYPE[damage_potential_type]]
                                                        for item in sublist]
                                    FILTERED_DAMAGE_POTENTIAL = [asset for asset in DAMAGE_POTENTIAL if
                                                                asset["AssetTypeID"] in OBJECT_CATEGORIES]

                                    # DO: Create affected assets list for each scenario depending on damage potential layer type
                                    for ann in [self.main_dialog.LABEL_SCENARIO_FREE.text().strip(), "30", "100",
                                                "300", self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()]:
                                        # Create an empty affected assets list
                                        AFFECTED_ASSETS = {
                                            "AffectedAssetList": []
                                        }
                                        # CHECK: Existing intensities
                                        if INTENSITY_LAYER_DICT[ann]["intensity_results"]:

                                            # For BUILDING and OTHER ASSET point features
                                            if current_damage_potential_layer.geometryType() == 0 or damage_potential_type == "BUILDING":
                                                intensity_max_column = INTENSITY_LAYER_DICT[ann]["intensity_column"] + "_max"
                                                affected_features = [feature for feature in INTENSITY_LAYER_DICT[ann][
                                                    "intensity_results"].getFeatures() if feature[intensity_max_column]]
                                                existing_special_parameter_fields_wrong_type = []
                                                for feature in affected_features:
                                                    AssetIdentID_user_name = \
                                                    [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if
                                                    p["API"] == "AssetIdentID"][0]
                                                    corresponding_asset = [asset for asset in FILTERED_DAMAGE_POTENTIAL if
                                                                        asset["AssetIdentID"] == str(
                                                                            feature[AssetIdentID_user_name])][0]

                                                    intensity_class = None

                                                    if int(feature[intensity_max_column]) == 1:
                                                        intensity_class = "weak"
                                                    elif int(feature[intensity_max_column]) == 2:
                                                        intensity_class = "moderate"
                                                    elif int(feature[intensity_max_column]) == 3:
                                                        intensity_class = "strong"

                                                    new_affected_asset = {
                                                        "AssetID": corresponding_asset['AssetID'],
                                                        "IntensityClass": intensity_class
                                                    }

                                                    ## Add special parameters
                                                    # TEST: Layers with same projection
                                                    affecting_intensity_area_test = [intens_feat for intens_feat in
                                                                                    INTENSITY_LAYER_DICT[ann][
                                                                                        "intensity_layer"].getFeatures() if
                                                                                    intens_feat.geometry().intersects(
                                                                                        feature.geometry())]
                                                    if len(affecting_intensity_area_test) != 0:
                                                        # Get corresponding intensity feature
                                                        affecting_intensity_area = [intens_feat for intens_feat in
                                                                                    INTENSITY_LAYER_DICT[ann][
                                                                                        "intensity_layer"].getFeatures() if
                                                                                    intens_feat.geometry().intersects(
                                                                                        feature.geometry()) and int(intens_feat[
                                                                                                                        INTENSITY_LAYER_DICT[
                                                                                                                            ann][
                                                                                                                            "intensity_column"]]) == int(
                                                                                        feature[intensity_max_column])][0]
                                                        # Get fields of intensity layer
                                                        intens_layer_fields = INTENSITY_LAYER_DICT[ann][
                                                            "intensity_layer"].fields()
                                                        # Existing field names
                                                        already_filled_parameters = []
                                                        # Loop through special parameters list for object category
                                                        for param in self.AFFECTED_ASSET_SPECIAL_PARAMETERS[
                                                            damage_potential_type]:
                                                            # Check if parameter field exists in intensity layer
                                                            possible_fieldnames = [name for name in intens_layer_fields.names()
                                                                                if name in param[0]]
                                                            if len(possible_fieldnames) != 0:
                                                                if max(possible_fieldnames) not in already_filled_parameters:
                                                                    # Get field name
                                                                    fieldname = max(possible_fieldnames)
                                                                    already_filled_parameters.append(fieldname)
                                                                    # Get field
                                                                    field = intens_layer_fields.field(fieldname)
                                                                    # Write special parameters to the affected asset if type is correct
                                                                    if param[1] == "string":
                                                                        if field.type() == QVariant.String:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                new_affected_asset[param[0]] = \
                                                                                affecting_intensity_area[fieldname]
                                                                        else:
                                                                            existing_special_parameter_fields_wrong_type.append(
                                                                                param[0])
                                                                    if param[1] == "integer":
                                                                        if field.type() == QVariant.Int or field.type() == QVariant.LongLong:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                new_affected_asset[param[0]] = \
                                                                                affecting_intensity_area[fieldname]
                                                                        else:
                                                                            existing_special_parameter_fields_wrong_type.append(
                                                                                param[0])
                                                                    if param[1] == "boolean":
                                                                        if field.type() == QVariant.Bool:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                new_affected_asset[param[0]] = \
                                                                                affecting_intensity_area[fieldname]
                                                                        elif field.type() == QVariant.Int:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                if affecting_intensity_area[fieldname] == 0:
                                                                                    new_affected_asset[param[0]] = False
                                                                                elif affecting_intensity_area[fieldname] == 1:
                                                                                    new_affected_asset[param[0]] = True
                                                                        else:
                                                                            existing_special_parameter_fields_wrong_type.append(
                                                                                param[0])
                                                                    if param[1] == "number":
                                                                        if field.type() == QVariant.Int or field.type() == QVariant.LongLong or field.type() == QVariant.Double:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                new_affected_asset[param[0]] = \
                                                                                affecting_intensity_area[fieldname]
                                                                        else:
                                                                            existing_special_parameter_fields_wrong_type.append(
                                                                                param[0])
                                                                    if param[1] == "numZeroToOne":
                                                                        if field.type() == QVariant.Int or field.type() == QVariant.LongLong or field.type() == QVariant.Double:
                                                                            if affecting_intensity_area[fieldname]:
                                                                                if affecting_intensity_area[fieldname] >= 0 and \
                                                                                        affecting_intensity_area[
                                                                                            fieldname] <= 1:
                                                                                    new_affected_asset[param[0]] = \
                                                                                    affecting_intensity_area[fieldname]
                                                                        else:
                                                                            existing_special_parameter_fields_wrong_type.append(
                                                                                param[0])

                                                        # USER INFO: Existing special parameter columns but wrong type
                                                        if len(existing_special_parameter_fields_wrong_type) != 0:
                                                            # USER INFO: Special parameter fields with wrong type
                                                            user_info_title = self.tr("Fehler")
                                                            user_info_text = self.tr("\nFolgende Attributspalten spezieller Parameter haben den falschen Typ:\n-> {existing_special_parameter_fields_wrong_type}\n").format(existing_special_parameter_fields_wrong_type=existing_special_parameter_fields_wrong_type)
                                                            worker.show_status.emit(user_info_title, user_info_text, 'error')
                                                            intensityLayerIsValid = False
                                                            break

                                                        # Check for p_rA and missing p_rA_Comment
                                                        if "p_rA" in new_affected_asset:
                                                            if "p_rA_Comment" in new_affected_asset:
                                                                if not new_affected_asset["p_rA_Comment"]:
                                                                    new_affected_asset[
                                                                        "p_rA_Comment"] = self.tr("p_rA wurde aus IK übernommen!")
                                                            else:
                                                                new_affected_asset[
                                                                    "p_rA_Comment"] = self.tr("p_rA wurde aus IK übernommen!")

                                                        # Check for PresenceValue and missing PresenceComment
                                                        if "PresenceValue" in new_affected_asset:
                                                            if "PresenceComment" in new_affected_asset:
                                                                if not new_affected_asset["PresenceComment"]:
                                                                    new_affected_asset[
                                                                        "PresenceComment"] = self.tr("PresenceValue wurde aus IK übernommen!")
                                                            else:
                                                                new_affected_asset[
                                                                    "PresenceComment"] = self.tr("PresenceValue wurde aus IK übernommen!")

                                                        # Check for ProtectionValue and missing ProtectionComment
                                                        if "ProtectionValue" in new_affected_asset:
                                                            if "ProtectionComment" in new_affected_asset:
                                                                if not new_affected_asset["ProtectionComment"]:
                                                                    new_affected_asset[
                                                                        "ProtectionComment"] = self.tr("ProtectionValue wurde aus IK übernommen!")
                                                            else:
                                                                new_affected_asset[
                                                                    "ProtectionComment"] = self.tr("ProtectionValue wurde aus IK übernommen!")

                                                        AFFECTED_ASSETS['AffectedAssetList'].append(new_affected_asset)
                                                    # Display warning message, layers with different projections
                                                    else:
                                                        # USER INFO: Damage potential layer and intensity maps with different coordinate systems
                                                        user_info_title = self.tr("Fehler")
                                                        user_info_text = self.tr("\nSchadenpotential-Layer und Intensitätskarten mit unterschiedlichem Koordinatensystem. Bitte anpassen!\n")
                                                        worker.show_status.emit(user_info_title, user_info_text, 'error')
                                                        intensityLayerIsValid = False
                                                        break
                                            # For ROAD, RAILWAY, CABLE TRANSPORT and OTHER ASSET line or polygon features
                                            elif current_damage_potential_layer.geometryType() == 1 or current_damage_potential_layer.geometryType() == 2:
                                                affected_features = [feature for feature in INTENSITY_LAYER_DICT[ann][
                                                    "intensity_results"].getFeatures() if
                                                                    feature['intensity_weak'] > 0 or feature[
                                                                        'intensity_moderate'] > 0 or feature[
                                                                        'intensity_strong'] > 0]
                                                existing_special_parameter_fields_wrong_type = []
                                                for feature in affected_features:
                                                    AssetIdentID_user_name = \
                                                    [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if
                                                    p["API"] == "AssetIdentID"][0]
                                                    corresponding_asset = [asset for asset in FILTERED_DAMAGE_POTENTIAL if
                                                                        asset["AssetIdentID"] == str(
                                                                            feature[AssetIdentID_user_name])][0]

                                                    intensity_weak = 0
                                                    intensity_moderate = 0
                                                    intensity_strong = 0

                                                    if feature['intensity_weak']:
                                                        intensity_weak = int(feature['intensity_weak'])
                                                    if feature['intensity_moderate']:
                                                        intensity_moderate = int(feature['intensity_moderate'])
                                                    if feature['intensity_strong']:
                                                        intensity_strong = int(feature['intensity_strong'])

                                                    if (intensity_weak + intensity_moderate + intensity_strong) > 0:
                                                        new_affected_asset = {
                                                            "AssetID": corresponding_asset['AssetID'],
                                                            "WeakIntensity": intensity_weak,
                                                            "ModerateIntensity": intensity_moderate,
                                                            "StrongIntensity": intensity_strong
                                                        }

                                                        ## Add special parameters (Methodik ?)

                                                        AFFECTED_ASSETS['AffectedAssetList'].append(new_affected_asset)

                                        if len(AFFECTED_ASSETS['AffectedAssetList']) != 0:
                                            AFFECTED_ASSETS_SCENARIO_COLLECTION[ann].extend(
                                                AFFECTED_ASSETS['AffectedAssetList'])

                                for ann in [self.main_dialog.LABEL_SCENARIO_FREE.text().strip(), "30", "100", "300",
                                            self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()]:

                                    # Get corresponding scene ID based on annuality, selected process IDs and variant IDs
                                    process_ID = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText().split(", ")[0]
                                    variant_ID = 0
                                    if self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(", ")[0]:
                                        variant_ID = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText().split(",")[0]
                                    # CHECK: Is annuality a digit ?
                                    if ann.isdigit():
                                        scenes_match = [scene["SceneID"] for scene in SCENES if
                                                        int(scene["Annuality"]) == int(ann) and int(scene["ProcessID"]) == int(
                                                            process_ID) and int(scene["VariantID"]) == int(variant_ID)]
                                        # CHECK: Matching scene
                                        if len(scenes_match) != 0:
                                            scene_ID = scenes_match[0]
                                            # Always delete consequence analysis before
                                            with requests.delete(
                                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                            ) as delete_before:

                                                # CHECK: Is the AffectedAssetList empty ?
                                                if AFFECTED_ASSETS_SCENARIO_COLLECTION[ann] and intensityLayerIsValid:

                                                    AFFECTED_ASSETS_PER_SCENARIO = {
                                                        'AffectedAssetList': AFFECTED_ASSETS_SCENARIO_COLLECTION[ann]
                                                    }

                                                    # Get existing affected assets
                                                    with requests.get(
                                                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                                            auth=HTTPBasicAuth(self.ECONOME_USER_NAME, self.ECONOME_PASSWORD)
                                                    ) as get_affected_assets:
                                                        UPLOADED_AFFECTED_ASSETS = json.loads(get_affected_assets.text)
                                                        # Existing affected assets, update affected assets
                                                        if len(UPLOADED_AFFECTED_ASSETS) != 0:
                                                            # Post new affected assets
                                                            with requests.post(
                                                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                                                    json=AFFECTED_ASSETS_PER_SCENARIO,
                                                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME,
                                                                                    self.ECONOME_PASSWORD)
                                                            ) as post_affected_assets:
                                                                if post_affected_assets.status_code != 200:
                                                                    UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT[
                                                                        ann] = post_affected_assets.json()
                                                        # No existing affected assets, post new affected assets
                                                        else:
                                                            # Post affected assets
                                                            with requests.post(
                                                                    self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/scene/{scene_ID}/asset",
                                                                    json=AFFECTED_ASSETS_PER_SCENARIO,
                                                                    auth=HTTPBasicAuth(self.ECONOME_USER_NAME,
                                                                                    self.ECONOME_PASSWORD)
                                                            ) as post_affected_assets:
                                                                if post_affected_assets.status_code != 200:
                                                                    UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT[ann] = json.loads(
                                                                        post_affected_assets.text)

                                if len(UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT) == 0 and intensityLayerIsValid:
                                    # USER INFO: Success
                                    user_info_title = self.tr("Erfolg")
                                    user_info_text = self.tr("\nKonsequenzanalyse erfolgreich hochgeladen!\n")
                                    worker.show_status.emit(user_info_title, user_info_text, 'success')
                                else:
                                    # USER INFO: Error
                                    user_info_title = self.tr("Fehler")
                                    user_info_text = self.tr("\nFehler beim Hochladen der Konsequenzanalyse!\n")
                                    user_info_text += " ".join(
                                        [f"{key}: {value['Message']}" for key, value in UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT.items()])
                                    user_info_text += " ".join(
                                        [f"{key}: {value['Details']}" for key, value in UPLOAD_CONSEQUENCE_ANALYSIS_ERRORS_DICT.items()])
                                    worker.show_status.emit(user_info_title, user_info_text, 'error')
                            
                            else:
                                # USER INFO: Intensity Values not valid
                                user_info_title = self.tr("Fehler")
                                user_info_text = self.tr("\nAusgewählte Intensitäts-Spalten nicht gültig!\nBitte wähle eine gültige Intensitäts-Spalte für jeden angewählten Intensitäts-Layer in der BAFU-Skala [(0), 1, 2, 3] aus.\n")
                                worker.show_status.emit(user_info_title, user_info_text, 'error')
                        else:
                            # USER INFO: Intensity column not selected
                            user_info_title = self.tr("Fehler")
                            user_info_text = self.tr("\nIntensitäts-Spalten nicht ausgewählt!\nBitte wähle für jeden angewählten Intensitäts-Layer die Intensitäts-Spalte aus.\n")
                            worker.show_status.emit(user_info_title, user_info_text, 'error')
                    else:
                        # USER INFO: No intensity layer selected
                        user_info_title = self.tr("Fehler")
                        user_info_text = self.tr("\nKein Intensitäts-Layer ausgewählt!\nBitte wähle mindestens 1 Intensitäts-Layer aus.\n")
                        worker.show_status.emit(user_info_title, user_info_text, 'error')
                else:
                    # USER INFO: At least one selected damage potential layer is not synced with EconoMe
                    user_info_title = self.tr("Fehler")
                    user_info_text = self.tr("\nNicht alle Schadenpotential-Layers sind bereit für den Verschnitt!\nBitte synchronisiere die ausgewählten Schadenpotential-Layer oder wähle sie ab, um fortzufahren!\n")
                    worker.show_status.emit(user_info_title, user_info_text, 'error')
            else:
                # USER INFO: No damage potential layer selected
                user_info_title = self.tr("Fehler")
                user_info_text = self.tr("\nKein Schadenpotential-Layer ausgewählt!\n")
                worker.show_status.emit(user_info_title, user_info_text, 'error')
        else:
            # USER INFO: No valid user info
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nUngültige Anmeldedaten: Projektnummer oder API-Keys!\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')

    def validate_econome_user_info(self):
        """Checks if API keys were given and are valid."""

        # Get user name and password from lineEdit
        ECONOME_USER_NAME = self.main_dialog.LINE_EDIT_ECONOME_USER_NAME.text().strip()
        ECONOME_PASSWORD = self.main_dialog.LINE_EDIT_ECONOME_PASSWORD.text().strip()

        # CHECK: User name and password filled
        if ECONOME_USER_NAME and ECONOME_PASSWORD:

            # Make test API request
            with requests.get(
                    self.ECONOME_API_PROJECT_BASE_PATH,
                    auth=HTTPBasicAuth(ECONOME_USER_NAME, ECONOME_PASSWORD),
                    timeout=10
            ) as make_test_request:
                # Check if API is responding
                try:
                    TEST_REQUEST = make_test_request.json()
                except ValueError:
                    TEST_REQUEST = []

            # Test API request was successful
            if 'ProjectList' in TEST_REQUEST:
                # Test if user has at least 1 project
                if len(TEST_REQUEST['ProjectList']) > 0:
                    # Set EconoMe user info
                    self.ECONOME_USER_NAME = ECONOME_USER_NAME
                    self.ECONOME_PASSWORD = ECONOME_PASSWORD
                    self.ECONOME_PROJECT_LIST = [(proj['ProjectIdentID'], proj['ProjectTitle']) for proj in TEST_REQUEST['ProjectList']]
                    if not self.ECONOME_PROJECT_NUMBER or self.ECONOME_PROJECT_NUMBER not in self.ECONOME_PROJECT_LIST or self.THREAD_PIPELINE == "update_settings":
                        if not self.ECONOME_PROJECT_NUMBER:
                            self.ECONOME_PROJECT_NUMBER = self.ECONOME_PROJECT_LIST[0][0]
                        if self.ECONOME_PROJECT_NUMBER == None or self.ECONOME_PROJECT_NUMBER not in [num[0] for num in self.ECONOME_PROJECT_LIST]:
                            self.ECONOME_PROJECT_NUMBER = self.ECONOME_PROJECT_LIST[0][0]
                    # Fill dropdown with ProjectIdentIDs
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
                    for project_info in self.ECONOME_PROJECT_LIST:
                        if project_info[0] not in [self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.itemText(i).split(', ') for i in range(self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.count())]:
                            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(f"{project_info[1]}, {project_info[0]}")
                    # Set current project
                    selected_project_name = [info[1] for info in self.ECONOME_PROJECT_LIST if info[0] == self.ECONOME_PROJECT_NUMBER]
                    if len(selected_project_name) != 0:
                        index = self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.findText(f"{selected_project_name[0]}, {self.ECONOME_PROJECT_NUMBER}")
                        if index != -1:
                            # Select the item by setting the current index
                            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentIndex(index)
                        else:
                            self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setCurrentIndex(0)
                    # Save project infos in plugin folder (if they were not existent)
                    if not self.get_econome_user_info():
                        # Create a temporary file in the plugin folder
                        self.ECONOME_USER_INFO_FILE_PATH = tempfile.NamedTemporaryFile(dir=self.plugin_dir, prefix="econome_user_info_", suffix=".txt", delete=False)
                        # Write project infos to a temporary text file which persists until QGIS is closed
                        with open(self.ECONOME_USER_INFO_FILE_PATH.name, "w") as file:
                            file.write(f"{self.ECONOME_PROJECT_NUMBER}\n")
                            file.write(f"{', '.join([item[0] for item in self.ECONOME_PROJECT_LIST])}\n")
                            file.write(f"{ECONOME_USER_NAME}\n")
                            file.write(f"{ECONOME_PASSWORD}\n")
                            file.flush()
                    # If existing user info, check for differences and overwrite
                    else:
                        existing_user_info = self.get_econome_user_info()
                        if existing_user_info['ECONOME_PROJECT_NUMBER'] != self.ECONOME_PROJECT_NUMBER or existing_user_info['ECONOME_PROJECT_LIST'] != self.ECONOME_PROJECT_LIST or existing_user_info['ECONOME_USER_NAME'] != self.ECONOME_USER_NAME or existing_user_info['ECONOME_PASSWORD'] != self.ECONOME_PASSWORD:
                            # Overwrite project infos to a temporary text file which persists until QGIS is closed
                            with open(os.path.join(self.plugin_dir, self.ECONOME_USER_INFO_FILE_PATH), "w") as file:
                                file.write(f"{self.ECONOME_PROJECT_NUMBER}\n")
                                file.write(f"{', '.join([item[0] for item in self.ECONOME_PROJECT_LIST])}\n")
                                file.write(f"{ECONOME_USER_NAME}\n")
                                file.write(f"{ECONOME_PASSWORD}\n")
                                file.flush()
                    # Set new label text
                    self.main_dialog.LABEL_ECONOME_PROJECT_NUMBER.setText(self.tr("Projekt(e)"))
                    # Color the login button green
                    self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet("background-color: #27ae60;")
                    # Black color for combo box items
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: black; }")
                    # EconoMe user info -> validated
                    self.USER_INFO_VALIDATED = True
                    # Enable buttons and dropdowns
                    self.disable_buttons_and_dropdowns(False)
                    # Get project status
                    self.get_project_status()
                else:
                    # EconoMe user info -> not validated
                    self.USER_INFO_VALIDATED = False
                    # Disable buttons and dropdowns
                    self.disable_buttons_and_dropdowns(True)
                    # Remove green color from login button
                    self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet("background-color: none;")
                    # Receive error messages and display them in the combo box
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem("No projects found ...")
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: red; }")
            # Test API request was not successful
            else:
                # EconoMe user info -> not validated
                self.USER_INFO_VALIDATED = False
                # Disable buttons and dropdowns
                self.disable_buttons_and_dropdowns(True)
                # Remove green color from login button
                self.main_dialog.BUTTON_ECONOME_PROJECT_CONNECT.setStyleSheet("background-color: none;")
                # Receive error messages and display them in the combo box
                if "Errorcode" in TEST_REQUEST:
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(f"{TEST_REQUEST['Errormessage']}")
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: red; }")
                elif TEST_REQUEST == []:
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem("API doesn't respond ...")
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: red; }")
                else:
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.clear()
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.addItem(f"{TEST_REQUEST['Message']}")
                    self.main_dialog.COMBO_BOX_ECONOME_PROJECT_LIST.setStyleSheet("QComboBox:editable { color: red; }")
                # Get project status
                self.get_project_status()
        # User name and/or password were not filled
        else:
            # EconoMe user info -> not validated
            self.USER_INFO_VALIDATED = False
            # Disable buttons and dropdowns
            self.disable_buttons_and_dropdowns(True)
            # Get project status
            self.get_project_status()

    def visualize_risks(self, worker):
        """Fetches risk from EconoMe and creates a visualized layer."""

        self.RISK_OUTPUT_TEMP_LAYERS = [] # Create empty result layers dict
        self.THREAD_PIPELINE = "visualize_risks" # Define thread pipeline

        # CHECK: Valid user info
        if self.USER_INFO_VALIDATED:
            # Get all currently selected damage potential layers and store them in a dict
            DAMAGE_POTENTIAL_LAYER_DICT = {
                "BUILDING": self.main_dialog.LAYER_DAM_POT_BUILDING.currentLayer(),
                "ROAD": self.main_dialog.LAYER_DAM_POT_ROAD.currentLayer(),
                "RAILWAY": self.main_dialog.LAYER_DAM_POT_RAILWAY.currentLayer(),
                "CABLE_TRANSPORT": self.main_dialog.LAYER_DAM_POT_CABLE_TRANSPORT.currentLayer(),
                "LINE_PIPE": self.main_dialog.LAYER_DAM_POT_LINE_PIPE.currentLayer(),
                "OTHER_ASSET": self.main_dialog.LAYER_DAM_POT_OTHER_ASSET.currentLayer()
            }
            # CHECK: At least one damage potential layer selected
            if any(layer is not None for layer in DAMAGE_POTENTIAL_LAYER_DICT.values()):
                # Create a dict of all input damage potential layers
                current_damage_potential_layers = {k: v for k, v in DAMAGE_POTENTIAL_LAYER_DICT.items() if v is not None}
                current_damage_potential_layer_status = [value for key, value in self.DAMAGE_POTENTIAL_LAYER_STATUS.items() if key in current_damage_potential_layers.keys()]
                
                # CHECK: Are all selected damage potential layers synced with EconoMe?
                if all(status == 1 for status in current_damage_potential_layer_status):
                    # Loop over damage potential layers
                    for key, current_damage_potential_layer in current_damage_potential_layers.items():

                        # Fetch API infos
                        API_GET_responses = self.multiple_API_GET_requests([
                            self.ECONOME_API_PROFILE_BASE_PATH + f"/plugin-{key.lower()}",
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/process",
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/damagepotential/asset",
                            self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + "/scene"
                        ])

                        PARAMETER_PROFILE = API_GET_responses[0].result()[1] # Get corresponding parameter profile
                        HAZARD_PROCESSES = API_GET_responses[1].result()[1] # Get hazard processes
                        UPLOADED_DAMAGE_POTENTIAL = API_GET_responses[2].result()[1] # Get damage potential
                        SCENES = API_GET_responses[3].result()[1] # Get scenes

                        # Read selected hazard process
                        SELECTED_HAZARD_PROCESS = self.main_dialog.COMBO_BOX_HAZARD_PROCESS.currentText()
                        # Read selected measure variant
                        SELECTED_MEASURE_VARIANT = self.main_dialog.COMBO_BOX_MEASURE_VARIANT.currentText()

                        # CHECK: Existing scenes
                        if len(SCENES) != 0:

                            # Free and extreme annualities
                            free_annuality = "-1"
                            extreme_annuality = "-1"

                            # Check for free and extreme annuality
                            if self.main_dialog.LABEL_SCENARIO_FREE.text().isdigit():
                                free_annuality = self.main_dialog.LABEL_SCENARIO_FREE.text().strip()
                            if self.main_dialog.LABEL_SCENARIO_EXTREME.text().isdigit():
                                extreme_annuality = self.main_dialog.LABEL_SCENARIO_EXTREME.text().strip()

                            # Store risk data output
                            RISK_OUTPUT_DICT = {
                                free_annuality: {
                                    "risk": None
                                },
                                "30": {
                                    "risk": None
                                },
                                "100": {
                                    "risk": None
                                },
                                "300": {
                                    "risk": None
                                },
                                extreme_annuality: {
                                    "risk": None
                                }
                            }

                            # Get risk from all scenes matching user selection

                            # Create scene ID dictionary
                            SCENE_ID_DICT = {
                                free_annuality: [scene["SceneID"] for scene in SCENES if scene["Annuality"] == int(free_annuality) and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "30": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 30 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "100": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 100 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                "300": [scene["SceneID"] for scene in SCENES if scene["Annuality"] == 300 and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])],
                                extreme_annuality: [scene["SceneID"] for scene in SCENES if scene["Annuality"] == int(extreme_annuality) and int(scene["ProcessID"]) == int(SELECTED_HAZARD_PROCESS.split(", ")[0]) and int(scene["VariantID"]) == int(SELECTED_MEASURE_VARIANT.split(", ")[0])]
                            }

                            FILTERED_GET_RISK_REQUESTS = {}
                            for annuality, scene_id in SCENE_ID_DICT.items():
                                if len(scene_id) != 0:
                                    FILTERED_GET_RISK_REQUESTS[annuality] = self.ECONOME_API_PROJECT_BASE_PATH + f"/{self.ECONOME_PROJECT_NUMBER}" + f"/risk/scene/{scene_id[0]}"

                            # Fetch API Risk values
                            API_GET_responses = self.multiple_API_GET_requests(list(FILTERED_GET_RISK_REQUESTS.values()))

                            index = 0
                            for response in API_GET_responses:
                                RISK_OUTPUT_DICT[list(FILTERED_GET_RISK_REQUESTS.keys())[index]]["risk"] = response.result()[1]
                                index += 1

                            has_risk = any(val["risk"] not in (None, []) for val in RISK_OUTPUT_DICT.values())

                            # CHECK: Are there risks in EconoMe
                            if has_risk:
                                self.OPEN_SAVE_LAYERS_DIALOG = True
                                # Create a copy of the damage potential layer as a memory layer with the prefix of the hazard process
                                type_and_projection = ""
                                if current_damage_potential_layer.geometryType() == 0:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"Point?crs={current_damage_potential_layer.crs().postgisSrid()}"
                                elif current_damage_potential_layer.geometryType() == 1:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"LineString?crs={current_damage_potential_layer.crs().postgisSrid()}"
                                elif current_damage_potential_layer.geometryType() == 2:
                                    # layer.geometryType() 0,1,2 Point Line Polygon  ---- layer.crs().postgisSrid()
                                    type_and_projection = f"Polygon?crs={current_damage_potential_layer.crs().postgisSrid()}"

                                current_damage_potential_layer_AssetIdentID = [p["User"] for p in PARAMETER_PROFILE['ParameterList'] if p["API"] == "AssetIdentID"][0]
                                current_damage_potential_layer_fields = current_damage_potential_layer.fields()
                                timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
                                risks_translation_str = self.tr("Risiken")
                                new_risk_layer = QgsVectorLayer(type_and_projection, f"{risks_translation_str}_{SELECTED_MEASURE_VARIANT.split(', ')[1][:25]}_{SELECTED_HAZARD_PROCESS.split(', ')[2][:25]}_{key}_{timestamp}", "memory")
                                new_risk_layer.dataProvider().addAttributes(current_damage_potential_layer_fields)

                                # Switch to edit mode
                                new_risk_layer.startEditing()

                                # Copy all features from the original layer to the new layer
                                for feature in current_damage_potential_layer.getFeatures():
                                    new_risk_layer.dataProvider().addFeature(feature)

                                # Loop over all risks and define new fields
                                for annuality in RISK_OUTPUT_DICT.keys():
                                    new_fields = QgsFields()
                                    # CHECK: Is there risk data ?
                                    if RISK_OUTPUT_DICT[annuality]["risk"]:
                                        # Define new fields if not existing
                                        for output_variable in self.RISK_OUTPUT_VARIABLES[key]:
                                            if output_variable not in current_damage_potential_layer_fields:
                                                new_fields.append(QgsField(f"{output_variable}_{annuality}", QVariant.Double))
                                                new_risk_layer.dataProvider().addAttributes(new_fields)
                                        # Update fields
                                        new_risk_layer.updateFields()

                                # Loop over all risks and fill fields
                                for annuality in RISK_OUTPUT_DICT.keys():
                                    # CHECK: Is there risk data ?
                                    if RISK_OUTPUT_DICT[annuality]["risk"]:
                                        # LOOP over all features
                                        for feature in new_risk_layer.getFeatures():
                                            # Get corresponding Asset ID
                                            corresponding_asset_id = [asset["AssetID"] for asset in UPLOADED_DAMAGE_POTENTIAL if asset["AssetIdentID"] == str(feature[current_damage_potential_layer_AssetIdentID])][0]
                                            result = [elem for elem in RISK_OUTPUT_DICT[annuality]["risk"] if int(elem["AssetID"]) == int(corresponding_asset_id)]
                                            if len(result) != 0:
                                                # Loop over all output variables
                                                for output_variable in self.RISK_OUTPUT_VARIABLES[key]:
                                                    # Enter attributes
                                                    feature.setAttribute(f"{output_variable}_{annuality}", result[0][output_variable])
                                            # Update feature
                                            new_risk_layer.updateFeature(feature)

                                # Save edits
                                new_risk_layer.commitChanges()
                                # Set color renderer

                                new_risk_layer_fields = [field.name() for field in new_risk_layer.fields()]

                                # Add summary risk fields
                                if f"R_{free_annuality}" in new_risk_layer_fields or "R_30" in new_risk_layer_fields or "R_100" in new_risk_layer_fields or "R_300" in new_risk_layer_fields or f"R_{extreme_annuality}" in new_risk_layer_fields:
                                    if key not in ["OTHER_ASSET", "LINE_PIPE"]:
                                        new_fields = QgsFields()
                                        new_fields.append(QgsField("R", QVariant.Double))
                                        new_fields.append(QgsField("r_indiv", QVariant.Double))
                                        new_risk_layer.dataProvider().addAttributes(new_fields)
                                        new_risk_layer.updateFields()

                                new_risk_layer_fields = [field.name() for field in new_risk_layer.fields()]
                                new_risk_layer.startEditing()

                                # Add summary risk fields
                                for feature in new_risk_layer.getFeatures():
                                    r_sum = 0
                                    if "R" in new_risk_layer_fields:
                                        r_sum = 0
                                        if f"R_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"R_{free_annuality}"] is not None:
                                                r_sum += feature[f"R_{free_annuality}"]
                                        if "R_30" in new_risk_layer_fields:
                                            if feature["R_30"] is not None:
                                                r_sum += feature["R_30"]
                                        if "R_100" in new_risk_layer_fields:
                                            if feature["R_100"] is not None:
                                                r_sum += feature["R_100"]
                                        if "R_300" in new_risk_layer_fields:
                                            if feature["R_300"] is not None:
                                                r_sum += feature["R_300"]
                                        if f"R_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"R_{extreme_annuality}"] is not None:
                                                r_sum += feature[f"R_{extreme_annuality}"]

                                        feature.setAttribute("R", r_sum)

                                    r_indiv_sum = 0
                                    if "r_indiv" in new_risk_layer_fields:
                                        r_indiv_sum = 0
                                        if f"r_indiv_{free_annuality}" in new_risk_layer_fields:
                                            if feature[f"r_indiv_{free_annuality}"] is not None:
                                                r_indiv_sum += feature[f"r_indiv_{free_annuality}"]
                                        if f"r_indiv_30" in new_risk_layer_fields:
                                            if feature["r_indiv_30"] is not None:
                                                r_indiv_sum += feature["r_indiv_30"]
                                        if f"r_indiv_100" in new_risk_layer_fields:
                                            if feature["r_indiv_100"] is not None:
                                                r_indiv_sum += feature["r_indiv_100"]
                                        if f"r_indiv_300" in new_risk_layer_fields:
                                            if feature["r_indiv_300"] is not None:
                                                r_indiv_sum += feature["r_indiv_300"]
                                        if f"r_indiv_{extreme_annuality}" in new_risk_layer_fields:
                                            if feature[f"r_indiv_{extreme_annuality}"] is not None:
                                                r_indiv_sum += feature[f"r_indiv_{extreme_annuality}"]

                                        feature.setAttribute("r_indiv", r_indiv_sum)

                                    new_risk_layer.updateFeature(feature)

                                new_risk_layer.commitChanges()

                                # Visualize r_indiv
                                if "r_indiv" in new_risk_layer_fields:

                                    new_risk_layer.startEditing()

                                    for feature in new_risk_layer.getFeatures():
                                        try:
                                            if feature["r_indiv"]:
                                                feature["r_indiv"] = float(feature["r_indiv"])
                                                new_risk_layer.updateFeature(feature)
                                        except ValueError:
                                            print(f"Invalid value: {feature['r_indiv']}")
                                    new_risk_layer.commitChanges()

                                    ranges = []

                                    # Define first range
                                    symbol1 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())
                                    symbol1.setColor(QColor("lightblue"))
                                    if key in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol1.setWidth(1.5)
                                    range1 = QgsRendererRange(0, 0.000001, symbol1, "< 1e-6")
                                    ranges.append(range1)

                                    # Define second range
                                    symbol2 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())
                                    if key in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol2.setWidth(1.5)
                                    symbol2.setColor(QColor("orange"))
                                    range2 = QgsRendererRange(0.000001, 0.00001, symbol2, "1e-6 - 1e-5")
                                    ranges.append(range2)

                                    # Define third range
                                    symbol3 = QgsSymbol.defaultSymbol(new_risk_layer.geometryType())
                                    if key in ["ROAD", "RAILWAY", "CABLE_TRANSPORT", "LINE_PIPE"]:
                                        symbol3.setWidth(1.5)
                                    symbol3.setColor(QColor("red"))
                                    range3 = QgsRendererRange(0.00001, 1, symbol3, "> 1e-5")
                                    ranges.append(range3)

                                    renderer = QgsGraduatedSymbolRenderer("r_indiv", ranges)

                                    # Set the renderer to the layer
                                    new_risk_layer.setRenderer(renderer)

                                # Collect risk layers
                                if new_risk_layer:
                                    self.RISK_OUTPUT_TEMP_LAYERS.append((new_risk_layer, key))

                        else:
                            # USER INFO: Keine Szenen definiert
                            self.THREAD_PIPELINE = ""
                            user_info_title = self.tr("Fehler")
                            user_info_text = self.tr("\nKeine Szenen vorhanden!\n")
                            worker.show_status.emit(user_info_title, user_info_text, 'error')
                else:
                    # USER INFO: Not synced damage potential layers
                    self.THREAD_PIPELINE = ""
                    user_info_title = self.tr("Fehler")
                    user_info_text = self.tr("\nBitte ausgewählte Schadenpotential-Layer mit EconoMe synchronisieren, um Risiken zu visualisieren!\n")
                    worker.show_status.emit(user_info_title, user_info_text, 'error')
            else:
                # USER INFO: Kein Schadenpotential ausgewählt
                self.THREAD_PIPELINE = ""
                user_info_title = self.tr("Fehler")
                user_info_text = self.tr("\nBitte Schadenpotential-Layer auswählen, um Risiken zu visualisieren!\n")
                worker.show_status.emit(user_info_title, user_info_text, 'error')
        else:
            # USER INFO: Ungültige Anmeldedaten
            self.THREAD_PIPELINE = ""
            user_info_title = self.tr("Fehler")
            user_info_text = self.tr("\nUngültige Anmeldedaten! (Projektnummer oder API-Keys)\n")
            worker.show_status.emit(user_info_title, user_info_text, 'error')
