from qgis.core import QgsProject, QgsSettings
from qgis.gui import QgsFloatingWidget, QgsLocatorWidget
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal, pyqtSlot
from qgis.PyQt.QtGui import QGuiApplication, QIcon
from qgis.PyQt.QtWidgets import QAction, QLineEdit, QToolBar
from qgis.utils import iface

from .locatorfilter import EasySearchFilter
from .menu import HistoryMenu, PatternMenu, SettingsMenu
from .switch import (
    AttributeDockSwitch,
    CaseSensitiveSwitch,
    InstantSearchSwitch,
    MapExtentSwitch,
)


class EasySearch(QToolBar):
    def __init__(self, parent=None):
        title = self.tr("Easy Search ToolBar")
        super().__init__(title, parent)
        self.setToolTip(self.tr("Easy Search"))
        self.cfg = QgsSettings()
        self.setup_ui()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("EasySearch", message)

    def setup_ui(self) -> None:
        self.search_box = QgsLocatorWidget(iface.mapCanvas())
        self.search_box.setMapCanvas(iface.mapCanvas())
        self.search_box.setResultContainerAnchors(
            # Added in QGIS 3.36.
            QgsFloatingWidget.TopLeft,  # relative position to the map canvas
            QgsFloatingWidget.BottomLeft,  # relative position to the search box
        )
        screen = QGuiApplication.primaryScreen()
        screen_width = screen.availableGeometry().width()
        self.search_box.setMaximumWidth(screen_width // 3)

        self.settings_menu = SettingsMenu()
        pattern_menu = PatternMenu()
        case_sensitive_switch = CaseSensitiveSwitch(self)
        map_extent_switch = MapExtentSwitch(self)
        instant_search_switch = InstantSearchSwitch(self)
        self.search_button = SearchButton(self.search_box)
        history_menu = HistoryMenu(self.search_box)
        self.attribute_dock = AttributeDockSwitch()

        search_filter = EasySearchFilter(history_menu)
        self.locator = self.search_box.locator()
        self.deregister_filters()
        self.locator.registerFilter(search_filter)
        search_filter.show_attribute.connect(self.attribute_dock.update)
        QgsProject.instance().cleared.connect(self.attribute_dock.update)
        QgsProject.instance().readProject.connect(
            self.settings_menu.update_on_read_project
        )

        self.settings_menu.conditions_changed.connect(self.re_search)
        pattern_menu.status_changed.connect(self.re_search)
        case_sensitive_switch.status_changed.connect(self.re_search)
        map_extent_switch.status_changed.connect(self.re_search)
        instant_search_switch.status_changed.connect(self.re_search)
        self.search_button.searched.connect(self.search)
        history_menu.history_selected.connect(self.search)

        self.addWidget(self.settings_menu)
        self.addWidget(pattern_menu)
        self.addAction(case_sensitive_switch)
        self.addAction(map_extent_switch)
        self.addAction(instant_search_switch)
        self.addSeparator()
        self.addWidget(self.search_box)
        self.addAction(self.search_button)
        self.addWidget(history_menu)
        self.addSeparator()
        self.addAction(self.attribute_dock)
        self.update_placeholder()

    def search(self, content: str) -> None:
        """called when search button clicked or search a history."""
        vlayer = self.settings_menu.layer_combobox.currentLayer()
        if not vlayer:
            return

        self.cfg.setValue("EasySearch/doSearch", True)
        self.locator.cancelWithoutBlocking()
        self.search_box.invalidateResults()
        self.search_box.search(content)

    def re_search(self) -> None:
        """called when search conditions changed."""
        self.update_placeholder()

        line_edit = self.search_box.findChild(QLineEdit)
        content = line_edit.text() if line_edit else ""
        if not content:
            return

        self.cfg.setValue("EasySearch/doSearch", True)
        self.search(content)

    def update_placeholder(self) -> None:
        """show usefull placeholder in searchbox.
        should be called on search conditions changed.
        """
        instant_search = self.cfg.value("EasySearch/instantSearch", False, type=bool)
        self.search_button.setVisible(not instant_search)

        search_layer = self.settings_menu.layer_combobox.currentLayer()
        if not search_layer:
            tip = self.tr("Select a layer to search")
            self.search_box.setPlaceholderText(tip)
            return

        max_len: int = 15
        layer_name = search_layer.name()
        if len(layer_name) > max_len:
            layer_name = layer_name[0:max_len] + "..."

        search_field = self.settings_menu.field_combobox.currentField()
        if search_field:
            if len(search_field) > max_len:
                fields = search_field[0:max_len] + "..."
            else:
                fields = search_field
        else:
            fields = self.tr("all")

        tip = self.tr("layer: {0}, field: {1}").format(layer_name, fields)
        self.search_box.setPlaceholderText(tip)

    def deregister_filters(self) -> None:
        """should be called before register filter or unload plugin."""
        for _filter in self.locator.filters():
            self.locator.deregisterFilter(_filter)

    def unload(self) -> None:
        """should be called when unload plugin."""
        self.deregister_filters()
        self.attribute_dock.unload()


class SearchButton(QAction):
    # this signal emitted on search button clicked
    searched = pyqtSignal(str)

    def __init__(self, searchBox, parent=None):
        super().__init__(parent)
        self.search_box = searchBox
        self.cfg = QgsSettings()
        icon = QIcon(":images/themes/default/console/iconSearchEditorConsole.svg")
        self.setIcon(icon)
        self.setText(self.tr("Search"))
        self.triggered.connect(self.on_search_triggered)

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("SearchButton", message)

    @pyqtSlot(bool)
    def on_search_triggered(self) -> None:
        line_edit = self.search_box.findChild(QLineEdit)
        content = line_edit.text() if line_edit else ""
        if content:
            self.searched.emit(content)
