from qgis.core import QgsFeature, QgsProject, QgsSettings
from qgis.gui import QgsAttributeForm, QgsDockWidget, QgsScrollArea, QgsAttributeEditorContext
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal, pyqtSlot
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.utils import iface

from ..compat import (
    DOCK_WIDGET_CLOSABLE,
    NO_DOCKWIDGET_AREA,
    NO_FRAME,
    RIGHT_DOCKWIDGET_AREA,
)


class CaseSensitiveSwitch(QAction):
    # this signal trigger update placeholder and re-search
    status_changed = pyqtSignal()

    def __init__(self, parent):
        super().__init__(parent)
        self.cfg = QgsSettings()
        icon = QIcon(":images/themes/default/mIconSearchCaseSensitive.svg")
        case_sensitive = self.cfg.value("EasySearch/caseSensitive", False, type=bool)
        self.setIcon(icon)
        self.setText(self.tr("Case Sensitive"))
        self.setCheckable(True)
        self.setChecked(case_sensitive)
        self.toggled.connect(self.on_toggled)

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("CaseSensitiveSwitch", message)

    @pyqtSlot(bool)
    def on_toggled(self, checked: bool) -> None:
        self.cfg.setValue("EasySearch/caseSensitive", checked)
        self.status_changed.emit()


class MapExtentSwitch(QAction):
    # this signal trigger update placeholder and re-search
    status_changed = pyqtSignal()

    def __init__(self, parent):
        super().__init__(parent)
        self.cfg = QgsSettings()
        icon = QIcon(":images/themes/default/propertyicons/overlay.svg")
        map_extent = self.cfg.value("EasySearch/mapExtent", False, type=bool)
        self.setIcon(icon)
        self.setText(self.tr("In Map Extent"))
        self.setCheckable(True)
        self.setChecked(map_extent)
        self.toggled.connect(self.on_toggled)

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("MapExtentSwitch", message)

    @pyqtSlot(bool)
    def on_toggled(self, checked: bool) -> None:
        self.cfg.setValue("EasySearch/mapExtent", checked)
        self.status_changed.emit()


class InstantSearchSwitch(QAction):
    # this signal trigger update placeholder and re-search
    status_changed = pyqtSignal()

    def __init__(self, parent):
        super().__init__(parent)
        self.cfg = QgsSettings()
        icon = QIcon(":images/themes/default/mActionShowAllLayers.svg")
        instant_search = self.cfg.value("EasySearch/instantSearch", False, type=bool)
        self.setIcon(icon)
        self.setText(self.tr("Instant Search"))
        self.setCheckable(True)
        self.setChecked(instant_search)
        self.toggled.connect(self.on_toggled)

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("InstantSearchSwitch", message)

    @pyqtSlot(bool)
    def on_toggled(self, checked: bool) -> None:
        self.cfg.setValue("EasySearch/instantSearch", checked)
        self.status_changed.emit()


class AttributeDockSwitch(QAction):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.cfg = QgsSettings()
        icon = QIcon(":images/themes/default/mActionOpenTableSelected.svg")
        self.setIcon(icon)
        self.setText(self.tr("Show Attributes"))
        self.setCheckable(True)
        self.toggled.connect(self.on_toggled)
        self.setup_ui()
        self.update()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("AttributeDockSwitch", message)

    def setup_ui(self) -> None:
        self.dock = QgsDockWidget(self.tr("Attributes"), iface.mainWindow())
        iface.addDockWidget(RIGHT_DOCKWIDGET_AREA, self.dock)
        self.dock.setAllowedAreas(NO_DOCKWIDGET_AREA)
        self.dock.setFeatures(self.dock.features() & ~DOCK_WIDGET_CLOSABLE)
        self.dock.setFloating(True)
        self.dock.resize(300, 500)
        # Set the dock to the bottom right corner as its initial position.
        dock_size = self.dock.size()
        main_window = iface.mainWindow()
        rect = main_window.rect()
        x = main_window.x() + rect.width() - dock_size.width() - 5
        y = main_window.y() + rect.height() - dock_size.height() - 35
        self.dock.move(x, y)

        self.scroll = QgsScrollArea()
        self.scroll.setWidgetResizable(True)
        self.scroll.setFrameShape(NO_FRAME)
        self.scroll.viewport().setContentsMargins(0, 0, 0, 0)
        self.scroll.setStyleSheet("""
            QScrollBar:vertical { width: 7px; }
            // QgsScrollArea { background-color: rgba(231, 247, 205, 150); }
            QgsScrollArea > QWidget > QWidget { background-color: transparent; }
        """)

    def update(self, feature: QgsFeature = None) -> None:
        is_visible = self.cfg.value("EasySearch/showAttributeDock", False, type=bool)
        self.dock.setUserVisible(is_visible)
        self.setChecked(is_visible)

        if not feature:
            self.dock.setUserVisible(False)
            return

        layer = QgsProject.instance().mapLayer(
            QgsSettings().value("EasySearch/searchLayer", None)
        )
        if not layer:
            return

        self.scroll.takeWidget()
        self.form = QgsAttributeForm(layer, feature)
        self.form.setMode(QgsAttributeEditorContext.SingleEditMode)
        self.scroll.setWidget(self.form)
        self.dock.setWidget(self.scroll)

    @pyqtSlot(bool)
    def on_toggled(self, checked: bool) -> None:
        self.cfg.setValue("EasySearch/showAttributeDock", checked)
        self.dock.setUserVisible(checked)

    def unload(self):
        iface.removeDockWidget(self.dock)
