from functools import partial

from qgis.core import QgsMapLayer, QgsMapLayerProxyModel, QgsProject, QgsSettings
from qgis.gui import QgsFieldComboBox, QgsMapLayerComboBox, QgsSpinBox
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal, pyqtSlot
from qgis.PyQt.QtGui import QFont, QIcon
from qgis.PyQt.QtWidgets import (
    QAction,
    QGridLayout,
    QLabel,
    QMenu,
    QToolButton,
    QWidget,
    QWidgetAction,
)

from ..compat import INSTANT_POPUP, PLUGIN_DIR


class SettingsMenu(QToolButton):
    # this signal trigger update placeholder and re-search
    conditions_changed = pyqtSignal()

    def __init__(self, parent=None):
        super().__init__(parent)
        self.cfg = QgsSettings()
        self.setup_ui()
        self.update()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("SettingsMenu", message)

    def setup_ui(self) -> None:
        self.setPopupMode(INSTANT_POPUP)
        self.setToolTip(self.tr("Settings Menu"))
        self.setIcon(QIcon(":images/themes/default/mActionReplace.svg"))

        widget = QWidget()
        layout = QGridLayout()
        widget.setLayout(layout)

        self.layer_combobox = QgsMapLayerComboBox()
        self.field_combobox = QgsFieldComboBox()
        self.descript_combobox = QgsFieldComboBox()
        self.layer_combobox.setFixedWidth(200)
        self.field_combobox.setFixedWidth(200)
        self.descript_combobox.setFixedWidth(200)
        self.layer_combobox.setFilters(QgsMapLayerProxyModel.VectorLayer)
        self.layer_combobox.setAllowEmptyLayer(True)
        self.field_combobox.setAllowEmptyFieldName(True)
        self.descript_combobox.setAllowEmptyFieldName(True)

        self.layer_combobox.layerChanged.connect(self.on_layer_changed)
        self.field_combobox.fieldChanged.connect(self.on_field_changed)
        self.descript_combobox.fieldChanged.connect(self.on_discript_changed)

        display_rows = self.cfg.value("/gui/locator/tree-height", 20, type=int)
        display_rows_spinbox = QgsSpinBox()
        display_rows_spinbox.setRange(5, 20)
        display_rows_spinbox.setSingleStep(5)
        display_rows_spinbox.setValue(display_rows)
        display_rows_spinbox.setShowClearButton(False)
        display_rows_spinbox.setToolTip(self.tr("Restart QGIS takes effect"))
        display_rows_spinbox.valueChanged.connect(self.one_display_rows_changed)

        layout.addWidget(QLabel(self.tr("Search layer")), 0, 0)
        layout.addWidget(self.layer_combobox, 0, 1)
        layout.addWidget(QLabel(self.tr("Search field")), 1, 0)
        layout.addWidget(self.field_combobox, 1, 1)
        layout.addWidget(QLabel(self.tr("Descript field")), 2, 0)
        layout.addWidget(self.descript_combobox, 2, 1)
        layout.addWidget(QLabel(self.tr("Display rows")), 3, 0)
        layout.addWidget(display_rows_spinbox, 3, 1)

        menu = QMenu()
        action = QWidgetAction(menu)
        action.setDefaultWidget(widget)
        menu.addAction(action)
        self.setMenu(menu)

    def update(self) -> None:
        layer_id = self.cfg.value("EasySearch/searchLayer", None)
        search_layer = QgsProject.instance().mapLayer(layer_id)
        self.descript_combobox.setLayer(search_layer)
        self.field_combobox.setLayer(search_layer)
        self.layer_combobox.setLayer(search_layer)
        self.update_fields()

    def update_on_read_project(self) -> None:
        layer_id = self.cfg.value("EasySearch/lastLayer", None)
        last_layer = QgsProject.instance().mapLayer(layer_id)
        if not last_layer:
            return

        self.layer_combobox.setLayer(last_layer)

    def update_fields(self) -> None:
        """to remove fid field. should be called when search layer changed."""
        fields = self.field_combobox.fields()
        idx = fields.indexFromName("fid")
        if idx != -1:
            fields.remove(idx)
        self.field_combobox.setFields(fields)

        search_field = self.cfg.value("EasySearch/searchField", None, type=str)
        self.field_combobox.setField(search_field)
        self.update_descript_fields()

    def update_descript_fields(self) -> None:
        """remove fid and search field.
        should be called when search layer/field changed.
        """
        search_field = self.cfg.value("EasySearch/searchField", None, type=str)
        fields = self.field_combobox.fields()
        for field in ["fid", search_field]:
            idx = fields.indexFromName(field)
            if idx != -1:
                fields.remove(idx)
        self.descript_combobox.setFields(fields)

        descript_field = self.cfg.value("EasySearch/descriptField", None, type=str)
        self.descript_combobox.setField(descript_field)

    @pyqtSlot("QgsMapLayer*")
    def on_layer_changed(self, layer: QgsMapLayer) -> None:
        # save last layer for reopen this project
        if layer:
            self.cfg.setValue("EasySearch/lastLayer", layer.id())
            self.cfg.setValue("EasySearch/searchLayer", layer.id())
        else:
            layer_id = self.cfg.value("EasySearch/searchLayer", None)
            self.cfg.setValue("EasySearch/lastLayer", layer_id)

        self.field_combobox.setLayer(layer)
        self.descript_combobox.setLayer(layer)
        self.update_fields()
        self.conditions_changed.emit()

    @pyqtSlot(str)
    def on_field_changed(self, field: str) -> None:
        self.cfg.setValue("EasySearch/searchField", field)
        self.update_descript_fields()
        self.conditions_changed.emit()

    @pyqtSlot(str)
    def on_discript_changed(self, field: str) -> None:
        self.cfg.setValue("EasySearch/descriptField", field)
        self.conditions_changed.emit()

    @pyqtSlot(int)
    def one_display_rows_changed(self, value: int) -> None:
        self.cfg.setValue("/gui/locator/tree-height", value)


class PatternMenu(QToolButton):
    # this signal trigger update placeholder and re-search
    status_changed = pyqtSignal()

    def __init__(self, parent=None):
        super().__init__(parent)
        self.cfg = QgsSettings()
        self.setup_ui()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("PatternMenu", message)

    def setup_ui(self) -> None:
        contains_icon = QIcon(":images/themes/default/mIconAlignCenter.svg")
        equals_icon = QIcon(f"{PLUGIN_DIR.joinpath('images/equal-line.svg')}")
        starts_icon = QIcon(":images/themes/default/mIconAlignLeft.svg")
        ends_icon = QIcon(":images/themes/default/mIconAlignRight.svg")
        regexp_icon = QIcon(":images/themes/default/mIconSearchRegex.svg")

        self.contains = QAction(contains_icon, self.tr("Contains"))
        self.equals = QAction(equals_icon, self.tr("Equals"))
        self.starts = QAction(starts_icon, self.tr("Starts"))
        self.ends = QAction(ends_icon, self.tr("Ends"))
        self.regexp = QAction(regexp_icon, self.tr("RegExp"))

        self.contains.triggered.connect(self.on_contains_triggered)
        self.equals.triggered.connect(self.on_equals_triggered)
        self.starts.triggered.connect(self.on_starts_triggered)
        self.ends.triggered.connect(self.on_ends_triggered)
        self.regexp.triggered.connect(self.on_regexp_triggered)

        menu = QMenu()
        menu.addAction(self.contains)
        menu.addAction(self.equals)
        menu.addAction(self.starts)
        menu.addAction(self.ends)
        menu.addAction(self.regexp)

        self.setMenu(menu)
        self.setPopupMode(INSTANT_POPUP)
        self.setAutoRaise(True)
        self.setToolTip(self.tr("Search pattern"))
        self.update_default_action()

    def update_default_action(self) -> None:
        _pattern = self.cfg.value("EasySearch/searchPattern", "contains")
        match _pattern:
            case "contains":
                self.setDefaultAction(self.contains)
            case "equals":
                self.setDefaultAction(self.equals)
            case "starts":
                self.setDefaultAction(self.starts)
            case "ends":
                self.setDefaultAction(self.ends)
            case "regexp":
                self.setDefaultAction(self.regexp)

    @pyqtSlot(bool)
    def on_contains_triggered(self) -> None:
        self.cfg.setValue("EasySearch/searchPattern", "contains")
        self.setDefaultAction(self.contains)
        self.status_changed.emit()

    @pyqtSlot(bool)
    def on_equals_triggered(self) -> None:
        self.cfg.setValue("EasySearch/searchPattern", "equals")
        self.setDefaultAction(self.equals)
        self.status_changed.emit()

    @pyqtSlot(bool)
    def on_starts_triggered(self) -> None:
        self.cfg.setValue("EasySearch/searchPattern", "starts")
        self.setDefaultAction(self.starts)
        self.status_changed.emit()

    @pyqtSlot(bool)
    def on_ends_triggered(self) -> None:
        self.cfg.setValue("EasySearch/searchPattern", "ends")
        self.setDefaultAction(self.ends)
        self.status_changed.emit()

    @pyqtSlot(bool)
    def on_regexp_triggered(self) -> None:
        self.cfg.setValue("EasySearch/searchPattern", "regexp")
        self.setDefaultAction(self.regexp)
        self.status_changed.emit()


class HistoryMenu(QToolButton):
    # this signal trigger search a history
    history_selected = pyqtSignal(str)

    def __init__(self, parent=None):
        super().__init__(parent)
        self.cfg = QgsSettings()
        self.setup_ui()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("HistoryMenu", message)

    def setup_ui(self) -> None:
        self.setPopupMode(INSTANT_POPUP)
        self.setAutoRaise(True)
        self.setToolTip(self.tr("Search history"))
        self.setIcon(QIcon(":images/themes/default/mIconHistory.svg"))

        self.menu = QMenu()
        self.setMenu(self.menu)
        self.update()

    def update(self) -> None:
        histories: list[str] = self.cfg.value("EasySearch/histories", [], type=list)
        if not histories:
            return

        self.menu.clear()
        for history in histories[::-1]:
            action = QAction(history, self.menu)
            action.triggered.connect(partial(self.on_history_selected, history))
            self.menu.addAction(action)

        font = QFont()
        font.setBold(True)
        icon = QIcon(":images/themes/default/console/iconClearConsole.svg")
        clear_action = QAction(icon, self.tr("Clear histories"), self.menu)
        clear_action.triggered.connect(self.clear)
        clear_action.setFont(font)
        self.menu.addSeparator()
        self.menu.addAction(clear_action)

    @pyqtSlot(bool)
    def on_history_selected(self, history: str) -> None:
        self.history_selected.emit(history)

    @pyqtSlot()
    def clear(self) -> None:
        self.menu.clear()
        self.cfg.remove("EasySearch/histories")
