from qgis.core import (
    QgsFeature,
    QgsFeatureRequest,
    QgsFeedback,
    QgsLocatorContext,
    QgsLocatorFilter,
    QgsLocatorResult,
    QgsMapLayer,
    QgsProject,
    QgsSettings,
)
from qgis.gui import QgsMapCanvas
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal
from qgis.utils import iface


class EasySearchFilter(QgsLocatorFilter):
    # this signal trigger show attribute dock
    show_attribute = pyqtSignal(QgsFeature)

    def __init__(self, history_menu=None):
        """history_menu: use to update history menu when do a search"""
        super().__init__()
        self.history_menu = history_menu
        self.cfg = QgsSettings()
        self._auto_set_fetch_results_delay()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("EasySearchFilter", message)

    def clone(self):
        return EasySearchFilter(self.history_menu)

    def name(self) -> str:
        return "EasySearchFilter"

    def displayName(self) -> str:
        return self.tr("Easy Search")

    def prefix(self) -> str:
        # prefix must >= 3 characters
        return "esf"

    def prepare(self, search: str, context: QgsLocatorContext) -> list:
        """searchbox foucs or each input character will tigger this method"""
        return []

    def fetchResults(
        self, search: str, context: QgsLocatorContext, feedback: QgsFeedback
    ) -> None:
        """searchbox focused or changed will tigger this method"""
        self._check_conditions(search)
        self._update_histories()
        self._bulid_feature_request()
        self._search_features(feedback)
        self.cfg.setValue("EasySearch/doSearch", False)
        self._auto_set_fetch_results_delay()

    def triggerResult(self, result) -> None:
        """tiggered by double click result."""
        self._zoomTo(result)

    def resultSelected(self, result) -> None:
        """tiggered by click result. Added in QGIS 3.40"""
        self._zoomTo(result)

    def _zoomTo(self, result) -> None:
        """zooms result to current scale"""
        layer =  QgsProject.instance().mapLayer(
            QgsSettings().value("EasySearch/searchLayer", None)
        )
        feature = result.userData
        canvas: QgsMapCanvas = iface.mapCanvas()
        scale = canvas.scale()
        canvas.zoomToFeatureIds(layer, [feature.id()])
        canvas.zoomScale(scale)
        canvas.flashFeatureIds(layer, [feature.id()])
        self.show_attribute.emit(feature)

    def _check_conditions(self, search: str) -> None:
        """check if conditions can be searched."""
        self.is_canceled: bool = False
        self.search_str: str = ""
        self.search_layer: QgsMapLayer = None
        self.search_fields: list[str] = []

        # skip focus search box
        if len(search) == 0:
            self.is_canceled = True
            return

        # allow search blank string
        if search.isspace():
            self.search_str = search
        else:
            self.search_str = search.strip()

        self.search_pattern = self.cfg.value("EasySearch/searchPattern", "contains")
        self.search_layer = QgsProject.instance().mapLayer(
            QgsSettings().value("EasySearch/searchLayer", None)
        )
        if not self.search_layer:
            self.is_canceled = True
            return

        if not self.search_layer.fields().names():
            self.is_canceled = True
            return

        field = self.cfg.value("EasySearch/searchField", None)
        if field:
            self.search_fields = [field]
        else:
            fields = self.search_layer.fields()
            self.search_fields = [f.name() for f in fields if f.name() != "fid"]

        doSearch = self.cfg.value("EasySearch/doSearch", False, type=bool)
        if doSearch:
            return

        instant_search = self.cfg.value("EasySearch/instantSearch", False, type=bool)
        if not instant_search:
            self.is_canceled = True
            return

    def _update_histories(self) -> None:
        if self.is_canceled:
            return
        histories = self.cfg.value("EasySearch/histories", [], type=list)
        # Do not record existing and blank string
        if (self.search_str not in histories) and (not self.search_str.isspace()):
            histories.append(self.search_str)
            histories = histories[-10:]
            self.cfg.setValue("EasySearch/histories", histories)
            self.history_menu.update()

    def _bulid_feature_request(self) -> None:
        if self.is_canceled:
            return

        caseSensitive = self.cfg.value("EasySearch/caseSensitive", False, type=bool)
        op = "LIKE" if caseSensitive else "ILIKE"
        fields = self.search_fields
        if self.search_pattern == "regexp":
            s = self.search_str
        else:
            s = self.search_str.replace("%", r"\\%")
        match self.search_pattern:
            case "contains":
                expr = " OR ".join([f""""{f}" {op} '%{s}%'""" for f in fields])
            case "equals":
                expr = " OR ".join([f""""{f}" {op} '{s}'""" for f in fields])
            case "starts":
                expr = " OR ".join([f""""{f}" {op} '{s}%'""" for f in fields])
            case "ends":
                expr = " OR ".join([f""""{f}" {op} '%{s}'""" for f in fields])
            case "regexp":
                expr = " OR ".join([f""""{f}" ~ '{s}'""" for f in fields])

        self.request = QgsFeatureRequest().setFilterExpression(expr)
        self.request.setFlags(QgsFeatureRequest.NoGeometry)
        map_extent = self.cfg.value("EasySearch/mapExtent", False, type=bool)
        if map_extent:
            self.request.setFilterRect(iface.mapCanvas().extent())

    def _search_features(self, feedback: QgsFeedback) -> None:
        if self.is_canceled:
            return

        display_field: str = self.search_fields[0]
        descript_field = self.cfg.value("EasySearch/descriptField", None)
        fields_count: int = len(self.search_fields)
        layer_fields = self.search_layer.fields()

        for feat in self.search_layer.getFeatures(self.request):
            if feedback.isCanceled():
                return

            result = QgsLocatorResult()
            result.filter = self
            if fields_count == 1:
                result.displayString = str(feat[display_field])
            else:
                result.displayString = f"fid: {feat.id()}"
            if layer_fields.lookupField(descript_field) != -1:
                result.description = str(feat[descript_field])
            else:
                pass
            result.userData = feat
            result.score = 1.0
            self.resultFetched.emit(result)

    def _auto_set_fetch_results_delay(self) -> None:
        instant_search = self.cfg.value("EasySearch/instantSearch", False, type=bool)
        if instant_search:
            self.setFetchResultsDelay(500)
        else:
            self.setFetchResultsDelay(50)
