# -*- coding: utf-8 -*-
"""
/***************************************************************************
 EasyPathDialog
                                 A QGIS plugin
 EasyPath
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-03-25
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Unicampania
        email                : email@prova.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import heapq
import csv

# PyQt5
from PyQt5.QtCore import Qt, QTimer, QVariant, QSettings, QSizeF, QPointF, QRectF
from PyQt5.QtWidgets import QMessageBox, QTableWidgetItem
from PyQt5.QtGui import QColor, QTextDocument, QFont

# QGIS UI
from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtXml import QDomDocument
from qgis.gui import QgsMapCanvas
from qgis.utils import iface

# QGIS Core
from qgis.core import (
    QgsTextAnnotation,
    QgsProject,
    QgsVectorLayer,
    QgsFeature,
    QgsField,
    QgsFields,
    QgsWkbTypes,
    QgsFeatureRequest,
    QgsGeometry,
    QgsPointXY,
    QgsVectorFileWriter,
    QgsVectorLayerUtils,
    QgsRendererRange,
    QgsSymbol,
    QgsGraduatedSymbolRenderer,
    QgsReadWriteContext,
    QgsLayout,
    QgsPrintLayout,
    QgsLayoutItemPage,
    QgsLayoutItemMap,
    QgsLayoutItemLabel,
    QgsLayoutItemLegend,
    QgsLayoutItemScaleBar,
    QgsLayoutItemPicture,
    QgsLayoutExporter,
    QgsLayoutSize,
    QgsLayoutPoint,
    QgsRectangle,
    QgsUnitTypes,
    QgsMargins,
    QgsUnitTypes,
    QgsLayerTreeGroup,
    QgsCoordinateTransform
)

# TIPOLOGIE DI GSI E PARAMETRI MINIMI
GSI_DEFINITIONS = {
    "Planter": {"width": 1.2, "length_min": 1, "length_max": 3, "with_trees": False},
    "Gutter": {"width": 0.5, "length_min": 1, "length_max": 3, "with_trees": False},
    "Swale": {"width": 2.0, "length_min": 2, "length_max": 4, "with_trees": True},
    "Tree Trench": {"width": 1.5, "length_min": 1.5, "length_max": 1.5, "with_trees": True},
    "Rain Garden": {"width": 8.0, "length_min": 5, "length_max": 5, "with_trees": True}
}

# DEFINIZIONI DIAMETRO CORONA (m) E TEMPO (t)
CROWN_DIAMETER_MAPPING = {
    1: 0,
    2: 3,
    4: 10,
    6: 15,
    8: 16  # consideriamo t > 15 come 16 per i calcoli
}

# SPECIFICHE ARREDI URBANI (ingombro: L x P in metri, distanza minima tra elementi in metri)
FURNITURE_SPECS = {
    "bench": {          # panchine
        "length_m": 1.8,
        "depth_m": 0.6,
        "min_spacing_m": 50.0
    },
    "waste_bin": {      # cestini
        "length_m": 0.3,
        "depth_m": 0.3,
        "min_spacing_m": 50.0
    },
    "fountain": {       # fontane
        "length_m": 0.8,
        "depth_m": 1.2,
        "min_spacing_m": 200.0
    },
    "bike_rack": {      # portabici
        "length_m": 1.5,
        "depth_m": 1.5,
        "min_spacing_m": 200.0
    },
}

# --- Larghezze standard pista ciclabile (m)
CYCLE_SINGLE_W = 1.0   # senso unico
CYCLE_DOUBLE_W = 2.5   # doppio senso


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'EasyPath_dialog_base.ui'))

class EasyPathDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(EasyPathDialog, self).__init__(parent)

        self.setupUi(self)

        # Spegnimento scheda output fase 3 all'avvio
        for i in range(self.Accessibility_tabWidget.count()):
            if self.Accessibility_tabWidget.widget(i).objectName() == "ParametricSimulationPath_OUTPUT":
                self._hidden_tab_index = i
                self._hidden_tab_widget = self.Accessibility_tabWidget.widget(i)
                self._hidden_tab_label = self.Accessibility_tabWidget.tabText(i)
                self.Accessibility_tabWidget.removeTab(i)
                break

        # Spegnimento tabelle output fase 3 all'avvio
        self.ScenariosEnvironmentalResults_tableWidget.setVisible(False)
        self.ScenariosPhysicalResults_tableWidget.setVisible(False)
        self.ScenariosHybridResults_tableWidget.setVisible(False)

        # Creazione mappa in plugin
        self.mini_canvas = QgsMapCanvas(self)
        self.mini_canvas.setCanvasColor(Qt.white)

        self.mini_canvas.setProject(QgsProject.instance())

        visible_layers = []
        root = QgsProject.instance().layerTreeRoot()

        for node in root.findLayers():
            if node.isVisible():
                layer = node.layer()
                if layer and layer.isValid():
                    visible_layers.append(layer)

        self.mini_canvas.setLayers(visible_layers)

        # 📍 Copia estensione della mappa principale
        main_canvas = iface.mapCanvas()
        self.mini_canvas.setDestinationCrs(main_canvas.mapSettings().destinationCrs())
        self.mini_canvas.setExtent(main_canvas.extent())
        self.mini_canvas.refresh()
        # Rende la mini mappa reattiva a cambiamenti nella visibilità dei layer
        QgsProject.instance().layerTreeRoot().visibilityChanged.connect(self.refresh_mini_canvas)
        QgsProject.instance().layerWasAdded.connect(lambda _: self.refresh_mini_canvas())
        QgsProject.instance().layerWillBeRemoved.connect(lambda _: self.refresh_mini_canvas())

        # Layout
        layout = QtWidgets.QVBoxLayout(self.miniMapWidget)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.addWidget(self.mini_canvas)

        # Pulisce la comboBox e aggiunge un elemento placeholder
        self.PathLayer_comboBox.clear()
        self.PathLayer_comboBox.addItem("Select layer", None)

        # Popola la comboBox con i layer del progetto
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            self.PathLayer_comboBox.addItem(layer.name(), layer.id())

        # Aggiorna attributi in combobox
        self.PathLayer_comboBox.currentIndexChanged.connect(self.update_attribute_comboboxes)

        # Popola PathLayer_comboBox_2 e NodeLayer_comboBox
        self.NodeLayer_comboBox.clear()
        self.NodeLayer_comboBox.addItem("Select Layer", None)

        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            self.NodeLayer_comboBox.addItem(layer.name(), layer.id())

        self.NodeLayer_comboBox.currentIndexChanged.connect(self.update_node_attribute_comboboxes)

        self.NodeID_comboBox.currentIndexChanged.connect(self.populate_start_and_arrival_nodes)
        self.NodeType_comboBox.currentIndexChanged.connect(self.populate_start_and_arrival_nodes)

        # Popola PhysicalAccessibilityLayer_comboBox
        self.PhysicalAccessibilityLayer_comboBox.clear()
        self.PhysicalAccessibilityLayer_comboBox.addItem("Select Layer", None)

        for layer in layers:
            self.PhysicalAccessibilityLayer_comboBox.addItem(layer.name(), layer.id())

        QgsProject.instance().layerWasAdded.connect(self.update_physical_accessibility_layer_combobox)
        QgsProject.instance().layerWillBeRemoved.connect(self.update_physical_accessibility_layer_combobox)

        # Popola EnvironmentalAccessibilityLayer_comboBox
        self.EnvironmentalAccessibilityLayer_comboBox.clear()
        self.EnvironmentalAccessibilityLayer_comboBox.addItem("Select Layer", None)

        for layer in layers:
            self.EnvironmentalAccessibilityLayer_comboBox.addItem(layer.name(), layer.id())

        QgsProject.instance().layerWasAdded.connect(self.update_environmental_accessibility_layer_combobox)
        QgsProject.instance().layerWillBeRemoved.connect(self.update_environmental_accessibility_layer_combobox)

        # Salva Stato Principale
        self.SaveMainState_pushButton.clicked.connect(self.save_main_state)
        self.restore_main_state()

        # Salva Stato Accessibilità fisica
        self.SaveStatePhysicalAccessibility_pushButton.clicked.connect(self.save_physical_accessibility_state)
        self.restore_physical_accessibility_state()

        # Salva Stato Accessibilità ambientale
        self.SaveState_pushButton_2.clicked.connect(self.save_environmental_accessibility_state)
        self.restore_environmental_accessibility_state()

        # Salva Stato Scheda 2
        self.SaveLayerPathState_pushButton.clicked.connect(self.save_path_layer_state)
        self.restore_path_layer_state()

        # Salva Stato scheda 3
        self.SaveLayerScenarioState_pushButton.clicked.connect(self.save_scenario_state)

        # Ripristino stato scenari
        self.restore_scenario_state()

        # Annotazioni
        self.route_annotations = {}

        # Rimozione annotazioni
        QgsProject.instance().layerWillBeRemoved.connect(self.remove_route_annotations)

        # Avvio calcolo accessibilità fisica
        self.CalculateAccessibilityGrade_pushButton.clicked.connect(self.calculate_accessibility_grade)

        # Avvio calcolo accessibilità ambientale
        self.CalculateAccessibilityGrade_pushButton.clicked.connect(self.calculate_environmental_accessibility_grade)

        # Avvio calcolo per generazione percorso ottimale
        self.GeneratePath_pushButton.clicked.connect(self.generate_optimal_path)

        # Refresh minimappa
        self.Refresh_pushButton.clicked.connect(self.refresh_mini_canvas)
        self.ShowPhysicalLayer_checkBox.stateChanged.connect(self.toggle_physical_layer_visibility)
        self.ShowEnvironmentalLayer_checkBox.stateChanged.connect(self.toggle_environmental_layer_visibility)
        self.ShowPathLayer_checkBox.stateChanged.connect(self.toggle_path_layer_visibility)

        # Imposta lo snapping degli slider ai soli valori -5, 0, 5
        self.Accessibility_horizontalSlider.valueChanged.connect(self.snap_slider_to_step)
        self.TimeAccessBalance_horizontalSlider.valueChanged.connect(self.snap_slider_to_step)

        # ************************************************************ __Init__ PER SCENARI ************************************************************ #
        # Popola ChosenPathLayer_comboBox
        self.ChosenPathLayer_comboBox.clear()
        self.ChosenPathLayer_comboBox.addItem("Select Layer", None)

        for layer in QgsProject.instance().mapLayers().values():
            self.ChosenPathLayer_comboBox.addItem(layer.name(), layer.id())
        # Aggiorna automaticamente quando viene aggiunto un nuovo layer
        QgsProject.instance().layerWasAdded.connect(self.update_chosen_path_layer_combobox)

        # Avvio calcolo scenario minimo
        self.GenerateScenarios_pushButton.clicked.connect(self.calculate_minimum_scenario)

        # Export indicatori (CSV) dal pulsante "CreateScenarioLayer"
        self.CreateScenarioLayer_pushButton.clicked.connect(self.export_indicators_csv)
        # Export layout A3 con planimetria e istogrammi
        self.CreateScenarioLayer_pushButton.clicked.connect(self.export_a3_layout_with_histograms)

        # assicura che la combo sia popolata/subito selezionata se esiste un Path
        self.update_chosen_path_layer_combobox()

    def snap_slider_to_step(self, val):
        step = 5
        nearest = round(val / step) * step
        sender = self.sender()
        if sender and sender.value() != nearest:
            sender.setValue(nearest)

    def update_attribute_comboboxes(self):
        # Ottiene il layer selezionato
        layer_id = self.PathLayer_comboBox.currentData()
        layer = QgsProject.instance().mapLayer(layer_id)

        # Controlla se è un layer vettoriale valido
        if not isinstance(layer, QgsVectorLayer):
            return

        # Ottieni la lista dei nomi degli attributi
        field_names = [field.name() for field in layer.fields()]

        # Riempie ogni combobox con questi attributi
        def fill_combo(combo):
            combo.clear()
            combo.addItem("Select attribute", None)
            for name in field_names:
                combo.addItem(name, name)

        # MAIN
        fill_combo(self.ID_comboBox)
        fill_combo(self.PathLength_comboBox)
        fill_combo(self.PathWidth_comboBox)
        fill_combo(self.PathName_comboBox)

        fill_combo(self.TreeNumberAB_comboBox)
        fill_combo(self.TreeNumberBA_comboBox)

        # ATTRIBUTI PERCORSO PER GENERAZIONE
        fill_combo(self.APathNode_comboBox)
        fill_combo(self.BPathNode_comboBox)
        fill_combo(self.PathDirection_comboBox)

        # PHYSICAL ACCESSIBILITY
        # Sidewalk
        fill_combo(self.WidthAB_comboBox)
        fill_combo(self.SlopeLongitudinal_comboBox)
        fill_combo(self.SlopeDownhillRoad_comboBox)
        fill_combo(self.WidthBA_comboBox)
        fill_combo(self.SlopeTransversal_comboBox)
        # Parking
        fill_combo(self.ParkingWidth_comboBox)
        fill_combo(self.ParkingNumber_comboBox)
        # Flooring
        fill_combo(self.FlooringAntiSlip_comboBox)
        # Bicycle Paths
        fill_combo(self.BicyclePathsLength_comboBox)
        # Public Transport
        fill_combo(self.PublicTransportStopsNumber_comboBox)
        # Street Benches
        fill_combo(self.StreetBenchesNumber_comboBox)
        # Waste Bins
        fill_combo(self.WasteBinsNumber_comboBox)
        # Drinking Water Fountains
        fill_combo(self.DrinkingWaterFountainsNumber_comboBox)
        # Bike Carrier
        fill_combo(self.BikeCarrierNumber_comboBox)
        # Crossing
        fill_combo(self.CrossingNumber_comboBox)

        # ENVIRONMENTAL ACCESSIBILITY
        # Microclimate
        fill_combo(self.MicroclimatePET9_comboBox)
        fill_combo(self.MicroclimatePET12_comboBox)
        fill_combo(self.MicroclimatePET15_comboBox)
        fill_combo(self.MicroclimateWind9_comboBox)
        fill_combo(self.MicroclimateWind12_comboBox)
        fill_combo(self.MicroclimateWind15_comboBox)
        fill_combo(self.MicroclimateShadow9_comboBox)
        fill_combo(self.MicroclimateShadow12_comboBox)
        fill_combo(self.MicroclimateShadow15_comboBox)
        fill_combo(self.MicroclimateSkyViewFactor_comboBox)
        fill_combo(self.MicroclimateTreeCover_comboBox)
        # Water
        fill_combo(self.WaterFloodingRisk_comboBox)
        fill_combo(self.WaterStormwaterVolume_comboBox)

        # RESIZE TABELLE
        self.ScenarioMinimoResults_tableWidget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        self.ScenarioMinimoResults_tableWidget.horizontalHeader().setStretchLastSection(False)

        self.ScenariosEnvironmentalResults_tableWidget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        self.ScenariosEnvironmentalResults_tableWidget.horizontalHeader().setStretchLastSection(False)

        self.ScenariosPhysicalResults_tableWidget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        self.ScenariosPhysicalResults_tableWidget.horizontalHeader().setStretchLastSection(False)

        self.ScenariosHybridResults_tableWidget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        self.ScenariosHybridResults_tableWidget.horizontalHeader().setStretchLastSection(False)

    def update_node_attribute_comboboxes(self):
        layer_id = self.NodeLayer_comboBox.currentData()
        layer = QgsProject.instance().mapLayer(layer_id)

        if not isinstance(layer, QgsVectorLayer):
            return

        field_names = [field.name() for field in layer.fields()]

        def fill_combo(combo):
            combo.clear()
            combo.addItem("Select attribute", None)
            for name in field_names:
                combo.addItem(name, name)

        fill_combo(self.NodeID_comboBox)
        fill_combo(self.NodeType_comboBox)

    def update_physical_accessibility_layer_combobox(self):
        current_value = self.PhysicalAccessibilityLayer_comboBox.currentData()

        self.PhysicalAccessibilityLayer_comboBox.clear()
        self.PhysicalAccessibilityLayer_comboBox.addItem("Select layer", None)

        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            self.PhysicalAccessibilityLayer_comboBox.addItem(layer.name(), layer.id())

        # Tenta di ripristinare il valore selezionato prima dell’aggiornamento
        if current_value:
            index = self.PhysicalAccessibilityLayer_comboBox.findData(current_value)
            if index >= 0:
                self.PhysicalAccessibilityLayer_comboBox.setCurrentIndex(index)

    def update_environmental_accessibility_layer_combobox(self):
        current_value = self.EnvironmentalAccessibilityLayer_comboBox.currentData()

        self.EnvironmentalAccessibilityLayer_comboBox.clear()
        self.EnvironmentalAccessibilityLayer_comboBox.addItem("Select layer", None)

        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            self.EnvironmentalAccessibilityLayer_comboBox.addItem(layer.name(), layer.id())

        # Ripristina valore selezionato
        if current_value:
            index = self.EnvironmentalAccessibilityLayer_comboBox.findData(current_value)
            if index >= 0:
                self.EnvironmentalAccessibilityLayer_comboBox.setCurrentIndex(index)

    def populate_start_and_arrival_nodes(self):
        layer_id = self.NodeLayer_comboBox.currentData()
        layer = QgsProject.instance().mapLayer(layer_id)

        if not isinstance(layer, QgsVectorLayer):
            return

        node_id_field = self.NodeID_comboBox.currentData()
        node_type_field = self.NodeType_comboBox.currentData()
        if not node_id_field or not node_type_field:
            return

        starting_nodes = set()
        arrival_nodes = set()

        for feature in layer.getFeatures():
            node_id = feature[node_id_field]
            node_type = feature[node_type_field]

            if node_id in (None, '') or node_type in (None, ''):
                continue

            node_id_str = str(node_id)
            node_type_str = str(node_type)

            if node_type_str == "Incrocio":
                starting_nodes.add(node_id_str)
            elif node_type_str == "Attrezzatura":
                arrival_nodes.add(node_id_str)

        starting_nodes_sorted = sorted(starting_nodes)
        arrival_nodes_sorted = sorted(arrival_nodes)

        def fill_combo(combo, values):
            combo.clear()
            combo.addItem("Select node", None)
            for val in values:
                combo.addItem(val, val)

        fill_combo(self.StartingNode_comboBox, starting_nodes_sorted)
        fill_combo(self.ArrivalNode_comboBox, arrival_nodes_sorted)

    # **************************** SALVATAGGIO MAIN STATE **************************** #
    def save_main_state(self):
        path_layer_id = self.PathLayer_comboBox.currentData()
        id_field = self.ID_comboBox.currentData()
        path_length_field = self.PathLength_comboBox.currentData()
        path_width_field = self.PathWidth_comboBox.currentData()
        path_name_field = self.PathName_comboBox.currentData()

        settings = QSettings()
        settings.setValue("EasyPath/layer_id", path_layer_id)
        settings.setValue("EasyPath/id_field", id_field)
        settings.setValue("EasyPath/path_length_field", path_length_field)
        settings.setValue("EasyPath/path_width_field", path_width_field)
        settings.setValue("EasyPath/path_name_field", path_name_field)

        QMessageBox.information(self, "Status saved", "Status saved successfully!", QMessageBox.Ok)

    def restore_main_state(self):
        settings = QSettings()
        saved_layer_id = settings.value("EasyPath/layer_id", None)
        saved_id_field = settings.value("EasyPath/id_field", None)
        saved_path_length = settings.value("EasyPath/path_length_field", None)
        saved_path_width = settings.value("EasyPath/path_width_field", None)
        saved_path_name = settings.value("EasyPath/path_name_field", None)

        if saved_layer_id:
            index = self.PathLayer_comboBox.findData(saved_layer_id)
            if index >= 0:
                self.PathLayer_comboBox.setCurrentIndex(index)

        # Gli altri campi verranno aggiornati automaticamente da update_attribute_comboboxes
        # Ma possiamo applicare anche questi dopo un piccolo delay/evento
        def select_fields_later():
            index_id = self.ID_comboBox.findData(saved_id_field)
            if index_id >= 0:
                self.ID_comboBox.setCurrentIndex(index_id)

            index_length = self.PathLength_comboBox.findData(saved_path_length)
            if index_length >= 0:
                self.PathLength_comboBox.setCurrentIndex(index_length)

            index_width = self.PathWidth_comboBox.findData(saved_path_width)
            if index_width >= 0:
                self.PathWidth_comboBox.setCurrentIndex(index_width)

            index_name = self.PathName_comboBox.findData(saved_path_name)
            if index_name >= 0:
                self.PathName_comboBox.setCurrentIndex(index_name)

        # Aspetta che i layer siano caricati prima di impostare
        QTimer.singleShot(100, select_fields_later)

    # **************************** SALVATAGGIO PHYSICAL ACCESSIBILITY STATE **************************** #
    def save_physical_accessibility_state(self):
        settings = QSettings()

        # Mappa delle comboBox da salvare: chiave settings -> oggetto combo
        combos = {
            "WidthAB": self.WidthAB_comboBox,
            "SlopeLongitudinal": self.SlopeLongitudinal_comboBox,
            "SlopeDownhillRoad": self.SlopeDownhillRoad_comboBox,
            "WidthBA": self.WidthBA_comboBox,
            "SlopeTransversal": self.SlopeTransversal_comboBox,
            "ParkingWidth": self.ParkingWidth_comboBox,
            "ParkingNumber": self.ParkingNumber_comboBox,
            "FlooringAntiSlip": self.FlooringAntiSlip_comboBox,
            "BicyclePathsLength": self.BicyclePathsLength_comboBox,
            "PublicTransportStopsNumber": self.PublicTransportStopsNumber_comboBox,
            "StreetBenchesNumber": self.StreetBenchesNumber_comboBox,
            "WasteBinsNumber": self.WasteBinsNumber_comboBox,
            "DrinkingWaterFountainsNumber": self.DrinkingWaterFountainsNumber_comboBox,
            "BikeCarrierNumber": self.BikeCarrierNumber_comboBox,
            "CrossingNumber": self.CrossingNumber_comboBox  # Assicurati che il nome sia corretto
        }

        for key, combo in combos.items():
            settings.setValue(f"EasyPath/{key}", combo.currentData())

        QMessageBox.information(self, "Status saved", "Status saved successfully!",
                                QMessageBox.Ok)

    def restore_physical_accessibility_state(self):
        settings = QSettings()

        combos = {
            "WidthAB": self.WidthAB_comboBox,
            "SlopeLongitudinal": self.SlopeLongitudinal_comboBox,
            "SlopeDownhillRoad": self.SlopeDownhillRoad_comboBox,
            "WidthBA": self.WidthBA_comboBox,
            "SlopeTransversal": self.SlopeTransversal_comboBox,
            "ParkingWidth": self.ParkingWidth_comboBox,
            "ParkingNumber": self.ParkingNumber_comboBox,
            "FlooringAntiSlip": self.FlooringAntiSlip_comboBox,
            "BicyclePathsLength": self.BicyclePathsLength_comboBox,
            "PublicTransportStopsNumber": self.PublicTransportStopsNumber_comboBox,
            "StreetBenchesNumber": self.StreetBenchesNumber_comboBox,
            "WasteBinsNumber": self.WasteBinsNumber_comboBox,
            "DrinkingWaterFountainsNumber": self.DrinkingWaterFountainsNumber_comboBox,
            "BikeCarrierNumber": self.BikeCarrierNumber_comboBox,
            "CrossingNumber": self.CrossingNumber_comboBox
        }

        # Delay per assicurarsi che i campi siano stati caricati
        def apply_restoration():
            for key, combo in combos.items():
                saved_value = settings.value(f"EasyPath/{key}", None)
                if saved_value is not None:
                    index = combo.findData(saved_value)
                    if index >= 0:
                        combo.setCurrentIndex(index)

        QTimer.singleShot(100, apply_restoration)

    # **************************** SALVATAGGIO ENVIRONMENTAL ACCESSIBILITY STATE **************************** #
    def save_environmental_accessibility_state(self):
        settings = QSettings()

        combos = {
            # Microclimate
            "MicroclimatePET9": self.MicroclimatePET9_comboBox,
            "MicroclimatePET12": self.MicroclimatePET12_comboBox,
            "MicroclimatePET15": self.MicroclimatePET15_comboBox,
            "MicroclimateWind9": self.MicroclimateWind9_comboBox,
            "MicroclimateWind12": self.MicroclimateWind12_comboBox,
            "MicroclimateWind15": self.MicroclimateWind15_comboBox,
            "MicroclimateShadow9": self.MicroclimateShadow9_comboBox,
            "MicroclimateShadow12": self.MicroclimateShadow12_comboBox,
            "MicroclimateShadow15": self.MicroclimateShadow15_comboBox,
            "MicroclimateSkyViewFactor": self.MicroclimateSkyViewFactor_comboBox,
            "MicroclimateTreeCover": self.MicroclimateTreeCover_comboBox,
            # Water
            "WaterFloodingRisk": self.WaterFloodingRisk_comboBox,
            "WaterStormwaterVolume": self.WaterStormwaterVolume_comboBox,
        }

        for key, combo in combos.items():
            settings.setValue(f"EasyPath/{key}", combo.currentData())

        QMessageBox.information(self, "Status saved", "Status saved successfully!", QMessageBox.Ok)

    def restore_environmental_accessibility_state(self):
        settings = QSettings()

        combos = {
            # Microclimate
            "MicroclimatePET9": self.MicroclimatePET9_comboBox,
            "MicroclimatePET12": self.MicroclimatePET12_comboBox,
            "MicroclimatePET15": self.MicroclimatePET15_comboBox,
            "MicroclimateWind9": self.MicroclimateWind9_comboBox,
            "MicroclimateWind12": self.MicroclimateWind12_comboBox,
            "MicroclimateWind15": self.MicroclimateWind15_comboBox,
            "MicroclimateShadow9": self.MicroclimateShadow9_comboBox,
            "MicroclimateShadow12": self.MicroclimateShadow12_comboBox,
            "MicroclimateShadow15": self.MicroclimateShadow15_comboBox,
            "MicroclimateSkyViewFactor": self.MicroclimateSkyViewFactor_comboBox,
            "MicroclimateTreeCover": self.MicroclimateTreeCover_comboBox,
            # Water
            "WaterFloodingRisk": self.WaterFloodingRisk_comboBox,
            "WaterStormwaterVolume": self.WaterStormwaterVolume_comboBox,
        }

        def apply_restoration():
            for key, combo in combos.items():
                saved_value = settings.value(f"EasyPath/{key}", None)
                if saved_value is not None:
                    index = combo.findData(saved_value)
                    if index >= 0:
                        combo.setCurrentIndex(index)

        QTimer.singleShot(100, apply_restoration)

    # **************************** SALVATAGGIO SCHEDA 2 **************************** #
    def save_path_layer_state(self):
        settings = QSettings()

        combos = {
            "PhysicalAccessibilityLayer": self.PhysicalAccessibilityLayer_comboBox,
            "EnvironmentalAccessibilityLayer": self.EnvironmentalAccessibilityLayer_comboBox,
            "NodeLayer": self.NodeLayer_comboBox,
            "NodeID": self.NodeID_comboBox,
            "NodeType": self.NodeType_comboBox,
            "APathNode": self.APathNode_comboBox,
            "BPathNode": self.BPathNode_comboBox,
            "PathDirection": self.PathDirection_comboBox
        }

        for key, combo in combos.items():
            settings.setValue(f"EasyPath/{key}", combo.currentData())

        QMessageBox.information(self, "Status saved", "Status saved successfully!", QMessageBox.Ok)

    def restore_path_layer_state(self):
        settings = QSettings()

        combos = {
            "PhysicalAccessibilityLayer": self.PhysicalAccessibilityLayer_comboBox,
            "EnvironmentalAccessibilityLayer": self.EnvironmentalAccessibilityLayer_comboBox,
            "NodeLayer": self.NodeLayer_comboBox,
            "NodeID": self.NodeID_comboBox,
            "NodeType": self.NodeType_comboBox,
            "APathNode": self.APathNode_comboBox,
            "BPathNode": self.BPathNode_comboBox,
            "PathDirection": self.PathDirection_comboBox
        }

        def apply_restoration():
            for key, combo in combos.items():
                saved_value = settings.value(f"EasyPath/{key}", None)
                if saved_value is not None:
                    index = combo.findData(saved_value)
                    if index >= 0:
                        combo.setCurrentIndex(index)

        QTimer.singleShot(100, apply_restoration)

    # **************************** SALVATAGGIO SCHEDA 3 **************************** #
    def save_scenario_state(self):
        settings = QSettings()

        # ComboBox (salvo .currentData() quando la combo ha dati-chiave significativi)
        combos_data = {
            "ChosenPathLayer": self.ChosenPathLayer_comboBox,  # layer id
            "TreeNumberAB": self.TreeNumberAB_comboBox,  # nome campo
            "TreeNumberBA": self.TreeNumberBA_comboBox  # nome campo
        }

        for key, combo in combos_data.items():
            try:
                settings.setValue(f"EasyPath/{key}", combo.currentData())
            except Exception:
                # fallback: se non ha dati associati
                settings.setValue(f"EasyPath/{key}", combo.currentText())

        # LineEdit (salvo testo così com’è)
        lineedits = [
            "AvailableBudget_lineEdit",
            "RoutineMaintentanceInterval_lineEdit",
            "SpecialMaintentanceInterval_lineEdit",
            "TreePruningInterval_lineEdit",
            "TreeStreetFurnitureCosts_lineEdit",
            "BenchesStreetFurnitureCosts_lineEdit",
            "BasketsStreetFurnitureCosts_lineEdit",
            "FountainsStreetFurnitureCosts_lineEdit",
            "BikeCarrierStreetFurnitureCosts_lineEdit",
            "PlanterGSICosts_lineEdit",
            "TreeTrenchGSICosts_lineEdit",
            "GutterGSICosts_lineEdit",
            "RainGardenGSICosts_lineEdit",
            "SwaleGSICosts_lineEdit",
            "VegetationAnnualMaintenanceCosts_lineEdit",
            "SidewalkAnnualMaintenanceCosts_lineEdit",
            "StreetAnnualMaintenanceCosts_lineEdit",
            "StreetFurnitureAnnualMaintenanceCosts_lineEdit",
            "AsphaltStreetCost_lineEdit",
            "CementStreetCost_lineEdit",
            "BasaltStreetCost_lineEdit",
            "StoneStreetCost_lineEdit",
            "CyclePathCost_lineEdit",
            "AsphaltSidewalkCost_lineEdit",
            "CementSidewalkCost_lineEdit",
            "BasaltSidewalkCost_lineEdit",
            "StoneSidewalkCost_lineEdit",
        ]

        for name in lineedits:
            w = getattr(self, name, None)
            if w is not None:
                settings.setValue(f"EasyPath/{name}", w.text())

        QMessageBox.information(self, "Status saved", "Status saved successfully!", QMessageBox.Ok)

    def restore_scenario_state(self):
        settings = QSettings()

        def apply_restoration():
            # ComboBox
            combos_data = {
                "ChosenPathLayer": self.ChosenPathLayer_comboBox,
                "TreeNumberAB": self.TreeNumberAB_comboBox,
                "TreeNumberBA": self.TreeNumberBA_comboBox
            }
            for key, combo in combos_data.items():
                saved = settings.value(f"EasyPath/{key}", None)
                if saved is None:
                    continue
                # prova match su data, se fallisce prova su text
                idx = combo.findData(saved)
                if idx < 0:
                    idx = combo.findText(str(saved))
                if idx >= 0:
                    combo.setCurrentIndex(idx)

            # LineEdit
            lineedits = [
                "AvailableBudget_lineEdit",
                "RoutineMaintentanceInterval_lineEdit",
                "SpecialMaintentanceInterval_lineEdit",
                "TreePruningInterval_lineEdit",
                "TreeStreetFurnitureCosts_lineEdit",
                "BenchesStreetFurnitureCosts_lineEdit",
                "BasketsStreetFurnitureCosts_lineEdit",
                "FountainsStreetFurnitureCosts_lineEdit",
                "BikeCarrierStreetFurnitureCosts_lineEdit",
                "PlanterGSICosts_lineEdit",
                "TreeTrenchGSICosts_lineEdit",
                "GutterGSICosts_lineEdit",
                "RainGardenGSICosts_lineEdit",
                "SwaleGSICosts_lineEdit",
                "VegetationAnnualMaintenanceCosts_lineEdit",
                "SidewalkAnnualMaintenanceCosts_lineEdit",
                "StreetAnnualMaintenanceCosts_lineEdit",
                "StreetFurnitureAnnualMaintenanceCosts_lineEdit",
                "AsphaltStreetCost_lineEdit",
                "CementStreetCost_lineEdit",
                "BasaltStreetCost_lineEdit",
                "StoneStreetCost_lineEdit",
                "CyclePathCost_lineEdit",
                "AsphaltSidewalkCost_lineEdit",
                "CementSidewalkCost_lineEdit",
                "BasaltSidewalkCost_lineEdit",
                "StoneSidewalkCost_lineEdit",
            ]
            for name in lineedits:
                w = getattr(self, name, None)
                if w is not None:
                    saved = settings.value(f"EasyPath/{name}", None)
                    if saved is not None:
                        w.setText(str(saved))

        # Aspetto che le combo siano popolate (ChosenPathLayer viene popolata nell'__init__)
        QTimer.singleShot(100, apply_restoration)

    # **************************** CALCOLO PHYSICAL ACCESSIBILITY **************************** #
    def calculate_accessibility_grade(self):
        # Recupera il layer selezionato
        layer_id = self.PathLayer_comboBox.currentData()
        vector_layer = QgsProject.instance().mapLayer(layer_id)
        if not isinstance(vector_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Error", "Invalid layer selected")
            return

        get_field = lambda c: c.currentText()
        field_ID = get_field(self.ID_comboBox)
        field_length = get_field(self.PathLength_comboBox)

        # Prepara i campi del nuovo layer
        fields = QgsFields()
        fields.append(QgsField("ID", QVariant.String))
        fields.append(QgsField("PathLength", QVariant.Double))
        fields.append(QgsField("SidewalkAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("ParkingAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("FlooringAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("BicyclePathsAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("PublicTransportAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("StreetBenchesAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("WasteBinsAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("DrinkingWaterAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("BikeCarrierAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("CrossingAccessibilityIndex", QVariant.Double))
        fields.append(QgsField("PhysicalAccessibilityIndex", QVariant.Double))

        # Crea layer in memoria
        geometry_type = vector_layer.wkbType()
        crs = vector_layer.crs().authid()
        new_layer = QgsVectorLayer(f"{QgsWkbTypes.displayString(geometry_type)}?crs={crs}", "Physical Accessibility",
                                   "memory")
        provider = new_layer.dataProvider()
        provider.addAttributes(fields)
        new_layer.updateFields()

        # Raggruppa per ID
        features_by_id = {}
        for feat in vector_layer.getFeatures():
            val = feat[field_ID]
            features_by_id.setdefault(val, []).append(feat)

        for id_val, feats in features_by_id.items():
            geom = feats[0].geometry()

            def safe_avg(combo, absolute=False):
                field = get_field(combo)
                values = []
                for f in feats:
                    try:
                        val = float(f[field])
                        if absolute:
                            val = abs(val)
                        values.append(val)
                    except (TypeError, ValueError, KeyError):
                        continue
                return sum(values) / len(values) if values else 0

            def clean(val):
                return round(min(val, 100), 2)

            try:
                total_length = sum(f[field_length] for f in feats if f[field_length] is not None)

                sidewalk = clean(25 * (
                        ((safe_avg(self.WidthAB_comboBox) + safe_avg(self.WidthBA_comboBox)) / 1.8) +
                        (safe_avg(self.SlopeLongitudinal_comboBox, absolute=True) / 0.05) +
                        (safe_avg(self.SlopeTransversal_comboBox, absolute=True) / 0.01) +
                        (safe_avg(self.SlopeDownhillRoad_comboBox, absolute=True) / 0.15)
                ))

                parking = clean(50 * (
                        (safe_avg(self.ParkingWidth_comboBox) / 3.2) +
                        (safe_avg(self.ParkingNumber_comboBox) / 0.02)
                ))

                flooring = clean(100 * safe_avg(self.FlooringAntiSlip_comboBox))
                bicycle = clean(
                    100 * (safe_avg(self.BicyclePathsLength_comboBox) / total_length)) if total_length else 0
                transport = clean(100 * safe_avg(self.PublicTransportStopsNumber_comboBox))
                benches = clean(
                    (safe_avg(self.StreetBenchesNumber_comboBox) / total_length * 100)) if total_length else 0
                bins = clean((safe_avg(self.WasteBinsNumber_comboBox) / total_length * 100)) if total_length else 0
                fountains = clean(
                    (safe_avg(self.DrinkingWaterFountainsNumber_comboBox) / total_length * 100)) if total_length else 0
                bike_carriers = clean(
                    (safe_avg(self.BikeCarrierNumber_comboBox) / total_length * 100)) if total_length else 0
                crossings = clean((safe_avg(self.CrossingNumber_comboBox) / total_length * 100)) if total_length else 0

                # Calcolo PhysicalAccessibilityIndex
                total_index = (
                        sidewalk + parking + flooring + bicycle + transport +
                        benches + bins + fountains + bike_carriers + crossings
                )
                physical_index = clean((total_index / 1000) * 5)

                # Aggiungi feature al layer
                new_feat = QgsFeature(new_layer.fields())
                new_feat.setGeometry(geom)
                new_feat.setAttributes([
                    id_val,
                    round(total_length, 2),
                    sidewalk,
                    parking,
                    flooring,
                    bicycle,
                    transport,
                    benches,
                    bins,
                    fountains,
                    bike_carriers,
                    crossings,
                    physical_index
                ])
                provider.addFeature(new_feat)

            except Exception as e:
                QMessageBox.warning(self, "Error", f"Error in calculation for ID '{id_val}': {e}")

        new_layer.updateExtents()
        QgsProject.instance().addMapLayer(new_layer)

        def on_layer_added(layer):
            if layer.id() == new_layer.id():
                self.update_physical_accessibility_layer_combobox()
                index = self.PhysicalAccessibilityLayer_comboBox.findData(new_layer.id())
                if index >= 0:
                    self.PhysicalAccessibilityLayer_comboBox.setCurrentIndex(index)
                # Dopo che ha fatto tutto, scollega il listener (per non accumulare listener)
                QgsProject.instance().layerWasAdded.disconnect(on_layer_added)

        QgsProject.instance().layerWasAdded.connect(on_layer_added)

        # Definisci i range e colori (dal rosso al verde)
        ranges = [
            (0.0, 1.0, "#b5241b"),
            (1.01, 1.5, "#e52318"),
            (1.51, 2.0, "#f3930e"),
            (2.01, 3.0, "#f6ff00"),
            (3.01, 4.0, "#57ec4f"),
            (4.01, 5.0, "#1bae13"),
        ]

        symbols = []
        for lower, upper, color in ranges:
            symbol = QgsSymbol.defaultSymbol(new_layer.geometryType())
            symbol.setColor(QColor(color))
            symbol.symbolLayer(0).setWidth(1)
            label = f"{lower} - {upper}"
            symbols.append(QgsRendererRange(lower, upper, symbol, label))

        renderer = QgsGraduatedSymbolRenderer("PhysicalAccessibilityIndex", symbols)
        renderer.setMode(QgsGraduatedSymbolRenderer.Custom)

        new_layer.setRenderer(renderer)
        new_layer.triggerRepaint()

        QMessageBox.information(self, "Done", "Physical accessibility layer generated correctly")

    # === Helpers per scenario ibrido ===
    def _hybrid_weights(self):
        """Pesi fisico/ambientale coerenti con lo slider già usato nel routing."""
        slider_value = self.Accessibility_horizontalSlider.value()  # valori -5, 0, 5
        w_phys = max(0, (5 - slider_value) / 10.0)
        w_env  = max(0, (5 + slider_value) / 10.0)
        return w_phys, w_env

    def _current_env_field(self) -> str:
        if self.H9_radioButton.isChecked():
            return "EnvironmentalAccessibility_9"
        if self.H12_radioButton.isChecked():
            return "EnvironmentalAccessibility_12"
        if self.H15_radioButton.isChecked():
            return "EnvironmentalAccessibility_15"
        # fallback
        return "EnvironmentalAccessibility_12"

    def _route_segment_ids_from_chosen(self):
        """
        Restituisce (segment_ids, route_layer) prendendo il layer dal
        ChosenPathLayer_comboBox. Se non è selezionato nulla, prova in fallback:
          1) ultimo layer che inizia con 'Path "'
          2) layer attivo se inizia con 'Path "'
        Se trova un layer, sincronizza anche la combo ChosenPathLayer_comboBox.
        """
        # 1) tenta dalla combo 'ChosenPathLayer'
        route_layer_id = self.ChosenPathLayer_comboBox.currentData()
        route_layer = QgsProject.instance().mapLayer(route_layer_id) if route_layer_id else None

        # 2) fallback: ultimo layer 'Path "…"' presente nel progetto
        if route_layer is None:
            candidates = []
            for lyr in QgsProject.instance().mapLayers().values():
                if isinstance(lyr, QgsVectorLayer) and lyr.name().startswith('Path "'):
                    candidates.append(lyr)
            if candidates:
                candidates.sort(key=lambda l: l.name())
                route_layer = candidates[-1]
                # prova a selezionarlo nella combo
                self.update_chosen_path_layer_combobox()
                idx = self.ChosenPathLayer_comboBox.findData(route_layer.id())
                if idx >= 0:
                    self.ChosenPathLayer_comboBox.setCurrentIndex(idx)

        # 3) ulteriore fallback: layer attivo se è un 'Path "…"'
        if route_layer is None:
            active = iface.activeLayer()
            if isinstance(active, QgsVectorLayer) and active.name().startswith('Path "'):
                route_layer = active
                self.update_chosen_path_layer_combobox()
                idx = self.ChosenPathLayer_comboBox.findData(route_layer.id())
                if idx >= 0:
                    self.ChosenPathLayer_comboBox.setCurrentIndex(idx)

        # 4) se ancora nulla: ritorna vuoto
        if route_layer is None:
            return [], None

        # 5) estrai gli ID dei segmenti
        segment_ids = []
        try:
            for f in route_layer.getFeatures():
                segment_ids.append(str(f["segment_id"]))
        except Exception:
            pass

        return segment_ids, route_layer

    def _avg_on_ids(self, layer: QgsVectorLayer, field: str, ids: list):
        """Media semplice del campo 'field' su features del layer con ID in ids."""
        if not isinstance(layer, QgsVectorLayer) or not field:
            return 0.0
        vals = []
        for sid in ids:
            for g in layer.getFeatures(QgsFeatureRequest().setFilterExpression(f'"ID" = \'{sid}\'')):
                v = g[field]
                try:
                    vals.append(float(v))
                except Exception:
                    pass
        return sum(vals) / len(vals) if vals else 0.0

    def _parse_hybrid_delta(self, text: str):
        """Parsa stringa tipo 'ΔPA=+0.123, ΔEA=-0.050' -> (dPA, dEA)."""
        dpa = 0.0;
        dea = 0.0
        if not text: return dpa, dea
        try:
            parts = text.replace("Δ", "").replace(" ", "").split(",")
            for p in parts:
                if p.upper().startswith("PA="):
                    dpa = float(p.split("=")[1])
                elif p.upper().startswith("EA="):
                    dea = float(p.split("=")[1])
        except Exception:
            pass
        return dpa, dea

    def _clean_score(self, v):
        try:
            return round(float(v), 6)
        except Exception:
            return 0.0

    def _safe_float(self, v, default=0.0):
        try:
            if v in [None, '', QVariant()]:
                return default
            return float(v)
        except Exception:
            return default

    def _binary_access(self, val):
        return 0 if val == 0 else 5

    def _sky_tree_access(self, val):
        thresholds = [0.1, 0.2, 0.35, 0.5, 0.7, 1]
        for i, t in enumerate(thresholds):
            if val <= t: return i
        return 5

    def _flooding_access(self, raw):
        if raw in [None, '', QVariant()]: return 5
        return {"Basso": 3, "Medio": 1, "Alto": 0}.get(str(raw), 5)


    # **************************** CALCOLO ENVIRONMENTAL ACCESSIBILITY **************************** #

    def _max_items_by_spacing(self, length_m: float, spacing_m: float) -> int:
        """Numero massimo posizionabile lungo il segmento rispettando il passo minimo."""
        try:
            L = float(length_m or 0.0)
            S = float(spacing_m or 0.01)
        except Exception:
            return 0
        if L <= 0 or S <= 0:
            return 0
        return int(L // S)

    def _cycle_choice(self, street_width: float, dir_field_value: str, length_m: float, cost_per_m: float):
        """
        Sceglie nessuna/monodirezionale/bidirezionale rispettando la carreggiata minima
        in base al senso (AB/BA = senso unico, altrimenti doppio senso).
        Ritorna: (cycle_w, cycle_cost, note)
        """
        direction = (dir_field_value or "").strip().upper()
        min_carreggiata = 3.8 if direction in ["AB", "BA"] else 6.6

        # Priorità qualitativa: prima doppio senso, poi senso unico, poi nulla
        for w, label in [(CYCLE_DOUBLE_W, "Two-way cycle path"),
                         (CYCLE_SINGLE_W, "One-way cycle path"),
                         (0.0, "No cycle path")]:
            if street_width - w >= min_carreggiata:
                cost = (cost_per_m or 0.0) * (length_m or 0.0) if w > 0 else 0.0
                return (w, cost, label)

        return (0.0, 0.0, "No cycle path")

    def calculate_environmental_accessibility_grade(self):
        layer_id = self.PathLayer_comboBox.currentData()
        vector_layer = QgsProject.instance().mapLayer(layer_id)
        if not isinstance(vector_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Error", "Invalid layer selected")
            return

        get_field = lambda c: c.currentText()
        field_ID = get_field(self.ID_comboBox)

        # ✅ Campi dettagliati in output
        fields = QgsFields()
        fields.append(QgsField("ID", QVariant.String))
        # Microclimate dettagliato
        fields.append(QgsField("PET9", QVariant.Double))
        fields.append(QgsField("PET12", QVariant.Double))
        fields.append(QgsField("PET15", QVariant.Double))
        fields.append(QgsField("Wind9", QVariant.Double))
        fields.append(QgsField("Wind12", QVariant.Double))
        fields.append(QgsField("Wind15", QVariant.Double))
        fields.append(QgsField("Shadow9", QVariant.Double))
        fields.append(QgsField("Shadow12", QVariant.Double))
        fields.append(QgsField("Shadow15", QVariant.Double))
        fields.append(QgsField("SkyViewFactor", QVariant.Double))
        fields.append(QgsField("TreeCover", QVariant.Double))
        # Altri elementi
        fields.append(QgsField("FloodingRisk", QVariant.Double))
        fields.append(QgsField("UtilitiesActivities", QVariant.Double))
        # Finali
        fields.append(QgsField("EnvironmentalAccessibility_9", QVariant.Double))
        fields.append(QgsField("EnvironmentalAccessibility_12", QVariant.Double))
        fields.append(QgsField("EnvironmentalAccessibility_15", QVariant.Double))

        geometry_type = vector_layer.wkbType()
        crs = vector_layer.crs().authid()
        new_layer = QgsVectorLayer(f"{QgsWkbTypes.displayString(geometry_type)}?crs={crs}",
                                   "Environmental Accessibility", "memory")
        provider = new_layer.dataProvider()
        provider.addAttributes(fields)
        new_layer.updateFields()

        def safe_avg(feats, field):
            values = []
            for f in feats:
                val = f[field]
                if val is None or isinstance(val, QVariant) or val == '':
                    continue
                try:
                    values.append(float(val))
                except (ValueError, TypeError):
                    continue
            return sum(values) / len(values) if values else 0

        def pet_access(val):
            thresholds = [4, 8, 13, 18, 20, 21, 23, 29, 35, 41]
            scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
            for t, s in zip(thresholds, scores):
                if val <= t: return s
            return 0

        def wind_access(val):
            thresholds = [0, 0.3, 0.9, 1.9, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5]
            scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
            for t, s in zip(thresholds, scores):
                if val <= t: return s
            return 0

        def binary_access(val):
            return 0 if val == 0 else 5

        def sky_tree_access(val):
            thresholds = [0.1, 0.2, 0.35, 0.5, 0.7, 1]
            for i, t in enumerate(thresholds):
                if val <= t: return i
            return 5

        def flooding_access(val):
            if val in [None, '', QVariant()]: return 5
            return {"Basso": 3, "Medio": 1, "Alto": 0}.get(str(val), 5)

        def activities_access(val):
            return 0 if val in [None, '', QVariant()] else 5

        features_by_id = {}
        for feat in vector_layer.getFeatures():
            val = feat[field_ID]
            features_by_id.setdefault(val, []).append(feat)

        for id_val, feats in features_by_id.items():
            geom = feats[0].geometry()

            try:
                pet9_val = safe_avg(feats, get_field(self.MicroclimatePET9_comboBox))
                pet12_val = safe_avg(feats, get_field(self.MicroclimatePET12_comboBox))
                pet15_val = safe_avg(feats, get_field(self.MicroclimatePET15_comboBox))
                wind9_val = safe_avg(feats, get_field(self.MicroclimateWind9_comboBox))
                wind12_val = safe_avg(feats, get_field(self.MicroclimateWind12_comboBox))
                wind15_val = safe_avg(feats, get_field(self.MicroclimateWind15_comboBox))
                shadow9_val = safe_avg(feats, get_field(self.MicroclimateShadow9_comboBox))
                shadow12_val = safe_avg(feats, get_field(self.MicroclimateShadow12_comboBox))
                shadow15_val = safe_avg(feats, get_field(self.MicroclimateShadow15_comboBox))
                sky_val = safe_avg(feats, get_field(self.MicroclimateSkyViewFactor_comboBox))
                tree_val = safe_avg(feats, get_field(self.MicroclimateTreeCover_comboBox))

                flood_field = get_field(self.WaterFloodingRisk_comboBox)
                flood_val_raw = next((f[flood_field] for f in feats if f[flood_field] not in [None, '', QVariant()]),
                                     None)
                flood_val = flooding_access(flood_val_raw)

                # Converte i valori in indici accessibilità
                pet9 = pet_access(pet9_val)
                pet12 = pet_access(pet12_val)
                pet15 = pet_access(pet15_val)
                wind9 = wind_access(wind9_val)
                wind12 = wind_access(wind12_val)
                wind15 = wind_access(wind15_val)
                shadow9 = binary_access(shadow9_val)
                shadow12 = binary_access(shadow12_val)
                shadow15 = binary_access(shadow15_val)
                sky = sky_tree_access(sky_val)
                tree = sky_tree_access(tree_val)

                # Media finale
                ea_9 = round((pet9 + wind9 + shadow9 + sky + tree + flood_val) / 6, 2)
                ea_12 = round((pet12 + wind12 + shadow12 + sky + tree + flood_val) / 6, 2)
                ea_15 = round((pet15 + wind15 + shadow15 + sky + tree + flood_val) / 6, 2)

                new_feat = QgsFeature(new_layer.fields())
                new_feat.setGeometry(geom)
                new_feat.setAttributes([
                    id_val,
                    pet9, pet12, pet15,
                    wind9, wind12, wind15,
                    shadow9, shadow12, shadow15,
                    sky, tree,
                    flood_val,
                    ea_9, ea_12, ea_15
                ])
                provider.addFeature(new_feat)

            except Exception as e:
                QMessageBox.warning(self, "Error", f"Error in calculation for ID '{id_val}': {e}")

        new_layer.updateExtents()

        # 🎨 Renderer graduato su EnvironmentalAccessibility_9
        ranges = [
            (0.0, 1.0, "#b5241b"),
            (1.01, 1.5, "#e52318"),
            (1.51, 2.0, "#f3930e"),
            (2.01, 3.0, "#f6ff00"),
            (3.01, 4.0, "#57ec4f"),
            (4.01, 5.0, "#1bae13"),
        ]

        symbols = []
        for lower, upper, color in ranges:
            symbol = QgsSymbol.defaultSymbol(new_layer.geometryType())
            symbol.setColor(QColor(color))
            symbol.symbolLayer(0).setWidth(1)
            label = f"{lower} - {upper}"
            symbols.append(QgsRendererRange(lower, upper, symbol, label))

        renderer = QgsGraduatedSymbolRenderer("EnvironmentalAccessibility_9", symbols)
        renderer.setMode(QgsGraduatedSymbolRenderer.Custom)
        new_layer.setRenderer(renderer)
        new_layer.triggerRepaint()

        QgsProject.instance().addMapLayer(new_layer)
        QMessageBox.information(self, "Done", "Environmental accessibility layer generated correctly.")

    # ****************************** MINIMAPPA UPDATE ****************************** #
    def refresh_mini_canvas(self):
        visible_layers = []
        root = QgsProject.instance().layerTreeRoot()

        # Ottieni gli ID dei due layer speciali
        physical_layer_id = self.PhysicalAccessibilityLayer_comboBox.currentData()
        environmental_layer_id = self.EnvironmentalAccessibilityLayer_comboBox.currentData()
        path_layer_id = self.PathSelector_comboBox.currentData()

        for node in root.findLayers():
            layer = node.layer()
            if not layer or not layer.isValid():
                continue

            # Controllo speciale sui layer Physical/Environmental
            if layer.id() == physical_layer_id:
                if not self.ShowPhysicalLayer_checkBox.isChecked():
                    continue
            if layer.id() == environmental_layer_id:
                if not self.ShowEnvironmentalLayer_checkBox.isChecked():
                    continue
            if layer.id() == path_layer_id:
                if not self.ShowPathLayer_checkBox.isChecked():
                    continue

            # Altrimenti lo aggiunge
            if node.isVisible():
                visible_layers.append(layer)

        self.mini_canvas.setLayers(visible_layers)
        self.mini_canvas.refresh()

    def toggle_physical_layer_visibility(self):
        physical_layer_id = self.PhysicalAccessibilityLayer_comboBox.currentData()
        physical_layer = QgsProject.instance().mapLayer(physical_layer_id)

        if not physical_layer:
            return

        root = QgsProject.instance().layerTreeRoot()
        node = root.findLayer(physical_layer.id())
        if node:
            node.setItemVisibilityChecked(self.ShowPhysicalLayer_checkBox.isChecked())

    def toggle_environmental_layer_visibility(self):
        environmental_layer_id = self.EnvironmentalAccessibilityLayer_comboBox.currentData()
        environmental_layer = QgsProject.instance().mapLayer(environmental_layer_id)

        if not environmental_layer:
            return

        root = QgsProject.instance().layerTreeRoot()
        node = root.findLayer(environmental_layer.id())
        if node:
            node.setItemVisibilityChecked(self.ShowEnvironmentalLayer_checkBox.isChecked())

    def toggle_path_layer_visibility(self):
        path_layer_id = self.PathSelector_comboBox.currentData()
        path_layer = QgsProject.instance().mapLayer(path_layer_id)

        if not path_layer:
            return

        root = QgsProject.instance().layerTreeRoot()
        node = root.findLayer(path_layer.id())
        if node:
            node.setItemVisibilityChecked(self.ShowPathLayer_checkBox.isChecked())

    def update_path_selector_combobox(self):
        self.PathSelector_comboBox.clear()

        layers = QgsProject.instance().mapLayers().values()
        path_layers = []

        for layer in layers:
            if layer.name().startswith('Path "') and isinstance(layer, QgsVectorLayer):
                path_layers.append(layer)

        # Ordina i layer per nome (opzionale: se vuoi più ordine alfabetico)
        path_layers.sort(key=lambda l: l.name())

        for layer in path_layers:
            self.PathSelector_comboBox.addItem(layer.name(), layer.id())

        # Seleziona automaticamente l'ultimo percorso aggiunto
        if path_layers:
            last_layer = path_layers[-1]
            index = self.PathSelector_comboBox.findData(last_layer.id())
            if index >= 0:
                self.PathSelector_comboBox.setCurrentIndex(index)

    # ************************************************************ CALCOLO PERCORSI ************************************************************ #
    def generate_optimal_path(self):
        start_node_val = self.StartingNode_comboBox.currentData()
        end_node_val = self.ArrivalNode_comboBox.currentData()

        if not start_node_val or not end_node_val:
            QMessageBox.warning(self, "Error", "Select departure and arrival nodes")
            return

        start_node_val = str(start_node_val)
        end_node_val = str(end_node_val)

        if not (
                self.OnFoot_radioButton.isChecked() or self.ByCar_radioButton.isChecked() or self.ByBike_radioButton.isChecked()
        ):
            QMessageBox.warning(self, "Error",
                                "Select at least one mode of transportation (walking, driving, or biking)")
            return

        layer_id = self.PathLayer_comboBox.currentData()
        vector_layer = QgsProject.instance().mapLayer(layer_id)
        if not isinstance(vector_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Error", "Invalid road layer")
            return

        path_access_id = self.PhysicalAccessibilityLayer_comboBox.currentData()
        access_layer = QgsProject.instance().mapLayer(path_access_id)

        env_access_id = self.EnvironmentalAccessibilityLayer_comboBox.currentData()
        env_layer = QgsProject.instance().mapLayer(env_access_id) if env_access_id else None

        if self.H9_radioButton.isChecked():
            env_field = "EnvironmentalAccessibility_9"
        elif self.H12_radioButton.isChecked():
            env_field = "EnvironmentalAccessibility_12"
        elif self.H15_radioButton.isChecked():
            env_field = "EnvironmentalAccessibility_15"
        else:
            QMessageBox.warning(self, "Error", "Select a time for environmental accessibility")
            return

        A_field = self.APathNode_comboBox.currentText()
        B_field = self.BPathNode_comboBox.currentText()
        dir_field = self.PathDirection_comboBox.currentText()
        id_field = self.ID_comboBox.currentText()
        length_field = self.PathLength_comboBox.currentText()

        if self.ByCar_radioButton.isChecked():
            speed = 30
            mode = "by car"
        elif self.ByBike_radioButton.isChecked():
            speed = 15
            mode = "by bike"
        else:
            speed = 4
            mode = "by foot"

        slider_value = self.Accessibility_horizontalSlider.value()
        weight_physical = max(0, (5 - slider_value) / 10)
        weight_environmental = max(0, (5 + slider_value) / 10)

        graph = {}
        segment_geoms = {}
        segment_lengths = {}
        segment_access = {}

        for feat in vector_layer.getFeatures():
            A = str(feat[A_field])
            B = str(feat[B_field])
            direction = (feat[dir_field] or "").strip().upper()
            fid = str(feat[id_field])
            try:
                length = float(feat[length_field])
            except (ValueError, TypeError):
                continue

            geom = feat.geometry()
            segment_lengths[fid] = length

            access_val = 1.0
            if access_layer:
                expr = f'"ID" = \'{fid}\''
                matches = access_layer.getFeatures(QgsFeatureRequest().setFilterExpression(expr))
                for af in matches:
                    access_val = af["PhysicalAccessibilityIndex"]
                    break

            env_val = 1.0
            if env_layer:
                expr = f'"ID" = \'{fid}\''
                matches = env_layer.getFeatures(QgsFeatureRequest().setFilterExpression(expr))
                for ef in matches:
                    env_val = ef[env_field]
                    break

            combined_access = weight_physical * access_val + weight_environmental * env_val
            segment_access[fid] = combined_access

            time_minutes = ((length / 1000) / speed) * 60
            penalty = max(0, 5 - combined_access) * 0.2
            cost = time_minutes * (1 + penalty)

            directions = []
            if direction == "AB":
                directions = [(A, B)]
            elif direction == "BA":
                directions = [(B, A)]
            else:
                directions = [(A, B), (B, A)]

            for u, v in directions:
                if u not in graph:
                    graph[u] = []
                graph[u].append((v, cost, fid, geom))
                segment_geoms[(u, v, fid)] = (geom, fid)

        if self.OnFoot_radioButton.isChecked():
            for feat in vector_layer.getFeatures():
                A = str(feat[A_field])
                B = str(feat[B_field])
                fid = str(feat[id_field])
                try:
                    length = float(feat[length_field])
                except (ValueError, TypeError):
                    continue

                geom = feat.geometry()
                combined_access = segment_access.get(fid, 1.0)
                time_minutes = (length / 1000) / speed * 60
                # Slider tempo/accessibilità
                balance_val = self.TimeAccessBalance_horizontalSlider.value()
                weight_time = max(0, (5 - balance_val) / 10)
                weight_access = max(0, (5 + balance_val) / 10)

                # Costo personalizzato
                cost = weight_time * time_minutes + weight_access * (5 - combined_access)

                if B not in graph:
                    graph[B] = []
                graph[B].append((A, cost, fid, geom))
                segment_geoms[(B, A, fid)] = (geom, fid)

        best_path = None
        final_cost = None
        total_length = 0
        used_ids = set()

        start_reachable = start_node_val in graph and len(graph[start_node_val]) > 0
        end_reachable = any(end_node_val == neighbor for neighbors in graph.values() for neighbor, *_ in neighbors)

        if not start_reachable or not end_reachable:
            problemi = []
            if not start_reachable:
                problemi.append(
                    f"The starting node '{start_node_val}' has no outgoing edges (blocked by the direction of travel).")
            if not end_reachable:
                problemi.append(
                    f"The starting node '{end_node_val}' has no incoming arcs (blocked by the direction of travel).")
            QMessageBox.warning(self, "Path not found", "Path not found\n\n" + "\n".join(problemi))
            return

        frontier = [(0, start_node_val, [], set(), 0)]

        visited = set()

        while frontier:
            curr_cost, curr_node, path, used_ids_set, path_len = heapq.heappop(frontier)
            state = (curr_node, tuple(path))
            if state in visited:
                continue
            visited.add(state)

            path = path + [curr_node]
            if curr_node == end_node_val:
                best_path = path
                final_cost = curr_cost
                total_length = path_len
                used_ids = used_ids_set
                break

            for neighbor, edge_cost, edge_fid, edge_geom in graph.get(curr_node, []):
                if neighbor in path or edge_fid in used_ids_set:
                    continue
                new_used = set(used_ids_set)
                new_used.add(edge_fid)
                new_len = path_len + segment_lengths.get(edge_fid, 0)
                heapq.heappush(frontier, (curr_cost + edge_cost, neighbor, path, new_used, new_len))

        if best_path is None:
            QMessageBox.warning(self, "Error", "Path not found")
            return

        crs = vector_layer.crs()

        # Recupera i nomi visibili dei nodi selezionati
        starting_node_text = self.StartingNode_comboBox.currentText()
        arrival_node_text = self.ArrivalNode_comboBox.currentText()

        # Costruisci nome layer
        layer_name = f'Path "{starting_node_text}" -> "{arrival_node_text}"'

        # Crea il layer con nome dinamico
        route_layer = QgsVectorLayer(f"LineString?crs={crs.authid()}", layer_name, "memory")

        pr = route_layer.dataProvider()
        pr.addAttributes([
            QgsField("segment_id", QVariant.String),
            QgsField("accessibility", QVariant.Double),
            QgsField("length_m", QVariant.Double),
            QgsField("duration_min", QVariant.Double),
            QgsField("mode", QVariant.String),
            QgsField("road_name", QVariant.String),
            QgsField("accessibility_priority", QVariant.Int),
            QgsField("time_access_balance", QVariant.Int),
        ])
        route_layer.updateFields()

        for i in range(len(best_path) - 1):
            u, v = best_path[i], best_path[i + 1]
            segment_data = next(((g, fid) for (uu, vv, fid), (g, fid) in segment_geoms.items()
                                 if ((uu == u and vv == v) or (uu == v and vv == u)) and fid in used_ids), None)

            if segment_data:
                geom, segment_fid = segment_data
                length = segment_lengths.get(segment_fid, 0)
                access = segment_access.get(segment_fid, 1.0)
                duration = (length / 1000) / speed * 60 * (1 + max(0, 5 - access) * 0.2)

                feat = QgsFeature()
                feat.setGeometry(geom)

                # Estrai il nome della strada per il segmento
                road_name = ""
                for f in vector_layer.getFeatures(
                        QgsFeatureRequest().setFilterExpression(f'"{id_field}" = \'{segment_fid}\'')):
                    road_name = f[self.PathName_comboBox.currentText()]
                    break

                feat.setAttributes([
                    segment_fid,
                    round(access, 2),
                    round(length, 2),
                    round(duration, 2),
                    mode,
                    road_name,
                    self.Accessibility_horizontalSlider.value(),
                    self.TimeAccessBalance_horizontalSlider.value()
                ])
                pr.addFeature(feat)

        route_layer.updateExtents()

        # Applica renderer graduato con la stessa legenda delle strade
        ranges = [
            (0.0, 1.0, "#b5241b"),
            (1.01, 1.5, "#e52318"),
            (1.51, 2.0, "#f3930e"),
            (2.01, 3.0, "#f6ff00"),
            (3.01, 4.0, "#57ec4f"),
            (4.01, 5.0, "#1bae13"),
        ]

        symbols = []
        for lower, upper, color in ranges:
            symbol = QgsSymbol.defaultSymbol(route_layer.geometryType())
            symbol_layer = symbol.symbolLayer(0)
            symbol_layer.setColor(QColor(color))
            symbol_layer.setWidth(2)  # Spessore linea
            symbol_layer.setPenStyle(Qt.DotLine)  # Linea tratteggiata
            label = f"{lower} - {upper}"
            symbols.append(QgsRendererRange(lower, upper, symbol, label))

        renderer = QgsGraduatedSymbolRenderer("accessibility", symbols)
        renderer.setMode(QgsGraduatedSymbolRenderer.Custom)

        route_layer.setRenderer(renderer)
        route_layer.triggerRepaint()

        QgsProject.instance().addMapLayer(route_layer)

        self.update_path_selector_combobox()

        QMessageBox.information(
            self,
            "Generated route",
            f"Generated segments: {len(used_ids)}\n"
            f"Length: {round(total_length, 1)} m\n"
            f"Estimated time: {round(final_cost, 1)} minutes\n"
            f"Mode: {mode}"
        )

        access_vals = [
            segment_access.get(fid, 1.0)
            for (uu, vv, fid) in segment_geoms
            if (uu, vv) in zip(best_path[:-1], best_path[1:]) and fid in used_ids
        ]
        avg_access = sum(access_vals) / len(access_vals) if access_vals else 0

        self.PathLengthValue_label.setText(f"{round(total_length, 2)} m")
        self.EasyPathValue_label.setText(f"{round(avg_access, 2)}")
        pure_time = (total_length / 1000) / speed * 60
        self.PathTimeValue_label.setText(f"{round(pure_time, 2)} min")

    def remove_route_annotations(self, layer_id):
        annotations = self.route_annotations.pop(layer_id, [])
        for annotation in annotations:
            QgsProject.instance().annotationManager().removeAnnotation(annotation)

    def update_chosen_path_layer_combobox(self):
        current_value = self.ChosenPathLayer_comboBox.currentData()

        self.ChosenPathLayer_comboBox.clear()
        self.ChosenPathLayer_comboBox.addItem("Select Layer", None)

        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer):
                self.ChosenPathLayer_comboBox.addItem(layer.name(), layer.id())

        if current_value:
            index = self.ChosenPathLayer_comboBox.findData(current_value)
            if index >= 0:
                self.ChosenPathLayer_comboBox.setCurrentIndex(index)


# ************************************************************ CALCOLO SCENARI FINALI ************************************************************ #
    def calculate_minimum_scenario(self):

        # Verifica selezione tipo scenario
        if not (self.EnvironmentalScenario_radioButton.isChecked() or
                self.PhysicalScenario_radioButton.isChecked() or
                self.HybridScenario_radioButton.isChecked()):
            QMessageBox.warning(self, "Error", "Select a scenario type (Environmental, Physical, or Hybrid)")
            return

        # Accensione scheda output fase 3
        # Mostra la tab nascosta se presente
        if hasattr(self, '_hidden_tab_widget'):
            self.Accessibility_tabWidget.addTab(self._hidden_tab_widget, self._hidden_tab_label)
            del self._hidden_tab_widget  # Mostrata, quindi non più necessaria

        self.minimum_scenario_modifications = {}  # Reset
        self.ScenarioMinimoResults_tableWidget.setRowCount(0)  # Clear output

        chosen_layer_id = self.ChosenPathLayer_comboBox.currentData()
        chosen_layer = QgsProject.instance().mapLayer(chosen_layer_id)
        if not chosen_layer:
            QMessageBox.warning(self, "Error", "Invalid layer path")
            return

        path_layer_id = self.PathLayer_comboBox.currentData()
        path_layer = QgsProject.instance().mapLayer(path_layer_id)
        if not path_layer:
            QMessageBox.warning(self, "Error", "Invalid path attribute layer")
            return

        # Recupera mapping attributi
        id_field = self.ID_comboBox.currentText()
        width_field = self.PathWidth_comboBox.currentText()
        length_field = self.PathLength_comboBox.currentText()
        widthAB_field = self.WidthAB_comboBox.currentText()
        widthBA_field = self.WidthBA_comboBox.currentText()
        treeAB_field = self.TreeNumberAB_comboBox.currentText()
        treeBA_field = self.TreeNumberBA_comboBox.currentText()
        direction_field = self.PathDirection_comboBox.currentText()

        # Materiali inseriti dall'utente
        street_material_cost = None
        for field in [self.AsphaltStreetCost_lineEdit, self.CementStreetCost_lineEdit,
                      self.BasaltStreetCost_lineEdit, self.StoneStreetCost_lineEdit]:
            if field.text():
                street_material_cost = float(field.text())
                break

        sidewalk_material_cost = None
        for field in [self.AsphaltSidewalkCost_lineEdit, self.CementSidewalkCost_lineEdit,
                      self.BasaltSidewalkCost_lineEdit, self.StoneSidewalkCost_lineEdit]:
            if field.text():
                sidewalk_material_cost = float(field.text())
                break

        if street_material_cost is None or sidewalk_material_cost is None:
            QMessageBox.warning(self, "Error", "Enter at least one cost for the road and one for the sidewalk")
            return

        # Indicizzazione layer attributi per ID
        path_features = {str(f[id_field]): f for f in path_layer.getFeatures()}

        for feat in chosen_layer.getFeatures():
            segment_ids = str(feat['segment_id'])
            if segment_ids not in path_features:
                continue

            path_feat = path_features[segment_ids]
            lunghezza = float(path_feat[length_field])
            larghezza_strada = float(path_feat[width_field])
            w_ab = float(path_feat[widthAB_field])
            w_ba = float(path_feat[widthBA_field])

            try:
                val_ab = path_feat[treeAB_field]
                val_ba = path_feat[treeBA_field]
            except KeyError:
                QMessageBox.warning(
                    self,
                    "Data error",
                    "Ensure that you have correctly selected the tree attributes (AB/BA) in the drop-down menu\n"
                    "Invalid value: ‘Select attribute’",
                    QMessageBox.Ok
                )
                return

            alberi_ab = int(val_ab) if val_ab is not None and val_ab != QVariant() else 0
            alberi_ba = int(val_ba) if val_ba is not None and val_ba != QVariant() else 0

            direzione = path_feat[direction_field].strip().upper()

            interventi = []
            costi = []
            descrizione = []
            spostamento_alberi_ab = "no action"
            spostamento_alberi_ba = "no action"

            # Adattamenti larghezza marciapiedi in base ad alberi
            extra_ab = 0.8 if alberi_ab > 0 else 0
            extra_ba = 0.8 if alberi_ba > 0 else 0

            w_ab_eff = w_ab + extra_ab
            w_ba_eff = w_ba + extra_ba
            larghezza_corsia = larghezza_strada - (w_ab_eff + w_ba_eff)

            # 🔹 memorizzo la corsia iniziale per calcolare il delta a fine logica
            corsia_init = larghezza_corsia

            soglia_corsia = 3.8 if direzione in ["AB", "BA"] else 6.6

            intervento_corsia = "no action"
            intervento_ab = "no action"
            intervento_ba = "no action"

            modifica_ab = 0
            modifica_ba = 0
            modifica_corsia = 0

            # LOGICA CORSIA
            if larghezza_corsia < soglia_corsia:
                # Tenta allargamento marciapiedi a 0.9
                target_ab = max(w_ab, 0.9)
                target_ba = max(w_ba, 0.9)
                new_corsia = larghezza_strada - (target_ab + target_ba + extra_ab + extra_ba)
                if new_corsia >= soglia_corsia:
                    modifica_ab = target_ab - w_ab
                    modifica_ba = target_ba - w_ba
                    w_ab = target_ab
                    w_ba = target_ba
                    larghezza_corsia = new_corsia
                else:
                    if extra_ab or extra_ba:
                        new_corsia2 = larghezza_strada - (w_ab + w_ba)
                        if new_corsia2 >= soglia_corsia:
                            spostamento_alberi_ab = "-0.8 m" if extra_ab else "no action"
                            spostamento_alberi_ba = "-0.8 m" if extra_ba else "no action"
                            larghezza_corsia = new_corsia2
                        else:
                            if w_ab >= 0.9 and w_ba >= 0.9:
                                if alberi_ab == 0:
                                    w_ab = 0
                                    modifica_ab = -w_ab
                                else:
                                    w_ba = 0
                                    modifica_ba = -w_ba
                                larghezza_corsia = larghezza_strada - (w_ab + w_ba)
                            else:
                                descrizione.append("No action possible, insufficient lane width")
            # LOGICA MARCIAPIEDI
            if w_ab >= 0.9 and w_ba >= 0.9:
                pass
            else:
                total_w = w_ab + w_ba
                spazio_corsia = larghezza_strada - (0.9 + 0.9)
                if spazio_corsia >= soglia_corsia:
                    if w_ab < 0.9:
                        modifica_ab = 0.9 - w_ab
                        w_ab = 0.9
                    if w_ba < 0.9:
                        modifica_ba = 0.9 - w_ba
                        w_ba = 0.9
                else:
                    if alberi_ab > 0 and w_ab < 0.9:
                        w_ab += 0.8
                        spostamento_alberi_ab = "-0.8 m"
                    if alberi_ba > 0 and w_ba < 0.9:
                        w_ba += 0.8
                        spostamento_alberi_ba = "-0.8 m"

            # COSTI E DESCRIZIONE
            costo_totale = 0
            if modifica_ab != 0:
                intervento_ab = ("+" if modifica_ab > 0 else "-") + f"{abs(modifica_ab):.2f} m"
                costo_totale += abs(modifica_ab + w_ab) * lunghezza * sidewalk_material_cost
            if modifica_ba != 0:
                intervento_ba = ("+" if modifica_ba > 0 else "-") + f"{abs(modifica_ba):.2f} m"
                costo_totale += abs(modifica_ba + w_ba) * lunghezza * sidewalk_material_cost
            if modifica_corsia != 0:
                intervento_corsia = ("+" if modifica_corsia > 0 else "-") + f"{abs(modifica_corsia):.2f} m"
                # 🔹 paga solo il DELTA corsia (m) * lunghezza (m) * €/m²
                costo_totale += abs(modifica_corsia) * lunghezza * street_material_cost

            # --- Ricalcolo finale coerente della carreggiata ---
            extra_ab_final = 0.0 if spostamento_alberi_ab != "no action" else (0.8 if alberi_ab > 0 else 0.0)
            extra_ba_final = 0.0 if spostamento_alberi_ba != "no action" else (0.8 if alberi_ba > 0 else 0.0)

            # larghezza corsia DOPO tutti gli interventi
            larghezza_corsia_final = larghezza_strada - (w_ab + w_ba + extra_ab_final + extra_ba_final)

            # 🔹 delta corsia (positivo = allargamento, negativo = restringimento)
            modifica_corsia = larghezza_corsia_final - corsia_init

            # mantengo una variabile con il valore finale (usata anche dopo)
            larghezza_corsia = larghezza_corsia_final

            # OUTPUT TABLE
            row = self.ScenarioMinimoResults_tableWidget.rowCount()
            self.ScenarioMinimoResults_tableWidget.insertRow(row)
            self.ScenarioMinimoResults_tableWidget.setItem(row, 0, QTableWidgetItem(segment_ids))
            self.ScenarioMinimoResults_tableWidget.setItem(row, 1, QTableWidgetItem(f"{costo_totale:.2f}"))
            descrizione_text = f"Lane Width: {intervento_corsia}; Sidewalk width AB: {intervento_ab}; Sidewalk width BA: {intervento_ba}; AB Tree Band Shift: {spostamento_alberi_ab}; BA Tree Band Shift: {spostamento_alberi_ba}"

            self.ScenarioMinimoResults_tableWidget.setItem(row, 2, QTableWidgetItem(descrizione_text))

            # SALVA MODIFICHE PER USO FUTURO
            self.minimum_scenario_modifications[segment_ids] = {
                "street_width": larghezza_corsia,
                "sidewalk_AB": w_ab,
                "sidewalk_BA": w_ba
            }

        # Verifica superamento budget
        try:
            user_budget_str = self.AvailableBudget_lineEdit.text()
            user_budget = float(user_budget_str.replace(",", "."))
        except ValueError:
            QMessageBox.warning(self, "Warning", "The value of the available budget is invalid")
            return

        # Somma tutti i costi dalla tabella
        total_cost = 0
        row_count = self.ScenarioMinimoResults_tableWidget.rowCount()
        for i in range(row_count):
            try:
                cost_str = self.ScenarioMinimoResults_tableWidget.item(i, 1).text()
                cost = float(cost_str)
                total_cost += cost
            except (ValueError, AttributeError):
                continue

        if total_cost > user_budget:
            QMessageBox.warning(
                self,
                "Insufficient budget",
                f"The total cost of the interventions ({total_cost:.2f} €) exceeds the available budget ({user_budget:.2f} €).",
                QMessageBox.Ok
            )

        self.CostoTotaleInterventiMinimiVALORE_label.setText(f"{total_cost:.2f} €")
        residuo = (user_budget - total_cost)
        self.ScostamentoInterventiMinimiVALORE_label.setText(f"{residuo:.2f} €")

        if self.EnvironmentalScenario_radioButton.isChecked():
            self.calculate_environmental_scenario(residuo)
        elif self.PhysicalScenario_radioButton.isChecked():
            self.calculate_physical_scenario(residuo)
        elif self.HybridScenario_radioButton.isChecked():
            self.calculate_hybrid_scenario(residuo)

        # Mostra solo la tabella relativa al tipo di scenario selezionato
        self.ScenariosEnvironmentalResults_tableWidget.setVisible(False)
        self.ScenariosPhysicalResults_tableWidget.setVisible(False)
        self.ScenariosHybridResults_tableWidget.setVisible(False)

        if self.EnvironmentalScenario_radioButton.isChecked():
            self.ScenariosEnvironmentalResults_tableWidget.setVisible(True)
        elif self.PhysicalScenario_radioButton.isChecked():
            self.ScenariosPhysicalResults_tableWidget.setVisible(True)
        elif self.HybridScenario_radioButton.isChecked():
            self.ScenariosHybridResults_tableWidget.setVisible(True)

    def calculate_environmental_scenario(self, budget_residuo: float):
        self.ScenariosEnvironmentalResults_tableWidget.setRowCount(0)

        get_val = lambda field: float(field.text().replace(",", ".")) if field.text() else 0

        # Costi inseriti dall'utente
        tree_cost = get_val(self.TreeStreetFurnitureCosts_lineEdit)

        gsi_costs = {
            'Planter': get_val(self.PlanterGSICosts_lineEdit),
            'TreeTrench': get_val(self.TreeTrenchGSICosts_lineEdit),
            'Gutter': get_val(self.GutterGSICosts_lineEdit),
            'RainGarden': get_val(self.RainGardenGSICosts_lineEdit),
            'Swale': get_val(self.SwaleGSICosts_lineEdit)
        }

        parking_costs = {
            'Asphalt': get_val(self.AsphaltStreetCost_lineEdit),
            'Cement': get_val(self.CementStreetCost_lineEdit),
            'Basalt': get_val(self.BasaltStreetCost_lineEdit),
            'Stone': get_val(self.StoneStreetCost_lineEdit)
        }

        street_material_cost = next((v for v in parking_costs.values() if v > 0), 0)

        self.environmental_scenario_modifications = {}

        for segment_id, modifiche in self.minimum_scenario_modifications.items():
            
            street_width = modifiche['street_width']
            sidewalk_ab = modifiche['sidewalk_AB']
            sidewalk_ba = modifiche['sidewalk_BA']

            length_field = self.PathLength_comboBox.currentText()
            width_field = self.PathWidth_comboBox.currentText()
            id_field = self.ID_comboBox.currentText()
            direction_field = self.PathDirection_comboBox.currentText()
            tree_ab_field = self.TreeNumberAB_comboBox.currentText()
            tree_ba_field = self.TreeNumberBA_comboBox.currentText()

            layer_id = self.PathLayer_comboBox.currentData()
            path_layer = QgsProject.instance().mapLayer(layer_id)
            path_feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)
            if not path_feat:
                continue

            length = float(path_feat[length_field])
            path_width = float(path_feat[width_field])
            direction = (path_feat[direction_field] or "").strip().upper()
            trees_ab = int(path_feat[tree_ab_field]) if path_feat[tree_ab_field] not in [None, '', QVariant()] else 0
            trees_ba = int(path_feat[tree_ba_field]) if path_feat[tree_ba_field] not in [None, '', QVariant()] else 0
            with_trees = trees_ab > 0 or trees_ba > 0

            water_volume_field = self.WaterStormwaterVolume_comboBox.currentText()
            try:
                water_volume = float(path_feat[water_volume_field])
            except (KeyError, TypeError, ValueError):
                water_volume = 0

            sup_min_gsi = water_volume / (1.5 * 0.35)
            permeabilita = (sup_min_gsi / (path_width * length)) * 100 if length * path_width != 0 else 0

            areaPerGSI = street_width - (3.8 if direction in ["AB", "BA"] else 6.6) + 0.5

            # Scegli il miglior GSI possibile
            candidates = [(name, spec) for name, spec in GSI_DEFINITIONS.items() if spec['width'] <= areaPerGSI]
            if not candidates:
                chosen_gsi_name, chosen_gsi_width, chosen_gsi_length = "-", 0, 0
                costo_gsi = 0
            else:
                candidates.sort(key=lambda x: -x[1]['width'])
                chosen_gsi_name, chosen_gsi_spec = candidates[0]
                chosen_gsi_width = chosen_gsi_spec['width']
                # chosen_gsi_length calcolo
                unit_min = chosen_gsi_spec['length_min']
                unit_max = chosen_gsi_spec['length_max']

                required_length = sup_min_gsi / chosen_gsi_width
                num_max_units = int(required_length // unit_max)
                residual = required_length % unit_max

                if residual >= unit_min:
                    chosen_gsi_length = (num_max_units * unit_max) + unit_max
                elif residual == 0:
                    chosen_gsi_length = (num_max_units * unit_max)
                else:
                    chosen_gsi_length = (num_max_units * unit_max) + unit_min

                costo_gsi = chosen_gsi_length * chosen_gsi_width * gsi_costs.get(chosen_gsi_name, 0)

            # Verifica se il budget è sufficiente per il GSI
            if costo_gsi > budget_residuo:
                print(f"Insufficient budget for GSI on segment {segment_id}")
                continue
            budget_residuo -= costo_gsi

            # Calcola Alberi
            alberi = int((length - chosen_gsi_length) // 6)
            costo_alberi = alberi * tree_cost
            if costo_alberi > budget_residuo:
                alberi = 0
                costo_alberi = 0
            else:
                budget_residuo -= costo_alberi

            # Calcola Parcheggi
            parcheggi = int((length - chosen_gsi_length - (alberi * 0.8)) // 5)
            costo_parcheggi = parcheggi * 10 * street_material_cost
            if costo_parcheggi > budget_residuo:
                parcheggi = 0
                costo_parcheggi = 0
            else:
                budget_residuo -= costo_parcheggi

            descrizione = (
                f"GSI : {chosen_gsi_name}, {chosen_gsi_length:.2f} m; "
                f"Trees: {alberi}; "
                f"Parkings: {parcheggi}"
            )

            costo_totale = costo_gsi + costo_alberi + costo_parcheggi

            # Calcola Benefici, VAN, Payback
            try:
                t_max = int(float(self.EstimationTime_lineEdit.text()))
            except (AttributeError, ValueError):
                t_max = 20  # fallback

            # Nuova Copertura Arborea (%)
            tree_cover_field = self.MicroclimateTreeCover_comboBox.currentText()
            existing_cover = float(path_feat[tree_cover_field]) if path_feat[tree_cover_field] else 0

            # Calcola nuova copertura per alberi aggiunti (in %)
            segment_area = path_width * length

            # Trova il diametro giusto per t_max
            diametro_selezionato = 0
            for diam, t in sorted(CROWN_DIAMETER_MAPPING.items(), key=lambda x: x[1]):
                if t <= t_max:
                    diametro_selezionato = diam
                else:
                    break

            added_trees_area = alberi * (3.1415 * (diametro_selezionato / 2) ** 2)

            added_tree_cover = (added_trees_area / segment_area) * 100 if segment_area else 0
            nuova_copertura = existing_cover + added_tree_cover

            # Calcola stoccaggio CO2 (tCO2)
            stoccaggio_co2 = 0.28 * nuova_copertura * 0.40

            # Costi di manutenzione annuali
            gsi_area = chosen_gsi_length * chosen_gsi_width
            costo_manutenzione_verde = gsi_area * get_val(self.VegetationAnnualMaintenanceCosts_lineEdit)

            larghezza_corsia_finale = street_width
            costo_manutenzione_strada = (
                                                (get_val(
                                                    self.StreetAnnualMaintenanceCosts_lineEdit) * larghezza_corsia_finale) +
                                                (get_val(self.SidewalkAnnualMaintenanceCosts_lineEdit) * (
                                                            sidewalk_ab + sidewalk_ba))
                                        ) * length

            # Calcolo VAN e Payback
            van = 0
            benefici_totali = 0
            payback_anno = "NULL"

            for anno in range(t_max + 1):
                discount = (1 + 0.03) ** anno
                co2_annua = stoccaggio_co2  # semplificazione: stesso valore ogni anno
                costi_annui = costo_manutenzione_verde
                if anno % 10 == 0:
                    costi_annui += costo_manutenzione_strada
                benefici_totali += co2_annua
                flusso = co2_annua - costi_annui
                van += flusso / discount
                if payback_anno == "NULL" and van > 0:
                    payback_anno = anno

            # === ΔEA (ambientale) — allineato allo scenario ibrido ===
            # helper punteggi per PET e Vento (come nell’ibrido)
            def _pet_access(val):
                thresholds = [4, 8, 13, 18, 20, 21, 23, 29, 35, 41]
                scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
                for t, s in zip(thresholds, scores):
                    if val <= t: return s
                return 0

            def _wind_access(val):
                thresholds = [0, 0.3, 0.9, 1.9, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5]
                scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
                for t, s in zip(thresholds, scores):
                    if val <= t: return s
                return 0

            # campi “12:00” come nell’ibrido
            pet12_f = self.MicroclimatePET12_comboBox.currentText()
            wind12_f = self.MicroclimateWind12_comboBox.currentText()
            shadow12_f = self.MicroclimateShadow12_comboBox.currentText()
            svf_f = self.MicroclimateSkyViewFactor_comboBox.currentText()
            tc_f = self.MicroclimateTreeCover_comboBox.currentText()
            flood_f = self.WaterFloodingRisk_comboBox.currentText()

            # valori "prima"
            PET_b = float(path_feat[pet12_f]) if path_feat[pet12_f] not in [None, '', QVariant()] else 0.0
            W_b = float(path_feat[wind12_f]) if path_feat[wind12_f] not in [None, '', QVariant()] else 0.0
            SH_b = float(path_feat[shadow12_f]) if path_feat[shadow12_f] not in [None, '', QVariant()] else 0.0
            SVF_b = float(path_feat[svf_f]) if path_feat[svf_f] not in [None, '', QVariant()] else 0.0
            FL_raw = path_feat[flood_f] if flood_f else None

            # UTC "prima" come frazione 0–1 (accetta % o frazione dal campo TreeCover)
            TC_b_raw = float(path_feat[tc_f]) if path_feat[tc_f] not in [None, '', QVariant()] else 0.0
            UTC_b = TC_b_raw / 100.0 if TC_b_raw > 1.0 else TC_b_raw

            # --- punteggi "prima" (EA_before) ---
            PET_b_s = _pet_access(PET_b)
            W_b_s = _wind_access(W_b)
            SH_b_s = self._binary_access(SH_b)
            SVF_b_s = self._sky_tree_access(SVF_b)
            FLO_b_s = self._flooding_access(FL_raw)

            # ✅ UTC prima: scala 0–5 dentro la media EA
            UTC_b_s = self._sky_tree_access(UTC_b)

            EA_before = (PET_b_s + W_b_s + FLO_b_s + UTC_b_s + SH_b_s + SVF_b_s) / 6.0

            # --- alberi inseriti: quelli “singoli” + quelli provenienti dalla GSI con alberi
            trees_from_gsi = int(chosen_gsi_length // 6) if GSI_DEFINITIONS.get(chosen_gsi_name, {}).get("with_trees",
                                                                                                         False) else 0
            n_trees = max(0, int(alberi) + int(trees_from_gsi))

            # area di segmento e incremento UTC (come nell’ibrido)
            import math
            segment_area = max(1e-6, path_width * length)

            # scegliamo il diametro coerente con t_max già calcolato più su (diametro_selezionato)
            area_chioma = n_trees * (math.pi * (diametro_selezionato ** 2) / 4.0)

            UTC_a = min(1.0, UTC_b + (area_chioma / segment_area))  # ⬅️ frazione 0–1

            # --- PET dopo (usa 0–1) ---
            deltaPET = -10.0 * (UTC_a - UTC_b)
            PET_a = PET_b + deltaPET
            PET_a_s = _pet_access(PET_a)

            # --- Vento dopo (usa 0–1) ---
            R = (0.7 * UTC_a) ** 1.2
            W_a = max(0.0, W_b * (1.0 - R))
            W_a_s = _wind_access(W_a)

            # --- Flood dopo (se c’è una GSI) ---
            FLO_a_s = FLO_b_s
            if chosen_gsi_name != "-":
                raw = str(FL_raw)
                new_raw = "Basso" if raw == "Medio" else ("Medio" if raw == "Alto" else raw)
                FLO_a_s = self._flooding_access(new_raw)

            # ✅ UTC dopo: nella media EA uso la scala 0–5
            UTC_a_s = self._sky_tree_access(UTC_a)

            # Nota: Ombra (SH) e SVF, Attività restano invariati come nell’ibrido
            EA_after = (PET_a_s + W_a_s + FLO_a_s + UTC_a_s + SH_b_s + SVF_b_s) / 6.0
            delta_ea = EA_after - EA_before

            # Aggiungi i valori alla tabella
            self.ScenariosEnvironmentalResults_tableWidget.insertRow(self.ScenariosEnvironmentalResults_tableWidget.rowCount())
            row_hybrid = self.ScenariosEnvironmentalResults_tableWidget.rowCount() - 1
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 0, QTableWidgetItem("Env_Scenario"))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 1, QTableWidgetItem(str(segment_id)))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 2, QTableWidgetItem(f"{costo_totale:.2f}"))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 3, QTableWidgetItem(f"{delta_ea:+.3f}"))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 4, QTableWidgetItem(descrizione))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 5, QTableWidgetItem(f"{permeabilita:.2f}"))
            self.ScenariosEnvironmentalResults_tableWidget.setItem(row_hybrid, 6, QTableWidgetItem(f"{sup_min_gsi:.2f}"))

            # --- ▶ SALVA DATI PER EXPORT CSV (scenario ambientale) ◀ ---
            if not hasattr(self, "environmental_scenario_modifications"):
                self.environmental_scenario_modifications = {}

            # alberi inseriti dal GSI (se previsto)
            trees_from_gsi = int(chosen_gsi_length // 6) if GSI_DEFINITIONS.get(chosen_gsi_name, {}).get("with_trees",
                                                                                                         False) else 0
            trees_added_total = alberi + trees_from_gsi

            # --- variabili "prima/dopo" coerenti con l'ibrido (UTC in 0–1) ---
            # prima: existing_cover può essere in % (0–100) o frazione (0–1)
            TC_b_raw = existing_cover
            UTC_b = (TC_b_raw / 100.0) if TC_b_raw is not None and TC_b_raw > 1 else float(TC_b_raw or 0.0)

            # dopo: nuova_copertura è già in %, ricaviamo UTC_a 0–1
            TC_a_pct = max(0.0, min(100.0, float(nuova_copertura)))
            UTC_a = TC_a_pct / 100.0

            # PET e Vento "dopo" coerenti con l'ibrido
            PET_a = PET_b + (-10.0 * (UTC_a - UTC_b))
            W_a = max(0.0, W_b * (1.0 - (0.7 * UTC_a) ** 1.2))

            self.environmental_scenario_modifications[segment_id] = {
                # costi
                "cost_env": float(costo_totale),
                # GSI
                "gsi_name": chosen_gsi_name,
                "gsi_len_m": float(chosen_gsi_length),
                "gsi_width_m": float(chosen_gsi_width),
                "gsi_area_m2": float(chosen_gsi_length * chosen_gsi_width),
                "permeability_pct": float(permeabilita),
                # alberi
                "trees_added": int(trees_added_total),
                # baseline microclima (12:00)
                "pet_before": float(PET_b) if 'PET_b' in locals() else 0.0,
                "wind_before": float(W_b) if 'W_b' in locals() else 0.0,
                "utc_before_pct": float(UTC_b * 100.0),
                "pet_after": float(PET_a),
                "wind_after": float(W_a),
                "utc_after_pct": float(UTC_a * 100.0),

                # EA
                "ea_before": float(EA_before),
                "ea_after": float(EA_after),
                "delta_ea": float(delta_ea),
            }

        try:
            user_budget_str = self.AvailableBudget_lineEdit.text()
            user_budget = float(user_budget_str.replace(",", "."))
        except ValueError:
            QMessageBox.warning(self, "Warning", "The available budget value is invalid")
            return

        # Calcola costo totale dalle due tabelle: ScenarioMinimoResults + ScenariosEnvironmentalResults
        total_cost_minimi = 0
        for i in range(self.ScenarioMinimoResults_tableWidget.rowCount()):
            try:
                cost_str = self.ScenarioMinimoResults_tableWidget.item(i, 1).text()
                total_cost_minimi += float(cost_str)
            except (ValueError, AttributeError):
                continue

        total_cost_ambientali = 0
        for i in range(self.ScenariosEnvironmentalResults_tableWidget.rowCount()):
            try:
                cost_str = self.ScenariosEnvironmentalResults_tableWidget.item(i, 2).text()
                total_cost_ambientali += float(cost_str)
            except (ValueError, AttributeError):
                continue

        # Calcola il totale complessivo
        total_cost = total_cost_minimi + total_cost_ambientali
        scostamento = user_budget - total_cost

        # Aggiorna le label
        self.CostoTotaleInterventiScenariVALORE_label.setText(f"{total_cost:.2f} €")
        self.ScostamentoInterventiScenariVALORE_label.setText(f"{scostamento:.2f} €")

        QMessageBox.information(self, "Done", "Scenario generated successfully")
        self.apply_row_colors_by_scenario(self.ScenariosEnvironmentalResults_tableWidget)

    def calculate_physical_scenario(self, budget_residuo: float):
        # Pulisci tabella fisico
        self.ScenariosPhysicalResults_tableWidget.setRowCount(0)

        # --- Costi da UI ---
        get_val = lambda w: float(w.text().replace(",", ".")) if getattr(w, "text", None) and w.text() else 0.0
        cost_bench = get_val(self.BenchesStreetFurnitureCosts_lineEdit)
        cost_bin = get_val(self.BasketsStreetFurnitureCosts_lineEdit)
        cost_fountain = get_val(self.FountainsStreetFurnitureCosts_lineEdit)
        cost_bikerack = get_val(self.BikeCarrierStreetFurnitureCosts_lineEdit)
        cost_cycle_m = get_val(self.CyclePathCost_lineEdit)

        # --- Layer/campi base ---
        layer_id = self.PathLayer_comboBox.currentData()
        path_layer = QgsProject.instance().mapLayer(layer_id)
        if not isinstance(path_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Error", "Invalid path attribute layer")
            return

        id_field = self.ID_comboBox.currentText()
        length_field = self.PathLength_comboBox.currentText()
        direction_field = self.PathDirection_comboBox.currentText()

        # Itera i segmenti dello scenario minimo (già dimensionati)
        for segment_id, modifiche in getattr(self, "minimum_scenario_modifications", {}).items():
            feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)
            if not feat:
                continue

            try:
                length = float(feat[length_field])
            except (TypeError, ValueError, KeyError):
                length = 0.0

            street_width = float(modifiche.get("street_width", 0.0))  # carreggiata disponibile dopo lo scenario minimo
            dir_value = str(feat[direction_field]) if direction_field else ""

            # ==== A) CICLABILE (prioritaria) ====
            cycle_w, cycle_cost, cycle_label = self._cycle_choice(street_width, dir_value, length, cost_cycle_m)
            min_carreggiata = 3.8 if (dir_value or "").strip().upper() in ["AB", "BA"] else 6.6

            descr_parts = []
            ciclo_posata = False
            if cycle_w > 0 and (cycle_cost <= budget_residuo) and (street_width - cycle_w >= min_carreggiata):
                budget_residuo -= cycle_cost
                street_width_after_cycle = street_width - cycle_w
                descr_parts.append(f"{cycle_label} (w={cycle_w} m, costo={cycle_cost:.2f} €)")
                ciclo_posata = True
            else:
                street_width_after_cycle = street_width
                if cycle_w == 0:
                    descr_parts.append("No cycle path")
                else:
                    if street_width - cycle_w < min_carreggiata:
                        descr_parts.append(f"No cycle path: insufficient section (min {min_carreggiata} m)")
                    elif cycle_cost > budget_residuo:
                        descr_parts.append("No cycle path: insufficient budget")

            # ==== B) ARREDI (solo sezione e budget; marciapiedi NON si toccano) ====
            def place_furniture(kind: str, unit_cost: float):
                spec = FURNITURE_SPECS[kind]
                depth = spec["depth_m"]
                spacing = spec["min_spacing_m"]

                # Vincolo trasversale (carreggiata minima per il senso)
                if (street_width_after_cycle - depth) < min_carreggiata:
                    return 0, 0.0, f"{kind}: not included (insufficient section)."

                # Capienza longitudinale
                n_max = self._max_items_by_spacing(length, spacing)
                if n_max <= 0:
                    return 0, 0.0, f"{kind}: not included (lenght/spacing)."

                # Vincolo budget
                n_budget = int(budget_residuo // unit_cost) if unit_cost > 0 else n_max
                n_place = min(n_max, n_budget)
                if n_place <= 0:
                    return 0, 0.0, f"{kind}: not included (insufficient budget)."

                costo = n_place * unit_cost
                return n_place, costo, f"{kind}: {n_place} pieces (spacing {spacing} m, cost={costo:.2f} €)"

            total_cost_segment = 0.0

            # Accumulatore ΔPA (coerente con l’ibrido)
            delta_pa = 0.0

            def _dpa_from_density(n, length_m):
                # ((n/length)*100)/1000*5  -> stessa logica usata nell’ibrido
                return (((n / length_m) * 100.0) / 1000.0 * 5.0) if (length_m and n > 0) else 0.0

            # Panchine
            n_bench, c_bench, msg_bench = place_furniture("bench", cost_bench)
            if n_bench > 0:
                budget_residuo -= c_bench
                total_cost_segment += c_bench
                delta_pa += _dpa_from_density(n_bench, length)

            descr_parts.append(msg_bench)

            # Cestini
            n_bin, c_bin, msg_bin = place_furniture("waste_bin", cost_bin)
            if n_bin > 0:
                budget_residuo -= c_bin
                total_cost_segment += c_bin
                delta_pa += _dpa_from_density(n_bin, length)

            descr_parts.append(msg_bin)

            # Fontane
            n_fountain, c_fountain, msg_fountain = place_furniture("fountain", cost_fountain)
            if n_fountain > 0:
                budget_residuo -= c_fountain
                total_cost_segment += c_fountain
                delta_pa += _dpa_from_density(n_fountain, length)

            descr_parts.append(msg_fountain)

            # Portabici
            n_rack, c_rack, msg_rack = place_furniture("bike_rack", cost_bikerack)
            if n_rack > 0:
                budget_residuo -= c_rack
                total_cost_segment += c_rack
                delta_pa += _dpa_from_density(n_rack, length)

            descr_parts.append(msg_rack)

            if ciclo_posata:
                total_cost_segment += cycle_cost
                # ΔPA per ciclabile (come nell’ibrido)
                delta_pa += (100.0 / 1000.0) * 5.0  # = 0.5

            # Scrittura tabella fisico
            # 0=Scenario, 1=Segmento, 2=Costo, 3=VAN, 4=Payback, 5=Benefici, 6=Δ, 7=Descrizione, 8=Permeabilità, 9=SupMinGSI
            row = self.ScenariosPhysicalResults_tableWidget.rowCount()
            self.ScenariosPhysicalResults_tableWidget.insertRow(row)

            self.ScenariosPhysicalResults_tableWidget.setItem(row, 0, QTableWidgetItem("Phy_Scenario"))
            self.ScenariosPhysicalResults_tableWidget.setItem(row, 1, QTableWidgetItem(str(segment_id)))
            self.ScenariosPhysicalResults_tableWidget.setItem(row, 2, QTableWidgetItem(f"{total_cost_segment:.2f}"))

            # ΔPA calcolato (coerente con l’ibrido)
            self.ScenariosPhysicalResults_tableWidget.setItem(row, 3, QTableWidgetItem(f"ΔPA=+{delta_pa:.3f}"))

            # Descrizione interventi → COLONNA 7 (giusta!)
            self.ScenariosPhysicalResults_tableWidget.setItem(row, 4, QTableWidgetItem(" | ".join(descr_parts)))

        # === Totali & scostamento (come nello scenario ambientale) ===
        try:
            user_budget = float(self.AvailableBudget_lineEdit.text().replace(",", "."))
        except ValueError:
            QMessageBox.warning(self, "Warning", "The available budget value is invalid")
            return

        total_cost_minimi = 0.0
        for i in range(self.ScenarioMinimoResults_tableWidget.rowCount()):
            try:
                total_cost_minimi += float(self.ScenarioMinimoResults_tableWidget.item(i, 1).text())
            except Exception:
                pass

        total_cost_fisico = 0.0
        for i in range(self.ScenariosPhysicalResults_tableWidget.rowCount()):
            try:
                total_cost_fisico += float(self.ScenariosPhysicalResults_tableWidget.item(i, 2).text())
            except Exception:
                pass

        total_cost = total_cost_minimi + total_cost_fisico
        scostamento = user_budget - total_cost
        self.CostoTotaleInterventiScenariVALORE_label.setText(f"{total_cost:.2f} €")
        self.ScostamentoInterventiScenariVALORE_label.setText(f"{scostamento:.2f} €")

        QMessageBox.information(self, "Done", "Physical scenario generated correctly")
        self.apply_row_colors_by_scenario(self.ScenariosPhysicalResults_tableWidget)

    def calculate_hybrid_scenario(self, budget_residuo: float):
        """
        Unica tabella ibrida con selezione greedy degli interventi (ambientali + fisici)
        massimizzando (w_phys*ΔPA + w_env*ΔEA)/costo e rispettando vincoli geometrici.
        """
        # pulisci tabella ibrida
        self.ScenariosHybridResults_tableWidget.setRowCount(0)

        # --- costi da UI ---
        val = lambda w: self._safe_float(w.text().replace(",", ".")) if getattr(w, "text", None) and w.text() else 0.0
        # fisico
        cost_bench    = val(self.BenchesStreetFurnitureCosts_lineEdit)
        cost_bin      = val(self.BasketsStreetFurnitureCosts_lineEdit)
        cost_fountain = val(self.FountainsStreetFurnitureCosts_lineEdit)
        cost_bikerack = val(self.BikeCarrierStreetFurnitureCosts_lineEdit)
        cost_cycle_m  = val(self.CyclePathCost_lineEdit)
        # ambientale
        cost_tree     = val(self.TreeStreetFurnitureCosts_lineEdit)
        gsi_costs = {
            "Planter":      val(self.PlanterGSICosts_lineEdit),
            "Tree Trench":  val(self.TreeTrenchGSICosts_lineEdit),
            "Gutter":       val(self.GutterGSICosts_lineEdit),
            "Rain Garden":  val(self.RainGardenGSICosts_lineEdit),
            "Swale":        val(self.SwaleGSICosts_lineEdit),
        }

        # layer/campi base
        path_layer_id = self.PathLayer_comboBox.currentData()
        path_layer = QgsProject.instance().mapLayer(path_layer_id)
        if not isinstance(path_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Error", "Invalid path attribute layer")
            return

        id_field        = self.ID_comboBox.currentText()
        length_field    = self.PathLength_comboBox.currentText()
        width_field     = self.PathWidth_comboBox.currentText()
        dir_field       = self.PathDirection_comboBox.currentText()
        tree_cover_fld  = self.MicroclimateTreeCover_comboBox.currentText()
        shadow12_field  = self.MicroclimateShadow12_comboBox.currentText()
        flood_field     = self.WaterFloodingRisk_comboBox.currentText()
        water_vol_field = self.WaterStormwaterVolume_comboBox.currentText()

        # pesi fisico/ambientale
        w_phys, w_env = self._hybrid_weights()

        # orizzonte per diametro-corona
        try:
            t_max = int(self._safe_float(self.EstimationTime_lineEdit.text(), 20))
        except Exception:
            t_max = 20

        # diametro-corona in base a t_max
        diametro = 0
        for d, t in sorted(CROWN_DIAMETER_MAPPING.items(), key=lambda x: x[1]):
            if t <= t_max: diametro = d
            else: break

        candidates = []  # lista di dict

        # scansiona i segmenti dello scenario minimo (sezioni consolidate)
        for segment_id, mod in getattr(self, "minimum_scenario_modifications", {}).items():

            feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)

            if not feat:
                continue

            length   = self._safe_float(feat[length_field])
            if length <= 0:
                continue

            path_w   = self._safe_float(feat[width_field])
            street_w = self._safe_float(mod.get("street_width", 0.0))  # carreggiata utile dopo "minimo"
            sw_ab    = self._safe_float(mod.get("sidewalk_AB", 0.0))
            sw_ba    = self._safe_float(mod.get("sidewalk_BA", 0.0))
            direction= (str(feat[dir_field]) if dir_field else "").strip().upper()
            min_car  = 3.8 if direction in ["AB","BA"] else 6.6

            # base per EA
            existing_cover = self._safe_float(feat[tree_cover_fld])
            existing_shadow_val = self._safe_float(feat[shadow12_field])
            existing_shadow_score = self._binary_access(existing_shadow_val)
            flood_raw = feat[flood_field] if flood_field else None
            existing_flood_score = self._flooding_access(flood_raw)

            # ---- A) Candidati fisici ----
            # ciclabile (DS poi SU)
            for w_cyc, label in [(CYCLE_DOUBLE_W, "Two-way Cycle Path"), (CYCLE_SINGLE_W, "One-way Cycle Path")]:
                if street_w - w_cyc >= min_car and cost_cycle_m > 0:
                    cost_cycle = cost_cycle_m * length
                    # ΔPA: contribuisco solo alla componente bicycle (100% sul segmento)
                    d_bicycle = 100.0
                    dPA = (d_bicycle) / 1000.0 * 5.0
                    dEA = 0.0
                    beneficio = w_phys * dPA + w_env * dEA
                    score = self._clean_score(beneficio / cost_cycle) if cost_cycle > 0 else 0.0
                    candidates.append({
                        "segment_ids": segment_id,
                        "costo": cost_cycle,
                        "dPA": dPA, "dEA": dEA,
                        "descr": f"{label} (w={w_cyc} m)",
                        "tag": "phys_cycle",
                        "score": score,
                        "mut": {"use_width": w_cyc, "type": "carriage"}
                    })

            # arredi
            def arredo_candidates(kind, unit_cost, spacing_m):
                if unit_cost <= 0: return []
                depth = FURNITURE_SPECS[kind]["depth_m"]
                if (street_w - depth) < min_car:
                    return []
                # capienza longitudinale
                n_max = self._max_items_by_spacing(length, spacing_m)
                out = []
                for n in range(1, n_max + 1):
                    costo = n * unit_cost
                    d_index = (n / length) * 100.0
                    dPA = (d_index) / 1000.0 * 5.0
                    dEA = 0.0
                    score = self._clean_score((w_phys * dPA + w_env * dEA) / costo) if costo > 0 else 0.0
                    out.append({
                        "segment_ids": segment_id,
                        "costo": costo,
                        "dPA": dPA, "dEA": dEA,
                        "descr": f"{kind}: {n} pz (passo {spacing_m} m)",
                        "tag": f"phys_{kind}",
                        "score": score,
                        "mut": {"reserve_long": n * spacing_m, "type": "longitudinal"}
                    })
                return out

            arredi_specs = [
                ("bench",    cost_bench,    FURNITURE_SPECS["bench"]["min_spacing_m"]),
                ("waste_bin",cost_bin,      FURNITURE_SPECS["waste_bin"]["min_spacing_m"]),
                ("fountain", cost_fountain, FURNITURE_SPECS["fountain"]["min_spacing_m"]),
                ("bike_rack",cost_bikerack, FURNITURE_SPECS["bike_rack"]["min_spacing_m"]),
            ]
            for kind, unit_cost, spacing in arredi_specs:
                candidates.extend(arredo_candidates(kind, unit_cost, spacing))

            # ---- B) Candidati ambientali ----
            # GSI (tipo più largo posabile)
            areaPerGSI = street_w - min_car + 0.5
            gsi_cands = [(n, s) for n, s in GSI_DEFINITIONS.items() if s["width"] <= areaPerGSI]
            water_volume = self._safe_float(feat[water_vol_field], 0.0)
            sup_min_gsi = water_volume / (1.5 * 0.35) if water_volume > 0 else 0.0

            if gsi_cands and sup_min_gsi > 0:
                gsi_cands.sort(key=lambda x: -x[1]["width"])
                name, spec = gsi_cands[0]
                width_gsi = spec["width"]
                Lmin, Lmax = spec["length_min"], spec["length_max"]

                required_len = sup_min_gsi / width_gsi
                units_max = int(required_len // Lmax)
                residual = required_len % Lmax
                if residual >= Lmin:
                    chosen_len = (units_max + 1) * Lmax
                elif residual == 0:
                    chosen_len = units_max * Lmax
                else:
                    chosen_len = (units_max * Lmax) + Lmin

                area_gsi = chosen_len * width_gsi
                costo_gsi = area_gsi * gsi_costs.get(name, 0.0)

                dEA = 0.0; delta_comp = 0
                # tree cover + shadow (se con alberi)
                if spec.get("with_trees", False):
                    new_trees = int(chosen_len // 6)
                    seg_area = max(1.0, path_w * length)
                    added_tree_area = new_trees * (3.1415 * (diametro / 2.0) ** 2)
                    new_cover = max(0.0, existing_cover + (added_tree_area / seg_area) * 100.0)
                    old_tc = self._sky_tree_access(existing_cover)
                    new_tc = self._sky_tree_access(new_cover)
                    dEA += max(0, new_tc - old_tc); delta_comp += 1
                    dEA += max(0, 5 - existing_shadow_score); delta_comp += 1

                # flooding piccolo boost
                old_f = existing_flood_score
                new_f = min(5, old_f + 1)
                dEA += max(0, new_f - old_f); delta_comp += 1

                if delta_comp > 0:
                    dEA = dEA / float(delta_comp)
                else:
                    dEA = 0.0

                dPA = 0.0
                beneficio = w_phys * dPA + w_env * dEA
                score = self._clean_score(beneficio / costo_gsi) if costo_gsi > 0 else 0.0

                new_trees_from_gsi = int(chosen_len // 6)

                candidates.append({
                    "segment_ids": segment_id,
                    "costo": costo_gsi,
                    "dPA": dPA, "dEA": dEA,
                    "descr": f"GSI {name} (w={width_gsi} m, L≈{chosen_len:.1f} m, A≈{area_gsi:.1f} m²)",
                    "tag": "env_gsi",
                    "score": score,
                    "mut": {"use_width": width_gsi, "type": "carriage", "reserve_long": chosen_len,
                            "sup_min_gsi": sup_min_gsi},
                    "n_trees": new_trees_from_gsi
                })

            # Alberi extra (passo 6 m) se sezione consente 0.8 m di fascia
            if street_w - 0.8 >= min_car and cost_tree > 0:
                n_trees_max = self._max_items_by_spacing(length, 6.0)
                for n in range(1, n_trees_max + 1):
                    costo = n * cost_tree
                    seg_area = max(1.0, path_w * length)
                    added_tree_area = n * (3.1415 * (diametro / 2.0) ** 2)
                    new_cover = max(0.0, existing_cover + (added_tree_area / seg_area) * 100.0)
                    old_tc = self._sky_tree_access(existing_cover)
                    new_tc = self._sky_tree_access(new_cover)

                    dEA_tc = max(0, new_tc - old_tc)
                    dEA_shadow = max(0, 5 - existing_shadow_score) if n > 0 else 0

                    comp_count = (1 if dEA_tc > 0 else 0) + (1 if dEA_shadow > 0 else 0)
                    if comp_count > 0:
                        dEA = (dEA_tc + dEA_shadow) / float(comp_count)  # media sulle componenti migliorate
                    else:
                        dEA = 0.0

                    dPA = 0.0
                    score = self._clean_score((w_phys * dPA + w_env * dEA) / costo)
                    candidates.append({
                        "segment_ids": segment_id,
                        "costo": costo,
                        "dPA": dPA, "dEA": dEA,
                        "descr": f"Trees: {n} pieces (spacing 6 m)",
                        "tag": "env_trees",
                        "score": score,
                        "mut": {"reserve_long": n * 6.0, "type": "longitudinal"},
                        "n_trees": n  # ⬅️ serve per accorpare dopo
                    })

        # --- Greedy: ordina per beneficio/costo e applica finché c'è budget ---
        state = {"carriage_used": {}, "long_used": {}}

        def can_apply(c):
            seg = c["segment_ids"]
            mut = c.get("mut", {});
            t = mut.get("type")
            if t == "carriage":
                use_w = self._safe_float(mut.get("use_width"), 0.0)
                if use_w <= 0: return True
                used = state["carriage_used"].get(seg, 0.0)
                feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)

                mod = self.minimum_scenario_modifications.get(seg, {})
                if not feat or not mod: return False
                direction = (str(feat[dir_field]) if dir_field else "").strip().upper()
                min_car = 3.8 if direction in ["AB", "BA"] else 6.6
                street_w = self._safe_float(mod.get("street_width", 0.0))
                return (street_w - used - use_w) >= min_car
            elif t == "longitudinal":
                resv = self._safe_float(mut.get("reserve_long"), 0.0)
                if resv <= 0: return True
                used = state["long_used"].get(seg, 0.0)
                feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == seg), None)
                if not feat: return False
                length = self._safe_float(feat[length_field])
                return (used + resv) <= length
            return True

        def apply_mutation(c):
            seg = c["segment_ids"]
            mut = c.get("mut", {});
            t = mut.get("type")
            if t == "carriage":
                use_w = self._safe_float(mut.get("use_width"), 0.0)
                if use_w > 0:
                    state["carriage_used"][seg] = state["carriage_used"].get(seg, 0.0) + use_w
            elif t == "longitudinal":
                resv = self._safe_float(mut.get("reserve_long"), 0.0)
                if resv > 0:
                    state["long_used"][seg] = state["long_used"].get(seg, 0.0) + resv

        candidates.sort(key=lambda c: (-c["score"], c["costo"]))

        applied_by_seg = {}  # segments# -> dict aggregato
        total_cost_hybrid = 0.0
        n_applied = 0

        for c in candidates:
            costo = c["costo"]
            if costo <= 0 or costo > budget_residuo:
                continue
            if not can_apply(c):
                continue

            budget_residuo -= costo
            total_cost_hybrid += costo
            n_applied += 1
            apply_mutation(c)

            seg = c["segment_ids"]
            agg = applied_by_seg.setdefault(seg, {
                "costo": 0.0,
                "dPA": 0.0,
                "dEA": 0.0,
                "descr": [],
                "sup_min_gsi": 0.0,  # per colonna GSI
                "has_gsi": False,
                "trees_count": 0  # ⬅️ nuovo: contatore alberi
            })

            agg["costo"] += costo
            agg["dPA"] += c.get("dPA", 0.0)
            agg["dEA"] += c.get("dEA", 0.0)

            if c.get("tag") == "env_trees":
                # gli alberi li riassumiamo a parte, niente descrizione qui
                agg["trees_count"] += int(c.get("n_trees", 0))

            elif c.get("tag") == "env_gsi":
                # conta eventuali alberi della GSI *e* aggiungi la descrizione del GSI
                agg["trees_count"] += int(c.get("n_trees", 0))
                descr = c.get("descr", "")
                if descr:
                    agg["descr"].append(descr)

            else:
                descr = c.get("descr", "")
                if descr:
                    agg["descr"].append(descr)

            # GSI: salva superficie minima per la colonna finale
            mut = c.get("mut", {})
            if c.get("tag") == "env_gsi" and "sup_min_gsi" in mut:
                agg["sup_min_gsi"] += self._safe_float(mut["sup_min_gsi"])
                agg["has_gsi"] = True

        for segmentagg in applied_by_seg.items():
            row = self.ScenariosHybridResults_tableWidget.rowCount()
            self.ScenariosHybridResults_tableWidget.insertRow(row)
            self.ScenariosHybridResults_tableWidget.setItem(row, 0, QTableWidgetItem("Hybrid"))
            self.ScenariosHybridResults_tableWidget.setItem(row, 1, QTableWidgetItem(str(segment_id)))
            self.ScenariosHybridResults_tableWidget.setItem(row, 2, QTableWidgetItem(f"{agg['costo']:.2f}"))

            # colonne economiche (non calcolate nello scenario ibrido): metto “-”
            self.ScenariosHybridResults_tableWidget.setItem(row, 3, QTableWidgetItem("-"))  # VAN
            self.ScenariosHybridResults_tableWidget.setItem(row, 4, QTableWidgetItem("-"))  # Payback
            self.ScenariosHybridResults_tableWidget.setItem(row, 5, QTableWidgetItem("-"))  # Benefici

            # === CALCOLO ΔEA (con UTC_before = TreeCover dal layer) ===
            # helper locali (copiati dalle mappe della scheda ambientale)
            def _pet_access(val):
                thresholds = [4, 8, 13, 18, 20, 21, 23, 29, 35, 41]
                scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
                for t, s in zip(thresholds, scores):
                    if val <= t: return s
                return 0

            def _wind_access(val):
                thresholds = [0, 0.3, 0.9, 1.9, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5]
                scores = [0, 1, 2, 3, 4, 5, 4, 3, 2, 1]
                for t, s in zip(thresholds, scores):
                    if val <= t: return s
                return 0

            feat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)

            if feat:
                # campi necessari
                pet_field = self.MicroclimatePET12_comboBox.currentText()
                wind_field = self.MicroclimateWind12_comboBox.currentText()
                sh_field = self.MicroclimateShadow12_comboBox.currentText()
                svf_field = self.MicroclimateSkyViewFactor_comboBox.currentText()
                flood_field = self.WaterFloodingRisk_comboBox.currentText()
                tc_field = self.MicroclimateTreeCover_comboBox.currentText()  # <-- TreeCover (UTC before)

                # valori "prima"
                PET_b = self._safe_float(feat[pet_field])
                W_b = self._safe_float(feat[wind_field])
                SH_b = self._safe_float(feat[sh_field])
                SVF_b = self._safe_float(feat[svf_field])
                FL_raw = feat[flood_field] if flood_field else None
                # UTC_before da TreeCover (accetto 0–1 o %)
                TC_b_raw = self._safe_float(feat[tc_field])
                UTC_b = TC_b_raw / 100.0 if TC_b_raw > 1.0 else TC_b_raw

                # --- punteggi "prima" (EA_before) ---
                PET_b_s = _pet_access(PET_b)
                W_b_s = _wind_access(W_b)
                SH_b_s = self._binary_access(SH_b)
                SVF_b_s = self._sky_tree_access(SVF_b)
                FLO_b_s = self._flooding_access(FL_raw)

                # ✅ UTC prima: scala 0–5 (non usare 0–1 qui)
                UTC_b_s = self._sky_tree_access(UTC_b)

                EA_before = (PET_b_s + W_b_s + FLO_b_s + UTC_b_s + SH_b_s + SVF_b_s) / 6.0

                # --- alberi inseriti (calcolo UTC_a come frazione 0–1 per PET/Wind) ---
                n_trees = int(agg.get("trees_count", 0))
                width_val = self._safe_float(feat[width_field])
                length_val = self._safe_float(feat[length_field])
                segment_area = max(1e-6, width_val * length_val)

                import math
                area_chioma = (n_trees * 9.0 * math.pi) / 4.0
                UTC_a = min(1.0, UTC_b + (area_chioma / segment_area))  # 0–1 per PET/Wind

                # --- PET dopo (usa 0–1) ---
                deltaPET = -10.0 * (UTC_a - UTC_b)
                PET_a = PET_b + deltaPET
                PET_a_s = _pet_access(PET_a)

                # --- Vento dopo (usa 0–1) ---
                R = (0.7 * UTC_a) ** 1.2
                W_a = max(0.0, W_b * (1.0 - R))
                W_a_s = _wind_access(W_a)

                # --- Flood dopo (se c'è GSI) ---
                FLO_a_s = FLO_b_s
                if agg.get("has_gsi", False):
                    raw = str(FL_raw)
                    new_raw = "Basso" if raw == "Medio" else ("Medio" if raw == "Alto" else raw)
                    FLO_a_s = self._flooding_access(new_raw)

                # ✅ UTC dopo: scala 0–5 nella media EA
                UTC_a_s = self._sky_tree_access(UTC_a)

                EA_after = (PET_a_s + W_a_s + FLO_a_s + UTC_a_s + SH_b_s + SVF_b_s) / 6.0
                delta_EA = EA_after - EA_before


            else:
                delta_EA = 0.0

            self.ScenariosHybridResults_tableWidget.setItem(
                row, 3, QTableWidgetItem(f"ΔPA=+{agg['dPA']:.3f}, ΔEA={delta_EA:+.3f}")
            )

            # descrizione interventi (concatenata)
            descr_list = [d for d in agg["descr"] if d]
            if agg.get("trees_count", 0) > 0:
                descr_list.append(f"Trees: {int(agg['trees_count'])} pieces (spacing 6 m)")
            self.ScenariosHybridResults_tableWidget.setItem(
                row, 4, QTableWidgetItem(" | ".join(descr_list))
            )

            # Permeabilità (%) solo se ho un GSI; altrimenti “-”
            permeab_txt = "-"
            if agg["has_gsi"] and agg["sup_min_gsi"] > 0:
                # ricalcolo la % sul tratto (Sup. Min. GSI / (larghezza*lunghezza) * 100)
                ffeat = next((f for f in path_layer.getFeatures() if str(f[id_field]) == str(segment_id)), None)

                if feat:
                    try:
                        length_val = float(feat[length_field])
                        width_val = float(feat[width_field])
                        area = width_val * length_val
                        if area > 0:
                            permeab_txt = f"{(agg['sup_min_gsi'] / area) * 100:.2f}"
                    except Exception:
                        permeab_txt = "-"
            self.ScenariosHybridResults_tableWidget.setItem(row, 5, QTableWidgetItem(permeab_txt))

            # Sup. Min. GSI (mq)
            sup_txt = f"{agg['sup_min_gsi']:.2f}" if agg["has_gsi"] else "-"
            self.ScenariosHybridResults_tableWidget.setItem(row, 6, QTableWidgetItem(sup_txt))

        # --- totali/scostamento come gli altri ---
        try:
            user_budget = float(self.AvailableBudget_lineEdit.text().replace(",", "."))
        except Exception:
            QMessageBox.warning(self, "Warning", "Invalid budget value")
            return

        total_cost_minimi = 0.0
        for i in range(self.ScenarioMinimoResults_tableWidget.rowCount()):
            try:
                total_cost_minimi += float(self.ScenarioMinimoResults_tableWidget.item(i, 1).text())
            except Exception:
                pass

        total_cost_h = 0.0
        for i in range(self.ScenariosHybridResults_tableWidget.rowCount()):
            try:
                total_cost_h += float(self.ScenariosHybridResults_tableWidget.item(i, 2).text())
            except Exception:
                pass

        totale = total_cost_minimi + total_cost_h
        scostamento = user_budget - totale
        self.CostoTotaleInterventiScenariVALORE_label.setText(f"{totale:.2f} €")
        self.ScostamentoInterventiScenariVALORE_label.setText(f"{scostamento:.2f} €")

        QMessageBox.information(self, "Done", f"Hybrid scenario generated. Measures applied: {n_applied}")
        self.apply_row_colors_by_scenario(self.ScenariosHybridResults_tableWidget)

    def export_indicators_csv(self):
        """
        Export indicators to CSV in the required order.

        Column order:
        1  Road ID
        2  Initial physical accessibility
        3  Initial environmental accessibility
        4  Road width
        5  Road Lenght
        6  Initial sidewalk width (AB)
        7  Initial sidewalk width (BA)
        8  Final sidewalk width (AB)
        9  Final sidewalk width (BA)
        10 Initial number of trees
        11 Final number of trees
        12 Initial UTC (%)
        13 Final UTC (%)
        14 Initial PET
        15 Final PET
        16 ΔPET
        17 Initial wind (m/s)
        18 Final wind (m/s)
        19 Δwind
        20 Average CO2 stored (= area covered by canopy * 0.28)
        21 Final physical accessibility
        22 Final environmental accessibility
        23 Δphy_accessibility
        24 Δenv_accessibility
        25 construction costs (minimum + environmental)
        """
        # --- prerequisiti ---
        if not hasattr(self, "minimum_scenario_modifications") or not self.minimum_scenario_modifications:
            QMessageBox.warning(self, "Export CSV", "First generate the Minimum Scenario (click the ‘Generate scenarios’ button)")
            return

        # layer base
        path_layer = QgsProject.instance().mapLayer(self.PathLayer_comboBox.currentData())
        if not isinstance(path_layer, QgsVectorLayer):
            QMessageBox.warning(self, "Export CSV", "Layer attributi percorso non valido.")
            return

        # layer accessibilità (se presenti)
        phys_layer = QgsProject.instance().mapLayer(self.PhysicalAccessibilityLayer_comboBox.currentData())
        env_layer = QgsProject.instance().mapLayer(self.EnvironmentalAccessibilityLayer_comboBox.currentData())

        # campi di base dal layer stradale
        id_f = self.ID_comboBox.currentText()
        length_f = self.PathLength_comboBox.currentText()
        width_f = self.PathWidth_comboBox.currentText()
        wAB_f = self.WidthAB_comboBox.currentText()
        wBA_f = self.WidthBA_comboBox.currentText()
        treeAB_f = self.TreeNumberAB_comboBox.currentText()
        treeBA_f = self.TreeNumberBA_comboBox.currentText()

        pet12_f = self.MicroclimatePET12_comboBox.currentText()
        wind12_f = self.MicroclimateWind12_comboBox.currentText()
        utc_f = self.MicroclimateTreeCover_comboBox.currentText()

        # campo EA iniziale dal layer ambientale
        env_field = "EnvironmentalAccessibility_12"
        if self.H9_radioButton.isChecked():
            env_field = "EnvironmentalAccessibility_9"
        elif self.H15_radioButton.isChecked():
            env_field = "EnvironmentalAccessibility_15"

        # indicizza le feature per ID
        feats_by_id = {str(f[id_f]): f for f in path_layer.getFeatures()}

        # util per safe-float
        def sfloat(v, d=0.0):
            try:
                if v in [None, '', QVariant()]:
                    return d
                return float(v)
            except Exception:
                return d

        # per PA iniziale/finale cerco sul layer “Physical Accessibility”
        def phys_by_id(segment_id):
            if not isinstance(phys_layer, QgsVectorLayer):
                return None
            expr = f'"ID" = \'{segment_id}\''
            for f in phys_layer.getFeatures(QgsFeatureRequest().setFilterExpression(expr)):
                return sfloat(f["PhysicalAccessibilityIndex"], None)
            return None

        # per EA iniziale prendo dal layer “Environmental Accessibility”
        def env_by_id(segment_id):
            if not isinstance(env_layer, QgsVectorLayer):
                return None
            expr = f'"ID" = \'{segment_id}\''
            for f in env_layer.getFeatures(QgsFeatureRequest().setFilterExpression(expr)):
                return sfloat(f[env_field], None)
            return None

        # Scegli file
        save_path, _ = QtWidgets.QFileDialog.getSaveFileName(
            self, "Save indicators", "", "CSV files (*.csv);;All files (*)"
        )
        if not save_path:
            return

        import csv, math

        headers = [
            "Segmento_ID",
            "Initial physical accessibility",
            "Initial environmental accessibility",
            "Road width (m)",
            "Road lenght (m)",
            "Initial Sidewalk_AB (m)",
            "Initial Sidewalk_BA (m)",
            "Final Sidewalk_AB (m)",
            "Final Sidewalk_BA (m)",
            "Initial Trees",
            "Final Trees",
            "Initial UTC (%)",
            "Final UTC (%)",
            "Initial PET (°C)",
            "Final PET (°C)",
            "ΔPET (°C)",
            "Initial wind (m/s)",
            "Final wind (m/s)",
            "Δwind",
            "Average CO2 storage (ton)",
            "Final Phy_Accessibility",
            "Final Env_Accessibility",
            "Δphy_acc",
            "Δenv_acc",
            "Production costs (€)"
        ]

        try:
            with open(save_path, "w", newline="", encoding="utf-8") as f:
                w = csv.DictWriter(f, fieldnames=headers, delimiter=";")
                w.writeheader()

                for segment_id, mod in self.minimum_scenario_modifications.items():
                    feat = feats_by_id.get(str(segment_id))
                    if not feat:
                        continue

                    # --- base dal layer strada ---
                    L = sfloat(feat[length_f])
                    W = sfloat(feat[width_f])
                    wAB_init = sfloat(feat[wAB_f])
                    wBA_init = sfloat(feat[wBA_f])
                    trees_init = int(sfloat(feat[treeAB_f])) + int(sfloat(feat[treeBA_f]))
                    utc_init_pct = sfloat(feat[utc_f])

                    if utc_init_pct <= 1.0:  # accetta 0–1 o %
                        utc_init_pct *= 100.0
                    pet_init = sfloat(feat[pet12_f])
                    wind_init = sfloat(feat[wind12_f])

                    # --- finali da scenario minimo / ambientale ---
                    wAB_final = sfloat(mod.get("sidewalk_AB"))
                    wBA_final = sfloat(mod.get("sidewalk_BA"))

                    env_mod = getattr(self, "environmental_scenario_modifications", {}).get(str(segment_id), {})
                    trees_added = int(env_mod.get("trees_added", 0))
                    trees_final = trees_init + trees_added

                    # ... (resto del calcolo invariato)

                    # UTC finale: aggiungo copertura stimata degli alberi inseriti (diametro coerente con t_max)
                    try:
                        t_max = int(float(self.EstimationTime_lineEdit.text()))
                    except Exception:
                        t_max = 20
                    diam = 0
                    for d, t in sorted(CROWN_DIAMETER_MAPPING.items(), key=lambda x: x[1]):
                        if t <= t_max:
                            diam = d
                        else:
                            break

                    seg_area = max(1e-6, W * L)
                    added_tree_area = trees_added * (math.pi * (diam / 2.0) ** 2)
                    utc_final_pct = min(100.0, utc_init_pct + (added_tree_area / seg_area) * 100.0)

                    # PET/Wind finali “come nell’ibrido”
                    deltaPET = -10.0 * ((utc_final_pct / 100.0) - (utc_init_pct / 100.0))
                    pet_final = pet_init + deltaPET

                    R = (0.7 * (utc_final_pct / 100.0)) ** 1.2
                    wind_final = max(0.0, wind_init * (1.0 - R))
                    deltaWind = wind_final - wind_init

                    # CO2 media = area coperta da chiome (m²) * 0.28 (tCO2/m²)
                    co2_mean = (utc_final_pct / 100.0) * seg_area * 0.28

                    # PA/EA iniziali dai layer (se non disponibili, None → uso 0)
                    pa_init = phys_by_id(segment_id)


                    ea_init = env_by_id(segment_id)


                    if pa_init is None: pa_init = 0.0
                    if ea_init is None:
                        # fallback: se non hai il layer ambientale usa EA calcolata nello scenario ambientale
                        ea_init = float(env_mod.get("ea_before", 0.0))

                    # PA finale: non modifichiamo PA nello scenario ambientale → rimane quella iniziale
                    pa_final = pa_init

                    # EA finale: se ho già calcolato nello scenario ambientale lo uso, altrimenti = EA iniziale
                    ea_final = float(env_mod.get("ea_after", ea_init))

                    deltaPA = pa_final - pa_init
                    deltaEA = ea_final - ea_init

                    # Costi: costo scenario minimo (tabella) + eventuale costo ambientale salvato sopra
                    costo_min = 0.0
                    # sommo le righe della tab “Scenario minimo” con lo stesso segment_id
                    for r in range(self.ScenarioMinimoResults_tableWidget.rowCount()):
                        if self.ScenarioMinimoResults_tableWidget.item(r, 0) and \
                                self.ScenarioMinimoResults_tableWidget.item(r, 0).text() == str(segment_id):

                            try:
                                costo_min += float(self.ScenarioMinimoResults_tableWidget.item(r, 1).text())
                            except Exception:
                                pass
                    costo_env = float(env_mod.get("cost_env", 0.0))
                    costo_tot = costo_min + costo_env

                    w.writerow({
                        "Segmento_ID": segment_id,
                        "Initial physical accessibility": round(pa_init, 3),
                        "Initial environmental accessibility": round(ea_init, 3),
                        "Road width (m)": round(W, 3),
                        "Road lenght (m)": round(L, 3),
                        "Initial Sidewalk_AB (m)": round(wAB_init, 3),
                        "Initial Sidewalk_BA (m)": round(wBA_init, 3),
                        "Final Sidewalk_AB (m)": round(wAB_final, 3),
                        "Final Sidewalk_BA (m)": round(wBA_final, 3),
                        "Initial Trees": trees_init,
                        "Final Trees": trees_final,
                        "Initial UTC (%)": round(utc_init_pct, 3),
                        "Final UTC (%)": round(utc_final_pct, 3),
                        "Initial PET (°C)": round(pet_init, 3),
                        "Final PET (°C)": round(pet_final, 3),
                        "ΔPET (°C)": round(deltaPET, 3),
                        "Initial wind (m/s)": round(wind_init, 3),
                        "Final wind (m/s)": round(wind_final, 3),
                        "Δwind": round(deltaWind, 3),
                        "Average CO2 storage (ton)": round(co2_mean, 3),
                        "Final Phy_Accessibility": round(pa_final, 3),
                        "Final Env_Accessibility": round(ea_final, 3),
                        "Δphy_acc": round(deltaPA, 3),
                        "Δenv_acc": round(deltaEA, 3),
                        "Production costs (€)": round(costo_tot, 2)
                    })

            QMessageBox.information(self, "Export CSV", f"Indicators saved in:\n{save_path}")

        except Exception as e:
            QMessageBox.critical(self, "Export CSV", f"Error during saving:\n{e}")

    def _collect_path_segment_ids(self):
        """Raccoglie gli ID dei segmenti che compongono il percorso selezionato in PathSelector_comboBox."""
        path_layer_id = self.PathSelector_comboBox.currentData()
        route_layer = QgsProject.instance().mapLayer(path_layer_id)
        if not isinstance(route_layer, QgsVectorLayer):
            return []

        segment_ids = []
        for f in route_layer.getFeatures():
            sid = f["segment_id"] if "segment_id" in route_layer.fields().names() else f["Segmento_ID"]
            if sid is not None:
                segment_ids.append(str(sid))

        # unici preservando l’ordine
        return list(dict.fromkeys(segment_ids))

    def _avg_over_path(self, layer: QgsVectorLayer, field_name: str, path_ids):
        """Media del campo field_name per i soli segmenti del percorso (per-ID)."""
        if not isinstance(layer, QgsVectorLayer) or not field_name:
            return 0.0
        vals = []
        if "ID" in layer.fields().names():
            id_field = "ID"
        elif "Segmento_ID" in layer.fields().names():
            id_field = "Segmento_ID"
        else:
            return 0.0

        req = QgsFeatureRequest().setSubsetOfAttributes([id_field, field_name], layer.fields())
        for f in layer.getFeatures(req):
            sid = str(f[id_field])
            if sid in path_ids:
                try:
                    vals.append(float(f[field_name]))
                except Exception:
                    pass
        return round(sum(vals) / len(vals), 3) if vals else 0.0

    def _get_before_after_metrics(self):
        """
        Restituisce un dict con: PA_before/after, EA_before/after, PET_before/after, WIND_before/after
        usando i layer selezionati nelle combo e i delta dallo scenario (se presenti).
        """
        # Percorso
        path_ids = self._collect_path_segment_ids()

        # Layer
        phys_id = self.PhysicalAccessibilityLayer_comboBox.currentData()
        env_id = self.EnvironmentalAccessibilityLayer_comboBox.currentData()
        phys_layer = QgsProject.instance().mapLayer(phys_id)
        env_layer = QgsProject.instance().mapLayer(env_id)

        # Campo orario per EA/PET/WIND
        if self.H9_radioButton.isChecked():
            ea_field = "EnvironmentalAccessibility_9";
            pet_f = "PET9";
            wind_f = "Wind9"
        elif self.H12_radioButton.isChecked():
            ea_field = "EnvironmentalAccessibility_12";
            pet_f = "PET12";
            wind_f = "Wind12"
        else:
            ea_field = "EnvironmentalAccessibility_15";
            pet_f = "PET15";
            wind_f = "Wind15"

        # Prima
        pa_before = self._avg_over_path(phys_layer, "PhysicalAccessibilityIndex", path_ids)
        ea_before = self._avg_over_path(env_layer, ea_field, path_ids)
        pet_before = self._avg_over_path(env_layer, pet_f, path_ids)
        wind_before = self._avg_over_path(env_layer, wind_f, path_ids)

        # Delta dagli scenari (se presenti)
        delta_pa = [];
        delta_ea = [];
        delta_pet = [];
        delta_wind = []

        mods = getattr(self, "minimum_scenario_modifications", {}) or {}
        for sid in path_ids:
            m = mods.get(str(sid), {})
            # coerenti con tue chiavi: deltaPA (nuovo), deltaEA (ambientale), deltaPET, deltaWIND
            if "deltaPA" in m:
                try:
                    delta_pa.append(float(m["deltaPA"]))
                except:
                    pass
            if "deltaEA" in m:
                try:
                    delta_ea.append(float(m["deltaEA"]))
                except:
                    pass
            if "deltaPET" in m:
                try:
                    delta_pet.append(float(m["deltaPET"]))
                except:
                    pass
            if "deltaWIND" in m:
                try:
                    delta_wind.append(float(m["deltaWIND"]))
                except:
                    pass

        def avg(lst):
            return sum(lst) / len(lst) if lst else 0.0

        pa_after = round(pa_before + avg(delta_pa), 3)
        ea_after = round(ea_before + avg(delta_ea), 3)
        pet_after = round(pet_before + avg(delta_pet), 3)
        wind_after = round(wind_before + avg(delta_wind), 3)

        return {
            "PA_before": pa_before, "PA_after": pa_after,
            "EA_before": ea_before, "EA_after": ea_after,
            "PET_before": pet_before, "PET_after": pet_after,
            "WIND_before": wind_before, "WIND_after": wind_after,
        }

    def export_a3_layout_with_histograms(self):
        # --- 0) Dati di base ---
        (segment_ids, route_layer) = self._route_segment_ids_from_chosen()
        if route_layer is None or not segment_ids:
            QMessageBox.warning(self, "Export drawing",
                                "No path selected in ChosenPathLayer")
            return

        # Layer base
        path_layer_id = self.PathLayer_comboBox.currentData()
        path_layer = QgsProject.instance().mapLayer(path_layer_id)

        # Layer accessibilità
        phys_id = self.PhysicalAccessibilityLayer_comboBox.currentData()
        env_id = self.EnvironmentalAccessibilityLayer_comboBox.currentData()
        phys_layer = QgsProject.instance().mapLayer(phys_id)
        env_layer = QgsProject.instance().mapLayer(env_id)

        # Campi per EA/PET/Wind "prima"
        env_field = self._current_env_field()
        pet_field = self.MicroclimatePET12_comboBox.currentText()  # 12 come specificato
        wind_field = self.MicroclimateWind12_comboBox.currentText()

        # --- 1) PRIMA: medie lungo il percorso ---
        EA_before = self._avg_on_ids(env_layer, env_field, segment_ids) if env_layer else 0.0
        PA_before = self._avg_on_ids(phys_layer, "PhysicalAccessibilityIndex", segment_ids) if phys_layer else 0.0

        # PET/vento "prima" direttamente dal layer strada
        def _avg_field_on_path(field):
            vals = []
            if not isinstance(path_layer, QgsVectorLayer) or not field:
                return 0.0
            for sid in segment_ids:
                for f in path_layer.getFeatures(
                        QgsFeatureRequest().setFilterExpression(f'"{self.ID_comboBox.currentText()}" = \'{sid}\'')):
                    try:
                        vals.append(float(f[field]))
                    except Exception:
                        pass
            return sum(vals) / len(vals) if vals else 0.0

        PET_before = _avg_field_on_path(pet_field)
        WIND_before = _avg_field_on_path(wind_field)

        # --- 2) DOPO: ricavati dalle tabelle scenario ---
        EA_after = EA_before
        PA_after = PA_before
        PET_after = PET_before
        WIND_after = WIND_before

        # a) scenario ambientale -> deltaEA in colonna 6 (indice 6), PA invariato
        if self.ScenariosEnvironmentalResults_tableWidget.isVisible():
            # sommo i delta per i segmenti presenti, poi media
            deltas = []
            for row in range(self.ScenariosEnvironmentalResults_tableWidget.rowCount()):
                sid_item = self.ScenariosEnvironmentalResults_tableWidget.item(row, 1)
                d_item = self.ScenariosEnvironmentalResults_tableWidget.item(row, 6)
                if not sid_item or not d_item:
                    continue
                if sid_item.text().strip() in segment_ids:
                    try:
                        deltas.append(float(d_item.text()))
                    except Exception:
                        pass
            dEA = (sum(deltas) / len(deltas)) if deltas else 0.0
            EA_after = EA_before + dEA
            # PET/vento nello scenario ambientale (12:00): usiamo le stesse formule del tuo calcolo EA se in futuro li salvi.
            # Per ora non abbiamo i “dopo” analitici -> lasciamo invariati se non hai già memorizzato PET/WIND after.

        # b) scenario fisico -> deltaPA (abbiamo allineato al metodo ibrido), EA invariato
        elif self.ScenariosPhysicalResults_tableWidget.isVisible():
            deltas = []
            # Hai messo ΔPA nella colonna 6 del fisico (coerente)? Se l’hai messa in altra colonna, aggiorna l’indice qui.
            for row in range(self.ScenariosPhysicalResults_tableWidget.rowCount()):
                sid_item = self.ScenariosPhysicalResults_tableWidget.item(row, 1)
                d_item = self.ScenariosPhysicalResults_tableWidget.item(row, 6)
                if not sid_item or not d_item:
                    continue
                if sid_item.text().strip() in segment_ids:
                    try:
                        deltas.append(float(d_item.text()))
                    except Exception:
                        pass
            dPA = (sum(deltas) / len(deltas)) if deltas else 0.0
            PA_after = PA_before + dPA

        # c) scenario ibrido -> colonna 6 contiene "ΔPA=..., ΔEA=..."
        elif self.ScenariosHybridResults_tableWidget.isVisible():
            dPA_list, dEA_list = [], []
            for row in range(self.ScenariosHybridResults_tableWidget.rowCount()):
                sid_item = self.ScenariosHybridResults_tableWidget.item(row, 1)
                mix_item = self.ScenariosHybridResults_tableWidget.item(row, 6)
                if not sid_item or not mix_item:
                    continue
                if sid_item.text().strip() in segment_ids:
                    dpa, dea = self._parse_hybrid_delta(mix_item.text())
                    dPA_list.append(dpa)
                    dEA_list.append(dea)
            if dPA_list:
                PA_after = PA_before + (sum(dPA_list) / len(dPA_list))
            if dEA_list:
                EA_after = EA_before + (sum(dEA_list) / len(dEA_list))

            # Se in ibrido hai già calcolato PET_after e WIND_after in codice, qui potresti ricalcolarli identicamente
            # e fare la media sui segmenti; in assenza dei valori salvati, li lasciamo invariati.

        # --- 3) Crea layout A3 orizzontale ---
        proj = QgsProject.instance()
        layout = QgsPrintLayout(proj)
        layout.initializeDefaults()
        layout.setName("Tavola percorso con istogrammi")

        # Pagina A3 orizzontale
        layout.pageCollection().clear()  # pulisco
        page = QgsLayoutItemPage(layout)
        layout.pageCollection().addPage(page)
        page.setPageSize(QgsLayoutSize(420, 297, QgsUnitTypes.LayoutMillimeters))  # A3 orizzontale

        # Margini “virtuali”
        M = 25  # mm

        # --- 4) Map frame centrato sul percorso, buffer 100 m, scala minima 1:200 ---
        # Extent percorso
        r_extent = route_layer.extent()
        # buffer 100 m in unità layer: assumo CRS metrico, come in routing
        buf = 100.0
        rect = QgsRectangle(r_extent.xMinimum() - buf, r_extent.yMinimum() - buf,
                            r_extent.xMaximum() + buf, r_extent.yMaximum() + buf)

        map_item = QgsLayoutItemMap(layout)
        map_item.attemptSetSceneRect(QRectF(0, 0, 1, 1))
        map_item.setFrameEnabled(True)

        # Trasforma l'extent del percorso nel CRS del progetto prima di impostarlo sulla mappa
        project_crs = QgsProject.instance().crs()
        route_crs = route_layer.crs()
        rect_proj = rect
        try:
            if route_crs.isValid() and project_crs.isValid() and route_crs != project_crs:
                tr = QgsCoordinateTransform(route_crs, project_crs, QgsProject.instance())
                rect_proj = tr.transformBoundingBox(rect)
        except Exception:
            rect_proj = rect

        # (opzionale) forza il CRS del map item
        try:
            map_item.setCrs(project_crs)
        except Exception:
            pass

        map_item.setExtent(rect_proj)

        # posiziono e dimensiono: occupa la metà sinistra
        map_item.attemptMove(QgsLayoutPoint(M, M, QgsUnitTypes.LayoutMillimeters))
        map_item.attemptResize(QgsLayoutSize(420 - 2 * M - 120, 297 - 2 * M, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(map_item)

        # Scala minima 1:200
        try:
            if map_item.scale() < 200:
                map_item.setScale(200)
        except Exception:
            pass

        # --- 5) Legenda, barra di scala, nord ---
        legend = QgsLayoutItemLegend(layout)
        legend.setAutoUpdateModel(True)
        legend.setTitle("Legenda")
        legend.setLinkedMap(map_item)
        legend.attemptMove(QgsLayoutPoint(420 - M - 110, M, QgsUnitTypes.LayoutMillimeters))
        legend.attemptResize(QgsLayoutSize(110, 80, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(legend)

        scalebar = QgsLayoutItemScaleBar(layout)
        scalebar.setLinkedMap(map_item)
        scalebar.setStyle('Single Box')
        scalebar.setNumberOfSegments(4)
        scalebar.setNumberOfSegmentsLeft(0)
        scalebar.setUnits(QgsUnitTypes.DistanceMeters)
        scalebar.setUnitLabel("m")
        scalebar.attemptMove(QgsLayoutPoint(420 - M - 110, M + 85, QgsUnitTypes.LayoutMillimeters))
        scalebar.attemptResize(QgsLayoutSize(110, 12, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(scalebar)

        north = QgsLayoutItemPicture(layout)
        north.setPicturePath(':/images/north_arrows/arrow_up.svg')
        north.attemptMove(QgsLayoutPoint(420 - M - 20, M + 100, QgsUnitTypes.LayoutMillimeters))
        north.attemptResize(QgsLayoutSize(15, 15, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(north)

        title = QgsLayoutItemLabel(layout)
        title.setText("Descrizione degli istogrammi")
        title.setFont(QFont("SansSerif", 14))
        title.setHAlign(Qt.AlignLeft)
        title.attemptMove(QgsLayoutPoint(420 - M - 110, M + 120, QgsUnitTypes.LayoutMillimeters))
        title.attemptResize(QgsLayoutSize(110, 12, QgsUnitTypes.LayoutMillimeters))
        layout.addLayoutItem(title)

        # --- 6) Istogrammi (prima/dopo) con matplotlib -> PNG temporanei ---
        try:
            import matplotlib
            matplotlib.use("Agg")
            import matplotlib.pyplot as plt
            import tempfile, os

            charts = [
                ("Accessibilità ambientale", EA_before, EA_after),
                ("Accessibilità fisica", PA_before, PA_after),
                ("PET (°C)", PET_before, PET_after),
                ("Vento (m/s)", WIND_before, WIND_after),
            ]
            pic_paths = []
            for name, b, a in charts:
                fig = plt.figure(figsize=(3.4, 2.2), dpi=150)  # piccolo riquadro
                plt.bar(["Prima", "Dopo"], [b, a])
                plt.title(name)
                plt.tight_layout()
                tmp = tempfile.mkstemp(suffix=".png")[1]
                plt.savefig(tmp, dpi=150)
                plt.close(fig)
                pic_paths.append((name, tmp))

            # Posiziono 2x2 riquadri a destra sotto il titolo
            slots = [
                (420 - M - 110, M + 135),
                (420 - M - 55, M + 135),
                (420 - M - 110, M + 200),
                (420 - M - 55, M + 200),
            ]
            for (name, pth), (x, y) in zip(pic_paths, slots):
                pic = QgsLayoutItemPicture(layout)
                pic.setPicturePath(pth)
                pic.attemptMove(QgsLayoutPoint(x, y, QgsUnitTypes.LayoutMillimeters))
                pic.attemptResize(QgsLayoutSize(50, 50, QgsUnitTypes.LayoutMillimeters))
                layout.addLayoutItem(pic)
        except Exception as e:
            QMessageBox.warning(self, "Histograms", f"Error generating histograms: {e}")

        # --- 6.b) Finestra per scegliere i layer da esportare ---
        # Ordine di disegno corrente del canvas
        canvas_layers = list(iface.mapCanvas().layers())

        # Mettiamo il layer del percorso in testa e lo forziamo sempre incluso
        # (puoi togliere il "blocco" se vuoi renderlo facoltativo)
        ordered_layers = [route_layer] + [lyr for lyr in canvas_layers if lyr and lyr.id() != route_layer.id()]

        # Visibilità attuale (per pre-spuntare gli elementi)
        root = QgsProject.instance().layerTreeRoot()
        def _is_visible(lyr):
            node = root.findLayer(lyr.id()) if lyr else None
            return node.isVisible() if node else True

        # Costruisco un dialogo minimale con lista spuntabile
        dlg = QtWidgets.QDialog(self)
        dlg.setWindowTitle("Select the layers to export")
        vbox = QtWidgets.QVBoxLayout(dlg)
        lst = QtWidgets.QListWidget(dlg)
        lst.setSelectionMode(QtWidgets.QAbstractItemView.NoSelection)
        vbox.addWidget(lst)

        # Mappa id->layer per recupero sicuro (nomi layer possono ripetersi)
        lyr_by_id = {lyr.id(): lyr for lyr in ordered_layers}

        for lyr in ordered_layers:
            item = QtWidgets.QListWidgetItem(lyr.name(), lst)
            item.setData(Qt.UserRole, lyr.id())
            # pre-spunta in base alla visibilità; il percorso è sempre spuntato e non deselezionabile
            if lyr.id() == route_layer.id():
                item.setCheckState(Qt.Checked)
                item.setFlags(item.flags() & ~Qt.ItemIsEnabled)   # bloccato: sempre incluso
            else:
                item.setCheckState(Qt.Checked if _is_visible(lyr) else Qt.Unchecked)

        # Pulsanti OK/Annulla
        btns = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel,
            parent=dlg
        )
        vbox.addWidget(btns)
        btns.accepted.connect(dlg.accept)
        btns.rejected.connect(dlg.reject)

        if dlg.exec_() != QtWidgets.QDialog.Accepted:
            return  # utente annulla

        # Costruisci il set di layer selezionati, mantenendo l'ordine
        selected_ids = []
        for i in range(lst.count()):
            it = lst.item(i)
            lid = it.data(Qt.UserRole)
            if (it.checkState() == Qt.Checked) or (lid == route_layer.id()):
                selected_ids.append(lid)

        selected_layers = [lyr for lyr in ordered_layers if lyr.id() in selected_ids]

        # Applica i layer alla mappa del layout (e quindi anche alla legenda)
        map_item.setLayers(selected_layers)
        # Mantieni questo set indipendente da cambiamenti futuri nel progetto
        try:
            map_item.setKeepLayerSet(True)
        except Exception:
            pass

        # === LEGENDA: mostra SOLO i layer selezionati ===
        try:
            # usa una radice "custom" per la legenda e disattiva l'auto-sincronizzazione col progetto
            legend.setAutoUpdateModel(False)

            # costruisci un gruppo con soltanto i layer selezionati, mantenendo l’ordine
            proj_root = QgsProject.instance().layerTreeRoot()
            custom_root = QgsLayerTreeGroup()
            for lyr in selected_layers:
                node = proj_root.findLayer(lyr.id())
                if node:
                    custom_root.addChildNode(node.clone())

            # imposta la radice del modello della legenda
            legend.model().setRootGroup(custom_root)

            # tieni comunque il collegamento alla mappa per il filtro per estensione/scala (opzionale)
            legend.setLinkedMap(map_item)
            legend.setLegendFilterByMapEnabled(True)

        except Exception:
            pass

        # Aggiorna mappa e legenda
        map_item.refresh()
        try:
            legend.refreshLegend()
        except Exception:
            pass

        # --- 7) Dialog di salvataggio + export immagine/PDF + salvataggio template .qpt ---
        last_dir = QSettings().value("EasyPath/lastExportDir", os.path.expanduser("~"))
        save_path, selected_filter = QtWidgets.QFileDialog.getSaveFileName(
            self,
            "Esporta tavola",
            os.path.join(last_dir, "tavola_percorso"),
            "PNG (*.png);;PDF (*.pdf);;SVG (*.svg)"
        )
        if not save_path:
            return  # utente ha annullato

        # ricorda la cartella per la prossima volta
        QSettings().setValue("EasyPath/lastExportDir", os.path.dirname(save_path))

        exporter = QgsLayoutExporter(layout)
        base, ext = os.path.splitext(save_path)
        ext = ext.lower()

        # se l'utente non ha scritto l'estensione, ricavala dal filtro
        if ext == "":
            if "PDF" in (selected_filter or ""):
                ext = ".pdf"
            elif "SVG" in (selected_filter or ""):
                ext = ".svg"
            else:
                ext = ".png"
            save_path = base + ext

        ok = False
        if ext == ".pdf":
            pdf = QgsLayoutExporter.PdfExportSettings()
            pdf.dpi = 300
            err = exporter.exportToPdf(save_path, pdf)
            ok = (err == QgsLayoutExporter.Success)

        elif ext == ".svg":
            svg = QgsLayoutExporter.SvgExportSettings()
            err = exporter.exportToSvg(save_path, svg)
            ok = (err == QgsLayoutExporter.Success)

        else:
            img = QgsLayoutExporter.ImageExportSettings()
            img.dpi = 300
            # ⚠️ esporta SEMPRE l'immagine (anche se l'estensione è già .png)
            err = exporter.exportToImage(save_path, img)
            ok = (err == QgsLayoutExporter.Success)

        if not ok:
            QMessageBox.warning(self, "Export", "Table export failed")
            return

        # salva anche il template QGIS .qpt affiancato al file esportato
        qpt_path = base + ".qpt"
        context = QgsReadWriteContext()
        ok_tmpl = layout.saveAsTemplate(qpt_path, context)
        if not ok_tmpl:
            QMessageBox.warning(self, "Export", "Table exported but QPT save failed.")
            # non ritorno: l’immagine/PDF è comunque stata esportata

        QMessageBox.information(self, "Export", f"Export completed:\n{save_path}\n{qpt_path}")

    # Coloramento righe in base a scenario
    def apply_row_colors_by_scenario(self, table_widget):
        """Applica colori distinti alle righe della tabella in base al valore della colonna 'Scenario'."""
        color_map = {}
        base_colors = [
            "#AED6F1", "#A9DFBF", "#F9E79F", "#F5B7B1", "#D7BDE2",
            "#FADBD8", "#D2B4DE", "#FAD7A0", "#ABEBC6", "#F5CBA7",
            "#D6DBDF", "#A3E4D7", "#F1948A", "#BB8FCE", "#FDEBD0",
            "#D1F2EB", "#F6DDCC", "#F0B27A", "#D98880", "#85C1E9"
        ]

        row_count = table_widget.rowCount()
        for row in range(row_count):
            scenario_item = table_widget.item(row, 0)
            if not scenario_item:
                continue
            scenario = scenario_item.text().strip()
            if scenario not in color_map:
                color_map[scenario] = QColor(base_colors[len(color_map) % len(base_colors)])
            color = color_map[scenario]
            for col in range(table_widget.columnCount()):
                item = table_widget.item(row, col)
                if item:
                    item.setBackground(color)