# -*- coding: utf-8 -*-

"""
***************************************************************************
    Date                 : September 2021
    Copyright            : Felix von Studsinske
    Email                : /
    Developer            : Felix von Studsinske
    Description          : -- optional --
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

import os
import traceback
import xml.etree.ElementTree as ET

from qgis.PyQt.QtXml import QDomDocument
from qgis.core import (QgsPrintLayout, QgsProject, QgsReadWriteContext,
                       Qgis, QgsTask, QgsApplication,
                       QgsPointXY, QgsRectangle)
from typing import Dict


from .plot_layout import PlotLayout
from ..template.base_class import ModuleBase
from ..template.gui.progressbar_extended import DoubleProgressGroup
from ...submodules.tools.path import get_files


class TaskReadTemplates(QgsTask):
    """ do not run to many tasks. QGIS do not like many tasks. """
    def __init__(self, description, packet, plots):
        QgsTask.__init__(self, description)
        self.packet = packet
        self.plots = plots
        self.layouts: Dict[str, PlotLayout] = {}
        self.exceptions = []
        self.text = ""
        self.exception = None

    def run(self):
        steps = 100 / len(self.packet)

        for i, item in enumerate(self.packet):
            step = i * steps
            self.setProgress(step)
            # xml child from plots.xml
            name = item.get('name')
            path = item.get('file')
            group = item.get('group')
            project = QgsProject.instance()
            layout = QgsPrintLayout(project)
            document = QDomDocument()

            self.text = path
            with open(os.path.join(self.plots, path), "rt") as file:
                # reads qpt template file
                document.setContent(file.read(), False)

            item_list = layout.loadFromTemplate(document, QgsReadWriteContext())[0]
            try:
                plot_layout = PlotLayout(name, path, group, layout, item_list, xml=item)
                self.layouts[path] = plot_layout
            except NameError:
                self.exceptions.append(str(traceback.format_exc()))
            except:
                self.exceptions.append(str(traceback.format_exc()))
        QgsApplication.messageLog().logMessage(str(self.progress()), "TaskReadTemplates", Qgis.Info)

        self.setProgress(100)
        return True

    def finished(self, result):
        QgsApplication.messageLog().logMessage(f'finished result: {result}', "TaskReadTemplates", Qgis.Info)

    def cancel(self):
        QgsApplication.messageLog().logMessage('TaskReadTemplates cancel', "TaskReadTemplates", Qgis.Critical)


class PlotLayoutTemplates(ModuleBase):
    """ Containing all configured layout files from QGIS.
        Must defined in templates/plots/xx/plots.xml.

        get layout from file: PlotLayoutTemplates["plots/xasd.qpt"]
    """

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.plots = os.path.join(self.get_plugin().plugin_dir, 'templates', 'plots')
        plot_files = [f for f in get_files(self.plots) if f.endswith(".xml")]

        if not plot_files:
            raise ValueError(f"no xml files found in {self.plots}")

        with open(plot_files[0], "r", encoding="utf-8") as file:
            self.xml = ET.parse(file)

        for path in plot_files[1:]:
            with open(path, "r", encoding="utf-8") as file:
                for child in ET.parse(file).getroot().getchildren():
                    self.xml.getroot().append(child)

        self.__layouts: Dict[str, PlotLayout] = {}
        self.__paths = set()
        self.exceptions = []

    def get_orientation(self, file: str):
        """ returns Qgis page orientation

            :param file: file
        """
        layout = self.__layouts[file]
        page = layout.page
        return page.orientation()

    def get_layout_extent(self, file: str, center: QgsPointXY, scale: int) -> QgsRectangle:
        """ Returns extent with given center and scale from layout

            :param file: file
            :param center: center of rectangle/extent (e.g. mouse position on plot layer
            :param scale: map scale
        """
        item_map = self.__layouts[file].item_map

        x_center = center.x()
        y_center = center.y()

        x_low = x_center - (x_center / 100)
        x_high = x_center + (x_center / 100)

        y_low = y_center - (y_center / 100)
        y_high = y_center + (y_center / 100)

        rectangle = QgsRectangle(x_low, y_low, x_high, y_high)
        item_map.zoomToExtent(rectangle)
        item_map.setScale(scale)

        return item_map.extent()

    @classmethod
    def tr_(cls, text: str):
        result = QgsApplication.translate("QgsApplication", text)
        return result

    def load_layouts(self):
        """ loads templates into local dictionary. """

        packets = [x for x in self.xml.getroot()]
        count = len(packets)

        progress: DoubleProgressGroup = getattr(self.get_parent(), "progress", None)
        progress.start_progressbars(0, count, use_subbar=False,
                                    can_cancel=False, hide_widgets=[self.get_parent().ScrollArea])

        task = TaskReadTemplates("TaskReadTemplates", packets, self.plots)
        task.progressChanged.connect(lambda: self.progress_changed(task=task))
        task.taskCompleted.connect(lambda: self.task_completed(task=task))
        task.taskTerminated.connect(lambda: self.task_completed(task=task))
        QgsApplication.taskManager().addTask(task)
        progress.set_text_main(self.tr_('Loading') + " " + self.tr_('Template') + f" '{packets[0].get('file')}'")

    def progress_changed(self, task: TaskReadTemplates):
        progress: DoubleProgressGroup = getattr(self.get_parent(), "progress", None)
        progress.add_main(1)
        progress.set_text_main(self.tr_('Loading') + " " + self.tr_('Template') + f" '{task.text}'")

    def task_completed(self, task: TaskReadTemplates):
        assert not task.exceptions, "some errors occurred: \n\n" + "\n\n".join(task.exceptions)
        self.__layouts.update(task.layouts)

        for layout in self.__layouts.values():
            layout.set_parent(self)

    @property
    def layouts(self):
        return self.__layouts.values()

    def __getitem__(self, item) -> PlotLayout:
        return self.__layouts[item]

    def __iter__(self):
        """ you can iterate over this object """
        for layout in self.__layouts.values():
            yield layout
