# -*- coding: utf-8 -*-
"""
/***************************************************************************
 EasyFilter
                                 A QGIS plugin
 Select features by list of values
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-10-03
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Łukasz Świątek
        email                : lukasz.swiatek1996@gmial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.core import QgsProject, Qgis, QgsVectorFileWriter, QgsAggregateCalculator, QgsMapLayer
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .easy_filter_and_selection_dockwidget import EasyFilterDockWidget
import os.path


class EasyFilter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'EasyFilter_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Easy Filter and Selection')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'EasyFilter')
        self.toolbar.setObjectName(u'EasyFilter')

        #print "** INITIALIZING EasyFilter"

        self.pluginIsActive = False
        self.dockwidget = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('EasyFilter', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/easy_filter_and_selection/icons/ico32.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Set easy filter/selection'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING EasyFilter"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD EasyFilter"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Easy Filter and Selection'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------
    
    def load_table(self):
    
        self.dockwidget.listWidget.clear()
        
        for l in QgsProject.instance().mapLayers().values():
            if l.type() == QgsMapLayer.VectorLayer:
                self.dockwidget.listWidget.addItem(l.name());
            
            
    def load_field_table(self):
    
        if self.dockwidget.listWidget.selectedItems():

            layers = QgsProject.instance().mapLayersByName(self.dockwidget.listWidget.selectedItems()[0].text())
            layer = layers[0]
                
            itemLabels = [field.name() for field in layer.fields()]
            self.dockwidget.listWidget_2.clear()
            
            for item in itemLabels:
                self.dockwidget.listWidget_2.addItem(item);
        else:
            self.dockwidget.listWidget_2.clear()
        
    def check_lists(self):
    
        if len(self.dockwidget.listWidget.selectedItems()) < 1:
        
            self.iface.messageBar().pushMessage("Can't be done:", "Select layer first.", level=Qgis.Warning, duration=30)
            return False
            
        if len(self.dockwidget.listWidget_2.selectedItems()) < 1:
        
            self.iface.messageBar().pushMessage("Can't be done:", "Select field first.", level=Qgis.Warning, duration=30)
            return False
            
        return True
    
            
    def sql_from_textEdit(self, selection=True, reverse=False):
    
        value = self.dockwidget.textEdit.toPlainText().strip()
        
        if value == "":
            self.iface.messageBar().pushMessage("Can't be done:", "Insert values first.", level=Qgis.Warning, duration=30)
            return
            
        if not self.check_lists():
            return
         
        
        field = self.dockwidget.listWidget_2.selectedItems()[0].text()
        splt = value.split("\n")
        
        if reverse:
            result = f'"{field}" not in ('
        else:
            result = f'"{field}" in ('
            
        numeric_field = self.dockwidget.checkBoxNumeric.isChecked() 

        if numeric_field:
            for s in splt:
                result = result + f"{s}, "
        else:
            for s in splt:
                result = result + f"'{s}', "
            
        result = result[:-2] + ")"
        
        layers = QgsProject.instance().mapLayersByName(self.dockwidget.listWidget.selectedItems()[0].text())
        layer = layers[0]
        
        if selection:
            layer.selectByExpression(result)
            self.iface.messageBar().pushMessage("Selection done!", result, level=Qgis.Success, duration=5)
        else:
            layer.setSubsetString(result)
            self.iface.messageBar().pushMessage("Filter done!", result, level=Qgis.Success, duration=5)
            
            
            
    def clear_selection_filter(self, selection=True):
    
        layers = QgsProject.instance().mapLayersByName(self.dockwidget.listWidget.selectedItems()[0].text())
        layer = layers[0]
        
        if selection:
            layer.selectByExpression('')
        else:
            layer.setSubsetString('')
            
    def select_features_with_duplicates(self, reverse: bool = False, filter: bool = False) -> None:
    
        """
        Selects features with duplicates in the reference field
        :param layer_name: name of the layer
        :param reference_field: name of the field to check e.g. id, fid etc.
        
        base on:
        https://gis.stackexchange.com/a/446561/244399
        """
    
        if not self.check_lists():
            return
    
        # warstwa z listy
        layers = QgsProject.instance().mapLayersByName(self.dockwidget.listWidget.selectedItems()[0].text())
        layer = layers[0]
        
        if filter: layer.setSubsetString('')
        
        # pole z listy
        reference_field = self.dockwidget.listWidget_2.selectedItems()[0].text()
        
        # lista wszystkich wartości pola referencyjnego 
        all_values = layer.aggregate(aggregate=QgsAggregateCalculator.ArrayAggregate, fieldOrExpression=reference_field)[0]

        if reverse:
            # lista nieduplikatów {wartość: ilość dupli}
            dict_with_values = {value : all_values.count(value) for value in all_values if all_values.count(value) == 1 }
        else:
            # lista duplikatów {wartość: ilość dupli}
            dict_with_values = {value : all_values.count(value) for value in all_values if all_values.count(value) > 1 }
        
        # tupl z kluczy dicta
        values_to_select = tuple([*dict_with_values])
        
        if len(values_to_select) == 0:
            self.iface.messageBar().pushMessage("Can't Select/Filter", "There is no value to select/filter.", level=Qgis.Success, duration=5)
            return
        
        # Selekcja
        result = f'"{reference_field}" in {values_to_select}'
        
        if result.endswith(',)'):
            result = result[0:-2] + ')'
        
        if filter:
            layer.setSubsetString(result)
            self.iface.messageBar().pushMessage("Filter done!", result, level=Qgis.Success, duration=5)
        
        else:
            layer.selectByExpression(result)
            self.iface.messageBar().pushMessage("Selection done!", result, level=Qgis.Success, duration=5)
        

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING EasyFilter"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = EasyFilterDockWidget()
                self.dockwidget.pushButton.clicked.connect(lambda: self.sql_from_textEdit(True))
                self.dockwidget.pushButton_RSelect.clicked.connect(lambda: self.sql_from_textEdit(True, True))
                self.dockwidget.pushButton_Unselect.clicked.connect(lambda: self.clear_selection_filter(True))
                
                self.dockwidget.pushButton_2.clicked.connect(lambda: self.sql_from_textEdit(False))
                self.dockwidget.pushButton_RFilter.clicked.connect(lambda: self.sql_from_textEdit(False, True))
                self.dockwidget.pushButton_Clear.clicked.connect(lambda: self.clear_selection_filter(False))
                
                self.dockwidget.pushButton_Refresh.clicked.connect(lambda: self.load_table())
                self.dockwidget.pushButton_Duplicates.clicked.connect(lambda: self.select_features_with_duplicates())
                self.dockwidget.pushButton_nonDuplicates.clicked.connect(lambda: self.select_features_with_duplicates(True))
                
                self.dockwidget.pushButton_FDuplicates.clicked.connect(lambda: self.select_features_with_duplicates(False, True))
                self.dockwidget.pushButton_nonFDuplicates.clicked.connect(lambda: self.select_features_with_duplicates(True, True))
                
                self.dockwidget.pushButton_TE_clear.clicked.connect(lambda: self.dockwidget.textEdit.setPlainText(''))
                
                self.dockwidget.listWidget.itemSelectionChanged.connect(lambda: self.load_field_table())
                
            self.load_table()
            self.load_field_table()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
