# flake8: noqa
from datetime import date
from typing import List, Optional

from xsdata.formats.dataclass.parsers import XmlParser

from eagris.common.plugin_constants import DEFAULT_NS
from eagris.common.xml import XmlNode, parseXmlString
from eagris.eagri.eagri_soap_service_type import EagriSoapServiceType
from eagris.eagri.ws.common.requests import injectSignatureTokenIntoRequest, doSendRequest, wrapWithCommonElements
from eagris.eagri.ws.generated.lpizzp01b import ResponseType
from eagris.eagri.ws.lpi_zzp01_b.model.request import ParcelCreationRequest, ParcelUpdateRequest, \
    ParcelUpdateRequestParcel, ParcelCreationRequestParcel, ParcelManagementRequestCrop


def createRequest(request: ParcelCreationRequest) -> ResponseType:
    login, wskey, szrid, _, _ = request.eagri_auth_data
    request_without_token = lpiZzp01bCreateRequest(login, szrid, request.iddpb, request.parcels)
    r = injectSignatureTokenIntoRequest(request.eagri_auth_data, request_without_token)
    return XmlParser().from_bytes(doSendRequest(r, request.eagri_auth_data.env, EagriSoapServiceType.LPI_ZZP01B))


"""
https://eagri.cz/public/portal/-q26171---i-YnMgUe/aktualizace-dvou-parcel
"""


def updateRequest(request: ParcelUpdateRequest) -> ResponseType:
    login, wskey, szrid, _, _ = request.eagri_auth_data
    request_without_token = lpiZzp01bUpdateRequest(login, szrid, request.iddpb, request.parcels)
    r = injectSignatureTokenIntoRequest(request.eagri_auth_data, request_without_token)
    return XmlParser().from_bytes(doSendRequest(r, request.eagri_auth_data.env, EagriSoapServiceType.LPI_ZZP01B))


def lpiZzp01bCreateRequest(login: str, szrid: str, iddpb: int, parcels: List[ParcelCreationRequestParcel]) -> XmlNode:
    return parseXmlString(
        wrapWithCommonElements(
            EagriSoapServiceType.LPI_GDP11B,
            login,
            szrid,
            inner_xml=f"<{DEFAULT_NS}:SZRID>{szrid}</{DEFAULT_NS}:SZRID><{DEFAULT_NS}:IDDPB>{iddpb}</{DEFAULT_NS}:IDDPB>{__createParcelsRequest(parcels)}"
        )
    )


def __createParcelsRequest(parcels: List[ParcelCreationRequestParcel]) -> str:
    return "".join([__createParcelRequest(p) for p in parcels])


def __createParcelRequest(p: ParcelCreationRequestParcel) -> str:
    return f"<{DEFAULT_NS}:PARCELA><{DEFAULT_NS}:NAZEV>{p.name}</{DEFAULT_NS}:NAZEV><{DEFAULT_NS}:GEOMETRIE>{p.geometry}</{DEFAULT_NS}:GEOMETRIE><{DEFAULT_NS}:VYMERA>{p.area_ha}</{DEFAULT_NS}:VYMERA><{DEFAULT_NS}:PARCELAOD>{p.valid_from}</{DEFAULT_NS}:PARCELAOD>{__parcelCropsRequest(p.crops)}</{DEFAULT_NS}:PARCELA>"


def lpiZzp01bUpdateRequest(login: str, szrid: str, iddpb: int, parcels: List[ParcelUpdateRequestParcel]) -> XmlNode:
    return parseXmlString(
        wrapWithCommonElements(
            EagriSoapServiceType.LPI_GDP11B,
            login,
            szrid,
            inner_xml=f"<{DEFAULT_NS}:SZRID>{szrid}</{DEFAULT_NS}:SZRID><{DEFAULT_NS}:IDDPB>{iddpb}</{DEFAULT_NS}:IDDPB>{__updateParcelsRequest(parcels)}"
        )
    )


def __updateParcelsRequest(parcels: List[ParcelUpdateRequestParcel]) -> str:
    return "".join([__updateParcelRequest(p) for p in parcels])


def __updateParcelRequest(p: ParcelUpdateRequestParcel) -> str:
    return f"<{DEFAULT_NS}:PARCELA><{DEFAULT_NS}:IDPARCELA>{p.id}</{DEFAULT_NS}:IDPARCELA><{DEFAULT_NS}:NAZEV>{p.name}</{DEFAULT_NS}:NAZEV><{DEFAULT_NS}:GEOMETRIE>{p.geometry}</{DEFAULT_NS}:GEOMETRIE><{DEFAULT_NS}:VYMERA>{p.area_ha}</{DEFAULT_NS}:VYMERA><{DEFAULT_NS}:PARCELAOD>{p.valid_from}</{DEFAULT_NS}:PARCELAOD>{__parcelCropsRequest(p.crops)}</{DEFAULT_NS}:PARCELA>"


def __parcelCropsRequest(crops: List[ParcelManagementRequestCrop]) -> str:
    return "".join([__updateParcelCropRequest(c) for c in crops])


def __updateParcelCropRequest(crop: ParcelManagementRequestCrop) -> str:
    return f"<{DEFAULT_NS}:PLODINA><{DEFAULT_NS}:KODPLODINY>{crop.legislative_code}</{DEFAULT_NS}:KODPLODINY><{DEFAULT_NS}:MEZIPLODINA>{crop.catch_crop}</{DEFAULT_NS}:MEZIPLODINA><{DEFAULT_NS}:PLATNOSTOD>{crop.valid_from}</{DEFAULT_NS}:PLATNOSTOD>{__cropValidTo(crop.valid_to)}</{DEFAULT_NS}:PLODINA>"


def __cropValidTo(valid_to: Optional[date]):
    if valid_to is None:
        return ""
    else:
        return f"<{DEFAULT_NS}:PLATNOSTDO>{valid_to}</{DEFAULT_NS}:PLATNOSTDO>"
