from typing import List

from PyQt5.QtCore import QThreadPool, Qt
from PyQt5.QtWidgets import QProgressDialog, QMessageBox
from qgis.core import QgsMessageLog, Qgis
from qgis.gui import QgisInterface

from eagris.common.eagris_vector_layer import EagrisVectorLayer
from eagris.common.vector_layer_type import VectorLayerType
from eagris.controller.vector_layer import createQgisLayer, updateQgisLayer
from eagris.controller.worker.base import EagrisWorker
from eagris.controller.worker.block_detail_downloader import BlockDetailDownloader
from eagris.controller.worker.signals import EagrisWorkerSignals
from eagris.eagri.dpb_list.service import dpbList
from eagris.eagri.ws.lpi_gdp11_b.model.response import DpbListResponse
from eagris.model.auth import EagriAuthData
from eagris.qgis.feature_mapper import dpb_to_feature


class EagriDataDownloadWorker:
    def __init__(self, iface: QgisInterface):
        self.__iface = iface
        self.__threadpool = QThreadPool.globalInstance()
        self.__layer_import_signals = EagrisWorkerSignals()
        self.__layer_import_signals.start.connect(self.__on_start)
        self.__layer_import_signals.finished.connect(self.__on_finished)
        self.__layer_import_signals.error.connect(self.__on_error)

    def start(self, auth_data: EagriAuthData):
        worker = EagrisWorker(self.__import_vector_layers, self.__layer_import_signals, auth_data)
        self.__threadpool.start(worker)

    def __import_vector_layers(self, auth_data: EagriAuthData) -> List[EagrisVectorLayer]:
        QgsMessageLog.logMessage("Importing eAGRI data...", level=Qgis.MessageLevel.Info)
        blocks = self.__dpb_list_with_progress_dialog(auth_data)
        importDpbDataLayer(auth_data, blocks)
        return BlockDetailDownloader(self.__create_dpb_detail_progress_dialog(blocks)) \
            .importDpbDetailDataLayer(auth_data, blocks)

    def __dpb_list_with_progress_dialog(self, auth_data: EagriAuthData):
        pd = self.__create_dpb_list_message()
        pd.show()
        blocks = dpbList(auth_data)
        pd.hide()
        return blocks

    def __create_dpb_detail_progress_dialog(self, blocks_response: DpbListResponse):
        progress_dialog = QProgressDialog()
        progress_dialog.setMinimum(0)
        progress_dialog.setMaximum(len(blocks_response.blocks))
        progress_dialog.setWindowTitle("DPB details")
        progress_dialog.setLabelText("Downloading DPB details...")
        progress_dialog.setWindowModality(Qt.WindowModal)
        return progress_dialog

    def __create_dpb_list_message(self):
        msg_box = QMessageBox()
        msg_box.setWindowTitle("DPB list")
        msg_box.setText("Downloading DPB blocks, this might take a moment...")
        msg_box.setStandardButtons(QMessageBox.NoButton)
        msg_box.setWindowModality(Qt.WindowModal)
        return msg_box

    def __on_start(self):
        QgsMessageLog.logMessage("Importing eAGRI vector layers...", level=Qgis.MessageLevel.Info)
        self.__iface.statusBarIface().showMessage("Importing eAGRI vector layers...")

    def __on_error(self, _exception: Exception):
        QgsMessageLog.logMessage("Import did not finish successfully.", level=Qgis.MessageLevel.Critical)

    def __on_finished(self):
        self.__iface.statusBarIface().clearMessage()

        for layer in self.__iface.mapCanvas().layers():
            layer.triggerRepaint()

        self.__iface.mapCanvas().refresh()
        QgsMessageLog.logMessage("eAGRI layer import finished.", level=Qgis.MessageLevel.Info)


def importDpbDataLayer(auth_data: EagriAuthData, dpb_list_response: DpbListResponse):
    blocks = dpb_list_response.blocks
    new_dpb_features = [dpb_to_feature(dpb) for dpb in blocks]
    layer = createQgisLayer(
        layer_type=VectorLayerType.DPB_LIST,
        eagri_auth_data=auth_data
    )
    return updateQgisLayer(layer, new_dpb_features)
