from multiprocessing import Pool, Manager, Queue
from threading import Thread
from time import sleep
from typing import Tuple, List

from PyQt5.QtWidgets import QProgressDialog

from eagris.common.eagris_vector_layer import EagrisVectorLayer
from eagris.common.plugin_constants import QGIS_TABLE_PARCEL_PREFIX
from eagris.common.vector_layer_type import VectorLayerType
from eagris.controller.vector_layer import createQgisLayer, updateQgisLayer, linkVectorLayers
from eagris.eagri.dpb_detail.service import dpbDetail
from eagris.eagri.ws.lpi_ddp01_b.model.response import DpbDetailResponse, Lpiddb01bParcelVersionAttribute, \
    Lpiddb01bParcelAttribute, Lpiddb01bParcelCropAttribute
from eagris.eagri.ws.lpi_gdp11_b.model.response import DpbListResponse, Dpb
from eagris.model.auth import EagriAuthData
from eagris.qgis.feature_mapper import parcel_version_to_feature, parcel_to_feature, parcel_crop_to_feature


class BlockDetailDownloader:
    """
    Component responsible for downloading DPB information.
    Download progress is shown to the user via QProgressDialog.
    """

    def __init__(self, progress_dialog: QProgressDialog):
        self.progress_dialog = progress_dialog

    def importDpbDetailDataLayer(
            self,
            auth_data: EagriAuthData,
            dpb_list_response: DpbListResponse
    ) -> List[EagrisVectorLayer]:
        blocks = [b for b in dpb_list_response.blocks if b.status == 'UCINNY']
        blocks_size = len(blocks)

        """
        We're creating internal multiprocess pool for parallel block download.

        Each block download then emits a event which is sent to an event queue
        shared across processes. These events are then consumed separately by a
        listener thread which then writes the new progress into a QProgressDialog.

        This method returns all new layers, together with their content type
        designated by VectorLayerType.
        """
        with Pool() as p, Manager() as queue_manager:
            queue = queue_manager.Queue()
            blocks_with_auth = [(auth_data, dpb, queue) for dpb in blocks]
            block_download_event_consumer = Thread(
                target=blockDownloadEventConsumer,
                args=[queue, self.progress_dialog, blocks_size]
            )
            block_download_event_consumer.start()

            self.progress_dialog.show()
            dpb_details = p.map(downloadDpbDetail, blocks_with_auth)
            # wait 5 secs for consumer thread to finish
            block_download_event_consumer.join(5.0)

            parcel_version = self.__mapDetailsToParcelVersionVectorLayer(auth_data, dpb_details)
            parcel = self.__mapDetailsToParcelsVectorLayer(auth_data, dpb_details)
            parcel_crop = self.__mapDetailsToParcelCropsVectorLayer(auth_data, dpb_details)
            linkVectorLayers(
                joined_fields_prefix=QGIS_TABLE_PARCEL_PREFIX,
                target_layer=parcel_version.qgis_vector_layer,
                source_layer=parcel.qgis_vector_layer,
                target_layer_field_name=Lpiddb01bParcelVersionAttribute.PARCEL_ID.value,
                source_layer_field_name=Lpiddb01bParcelAttribute.ID.value
            )
            linkVectorLayers(
                joined_fields_prefix=QGIS_TABLE_PARCEL_PREFIX,
                target_layer=parcel_crop.qgis_vector_layer,
                source_layer=parcel.qgis_vector_layer,
                target_layer_field_name=Lpiddb01bParcelCropAttribute.PARCEL_ID.value,
                source_layer_field_name=Lpiddb01bParcelAttribute.ID.value
            )
            return [parcel_version, parcel, parcel_crop]

    def __mapDetailsToParcelVersionVectorLayer(self, auth_data, dpb_details):
        """
        Maps block details retrieved from eAGRI to parcel version features and publishes them into a new layer.
        This is the only parcel layer which contains geometries.
        :return: new vector layer consisting of parcel version features
        """
        layer_type = VectorLayerType.AGRICULTURAL_PARCEL_VERSION
        features = \
            [parcel_version_to_feature(parcel.id, parcel_version)
             for dpb_detail in dpb_details
             for parcel in dpb_detail.parcels
             for parcel_version in parcel.versions
             ]
        layer = createQgisLayer(
            layer_type=layer_type,
            eagri_auth_data=auth_data
        )
        return EagrisVectorLayer(
            updateQgisLayer(layer, features),
            layer_type=layer_type
        )

    def __mapDetailsToParcelCropsVectorLayer(self, auth_data, dpb_details):
        layer_type = VectorLayerType.AGRICULTURAL_PARCEL_CROP
        features = \
            [parcel_crop_to_feature(parcel.id, parcel_crop)
             for dpb_detail in dpb_details
             for parcel in dpb_detail.parcels
             for parcel_crop in parcel.crops
             ]
        layer = createQgisLayer(
            layer_type=layer_type,
            eagri_auth_data=auth_data
        )
        return EagrisVectorLayer(
            qgis_vector_layer=updateQgisLayer(layer, features),
            layer_type=layer_type
        )

    def __mapDetailsToParcelsVectorLayer(self, auth_data, dpb_details):
        layer_type = VectorLayerType.AGRICULTURAL_PARCEL_PARCEL
        features = \
            [parcel_to_feature(parcel)
             for dpb_detail in dpb_details
             for parcel in dpb_detail.parcels
             ]
        layer = createQgisLayer(
            layer_type=layer_type,
            eagri_auth_data=auth_data
        )
        return EagrisVectorLayer(
            qgis_vector_layer=updateQgisLayer(layer, features),
            layer_type=layer_type
        )


def blockDownloadEventConsumer(queue, progress_dialog: QProgressDialog, blocks_size: int):
    progress_dialog.setValue(0)
    blocks_downloaded_so_far = 0
    while blocks_downloaded_so_far < blocks_size:
        sleep(0.05)  # keep the thread from spamming CPU if there are no blocks available
        if not queue.empty():
            new_event = queue.get()
            blocks_downloaded_so_far = progress_dialog.value() + 1
            print(f"NEW EVENT: {new_event} ({blocks_downloaded_so_far}/{blocks_size})")
            progress_dialog.setValue(blocks_downloaded_so_far)


def downloadDpbDetail(item: Tuple[EagriAuthData, Dpb, Queue]) -> DpbDetailResponse:
    auth_data, dpb, queue = item
    queue.put(dpb.id)
    return dpbDetail(auth_data, dpb)
