# -*- coding: utf-8 -*-
from typing import Callable

from PyQt5.QtCore import Qt, QUrl, pyqtSlot
from PyQt5.QtGui import QKeyEvent, QDesktopServices
from PyQt5.QtWidgets import QDockWidget, QTreeWidgetItem, QToolButton, QAction, QTreeWidget, QLineEdit, QWidget, \
    QTabWidget, QTableWidget
from qgis.gui import QgisInterface

from eagris.common.plugin_constants import DOCUMENTATION_URL
from eagris.controller.panel.credtree import EagrisCredentialsTree
from eagris.controller.constants import EAGRI_PANEL_UI_PATH

from qgis.PyQt import uic
import os

from eagris.controller.panel.update_handler import ParcelDetailUpdateHandler
from eagris.controller.worker.eagri_data_download_worker import EagriDataDownloadWorker


class QgisEagriPanel(QDockWidget):
    eagriUsersTreeWidget: QTreeWidget
    parcelDetail: QWidget
    parcelDetailCrops: QTableWidget
    eagriUsers: QWidget
    tabWidget: QTabWidget
    toolButton: QToolButton
    helpButton: QToolButton
    downloadButton: QToolButton
    deleteButton: QToolButton
    parcelDetailNameEdit: QLineEdit

    def __init__(
            self,
            iface: QgisInterface,
            # callback contains reference to the modal, evade circular dependency between components
            default_action_on_click_callback: Callable[[], None]
    ):
        QDockWidget.__init__(self)
        ui_path = os.path.join(os.path.dirname(__file__), "..", EAGRI_PANEL_UI_PATH)
        uic.loadUi(ui_path, self)
        self.__default_action_on_click_callback = default_action_on_click_callback
        self.__iface = iface
        self.__parcel_update_handler = ParcelDetailUpdateHandler(
            self.parcelDetailNameEdit,
            self.parcelDetailCrops
        )
        self.__setup_eagri_users_tree_widget()
        self.__setup_parcel_detail_components()
        self.credsTree = EagrisCredentialsTree(self.eagriUsersTreeWidget)
        self.__downloadDataWorker = EagriDataDownloadWorker(self.__iface)
        self.__setup_tool_button_actions()
        self.__setup_parcel_detail_tool()

    def __setup_eagri_users_tree_widget(self):
        self.eagriUsersTreeWidget.keyPressEvent = self.__handle_key_press
        # this signal does not seem to work with @pyqtSlot
        self.eagriUsersTreeWidget.itemDoubleClicked.connect(self.__handle_creds_tree_double_click)

    def __setup_parcel_detail_components(self):
        self.parcelDetailNameEdit.textChanged.connect(self.__handle_parcel_detail_toggle)
        self.parcelDetailCrops.setSortingEnabled(True)
        self.parcelDetailCrops.sortByColumn(2, Qt.SortOrder.AscendingOrder)

    def __setup_parcel_detail_tool(self):
        self.__iface.mapCanvas().selectionChanged.connect(self.__parcel_update_handler.update_parcel_detail)

    def __handle_parcel_detail_toggle(self, text: str):
        if not text:
            self.tabWidget.setCurrentIndex(0)
            self.parcelDetail.setEnabled(False)
            self.eagriUsers.activateWindow()
        else:
            self.tabWidget.setCurrentIndex(1)
            self.parcelDetail.setEnabled(True)
            self.parcelDetail.activateWindow()

    def __tool_button_actions(self):
        import_manually = QAction("Manually", parent=self.toolButton)
        import_from_db = QAction("DB (TODO)", parent=self.toolButton)
        import_manually.triggered.connect(self.__default_action_on_click_callback)
        import_from_db.setEnabled(False)
        return [import_manually, import_from_db]

    def __setup_tool_button_actions(self):
        icon = self.toolButton.icon()
        actions = self.__tool_button_actions()
        self.toolButton.setText(None)
        self.toolButton.setIcon(icon)
        self.toolButton.setDefaultAction(actions[0])
        self.toolButton.addActions(actions)
        self.toolButton.setText(None)
        self.toolButton.setIcon(icon)
        self.toolButton.setPopupMode(QToolButton.ToolButtonPopupMode.MenuButtonPopup)

    def __handle_creds_tree_double_click(self, item: QTreeWidgetItem):
        auth_data = item.data(0, Qt.UserRole)
        print(f"Importing eAGRI vector layers for auth item {auth_data}...")
        self.__downloadDataWorker.start(auth_data=auth_data)

    @pyqtSlot(name="on_deleteButton_clicked")
    def __handle_delete_button_click(self):
        self.credsTree.removeAllSelectedItems()

    @pyqtSlot(name="on_toolButton_clicked")
    def __handle_tool_button_click(self):
        self.__default_action_on_click_callback()

    @pyqtSlot(name="on_downloadButton_clicked")
    def __handle_download_button_click(self):
        for i in self.eagriUsersTreeWidget.selectedItems():
            auth_data = i.data(0, Qt.UserRole)
            print(f"Importing eAGRI vector layers for auth item {auth_data}...")
            self.__downloadDataWorker.start(auth_data)

    @pyqtSlot(name="on_helpButton_clicked")
    def __handle_help_button_click(self):
        QDesktopServices.openUrl(QUrl(DOCUMENTATION_URL))

    def __handle_key_press(self, event: QKeyEvent):
        if event.key() == Qt.Key_Delete:
            self.credsTree.removeAllSelectedItems()
