# flake8: noqa
from typing import TypeVar
from requests import post

from eagris.eagri.auth.soap_headers import computeAuthenticationToken
from eagris.eagri.env import EagriEnv, requestUrl
from eagris.eagri.error.error_parser import parseErrorResponse
from eagris.eagri.namespaces import IMPLICIT_NS
from eagris.common.plugin_constants import DEFAULT_NS
from eagris.common.xml import XmlNode, findXmlSubnode, serializeXml, parseXmlString
from eagris.eagri.eagri_soap_service_type import EagriSoapServiceType
from eagris.error.exceptions import EagriRequestException
from eagris.model.auth import EagriAuthData


def wrapWithCommonElements(service: EagriSoapServiceType, login: str, szrid: str, inner_xml: str):
    return f'<soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"><soap:Header><vOKO-wss:Token type="A01" xmlns:vOKO-wss="http://www.pds.eu/vOKO/v0200/wss"></vOKO-wss:Token></soap:Header><soap:Body xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"><Request xmlns="http://www.pds.eu/vOKO/v0200" xmlns:{DEFAULT_NS}="http://sitewell.cz/lpis/schemas/{service.value}" vOKOid="{service.value}"><UIDkey addressAD="default" dn="{login}"></UIDkey><RequestHeader><Subject subjectID="{szrid}"></Subject></RequestHeader><RequestContent><{DEFAULT_NS}:Request>{inner_xml}</{DEFAULT_NS}:Request></RequestContent></Request></soap:Body></soap:Envelope>'


def injectSignatureTokenIntoRequest(eagri_auth_data: EagriAuthData, request_without_token: XmlNode) -> XmlNode:
    login, wskey, szrid, _, _ = eagri_auth_data
    auth_token = computeAuthenticationToken(request_without_token, wskey)
    token = findXmlSubnode(request_without_token, 'Token', strict=True, namespace=IMPLICIT_NS)
    token.text = auth_token
    print(eagri_auth_data)
    return request_without_token


ResponseType = TypeVar("ResponseType")


def doSendRequest(soap_request: XmlNode, env: EagriEnv, service: EagriSoapServiceType) -> XmlNode:
    headers = {'content-type': 'text/xml'}

    request_url = requestUrl(env, service)
    response = post(request_url,
                    data=serializeXml(soap_request),
                    headers=headers)
    # TODO debugging logs should be a bit more clever than this :)
    #      ideally, the plugin should put its debugging logs to
    #      - stdout
    #      - QGIS plugin message log
    #      - (optionally) a file
    #     using print has one great benefit - we do not couple SOAP clients with QGIS
    print(f"URL: {request_url}")
    print(f"REQUEST: {serializeXml(soap_request)}")
    print(f"RESPONSE: {response.content}")
    print(f"NAMESPACE: {service.namespace()}")
    if response.ok:
        soap_response_body = parseXmlString(response.content)
        return serializeXml(findXmlSubnode(soap_response_body, 'Response', strict=True, namespace=service.namespace()))
    else:
        raise EagriRequestException(
            response.status_code,
            parseErrorResponse(parseXmlString(response.content)))
