from PyQt5.QtCore import QThreadPool, Qt
from PyQt5.QtWidgets import QProgressDialog, QMessageBox
from qgis.core import QgsMessageLog, Qgis
from qgis.gui import QgisInterface

from eagris.common.vector_layer_type import VectorLayerType
from eagris.controller.vector_layer import getOrCreateQgisLayer, updateQgisLayer
from eagris.controller.worker.base import EagrisWorker
from eagris.controller.worker.block_detail_downloader import BlockDetailDownloader
from eagris.controller.worker.signals import EagrisWorkerSignals
from eagris.eagri.dpb_list.service import dpbList
from eagris.eagri.ws.lpi_gdp11_b.model.response import DpbListResponse
from eagris.model.auth import EagriAuthData
from eagris.qgis.feature_mapper import dpbToFeature


class EagriDataDownloadWorker:
    def __init__(self, iface: QgisInterface):
        self.__iface = iface
        self.__threadpool = QThreadPool.globalInstance()
        self.__layer_import_signals = EagrisWorkerSignals()
        self.__layer_import_signals.start.connect(self.__onStart)
        self.__layer_import_signals.finished.connect(self.__onFinished)
        self.__layer_import_signals.error.connect(self.__onError)

    def start(self, auth_data: EagriAuthData):
        worker = EagrisWorker(self.__importVectorLayers, self.__layer_import_signals, auth_data)
        self.__threadpool.start(worker)

    def __importVectorLayers(self, auth_data: EagriAuthData):
        QgsMessageLog.logMessage("Importing eAGRI data...", level=Qgis.MessageLevel.Info)
        blocks = self.__dpbListWithProgressDialog(auth_data)
        importDpbDataLayer(auth_data, blocks)
        BlockDetailDownloader(self.__createDpbDetailProgressDialog(blocks)) \
            .importDpbDetailDataLayer(auth_data, blocks)

    def __dpbListWithProgressDialog(self, auth_data: EagriAuthData):
        pd = self.__createDpbListMessage()
        pd.show()
        blocks = dpbList(auth_data)
        pd.hide()
        return blocks

    def __createDpbDetailProgressDialog(self, blocks_response: DpbListResponse):
        progress_dialog = QProgressDialog()
        progress_dialog.setMinimum(0)
        progress_dialog.setMaximum(len(blocks_response.blocks))
        progress_dialog.setWindowTitle("DPB details")
        progress_dialog.setLabelText("Downloading DPB details...")
        progress_dialog.setWindowModality(Qt.WindowModal)
        return progress_dialog

    def __createDpbListMessage(self):
        msg_box = QMessageBox()
        msg_box.setWindowTitle("DPB list")
        msg_box.setText("Downloading DPB blocks, this might take a moment...")
        msg_box.setStandardButtons(QMessageBox.NoButton)
        msg_box.setWindowModality(Qt.WindowModal)
        return msg_box

    def __onStart(self):
        QgsMessageLog.logMessage("Importing eAGRI vector layers...", level=Qgis.MessageLevel.Info)
        self.__iface.statusBarIface().showMessage("Importing eAGRI vector layers...")

    def __onError(self, exception: Exception):
        QgsMessageLog.logMessage("Import did not finish successfully.", level=Qgis.MessageLevel.Critical)

    def __onFinished(self):
        self.__iface.statusBarIface().clearMessage()
        QgsMessageLog.logMessage("eAGRI layer import finished.", level=Qgis.MessageLevel.Info)


def importDpbDataLayer(auth_data: EagriAuthData, dpb_list_response: DpbListResponse):
    blocks = dpb_list_response.blocks
    new_dpb_features = [dpbToFeature(dpb) for dpb in blocks]
    layer = getOrCreateQgisLayer(
        layer_type=VectorLayerType.DPB_LIST,
        subject_name=auth_data.name
    )
    return updateQgisLayer(layer, new_dpb_features)
