from datetime import datetime
from typing import List

from PyQt5 import QtGui
from qgis.core import QgsCoordinateReferenceSystem, QgsFeature, QgsSymbol, QgsVectorLayerJoinInfo, QgsVectorLayer, QgsProject

from .attribute_table_fields import LpisBlockListAttributeTableField, LpisBlockDetailParcelVersionAttributeTableField, \
    LpisBlockDetailParcelAttributeTableField, LpisBlockDetailParcelCropAttributeTableField
from .labeling import enableDpbLayerLabeling
from ..common.plugin_constants import GEOMETRYLESS_VECTOR_LAYER_COLOR
from ..common.vector_layer_type import VectorLayerType
from ..common.qgis_constants import QGIS_REGISTRY, KROVAK_CRS
from ..common.geometry_type import GeometryType
from ..common.layer_provider_type import FeatureLayerProviderType
from ..common.data_provider import memoryDataProviderUrl
from ..error.exceptions import VectorLayerJoinException


def getOrCreateQgisLayer(
        layer_type: VectorLayerType,
        subject_name: str,
        registry: QgsProject = QGIS_REGISTRY,
        crs: QgsCoordinateReferenceSystem = KROVAK_CRS,
        geometry_type: GeometryType = GeometryType.POLYGON,
        layer_provider: FeatureLayerProviderType = FeatureLayerProviderType.MEMORY
) -> QgsVectorLayer:
    layer_title: str = f"eAGRI / {layer_type.layer_name()} / {subject_name} {datetime.now().strftime('%Y-%m-%d %H:%M')}"
    # search for a layer with name [layer_title], if such one exists, return it
    existing_layers_with_the_same_name = registry.mapLayersByName(layer_title)
    if existing_layers_with_the_same_name:
        return existing_layers_with_the_same_name[0]
    layer = QgsVectorLayer(memoryDataProviderUrl(geometry_type.value, crs), layer_title, layer_provider.value)
    # add attribute id, purely to make the features selectable from within attribute table
    layer.startEditing()
    layer.dataProvider().addAttributes(__attributesForLayerType(layer_type))
    layer.commitChanges()
    __setGeometryColor(layer, layer_type)
    __setDpbLayerLabeling(layer, layer_type)
    registry.addMapLayer(layer)
    return layer


def __setDpbLayerLabeling(
        vector_layer: QgsVectorLayer,
        layer_type: VectorLayerType
):
    if layer_type == VectorLayerType.DPB_LIST:
        enableDpbLayerLabeling(vector_layer)
    else:
        pass


def __setGeometryColor(
        vector_layer: QgsVectorLayer,
        layer_type: VectorLayerType
):
    symbol = QgsSymbol.defaultSymbol(vector_layer.geometryType())
    symbol.setColor(QtGui.QColor(__colorForLayerType(layer_type)))
    symbol.setOpacity(1)
    vector_layer.renderer().setSymbol(symbol)


# https://signatureedits.com/wp-content/uploads/2023/05/Complementary-Color-Scheme-Examples-8-min.jpg
def __colorForLayerType(layer_type: VectorLayerType):
    if layer_type == VectorLayerType.DPB_LIST:
        return "#BE7333"
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_VERSION:
        return "#678CA8"
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_PARCEL:
        return GEOMETRYLESS_VECTOR_LAYER_COLOR
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_CROP:
        return GEOMETRYLESS_VECTOR_LAYER_COLOR
    else:
        raise ValueError(f"Could not determine layer color. Unsupported vector layer type: {layer_type}")


def __attributesForLayerType(layer_type: VectorLayerType):
    if layer_type == VectorLayerType.DPB_LIST:
        return [f.value for f in LpisBlockListAttributeTableField]
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_VERSION:
        return [f.value for f in LpisBlockDetailParcelVersionAttributeTableField]
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_PARCEL:
        return [f.value for f in LpisBlockDetailParcelAttributeTableField]
    elif layer_type == VectorLayerType.AGRICULTURAL_PARCEL_CROP:
        return [f.value for f in LpisBlockDetailParcelCropAttributeTableField]
    else:
        raise ValueError(f"Could not determine attribute fields. Unsupported vector layer type: {layer_type}")


def updateQgisLayer(layer: QgsVectorLayer, new_features: List[QgsFeature]) -> QgsVectorLayer:
    layer.startEditing()
    layer.addFeatures(new_features)
    layer.commitChanges()
    layer.updateExtents()
    layer.reload()
    return layer


def linkVectorLayers(
        joined_fields_prefix: str,
        target_layer: QgsVectorLayer,
        source_layer: QgsVectorLayer,
        target_layer_field_name,
        source_layer_field_name
):
    """
    Links two vector layers using a column present in both :target_layer and :source_layer.
    Joined columns are added to the :target_layer.
    """
    join_info = QgsVectorLayerJoinInfo()
    join_info.setJoinFieldName(source_layer_field_name)
    join_info.setTargetFieldName(target_layer_field_name)
    join_info.setJoinLayerId(source_layer.id())
    join_info.setJoinLayer(source_layer)
    join_info.setPrefix(joined_fields_prefix)
    join_info.setUsingMemoryCache(True)

    target_layer.startEditing()
    if not target_layer.addJoin(join_info):
        target_layer.commitChanges()
        raise VectorLayerJoinException(source_layer.name(), target_layer.name())
    else:
        target_layer.commitChanges()
        target_layer.updateExtents()
        target_layer.reload()
        target_layer.triggerRepaint()
