# -*- coding: utf-8 -*-
from typing import Callable

from PyQt5.QtCore import Qt, QUrl, pyqtSlot
from PyQt5.QtGui import QKeyEvent, QDesktopServices
from PyQt5.QtWidgets import QDockWidget, QTreeWidgetItem, QToolButton, QAction, QTreeWidget
from qgis.gui import QgisInterface

from eagris.common.plugin_constants import DOCUMENTATION_URL
from eagris.controller.panel.credtree import EagrisCredentialsTree
from eagris.controller.constants import EAGRI_PANEL_UI_PATH

from qgis.PyQt import uic
import os

from eagris.controller.worker.eagri_data_download_worker import EagriDataDownloadWorker


class QgisEagriPanel(QDockWidget):
    treeWidget: QTreeWidget
    toolButton: QToolButton
    helpButton: QToolButton
    downloadButton: QToolButton
    deleteButton: QToolButton

    def __init__(
            self,
            iface: QgisInterface,
            # callback contains reference to the modal, evade circular dependency between components
            default_action_on_click_callback: Callable[[], None]
    ):
        QDockWidget.__init__(self)
        ui_path = os.path.join(os.path.dirname(__file__), "..", EAGRI_PANEL_UI_PATH)
        uic.loadUi(ui_path, self)
        self.__default_action_on_click_callback = default_action_on_click_callback
        self.__iface = iface
        self.treeWidget.keyPressEvent = self.__handleKeyPress
        # this signal does not seem to work with @pyqtSlot
        self.treeWidget.itemDoubleClicked.connect(self.__handleCredsTreeDoubleClick)
        self.credsTree = EagrisCredentialsTree(self.treeWidget)
        self.__downloadDataWorker = EagriDataDownloadWorker(self.__iface)
        self.__setupToolButtonActions()

    def __toolButtonActions(self):
        import_manually = QAction("Manually", parent=self.toolButton)
        import_from_db = QAction("DB (TODO)", parent=self.toolButton)
        import_manually.triggered.connect(self.__default_action_on_click_callback)
        import_from_db.setEnabled(False)
        return [import_manually, import_from_db]

    def __setupToolButtonActions(self):
        icon = self.toolButton.icon()
        actions = self.__toolButtonActions()
        self.toolButton.setText(None)
        self.toolButton.setIcon(icon)
        self.toolButton.setDefaultAction(actions[0])
        self.toolButton.addActions(actions)
        self.toolButton.setText(None)
        self.toolButton.setIcon(icon)
        self.toolButton.setPopupMode(QToolButton.ToolButtonPopupMode.MenuButtonPopup)

    def __handleCredsTreeDoubleClick(self, item: QTreeWidgetItem):
        auth_data = item.data(0, Qt.UserRole)
        print(f"Importing eAGRI vector layers for auth item {auth_data}...")
        self.__downloadDataWorker.start(auth_data=auth_data)

    @pyqtSlot(name="on_deleteButton_clicked")
    def __handleDeleteButtonClick(self):
        self.credsTree.removeAllSelectedItems()

    @pyqtSlot(name="on_toolButton_clicked")
    def __handleToolButtonClick(self):
        self.__default_action_on_click_callback()

    @pyqtSlot(name="on_downloadButton_clicked")
    def __handleDownloadButtonClick(self):
        for i in self.treeWidget.selectedItems():
            auth_data = i.data(0, Qt.UserRole)
            print(f"Importing eAGRI vector layers for auth item {auth_data}...")
            self.__downloadDataWorker.start(auth_data)

    @pyqtSlot(name="on_helpButton_clicked")
    def __handleHelpButtonClick(self):
        QDesktopServices.openUrl(QUrl(DOCUMENTATION_URL))

    def __handleKeyPress(self, event: QKeyEvent):
        if event.key() == Qt.Key_Delete:
            self.credsTree.removeAllSelectedItems()
