from enum import Enum
from typing import List

from ..common.requests import wrapWithCommonElements
from ...eagri_soap_service_type import EagriSoapServiceType
from ....common.xml import parseXmlString
from ....eagri.ws.generated.lpiddp01b import TypDataKodType

"""
Why do we keep XML documents as string oneliners?
Because eAGRI does not care about c14n spec and evidently strips whitespaces to compute document signature. :)
#https://www.w3.org/TR/2001/REC-xml-c14n-20010315#Example-WhitespaceInContent

That's why standard c14n is not a good idea.
"""


def lpiddpRequest(login: str, szrid: str, iddpb: str, data_types: List[TypDataKodType]):
    return parseXmlString(
        wrapWithCommonElements(
            EagriSoapServiceType.LPI_DDP01B,
            login,
            szrid,
            inner_xml=f"<ns2:IDDPB>{iddpb}</ns2:IDDPB>${__typeDataElement(data_types)}"
        )
    )


def __typeDataElement(data_types: List[TypDataKodType]):
    return "".join([f"<ns2:TYPDATAKOD>{kod.value}</ns2:TYPDATAKOD>" for kod in data_types])


class Lpiddb01bParcelAttribute(Enum):
    ID = 'IDPARCELA'
    NAME = 'NAZEV'
    VALID_FROM = 'PARCELAOD'
    VALID_TO = 'PARCELADO'


class Lpiddb01bParcelVersionAttribute(Enum):
    # join field
    ID = 'IDPARCELA'
    VERSION_ID = 'IDPARCELAVERZE'
    AREA = 'VYMERA'
    VALID_FROM = 'PARCELAVERZEOD'
    VALID_TO = 'PARCELAVERZEDO'


class Lpiddb01bParcelCropAttribute(Enum):
    EPH_ID = 'PESTOVANIID'
    CROP_ID = 'KODPLODINY'
    CATCH_CROP = 'MEZIPLODINA'
    VALID_FROM = 'PLATNOSTOD'
    VALID_TO = 'PLATNOSTDO'
