from datetime import date

from PyQt5.QtCore import QThreadPool, QDate
from qgis.core import QgsMessageLog, Qgis, QgsGeometry, QgsFeature
from qgis.gui import QgisInterface

from eagris.controller.vector_layer import getOrCreateQgisLayer, updateQgisLayer
from eagris.controller.worker.base import EagrisWorker
from eagris.controller.worker.signals import EagrisWorkerSignals
from eagris.eagri.dpb_list.service import dpbList
from eagris.eagri.ws.lpi_gdp11_b.model.response import Dpb
from eagris.model.auth import EagriAuthData


class DownloadEagriDataWorker:
    def __init__(self, iface: QgisInterface):
        self.__iface = iface
        self.__threadpool = QThreadPool.globalInstance()
        self.__layer_import_signals = EagrisWorkerSignals()
        self.__layer_import_signals.start.connect(self.__onStart)
        self.__layer_import_signals.finished.connect(self.__onFinished)
        self.__layer_import_signals.error.connect(self.__onError)

    def start(self, auth_data: EagriAuthData):
        worker = EagrisWorker(self.__importVectorLayers, self.__layer_import_signals, auth_data)
        self.__threadpool.start(worker)

    def __importVectorLayers(self, auth_data: EagriAuthData):
        QgsMessageLog.logMessage("Importing eAGRI data...", level=Qgis.MessageLevel.Info)
        importDpbDataLayer(auth_data)

    def __onStart(self):
        QgsMessageLog.logMessage("Importing eAGRI vector layers...", level=Qgis.MessageLevel.Info)
        self.__iface.statusBarIface().showMessage("Importing eAGRI vector layers...")

    def __onError(self, exception: Exception):
        QgsMessageLog.logMessage("Import did not finish successfully.", level=Qgis.MessageLevel.Critical)

    def __onFinished(self):
        self.__iface.statusBarIface().clearMessage()
        QgsMessageLog.logMessage("eAGRI layer import finished.", level=Qgis.MessageLevel.Info)


def __dateToQDate(d: date):
    return QDate(d.year, d.month, d.day) if d else None


def dpbToFeature(dpb: Dpb):
    geom = QgsGeometry.fromWkt(dpb.geometry)

    f = QgsFeature()
    f.setGeometry(geom)

    f.setAttributes([
        dpb.id,
        dpb.square,
        dpb.zkod,
        dpb.status,
        __dateToQDate(dpb.valid_from),
        dpb.area,
        dpb.culture,
        dpb.culture_name,
        dpb.geometry,
        __dateToQDate(dpb.valid_to)
    ])
    return f


def importDpbDataLayer(auth_data: EagriAuthData):
    dpb_response = dpbList(auth_data)
    new_features = list(map(dpbToFeature, dpb_response.blocks))

    layer = getOrCreateQgisLayer()
    return updateQgisLayer(layer, new_features)
