from datetime import datetime
from typing import List

from qgis.core import QgsCoordinateReferenceSystem, QgsFeature, QgsVectorLayer, QgsProject

from .labeling import enableDpbLayerLabeling
from ..model.lpis_block_fields import LpisBlockAttributeTableField
from ..common.qgis_constants import QGIS_REGISTRY, KROVAK_CRS
from ..common.geometry_type import GeometryType
from ..common.layer_provider_type import FeatureLayerProviderType
from ..common.data_provider import memoryDataProviderUrl


def getOrCreateQgisLayer(
    registry: QgsProject = QGIS_REGISTRY,
    # TODO name the layer using subject name, the current title is just a dummy placeholder
    layer_title: str = 'eAGRI (%s)' % datetime.now().strftime('%Y-%m-%d %H:%M'),
    crs: QgsCoordinateReferenceSystem = KROVAK_CRS,
    geometry_type: GeometryType = GeometryType.POLYGON,
    layer_provider: FeatureLayerProviderType = FeatureLayerProviderType.MEMORY
) -> QgsVectorLayer:
    # search for a layer with name [layer_title], if such one exists, return it
    existing_layers_with_the_same_name = registry.mapLayersByName(layer_title)
    if existing_layers_with_the_same_name:
        return existing_layers_with_the_same_name[0]
    layer = QgsVectorLayer(memoryDataProviderUrl(geometry_type.value, crs), layer_title, layer_provider.value)
    # add attribute id, purely to make the features selectable from within attribute table
    layer.startEditing()
    layer.dataProvider().addAttributes([f.value for f in LpisBlockAttributeTableField])
    layer.commitChanges()
    enableDpbLayerLabeling(layer)
    registry.addMapLayer(layer)
    return layer


def updateQgisLayer(layer: QgsVectorLayer, new_features: List[QgsFeature]) -> QgsVectorLayer:
    layer.startEditing()
    layer.addFeatures(new_features)
    layer.commitChanges()
    layer.updateExtents()
    layer.reload()
    return layer
