from typing import List

from .default_cred_settings import DEFAULT_AUTH_SETTINGS
from ..auth.sqlite.utils import executeSqliteStatement, create_session
from ..configuration.auth_cred_setting import AuthCredSetting, OptionalAuthCredSetting
from ..auth.auth_settings import AuthCredSettings
from ..common.qgis_constants import QGIS_SETTINGS
from ..error.exceptions import EagrisException


class QgisEagriConfigurationException(EagrisException):
    def __init__(self, message):
        super().__init__(message)


def loadSettings():
    qgis_keys = [prop for prop in QGIS_SETTINGS.allKeys() if prop.startswith("eagris/")]
    __checkConfigDataSanity(qgis_keys)
    qgis_variables = {key: QGIS_SETTINGS.value(key) for key in qgis_keys}

    auth_cred_settings = AuthCredSettings(
        qgis_variables[AuthCredSetting.AUTH_CREDS_PATH.value],
        qgis_variables[AuthCredSetting.TABLE_NAME.value],
        qgis_variables[AuthCredSetting.LOGIN_FIELD_NAME.value],
        qgis_variables[AuthCredSetting.SZRID_FIELD_NAME.value],
        qgis_variables[AuthCredSetting.WSKEY_FIELD_NAME.value],
        qgis_variables.get(OptionalAuthCredSetting.NAME_FIELD_NAME.value, None),
        qgis_variables.get(OptionalAuthCredSetting.ENV_FIELD_NAME.value, None)
    )
    # TODO deletegate to multiple streams
    print(f"Successfully loaded Eagris configuration, got {auth_cred_settings}")
    return auth_cred_settings


def setupLocalCreds(auth_creds_path: str = DEFAULT_AUTH_SETTINGS.authCredsPath):
    print(f"setting up local credentials using file {auth_creds_path}")
    QGIS_SETTINGS.setValue('eagris', True)
    QGIS_SETTINGS.setValue(AuthCredSetting.AUTH_CREDS_PATH.value, auth_creds_path)
    QGIS_SETTINGS.setValue(AuthCredSetting.TABLE_NAME.value, DEFAULT_AUTH_SETTINGS.tableName)
    QGIS_SETTINGS.setValue(AuthCredSetting.WSKEY_FIELD_NAME.value, DEFAULT_AUTH_SETTINGS.wsKeyField)
    QGIS_SETTINGS.setValue(AuthCredSetting.SZRID_FIELD_NAME.value, DEFAULT_AUTH_SETTINGS.szridField)
    QGIS_SETTINGS.setValue(AuthCredSetting.LOGIN_FIELD_NAME.value, DEFAULT_AUTH_SETTINGS.loginField)

    session = create_session(auth_creds_path)
    executeSqliteStatement(
        f"""
            CREATE TABLE IF NOT EXISTS {DEFAULT_AUTH_SETTINGS.tableName}(
                {DEFAULT_AUTH_SETTINGS.loginField} text NOT NULL,
                {DEFAULT_AUTH_SETTINGS.szridField} text NOT NULL,
                {DEFAULT_AUTH_SETTINGS.wsKeyField} text NOT NULL,
                {DEFAULT_AUTH_SETTINGS.nameField} text NULL,
                {DEFAULT_AUTH_SETTINGS.envField} text NULL
            );
            """,
        session
    )


def __checkConfigExists(setting: AuthCredSetting, qgis_keys: List[str]):
    if setting.value not in QGIS_SETTINGS.allKeys():
        __eraseConfig(qgis_keys)
        raise QgisEagriConfigurationException(f"Setting {setting.value} does not exist in user config. The remaining "
                                              f"config (if there was any) was erased.")


def __eraseConfig(qgis_keys: List[str]):
    [QGIS_SETTINGS.remove(key) for key in qgis_keys]


def __checkConfigDataSanity(qgis_keys: List[str]):
    [__checkConfigExists(e, qgis_keys) for e in AuthCredSetting]


def isEagrisPluginSetUp() -> bool:
    return QGIS_SETTINGS.value('eagris', defaultValue=False, type=bool)
