# -*- coding: utf-8 -*-

"""
This is part of eMapTool plugin. 

This one creates retention tree areas from points based on ecological values and weighing of the values.

"""

__author__ = 'Mikko Kesälä'
__date__ = '2024-01-01'
__copyright__ = '(C) 2024 by eMap modeling'

# This will get replaced with a git SHA1 when you do a git archive

from stat import S_ISLNK
from qgis import processing
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication,QVariant
from qgis.core import (QgsProcessing,
                       QgsField,
                       QgsFeatureSink,QgsProcessingParameterField,
                       QgsProcessingParameterFeatureSource,QgsProcessingParameterRasterDestination,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterDefinition,
                       QgsProcessingUtils,QgsRasterLayer,QgsVectorLayer)
import os,time,sys
import geopandas as gpd
import pandas as pd
from .algorithms.geotools2 import layer2gpd,gpd2qgis
from .algorithms.retreettools import ecosystemIndicators,spatialPlanning

class points2retreeareas(QgsProcessingAlgorithm):


    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        #inputs
        self.addParameter(QgsProcessingParameterFeatureSource('trees','Trees',[QgsProcessing.TypeVectorPoint]))
        #self.addParameter(QgsProcessingParameterMapLayer('trees', 'Trees', types=[QgsProcessing.TypeVectorPoint]))
        self.addParameter(QgsProcessingParameterField('height_field', 'Height', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='height'))
        self.addParameter(QgsProcessingParameterField('diameter_field', 'Diameter', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='dbh'))
        self.addParameter(QgsProcessingParameterField('species_field', 'Tree species', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='treespecies'))
        self.addParameter(QgsProcessingParameterField('fertilityclass_field', 'Fertility class', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='fertilityclass'))
        self.addParameter(QgsProcessingParameterField('vegezone_field', 'Vegetation zone', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='paajakonro'))
        self.addParameter(QgsProcessingParameterField('dtw_field', 'DTW', type=QgsProcessingParameterField.Numeric, parentLayerParameterName='trees', allowMultiple=False, defaultValue='dtw'))
        
        
        #parameters
        params = []
        params.append(QgsProcessingParameterEnum('ds_w','Tree diversity',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('gc_w','Tree heterogeneity',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('dwp_w','Deadwood potential',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('dtw_w',"Depth-to-Water",options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        
        for p in params:
            p.setFlags(p.flags() | QgsProcessingParameterDefinition.FlagAdvanced) 
            self.addParameter(p)

        self.addParameter(QgsProcessingParameterNumber('treecount','Retention tree count (pcs. / ha)',type=QgsProcessingParameterNumber.Integer,minValue=5,maxValue=30,defaultValue=10))
        #outputs
        self.addParameter(QgsProcessingParameterFeatureSink('out_trees','Trees', type=QgsProcessing.TypeVectorPoint, createByDefault=True, defaultValue=None))
        self.addParameter(QgsProcessingParameterFeatureSink('retention','Retention tree groups', type=QgsProcessing.TypeVectorPolygon, createByDefault=True, defaultValue=None))
    
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        results = {}

        #get inputs and parameters
        ds_w = self.parameterAsInt(parameters,'ds_w',context)
        gc_w = self.parameterAsInt(parameters,'gc_w',context)
        dwp_w = self.parameterAsInt(parameters,'dwp_w',context)
        dtw_w = self.parameterAsInt(parameters,'dtw_w',context)
        w_translate = {0:'',1:'A',2:'S',3:'P'}

        weights = {'ds':w_translate[ds_w],
            'edtw':w_translate[dtw_w],
            'dwp':w_translate[dwp_w],
            'gini':w_translate[gc_w]}
        
        treecount = self.parameterAsInt(parameters,'treecount',context)

        fields = {self.parameterAsString(parameters,'species_field',context):'treespecie',
                  self.parameterAsString(parameters,'diameter_field',context):'dbh',
                  self.parameterAsString(parameters,'height_field',context):'height',
                  self.parameterAsString(parameters,'dtw_field',context):'dtw',
                  self.parameterAsString(parameters,'vegezone_field',context):'paajakonro',
                  self.parameterAsString(parameters,'fertilityclass_field',context):'fertilityclass'}
        trees = QgsProcessingUtils.mapLayerFromString(parameters['trees'],context)
        
        #convert to geopandas
        trees_gdf = layer2gpd(trees)
        trees_gdf.rename(columns=fields, inplace=True)

        #process
        feedback.pushInfo("Calcuating ecosystem service indicators ... ")
        trees_gdf = ecosystemIndicators(trees_gdf,30,30)
        feedback.pushInfo("Proposing retention tree groups")
        trees_gdf,retention = spatialPlanning(trees_gdf,treecount,30,weights)
        feedback.pushInfo("... ready! ("+str(len(retention))+" tree groups)")
        
        #convert to qgis format
        #print (trees)
        trees_qs = gpd2qgis(trees_gdf)
        retention_qs = gpd2qgis(retention)

        #write to user output
        (sink, dest_id) = self.parameterAsSink(parameters, 'out_trees',context,
                    trees_qs.fields(), trees_qs.wkbType(), trees_qs.crs())
        results['out_trees'] = dest_id

        for feature in trees_qs.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
                
        (sink, area_id) = self.parameterAsSink(parameters, 'retention',context,
                    retention_qs.fields(), retention_qs.wkbType(), retention_qs.crs())
        
        for feature in retention_qs.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
        results['retention'] = area_id

        #define styles for output
        style = os.path.join(os.path.dirname(__file__),"styles/reTree5.qml")
        style2 = os.path.join(os.path.dirname(__file__),"styles/retreet_areas.qml")

        trees_qs = QgsProcessingUtils.mapLayerFromString(dest_id, context)
        trees_qs.loadNamedStyle(style)
        
        layer2 = QgsProcessingUtils.mapLayerFromString(area_id, context)
        layer2.loadNamedStyle(style2)

        return results

    
    
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'points2retreeareas'


    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return 'Plan retention tree groups (trees)'

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'ReTreeT planning'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def shortHelpString(self):
        helpfile = open(os.path.dirname(__file__) + '/descriptions/points2retreetareas.html',encoding="utf-8")
        help = helpfile.read()
        return help
    
    def createInstance(self):
        return points2retreeareas()
