# -*- coding: utf-8 -*-

"""
This is part of eMapTool plugin. 

This one plans retention tree areas within planning area which is polygon input of the tool.
Retention tree planning is based on ecological values and weighing of the values.

"""

__author__ = 'Mikko Kesälä'
__date__ = '2024-01-01'
__copyright__ = '(C) 2024 by eMap modeling'

# This will get replaced with a git SHA1 when you do a git archive


from stat import S_ISLNK
from qgis import processing
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication,QVariant
from qgis.core import (QgsProcessing,
                       QgsField,
                       QgsFeatureSink,QgsProcessingParameterField,
                       QgsProcessingParameterFeatureSource,QgsProcessingParameterRasterDestination,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterDefinition,
                       QgsProcessingUtils,QgsRasterLayer,QgsVectorLayer)
import os,time,sys,tempfile
import numpy as np
from urllib.parse import urlparse, parse_qs
#from PIL import Image

from .algorithms.geotools2 import clipByFeature,clipBigVRTrasterByFeature,layer2gpd,gpd2qgis
from .algorithms.importTools import importWfsByGDB
from .algorithms.retreettools import treeDeduct,ecosystemIndicators,spatialPlanning

class planReTreeAreas(QgsProcessingAlgorithm):

 
    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        #inputs
        self.addParameter(QgsProcessingParameterFeatureSource('sites','Harvesting areas',[QgsProcessing.TypeVectorPolygon]))
        #self.addParameter(QgsProcessingParameterField('id_field', 'ID field', type=QgsProcessingParameterField.Any, parentLayerParameterName=self.INPUT, allowMultiple=False))
        self.addParameter(QgsProcessingParameterMapLayer('chm', 'CHM', types=[QgsProcessing.TypeRaster]))
        self.addParameter(QgsProcessingParameterMapLayer('dtw', 'DTW',types=[QgsProcessing.TypeRaster]))
        self.addParameter(QgsProcessingParameterMapLayer('vegetationzone', 'Vegetation zones', types=[QgsProcessing.TypeVectorPolygon]))
        #self.addParameter(QgsProcessingParameterMapLayer('waterbody', 'Water body', types=[QgsProcessing.TypeVectorPolygon]))
        self.addParameter(QgsProcessingParameterMapLayer('forestgrid', 'Forest grid',types=[QgsProcessing.TypeVectorPolygon]))
 
        
        #parameters
        params = []
        params.append(QgsProcessingParameterEnum('ds_w','Tree diversity',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('gc_w','Tree heterogeneity',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('dwp_w','Deadwood potential',options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        params.append(QgsProcessingParameterEnum('dtw_w',"Depth-to-Water",options=['No weighting','Low','Moderate','Significant'],defaultValue=1))
        
        for p in params:
            p.setFlags(p.flags() | QgsProcessingParameterDefinition.FlagAdvanced) 
            self.addParameter(p)

        self.addParameter(QgsProcessingParameterNumber('treecount','Retention tree count (pcs. / ha)',type=QgsProcessingParameterNumber.Integer,minValue=5,maxValue=30,defaultValue=10))
        #outputs
        self.addParameter(QgsProcessingParameterFeatureSink('trees','Trees', type=QgsProcessing.TypeVectorPoint, createByDefault=True, defaultValue=None))
        self.addParameter(QgsProcessingParameterFeatureSink('retention','Retention tree groups', type=QgsProcessing.TypeVectorPolygon, createByDefault=True, defaultValue=None))

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        results = {}
        
        #get data
        #sites = self.parameterAsSource(parameters, 'trees', context)
        #sites = QgsProcessingUtils.mapLayerFromString(parameters['sites'],context)
        sites = self.parameterAsSource(parameters, 'sites', context)
        if sites.featureCount() > 50:
            feedback.reportError("Input layer has too many features. 20 features is maximum. Process failed.")
            sys.exit()
        sites = processing.run("native:dissolve", {'INPUT':parameters['sites'],'OUTPUT':'TEMPORARY_OUTPUT'})
        sites = processing.run("native:multiparttosingleparts", {'INPUT':sites['OUTPUT'],'OUTPUT':'TEMPORARY_OUTPUT'})['OUTPUT']
        vege = QgsProcessingUtils.mapLayerFromString(parameters['vegetationzone'],context)
        fgrid = QgsProcessingUtils.mapLayerFromString(parameters['forestgrid'],context)
        chm = QgsProcessingUtils.mapLayerFromString(parameters['chm'],context)
        dtw = QgsProcessingUtils.mapLayerFromString(parameters['dtw'],context)

        sites_gdf = layer2gpd(sites)
        vege_gdf = layer2gpd(vege)
        vege_fields = ["paajakonro","geometry"]
        vege_gdf = vege_gdf[vege_fields]

        if 'id' in sites_gdf.columns:
            sites_gdf['id'] = sites_gdf.index

        #get parameters
        ds_w = self.parameterAsInt(parameters,'ds_w',context)
        gc_w = self.parameterAsInt(parameters,'gc_w',context)
        dwp_w = self.parameterAsInt(parameters,'dwp_w',context)
        dtw_w = self.parameterAsInt(parameters,'dtw_w',context)
        w_translate = {0:'',1:'A',2:'S',3:'P'}
        
        weights = {'ds':w_translate[ds_w],
            'edtw':w_translate[dtw_w],
            'dwp':w_translate[dwp_w],
            'gini':w_translate[gc_w]}
        
        treecount = self.parameterAsInt(parameters,'treecount',context)


        #clip grids
        if "url=" in fgrid.source():
            # Split parameters (usually separated by '&')
            parts = fgrid.source().split(" ")
            for part in parts:
                if part.startswith("url="):
                    url_base = part.split("=",1)[1]+"?"
                    url_base = url_base.replace("'","")
                    parts.remove(part)
            parts = [p.replace("'","") for p in parts]
            query_string = "&".join(parts)    # The rest is the query string

            # Optionally reconstruct full URL and parse it
            full_url = f"{url_base}?{query_string}"
            parsed = urlparse(full_url)
            params = parse_qs(parsed.query)
            flattened_params = {k: v[0]  for k, v in params.items() if k in ['typename','srsname']}
            grid_fields = 'FERTILITYCLASS,MEANHEIGHTPINE,MEANHEIGHTSPRUCE,MEANHEIGHTDECIDUOUS,GEOMETRY'
            grid_gdf = importWfsByGDB(url_base,flattened_params['typename'],grid_fields,sites_gdf)
        
            feedback.pushInfo("Data clipped")
        

        if "url=" in dtw.source():
            parts = dtw.source().split("&")
            for part in parts:
                if part.startswith("url="):
                    url_base = part.split("=",1)[1]
                    parts.remove(part)
            parts = [p for p in parts if "identifier" in p]
            query_string = "&".join(parts)
            query_string = query_string.replace("identifier","coverageId")
            base_string = "service=WCS&version=1.0.0&request=GetCoverage&" 
            dtw =  f"WCS:{url_base}?{base_string}{query_string}"
        else:
            dtw = dtw.source()
                
        #process
        feedback.pushInfo("Mapping trees ... ")
        trees = treeDeduct(sites_gdf,vege_gdf,grid_gdf,chm.source(),dtw)
        feedback.pushInfo("... ready! ("+str(len(trees))+" trees)")
        #feedback.pushInfo("\tReady in")
        feedback.pushInfo("Calcuating ecosystem service indicators ... ")
        trees = ecosystemIndicators(trees,30,30)
        feedback.pushInfo("Proposing retention tree groups")
        trees,retention = spatialPlanning(trees,treecount,30,weights)
        feedback.pushInfo("... ready! ("+str(len(retention))+" tree groups)")
        
        #translate to qgis format
        #print (trees)
        trees_qs = gpd2qgis(trees)
        retention_qs = gpd2qgis(retention)

        #write to user output
        (sink, dest_id) = self.parameterAsSink(parameters, 'trees',context,
                    trees_qs.fields(), trees_qs.wkbType(), trees_qs.crs())
        results['trees'] = dest_id

        for feature in trees_qs.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
                
        (sink, area_id) = self.parameterAsSink(parameters, 'retention',context,
                    retention_qs.fields(), retention_qs.wkbType(), retention_qs.crs())
        
        for feature in retention_qs.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
        results['retention'] = area_id

        #define styles for output
        style = os.path.join(os.path.dirname(__file__),"styles/reTree5.qml")
        style2 = os.path.join(os.path.dirname(__file__),"styles/retreet_areas.qml")

        trees_qs = QgsProcessingUtils.mapLayerFromString(dest_id, context)
        trees_qs.loadNamedStyle(style)
        
        layer2 = QgsProcessingUtils.mapLayerFromString(area_id, context)
        layer2.loadNamedStyle(style2)

        return results


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'planretreeareas'


    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return 'Plan retention tree groups (harvesting area)'

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'ReTreeT planning'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def shortHelpString(self):
        helpfile = open(os.path.dirname(__file__) + '/descriptions/planretreetareas.html',encoding="utf-8")
        help = helpfile.read()
        return help


    def createInstance(self):
        return planReTreeAreas()
