"""Nested Cross-Validation Algorithm for dzetsaka.

This module provides a QGIS Processing algorithm for performing nested
cross-validation to evaluate model performance without bias.
"""

import os

from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterEnum,
    QgsProcessingParameterField,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterNumber,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterVectorLayer,
)
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon

from dzetsaka import classifier_config
from dzetsaka.logging import show_error_dialog
from dzetsaka.qgis.processing import metadata_helpers

# Try to import nested CV
try:
    from dzetsaka.scripts.validation.nested_cv import NestedCrossValidator

    NESTED_CV_AVAILABLE = True
except ImportError:
    NESTED_CV_AVAILABLE = False
    NestedCrossValidator = None

# Try to import sampling
try:
    from dzetsaka.scripts.sampling import class_weights as _class_weights  # noqa: F401
    from dzetsaka.scripts.sampling import smote_sampler as _smote_sampler  # noqa: F401

    SAMPLING_AVAILABLE = True
except ImportError:
    SAMPLING_AVAILABLE = False

plugin_path = os.path.abspath(os.path.join(os.path.dirname(__file__), *([".."] * 7)))


class NestedCVAlgorithm(QgsProcessingAlgorithm):
    """Nested cross-validation for unbiased model evaluation.

    Evaluates classification models using nested cross-validation,
    which separates hyperparameter tuning from performance estimation.
    """

    INPUT_RASTER = "INPUT_RASTER"
    INPUT_LAYER = "INPUT_LAYER"
    INPUT_COLUMN = "INPUT_COLUMN"
    TRAIN = "TRAIN"
    TRAIN_ALGORITHMS = classifier_config.UI_DISPLAY_NAMES
    TRAIN_ALGORITHMS_CODE = classifier_config.CLASSIFIER_CODES
    INNER_CV = "INNER_CV"
    OUTER_CV = "OUTER_CV"
    USE_SMOTE = "USE_SMOTE"
    USE_CLASS_WEIGHTS = "USE_CLASS_WEIGHTS"
    OUTPUT_RESULTS = "OUTPUT_RESULTS"

    def shortHelpString(self):
        """Return the short help string for this algorithm."""
        return self.tr(
            """Nested Cross-Validation for Unbiased Model Evaluation.

<h3>What is Nested CV?</h3>
Standard cross-validation can overestimate model performance because
the same data is used for both hyperparameter tuning AND evaluation.

Nested CV solves this by using TWO loops:
- <b>Inner loop</b>: Tunes hyperparameters (selects best parameters)
- <b>Outer loop</b>: Evaluates model performance (estimates true accuracy)

This separation ensures test data is never used for tuning.

<h3>Parameters</h3>
- <b>Inner CV folds</b>: Number of folds for hyperparameter tuning (3-5)
- <b>Outer CV folds</b>: Number of folds for model evaluation (3-10)
- <b>Use SMOTE</b>: Apply oversampling for imbalanced datasets
- <b>Use Class Weights</b>: Apply cost-sensitive learning

<h3>Output</h3>
CSV file with:
- Per-fold accuracy scores
- Mean and std of performance
- Best hyperparameters per fold
- Class distribution information

<h3>Performance Comparison</h3>
<table>
<tr><th>Method</th><th>Bias</th><th>Speed</th></tr>
<tr><td>Standard CV</td><td>Overestimates</td><td>Fast</td></tr>
<tr><td>Nested CV</td><td>Unbiased</td><td>Slower</td></tr>
</table>

<h3>Tips</h3>
- Start with inner=3, outer=5 for balanced speed/accuracy
- Use SMOTE for datasets with >2x class imbalance
- Larger outer_cv gives more reliable performance estimates
- Compare nested CV results with standard training for reference
""",
        )

    def name(self):
        """Returns the algorithm name."""
        return "Nested cross-validation"

    def icon(self):
        """Return the algorithm icon."""
        return QIcon(os.path.join(plugin_path, "icon.png"))

    def initAlgorithm(self, config=None):
        """Initialize the algorithm parameters."""
        # Input raster
        self.addParameter(QgsProcessingParameterRasterLayer(self.INPUT_RASTER, self.tr("Input raster")))

        # Training vector layer
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_LAYER,
                self.tr("Training layer"),
            ),
        )

        # Class field
        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_COLUMN,
                self.tr("Class field"),
                parentLayerParameterName=self.INPUT_LAYER,
                optional=False,
            ),
        )

        # Classifier selection
        self.addParameter(QgsProcessingParameterEnum(self.TRAIN, self.tr("Classifier"), self.TRAIN_ALGORITHMS, 0))

        # Inner CV folds
        self.addParameter(
            QgsProcessingParameterNumber(
                self.INNER_CV,
                self.tr("Inner CV folds (hyperparameter tuning)"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=2,
                maxValue=10,
                defaultValue=3,
            ),
        )

        # Outer CV folds
        self.addParameter(
            QgsProcessingParameterNumber(
                self.OUTER_CV,
                self.tr("Outer CV folds (model evaluation)"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=3,
                maxValue=10,
                defaultValue=5,
            ),
        )

        # Use SMOTE
        self.addParameter(
            QgsProcessingParameterEnum(
                self.USE_SMOTE,
                self.tr("Apply SMOTE oversampling"),
                ["No", "Yes"],
                defaultValue=0,
            ),
        )

        # Use class weights
        self.addParameter(
            QgsProcessingParameterEnum(
                self.USE_CLASS_WEIGHTS,
                self.tr("Apply class weights"),
                ["No", "Yes"],
                defaultValue=0,
            ),
        )

        # Output results file
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_RESULTS,
                self.tr("Output results file"),
                fileFilter="CSV (*.csv)",
            ),
        )

    def processAlgorithm(self, parameters, context, feedback):
        """Process the algorithm with given parameters."""
        # Check nested CV availability
        if not NESTED_CV_AVAILABLE:
            feedback.reportError("Nested CV requires scikit-learn. Install with: pip install scikit-learn>=1.0.0")
            show_error_dialog(
                "dzetsaka Nested CV Error",
                "Nested CV requires scikit-learn>=1.0.0. Install with pip install scikit-learn>=1.0.0.",
            )
            return {}

        # Get parameters
        self.parameterAsRasterLayer(parameters, self.INPUT_RASTER, context)
        self.parameterAsVectorLayer(parameters, self.INPUT_LAYER, context)
        self.parameterAsFields(parameters, self.INPUT_COLUMN, context)
        classifier_idx = self.parameterAsEnums(parameters, self.TRAIN, context)
        inner_cv = self.parameterAsInt(parameters, self.INNER_CV, context)
        outer_cv = self.parameterAsInt(parameters, self.OUTER_CV, context)
        use_smote = self.parameterAsEnums(parameters, self.USE_SMOTE, context)[0] == 1
        use_class_weights = self.parameterAsEnums(parameters, self.USE_CLASS_WEIGHTS, context)[0] == 1
        output_path = self.parameterAsFileOutput(parameters, self.OUTPUT_RESULTS, context)

        # Get classifier code
        classifier_code = self.TRAIN_ALGORITHMS_CODE[classifier_idx[0]]
        feedback.pushInfo(f"Classifier: {classifier_code}")
        feedback.pushInfo(f"Inner CV folds: {inner_cv}")
        feedback.pushInfo(f"Outer CV folds: {outer_cv}")
        feedback.pushInfo(f"SMOTE: {'Yes' if use_smote else 'No'}")
        feedback.pushInfo(f"Class weights: {'Yes' if use_class_weights else 'No'}")

        # Load data using classification_pipeline
        try:
            from dzetsaka.scripts import classification_pipeline as _classification_pipeline  # noqa: F401

            # Create a LearnModel instance just for data loading
            feedback.pushInfo("Loading data...")
            # Note: Full integration would use LearnModel's data loading
            # This is a placeholder for the data extraction logic
            feedback.pushInfo("Data loading requires full QGIS environment")
            feedback.pushInfo("Please use LearnModel with USE_NESTED_CV=True instead")
            feedback.pushInfo(f"Output path: {output_path}")

        except ImportError as e:
            feedback.reportError(f"Failed to import classification_pipeline: {e!s}")
            show_error_dialog("dzetsaka Nested CV Error", f"Failed to import classification_pipeline: {e!s}")
            return {}

        # Log configuration
        feedback.pushInfo("\n=== Nested CV Configuration ===")
        feedback.pushInfo(f"Algorithm: {classifier_code}")
        feedback.pushInfo(f"Inner folds: {inner_cv}")
        feedback.pushInfo(f"Outer folds: {outer_cv}")
        feedback.pushInfo(f"SMOTE enabled: {use_smote}")
        feedback.pushInfo(f"Class weights: {use_class_weights}")
        feedback.pushInfo("\nUse LearnModel API for full nested CV:")
        feedback.pushInfo("  extraParam = {")
        feedback.pushInfo('    "USE_NESTED_CV": True,')
        feedback.pushInfo(f'    "NESTED_INNER_CV": {inner_cv},')
        feedback.pushInfo(f'    "NESTED_OUTER_CV": {outer_cv},')
        if use_smote:
            feedback.pushInfo('    "USE_SMOTE": True,')
        if use_class_weights:
            feedback.pushInfo('    "USE_CLASS_WEIGHTS": True,')
        feedback.pushInfo("  }")

        return {self.OUTPUT_RESULTS: output_path}

    def tr(self, string):
        """Translate string using Qt's translation system."""
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        """Create a new instance of this algorithm."""
        return NestedCVAlgorithm()

    def displayName(self):
        """Returns the translated algorithm name."""
        return self.tr(self.name())

    def group(self):
        """Returns the name of the group this algorithm belongs to."""
        return self.tr(self.groupId())

    def groupId(self):
        """Returns the unique ID of the group this algorithm belongs to."""
        return metadata_helpers.get_group_id()

    def helpUrl(self):
        """Returns a URL to the algorithm's help/documentation."""
        return metadata_helpers.get_help_url("nested_cv")

    def tags(self):
        """Returns tags for the algorithm for better searchability."""
        common = metadata_helpers.get_common_tags()
        specific = metadata_helpers.get_algorithm_specific_tags("validation")
        return common + specific
