# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DigiAgriApp
                                 A QGIS plugin
 Plugins to work with DigiAgriApp server
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-02-23
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Luca Delucchi
        email                : lucadeluge@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import sys
import os
import json
import requests
import shapely
import tempfile
from datetime import datetime
from shapely.geometry import Point
from shapely.geometry import LineString
import numpy

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtCore import QSettings
from qgis.PyQt.QtCore import QTranslator
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import QMenu
from qgis.PyQt.QtWidgets import QTableWidgetItem
from qgis.PyQt.QtWidgets import QComboBox
from qgis.PyQt.QtWidgets import QAbstractScrollArea
from qgis.PyQt.QtWidgets import QLineEdit
from qgis.PyQt.QtWidgets import QFileDialog
from qgis.PyQt.QtWidgets import QAbstractButton
from qgis.PyQt.QtCore import QVariant

from qgis.core import QgsProject
from qgis.core import Qgis
from qgis.core import QgsVectorLayer
from qgis.core import QgsRasterLayer
from qgis.core import QgsJsonExporter
from qgis.core import QgsVectorFileWriter
from qgis.core import QgsGeometry
from qgis.core import QgsFields
from qgis.core import QgsField
from qgis.core import QgsFeature
from qgis.core import QgsJsonUtils
from qgis.core import QgsPointXY
from qgis.core import QgsSymbol
from qgis.core import QgsRendererCategory
from qgis.core import QgsCategorizedSymbolRenderer

## also import pyqtRemoveInputHook
# from qgis.PyQt.QtCore import pyqtRemoveInputHook
# import pdb

# pyqtRemoveInputHook()
# pdb.set_trace()
## Initialize Qt resources from file resources.py
# from .resources_rc import *

from .functions import QuestionDialog
from .functions import createPlantPoleFeature
from .functions import ProgressBarDialog
from .functions import field_to_value_map
from .functions import join_layer_csv
from .functions import get_project_path
from .functions import set_style_plant_material_svg
from .functions import set_single_style
from .functions import set_style_plant_material_simple
from .functions import set_label
# Import the code for the dialog
from .digiagriapp_dialog_import import DigiAgriAppImport
from .digiagriapp_dialog_settings import DigiAgriAppSettings
from .digiagriapp_dialog_about import DigiAgriAppAbout
from .digiagriapp_dialog_export_vector import DigiAgriAppExportVector
from .digiagriapp_dialog_plants import DigiAgriAppPlantCreation
from .digiagriapp_dialog_style import DigiAgriAppStyle

sys.path.append(os.path.dirname(__file__))

FARM_URL = "{server}api/v1/farms/"
FARM_STATS_URL = FARM_URL + "{farmid}/stats/"
FIELD_URL = FARM_URL + "{farmid}/fields/"
SUBFIELD_URL = FIELD_URL + "{fieldid}/subfields/"
FIELD_ROW_URL = FIELD_URL + "{fieldid}/rows/"
SUBFIELD_ROW_URL = SUBFIELD_URL + "{subfieldid}/rows/"
PLANT_URL = FIELD_URL + "{fieldid}/plants/"
POLE_URL = FIELD_URL + "{fieldid}/poles/"
MATERIAL_URL = FARM_URL + "{farmid}/materials/"
FIELD_IMPORT = FARM_URL + "{farmid}/{typ}-import"
PLANTS_CREATE = "{server}api/v1/plant-full-import/"
SENSOR_URL = FARM_URL + "{farmid}/remote-sensors/"


class DigiAgriApp:
    """QGIS Plugin Implementation."""

    @staticmethod
    def _request_error(req):
        """Return the error in a request"""
        if "detail" in req.keys():
            return req["detail"]
        else:
            return req

    def _refresh_table(self):
        """Refresh table"""
        self.import_dlg.tableWidget.setSizeAdjustPolicy(
            QAbstractScrollArea.AdjustToContents
        )
        self.import_dlg.tableWidget.resizeColumnsToContents()

    def _get_farm_stats(self):
        """Return statistics about farm"""
        if self.farm_id:
            farm_id = self.farm_id
        else:
            farm_id = self.settings_dlg.farms_comboBox.currentData()
        farm_url = FARM_STATS_URL.format(
            server=self.server,
            farmid=farm_id
        )
        with requests.Session() as ses:
            try:
                req = ses.get(
                    farm_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
            except Exception as er:
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem qetting the farms: {er}"),
                    level=Qgis.Critical,
                )
                return False
            if not req.ok:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = req.text
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem to get farms; error: {reason}"),
                    level=Qgis.Critical,
                )
                return False
        return req.json()["data"]

    def _get_fields(self, farm_id=None):
        """Return available fields for farm"""
        if not farm_id:
            farm_id = self.farm_id
        server = self.settings_dlg.server_lineEdit.text()
        fields_url = FIELD_URL.format(server=server, farmid=farm_id)
        if self.year:
            fields_url += f"?year={self.year}"
        with requests.Session() as ses:
            req = ses.get(
                fields_url,
                headers={"Authorization": f"Token {self.token}"}
            )
            if not req.ok:
                reason = self._request_error(req.json())
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(
                        f"Problem to get fields for farm with ID {farm_id}; error: {reason}"
                    ),
                    level=Qgis.Critical,
                )
                return False
            self.fields = req.json()
            self.fields_id = {}
            for field in self.fields:
                self.fields_id[field["id"]] = field["name"]
        return True

    def _get_subfields(self, fieldid):
        """API call to get subfields for a field

        Args:
            fieldid (int): the ID of the field

        Returns:
            JSON: all the subfields for a field or an error
        """
        subfields_url = SUBFIELD_URL.format(
            server=self.server,
            farmid=self.farm_id,
            fieldid=fieldid
        )
        subfields_url += f"?year={self.year}"
        print(subfields_url)
        with requests.Session() as ses:
            req = ses.get(
                subfields_url,
                headers={"Authorization": f"Token {self.token}"}
            )
            if not req.ok:
                reason = self._request_error(req.json())
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr(
                        f"Problem to get subfield for farm with ID {self.farm_id} with subfield with ID {fieldid}; error: {reason}"
                    ),
                    level=Qgis.Warning,
                )
                return False
        return req.json()

    def _get_subfields_all(self):
        """Function to return all subfields for all fields

        Returns:
            bool: True if everything is fine
        """
        self.subfields_id = {}
        if not self.fields_id:
            self._get_fields()
        for key in self.fields_id.keys():
            subfields = self._get_subfields(key)
            if subfields:
                for sub in subfields:
                    self.subfields_id[sub["id"]] = sub["name"]
        return True

    def _get_rows(self, fieldid, subfieldid=None):
        """API call to get rows for a field or a subfield

        Args:
            fieldid (_type_): _description_
            subfieldid (_type_): _description_

        Returns:
            JSON: all the row for a field and subfile or an error
        """
        if subfieldid:
            rows_url = SUBFIELD_ROW_URL.format(
                server=self.server,
                farmid=self.farm_id,
                fieldid=fieldid,
                subfieldid=subfieldid
            )
        else:
            rows_url = FIELD_ROW_URL.format(
                server=self.server,
                farmid=self.farm_id,
                fieldid=fieldid
            )
        rows_url += f"?year={self.year}"
        print(rows_url)
        with requests.Session() as ses:
            try:
                req = ses.get(
                    rows_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
            except Exception as err:
                self.iface.messageBar().pushMessage(
                self.tr("Warning"),
                self.tr(f"Problem qetting the rows: {err}"),
                    level=Qgis.Warning,
                )
                return False
            if not req.ok:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = req.text
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr(f"Problem to get rows; error: {reason}"),
                    level=Qgis.Warning,
                )
                return False
        return req.json()

    def _get_rows_all(self):
        self.rows_id = {}
        if not self.fields_id:
            self._get_fields()
        for key in self.fields_id.keys():
            rows = self._get_rows(key)
            if rows:
                for row in rows:
                    self.rows_id[row["id"]] = row["name"]
        return True

    def _get_farms(self):
        """Return available farms for user"""
        print("inn _get_faarm")
        server = self.settings_dlg.server_lineEdit.text()
        farms_url = FARM_URL.format(server=server)
        with requests.Session() as ses:
            try:
                req = ses.get(
                    farms_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
            except Exception as er:
                self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(f"Problem qetting the farms: {er}"),
                    level=Qgis.Warning,
                )
                return False
            if not req.ok:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = req.text
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem to get farms; error: {reason}"),
                    level=Qgis.Warning,
                )
                return False
            self.settings_dlg.farms_comboBox.clear()
            self.settings_dlg.farms_comboBox.addItem("---", None)
            for farm in req.json():
                self.settings_dlg.farms_comboBox.addItem(
                    farm["name"],
                    farm["id"]
                )
        self.settings_dlg.farms_comboBox.setEnabled(True)
        if not self.farm:
            if self.settings.contains("/DigiAgriApp/farm"):
                self.farm = self.settings.value("/DigiAgriApp/farm", self.farm)
            if self.settings.contains("/DigiAgriApp/farm_id"):
                self.farm_id = self.settings.value("/DigiAgriApp/farm_id", self.farm_id)
        if self.farm and self.farm_id:
            farmindex = self.settings_dlg.farms_comboBox.findData(self.farm_id)
            self.settings_dlg.farms_comboBox.setCurrentIndex(farmindex)
            self.farm_stats = self._get_farm_stats()
            self._get_fields()
        else:
            return False
        return True

    def _get_material(self, csv=False):
        """Return material in a QGIS table"""
        print("inside get material")
        farm = self.settings_dlg.farms_comboBox.currentText()
        farm_id = self.settings_dlg.farms_comboBox.currentData()
        server = self.settings_dlg.server_lineEdit.text()
        if not farm_id:
            return False
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading material for farm {farm}"),
            title=self.tr("DigiAgriApp: download material")
        )
        if csv:
            mats_url = MATERIAL_URL.format(server=server, farmid=farm_id) + "?format=csv"
        else:
            mats_url = MATERIAL_URL.format(server=server, farmid=farm_id) + "?full=yes"
        with requests.Session() as ses:
            try:
                req = ses.get(
                    mats_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
            except Exception as er:
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem qetting the farms: {er}"),
                        level=Qgis.Warning,
                )
                pbd.hide()
                return False
            if not req.ok:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = req.text
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem to get farms; error: {reason}")
                )
                pbd.hide()
                return False
        if csv:
            if self.output_dir:
                csv_path = os.path.join(self.output_dir, f"{farm}_material.csv")
            else:
                csv_path = f"{tempfile.mkstemp()[1]}.csv"
            fil = open(csv_path, mode='w')
            fil.write(req.text)
            fil.close()
            pbd.setValue(50)
            csv_uri = f"file://{csv_path}?&delimiter=,&trimFields=yes&geomType=none"
            self.csv_layer = QgsVectorLayer(
                csv_uri, f"{self.farm.replace(' ', '_')}_material_csv", "delimitedtext"
            )
            QgsProject.instance().addMapLayer(self.csv_layer)
            pbd.setValue(100)
        else:
            mats = req.json()
            total_fields = len(mats)
            n_fields = 1
            year = datetime.today().year
            minyear = year
            maxyear = year
            for mat in mats:
                try:
                    year = int(mat["start_date"].split("-")[0])
                    if year < minyear and year >= 1950:
                        minyear = year
                except:
                    pass
                self.material[mat["id"]] = mat
                if total_fields != 0:
                    pbd.setValue(int((n_fields / total_fields) * 100))
                n_fields += 1
            self.minmax_years = [minyear, maxyear]
        print("finish get material")
        pbd.hide()
        return True

    def _get_sensors(self):
        """Get the sensor for the farm

        """
        farm = self.settings_dlg.farms_comboBox.currentText()
        farm_id = self.settings_dlg.farms_comboBox.currentData()
        server = self.settings_dlg.server_lineEdit.text()
        if not farm_id:
            return False
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading remote sensing sensors for farm {farm}"),
            title=self.tr("DigiAgriApp: download remote sensing sensors")
        )
        sensor_url = SENSOR_URL.format(server=server, farmid=farm_id)
        with requests.Session() as ses:
            try:
                req = ses.get(
                    sensor_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
            except Exception as er:
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem qetting the farms: {er}"),
                        level=Qgis.Warning,
                )
                pbd.hide()
                return False
            if not req.ok:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = req.text
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr(f"Problem to get farms; error: {reason}")
                )
                pbd.hide()
                return False
            pbd.setValue(100)
        pbd.hide()
        return req.json()

    def _check_csv_layer(self):
        farm = self.settings_dlg.farms_comboBox.currentText()
        if self.output_dir:
            csv_path = os.path.join(self.output_dir, f"{farm.replace(' ', '_')}_material.csv")
            csv_uri = f"file://{csv_path}?&delimiter=,&trimFields=yes&geomType=none"
            self.csv_layer = QgsVectorLayer(
                csv_uri, f"{self.farm.replace(' ', '_')}_material_csv", "delimitedtext"
            )
            if not QgsProject.instance().mapLayersByName(self.csv_layer.name()):
                QgsProject.instance().addMapLayer(self.csv_layer)
        else:
            for layer in QgsProject.instance().mapLayers().values():
                if layer.name() == f"{farm.replace(' ', '_')}_material_csv":
                    self.csv_layer = layer
                    break

    def _check_year(self):
        if len(self.minmax_years) < 2:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(f"Problem with minimum and maximum year, no material could be assigned to this farm"),
            )
            return False
        self.settings_dlg.comboBox_year.clear()
        self.settings_dlg.comboBox_year.addItem("---", None)
        for year in range(self.minmax_years[0], self.minmax_years[1] + 1):
            self.settings_dlg.comboBox_year.addItem(
                str(year), year
            )
        self.settings_dlg.comboBox_year.setEnabled(True)
        if self.settings.contains("/DigiAgriApp/year"):
            self.year = self.settings.value("/DigiAgriApp/year")
            yearindex = self.settings_dlg.comboBox_year.findData(self.year)
            self.settings_dlg.comboBox_year.setCurrentIndex(yearindex)
            return True
        return False

    def _update_materials_years(self):
        """Function to renew material and year when farm change"""
        current = self.settings_dlg.farms_comboBox.currentData()
        if current:
            print(current, self.tmp_farm_id)
            if current != self.tmp_farm_id:
                self._get_material()
                self._check_year()
                self.farm_stats = self._get_farm_stats()
                self._get_fields(self.tmp_farm_id)
                self.tmp_farm_id = current
            return True
        return False

    def _check_server_ending(self):
        """Check if the server string ends with / otherwise added it"""
        server = self.settings_dlg.server_lineEdit.text()
        if not server.endswith("/"):
            server += "/"
            self.settings_dlg.server_lineEdit.setText(server)
            self.server = server
        return True

    def _check_server(self):
        """Check is server is save to settings"""
        if self.settings.contains("/DigiAgriApp/server"):
            self.settings_dlg.server_lineEdit.setText(
                self.settings.value("/DigiAgriApp/server")
            )
            self.server = self.settings.value("/DigiAgriApp/server")
            return True
        return False

    def _check_user(self):
        if self.settings.contains("/DigiAgriApp/user"):
            self.settings_dlg.user_lineEdit.setText(
                self.settings.value("/DigiAgriApp/user")
            )
            self.user = self.settings.value("/DigiAgriApp/user")
        if self.settings.contains("/DigiAgriApp/password"):
            self.settings_dlg.password_lineEdit.setText(
                self.settings.value("/DigiAgriApp/password")
            )
            self.password = self.settings.value("/DigiAgriApp/user")

    def _check_token(self):
        """Check if the token is saved in the settings"""
        if self.token:
            return True
        return False

    def _check_farm(self):
        """Check if the farm is saved in the setting"""
        if self.settings.contains("/DigiAgriApp/farm"):
            self.farm = self.settings.value("/DigiAgriApp/farm")
        if self.settings.contains("/DigiAgriApp/farm_id"):
            self.farm_id = self.settings.value("/DigiAgriApp/farm_id")
        if self.farm and self.farm_id:
            self.settings_dlg.comboBox_year.setEnabled(True)
            return True
        else:
            return False

    def _check_style(self):
        if self.settings.contains("/DigiAgriApp/style"):
            if self.settings.value("/DigiAgriApp/style") == "svg":
                self.settings_dlg.radioButtonMaterialSvg.setChecked(True)
            elif self.settings.value("/DigiAgriApp/style") == "icon":
                self.settings_dlg.radioButtonMaterialIcon.setChecked(True)
            else:
                self.settings_dlg.radioButtonSingle.setChecked(True)
            self._check_material()
        if self.settings.contains("/DigiAgriApp/style_colname"):
            self.style_colname = self.settings.value("/DigiAgriApp/style_colname")
            nameindex = self.settings_dlg.comboBoxColumnMaterial.findText(self.style_colname)
            self.settings_dlg.comboBoxColumnMaterial.setCurrentIndex(nameindex)
        if self.settings.contains("/DigiAgriApp/style_labelname"):
            self.style_labelname = self.settings.value("/DigiAgriApp/style_labelname")
            nameindex = self.settings_dlg.comboBoxLabel.findText(self.style_labelname)
            self.settings_dlg.comboBoxLabel.setCurrentIndex(nameindex)
        if self.settings.contains("/DigiAgriApp/style_iconsize"):
            self.style_icon_size = self.settings.value("/DigiAgriApp/style_iconsize")
            self.settings_dlg.spinBoxIconSize.setValue(int(self.style_icon_size))
        if self.settings.contains("/DigiAgriApp/style_textsize"):
            self.style_text_size = self.settings.value("/DigiAgriApp/style_textsize")
            self.settings_dlg.spinBoxTextSize.setValue(int(self.style_text_size))
        return True

    def _get_token(self):
        user = self.settings_dlg.user_lineEdit.text()
        password = self.settings_dlg.password_lineEdit.text()
        server = self.settings_dlg.server_lineEdit.text()
        loginurl = server + "api/token-auth/"
        try:
            req = requests.post(
                loginurl,
                data={
                    "username": user,
                    "password": password,
                },
                timeout=400,
            )
        except Exception as er:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(f"Problem requesting the token: {er}"),
                level=Qgis.Critical,
            )
            raise er
        try:
            self.token = req.json()["token"]
        except:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(
                    "Problem to get the token, please check server, user and password"
                ),
                level=Qgis.Critical,
            )
            return False
        print("finished get token")
        self._get_farms()
        self.iface.messageBar().pushMessage(
            self.tr("Success"),
            self.tr("Token acquired correctly"),
            level=Qgis.Success,
        )
        self.settings_dlg.tabWidget.setCurrentIndex(1)
        return True


    def _save_settings(self):
        """Save settings"""
        user = self.settings_dlg.user_lineEdit.text()
        password = self.settings_dlg.password_lineEdit.text()
        self.server = self.settings_dlg.server_lineEdit.text()
        self.farm = self.settings_dlg.farms_comboBox.currentText()
        self.farm_id = self.settings_dlg.farms_comboBox.currentData()
        self.year = self.settings_dlg.comboBox_year.currentText()
        if self.settings_dlg.radioButtonMaterialSvg.isChecked():
            self.style_checkbox = "svg"
        elif self.settings_dlg.radioButtonMaterialIcon.isChecked():
            self.style_checkbox = "icon"
        else:
            self.style_checkbox = "single"

        if self.settings_dlg.comboBoxColumnMaterial.isEnabled():
            self.style_colname = self.settings_dlg.comboBoxColumnMaterial.currentText()
        else:
            self.style_colname = None
        self.style_labelname = self.settings_dlg.comboBoxLabel.currentText()
        self.style_icon_size = self.settings_dlg.spinBoxIconSize.value()
        self.style_text_size = self.settings_dlg.spinBoxTextSize.value()

        missing = []
        if not self.server:
            missing.append("server")
        else:
            if not self.server.endswith("/"):
                self.server += "/"
            self.settings.setValue("/DigiAgriApp/server", self.server)
        if not user:
            missing.append("user")
        else:
            self.settings.setValue("/DigiAgriApp/user", user)
        if not password:
            missing.append("password")
        else:
            self.settings.setValue("/DigiAgriApp/password", password)
        # if not self.token:
        #     missing.append("token")
        # else:
        #     self.settings.setValue("/DigiAgriApp/token", self.token)
        if self.farm:
            self.settings.setValue("/DigiAgriApp/farm", self.farm)
            self.settings.setValue("/DigiAgriApp/farm_id", self.farm_id)
        else:
            missing.append("field")
        if self.year:
            self.settings.setValue("/DigiAgriApp/year", self.year)
        else:
            missing.append("year")
        if len(missing) != 0:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr("Please set {} values".format(",".join(missing))),
                level=Qgis.Critical,
            )
            return False
        if self.style_checkbox == "svg":
            self.settings.setValue("/DigiAgriApp/style", "svg")
        elif self.style_checkbox == "icon":
            self.settings.setValue("/DigiAgriApp/style", "icon")
        else:
            self.settings.setValue("/DigiAgriApp/style", "single")
        if self.style_colname:
            self.settings.setValue("/DigiAgriApp/style_colname", self.style_colname)
        if self.style_labelname:
            self.settings.setValue("/DigiAgriApp/style_labelname", self.style_labelname)
        if self.style_icon_size:
            self.settings.setValue("/DigiAgriApp/style_iconsize", self.style_icon_size)
        if self.style_text_size:
            self.settings.setValue("/DigiAgriApp/style_textsize", self.style_text_size)
        self._get_fields()
        self.settings_dlg.tabWidget.setCurrentIndex(0)
        self.settings_dlg.close()

    def _working_server(self):
        """Function to check if everything is working properly with the server"""
        if not self._check_server():
            self.iface.messageBar().pushMessage(
                                self.tr("Error"),
                                self.tr("Impossible get the server url, please check the settings."),
                                level=Qgis.Critical,
                            )
            self.show_settings_dialog()
            return False
        token_exist = self._check_token()
        if not token_exist:
            self.iface.messageBar().pushMessage(
                                self.tr("Error"),
                                self.tr("Impossible connect to the server, please check the settings."),
                                level=Qgis.Critical,
                            )
            self.show_settings_dialog()
            return False
        if not self._check_farm():
            self.iface.messageBar().pushMessage(
                self.tr("Warning"),
                self.tr("Farms seems not set, please set it"),
                level=Qgis.Warning,
            )
            self.show_settings_dialog()
            return False
        return True

    def _get_directory(self, dialog):
        """Select a file"""
        dlg = QFileDialog.getExistingDirectory(None, caption="", directory="", options=QFileDialog.DontConfirmOverwrite)
        if dlg:
            if dialog == "settings_out_dif":
                self.settings_dlg.lineEditOutput.setText(dlg)
                self.output_dir = dlg
                self.settings.setValue("/DigiAgriApp/output_directory", self.output_dir)
            if dialog == "settings_svg_dir":
                self.settings_dlg.lineEditSvg.setText(dlg)
                self.svg_dir = dlg
                self.settings.setValue("/DigiAgriApp/svg_directory", self.svg_dir)
        return True

    def _get_file(self, dialog):
        """Select a file"""
        dlg = QFileDialog.getSaveFileName(None, 'inFile', options=QFileDialog.DontConfirmOverwrite)
        if dlg[0]:
            if dialog == "export":
                self.export_vector_dlg.lineEditOutput.setText(dlg[0])
            elif dialog == "plants":
                self.plants_dlg.lineEditOutput.setText(dlg[0])
        return True

    def _import_vector(self, layer, allfeats, import_url):
        fields = [field.name() for field in layer.fields()]
        layer_name = layer.name
        export = QgsJsonExporter(layer)
        try:
            fidind = fields.index("fid")
        except ValueError:
            fidind = None
        if fidind is not None:
            export.setExcludedAttributes([fidind])
        try:
            export.setUseFieldFormatters(False)
        except:
            pass
        if allfeats:
            layer_json = export.exportFeatures(
                layer.getFeatures()
            )
        else:
            layer_json = export.exportFeatures(
                layer.selectedFeatures()
            )
        if import_url[-1] != "/":
            import_url += "/"
        # print(import_url)
        # print(layer_json)
        with requests.Session() as ses:
            req = ses.post(
                import_url,
                data=layer_json,
                headers={
                    "Authorization": f"Token {self.token}",
                    "Content-type": "application/json"
                },
            )
            print(req.status_code)
            print(req.text)
            if req.ok:
                self.iface.messageBar().pushMessage(
                    self.tr("Success"),
                    self.tr(f"Layer '{layer_name}' imported correctly"),
                    level=Qgis.Success,
                )
                return True
            else:
                try:
                    reason = self._request_error(req.json())
                except:
                    reason = "Server failing with unknown error"
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr(
                        f"Layer '{layer_name}' not imported reason: {reason}"
                    ),
                    level=Qgis.Warning,
                )
                return False

    def _export_field(self, file_output, year=None):
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading fields for farm {self.farm}"),
            title=self.tr("DigiAgriApp: download fields")
        )
        field_fields = QgsFields()
        field_fields.append(QgsField("id", QVariant.Int))
        field_fields.append(QgsField("farm", QVariant.Int))
        field_fields.append(QgsField("name", QVariant.String))
        field_fields.append(QgsField("management", QVariant.String))
        field_fields.append(QgsField("start_date", QVariant.String))
        field_fields.append(QgsField("end_date", QVariant.String))
        field_fields.append(QgsField("soil", QVariant.Int))
        field_fields.append(QgsField("note", QVariant.String))
        field_fields.append(QgsField("active_materials", QVariant.String))
        field_fields.append(QgsField("last_treatment", QVariant.String))
        field_fields.append(QgsField("last_operation", QVariant.String))

        vlf = QgsVectorLayer(
            "MultiPolygon",
            f"{self.farm.replace(' ', '_')}_digiagriapp_field",
            "memory"
        )
        prf = vlf.dataProvider()
        prf.addAttributes(field_fields)
        vlf.updateFields()
        total_fields = len(self.fields)
        n_fields = 0
        for field in self.fields:
            feat = QgsJsonUtils.stringToFeatureList(
                json.dumps(field["geom"]),
                field_fields,
                None
            )[0]
            print("prima", field["id"], feat.id())
            #feat.setId(field["id"])
            print("dopo", field["id"], feat.id())
            try:
                active_materials = " | ".join(field["active_materials"])
            except:
                active_materials = None
            feat.setAttributes([
                field["id"],
                field["farm"],
                field["name"],
                field["management"],
                field["start_date"],
                field["end_date"],
                field["soil"],
                field["note"],
                active_materials,
                field["last_treatment"],
                field["last_operation"],
            ])
            prf.addFeatures([feat])
            n_fields += 1
            if total_fields != 0:
                pbd.setValue(int((n_fields / total_fields) * 100))
        self._write_vector(vlf, file_output)
        pbd.hide()
        return True

    def _export_subfield(self, file_output, year=None):
        """Function to export to geopackage the subfields

        Args:
            file_output (str): file path to the geopackage file
        """
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading subfields for farm {self.farm}"),
            title=self.tr("DigiAgriApp: download subfields")
        )
        subfield_fields = QgsFields()
        subfield_fields.append(QgsField("id", QVariant.Int))
        subfield_fields.append(QgsField("field", QVariant.Int))
        subfield_fields.append(QgsField("name", QVariant.String))
        subfield_fields.append(QgsField("start_date", QVariant.String))
        subfield_fields.append(QgsField("end_date", QVariant.String))
        subfield_fields.append(QgsField("note", QVariant.String))
        subfield_fields.append(QgsField("active_materials", QVariant.String))
        subfield_fields.append(QgsField("last_treatment", QVariant.String))
        subfield_fields.append(QgsField("last_operation", QVariant.String))

        vls = QgsVectorLayer(
            "Polygon",
            f"{self.farm.replace(' ', '_')}_digiagriapp_subfield",
            "memory"
        )
        prs = vls.dataProvider()
        prs.addAttributes(subfield_fields)
        vls.updateFields()
        if not self.farm_stats:
            self.farm_stats = self._get_farm_stats()
        total_subfields = self.farm_stats["subfields_count"]
        n_subfields = 0
        for field in self.fields:
            subfields = self._get_subfields(field["id"])
            if not subfields:
                continue
            for subfield in subfields:
                feat = QgsJsonUtils.stringToFeatureList(
                    json.dumps(subfield["geom"]),
                    subfield_fields,
                    None
                )[0]
                try:
                    active_materials = " | ".join(subfield["active_materials"])
                except:
                    active_materials = None
                feat.setAttributes([
                    subfield["id"],
                    subfield["field"],
                    subfield["name"],
                    subfield["start_date"],
                    subfield["end_date"],
                    subfield["note"],
                    active_materials,
                    subfield["last_treatment"],
                    subfield["last_operation"],
                ])
                prs.addFeatures([feat])
                n_subfields += 1
                if total_subfields != 0:
                    pbd.setValue(int((n_subfields / total_subfields) * 100))
        self._write_vector(vls, file_output)
        pbd.hide()
        return True

    def _export_row(self, file_output):
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading rows for farm {self.farm}"),
            title=self.tr("DigiAgriApp: download rows")
        )
        row_fields = QgsFields()
        row_fields.append(QgsField("id", QVariant.Int))
        row_fields.append(QgsField("field", QVariant.Int))
        row_fields.append(QgsField("subfield", QVariant.Int))
        row_fields.append(QgsField("name", QVariant.String))
        row_fields.append(QgsField("note", QVariant.String))
        row_fields.append(QgsField("active_materials", QVariant.String))
        row_fields.append(QgsField("last_treatment", QVariant.String))
        row_fields.append(QgsField("last_operation", QVariant.String))

        vlr = QgsVectorLayer(
            "LineString",
            f"{self.farm.replace(' ', '_')}_digiagriapp_row",
            "memory"
        )
        prr = vlr.dataProvider()
        prr.addAttributes(row_fields)
        vlr.updateFields()
        if not self.farm_stats:
            self.farm_stats = self._get_farm_stats()
        total_rows = self.farm_stats["rows_count"]
        n_rows = 0
        for fieldid, fieldname in self.fields_id.items():
            rows = self._get_rows(fieldid)
            if not rows:
                continue
            for row in rows:
                feat = QgsJsonUtils.stringToFeatureList(
                    json.dumps(row["geom"]),
                    row_fields,
                    None
                )[0]
                try:
                    active_materials = " | ".join(row["active_materials"])
                except:
                    active_materials = None
                feat.setAttributes([
                    row["id"],
                    row["field"],
                    row["subfield"],
                    row["name"],
                    row["note"],
                    active_materials,
                    row["last_treatment"],
                    row["last_operation"],
                ])
                prr.addFeatures([feat])
                n_rows += 1
                if total_rows != 0:
                    pbd.setValue(int((n_rows / total_rows) * 100))
        self._write_vector(vlr, file_output)
        pbd.hide()
        return True

    def _export_plant(self, file_output):
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading plants for farm {self.farm}"),
            title=self.tr("DigiAgriApp: download plant")
        )
        plant_fields = QgsFields()
        plant_fields.append(QgsField("id", QVariant.Int))
        plant_fields.append(QgsField("field", QVariant.Int))
        plant_fields.append(QgsField("subfield", QVariant.Int))
        plant_fields.append(QgsField("row", QVariant.Int))
        plant_fields.append(QgsField("name", QVariant.String))
        plant_fields.append(QgsField("full_name", QVariant.String))
        plant_fields.append(QgsField("block", QVariant.Int))
        plant_fields.append(QgsField("position", QVariant.Int))
        plant_fields.append(QgsField("tag", QVariant.String))
        plant_fields.append(QgsField("note", QVariant.String))
        plant_fields.append(QgsField("active_material_id", QVariant.Int))
        plant_fields.append(QgsField("active_materials", QVariant.String))
        plant_fields.append(QgsField("last_treatment", QVariant.String))
        plant_fields.append(QgsField("last_operation", QVariant.String))
        plant_fields.append(QgsField("experiment", QVariant.String))
        plant_fields.append(QgsField("thesis", QVariant.String))
        plant_fields.append(QgsField("icon", QVariant.String))

        vlp = QgsVectorLayer(
            "Point",
            f"{self.farm.replace(' ', '_')}_digiagriapp_plant",
            "memory"
        )
        prp = vlp.dataProvider()
        prp.addAttributes(plant_fields)
        vlp.updateFields()
        if not self.farm_stats:
            self.farm_stats = self._get_farm_stats()
        total_plants = self.farm_stats["plants_count"]
        n_plants = 0
        for fieldid, fieldname in self.fields_id.items():
            plants_url = PLANT_URL.format(
                server=self.server,
                farmid=self.farm_id,
                fieldid=fieldid
            )
            plants_url += f"?year={self.year}"
            with requests.Session() as ses:
                req = ses.get(
                    plants_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
                if not req.ok:
                    reason = self._request_error(req.json())
                    self.iface.messageBar().pushMessage(
                        self.tr("Warning"),
                        self.tr(
                            f"Problem to get plants for farm with ID {self.farm_id} with field with ID {fieldid}; error: {reason}"
                        ),
                        level=Qgis.Warning,
                    )
                    continue
                plants = req.json()
                for plant in plants:
                    feat = QgsJsonUtils.stringToFeatureList(
                        json.dumps(plant["geom"]),
                        plant_fields,
                        None
                    )[0]
                    #feat.setId(plant["id"])
                    try:
                        active_materials = " | ".join(plant["active_materials"])
                    except Exception:
                        active_materials = None
                    try:
                        if len(plant["active_materials_id"]) > 0:
                            active_material_id = plant["active_materials_id"][0]
                        else:
                            active_material_id = None
                    except Exception:
                        active_material_id = None
                    feat.setAttributes([
                        plant["id"],
                        plant["field"],
                        plant["subfield"],
                        plant["row"],
                        plant["name"],
                        plant["full_name"],
                        plant["block"],
                        plant["position"],
                        plant["tag"],
                        plant["note"],
                        active_material_id,
                        active_materials,
                        plant["last_treatment"],
                        plant["last_operation"],
                        plant["experiment_name"],
                        plant["thesis_name"],
                        plant["icon"]
                    ])
                    prp.addFeatures([feat])
                    n_plants += 1
                    if total_plants != 0:
                        pbd.setValue(int((n_plants / total_plants) * 100))
        self._write_vector(vlp, file_output)
        pbd.hide()
        return True

    def _export_pole(self, file_output):
        """Function to export infrestructure pole"""
        pbd = ProgressBarDialog(
            parent=self.iface.mainWindow(),
            label=self.tr(f"Downloading poles for farm {self.farm}"),
            title=self.tr("DigiAgriApp: download poles")
        )
        pole_fields = QgsFields()
        pole_fields.append(QgsField("id", QVariant.Int))
        pole_fields.append(QgsField("farm", QVariant.Int))
        pole_fields.append(QgsField("field", QVariant.Int))
        pole_fields.append(QgsField("subfield", QVariant.Int))
        pole_fields.append(QgsField("row", QVariant.Int))
        pole_fields.append(QgsField("typee", QVariant.String))
        pole_fields.append(QgsField("note", QVariant.String))

        vlp = QgsVectorLayer(
            "Point",
            f"{self.farm.replace(' ', '_')}_digiagriapp_pole",
            "memory"
        )
        prp = vlp.dataProvider()
        prp.addAttributes(pole_fields)
        vlp.updateFields()
        if not self.farm_stats:
            self.farm_stats = self._get_farm_stats()
        total_poles = self.farm_stats["poles_count"]
        n_poles = 0
        for fieldid, fieldname in self.fields_id.items():
            poles_url = POLE_URL.format(
                server=self.server,
                farmid=self.farm_id,
                fieldid=fieldid
            )
            print(poles_url)
            with requests.Session() as ses:
                req = ses.get(
                    poles_url,
                    headers={"Authorization": f"Token {self.token}"}
                )
                if not req.ok:
                    reason = self._request_error(req.json())
                    self.iface.messageBar().pushMessage(
                        self.tr("Warning"),
                        self.tr(
                            f"Problem to get poles for farm with ID {self.farm_id} with field with ID {fieldid}; error: {reason}"
                        ),
                        level=Qgis.Warning,
                    )
                    continue
                poles = req.json()
                for plant in poles:
                    feat = QgsJsonUtils.stringToFeatureList(
                        json.dumps(plant["geom"]),
                        pole_fields,
                        None
                    )[0]
                    #feat.setId(plant["id"])
                    feat.setAttributes([
                        plant["id"],
                        plant["farm"],
                        plant["field"],
                        plant["subfield"],
                        plant["row"],
                        plant["typee"],
                        plant["note"],
                    ])
                    prp.addFeatures([feat])
                    n_poles += 1
                    if total_poles != 0:
                        pbd.setValue(int((n_poles / total_poles) * 100))
        self._write_vector(vlp, file_output)
        pbd.hide()
        return True

    def _reload_layer(self, infile, layername):
        """Function to reload a layer in QGIS canva

        Args:
            infile (str): the path to gpkg file
            layername (str): the name of layer to load, it can be field, subfield, row, plant
        """
        layers = QgsProject.instance().mapLayersByName(f"{self.farm.replace(' ', '_')}_digiagriapp_{layername}")
        #csv = QgsProject.instance().mapLayersByName("material_csv")[0]
        layer = None
        if len(layers) == 0:
            layer = self.iface.addVectorLayer(
                f"{infile}|layername={self.farm.replace(' ', '_')}_digiagriapp_{layername}",
                f"{self.farm.replace(' ', '_')}_digiagriapp_{layername}",
                "ogr"
            )
        elif len(layers) == 1:
            layer = layers[0]
            layer.reload()
        else:
            QgsProject.instance().reloadAllLayers()
        return layer

    def _create_points(self, line, info, row_name, field, subfield, rowid, check_poles = False, setted_distance=0.2):
        """Function to create points along the line"""
        linecoors = line.coords
        npoints = info["pole_number"]
        plants_block = info["plants_block"]
        first = linecoors[0]
        last = linecoors[-1]
        # work only with shapely > 2
        if shapely.__version__ > "2.0.0":
            if last[1] < first[1]:
                line = line.reverse()
                linecoors = line.coords
        else:
            if last[1] < first[1]:
                linelist = list(linecoors)
                linelist.reverse()
                line = LineString(linelist)
                linecoors = line.coords
        # create qgis vector for the line
        row = QgsVectorLayer("LineString", f"temporary_{row_name}_row", "memory")
        row_prov = row.dataProvider()
        row_prov.addAttributes([QgsField("name", QVariant.String)])
        row.updateFields()
        row_feat = QgsFeature()
        row_feat.setGeometry(QgsGeometry.fromWkt(line.wkt))
        row_feat.setAttributes([row_name])
        print(row_feat.isValid(), row_prov.featureCount())
        if not row_prov.addFeatures([row_feat]):
            print(f"{row_feat} not added")
        row.updateExtents()
        QgsProject.instance().addMapLayer(row)
        line_coords = []
        for coor in linecoors:
            line_coords.append(coor)
        usepoints = npoints - 1

        # epsg = get_utm_srid_from_extent(line.bounds)
        # utm = f"EPSG:{epsg}"
        # wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        # transformContext = QgsProject.instance().transformContext()
        # xform = QgsCoordinateTransform(wgs84, utm, transformContext)
        # utm_line = xform.transform(line)

        # epsg = utm_from_extent(line.bounds)
        # wgs84 = pyproj.CRS("EPSG:4326")
        # project = pyproj.Transformer.from_crs(wgs84, utm, always_xy=True).transform
        # utm_line = geotransform(project, line)

        # create the point of poles interpolation the line
        polepoints = [line.interpolate((i/usepoints), normalized=True) for i in range(1, usepoints)]
        segment_num = 0
        final_points = [line_coords[0]]
        # create vector for points
        plants = QgsVectorLayer("Point", f"temporary_{row_name}_plants", "memory")
        plants_prov = plants.dataProvider()
        plants_prov.addAttributes([
            QgsField("type", QVariant.String),
            QgsField("material", QVariant.Int),
            QgsField("label", QVariant.String),
            QgsField("thesis", QVariant.Int),
            QgsField("experiment", QVariant.Int),
            QgsField("farm", QVariant.Int),
            QgsField("field", QVariant.Int),
            QgsField("subfield", QVariant.Int),
            QgsField("row", QVariant.Int),
        ])
        plants.updateFields()

        totalpoints = 1
        plantsnumb = 0
        # check if first is pole or plant
        if info["first"] == "pole":
            plant_feat = QgsFeature()
            plant_feat.setGeometry(
                QgsGeometry.fromPointXY(
                    QgsPointXY(line_coords[0][0], line_coords[0][1])
                )
            )
            plant_feat.setAttributes(
                ["pole", None, None, None, None, self.farm_id, field, subfield, rowid]
            )
            print(plant_feat.isValid(), plants_prov.featureCount())
            if not plants_prov.addFeatures([plant_feat]):
                print(f"{plant_feat} not added")
            #
            norm = numpy.random.normal(plants_block[1:-1])
            minor_value = round(norm.mean() - norm.std() -1)
            next_poi = None
            secondtolast_poi = None
            final_line  = None
            last_poi = Point(line_coords[-1])
            # first segment is for shorter block
            if plants_block[0] < minor_value and not check_poles:
                thisline = LineString([line_coords[0], list(polepoints[0].coords)[0]])
                leng = thisline.length
                space = leng / max(plants_block)
                maxleng = space * plants_block[0]
                next_poi = thisline.interpolate(maxleng)
            # last segment
            if info["last"] == "plant":
                if next_poi:
                    linereverse = LineString([line_coords[-1], next_poi])
                else:
                    linereverse = LineString([line_coords[-1], line_coords[0]])
                last_poi = linereverse.interpolate(0.0000045)
                if plants_block[-2] < minor_value and not check_poles:
                    final_line = LineString([polepoints[-1].coords[0], last_poi.coords[0]])
            else:
                if plants_block[-1] < minor_value and not check_poles:
                    final_line = LineString([polepoints[-1].coords[0], line_coords[-1]])
            if final_line:
                leng = final_line.length
                space = leng / max(plants_block)
                maxleng = space * plants_block[-1]
                if shapely.__version__ > "2.0.0":
                    linereverse = final_line.reverse()
                else:
                    linereverse = LineString([line_coords[-1], list(polepoints[-1].coords)[0]])
                secondtolast_poi = linereverse.interpolate(maxleng)
            finalpolepoints = [Point(line.coords[0])]
            if next_poi and secondtolast_poi:
                newline = LineString([next_poi, secondtolast_poi])
                newusepoints = usepoints - 2
                newpolepoints = [newline.interpolate((i/newusepoints), normalized=True) for i in range(1, newusepoints)]
                finalpolepoints.append(next_poi)
                finalpolepoints.extend(newpolepoints)
                finalpolepoints.append(secondtolast_poi)
            elif next_poi and not secondtolast_poi:
                newline = LineString([next_poi, line_coords[-1]])
                newusepoints = usepoints - 1
                newpolepoints = [newline.interpolate((i/newusepoints), normalized=True) for i in range(1, newusepoints)]
                finalpolepoints.append(next_poi)
                finalpolepoints.extend(newpolepoints)
            elif secondtolast_poi and not next_poi:
                newline = LineString([line_coords[0], secondtolast_poi])
                newusepoints = usepoints - 1
                newpolepoints = [newline.interpolate((i/newusepoints), normalized=True) for i in range(1, newusepoints)]
                finalpolepoints.extend(newpolepoints)
                finalpolepoints.append(secondtolast_poi)
            else:
                finalpolepoints.extend(polepoints)
            finalpolepoints.append(last_poi)
        # first is a plant
        else:
            next_poi = None
            distance = 0
            for npoi in range(info["plants_block"][0]):
                plant_info = info["points"][npoi]
                if plant_info["material"] in ["F", "f"]:
                    # distance 60cm
                    distance += 0.6 / 111139
                elif plant_info["material"] in ["", "p", "P"]:
                    raise ValueError("This plant seem to be a pole")
                else:
                    if "distance" in plant_info.keys():
                        distance += plant_info["distance"] / 111139
                    else:
                        distance += 0.6 / 111139
                if next_poi is None:
                    geom = QgsGeometry.fromPointXY(
                        QgsPointXY(line_coords[0][0], line_coords[0][1])
                    )
                else:
                    nextcoords = list(next_poi.coords)[0]
                    geom = QgsGeometry.fromPointXY(
                        QgsPointXY(nextcoords[0], nextcoords[1])
                    )

                plant_info["type"] = "plant"
                plant_feat = createPlantPoleFeature(
                    plant_info,
                    geom,
                    self.farm_id,
                    field,
                    subfield,
                    rowid
                )
                print(plant_feat.isValid(), plants_prov.featureCount())
                if plant_feat.isValid():
                    if not plants_prov.addFeature(plant_feat):
                        print(f"{plant_feat} not added")


                # move the first pole 50cm (second row in the csv file)
                next_poi = line.interpolate(distance)
                plantsnumb += 1
            secondtolast_poi = None
            # if last point not pole
            finalpolepoints = [next_poi]
            if info["last"] == "plant":
                final_distance = 0
                for npoi in range(info["plants_block"][-1]):
                    if info["points"][npoi]["material"] in ["F", "f"]:
                        material = None
                        # distance 60cm
                        final_distance += 0.6 / 111139
                    elif info["points"][npoi]["material"] in ["", "p", "P"]:
                        raise ValueError("This plant seem to be a pole")
                    else:
                        if "distance" in plant_info.keys():
                            final_distance += plant_info["distance"] / 111139
                        else:
                            final_distance += 0.6 / 111139
                    linereverse = LineString([line_coords[-1], next_poi])
                    secondtolast_poi = linereverse.interpolate(distance)
            if secondtolast_poi is not None:
                newline = LineString([next_poi, secondtolast_poi])
                newpolepoints = [newline.interpolate((i/usepoints), normalized=True) for i in range(1, usepoints)]
                finalpolepoints.extend(newpolepoints)
                finalpolepoints.append(secondtolast_poi)
            else:
                newline = LineString([next_poi, line_coords[-1]])
                newpolepoints = [newline.interpolate((i/usepoints), normalized=True) for i in range(1, usepoints)]
                finalpolepoints.extend(newpolepoints)
                finalpolepoints.append(Point(line_coords[-1]))

        # for each pole create the block line and interpolate all the plants of that block
        for poi in finalpolepoints[1:]:
            poicoords = list(poi.coords)[0]
            if segment_num == 0:
                polecoords = list(finalpolepoints[0].coords[0])
                thisline = LineString([polecoords, poicoords])
                # first was already added before
                if info["first"] == "plant":
                    final_points.append(poicoords)
                    plant_feat = QgsFeature()
                    plant_feat.setGeometry(
                        QgsGeometry.fromPointXY(
                            QgsPointXY(polecoords[0], polecoords[1])
                        )
                    )
                    plant_feat.setAttributes(
                        ["pole", None, None, None, None, self.farm_id, field, subfield, rowid]
                    )
                    if not plants_prov.addFeatures([plant_feat]):
                        print(f"{plant_feat} not added")
                    totalpoints += 1
                    #segment_num += 1

            print(segment_num, npoints)
            if segment_num == npoints:
                if info["last"] == "plant":
                    continue
            else:
                thisline = LineString([finalpolepoints[segment_num], poicoords])
            if info["first"] == "plant":
                thispoints = [thisline.interpolate((i/(plants_block[segment_num + 1] + 1)), normalized=True) for i in range(1, plants_block[segment_num +1] + 1)]
            else:
                thispoints = [thisline.interpolate((i/(plants_block[segment_num] + 1)), normalized=True) for i in range(1, plants_block[segment_num] + 1)]
            print(f"Number of points {thispoints}")

            #for tpoi in thispoints:
            jump_val = 0
            for tpoi_index in range(len(thispoints)):
                right_index = tpoi_index + jump_val
                if right_index < len(thispoints):
                    tpoi = thispoints[right_index]
                else:
                    continue
                tpoi_coord = tuple(tpoi.coords)[0]
                final_points.append(tpoi_coord)
                geom = QgsGeometry.fromPointXY(
                        QgsPointXY(tpoi_coord[0], tpoi_coord[1])
                    )
                if plantsnumb < len(info["points"]):
                    plant_info = info["points"][plantsnumb]
                else:
                    continue
                plant_info["type"] = "plant"
                plant_feat = createPlantPoleFeature(
                    plant_info,
                    geom,
                    self.farm_id,
                    field,
                    subfield,
                    rowid
                )

                if plant_feat.isValid():
                    print(plant_feat.isValid(), plants_prov.featureCount())
                    if not plants_prov.addFeature(plant_feat):
                        print(f"{plant_feat} not added")
                    totalpoints += 1
                    plantsnumb += 1
                else:
                    raise ValueError("Plant not valid")
                multiplant = False
                if "multiplant" in plant_info.keys():
                    if plant_info["multiplant"] and (right_index + 1) < len(thispoints):
                        plant_info = info["points"][plantsnumb]
                        if plant_info["material"] in ["f", "F"] or (plant_info["material"] not in ["", "p", "P"] and plant_info["multiplant"]):
                            jump_val += 1
                            next_poi = thispoints[right_index + 1]
                            next_coords = tuple(next_poi.coords)[0]
                            next_line = LineString([tpoi_coord, next_coords])
                            # interpolate converting meters in degree
                            final_next_poi = next_line.interpolate(setted_distance / 111139)
                            final_next_coords = tuple(final_next_poi.coords)[0]
                            next_geom = QgsGeometry.fromPointXY(
                                QgsPointXY(final_next_coords[0], final_next_coords[1])
                            )

                            plant_info["type"] = "plant"
                            plant_feat = createPlantPoleFeature(
                                plant_info,
                                next_geom,
                                self.farm_id,
                                field,
                                subfield,
                                rowid
                            )

                            if plant_feat.isValid():
                                print(plant_feat.isValid(), plants_prov.featureCount())
                                if not plants_prov.addFeature(plant_feat):
                                    print(f"{plant_feat} not added")
                                totalpoints += 1
                                plantsnumb += 1
                                multiplant == True

            print(plants_prov.featureCount())
            final_points.append(poicoords)
            plant_feat = QgsFeature()
            plant_feat.setGeometry(
                QgsGeometry.fromPointXY(
                    QgsPointXY(poicoords[0], poicoords[1])
                )
            )
            plant_feat.setAttributes(
                ["pole", None, None, None, None, self.farm_id, field, subfield, rowid]
            )
            if not plants_prov.addFeatures([plant_feat]):
                print(f"{plant_feat} not added")
            totalpoints += 1
            segment_num += 1
        # if there is a last block of plant after last pole
        if info["last"] == "plant":
            linereverse = LineString([line_coords[-1], finalpolepoints[-1]])
            plantpoints = [linereverse.interpolate((i/info["plants_block"][-1]), normalized=True) for i in range(1, info["plants_block"][-1])]
            for plantpoi in plantpoints:
                if multiplant:
                    plant_info =  info["points"][plantsnumb - 1]
                else:
                    try:
                        plant_info =  info["points"][plantsnumb]
                    except Exception as err:
                        self.iface.messageBar().pushMessage(
                            self.tr("Error"),
                            self.tr("Something is probably wrong in the data, please check double poles close each others, empty lines, etc"),
                            level=Qgis.Info,
                        )
                        raise ValueError(err)
                plant_info["type"] = "plant"
                plant_coords = list(plantpoi.coords)[0]
                geom = QgsGeometry.fromPointXY(
                    QgsPointXY(plant_coords[0], plant_coords[1])
                )
                plant_feat = createPlantPoleFeature(
                    plant_info,
                    geom,
                    self.farm_id,
                    field,
                    subfield,
                    rowid
                )
                if plant_feat.isValid():
                    print(plant_feat.isValid(), plants_prov.featureCount())
                    if not plants_prov.addFeatures([plant_feat]):
                        print(f"{plant_feat} not added")
                totalpoints += 1
                plantsnumb += 1
            # add last plant
            if plantsnumb < len(info["points"]):
                plant_info =  info["points"][plantsnumb]
                plant_info["type"] = "plant"
                geom = QgsGeometry.fromPointXY(
                    QgsPointXY(line_coords[-1][0], line_coords[-1][1])
                )
                plant_feat = createPlantPoleFeature(
                    plant_info,
                    geom,
                    self.farm_id,
                    field,
                    subfield,
                    rowid
                )
                if plant_feat.isValid():
                    print(plant_feat.isValid(), plants_prov.featureCount())
                    if not plants_prov.addFeatures([plant_feat]):
                        print(f"{plant_feat} not added")
                totalpoints += 1
                plantsnumb += 1
        plants.updateExtents()
        QgsProject.instance().addMapLayer(plants)
        # set style
        # TODO improve style getting info from material
        classes = {"plant": ("#00FF00", "Plant"),
                   "pole": ("#808080", "Pole")}
        categories = []
        # Iterate through the dictionary
        for classes, (color, label) in classes.items():
            # Automatically set symbols based on layer's geometry
            symbol = QgsSymbol.defaultSymbol(plants.geometryType())
            # Set colour
            symbol.setColor(QColor(color))
            # Set symbol with value = 0 to be transparent
            if classes == "0":
                symbol.setOpacity(0)
            # Set the renderer properties
            category = QgsRendererCategory(classes, symbol, label)
            categories.append(category)

        # Field name
        expression = "type"
        # Set the categorized renderer
        renderer = QgsCategorizedSymbolRenderer(expression, categories)
        plants.setRenderer(renderer)
        # Refresh layer
        plants.triggerRepaint()
        set_label(plants, "label", 8, "left")

        return plants

    def _check_material(self):
        """Check if checkbox is enable and activate label and checkbox
        """
        if self.settings_dlg.radioButtonMaterialSvg.isChecked():
            self.settings_dlg.labelColumnMaterial.setEnabled(True)
            self.settings_dlg.comboBoxColumnMaterial.setEnabled(True)
        else:
            self.settings_dlg.labelColumnMaterial.setEnabled(False)
            self.settings_dlg.comboBoxColumnMaterial.setEnabled(False)
        return True

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # settings to save server, user, password
        self.settings = QSettings()
        # initialize locale
        locale = self.settings.value("locale/userLocale")#[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            "i18n",
            f"DigiAgriApp_{locale}.qm"
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # settings useful variable
        self.server = None
        self.token = None
        self.farm = None
        self.farm_id = None
        self.tmp_farm_id = None
        self.farm_stats = None
        self.fields = None
        self.fields_id = {}
        self.subfields_id = {}
        self.rows_id = {}
        self.layers_list = {}
        self.actions = []
        self.menu = None
        self.data_import = None
        self.material = {}
        self.minmax_years = []
        self.year = None
        self.csv_layer = None
        self.output_dir = None
        self.svg_dir = None
        self.style_checkbox = None
        self.style_colname = None
        self.style_labelname = None
        self.style_icon_size = 5
        self.style_text_size = 5

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads

        # import
        self.import_dlg = DigiAgriAppImport()
        self.import_dlg.button_box.accepted.connect(self.import_data)
        self.import_dlg.button_box.rejected.connect(self.import_dlg.close)

        #about
        self.about_dlg = DigiAgriAppAbout()

        #settings
        self.settings_dlg = DigiAgriAppSettings()
        self.settings_dlg.tokenbutton.clicked.connect(self._get_token)
        self.settings_dlg.savebutton.clicked.connect(self._save_settings)
        self.settings_dlg.closebutton.clicked.connect(self.settings_dlg.close)
        self.settings_dlg.farms_comboBox.currentTextChanged.connect(self._update_materials_years)
        self.settings_dlg.server_lineEdit.editingFinished.connect(self._check_server_ending)
        self.settings_dlg.server_lineEdit.returnPressed.connect(self._check_server_ending)
        self.settings_dlg.buttonGroup.buttonToggled[QAbstractButton, bool].connect(self._check_material)
        self.settings_dlg.comboBoxColumnMaterial.addItems(["variety", "clone", "rootstock"])
        self.settings_dlg.selectOutputDir.clicked.connect(
            lambda: self._get_directory("settings_out_dif")
        )
        self.settings_dlg.comboBoxLabel.addItems(["full_name", "name"])
        project_path = get_project_path()
        if project_path:
            self.output_dir = project_path
            self.settings_dlg.lineEditOutput.setText(project_path)
            self.settings.setValue("/DigiAgriApp/output_directory", self.output_dir)
        self.settings_dlg.selectSvgDir.clicked.connect(
            lambda: self._get_directory("settings_svg_dir")
        )

        # export
        self.export_vector_dlg = DigiAgriAppExportVector()
        #self.export_vector_dlg.button_box.accepted.connect(self.export_vector)
        self.export_vector_dlg.button_box.rejected.connect(
            self.export_vector_dlg.close
        )
        self.export_vector_dlg.pushButton.clicked.connect(
            lambda: self._get_file("export")
        )

        # plants
        self.plants_dlg = DigiAgriAppPlantCreation()
        self.plants_dlg.button_box.accepted.connect(self.plant_creation)
        self.plants_dlg.button_box.rejected.connect(self.plants_dlg.close)
        self.plants_dlg.comboBoxField.currentIndexChanged.connect(self.populate_subfields)
        self.plants_dlg.comboBoxSubfield.currentIndexChanged.connect(self.populate_rows)
        self.plants_dlg.pushButton.clicked.connect(lambda: self._get_file("plants"))

        #style
        self.style_dlg = DigiAgriAppStyle()
        self.style_dlg.buttonBox.accepted.connect(self.set_layer_style)
        self.style_dlg.buttonBox.rejected.connect(self.style_dlg.close)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("DigiAgriApp", message)

    def build_menu(self):
        """Function to create the menu"""
        # import menu
        icon_import_path = self.plugin_dir + "/icons/icon_import.svg"
        self.import_menu = QMenu(self.tr("Import"), self.iface.mainWindow())
        self.import_menu.setIcon(QIcon(icon_import_path))
        ## vector
        icon_import_vector_path = self.plugin_dir + "/icons/icon_import_vector.svg"
        self.import_vector_act = QAction(
            QIcon(icon_import_vector_path),
            self.tr("Vector"),
            self.iface.mainWindow()
        )
        self.import_vector_act.triggered.connect(self.show_import_dialog)
        self.import_menu.addAction(self.import_vector_act)
        ## create plants
        icon_create_plant_path = self.plugin_dir + "/icons/icon_import_vector.svg"
        self.create_plant_act = QAction(
            QIcon(icon_create_plant_path),
            self.tr("Create plants from CSV file"),
            self.iface.mainWindow()
        )
        self.create_plant_act.triggered.connect(self.show_create_plant_dialog)
        self.import_menu.addAction(self.create_plant_act)
        ## raster
        icon_import_raster_path = self.plugin_dir + "/icons/icon_import_raster.svg"
        self.import_raster_act = QAction(
            QIcon(icon_import_raster_path),
            self.tr("Raster"),
            self.iface.mainWindow()
        )
        self.import_raster_act.setEnabled(False)
        #self.import_raster_act.triggered.connect(self.show_import_dialog)
        self.import_menu.addAction(self.import_raster_act)
        self.menu.addMenu(self.import_menu)

        # export menu
        icon_export_path = self.plugin_dir + "/icons/icon_export.svg"
        self.export_menu = QMenu(self.tr("Export"), self.iface.mainWindow())
        self.export_menu.setIcon(QIcon(icon_export_path))
        ## vector
        icon_export_vector_path = self.plugin_dir + "/icons/icon_export_vector.svg"
        self.export_vector_act = QAction(
            QIcon(icon_export_vector_path),
            self.tr("Vector"),
            self.iface.mainWindow()
        )
        self.export_vector_act.triggered.connect(self.show_export_vector_dialog)
        self.export_menu.addAction(self.export_vector_act)
        ## raster
        icon_export_raster_path = self.plugin_dir + "/icons/icon_export_raster.svg"
        self.export_raster_act = QAction(
            QIcon(icon_export_raster_path),
            self.tr("Raster"),
            self.iface.mainWindow()
        )
        self.export_raster_act.setEnabled(False)
        #self.export_raster_act.triggered.connect(self.show_export_raster_dialog)
        self.export_menu.addAction(self.export_raster_act)
        self.menu.addMenu(self.export_menu)

        # style
        icon_style_path = self.plugin_dir + "/icons/icon_about.svg"
        self.style_act = QAction(
            QIcon(icon_style_path),
            self.tr("Style"),
            self.iface.mainWindow()
        )
        self.actions.append(self.style_act)
        self.style_act.triggered.connect(self.show_style_dialog)
        self.menu.addAction(self.style_act)

        # settings
        icon_settings_path = self.plugin_dir + "/icons/icon_settings.svg"
        self.settings_act = QAction(
            QIcon(icon_settings_path),
            self.tr("Settings"),
            self.iface.mainWindow()
        )
        self.actions.append(self.settings_act)
        self.settings_act.triggered.connect(self.show_settings_dialog)
        self.menu.addAction(self.settings_act)

        # about
        icon_about_path = self.plugin_dir + "/icons/icon_about.svg"
        self.about_act = QAction(
            QIcon(icon_about_path),
            self.tr("About"),
            self.iface.mainWindow()
        )
        self.actions.append(self.about_act)
        self.about_act.triggered.connect(self.about_dlg.show)
        self.menu.addAction(self.about_act)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(None, action)
        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = self.plugin_dir + "/icons/icon.svg"
        self.menu = QMenu(self.tr("DigiAgriApp"))
        self.menu.setIcon(QIcon(icon_path))
        # self.add_action(
        #     icon_path,
        #     text=self.tr("DigiAgriApp"),
        #     callback=self.run,
        #     parent=self.iface.mainWindow(),
        # )

        # need workaround for WebMenu for QuickMapService
        _temp_act = QAction('temp', self.iface.mainWindow())
        self.iface.addPluginToWebMenu("_tmp", _temp_act)
        self.iface.webMenu().addMenu(self.menu)
        self.iface.removePluginWebMenu("_tmp", _temp_act)

        self.build_menu()
        self.iface.webMenu().addMenu(self.menu)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        if self.menu:
            self.iface.webMenu().removeAction(self.menu.menuAction())
            self.iface.addLayerMenu().removeAction(self.menu.menuAction())
        for action in self.actions:
            self.iface.webMenu().removeAction(action)
            self.iface.removeWebToolBarIcon(action)
            self.iface.removePluginWebMenu(self.tr("&DigiAgriApp"), action)
            self.iface.removeToolBarIcon(action)

    def show_settings_dialog(self):
        """Show settings dialog"""
        self._check_server()
        self._check_user()
        self._check_style()
        self.settings_dlg.show()

    def show_import_dialog(self):
        """Function to show import"""
        if not self._working_server():
            self.import_dlg.close()
            return False
        action = self.import_dlg.sender()
        self.import_dlg.tableWidget.clear()
        self.layers_list = {}
        col = 0
        row = 0
        actual_rows = self.import_dlg.tableWidget.rowCount()
        addtype = None
        fields_to_add = ["Layer name", "Upload"]
        if action.text() == self.tr("Vector"):
            addtype = "vector"
            fields_to_add.extend(["DigiAgriApp feature type", "Selected features"])
        elif action.text() == self.tr("Raster"):
            addtype = "raster"
            fields_to_add.extend(["Field", "Time", "Sensor"])
            self.import_dlg.checkBoxDown.hide()
            self.import_dlg.checkBoxDown.enabled = False
        descr = f"Upload {addtype} layers to you DigiAgriApp server."
        descr += " You can select which layer upload with its metadata."
        self.import_dlg.label_descr.setText(descr)
        self.import_dlg.label_farm.setText(self.tr(f"<b>Selected farm is {self.farm}</b>"))
        self.import_dlg.label_farm.setStyleSheet(''' font-size: 24px; ''')
        self.import_dlg.label_farm.setAlignment(Qt.AlignCenter)
        if self.import_dlg.tableWidget.columnCount() < len(fields_to_add):
            while self.import_dlg.tableWidget.columnCount() != len(fields_to_add):
                self.import_dlg.tableWidget.insertColumn(col)
                col += 1
        elif self.import_dlg.tableWidget.columnCount() > len(fields_to_add):
            col_count = self.import_dlg.tableWidget.columnCount()
            while self.import_dlg.tableWidget.columnCount() != len(fields_to_add):
                self.import_dlg.tableWidget.removeColumn(col_count + 1)
                col_count -= 1

        self.import_dlg.tableWidget.setHorizontalHeaderLabels(fields_to_add)

        layer_type_count = 0
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer) and addtype == "vector":
                self.layers_list[layer.name()] = layer
                layer_type_count += 1
            elif isinstance(layer, QgsRasterLayer)  and addtype == "raster":
                self.layers_list[layer.name()] = layer
                layer_type_count += 1
        for layer in self.layers_list.values():
            if actual_rows == 0:
                self.import_dlg.tableWidget.insertRow(row)
            elif actual_rows != 0 and row >= actual_rows:
                self.import_dlg.tableWidget.insertRow(row)
            upload_checkbox = QTableWidgetItem()
            upload_checkbox.setFlags(
                Qt.ItemIsUserCheckable | Qt.ItemIsEnabled
            )
            upload_checkbox.setCheckState(Qt.Unchecked)
            self.import_dlg.tableWidget.setItem(
                row, 0, QTableWidgetItem(layer.name())
            )
            self.import_dlg.tableWidget.setItem(row, 1, upload_checkbox)

            if isinstance(layer, QgsVectorLayer) and addtype == "vector":
                digiagriapp_type = QComboBox()
                digiagriapp_type.addItems(
                    [
                        self.tr("field"),
                        self.tr("subfield"),
                        self.tr("row"),
                        self.tr("plant"),
                        self.tr("pole")
                    ]
                )
                self.import_dlg.tableWidget.setCellWidget(
                    row, 2, digiagriapp_type
                )
                features_checkbox = QTableWidgetItem()
                features_checkbox.setFlags(
                    Qt.ItemIsUserCheckable | Qt.ItemIsEnabled
                )
                features_checkbox.setCheckState(Qt.Unchecked)
                self.import_dlg.tableWidget.setItem(
                    row, 3, QTableWidgetItem(features_checkbox)
                )
            elif isinstance(layer, QgsRasterLayer)  and addtype == "raster":
                fields_combo = QComboBox()
                for fid, finame in self.fields_id.items():
                    fields_combo.addItem(finame, fid)
                self.import_dlg.tableWidget.setCellWidget(row, 2, fields_combo)
                timeItem = QLineEdit()
                self.import_dlg.tableWidget.setCellWidget(row, 3, timeItem)
                sensor_combo = QComboBox()
                sensors = self._get_sensors()
                for sensor in sensors:
                    sensor_combo.addItem(sensor["name"], sensor["id"])
                self.import_dlg.tableWidget.setCellWidget(row, 4, sensor_combo)
            row += 1

        while self.import_dlg.tableWidget.rowCount() > len(self.layers_list):
            self.import_dlg.tableWidget.removeRow(
                self.import_dlg.tableWidget.rowCount()-1
            )
        self._refresh_table()
        self.import_dlg.exec()

    def import_data(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback,
        # so that it will only load when the plugin is started

        # Run the dialog event loop
        self._check_farm()
        result = self.import_dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            exportCheck = self.import_dlg.checkBoxDown.isChecked()
            for rownum in range(self.import_dlg.tableWidget.rowCount()):
                if self.import_dlg.tableWidget.item(rownum, 1).checkState() == 2:
                    layer_name = self.import_dlg.tableWidget.item(rownum, 0).text()
                    layer = self.layers_list[layer_name]
                    if isinstance(layer, QgsVectorLayer):
                        layer_digi_type = self.import_dlg.tableWidget.cellWidget(
                            rownum, 2
                        ).currentText()
                        if self.import_dlg.tableWidget.item(rownum, 3).checkState() == 2:
                            layer_allfeats = False
                        else:
                            layer_allfeats = True

                        import_url = FIELD_IMPORT.format(
                            server=self.server, typ=layer_digi_type, farmid=self.farm_id
                        )
                        self._import_vector(layer, layer_allfeats, import_url)
                        if exportCheck:
                            filepath = layer.dataProvider().uri().uri().split('|')[0].strip()
                            QgsProject.instance().removeMapLayer(layer)
                            if layer_digi_type == "field":
                                self._export_field(filepath)
                                layer = self._reload_layer(filepath, "field")
                                set_single_style(layer, "#FFFF00", iface=self.iface)
                                field_to_value_map(layer, 'farm', {self.farm: self.farm_id})
                            elif layer_digi_type == "subfield":
                                self._export_subfield(filepath)
                                layer = self._reload_layer(filepath, "subfield")
                                set_single_style(layer, "#00FF00", iface=self.iface)
                                if not self.fields_id:
                                    self._get_fields()
                                field_to_value_map(layer, 'field', self.fields_id)
                            elif layer_digi_type == "row":
                                self._export_row(filepath)
                                layer = self._reload_layer(filepath, "row")
                                set_single_style(layer, "#8b4513", iface=self.iface)
                                if not self.fields_id:
                                    self._get_fields()
                                if not self.subfields_id:
                                    self._get_subfields_all()
                                field_to_value_map(layer, 'field', self.fields_id)
                                field_to_value_map(layer, 'subfield', self.subfields_id)
                            elif layer_digi_type == "plant":
                                self._export_plant(filepath)
                                layer = self._reload_layer(filepath, "plant")
                                if not self.fields_id:
                                    self._get_fields()
                                if not self.subfields_id:
                                    self._get_subfields_all()
                                if not self.rows_id:
                                    self._get_rows_all()
                                field_to_value_map(layer, 'field', self.fields_id)
                                field_to_value_map(layer, 'subfield', self.subfields_id)
                                field_to_value_map(layer, 'row', self.rows_id)
                                if not self.csv_layer:
                                    print("in CSV layer")
                                    self._check_csv_layer()
                                join = join_layer_csv(layer, "active_material_id", self.csv_layer, "id")
                                if join:
                                    if self.settings_dlg.radioButtonMaterialSvg.isChecked():
                                        col = self.settings_dlg.comboBoxColumnMaterial.currentText()
                                        if not set_style_plant_material_svg(
                                            layer,
                                            self.csv_layer,
                                            col,
                                            self.settings_dlg.comboBoxLabel.currentText(),
                                            icon_size=self.style_icon_size,
                                            text_size=self.style_text_size,
                                            iface=self.iface
                                        ):
                                            self.iface.messageBar().pushMessage(
                                                self.tr("Info"),
                                                self.tr("Using single style since the selected returned a problem"),
                                                level=Qgis.Info,
                                            )
                                            set_single_style(
                                                layer,
                                                "#006400",
                                                self.settings_dlg.comboBoxLabel.currentText(),
                                                icon_size=self.style_icon_size,
                                                text_size=self.style_text_size,
                                                iface=self.iface
                                            )
                                    elif self.settings_dlg.radioButtonMaterialIcon.isChecked():
                                        col = self.settings_dlg.comboBoxColumnMaterial.currentText()
                                        if not set_style_plant_material_simple(
                                            layer,
                                            col,
                                            self.settings_dlg.comboBoxLabel.currentText(),
                                            icon_size=self.style_icon_size,
                                            text_size=self.style_text_size
                                        ):
                                            self.iface.messageBar().pushMessage(
                                                self.tr("Info"),
                                                self.tr("Using single style since the selected returned a problem"),
                                                level=Qgis.Info,
                                            )
                                            set_single_style(
                                                layer,
                                                "#006400",
                                                self.settings_dlg.comboBoxLabel.currentText(),
                                                icon_size=self.style_icon_size,
                                                text_size=self.style_text_size,
                                                iface=self.iface
                                            )
                                    else:
                                        set_single_style(
                                            layer,
                                            "#006400",
                                            self.settings_dlg.comboBoxLabel.currentText(),
                                            icon_size=self.style_icon_size,
                                            text_size=self.style_text_size,
                                            iface=self.iface
                                        )

                            elif layer_digi_type == "pole":
                                self._export_pole(filepath)
                                layer = self._reload_layer(filepath, "pole")
                                set_single_style(layer, "#808080", iface=self.iface)
                                field_to_value_map(layer, 'field', self.fields_id)
                                field_to_value_map(layer, 'subfield', self.subfields_id)
                                field_to_value_map(layer, 'row', self.rows_id)
                    elif isinstance(layer, QgsRasterLayer):
                        self.iface.messageBar().pushMessage(
                                self.tr("Warning"),
                                self.tr("Raster not yet implemented"),
                                level=Qgis.Warning,
                            )
            self.import_dlg.close()
        else:
            self.import_dlg.close()
        return True

    def _write_vector(self, layer, file_output):
        """ Write vector layer to file

        :param layer: Vector layer object.
        :type layer: obj, QgsVectorLayer

        :returns: Return True or False if vector is correctly written.
        :rtype: Boolean
        """
        save_options = QgsVectorFileWriter.SaveVectorOptions()
        if os.path.isfile(file_output):
            save_options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer
        save_options.layerName = layer.name()
        save_options.fileEncoding = layer.dataProvider().encoding()
        ##save_options.driverName = "GeoJSON"
        transform_context = QgsProject.instance().transformContext()
        error = QgsVectorFileWriter.writeAsVectorFormatV3(
            layer,
            file_output,
            transform_context,
            save_options,
        )
        if error[0] == QgsVectorFileWriter.NoError:
            self.iface.messageBar().pushMessage(
                self.tr("Success"),
                self.tr(f"Layer '{layer.name()}' correctly written to {file_output}"),
                level=Qgis.Success,
            )
            return True
        else:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(f"Layer '{layer.name()}' not written correctly: {error}"),
                level=Qgis.Critical,
            )
            return False

    def show_create_plant_dialog(self):
        """Function to show the create plants dialog"""
        if not self._working_server():
            self.plants_dlg.close()
            return False
        self.plants_dlg.comboBoxField.clear()
        self._get_fields()
        for fieldid, fieldname in self.fields_id.items():
            self.plants_dlg.comboBoxField.addItem(fieldname, fieldid)
        self.plants_dlg.label_farm.setText(self.tr(f"<b>Selected farm is {self.farm}</b>"))
        self.plants_dlg.label_farm.setStyleSheet(''' font-size: 24px; ''')
        self.plants_dlg.label_farm.setAlignment(Qt.AlignCenter)
        self.plants_dlg.exec()

    def plant_creation(self):
        """Function to create plants"""
        file_in = self.plants_dlg.lineEditOutput.text()
        check_material = self.plants_dlg.material_checkBox.isChecked()
        check_poles = self.plants_dlg.poles_checkBox.isChecked()
        distance = self.plants_dlg.doublePlantSpinBox.value()
        if not file_in:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr("Please set the input CSV file path."),
                level=Qgis.Critical,
            )
            return False
        if os.path.exists(file_in) is False:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr(f"The input CSV file path {file_in} does not exist."),
                level=Qgis.Critical,
            )
            return False
        fieldid = self.plants_dlg.comboBoxField.currentData()
        if not fieldid:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr("Please set the field ID selecting a field"),
                level=Qgis.Critical,
            )
            return False
        subfieldid = self.plants_dlg.comboBoxSubfield.currentData()
        if not subfieldid:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr("Please set the subfield ID selecting a subfield"),
                level=Qgis.Critical,
            )
            return False
        with open(file_in, "r") as csvfile:
            readlines = csvfile.readlines()

        if readlines[0].count("|") > 0:
            sep = "|"
        elif readlines[0].count(";") > 0:
            sep = ";"
        elif readlines[0].count(",") > 0:
            sep = ","
        else:
            self.iface.messageBar().pushMessage(
                self.tr("Error"),
                self.tr("CSV separator not recognized, please use one between '|', ';', ','"),
                level=Qgis.Critical,
            )
            return False
        headers = readlines[0].strip().split(sep)
        material_index = headers.index("material")
        row_index = headers.index("row")
        label_index = headers.index("label")
        thesis_index = headers.index("thesis")
        experiment_index = headers.index("experiment")
        self.data_import = {}
        old_row = None
        pole_number = 0
        plant_number = 0
        for line in readlines[1:]:
            infos = line.strip().split(sep)
            row_name = infos[row_index]
            if row_name not in self.data_import.keys():
                if old_row:
                    if plant_number > 0:
                        self.data_import[old_row]["plants_block"].append(plant_number)
                        self.data_import[old_row]["last"] = "plant"
                    else:
                        self.data_import[old_row]["last"] = "pole"
                    if "pole_number" not in self.data_import[old_row].keys() and pole_number > 0:
                        self.data_import[old_row]["pole_number"] = pole_number
                self.data_import[row_name] = {}
                self.data_import[row_name]["points"] = []
                self.data_import[row_name]["plants_block"] = []
                if infos[material_index] in ["", "p", "P"]:
                    self.data_import[row_name]["first"] = "pole"
                else:
                    self.data_import[row_name]["first"] = "plant"
                    self.data_import[row_name]["last"] = "pole"
                plant_number = 0
                pole_number = 0
                old_row = row_name
            if infos[material_index] in ["", "p", "P"]:
                print(f"plant number {plant_number}")
                if plant_number > 0:
                    self.data_import[row_name]["plants_block"].append(plant_number)
                pole_number += 1
                plant_number = 0
            else:
                plant_number += 1
                poi = {
                    "material": infos[material_index],
                    "label": infos[label_index],
                    "thesis": infos[thesis_index],
                    "experiment": infos[experiment_index]
                }
                if check_material:

                    if infos[material_index] in ["F", "f"]:
                        # pole is in the half of distance between two plants
                        try:
                            poi["spacing"] = self.data_import[row_name]["points"][-1]["spacing"] / 2
                        except:
                            poi["spacing"] = None
                        poi["multiplant"] = False
                    else:
                        mat = self.material.get(int(infos[material_index]))
                        try:
                            poi["spacing"] = mat["spacing_dict"]["plant"]
                            if mat["spacing_dict"]["multiplant"] in (True, 't'):
                                poi["multiplant"] = True
                            else:
                                poi["multiplant"] = False
                        except:
                            poi["spacing"] = None
                            poi["multiplant"] = False
                self.data_import[row_name]["points"].append(poi)
        if plant_number > 0:
            self.data_import[row_name]["plants_block"].append(plant_number)
            self.data_import[row_name]["last"] = "plant"
        else:
            self.data_import[row_name]["last"] = "pole"
        if "pole_number" not in self.data_import[old_row].keys():
            self.data_import[old_row]["pole_number"] = pole_number
        try:
            rowid = self.plants_dlg.comboBoxRow.currentData()
        except Exception:
            rowid = None
        if rowid:
            if subfieldid:
                url = SUBFIELD_ROW_URL.format(
                    server=self.server,
                    farmid=self.farm_id,
                    fieldid=fieldid,
                    subfieldid=subfieldid
                ) + f"{rowid}/"
            else:
                url = FIELD_ROW_URL.format(
                    server=self.server,
                    farmid=self.farm_id,
                    fieldid=fieldid
                ) + f"{rowid}/"
        elif subfieldid:
            url = SUBFIELD_URL.format(
                server=self.server,
                farmid=self.farm_id,
                fieldid=fieldid
            ) + f"{subfieldid}/rows/"
        with requests.Session() as ses:
            req = ses.get(
                url,
                headers={"Authorization": f"Token {self.token}"}
            )
            if not req.ok:
                reason = self._request_error(req.json())
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr(
                        f"Problem to get subfield for farm with ID {self.farm_id} with subfield with ID {fieldid}; error: {reason}"
                    ),
                    level=Qgis.Warning,
                )
                return False
        rows = req.json()
        data_keys = self.data_import.keys()
        if isinstance(rows, list):
            plant_layers = []
            for row in rows:
                geom = LineString(row["geom"]["coordinates"])
                if row["name"] in data_keys:
                    plant_layers.append(self._create_points(
                        geom,
                        self.data_import[row["name"]],
                        row["name"],
                        fieldid,
                        subfieldid,
                        row["id"],
                        check_poles,
                        distance
                    ))
                else:
                    self.iface.messageBar().pushMessage(
                        self.tr("Warning"),
                        self.tr(
                            f"Row {row['name']} not found in the CSV file skipping"
                        ),
                        level=Qgis.Warning,
                    )
                    continue
                # if len(outcoords) != len(self.data_import[row["name"]]["points"]):
                #     self.iface.messageBar().pushMessage(
                #         self.tr("Warning"),
                #         self.tr(
                #             "Number of plants from CSV is different from coordinates calculated"
                #         ),
                #     )
                # for x in range(len(outcoords)):
                #     self.data_import[row["name"]]["points"][x]["geom"] = outcoords[x]
        elif isinstance(rows, dict):
            geom = LineString(rows["geom"]["coordinates"])
            if rows["name"] in data_keys:
                plant_layers = [self._create_points(
                    geom,
                    self.data_import[rows["name"]],
                    rows["name"],
                    fieldid,
                    subfieldid,
                    rows["id"],
                    check_poles,
                    distance
                )]
            else:
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr(
                        f"Row {rows['name']} not found in the CSV file skipping"
                    ),
                    level=Qgis.Warning,
                )
            # if len(outcoords) != len(self.data_import[rows["name"]]["points"]):
            #     self.iface.messageBar().pushMessage(
            #         self.tr("Warning"),
            #         self.tr(
            #             "Number of plants from CSV is different from coordinates calculated"
            #         ),
            #     )
            # for x in range(len(outcoords)):
            #     self.data_import[rows["name"]]["points"][x]["geom"] = outcoords[x]
        canvas = self.iface.mapCanvas()
        extent = None
        for layer in plant_layers:
            if extent is None:
                extent = layer.extent()
            else:
                extent.combineExtentWith(layer.extent())
        canvas.setExtent(extent)
        canvas.refresh()
        qd = QuestionDialog(parent=self.iface.mainWindow())
        if qd.exec():
            print(plant_layers)
            plant_url = PLANTS_CREATE.format(server=self.server)
            for layer in plant_layers:
                self._import_vector(layer, True, plant_url)
        else:
            print("Cancel!")
        return True

    def populate_subfields(self):
        self.plants_dlg.comboBoxSubfield.clear()
        self.plants_dlg.comboBoxSubfield.addItem("---", None)
        fieldid = self.plants_dlg.comboBoxField.currentData()

        subfields = self._get_subfields(fieldid)
        if not subfields:
            return False
        for subfield in subfields:
            self.plants_dlg.comboBoxSubfield.addItem(subfield["name"], subfield["id"])
        return True

    def populate_rows(self):
        self.plants_dlg.comboBoxRow.clear()
        self.plants_dlg.comboBoxRow.addItem("---", None)
        fieldid = self.plants_dlg.comboBoxField.currentData()
        subfieldid = self.plants_dlg.comboBoxSubfield.currentData()
        rows = self._get_rows(fieldid, subfieldid)
        if not rows:
            return False
        for row in rows:
            self.plants_dlg.comboBoxRow.addItem(row["name"], row["id"])
        return True

    def show_export_vector_dialog(self):
        """Function to show export vector data dialog"""
        # See if OK was pressed
        if not self._working_server():
            self.export_vector_dlg.close()
            return False
        self.export_vector_dlg.label_descr.setWordWrap(True)
        self.export_vector_dlg.label_farm.setText(self.tr(f"<b>Selected farm is {self.farm} for year {self.year}</b>"))
        self.export_vector_dlg.label_farm.setStyleSheet(''' font-size: 24px; ''')
        self.export_vector_dlg.label_farm.setAlignment(Qt.AlignCenter)
        result = self.export_vector_dlg.exec()
        if result:
            if not self._check_farm():
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr("Please set the farm in the settings window."),
                    level=Qgis.Critical,
                )
                return False
            file_output = self.export_vector_dlg.lineEditOutput.text()
            if not file_output:
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr("Please set the output file path."),
                    level=Qgis.Critical,
                )
                return False
            token_exist = self._check_token()
            if not token_exist:
                self.iface.messageBar().pushMessage(
                    self.tr("Error"),
                    self.tr("Impossible connect to the server, please check the settings."),
                    level=Qgis.Critical,
                )
                return False
            print(self.farm)
            fields_check = self.export_vector_dlg.fields_checkBox.isChecked()
            subfields_check = self.export_vector_dlg.subfields_checkBox.isChecked()
            rows_check = self.export_vector_dlg.rows_checkBox.isChecked()
            plants_check = self.export_vector_dlg.plants_checkBox.isChecked()
            poles_check = self.export_vector_dlg.poles_checkBox.isChecked()
            if fields_check is True or subfields_check is True or rows_check is True or plants_check is True or poles_check is True:
                print("in self._get_fields()")
                self._get_fields()
            else:
                return True
            if len(self.fields) < 1:
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr("There are no fields in the server"),
                    level=Qgis.Warning,
                )
                return False
            if not self.farm_stats:
                self.farm_stats = self._get_farm_stats()
            if fields_check is True or subfields_check is True or rows_check is True or plants_check is True:
                print("in self._get_material()")
                if self.csv_layer is None:
                    self._check_csv_layer()
                if self.csv_layer is None:
                    self._get_material(csv=True)

            if fields_check:
                self._export_field(file_output, self.year)
                layer = self._reload_layer(file_output, "field")
                if not layer:
                    try:
                        layer = QgsProject.instance().mapLayersByName(
                            f"{self.farm.replace(' ', '_')}_digiagriapp_field"
                        )[0]
                    except IndexError:
                        pass
                if layer:
                    set_single_style(layer, "#FFFF00", iface=self.iface)
                    if not self.farm_id:
                        self._get_fields()
                    field_to_value_map(layer, 'farm', {self.farm: self.farm_id})
            if subfields_check:
                self._export_subfield(file_output, self.year)
                self._reload_layer(file_output, "subfield")
                try:
                    layer = QgsProject.instance().mapLayersByName(f"{self.farm.replace(' ', '_')}_digiagriapp_subfield")[0]
                except IndexError:
                    pass
                if layer:
                    set_single_style(layer, "#00FF00", iface=self.iface)
                    if not self.fields_id:
                        self._get_fields()
                    if not self.subfields_id:
                        self._get_subfields_all()
                    field_to_value_map(layer, 'field', self.fields_id)
                    field_to_value_map(layer, 'subfield', self.subfields_id)
            if rows_check:
                self._export_row(file_output)
                layer = self._reload_layer(file_output, "row")
                if not layer:
                    try:
                        layer = QgsProject.instance().mapLayersByName(f"{self.farm.replace(' ', '_')}_digiagriapp_row")[0]
                    except IndexError:
                        pass
                if layer:
                    set_single_style(layer, "#8b4513", iface=self.iface)
                    if not self.fields_id:
                        self._get_fields()
                    if not self.subfields_id:
                        self._get_subfields_all()
                    field_to_value_map(layer, 'field', self.fields_id)
                    field_to_value_map(layer, 'subfield', self.subfields_id)
            if plants_check:
                self._export_plant(file_output)
                layer = self._reload_layer(file_output, "plant")
                if not layer:
                    try:
                        layer = QgsProject.instance().mapLayersByName(f"{self.farm.replace(' ', '_')}_digiagriapp_plant")[0]
                    except IndexError:
                        pass
                if layer:
                    if not self.fields_id:
                        self._get_fields()
                    if not self.subfields_id:
                        self._get_subfields_all()
                    if not self.rows_id:
                        self._get_rows_all()
                    field_to_value_map(layer, 'field', self.fields_id)
                    field_to_value_map(layer, 'subfield', self.subfields_id)
                    field_to_value_map(layer, 'row', self.rows_id)
                    join = join_layer_csv(layer, "active_material_id", self.csv_layer, "id")
                    if join:
                        if self.settings_dlg.radioButtonMaterialSvg.isChecked():
                            col = self.settings_dlg.comboBoxColumnMaterial.currentText()
                            if not set_style_plant_material_svg(
                                layer,
                                self.csv_layer,
                                col,
                                self.settings_dlg.comboBoxLabel.currentText(),
                                iface=self.iface
                            ):
                                self.iface.messageBar().pushMessage(
                                    self.tr("Info"),
                                    self.tr("Using single style since the selected returned a problem"),
                                    level=Qgis.Info,
                                )
                                set_single_style(
                                    layer,
                                    "#006400",
                                    self.settings_dlg.comboBoxLabel.currentText(),
                                    iface=self.iface
                                )
                        else:
                            set_single_style(
                                layer,
                                "#006400",
                                self.settings_dlg.comboBoxLabel.currentText(),
                                iface=self.iface
                            )
            if poles_check:
                self._export_pole(file_output)
                layer = self._reload_layer(file_output, "pole")
                if not layer:
                    try:
                        layer = QgsProject.instance().mapLayersByName(f"{self.farm.replace(' ', '_')}_digiagriapp_pole")[0]
                    except IndexError:
                        pass
                if layer:
                    set_single_style(layer, "#808080", icon_size=4, iface=self.iface)
                    field_to_value_map(layer, 'field', self.fields_id)
                    field_to_value_map(layer, 'subfield', self.subfields_id)
                    field_to_value_map(layer, 'row', self.rows_id)
            return True
        else:
            self.import_dlg.close()
        return True

    # def show_export_raster_dialog(self):
    #     """Function to export vector data"""

    def show_style_dialog(self):
        if self.settings_dlg.radioButtonMaterialIcon.isChecked() or self.settings_dlg.radioButtonMaterialSvg.isChecked():
            self.style_dlg.labelSingleColor.setEnabled(False)
            self.style_dlg.mColorButton.setEnabled(False)
        else:
            self.style_dlg.labelSingleColor.setEnabled(True)
            self.style_dlg.mColorButton.setEnabled(True)
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer):
                self.style_dlg.comboBox.addItem(layer.name(), layer.id())
        self.style_dlg.exec()

    def set_layer_style(self):
        result = self.style_dlg.exec_()
        # See if OK was pressed
        if result:
            layer_id = self.style_dlg.comboBox.currentData()
            layer = QgsProject.instance().mapLayer(layer_id)
            if self.settings_dlg.radioButtonMaterialSvg.isChecked():
                col = self.settings_dlg.comboBoxColumnMaterial.currentText()
                if self.csv_layer is None:
                    print("in CSV layer")
                    self._check_csv_layer()
                if len(layer.vectorJoins()) >= 1:
                    if len(layer.vectorJoins()) > 1:
                        self.iface.messageBar().pushMessage(
                            self.tr("Warning"),
                            self.tr("To many tables connected with vector, choosing first working"),
                            level=Qgis.Warning,
                        )
                    for vjoin in layer.vectorJoins():
                        if vjoin.joinLayer():
                            self.csv_layer = vjoin.joinLayer()
                if not set_style_plant_material_svg(
                    layer,
                    self.csv_layer,
                    col,
                    self.settings_dlg.comboBoxLabel.currentText(),
                    self.style_icon_size,
                    self.style_text_size,
                    self.iface
                ):
                    self.iface.messageBar().pushMessage(
                        self.tr("Info"),
                        self.tr("Using single style since the selected returned a problem"),
                        level=Qgis.Info,
                    )
                    color = self.style_dlg.mColorButton.color()
                    set_single_style(
                        layer,
                        color.name(),
                        self.settings_dlg.comboBoxLabel.currentText(),
                        self.style_icon_size,
                        self.style_text_size,
                        iface=self.iface
                    )
            elif self.settings_dlg.radioButtonMaterialIcon.isChecked():
                #col = self.settings_dlg.comboBoxColumnMaterial.currentText()
                col = "active_materials"
                if not set_style_plant_material_simple(
                    layer,
                    col,
                    self.settings_dlg.comboBoxLabel.currentText(),
                    icon_size=self.style_icon_size,
                    text_size=self.style_text_size
                ):
                    self.iface.messageBar().pushMessage(
                        self.tr("Info"),
                        self.tr("Using single style since the selected returned a problem"),
                        level=Qgis.Info,
                    )
                    color = self.style_dlg.mColorButton.color()
                    set_single_style(
                        layer,
                        color.name(),
                        self.settings_dlg.comboBoxLabel.currentText(),
                        self.style_icon_size,
                        self.style_text_size,
                        iface=self.iface
                    )
            else:
                color = self.style_dlg.mColorButton.color()
                set_single_style(
                    layer,
                    color.name(),
                    self.settings_dlg.comboBoxLabel.currentText(),
                    self.style_icon_size,
                    self.style_text_size,
                    iface=self.iface
                )
        return True
