# from pyproj import CRS
# from pyproj.aoi import AreaOfInterest
# from pyproj.database import query_utm_crs_info
import os
import csv
import random

from qgis.PyQt.QtWidgets import QDialog
from qgis.PyQt.QtWidgets import QDialogButtonBox
from qgis.PyQt.QtWidgets import QLabel
from qgis.PyQt.QtWidgets import QVBoxLayout
from qgis.PyQt.QtWidgets import QProgressDialog
from qgis.PyQt.QtGui import QColor

from qgis.core import QgsFeature
from qgis.core import QgsEditorWidgetSetup
from qgis.core import QgsVectorLayerJoinInfo
from qgis.core import QgsProject
from qgis.core import QgsSymbol
from qgis.core import QgsRendererCategory
from qgis.core import QgsCategorizedSymbolRenderer
from qgis.core import QgsSingleSymbolRenderer
from qgis.core import QgsLineSymbol
from qgis.core import QgsMarkerSymbol
from qgis.core import QgsPalLayerSettings
from qgis.core import QgsTextFormat
from qgis.core import QgsVectorLayerSimpleLabeling
from qgis.core import QgsTextBufferSettings
from qgis.core import QgsSvgMarkerSymbolLayer

THIS_PATH = os.path.dirname(__file__)

class QuestionDialog(QDialog):
    """Dialog with question with button box"""
    def __init__(self, parent=None):
        super().__init__(parent)

        self.setWindowTitle("Upload data!")

        QBtn = QDialogButtonBox.Ok | QDialogButtonBox.Cancel

        self.buttonBox = QDialogButtonBox(QBtn)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)

        self.layout = QVBoxLayout()
        message = QLabel("Do you want to upload the last create plants?")
        self.layout.addWidget(message)
        self.layout.addWidget(self.buttonBox)
        self.setLayout(self.layout)

    # def accept(self):
    #     """Upload create data
    #     """


    # def reject(self):
    #     """Not upload data"""


class ProgressBarDialog(QProgressDialog):
    """Dialog with progress bar"""
    def __init__(self, parent=None, label=None, title=None):
        super().__init__(parent)
        if title:
            self.setWindowTitle(title)
        else:
            self.setWindowTitle("DigiAgriApp: operation progress")
        if label:
            self.setLabelText(label)
        self.setMinimumWidth(300)
        self.setMinimumDuration(0)
        self.setMinimum(0)
        self.setMaximum(100)
        self.setValue(0)
        self.setAutoClose(True)
        self.setCancelButton(None)
        self.setModal(True)
        self.show()


def get_utm_srid_from_extent(extent):
    """Return the right UTM code from Django geometry envelope

    Args:
        extent (obj): list or Envelope geometry in longitude and latitude (EPSG 4326)

    Returns:
        CRS object
    """
    lon = (extent[0] + extent[2]) / 2
    lat = (extent[1] + extent[3]) / 2
    EPSG_srid = int(32700-round((45+lat)/90,0)*100 + round((183+lon)/6,0))
    return EPSG_srid

# def utm_from_extent(extent):
#     """Return the right UTM code from Django geometry envelope

#     Args:
#         extent (obj): list or Envelope geometry in longitude and latitude (EPSG 4326)

#     Returns:
#         CRS object
#     """
#     utm_crs_list = query_utm_crs_info(
#         datum_name="WGS 84",
#         area_of_interest=AreaOfInterest(
#             west_lon_degree=extent[0],
#             south_lat_degree=extent[1],
#             east_lon_degree=extent[2],
#             north_lat_degree=extent[3],
#         ),
#     )
#     return CRS.from_epsg(utm_crs_list[0].code)


def createPlantPoleFeature(in_dict, geom, farm, field, subfield, row):
    """Create a QGIS feature from a dictionary with keys

    Args:
        in_dict (dict): A dictionary with the following fields
        geom (obj): a QGIS Point geometry
        farm (int): The farm ID
        field (int): The field ID
        subfield (int): The subfield ID
        row (int): The row ID

    Returns:
        obj: A QGIS feature object
    """
    if not in_dict["label"]:
        label = None
    else:
        label = str(in_dict["label"])
    if not in_dict["theory"]:
        theory = None
    else:
        theory = in_dict["theory"]
    if not in_dict["evidence"]:
        evidence = None
    else:
        evidence = in_dict["evidence"]
    # check if is a foro F/f or empty E/e
    if in_dict["material"] in ["F", "f", "E", "e"]:
        material = None
    # check if is a pole P/p
    elif in_dict["material"] in ["", "p", "P"]:
        raise ValueError("The first plant seem to be a pole")
    else:
        material = in_dict["material"]
    plant_feat = QgsFeature()
    plant_feat.setGeometry(geom)
    plant_feat.setAttributes([
        "plant",
        material,
        label,
        theory,
        evidence,
        farm,
        field,
        subfield,
        row
    ])
    return plant_feat


def field_to_value_map(layer, field, inputlayer):
    """Function to set editing form to map values

    Args:
        layer (obj): The QGIS layer object to be used to set the map value
        field (str): The field to be set
        inputlayer (obj): The QGIS layer object to get values from

    Returns:
        boolean: return True if everything works well
    """
    list_values = {}
    for idd, name in inputlayer.items():
        list_values[name] = idd
    config = {'map' : list_values}
    widget_setup = QgsEditorWidgetSetup('ValueMap',config)
    field_idx = layer.fields().indexFromName(field)
    layer.setEditorWidgetSetup(field_idx, widget_setup)
    return True


def join_layer_csv(vect_layer, vect_col, csv_layer, csv_col):
    """Function to join a csv layer with a vector

    Args:
        vect_layer (obj): the vector layer to join
        vect_col (str): the column of vector to make the join
        csv_layer (obj): the csv layer to join with vector layer
        csv_col (str): the column of CSV to make the join

    Returns:
        bool: if everything is work properly
    """
    print("before join")
    joinObject = QgsVectorLayerJoinInfo()
    joinObject.setJoinLayer(csv_layer)
    # csv column is id for material id
    joinObject.setJoinFieldName(csv_col)
    joinObject.setTargetFieldName(vect_col)
    joinObject.setUsingMemoryCache(True)
    vect_layer.addJoin(joinObject)
    print("after join")
    return True

def get_project_path():
    """Function to return the directory where project is saved

    Returns:
        str: the path of the project's directory
    """
    path = QgsProject.instance().absolutePath()
    if path:
        return path
    return None

def set_label(layer, field_name):
    label_settings = QgsPalLayerSettings()
    #label_settings.drawBackground = True
    label_settings.fieldName = field_name
    text_format = QgsTextFormat()
    text_buffer = QgsTextBufferSettings()
    text_buffer.setColor(QColor("#FFFFFF"))
    text_buffer.setSize(1)
    text_buffer.setEnabled(True)
    text_format.setBuffer(text_buffer)
    label_settings.setFormat(text_format)
    layer.setLabeling(QgsVectorLayerSimpleLabeling(label_settings))
    layer.setLabelsEnabled(True)

def set_style_plant_material(layer, csv_layer, colname, field_name="name"):
    if csv_layer is None:
        raise ValueError("CSV layer is empty but it is required by this function")
    full_path = csv_layer.dataProvider().dataSourceUri().split("?")[0].replace("file://","")
    csvfile = open(full_path, newline='')
    csvdata = csv.reader(csvfile, delimiter=',')
    try:
        headers = next(csvdata)
    except:
        raise ValueError("Problem reading the CSV file, has it some rows?")
    col_index = headers.index(colname)
    icon_index = headers.index("icon")
    classes = {}
    for row in csvdata:
        var = row[col_index]
        svg = row[icon_index]
        if var not in classes.keys():
            classes[f"{var}_{svg}"] = (
                "#"+''.join([random.choice('0123456789ABCDEF') for j in range(6)]),
                var,
                svg
            )
    csvfile.close()
    # Create list to store symbology properties
    categories = []
    # Iterate through the dictionary
    for classes, (color, variety, svg) in classes.items():
        # Automatically set symbols based on layer's geometry
        symbol = QgsSymbol.defaultSymbol(layer.geometryType())
        # Set colour
        symbol.setColor(QColor(color))
        # Set symbol with value = 0 to be transparent
        symbol.setOpacity(0.6)
        symbol.setSize(5)
        # Set the renderer properties

        svgStyle = {}
        svgStyle['fill'] = '#000000'
        svgStyle['name'] = os.path.join(THIS_PATH, "plant_icons", svg)
        svgStyle['size'] = 5

        symbol_layer = QgsSvgMarkerSymbolLayer.create(svgStyle)
        symbol.appendSymbolLayer(symbol_layer.clone())
        category = QgsRendererCategory(variety, symbol, classes.replace("_", " "))
        categories.append(category)

    # Field name
    expression = f"{csv_layer.name()}_{colname}"
    # Set the categorized renderer
    renderer = QgsCategorizedSymbolRenderer(expression, categories)
    layer.setRenderer(renderer)

    if "_row" not in layer.name() and "_pole" not in layer.name():
        set_label(layer, field_name)
    # Refresh layer

    layer.triggerRepaint()
    return True

def set_single_style(layer, color, fieldName="name"):
    # Set the categorized renderer
    symbol = QgsSymbol.defaultSymbol(layer.geometryType())
    symbol.setColor(QColor(color))
    if isinstance(symbol, QgsLineSymbol):
        symbol.setWidth(1)
    elif isinstance(symbol, QgsMarkerSymbol):
        symbol.setSize(2)
    elif isinstance(symbol, QgsMarkerSymbol):
        symbol.setOpacity(0.6)
    renderer = QgsSingleSymbolRenderer(symbol)
    layer.setRenderer(renderer)
    if "_row" not in layer.name() and "_pole" not in layer.name():
        set_label(layer, fieldName)
    # Refresh layer
    layer.triggerRepaint()
    return True