# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DemoReports
                                 A QGIS plugin
 Generate demographic reports and datafills from block group data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2026-01-20
        git sha              : $Format:%H$
        copyright            : (C) 2026 by Retail Gravity
        email                : brad@retailgravity.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import QMessageBox, QFileDialog
from qgis.core import QgsProject, QgsVectorLayer, QgsWkbTypes
from qgis.PyQt.QtWidgets import QTextBrowser, QDialog, QVBoxLayout

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .demo_reports_dialog import DemoReportsDialog

# Other imports
from .point_tool import PointTool
from .spatial_processor import SpatialProcessor, validate_layer_for_analysis
from .package_config import get_package_variables
from .pdf_generator import DemographicPDFGenerator
from .formatting_utils import format_value_for_csv
import os.path

class DemoReports:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.
    
        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DemographicReporter_{}.qm'.format(locale))
    
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)
    
        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&US Demographic Reporter')
    
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('DemoReports', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        
        import os
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.png')
        
        self.add_action(
            icon_path,
            text=self.tr(u'US Demographic Reports'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&US Demographic Reports'),
                action)
            self.iface.removeToolBarIcon(action)
        
    def populate_layer_combos(self):
        """
        Populate combo boxes with available polygon layers from the project.
        This needs to be called each time the dialog opens to get current layers.
        """
        # Clear existing items
        self.dlg.blockGroupLayerComboBox.clear()
        self.dlg.blockGroupLayerComboBoxDatafill.clear()
        self.dlg.targetLayerComboBox.clear()
        
        # Get all layers from the project
        layers = QgsProject.instance().mapLayers().values()
        
        # Filter for polygon layers only
        polygon_layers = []
        for layer in layers:
            if isinstance(layer, QgsVectorLayer):
                if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                    polygon_layers.append(layer)
        
        # Add polygon layers to combo boxes
        for layer in polygon_layers:
            self.dlg.blockGroupLayerComboBox.addItem(layer.name(), layer)
            self.dlg.blockGroupLayerComboBoxDatafill.addItem(layer.name(), layer)
            self.dlg.targetLayerComboBox.addItem(layer.name(), layer)
        
        # If no layers found, add a message
        if not polygon_layers:
            self.dlg.blockGroupLayerComboBox.addItem("(No polygon layers loaded)", None)
            self.dlg.blockGroupLayerComboBoxDatafill.addItem("(No polygon layers loaded)", None)
            self.dlg.targetLayerComboBox.addItem("(No polygon layers loaded)", None)

    def get_layer_from_combo(self, combo_box):
        """
        Get the QgsVectorLayer from a combo box.
        
        Args:
            combo_box: QComboBox containing layer names
        
        Returns:
            QgsVectorLayer or None
        """
        current_index = combo_box.currentIndex()
        if current_index < 0:
            return None
        
        # Get the layer object stored as user data
        layer = combo_box.itemData(current_index)
        return layer

    def browse_output_file(self):
        """
        Open file dialog to select output CSV file location.
        """
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg,
            "Select Output File",
            "",
            "CSV Files (*.csv)"
        )
        
        if filename:
            # Make sure it has .csv extension
            if not filename.lower().endswith('.csv'):
                filename += '.csv'
            
            # Update the line edit
            self.dlg.outputFileLineEdit.setText(filename)

    def select_point_on_map(self):
        """
        Activate the map tool to allow user to click on map.
        """
        # Hide the dialog while selecting point
        self.dlg.hide()
        
        # Activate the point selection tool
        self.iface.mapCanvas().setMapTool(self.point_tool)
        
        # Show message to user
        self.iface.messageBar().pushMessage(
            "Point Selection",
            "Click on the map to select a location for your report",
            level=0,  # Info level
            duration=5
        )
    
    def generate_point_report(self):
        """
        Generate demographic report for selected point and radius.
        Uses proportional area allocation for block groups.
        """
        print("=== Starting Point Report Generation ===")
        
        # Get inputs
        package = self.dlg.packageComboBox.currentText()
        layer = self.get_layer_from_combo(self.dlg.blockGroupLayerComboBox)
        radius_text = self.dlg.radiusLineEdit.text()
        
        # Validate layer selection
        if not layer:
            QMessageBox.warning(self.dlg, "Error", "Please select a block group layer")
            return
        
        # Validate point selection
        if not self.selected_point:
            QMessageBox.warning(
                self.dlg,
                "Error",
                "Please click 'Select Point on Map' and choose a location first"
            )
            return
        
        # Validate radius
        if not radius_text:
            QMessageBox.warning(self.dlg, "Error", "Please enter a radius in miles")
            return
        
        try:
            radius_miles = float(radius_text)
            if radius_miles <= 0:
                raise ValueError("Radius must be positive")
            # Convert miles to meters (1 mile = 1609.34 meters)
            radius_meters = radius_miles * 1609.34
            print(f"Radius: {radius_miles} miles = {radius_meters:.2f} meters")
        except ValueError as e:
            QMessageBox.warning(self.dlg, "Error", f"Invalid radius: {str(e)}")
            return
        
        # Get variables for this package
        variables = get_package_variables(package)
        
        if not variables:
            QMessageBox.warning(self.dlg, "Error", f"No variables defined for {package}")
            return
        
        # Validate layer has required fields
        is_valid, missing, msg = validate_layer_for_analysis(layer, variables)
        if not is_valid:
            QMessageBox.warning(
                self.dlg,
                "Data Validation Error",
                f"{msg}\n\nMissing variables: {', '.join(missing[:10])}"
            )
            return
        
        print("✓ Validation passed")
        print(f"  Package: {package}")
        print(f"  Layer: {layer.name()}")
        print(f"  Point: {self.selected_point.x():.6f}, {self.selected_point.y():.6f}")
        print(f"  Radius: {radius_miles} miles ({radius_meters:.2f} meters)")
        print(f"  Variables: {len(variables)}")
        
        # Create spatial processor
        processor = SpatialProcessor()
        
        # Create buffer around selected point (using meters)
        print("Creating buffer...")
        buffer_geom = processor.create_buffer(
            self.selected_point,
            radius_meters,
            layer.crs()
        )
        
        # Aggregate demographic data
        print("Aggregating demographic data...")
        # Now pass layer.crs() as the analysis_crs parameter
        results, metadata = processor.aggregate_demographics(
            layer,
            buffer_geom,
            layer.crs(),  
            variables,
            package
        )
        
        print("✓ Aggregation complete")
        print(f"  Block groups processed: {metadata['block_groups_processed']}")
        print(f"  Block groups intersecting: {metadata['block_groups_intersecting']}")
        
        # Display results (pass miles for display)
        self.display_report_results(results, metadata, package, radius_miles)

    def point_selected(self, point):
        """
        Called when user clicks on map with point tool.
        
        Args:
            point: QgsPointXY - The clicked location
        """
        # Store the selected point
        self.selected_point = point
        
        # Update the label to show coordinates
        coord_text = f"Selected Location: {point.x():.6f}, {point.y():.6f}"
        self.dlg.locationLabel.setText(coord_text)
        
        # Show success message
        self.iface.messageBar().pushMessage(
            "Success",
            "Point selected! You can now generate a report.",
            level=3,  # Success level
            duration=3
        )
        
        # Show the dialog again
        self.dlg.show()
        
        # Deactivate the point tool
        self.iface.mapCanvas().unsetMapTool(self.point_tool)
        
    def display_report_results(self, results, metadata, package, radius_miles):
        """
        Display aggregated demographic results and offer to save as PDF.
        
        Args:
            results: dict - Aggregated demographic values
            metadata: dict - Processing metadata
            package: str - Package name
            radius_miles: float - Analysis radius in miles
        """
        from qgis.PyQt.QtWidgets import QMessageBox
        
        # Ask user if they want to save as PDF
        reply = QMessageBox.question(
            self.dlg,
            'Save Report',
            'Would you like to save this report as a PDF?',
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.Yes
        )
        
        if reply == QMessageBox.Yes:
            # Get output file path
            filename, _filter = QFileDialog.getSaveFileName(
                self.dlg,
                "Save Report as PDF",
                "",
                "PDF Files (*.pdf)"
            )
            
            if filename:
                # Make sure it has .pdf extension
                if not filename.lower().endswith('.pdf'):
                    filename += '.pdf'
                
                # Generate PDF
                try:
                    from .package_config import VARIABLE_CATEGORIES, VARIABLE_DEFINITIONS
                    
                    # CUSTOMIZE THESE VALUES FOR YOUR BRANDING
                    # Put your logo file in the plugin directory, or use full path
                    logo_path = os.path.join(self.plugin_dir, "logo.png")  # Change to your logo filename
                    if not os.path.exists(logo_path):
                        logo_path = None  # If logo doesn't exist, don't use it
                    
                    generator = DemographicPDFGenerator(
                        logo_path=logo_path,
                        company_name="Retail Gravity",  # CHANGE THIS
                        website="www.retailgravity.com",     # CHANGE THIS
                        brand_color="#3498db"              # CHANGE THIS to your brand color (hex code)
                    )
                    
                    generator.generate_point_report(
                        filename,
                        results,
                        metadata,
                        package,
                        radius_miles,
                        (self.selected_point.x(), self.selected_point.y()),
                        VARIABLE_CATEGORIES,
                        VARIABLE_DEFINITIONS
                    )
                    
                    print(f"✓ PDF report saved to: {filename}")
                    
                    # Ask if user wants to open the PDF
                    open_reply = QMessageBox.question(
                        self.dlg,
                        'Success',
                        f'Report saved successfully!\n\nWould you like to open it?',
                        QMessageBox.Yes | QMessageBox.No,
                        QMessageBox.Yes
                    )
                    
                    if open_reply == QMessageBox.Yes:
                        import subprocess
                        import platform
                        
                        # Open PDF with default application
                        if platform.system() == 'Darwin':       # macOS
                            subprocess.call(('open', filename))
                        elif platform.system() == 'Windows':    # Windows
                            os.startfile(filename)
                        else:                                   # linux variants
                            subprocess.call(('xdg-open', filename))
                    
                except Exception as e:
                    import traceback
                    print(f"ERROR generating PDF: {str(e)}")
                    print(traceback.format_exc())
                    QMessageBox.critical(
                        self.dlg,
                        "Error",
                        f"Failed to generate PDF:\n{str(e)}"
                    )
        else:
            # Just show in dialog
            dialog = QDialog(self.dlg)
            dialog.setWindowTitle("Demographic Report Results")
            dialog.resize(600, 800)
            
            from qgis.PyQt.QtWidgets import QVBoxLayout, QTextBrowser
            layout = QVBoxLayout()
            text_browser = QTextBrowser()
            text_browser.setOpenExternalLinks(False)
            
            html = self._build_html_report(results, metadata, package, radius_miles)
            text_browser.setHtml(html)
            
            layout.addWidget(text_browser)
            dialog.setLayout(layout)
            dialog.exec_()
    
    def _build_html_report(self, results, metadata, package, radius):
        """
        Build HTML formatted report from results.
        
        Args:
            results: dict - Aggregated values
            metadata: dict - Processing metadata
            package: str - Package name
            radius: float - Analysis radius in MILES
        
        Returns:
            str - HTML formatted report
        """
        from .package_config import VARIABLE_CATEGORIES, VARIABLE_DEFINITIONS
        
        # Convert miles to km for display
        radius_km = radius * 1.60934
        
        html = f"""
        <html>
        <head>
            <style>
                body {{ font-family: Arial, sans-serif; margin: 20px; }}
                h1 {{ color: #2c3e50; border-bottom: 2px solid #3498db; padding-bottom: 10px; }}
                h2 {{ color: #34495e; margin-top: 30px; border-bottom: 1px solid #bdc3c7; }}
                .metadata {{ background-color: #ecf0f1; padding: 15px; border-radius: 5px; margin-bottom: 20px; }}
                .variable {{ margin: 10px 0; padding: 8px; background-color: #f8f9fa; border-left: 3px solid #3498db; }}
                .variable-name {{ font-weight: bold; color: #2c3e50; }}
                .variable-value {{ color: #27ae60; font-weight: bold; }}
                .definition {{ font-size: 0.9em; color: #7f8c8d; font-style: italic; }}
            </style>
        </head>
        <body>
            <h1>Demographic Report</h1>
            
            <div class="metadata">
                <strong>Package:</strong> {package}<br>
                <strong>Analysis Radius:</strong> {radius:.2f} miles ({radius_km:.2f} km)<br>
                <strong>Analysis Area:</strong> {metadata.get('total_analysis_area_sqm', 0)/1000000:.2f} sq km<br>
                <strong>Block Groups Processed:</strong> {metadata.get('block_groups_processed', 0)}<br>
                <strong>Block Groups Intersecting:</strong> {metadata.get('block_groups_intersecting', 0)}
            </div>
        """
        
        # Group variables by category
        for category, var_list in VARIABLE_CATEGORIES.items():
            # Check if any variables in this category are in results
            category_vars = [v for v in var_list if v in results]
            
            if not category_vars:
                continue
            
            html += f"<h2>{category}</h2>"
            
            for var in category_vars:
                value = results[var]
                definition = VARIABLE_DEFINITIONS.get(var, "No definition available")
                
                # Format value based on type
                if isinstance(value, float):
                    if value > 1000:
                        formatted_value = f"{value:,.0f}"
                    else:
                        formatted_value = f"{value:,.2f}"
                else:
                    formatted_value = str(value)
                
                html += f"""
                <div class="variable">
                    <span class="variable-name">{var}:</span> 
                    <span class="variable-value">{formatted_value}</span><br>
                    <span class="definition">{definition}</span>
                </div>
                """
        
        html += """
        </body>
        </html>
        """
        
        return html
    
    def generate_datafill(self):
        """
        Generate datafill CSV for target layer.
        Aggregates block group data into target polygons using proportional allocation.
        """
        print("=== Starting Layer Datafill Generation ===")
        
        # Get inputs
        package = self.dlg.packageComboBoxDatafill.currentText()
        bg_layer = self.get_layer_from_combo(self.dlg.blockGroupLayerComboBoxDatafill)
        target_layer = self.get_layer_from_combo(self.dlg.targetLayerComboBox)
        output_file = self.dlg.outputFileLineEdit.text()
        
        # Validate inputs
        if not bg_layer:
            QMessageBox.warning(self.dlg, "Error", "Please select a block group layer")
            return
        
        if not target_layer:
            QMessageBox.warning(self.dlg, "Error", "Please select a target polygon layer")
            return
        
        if not output_file:
            QMessageBox.warning(self.dlg, "Error", "Please select an output CSV file")
            return
        
        # Get variables for this package
        variables = get_package_variables(package)
        
        if not variables:
            QMessageBox.warning(self.dlg, "Error", f"No variables defined for {package}")
            return
        
        # Validate block group layer has required fields
        is_valid, missing, msg = validate_layer_for_analysis(bg_layer, variables)
        if not is_valid:
            QMessageBox.warning(
                self.dlg,
                "Data Validation Error",
                f"{msg}\n\nMissing variables: {', '.join(missing[:10])}"
            )
            return
        
        print("✓ Validation passed")
        print(f"  Package: {package}")
        print(f"  Block Group Layer: {bg_layer.name()}")
        print(f"  Target Layer: {target_layer.name()}")
        print(f"  Output File: {output_file}")
        print(f"  Variables: {len(variables)}")
        print(f"  Target Features: {target_layer.featureCount()}")
        
        # Process datafill
        self._process_datafill(bg_layer, target_layer, variables, output_file)
    
    def _process_datafill(self, bg_layer, target_layer, variables, output_file):
        """
        Process datafill by aggregating demographics for each target polygon.
        Enhanced version with metadata header and proper rounding.
        
        Args:
            bg_layer: QgsVectorLayer - Block groups with demographic data
            target_layer: QgsVectorLayer - Target polygons to aggregate into
            variables: list - Variables to aggregate
            output_file: str - Path to output CSV file
        """
        import csv
        from qgis.PyQt.QtWidgets import QProgressDialog
        from qgis.PyQt.QtCore import Qt
        from datetime import datetime
        from .formatting_utils import format_value_for_csv
        
        # Create spatial processor
        processor = SpatialProcessor()
        
        # Get target features
        target_features = list(target_layer.getFeatures())
        total_features = len(target_features)
        
        # Create progress dialog
        progress = QProgressDialog(
            "Processing datafill...",
            "Cancel",
            0,
            total_features,
            self.dlg
        )
        progress.setWindowModality(Qt.WindowModal)
        progress.setWindowTitle("Generating Datafill")
        
        # Prepare output data
        output_rows = []
        total_intersecting = 0
        
        # Process each target feature
        for idx, target_feature in enumerate(target_features):
            # Update progress
            progress.setValue(idx)
            if progress.wasCanceled():
                print("Datafill cancelled by user")
                return
            
            progress.setLabelText(f"Processing feature {idx + 1} of {total_features}")
            
            # Get target geometry
            target_geom = target_feature.geometry()
            
            # Aggregate demographics for this target polygon
            results, metadata = processor.aggregate_demographics(
                bg_layer,
                target_geom,
                target_layer.crs(),
                variables,
                None
            )
            
            total_intersecting += metadata['block_groups_intersecting']
            
            # Create output row
            row = {}
            
            # Add target feature ID if it has one
            if 'id' in [f.name().lower() for f in target_layer.fields()]:
                row['target_id'] = target_feature['id']
            else:
                row['target_id'] = target_feature.id()
            
            # Add all variable values with proper rounding
            for var in variables:
                value = results.get(var, 0.0)
                # Apply rounding rules
                row[var] = format_value_for_csv(var, value)
            
            output_rows.append(row)
        
        # Close progress dialog
        progress.setValue(total_features)
        
        # Write to CSV with metadata header
        print(f"Writing results to {output_file}")
        
        try:
            with open(output_file, 'w', newline='', encoding='utf-8') as csvfile:
                # Write metadata as comments at the top
                package = self.dlg.packageComboBoxDatafill.currentText()
                
                # Create header with target_id + all variables
                fieldnames = ['target_id'] + variables
                writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                
                writer.writeheader()
                for row in output_rows:
                    writer.writerow(row)
            
            print(f"✓ Datafill complete: {len(output_rows)} features processed")
            
            # Show success message
            QMessageBox.information(
                self.dlg,
                "Success",
                f"Datafill complete!\n\n{len(output_rows)} features processed\n{total_intersecting} block group intersections\n\nOutput saved to:\n{output_file}"
            )
            
        except Exception as e:
            print(f"ERROR writing CSV: {str(e)}")
            QMessageBox.critical(
                self.dlg,
                "Error",
                f"Failed to write output file:\n{str(e)}"
            )
            
    def run(self):
        """Run method that performs all the real work"""
        
        # Create the dialog if it doesn't exist yet (first time)
        if self.first_start:
            self.first_start = False
            self.dlg = DemoReportsDialog()
            
            # Create map tool for point selection
            self.point_tool = PointTool(self.iface.mapCanvas())
            self.point_tool.pointSelected.connect(self.point_selected)
            self.selected_point = None  # Store the selected point
            
            # Connect button signals ONLY ONCE HERE
            self.dlg.selectPointButton.clicked.connect(self.select_point_on_map)
            self.dlg.generateReportButton.clicked.connect(self.generate_point_report)
            self.dlg.generateDatafillButton.clicked.connect(self.generate_datafill)
            self.dlg.browseButton.clicked.connect(self.browse_output_file)
            self.dlg.closeButton.clicked.connect(self.dlg.close) 
        
        # Populate layer combo boxes with current layers
        self.populate_layer_combos()
        
        # Show the dialog (non-modal)
        self.dlg.show()