from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterNumber,
    QgsProcessingParameterCrs,
    QgsProcessingParameterFolderDestination,
    QgsProcessingOutputVectorLayer,
    QgsProcessingException,
    QgsProcessingFeedback,
    QgsProcessingContext,
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsProject
)
import os

# Import your core logic
from .allocation_logic import run_allocation


class DemandAllocatorAlgorithm(QgsProcessingAlgorithm):
    INPUT_HOUSE_POLYGONS = 'INPUT_HOUSE_POLYGONS'
    INPUT_PIPE_NETWORK = 'INPUT_PIPE_NETWORK'
    THRESHOLD = 'THRESHOLD'
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'
    OUTPUT_CRS = 'OUTPUT_CRS'
    OUTPUT_HOUSE_POINTS = 'OUTPUT_HOUSE_POINTS'
    OUTPUT_REFERENCE_LINES = 'OUTPUT_REFERENCE_LINES'

    def initAlgorithm(self, config=None):
        # ─── Inputs ───────────────────────────────────────────────────────
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_HOUSE_POLYGONS,
                'House Polygons',
                types=[QgsProcessing.TypeVectorPolygon],
                defaultValue=None,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_PIPE_NETWORK,
                'Pipe Network',
                types=[QgsProcessing.TypeVectorLine],
                defaultValue=None,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.THRESHOLD,
                'Clustering Threshold (meters)',
                type=QgsProcessingParameterNumber.Double,
                defaultValue=50.0,
                minValue=0.0,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterCrs(
                self.OUTPUT_CRS,
                'Output Coordinate Reference System',
                defaultValue='ProjectCrs'
            )
        )

        # ─── Optional save location ───────────────────────────────────────
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_FOLDER,
                'Output Folder (optional – leave empty for temporary layers)',
                defaultValue=None,
                optional=True
            )
        )

        # ─── Outputs ──────────────────────────────────────────────────────
        self.addOutput(
            QgsProcessingOutputVectorLayer(
                self.OUTPUT_HOUSE_POINTS,
                'House Load Points',
                QgsProcessing.TypeVectorPoint
            )
        )

        self.addOutput(
            QgsProcessingOutputVectorLayer(
                self.OUTPUT_REFERENCE_LINES,
                'Reference Lines (centroid to demand point)',
                QgsProcessing.TypeVectorLine
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        polygons = self.parameterAsVectorLayer(parameters, self.INPUT_HOUSE_POLYGONS, context)
        pipes = self.parameterAsVectorLayer(parameters, self.INPUT_PIPE_NETWORK, context)
        threshold = self.parameterAsDouble(parameters, self.THRESHOLD, context)
        output_crs = self.parameterAsCrs(parameters, self.OUTPUT_CRS, context)
        output_folder = self.parameterAsString(parameters, self.OUTPUT_FOLDER, context)

        if not polygons.isValid() or not pipes.isValid():
            raise QgsProcessingException("One or both input layers are invalid or missing.")

        feedback.pushInfo(f"Processing {polygons.featureCount()} house polygons and {pipes.featureCount()} pipe segments")

        # Run core logic → memory layers
        point_layer_mem, line_layer_mem = run_allocation(
            polygons,
            pipes,
            threshold,
            feedback,
            output_crs
        )

        # ─── Decide what to add to project ─────────────────────────────────
        final_point_layer = point_layer_mem
        final_line_layer = line_layer_mem

        if output_folder and os.path.isdir(output_folder):
            try:
                point_path = os.path.join(output_folder, "house_load_points.shp")
                line_path = os.path.join(output_folder, "reference_lines.shp")

                # Save to shapefile
                QgsVectorFileWriter.writeAsVectorFormat(
                    point_layer_mem,
                    point_path,
                    "utf-8",
                    point_layer_mem.crs(),
                    "ESRI Shapefile"
                )

                QgsVectorFileWriter.writeAsVectorFormat(
                    line_layer_mem,
                    line_path,
                    "utf-8",
                    line_layer_mem.crs(),
                    "ESRI Shapefile"
                )

                feedback.pushInfo(f"Results saved as Shapefiles:\n  {point_path}\n  {line_path}")

                # Load saved shapefiles instead of memory layers
                point_shp = QgsVectorLayer(point_path, "House Load Points", "ogr")
                line_shp = QgsVectorLayer(line_path, "Reference Lines (centroid to demand)", "ogr")

                if point_shp.isValid() and line_shp.isValid():
                    final_point_layer = point_shp
                    final_line_layer = line_shp
                    feedback.pushInfo("Loaded permanent shapefile layers into project")
                else:
                    feedback.reportError("Saved files created but could not be loaded — using temporary layers")

            except Exception as e:
                feedback.reportError(f"Failed to save or load shapefiles: {str(e)}")
                # fall back to memory layers

        # ─── Add the final (chosen) layers to project ─────────────────────
        QgsProject.instance().addMapLayer(final_point_layer)
        QgsProject.instance().addMapLayer(final_line_layer)

        feedback.pushInfo("Processing completed successfully")

        return {
            self.OUTPUT_HOUSE_POINTS: final_point_layer,
            self.OUTPUT_REFERENCE_LINES: final_line_layer
        }

    def name(self):
        return 'allocate_demand_points'

    def displayName(self):
        return 'Allocate Demand Points'

    def group(self):
        return 'Water Utilities'

    def groupId(self):
        return 'water_utilities'

    def shortHelpString(self):
        return """<html><body>
<h2>Algorithm Description</h2>

<p>This processing algorithm allocates demand from house/building polygons to a pipe network for water supply, sewer, or utility purposes.</p>

<h2>Workflow</h2>
<ol>
  <li>Compute centroid of each house polygon</li>
  <li>Snap centroid to the nearest point on the pipe network</li>
  <li>Cluster snapped points within the user-defined threshold distance</li>
  <li>Generate aggregated demand points at cluster centroids</li>
  <li>Create reference lines linking each original house centroid to its assigned demand point</li>
</ol>

<h2>Outputs</h2>

<h3>House Load Points</h3>
<p>Point layer with aggregated demand locations</p>
<ul>
  <li><b>NUM</b>: number of houses aggregated at this point</li>
</ul>

<h3>Reference Lines</h3>
<p>Line layer showing connections from each house centroid to its demand point</p>
<ul>
  <li><b>NUM_HOUSES</b>: number of houses per cluster</li>
  <li><b>DISTANCE_M</b>: length of the connection line</li>
</ul>

<h2>Parameters</h2>
<dl>
  <dt><b>House Polygons</b></dt>
  <dd>Vector polygon layer representing buildings or residential units</dd>

  <dt><b>Pipe Network</b></dt>
  <dd>Vector line layer of water supply or sewer pipes</dd>

  <dt><b>Clustering Threshold (m)</b></dt>
  <dd>Maximum distance for grouping nearby snapped points (default: 50 m)</dd>

  <dt><b>Output CRS</b></dt>
  <dd>Coordinate reference system for output layers (defaults to project CRS)</dd>
  <span style="color: #c33; font-weight: bold;">Important:</span> Use a <strong>projected CRS</strong>.
</dd>

  <dt><b>Output Folder</b></dt>
  <dd>Optional folder to save results permanently as Shapefiles (.shp)</dd>
</dl>

<h2>Output Behavior</h2>
<ul>
  <li>If a folder is selected → saves <code>house_load_points.shp</code> and <code>reference_lines.shp</code> and loads them into the project</li>
  <li>If no folder is selected → creates temporary memory layers added to the current project</li>
</ul>

</body></html>"""

    def createInstance(self):
        return DemandAllocatorAlgorithm()