"""
/***************************************************************************
 DemSlicerDockWidget
                                 A QGIS plugin
 D.E.M. slicer, produces lines
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2019-10-24
        git sha              : $Format:%H$
        copyright            : (C) 2019 by xc
        email                : xavier.culos@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

TODO : compass : grid
TODO : poi - azimuth en mode ortho
TODO : attributs : remplacer num+prof par cutid
"""

import os

from .__about__ import DIR_PLUGIN_ROOT
from .logic import tools

from qgis.PyQt import QtWidgets, uic
from qgis.PyQt.QtWidgets import (
    QFileDialog,
    QApplication
)
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QTranslator,
    pyqtSignal,
    Qt,
    QVariant,
    QSettings,
)
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsApplication,
    Qgis,
    QgsWkbTypes,
    QgsGeometry,
    QgsPoint, QgsPointXY,
    QgsLineString,
    QgsMessageLog,
    QgsProject,
    QgsMapLayer,
    QgsVectorLayer,
    QgsField,
    QgsUnitTypes,
    QgsCoordinateReferenceSystem,
    QgsFeature,
    QgsCoordinateTransform,
    QgsFeatureRequest,
    QgsRectangle,
)

from qgis.gui import QgsRubberBand, QgsMapTool
import math
import inspect

FORM_CLASS, _ = uic.loadUiType(
    str(DIR_PLUGIN_ROOT / "ui/dem_slicer_dockwidget_base.ui")
)


class DemSlicerDockWidget(QtWidgets.QDockWidget, FORM_CLASS):

    closingPlugin = pyqtSignal()
    plugin = None

    def __init__(self, _plugin, parent=None):
        """Constructor."""
        super(DemSlicerDockWidget, self).__init__(parent)
        self.setupUi(self)
        self.plugin = _plugin
        self.canvas = self.plugin.iface.mapCanvas()
        self.started = False
        self.btnStart.setCheckable(True)

        self.mt = MapTool(self)

        locale = QSettings().value("locale/userLocale")[0:2]
        localePath = os.path.join(str(DIR_PLUGIN_ROOT / "i18n"), "{}.qm".format(locale))
        self.info(localePath)

        if os.path.exists(localePath):
            self.translator = QTranslator()
            self.translator.load(localePath)
            QCoreApplication.installTranslator(self.translator)

        for widget in [
            self.rasterListLabel,
            self.btnBuild,
            self.btnStart,
            self.lineCountLabel,
            self.zFactorLabel,
            self.xStepLabel,
            self.zShiftLabel,
            self.renderLines,
            self.renderPolygons,
            self.renderRidges,
            self.renderCompass,
            self.toSmooth,
            self.renderSource,
            self.parallelView,
            self.poiListLabel,
            self.labelElevation,
            self.btnLoad,
            self.labelBase,
        ]:
            widget.setText(self.tr(widget.text()))

        for widget in [self.btnSave, self.reset, self.btnLoad]:
            widget.setToolTip(self.tr(widget.toolTip()))

        self.widget2Enable = [
            self.lineCount,
            self.rasterList,
            self.poiList,
            self.xStep,
            self.zShift,
            self.zFactor,
            self.renderLines,
            self.renderPolygons,
            self.renderRidges,
            self.renderCompass,
            self.toSmooth,
            self.renderSource,
            self.btnBuild,
            self.progressBar,
            self.parallelView,
            self.reset,
            self.btnLoad,
            self.btnSave,
            self.elevation,
            self.base,
        ]

        self.setAlert("")

    def tr(self, message):
        return QCoreApplication.translate("DemSlicer", message)

    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()

    def info(self, message, d=2):
        self.plugin.iface.messageBar().pushMessage(
            "DEM Slicer", self.tr(message), level=Qgis.Info, duration=d
        )

    def warning(self, message, d=2):
        self.plugin.iface.messageBar().pushMessage(
            "DEM Slicer", self.tr(message), level=Qgis.Warning, duration=d
        )

    def log(self, message):
        QgsMessageLog.logMessage(str(message), "Extensions")
        QApplication.processEvents()

    def setAlert(self, message):
        self.alert.setText(message)
        QApplication.processEvents()

    def start(self):
        """Start the processus : activates tools, builds rubber bands..."""
        # check ap units
        if self.canvas.mapSettings().destinationCrs().mapUnits() == QgsUnitTypes.DistanceDegrees:
            self.warning(self.tr("Bad map unit (Degrees)"))
            self.setAlert(self.tr("Bad map unit (Degrees)"))
        else:
            self.setAlert("")

        rId = self.rasterList.itemData(self.rasterList.currentIndex())
        poiId = self.poiList.itemData(self.poiList.currentIndex())
        self.rasterList.clear()
        self.poiList.clear()
        self.poiList.addItem("- None -", 0)
        i, j = 0, 1
        for layer in QgsProject.instance().mapLayers().values():
            # If layer is a raster and it is not a multiband type
            if layer.type() == QgsMapLayer.RasterLayer:
                # Add to list
                self.rasterList.addItem(layer.name(), layer.id())
                if layer.id() == rId:
                    self.rasterList.setCurrentIndex(i)
                i = i + 1

            if layer.type() == QgsMapLayer.VectorLayer:
                # Add to list
                self.poiList.addItem(layer.name(), layer.id())
                if layer.id() == poiId:
                    self.poiList.setCurrentIndex(j)
                j = j + 1

        if i == 0:
            return False

        for w in self.widget2Enable:
            w.setEnabled(True)
        self.renderCompass.setEnabled(not self.parallelView.isChecked())
        self.zShift.setEnabled(self.parallelView.isChecked())

        self.canvas.setMapTool(self.mt)

        self.mt.newRubber()

        return True

    def cancel(self):
        self.mt.hide()
        self.canvas.unsetMapTool(self.mt)

        for w in self.widget2Enable:
            w.setEnabled(False)

    def getElevation(self, point):
        """
        Returns elevation

        :param point: The point to identify elevation (mntLayer crs)
        :type point: QgsPointXY

        :return: The elevation (first band value)
        :rtype: int
        """
        try:
            v, ok = self.mntLayer.dataProvider().sample(point, 1)
            return v if ok else 0
        except Exception:
            return 0

    def getGaz(self, pt, polys, prof):
        """
        Calculates the number of hidden profiles, behind a segment

        :param pt: point to cjeck
        :param polys: profiles
        :param prof: 'depth' of the current segment

        :return: Number of hidden profiles
        :rtype: int
        """
        for i, p in enumerate(polys[::-1]):
            if i > prof and p.contains(pt):
                return i - prof

        return None

    def getVisibility(self, pt, polys, prof):
        """
        Returns if then point is visible (> 0) or hidden (<= 0) by the profiles
        """
        visi = 1
        log = ""
        for i, p in enumerate(polys[::-1]):
            if prof >= i and p.contains(pt):
                visi = visi - 1

            log = log + "prof {} i {} visi {}\n".format(prof, i, visi)
            if visi <= -2:
                break

        return visi, log

    def getNewZ(self, zTarget, depth):
        """
        :param zTarget : point target altitude
        :param depth : distance (depth) between observer and point (2d)

        :return: new projected z
        """
        # Altitude de la cible par rapport à l'observateur
        h = zTarget - self.elevation.value() - self.altY
        return h * ((self.mt.horizon) / depth)

    def getProf(self, depth):
        # distance entre deux profils
        dd = self.mt.zoneDepth / (self.lineCount.value() - 1)
        # indice du profil
        prof = (depth - self.mt.d0) / dd
        # print("{} {} {} {}".format(self.mt.zoneDepth, (depth - self.mt.d0), self.lineCount.value(), prof))
        return prof

    def getProjectionPointAlti(self, pt, alti):
        # new Y
        d = (self.mt.d0 + self.mt.segCD.distance(QgsGeometry.fromPointXY(pt))) \
            if self.parallelView.isChecked() else self.mt.pointXY('Y').distance(pt)
        newZ = self.getNewZ(alti, d)
        zf = self.zFactor.value()
        if self.parallelView.isChecked():
            newY = (
                self.mt.y('R')
                + (newZ * zf)
                + (d * self.zShift.value() / self.mt.zoneDepth)
            )
        else:
            newY = (
                self.mt.y('R')
                + (newZ * zf)
            )

        # new X'
        if self.parallelView.isChecked():
            newX = self.mt.x('R') + self.mt.segAD.distance(QgsGeometry.fromPointXY(pt))
        else:
            aPeak = self.mt.pointXY('Y').azimuth(pt)
            if aPeak < self.mt.azimuthLeft:
                aPeak = aPeak + 360
            newX = self.mt.x('R') + ((aPeak-self.mt.azimuthLeft)/(self.mt.azimuthRight-self.mt.azimuthLeft))*self.mt.finalWidth

        return newX, newY

    def getProjectionPoint(self, pt):
        alti = self.getElevation(self.xMap2Raster.transform(pt.x(), pt.y()))
        newX, newY = self.getProjectionPointAlti(pt, alti)
        return QgsPointXY(newX, newY)

    def getThumbnailGeom(self) -> QgsGeometry:
        """
            Return first line, horizon line and all polygons
        """
        _, _, aPolys = self.getLinesAndPolys(sample=True)
        return (
            QgsGeometry.fromPolygonXY(aPolys[-1]),
            QgsGeometry.fromPolygonXY(aPolys[0]),
            QgsGeometry.fromMultiPolygonXY(aPolys[1:-1])
        )

    def getPeakGeom(self, ptPeak):
        ptXY = self.getProjectionPoint(ptPeak)
        return QgsGeometry.fromPointXY(ptXY)

    def buildLayer(self, layer, aLinesOrPolys):
        layer.startEditing()
        layer.dataProvider().addAttributes(
            [QgsField("demslicer_cutid", QVariant.Int)]
        )
        layer.updateFields()
        feats = []
        # croissant en partant du fond (ligne d'horizon vers premier profil)
        for fid, lin in enumerate(aLinesOrPolys):
            self.progress(1, '')
            feature = QgsFeature(fid)
            feature.setAttributes([str(fid)])
            feature.setGeometry(lin)
            feats.append(feature)

        layer.dataProvider().addFeatures(feats)
        layer.commitChanges()

    def getSourcePolylines(self, dx, sample=False):
        if sample:
            return self.mt.getSampleLines()
        else:
            return self.mt.getLines()

    def getProjPolylines(self, dx, sample=False):
        if sample:
            geom = QgsGeometry.fromMultiPolylineXY(self.mt.getSampleSkylines())
        else:
            geom = QgsGeometry.fromMultiPolylineXY(self.mt.getSkylines())

        return geom.boundingBox(), geom.densifyByDistance(dx).asMultiPolyline()

    def getLinesAndPolys(self, sample=False):
        vSource = None
        if sample:
            dx = self.mt.finalWidth / 15.0
        else:
            dx = self.xStep.value()

            if self.renderSource.isChecked():
                vSource = QgsVectorLayer(
                    "Point?crs={}".format(QgsProject.instance().crs().authid()),
                    "Source",
                    "memory",
                )
                vSource.startEditing()
                vSource.dataProvider().addAttributes([QgsField("demslicer_cutid", QVariant.Int), QgsField("demslicer_z", QVariant.Int)])
                vSource.updateFields()
                feats = []

        # source
        polylineIn = self.getSourcePolylines(dx, sample)

        # projection
        self.projBox, polylineOut = self.getProjPolylines(dx, sample)
        ymin = self.projBox.yMaximum()
        ymax = self.projBox.yMinimum()

        aLines = []
        aPolys = []
        zf = self.zFactor.value()

        # search for Z values
        # print("finalw:{} dx:{} - {} polylineIn, {} polylineOut".format(self.mt.finalWidth, dx, len(polylineIn), len(polylineOut)))
        id = self.lineCount.value()-1
        for lineIn, lineOut in zip(polylineIn, polylineOut):
            ds, zs = map(
                list,
                zip(
                    *[
                        (
                            # distance ligne observateur - point si //
                            # distance entre les deux points sinon
                            self.mt.d0
                            + self.mt.segCD.distance(QgsGeometry.fromPointXY(point))
                            if self.parallelView.isChecked()
                            else self.mt.pointXY('Y').distance(point),
                            self.getElevation(
                                self.xMap2Raster.transform(point.x(), point.y())
                            ),
                        )
                        for point in lineIn
                    ]
                ),
            )

            for d, z, point in zip(ds, zs, lineOut):
                newZ = self.getNewZ(z, d)
                newY = (
                    self.mt.y('R')
                    + (newZ * zf)
                    + ((d * self.zShift.value() / self.mt.zoneDepth) if self.parallelView.isChecked() else 0)
                )
                point.setY(newY)
                if newY < ymin:
                    ymin = newY
                if newY > ymax:
                    ymax = newY

            if sample or not self.toSmooth.isChecked():
                aLines.append(lineOut)
            else:
                aLines.append(QgsGeometry.fromPolylineXY(lineOut).smooth(2, 0.25).asPolyline())

            if not sample and self.renderSource.isChecked():
                for z, pt in zip(zs, lineIn):
                    vtx = QgsPoint(pt.x(), pt.y())
                    newG = QgsGeometry(vtx)
                    newF = QgsFeature()
                    newF.setGeometry(newG)
                    newF.setFields(vSource.fields())
                    newF.setAttribute("demslicer_cutid", id)
                    newF.setAttribute("demslicer_z", z)
                    feats.append(newF)
                id = id - 1

            self.yMin = ymin
            self.yMax = ymax

        if sample:
            # pas de copy necessaire
            for lineOut in aLines:
                lineOut.append(QgsPointXY(self.projBox.xMaximum(), self.yMin-self.base.value()))
                lineOut.append(QgsPointXY(self.projBox.xMinimum(), self.yMin-self.base.value()))
                aPolys.append([lineOut])
        else:
            # copy
            for lineOut in [lin.copy() for lin in aLines]:
                lineOut.append(QgsPointXY(self.projBox.xMaximum(), self.yMin-self.base.value()))
                lineOut.append(QgsPointXY(self.projBox.xMinimum(), self.yMin-self.base.value()))
                aPolys.append([lineOut])

            if self.renderSource.isChecked():
                # add observer to source layer
                newG = QgsGeometry.fromPointXY(self.mt.points['Y'].pxy)
                newF = QgsFeature()
                newF.setGeometry(newG)
                newF.setFields(vSource.fields())
                newF.setAttribute("demslicer_cutid", -1)
                newF.setAttribute("demslicer_z", self.getElevation(self.xMap2Raster.transform(self.mt.points['Y'].x(), self.mt.points['Y'].y())))
                feats.append(newF)

                vSource.dataProvider().addFeatures(feats)
                vSource.commitChanges()

        return vSource, aLines, aPolys

    def progress(self, n, s):
        self.progressBar.setValue(self.progressBar.value()+n)
        QApplication.processEvents()
        # self.log("progress {} : {}".format(s, self.progressBar.value()))

    def buildSlices(self):

        try:
            QgsApplication.setOverrideCursor(Qt.WaitCursor)
            self.mt.rebuildCuttingLines(False)

            # initial bbox ... lines
            vSource, aLines, aPolys = self.getLinesAndPolys(sample=False)
            aLines = [QgsGeometry.fromPolylineXY(line) for line in aLines]
            aPolys = [QgsGeometry.fromPolygonXY(poly) for poly in aPolys]

            # prepare POIs
            poiId = self.poiList.itemData(self.poiList.currentIndex())
            poiLayer = None
            poiFeatures = None
            nPoi = 0
            if poiId != 0:
                poiLayer = QgsProject.instance().mapLayer(poiId)
                mapcrs = self.canvas.mapSettings().destinationCrs()
                xMap2Poi = QgsCoordinateTransform(
                    mapcrs, poiLayer.crs(), QgsProject.instance()
                )
                # filter points (convex envelope)
                zon = QgsGeometry.fromMultiPolylineXY(self.mt.getLines())
                rq = QgsFeatureRequest().setFilterRect(
                    xMap2Poi.transform(zon.boundingBox())
                )
                rq.setFlags(QgsFeatureRequest.ExactIntersect)
                poiFeatures = poiLayer.getFeatures(rq)
                nPoi = (sum(1 for _ in poiFeatures) if poiFeatures is not None else 0)
                poiFeatures = poiLayer.getFeatures(rq)

            self.progressBar.setMaximum(
                10  # init
                + (len(aLines) if self.renderLines.isChecked() else 0)  # lines
                + (len(aPolys) if self.renderPolygons.isChecked() else 0)  # polygons
                + (9*10 + 2*len(aLines) if self.renderRidges.isChecked() else 0)  # ridges
                + nPoi
            )

            self.progress(10, 'init')

            # Compass ------------------------------------------------------------------------
            if not self.parallelView.isChecked() and self.renderCompass.isChecked():
                self.setAlert(self.tr("build compass"))
                compass = QgsVectorLayer(
                    "Point?crs={}".format(QgsProject.instance().crs().authid()),
                    "Compass",
                    "memory",
                )

                compass.startEditing()
                feats = []
                compass.dataProvider().addAttributes([
                    QgsField("demslicer_orientation", QVariant.String),
                    QgsField("demslicer_azimuth", QVariant.Double)
                ])
                compass.updateFields()
                for i, alpha in enumerate(range(round(self.mt.azimuthLeft), round(self.mt.azimuthRight))):
                    c = QgsPoint(
                        self.projBox.xMinimum() + (i*((self.projBox.xMaximum()-self.projBox.xMinimum())/(self.mt.azimuthRight-self.mt.azimuthLeft))),
                        self.yMin-self.base.value())
                    feature = QgsFeature()
                    feature.setAttributes(['H', alpha if alpha >= 0 else alpha+360])
                    feature.setGeometry(c)
                    feats.append(feature)

                # verticalement
                yMinNewZ = (self.yMin - self.mt.y('R')) / self.zFactor.value()
                # h = newZ si depth = horizon
                alphaBas = math.degrees(math.atan(yMinNewZ / self.mt.horizon))

                yMaxNewZ = (self.yMax - self.mt.y('R')) / self.zFactor.value()
                alphaHaut = math.degrees(math.atan(yMaxNewZ / self.mt.horizon))
                # self.log("{} {}".format(alphaBas, alphaHaut))

                for i, alpha in enumerate(range(round(alphaBas), round(alphaHaut))):
                    h = self.mt.y('R')+self.zFactor.value()*(self.mt.horizon*math.tan(math.radians(alpha)))
                    c = QgsPoint(
                        self.projBox.xMinimum(),
                        h
                    )
                    feature = QgsFeature()
                    feature.setAttributes(['V', alpha])
                    feature.setGeometry(c)
                    feats.append(feature)

                compass.dataProvider().addFeatures(feats)
                compass.commitChanges()

                # TODO : grid

                QgsProject.instance().addMapLayer(compass)
                compass.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/compass.qml"))

            # Source lines -------------------------------------------------------------------
            if self.renderSource.isChecked():
                QgsProject.instance().addMapLayer(vSource)
                vSource.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/source-points.qml"))

            # Line Slices --------------------------------------------------------------------
            if self.renderLines.isChecked():
                self.setAlert(self.tr("build lines"))
                projectedLineslayer = QgsVectorLayer(
                    "MultiLineString?crs={}".format(QgsProject.instance().crs().authid()),
                    "Lines",
                    "memory",
                )
                self.buildLayer(projectedLineslayer, aLines)
                QgsProject.instance().addMapLayer(projectedLineslayer)
                projectedLineslayer.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/lines.qml"))

            # Poly Slices --------------------------------------------------------------------
            if self.renderPolygons.isChecked():
                self.setAlert(self.tr("build polygons"))
                pLayer = QgsVectorLayer(
                    "Polygon?crs={}".format(QgsProject.instance().crs().authid()),
                    "Polygons",
                    "memory",
                )
                self.buildLayer(pLayer, aPolys)
                QgsProject.instance().addMapLayer(pLayer)
                if self.renderRidges.isChecked():
                    pLayer.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/polygons_ridges.qml"))
                else:
                    pLayer.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/polygons.qml"))

            # RIDGES --------------------------------------------------------------------
            if self.renderRidges.isChecked():
                self.setAlert(self.tr("build ridges"))
                try:
                    hLayer = QgsVectorLayer(
                        "LineString?crs={}".format(QgsProject.instance().crs().authid()),
                        "Ridges",
                        "memory",
                    )
                    hLayer.startEditing()
                    hLayer.dataProvider().addAttributes([
                        QgsField("demslicer_cutid", QVariant.Int),
                        QgsField("demslicer_gaz", QVariant.Int)
                    ])
                    hLayer.updateFields()

                    horizons = []
                    for lnum, linG in enumerate(aLines):
                        #  ("# lnum {}".format(lnum))
                        lin = QgsGeometry(linG)
                        # self.progress(2, 'ridge')

                        polyMax = None
                        for poly in aPolys[lnum+1:]:
                            # if not poly.isGeosValid():
                            #    print("! {} invalid".format(pnum+lnum+1))

                            if polyMax is None:
                                polyMax = poly.makeValid()
                            else:
                                polyMax = polyMax.combine(poly.makeValid())

                            lin = lin.difference(polyMax)

                        horizons.append(lin)

                    feats = []
                    for fid, g in enumerate(horizons[::-1]):
                        feature = QgsFeature(fid)
                        feature.setAttributes([fid, 0])
                        feature.setGeometry(g)
                        feats.append(feature)

                    hLayer.dataProvider().addFeatures(feats)
                    hLayer.commitChanges()

                    # processing pour poursuivre...
                    try:
                        # prolonger les lignes
                        self.setAlert(self.tr("build ridges 2/10"))
                        extendlines = tools.run("native:extendlines", hLayer, params={'START_DISTANCE': 1, 'END_DISTANCE': 1})
                        tools.run("native:createspatialindex", extendlines, {})
                        self.progress(10, 'ridges')
                        # intersection -> points
                        self.setAlert(self.tr("build ridges 3/10"))
                        lineintersections = tools.run("native:lineintersections", hLayer, params={'INTERSECT': extendlines})
                        tools.run("native:createspatialindex", lineintersections, {})
                        self.progress(10, 'ridges')
                        # filtrer
                        self.setAlert(self.tr("build ridges 4/10"))
                        filtered = tools.run("qgis:extractbyexpression", lineintersections, {'EXPRESSION': ' "demslicer_cutid" <  "demslicer_cutid_2" '})
                        tools.run("native:createspatialindex", filtered, {})
                        self.progress(10, 'ridges')
                        # générer des verticales pour découpage (expression) -> segments
                        self.setAlert(self.tr("build ridges 5/10"))
                        verticals = tools.run(
                            "native:geometrybyexpression", filtered,
                            params={
                                'EXPRESSION': 'make_line( make_point(x($geometry), y($geometry)-1), make_point(x($geometry), y($geometry)+1))'
                            }
                        )
                        tools.run("native:createspatialindex", verticals, {})
                        self.progress(10, 'ridges')
                        # couper
                        self.setAlert(self.tr("build ridges 6/10"))
                        splitwithlines = tools.run("native:splitwithlines", hLayer, params={'LINES': verticals})
                        tools.run("native:createspatialindex", splitwithlines, {})
                        self.progress(10, 'ridges')
                        # exploser
                        self.setAlert(self.tr("build ridges 7/10"))
                        ridges = tools.run("native:explodelines", splitwithlines, {}, name="ridges")
                        tools.run("native:createspatialindex", ridges, {})
                        self.progress(10, 'ridges')

                        self.setAlert(self.tr("build ridges 8/10"))
                        ridges.startEditing()
                        for f in ridges.getFeatures():
                            gz = self.getGaz(f.geometry().centroid(), aPolys, f["demslicer_cutid"])
                            f["demslicer_gaz"] = gz
                            ridges.updateFeature(f)
                        self.progress(10, 'ridges')
                        ridges.commitChanges()

                        # collect (réduire le nb d'entités)
                        self.setAlert(self.tr("build ridges 10/10"))
                        ridges = tools.run(
                            "native:collect", ridges, {'FIELD': ['demslicer_cutid', 'demslicer_gaz']}, name="ridges"
                        )
                        tools.run("native:createspatialindex", ridges, {})
                        self.progress(10, 'ridges')

                        ridges.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/ridges.qml"))

                        QgsProject.instance().addMapLayer(ridges)

                    except Exception:
                        self.warning(self.tr("Error in processing step (is processing tools actived ?)"))

                except Exception:
                    self.warning(self.tr("Error in ridges construction."))

            # POI --------------------------------------------------------------------
            if poiLayer is not None:
                self.setAlert(self.tr("project POIs"))

                mapcrs = self.canvas.mapSettings().destinationCrs()
                xPoi2Map = QgsCoordinateTransform(
                    poiLayer.crs(), mapcrs, QgsProject.instance()
                )

                if (
                    poiLayer.wkbType() == QgsWkbTypes.Point
                    or poiLayer.wkbType() == QgsWkbTypes.PointZ
                    or poiLayer.wkbType() == QgsWkbTypes.MultiPoint
                    or poiLayer.wkbType() == QgsWkbTypes.MultiPointZ
                ):
                    feats = []
                    for feat in poiFeatures:
                        self.progress(1, 'poi')

                        poiPointXY = xPoi2Map.transform(
                            feat.geometry().asPoint().x(), feat.geometry().asPoint().y()
                        )
                        if self.mt.geomZone.contains(poiPointXY):
                            z = self.getElevation(
                                self.xMap2Raster.transform(poiPointXY.x(), poiPointXY.y())
                            )
                            depth = self.mt.pointXY('Y').distance(poiPointXY)
                            prof = self.getProf(depth)
                            newX, newY = self.getProjectionPointAlti(
                                QgsPointXY(poiPointXY.x(), poiPointXY.y()), z
                            )
                            if self.parallelView.isChecked():
                                azimuth = self.mt.azimuth('Y', 'M')
                            else:
                                azimuth = self.mt.pointXY('Y').azimuth(poiPointXY)
                            
                            pt = QgsGeometry.fromPointXY(QgsPointXY(newX, newY))
                            visi, _ = self.getVisibility(pt, aPolys, prof)
                            fet0 = QgsFeature()
                            fet0.setAttributes(
                                feat.attributes()
                                + [int(prof)+1, z, depth, visi, azimuth]
                            )
                            fet0.setGeometry(pt)
                            feats.append(fet0)

                    if len(feats) > 0:
                        layer = QgsVectorLayer(
                            "MultiPoint?crs={}".format(
                                QgsProject.instance().crs().authid()
                            ),
                            'projection - ' + poiLayer.name(),
                            "memory",
                        )
                        QgsProject.instance().addMapLayer(layer)
                        layer.startEditing()
                        layer.dataProvider().addAttributes(
                            poiLayer.dataProvider().fields().toList()
                            + [
                                QgsField("demslicer_cutid", QVariant.Int),
                                QgsField("demslicer_z", QVariant.Int),
                                QgsField("demslicer_dist", QVariant.Double, "double", 4, 1),
                                QgsField("demslicer_visi", QVariant.Int),
                                QgsField("demslicer_azimuth", QVariant.Double, "double", 4, 1),
                            ]
                        )
                        layer.dataProvider().setEncoding(poiLayer.dataProvider().encoding())
                        layer.updateFields()

                        layer.dataProvider().addFeatures(feats)
                        layer.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/poi.qml"))
                        layer.commitChanges()

                # Projeter des lignes ou polygones

                if (
                    poiLayer.wkbType() == QgsWkbTypes.LineString
                    or poiLayer.wkbType() == QgsWkbTypes.LineStringZ
                    or poiLayer.wkbType() == QgsWkbTypes.MultiLineString
                    or poiLayer.wkbType() == QgsWkbTypes.MultiLineStringZ
                    or poiLayer.wkbType() == QgsWkbTypes.Polygon
                    or poiLayer.wkbType() == QgsWkbTypes.PolygonZ
                    or poiLayer.wkbType() == QgsWkbTypes.MultiPolygon
                    or poiLayer.wkbType() == QgsWkbTypes.MultiPolygonZ
                ):
                    # couche "zone"
                    self.setAlert(self.tr("projection 1/10"))
                    lZone = QgsVectorLayer("Polygon?crs={}".format(QgsProject.instance().crs().authid()), "lzone", "memory")
                    lZone.startEditing()
                    fetZone = QgsFeature()
                    fetZone.setGeometry(self.mt.geomZone)
                    lZone.dataProvider().addFeature(fetZone)
                    lZone.commitChanges()
                    # QgsProject.instance().addMapLayer(lZone)

                    # couche lignes source
                    lCut = QgsVectorLayer("MultiLineString?crs={}".format(QgsProject.instance().crs().authid()), "lcut", "memory")
                    lCut.startEditing()
                    for polyline in self.mt.getLines():
                        f = QgsFeature()
                        f.setGeometry(QgsGeometry.fromPolylineXY(polyline))
                        lCut.dataProvider().addFeature(f)
                    lCut.commitChanges()
                    # QgsProject.instance().addMapLayer(lCut)

                    # cliper (emprise zone)
                    self.setAlert(self.tr("projection 2/10"))
                    poiZLayer = tools.run("native:clip", poiLayer, {'OVERLAY': lZone})

                    self.setAlert(self.tr("projection 3/10"))
                    poiZLayer = tools.run("native:reprojectlayer", poiZLayer,
                        {'TARGET_CRS': QgsCoordinateReferenceSystem("{}".format(QgsProject.instance().crs().authid()))}
                    )

                    # découper couche d'habillage par les lignes sources
                    self.setAlert(self.tr("projection 4/10"))
                    poiZLayer = tools.run("native:splitwithlines", poiZLayer, {'LINES': lCut})
                    self.setAlert(self.tr("projection 5/10"))
                    poiZLayer = tools.run("native:multiparttosingleparts", poiZLayer)
                    poiZLayer.dataProvider().addAttributes([
                            QgsField("demslicer_visi", QVariant.Int),
                            QgsField("demslicer_cutid", QVariant.Int),
                            # QgsField("demslicer_log", QVariant.String)
                        ]
                    )
                    poiZLayer.updateFields()

                    # QgsProject.instance().addMapLayer(poiZLayer)

                    # affecter une profondeur à chaque entité
                    self.setAlert(self.tr("projection 6/10"))
                    poiZLayer.startEditing()
                    for feat in poiZLayer.getFeatures():
                        depth = self.mt.pointXY('Y').distance(feat.geometry().centroid().asPoint())
                        feat['demslicer_cutid'] = int(self.getProf(depth))+1
                        poiZLayer.updateFeature(feat)
                    poiZLayer.commitChanges()

                    # récupérer alti (valeur m)
                    self.setAlert(self.tr("projection 7/10"))
                    poiZLayer = tools.run("native:setmfromraster", poiZLayer, {'RASTER': self.mntLayer, 'BAND': 1, 'NODATA': 0, 'SCALE': 1})
                    # QgsProject.instance().addMapLayer(poiZLayer)

                    self.setAlert(self.tr("projection 8/10"))
                    finalType = QgsWkbTypes.MultiLineString
                    gType = "MultiLineString"
                    style = "ornementation-line.qml"
                    # projeter (nouvelle couche liée à la manipulation des vertices)
                    if (
                        poiLayer.wkbType() == QgsWkbTypes.Polygon
                        or poiLayer.wkbType() == QgsWkbTypes.PolygonZ
                        or poiLayer.wkbType() == QgsWkbTypes.MultiPolygon
                        or poiLayer.wkbType() == QgsWkbTypes.MultiPolygonZ
                    ) :
                        finalType = QgsWkbTypes.MultiPolygon
                        style = "ornementation-polygon.qml"

                    projectedLayer = QgsVectorLayer(
                        "{}?crs={}".format(gType, QgsProject.instance().crs().authid()),
                        'projection - ' + poiLayer.name(),
                        "memory"
                    )

                    self.setAlert(self.tr("projection 9/10"))
                    projectedLayer.startEditing()
                    projectedLayer.dataProvider().addAttributes(poiZLayer.fields())
                    feats = []
                    for f in poiZLayer.getFeatures():
                        self.progress(1, 'poi')

                        newF = QgsFeature()
                        newG = None
                        for part in f.geometry().constParts():
                            newPart = QgsLineString()
                            vtx0 = None
                            for vertexQgsPoint in part.vertices():
                                newX, newY = self.getProjectionPointAlti(
                                    QgsPointXY(vertexQgsPoint.x(), vertexQgsPoint.y()),
                                    vertexQgsPoint.m()
                                )
                                if vtx0 is None:
                                    vtx0 = QgsPoint(newX, newY)
                                vtx = QgsPoint(newX, newY)
                                newPart.addVertex(vtx)

                            if gType == "MultiPolygon" and vtx0 is not None:
                                newPart.addVertex(vtx0)

                            if newG is None:
                                newG = QgsGeometry(newPart)
                                newG.convertToMultiType()
                            else:
                                r = newG.addPart(newPart)
                                if r != 0:
                                    pass

                        if gType == "MultiLineString":
                            newF.setGeometry(newG)
                        else:
                            newF.setGeometry(newG.convertToType(QgsWkbTypes.MultiPolygon))

                        newF.setAttributes(f.attributes())
                        feats.append(newF)

                    projectedLayer.dataProvider().addFeatures(feats)
                    projectedLayer.commitChanges()

                    # découper à nouveau (par les profils projetés)
                    # projectedLayer = tools.run("native:splitwithlines", projectedLayer, {'LINES': projectedLineslayer})

                    # calculer visibilité
                    self.setAlert(self.tr("projection 10/10"))
                    projectedLayer.startEditing()
                    for feat in projectedLayer.getFeatures():
                        visi, _ = self.getVisibility(feat.geometry().centroid(), aPolys, feat['demslicer_cutid'])
                        feat['demslicer_visi'] = visi
                        # feat['demslicer_log'] = log
                        projectedLayer.updateFeature(feat)
                    projectedLayer.commitChanges()

                    if finalType == QgsWkbTypes.MultiPolygon:
                        projectedLayer = tools.run("qgis:linestopolygons", projectedLayer, {}, name='projection - ' + poiLayer.name())

                    # ajouter au projet
                    QgsProject.instance().addMapLayer(projectedLayer)
                    projectedLayer.loadNamedStyle(str(DIR_PLUGIN_ROOT / "resources/{}".format(style)))

        finally:
            self.setAlert("")
            self.progressBar.setValue(0)
            QgsApplication.restoreOverrideCursor()

    def build(self):
        """
        Build/Draw the final skylines
        """
        self.info("Build")
        self.buildSlices()

    def on_btnStart_toggled(self, checked):
        self.started = checked
        if checked:
            if self.start():
                self.btnStart.setText("Cancel")
            else:
                self.btnStart.toggle()
        else:
            self.btnStart.setText("Start")
            self.cancel()

    def on_lineCount_valueChanged(self, v):
        self.mt.updateRubberGeom()
        self.lineCount.setSingleStep(min(max(int(int(v)/10), 1), 10))

    def on_zShift_valueChanged(self, v):
        self.mt.updateRubberGeom()
        self.zShift.setSingleStep(min(max(int(int(v)/10), 1), 100))

    def on_xStep_valueChanged(self, v):
        self.mt.updateRubberGeom()
        self.xStep.setSingleStep(min(max(int(int(v)/10), 1), 100))

    def on_base_valueChanged(self, v):
        self.mt.updateRubberGeom()
        self.base.setSingleStep(min(max(int(int(v)/10), 1), 100))

    def on_elevation_valueChanged(self, v):
        self.mt.updateRubberGeom()
        self.elevation.setSingleStep(min(max(int(int(v)/10), 10), 100))

    def on_zFactor_valueChanged(self, v):
        self.mt.updateRubberGeom()

    def on_parallelView_stateChanged(self, v):
        self.renderCompass.setEnabled(not self.parallelView.isChecked())
        self.zShift.setEnabled(self.parallelView.isChecked())
        self.mt.updateRubberGeom()

    def on_btnBuild_released(self):
        self.build()

    def on_reset_released(self):
        self.mt.points['X'].pxy = None
        self.start()

    def on_rasterList_currentIndexChanged(self, i):
        id = self.rasterList.itemData(self.rasterList.currentIndex())
        self.mntLayer = QgsProject.instance().mapLayer(id)

        try:
            mapcrs = self.canvas.mapSettings().destinationCrs()
            self.xMap2Raster = QgsCoordinateTransform(
                mapcrs, self.mntLayer.crs(), QgsProject.instance()
            )
        except Exception:
            pass

    def on_btnHelp_released(self):
        tools.showPluginHelp(filename="help/index")

    def on_btnSave_released(self):
        fileName, _ = QFileDialog.getSaveFileName(
            self, self.tr("Save parameters"), "", self.tr("Ini files (*.ini)")
        )
        if fileName:
            s = QSettings(fileName, QSettings.IniFormat)
            for p in self.mt.ALL_POINTS:
                s.setValue("dem_slicer/{}x".format(p), self.mt.x(p))
                s.setValue("dem_slicer/{}y".format(p), self.mt.y(p))

            rId = self.rasterList.itemData(self.rasterList.currentIndex())
            s.setValue("dem_slicer/demLayerId", rId)
            poiId = self.poiList.itemData(self.poiList.currentIndex())
            s.setValue("dem_slicer/decoLayerId", poiId)
            s.setValue("dem_slicer/lineCount", self.lineCount.value())
            s.setValue("dem_slicer/elevation", self.elevation.value())
            s.setValue("dem_slicer/base", self.base.value())
            s.setValue(
                "dem_slicer/parallelView",
                "true" if self.parallelView.isChecked() else "false",
            )
            s.setValue("dem_slicer/xStep", self.xStep.value())
            s.setValue("dem_slicer/zShift", self.zShift.value())
            s.setValue("dem_slicer/zFactor", self.zFactor.value())
            s.setValue("dem_slicer/renderLines", self.renderLines.isChecked())
            s.setValue("dem_slicer/renderPolygons", self.renderPolygons.isChecked())
            s.setValue("dem_slicer/renderRidges", self.renderRidges.isChecked())
            s.setValue("dem_slicer/renderCompass", self.renderCompass.isChecked())
            s.setValue("dem_slicer/renderSource", self.renderSource.isChecked())

            s.sync()

    def on_btnLoad_released(self):
        fileName, _ = QFileDialog.getOpenFileName(
            self, self.tr("Load parameters"), "", self.tr("Ini files (*.ini)")
        )
        if fileName:
            s = QSettings(fileName, QSettings.IniFormat)
            for p in self.mt.ALL_POINTS:
                self.mt.points[p].setXY(
                    float(s.value("dem_slicer/{}x".format(p))),
                    float(s.value("dem_slicer/{}y".format(p))))

            self.rasterList.setCurrentIndex(0)
            for i in range(self.rasterList.count()):
                if self.rasterList.itemData(i) == s.value("dem_slicer/demLayerId"):
                    self.rasterList.setCurrentIndex(i)

            self.poiList.setCurrentIndex(0)
            for i in range(self.poiList.count()):
                if self.poiList.itemData(i) == s.value("dem_slicer/decoLayerId"):
                    self.poiList.setCurrentIndex(i)

            self.lineCount.setValue(int(s.value("dem_slicer/lineCount")))
            try:
                self.elevation.setValue(int(s.value("dem_slicer/elevation")))
            except Exception:
                self.elevation.setValue(0)
            try:
                self.base.setValue(int(s.value("dem_slicer/base")))
            except Exception:
                self.base.setValue(0)

            self.parallelView.setChecked(s.value("dem_slicer/parallelView") == "true")
            self.xStep.setValue(float(s.value("dem_slicer/xStep")))
            self.zShift.setValue(float(s.value("dem_slicer/zShift")))
            self.zFactor.setValue(float(s.value("dem_slicer/zFactor")))
            self.renderLines.setChecked(
                True if s.value("dem_slicer/renderLines") == 'true' else False
            )
            self.renderPolygons.setChecked(
                True if s.value("dem_slicer/renderPolygons") == 'true' else False
            )
            self.renderRidges.setChecked(
                True if s.value("dem_slicer/renderRidges") == 'true' else False
            )
            self.renderCompass.setChecked(
                True if s.value("dem_slicer/renderCompass") == 'true' else False
            )
            self.renderSource.setChecked(
                True if s.value("dem_slicer/renderSource") == 'true' else False
            )

            for p in self.mt.ALL_POINTS:
                self.mt.initpos[p] = QgsPointXY(self.mt.points[p].pxy)

            self.mt.updateRubberGeom()

    def updateZ(self, pt):
        self.altY = self.getElevation(self.xMap2Raster.transform(pt))
        try:
            self.labelElevation.setText(
                "Obs. : {} m   +".format(
                    round(self.altY, int(2 - math.log10(self.altY)))
                )
            )
        except Exception:
            self.labelElevation.setText("Obs. : {} m   +".format(self.altY))


class Point():
    """
    Outils de manipulation de l'emprise et affichage d'un échantillon

            mode ortho             non ortho

        A------H-------B         _----H----_
        |              |      A2              B2
        |  S   X       L       \   S  X      L2
        |              |        \           /                  ~~~~~~~~~~~~~~~~
        D------M-------C         \ _--M--_ /        ->         ~~~~~~~~~~~~~~~~
        \      |      /          D2   |   C2                   R ~~~~~~~~~~~~~~
          \    d0   /              \  d0 /
            \  |  /                 \ | /
               Y                      Y

    Les manipulations possibles :
    - déplacement de l'observateur (Y) [DC] reste constant, rotation autour de M (sommet)
    - déplacement de l'horizon AB (H) sur droite YH : M reste en place
    - déplacement 1er profil CD (M) sur droite YH: H reste en place
    - élargissement (L) largeur en mode ortho ou (L2) angle
    - rotation centre Y (B) ou (B2)

    les poignées visibles : Y (obs), H, M, L, L2, B, B2, et R

    """

    def __init__(self, pt=None):
        self.pxy = pt  # QgsPointXY

    def setXY(self, x, y=None):
        if isinstance(x, QgsPointXY):
            self.pxy = QgsPointXY(x)
        else:
            self.pxy = QgsPointXY(x, y)

    def x(self):
        return self.pxy.x()

    def y(self):
        return self.pxy.y()

    def isOk(self):
        return self.pxy is not None

    def distance(self, pt):
        return self.pxy.distance(pt.pxy)

    def azimuth(self, pt):
        return self.pxy.azimuth(pt.pxy)


def snap_to_line(A, B, C):
    # snap a point to a 2d line
    # parameters:
    #   A,B: the endpoints of the line
    #   C: the point we want to snap to the line AB
    # all parameters must be a tuple/list of float numbers

    Ax, Ay = A.x(), A.y()
    Bx, By = B.x(), B.y()
    Cx, Cy = C.x(), C.y()

    eps = 0.0000001
    if abs(Ax-Bx) < eps and abs(Ay-By) < eps:
        return QgsPointXY(Ax, Ay)

    dx = Bx-Ax
    dy = By-Ay
    d2 = dx*dx + dy*dy
    t = ((Cx-Ax)*dx + (Cy-Ay)*dy) / d2
    if t <= 0:
        return QgsPointXY(Ax, Ay)
    if t >= 1:
        return QgsPointXY(Bx, By)
    return QgsPointXY(dx*t + Ax, dy*t + Ay)


class MapTool(QgsMapTool):
    MODE_NONE = 0
    HANDLES = ('Y', 'H', 'M', 'L', 'L2', 'B', 'B2', 'peak', 'R')
    HANDLES_1 = ('Y', 'H', 'M', 'L', 'B', 'peak', 'R')
    HANDLES_2 = ('Y', 'H', 'M', 'B2', 'L2', 'peak', 'R')
    ALL_POINTS = ('X', 'A', 'A2', 'B', 'B2', 'C', 'C2', 'D', 'D2',
                  'L', 'Y', 'L2', 'M', 'R', 'peak', 'H')
    ALL_POINTS_R = ('X', 'A', 'A2', 'B', 'B2', 'C', 'C2', 'D', 'D2',
                    'L', 'Y', 'L2', 'M', 'peak', 'H')

    def getRubber(self, typ, color=Qt.red, w=6):
        r = QgsRubberBand(self.canvas, typ)
        r.setStrokeColor(color)
        r.setWidth(w)
        return r

    def geomPolyline(self, pList):
        return QgsGeometry.fromPolylineXY(
            [(self.points[p].pxy if isinstance(p, str) else p) for p in pList]
        )

    def geomPolygon(self, pList):
        return QgsGeometry.fromPolygonXY(
            [[(self.points[p].pxy if isinstance(p, str) else p) for p in pList]]
        )

    def geomPoint(self, p):
        if isinstance(p, QgsPointXY):
            return QgsGeometry.fromPointXY(p)
        if isinstance(p, str):
            return QgsGeometry.fromPointXY(self.points[p].pxy)
        return None

    def pointXY(self, p):
        if isinstance(p, QgsPointXY):
            return p
        else:
            return self.points[p].pxy

    def x(self, p):
        return self.points[p].x()

    def y(self, p):
        return self.points[p].y()

    def dx(self, p1, p2):
        return self.points[p2].x()-self.points[p1].x()

    def dy(self, p1, p2):
        return self.points[p2].y()-self.points[p1].y()

    def milieu(self, p1, p2):
        p1 = self.pointXY(p1)
        p2 = self.pointXY(p2)
        return QgsPointXY((p1.x()+p2.x())/2, (p1.y()+p2.y())/2)

    def distance(self, p1, p2):
        p1 = self.pointXY(p1)
        p2 = self.pointXY(p2)
        return p1.distance(p2)

    def azimuth(self, p1, p2):
        return self.points[p1].azimuth(self.points[p2])

    def angle(self, p1, c, p2):
        p1 = self.pointXY(p1)
        c = self.pointXY(c)
        p2 = self.pointXY(p2)
        az1 = c.azimuth(p1)
        az2 = c.azimuth(p2)
        if az1 > az2:
            az2 = az2 + 360
        return az2-az1

    def rotatePoint(self, p, delta, c):
        p = self.geomPoint(p)
        c = self.pointXY(c)

        p.rotate(delta, c)
        return p.asPoint()

    def __init__(self, widget):
        QgsMapTool.__init__(self, widget.canvas)
        self.widget = widget
        self.canvas = widget.canvas
        self.mode = self.MODE_NONE

        # clicked position
        self.p0 = None

        self.points = {}
        for p in self.ALL_POINTS:
            self.points[p] = Point()

        # rectangle vertices (handles)
        self.zoneDepth = None

        # Rubbers (éléments graphiques visibles) -----------
        # Ordre de la déclaration = ordre d'affichage

        self.rubbers = {}
        for p in ('horizon',):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.LineGeometry)

        for p in ('thumbnail', 'box'):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.PolygonGeometry)

        for p in ('first',):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.LineGeometry)

        for p in ('foc', 'lines'):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.LineGeometry)

        for p in ('peak', 'peakProj', 'B', 'H', 'L', 'Y', 'B2', 'L2', 'M', 'R'):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.PointGeometry)

        """ tests
        for p in ('A', 'B', 'C', 'D'):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.PointGeometry, color=Qt.green, w=2)
        for p in ('A2', 'C2', 'D2'):
            self.rubbers[p] = self.getRubber(QgsWkbTypes.PointGeometry, color=Qt.darkGray, w=2)"""

        # last line (blue)
        self.rubbers['horizon'].setStrokeColor(QColor(70, 100, 255, 200))
        self.rubbers['horizon'].setWidth(3)
        self.rubbers['first'].setStrokeColor(QColor(230, 170, 70, 200))
        self.rubbers['first'].setWidth(2)
        # thumbnails skylines - profil échantillon projeté
        self.rubbers['thumbnail'].setStrokeColor(QColor(200, 120, 70, 130))
        self.rubbers['thumbnail'].setWidth(1)
        self.rubbers['R'].setStrokeColor(QColor(120, 110, 130, 200))
        # peak projection
        self.rubbers['peakProj'].setStrokeColor(QColor(255, 239, 15, 200))
        self.rubbers['peakProj'].setWidth(4)
        # rectangle or cone
        self.rubbers['box'].setStrokeColor(QColor(70, 100, 255, 200))
        self.rubbers['box'].setWidth(3)
        # view angle
        self.rubbers['foc'].setStrokeColor(Qt.blue)
        self.rubbers['foc'].setWidth(1)
        # cutting lines
        self.rubbers['lines'].setStrokeColor(QColor(40, 180, 30, 200))
        self.rubbers['lines'].setWidth(2)
        # ROTATE node (eye)
        self.rubbers['Y'].setStrokeColor(Qt.blue)
        # PEAK
        self.rubbers['peak'].setStrokeColor(QColor(255, 239, 15, 200))

        # azimuth left and right
        self.azimuthLeft = -45
        self.azimuthRight = 45

    def hide(self):
        for rb in self.rubbers.values():
            rb.reset()

    def rebuildCuttingLines(self, sample=True):
        if sample:
            dx = self.finalWidth / 15.0
        else:
            dx = self.widget.xStep.value()

        if self.widget.parallelView.isChecked():
            leftEdge = (
                self.geomPolyline(['A', 'D'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )
            rightEdge = (
                self.geomPolyline(['B', 'C'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )
            polyline = list(zip(leftEdge, rightEdge))

            self.cuttingLines = QgsGeometry.fromMultiPolylineXY(polyline).densifyByDistance(dx).asMultiPolyline()
        else:
            dAlpha = self.angle('D2', 'Y', 'C2') / (self.finalWidth / dx)

            leftEdge = (
                self.geomPolyline(['A2', 'D2'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )

            polyline = []
            for p in leftEdge:
                line = []
                for nx in range(2 + int((self.finalWidth / dx))):
                    g = self.geomPoint(p)
                    g.rotate(nx * dAlpha, self.pointXY('Y'))
                    line.append(g.asPoint())

                polyline.append(line)

            self.cuttingLines = polyline

    def updateRubberGeom(self):
        if not self.points['A'].isOk():
            return

        self.hide()

        self.azimuthLeft = self.azimuth('Y', 'D2')
        self.azimuthRight = self.azimuth('Y', 'C2')
        if self.azimuthRight < self.azimuthLeft:
            self.azimuthRight = self.azimuthRight + 360

        self.segCD = self.geomPolyline(['C', 'D'])
        self.segAD = self.geomPolyline(['A', 'D'])

        self.d0 = self.distance('M', 'Y')
        self.horizon = self.distance('H', 'Y')
        self.widget.updateZ(self.points['Y'].pxy)

        self.rubbers['R'].setToGeometry(self.geomPoint('R'))

        # cutting lines
        polylineX = []
        if self.widget.parallelView.isChecked():
            self.rubbers['foc'].setToGeometry(self.geomPolyline(['D', 'Y', 'C']))

            leftEdge = (
                self.geomPolyline(['A', 'D'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )
            rightEdge = (
                self.geomPolyline(['B', 'C'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )
            polyline = list(zip(leftEdge, rightEdge))

            backSide = (
                self.geomPolyline(['A', 'B'])
                .densifyByDistance(self.widget.xStep.value())
                .asPolyline()
            )
            frontSide = (
                self.geomPolyline(['D', 'C'])
                .densifyByDistance(self.widget.xStep.value())
                .asPolyline()
            )
            polylineX = list(zip(frontSide[:5], backSide[:5]))

            self.finalWidth = self.distance('A', 'B')
            self.zoneDepth = self.distance('A', 'D')

        else:
            self.rubbers['foc'].setToGeometry(self.geomPolyline(['D2', 'Y', 'C2']))

            alpha = self.angle('D2', 'Y', 'C2')
            dAlpha = alpha / 12

            leftEdge = (
                self.geomPolyline(['A2', 'D2'])
                .densifyByCount(self.widget.lineCount.value() - 2)
                .asPolyline()
            )
            rightEdge = (
                self.geomPolyline(['B2', 'C2'])
                .asPolyline()
            )
            polyline = []
            for p in leftEdge:
                line = []
                for nx in range(1 + 12):
                    g = self.geomPoint(p)
                    g.rotate(nx * dAlpha, self.pointXY('Y'))
                    line.append(g.asPoint())

                polyline.append(line)

            self.finalWidth = self.geomPolyline(polyline[int(len(polyline) / 2)]).length()
            self.zoneDepth = self.distance('A2', 'D2')

            dAlphaDetail = alpha / (self.finalWidth / self.widget.xStep.value())

            aPrim = self.geomPoint('A2')
            dPrim = self.geomPoint('D2')
            for _ in range(5):
                seg = self.geomPolyline([aPrim.asPoint(), dPrim.asPoint()]).asPolyline()
                polylineX.append(seg)
                aPrim.rotate(dAlphaDetail, self.pointXY('Y'))
                dPrim.rotate(dAlphaDetail, self.pointXY('Y'))

        self.rebuildCuttingLines(True)

        self.rubbers['lines'].setToGeometry(
            QgsGeometry.fromMultiPolylineXY(
                polylineX
                + polyline[0:50]
                + polyline[:: max(1, 1 + int(len(polyline) / 50))][-50:]
            )
        )

        # final result

        # box
        if self.widget.parallelView.isChecked():
            self.geomZone = self.geomPolygon(['D', 'A', 'B', 'C', 'D'])
            self.geomZone = self.geomZone.buffer(0, 1)
            self.rubbers['box'].setToGeometry(self.geomZone)
            for p in self.HANDLES_1:
                try:
                    self.rubbers[p].setToGeometry(self.geomPoint(p))
                except Exception:
                    pass
        else:
            self.geomZone = QgsGeometry.fromMultiPolygonXY(
                [[
                    leftEdge + polyline[0] + rightEdge[::-1] + polyline[-1][::-1]
                    + [leftEdge[0]]
                ]]
            )
            self.geomZone = self.geomZone.buffer(0, 1)
            self.rubbers['box'].setToGeometry(self.geomZone)

            for p in self.HANDLES_2:
                try:
                    self.rubbers[p].setToGeometry(self.geomPoint(p))
                except Exception:
                    pass

        # peak
        if self.geomZone.contains(self.pointXY('peak')):
            proj = self.widget.getPeakGeom(self.pointXY('peak'))
            self.rubbers['peakProj'].setToGeometry(proj)

        # build projected base lines
        polyline = []
        nbLines = self.widget.lineCount.value()
        for _ in range(nbLines)[::-1]:
            p1 = QgsPointXY(self.x('R'), self.y('R'))
            p2 = QgsPointXY(self.x('R') + self.finalWidth, self.y('R'))
            polyline.append([p1, p2])

        self.skyLines = polyline

        # Echantillon
        try:
            first, horiz, thumbnailGeom = self.widget.getThumbnailGeom()
            self.rubbers['first'].setToGeometry(first)
            self.rubbers['horizon'].setToGeometry(horiz)
            self.rubbers['thumbnail'].setToGeometry(thumbnailGeom)
        except Exception as e:
            self.widget.log("Err ds {} ligne {} ".format(
                inspect.stack()[0][3], inspect.currentframe().f_back.f_lineno)
            )
            self.widget.log(e)
            raise

        # alert
        nbPoints = int(len(polyline) * (self.finalWidth / self.widget.xStep.value()))
        alert = ""
        if nbPoints > 100000:
            alert = alert + "Attention : {} points\n".format(nbPoints)

        self.widget.setAlert(alert)

    def getLines(self):
        return self.cuttingLines

    def getSampleLines(self):
        return (
            [self.cuttingLines[0], self.cuttingLines[1]]
            + self.cuttingLines[2:-1][:: max(1, 1 + round((len(self.cuttingLines) - 3) / 15))]
            + [self.cuttingLines[-1]]
        )

    def getSkylines(self):
        return self.skyLines

    def getSampleSkylines(self):
        return (
            [self.skyLines[0], self.skyLines[1]]
            + self.skyLines[2:-1][:: max(1, 1 + round((len(self.skyLines) - 3) / 15))]
            + [self.skyLines[-1]]
        )

    def newRubber(self):
        # self.widget.log('newRubber')
        if self.points['X'].isOk():
            self.updateRubberGeom()
            return

        # self.widget.log('  ...')

        # default parameters
        h = 2 * self.widget.canvas.extent().height() / 3 / 20
        self.widget.xStep.setValue(round(h / 2, int(2 - math.log10(h / 2))))

        # first bbox, according to current view
        h = self.canvas.extent().height() / 6
        c = self.canvas.extent().center()
        rubberExtent = QgsRectangle(
            QgsPointXY(c.x() - h, c.y() - h), QgsPointXY(c.x() + h, c.y() + h)
        )
        self.rotation = 0.0
        width = rubberExtent.xMaximum() - rubberExtent.xMinimum()
        height = rubberExtent.yMaximum() - rubberExtent.yMinimum()

        self.points['A'].setXY(rubberExtent.xMinimum(), rubberExtent.yMaximum())
        self.points['B'].setXY(rubberExtent.xMaximum(), rubberExtent.yMaximum())
        self.points['C'].setXY(rubberExtent.xMaximum(), rubberExtent.yMinimum())
        self.points['D'].setXY(rubberExtent.xMinimum(), rubberExtent.yMinimum())
        self.points['X'].setXY(self.milieu('A', 'C'))
        self.segCD = self.geomPolyline(['C', 'D'])
        self.segAD = self.geomPolyline(['A', 'D'])

        # handles H / L
        self.points['H'].setXY(self.milieu('A', 'B'))
        self.points['L'].setXY(self.milieu('B', 'C'))
        self.points['M'].setXY(self.milieu('C', 'D'))

        # eye (rotation)
        self.points['Y'].setXY(self.x('X'), self.y('X') - height)

        # peak
        self.points['peak'].setXY(self.milieu('A', 'C'))

        self.points['R'].setXY(self.x('L') + width/2, self.y('C'))

        # perspective handles
        a = self.angle('D', 'Y', 'H')
        self.points['B2'].setXY(self.rotatePoint('H', a, 'Y'))
        self.points['L2'].setXY(self.rotatePoint('X', a, 'Y'))
        self.points['A2'].setXY(self.rotatePoint('H', -a, 'Y'))
        self.points['C2'].setXY(self.rotatePoint('M', a, 'Y'))
        self.points['D2'].setXY(self.rotatePoint('M', -a, 'Y'))

        # Pos INIT
        self.initpos = {}
        for p in self.ALL_POINTS:
            try:
                self.initpos[p] = QgsPointXY(self.points[p].x(), self.points[p].y())
            except AttributeError:
                self.widget.log("points {} non renseigné".format(p))

        self.updateRubberGeom()

    def canvasPressEvent(self, event):
        DIST = 8
        x = event.pos().x()
        y = event.pos().y()
        self.p0 = self.canvas.getCoordinateTransform().toMapCoordinates(x, y)

        self.mode == self.MODE_NONE
        for p in self.HANDLES_1 if self.widget.parallelView.isChecked() else self.HANDLES_2:
            if self.p0.distance(self.pointXY(p)) / self.canvas.mapUnitsPerPixel() < DIST:
                self.mode = p
                # self.widget.log("mode "+p)
                return

        if self.geomZone.contains(self.p0):
            self.mode = 'box'
            return

        if self.rubbers['thumbnail'].asGeometry().convexHull().contains(self.p0):
            self.mode = 'R'
            return

    def move(self, pt, toMove, segOrPoint):
        if isinstance(segOrPoint, list):
            # cible projetée sur segment
            target = snap_to_line(
                self.initpos[segOrPoint[0]], self.initpos[segOrPoint[1]], self.initpos[toMove]
            )
        else:
            # point cible
            target = self.initpos[segOrPoint]

        # position initiale du point à déplacer
        p1_init = self.initpos[toMove]
        # distance à la cible (avant, après)
        d_old = p1_init.distance(target)
        d_new = pt.distance(target)
        # ratio
        dd = min(max(d_new / d_old, 0.005), 10)

        dx = dd * (p1_init.x()-target.x())
        dy = dd * (p1_init.y()-target.y())
        return (dd, (target.x()+dx)-p1_init.x(), (target.y()+dy)-p1_init.y())

    def deltaRotation(self, mousePXY, p, c):
        az_init = self.initpos[p].azimuth(self.pointXY(c))
        az_new = mousePXY.azimuth(self.pointXY(c))
        if az_init > az_new:
            az_new = az_new + 360
        return az_new - az_init

    def updateCD(self):
        # faire suivre D C A B
        a = self.angle('D2', 'Y', 'M')
        dYD = self.distance('Y', 'M') / math.cos(math.radians(a))

        a = self.azimuth('Y', 'A2')
        self.points['D'].setXY(
            self.x('Y')+math.sin(math.radians(a))*dYD,
            self.y('Y')+math.cos(math.radians(a))*dYD
        )
        a = self.azimuth('Y', 'B2')
        self.points['C'].setXY(
            self.x('Y')+math.sin(math.radians(a))*dYD,
            self.y('Y')+math.cos(math.radians(a))*dYD
        )

    def updateABCD(self):
        self.updateCD()
        self.points['A'].setXY(self.x('H')+self.dx('M', 'D'), self.y('H')+self.dy('M', 'D'))
        self.points['B'].setXY(self.x('H')+self.dx('M', 'C'), self.y('H')+self.dy('M', 'C'))
        self.points['X'].setXY(self.milieu('A', 'C'))
        self.points['L'].setXY(self.milieu('B', 'C'))

    def updateA2B2C2D2(self):
        a = self.angle('D', 'Y', 'H')
        self.points['A2'].setXY(self.rotatePoint('H', -a, 'Y'))
        self.points['B2'].setXY(self.rotatePoint('H', a, 'Y'))
        self.points['C2'].setXY(self.rotatePoint('M', a, 'Y'))
        self.points['D2'].setXY(self.rotatePoint('M', -a, 'Y'))
        self.points['L2'].setXY(self.rotatePoint('X', a, 'Y'))

    def canvasMoveEvent(self, event):
        if self.mode == self.MODE_NONE:
            return

        x = event.pos().x()
        y = event.pos().y()
        pt = self.canvas.getCoordinateTransform().toMapCoordinates(x, y)
        dx = pt.x() - self.p0.x()
        dy = pt.y() - self.p0.y()

        try:
            toMove = self.points[self.mode]
            xi, yi = toMove.x(), toMove.y()
        except Exception:
            xi, yi = self.p0.x(), self.p0.y()

        # pan mode
        if self.mode == 'box':
            # déplacer l'ensemble des points sauf R
            for p in self.ALL_POINTS_R:
                self.points[p].setXY(self.initpos[p].x()+dx, self.initpos[p].y()+dy)

        # result pan
        if self.mode == 'R':
            # déplacer seulement l'échantillon
            self.points[self.mode].setXY(
                self.initpos[self.mode].x() + dx,
                self.initpos[self.mode].y() + dy
            )

        # horizontal sizing
        if self.mode == 'L':
            # On déplace L
            # distance L <-> [A,D]
            _, dx2, dy2 = self.move(pt, 'L', ['A', 'D'])
            toMove.setXY(self.initpos[self.mode].x()+dx2, self.initpos[self.mode].y() + dy2)

            # contrainte
            if self.distance('A', 'B') < 2 * self.widget.xStep.value():
                self.points[self.mode].setXY(xi, yi)
                return

            # faire suivre A, B, D, C
            for p in ['B', 'C']:
                toMove = self.points[p]
                toMove.setXY(self.initpos[p].x()+dx2, self.initpos[p].y()+dy2)
            for p in ['A', 'D']:
                toMove = self.points[p]
                toMove.setXY(self.initpos[p].x()-dx2, self.initpos[p].y()-dy2)

            # A2 B2 C2 Y2
            self.updateA2B2C2D2()

        if self.mode == 'L2':
            center = 'Y'
            delta = self.deltaRotation(pt, self.mode, center)

            # déplacer la poignée (rotation)
            self.points['L2'].setXY(self.rotatePoint(self.initpos['L2'], delta, 'Y'))

            # contrainte
            if self.distance('A2', 'B2') < self.distance(self.initpos['A2'], self.initpos['B2']) and self.distance('L2', 'X') < 2 * self.widget.xStep.value():
                self.points[self.mode].setXY(xi, yi)
                return

            a = self.angle('M', 'Y', 'L2')
            # self.widget.log("a {} ".format(a))
            if (a >= 89):
                self.points[self.mode].setXY(xi, yi)
                return

            self.points['C2'].setXY(self.rotatePoint(self.initpos['C2'], delta, 'Y'))
            self.points['B2'].setXY(self.rotatePoint(self.initpos['B2'], delta, 'Y'))
            self.points['D2'].setXY(self.rotatePoint(self.initpos['D2'], -delta, 'Y'))
            self.points['A2'].setXY(self.rotatePoint(self.initpos['A2'], -delta, 'Y'))
            self.updateABCD()

        # horizon deplacement
        if self.mode == 'H':
            # On déplace H
            # distance H <-> [C,D]
            _, dx2, dy2 = self.move(pt, 'H', ['C', 'D'])
            toMove.setXY(self.initpos[self.mode].x()+dx2, self.initpos[self.mode].y() + dy2)

            # faire suivre A, B
            for p in ['A', 'B']:
                toMove = self.points[p]
                toMove.setXY(self.initpos[p].x()+dx2, self.initpos[p].y()+dy2)

            self.updateCD()
            self.updateA2B2C2D2()

        # first line deplacement
        if self.mode == 'M':
            # On déplace M le long de (Y,H)
            # distance M <-> [A,B]
            _, dx2, dy2 = self.move(pt, 'M', ['A', 'B'])
            toMove.setXY(self.initpos[self.mode].x()+dx2, self.initpos[self.mode].y() + dy2)

            # empecher de passer sur seg [MH]
            if self.distance('H', 'M') > self.distance('H', 'Y'):
                self.points[self.mode].setXY(xi, yi)
                return

            self.updateABCD()
            self.updateA2B2C2D2()

        if self.mode == 'Y':  # CD fixe en largeur
            center = 'M'
            self.points[self.mode].setXY(pt.x(), pt.y())

            # contrainte
            dYM = self.distance('Y', 'M')
            if dYM < self.canvas.mapUnitsPerPixel():
                self.points[self.mode].setXY(xi, yi)
                return

            # rotation H, C, D
            az_init = self.initpos[self.mode].azimuth(self.pointXY(center))
            az_new = pt.azimuth(self.pointXY(center))
            if az_init > az_new:
                az_new = az_new + 360
            theta = az_new - az_init

            for p in ['H', 'C', 'D']:
                self.points[p].setXY(self.rotatePoint(self.initpos[p], theta, center))

            self.updateA2B2C2D2()
            self.updateABCD()

        if self.mode == 'B' or self.mode == 'B2':
            center = 'Y'
            az_init = self.initpos[self.mode].azimuth(self.pointXY(center))
            az_new = pt.azimuth(self.pointXY(center))
            if az_init > az_new:
                az_new = az_new + 360
            theta = az_new - az_init

            for p in list(set(self.ALL_POINTS_R)-set(['Y'])):
                A = self.geomPoint(self.initpos[p])
                A.rotate(theta, self.pointXY(center))
                self.points[p].setXY(A.asPoint().x(), A.asPoint().y())

        if self.mode == 'peak':
            self.points[self.mode].setXY(pt)

        self.points['X'].setXY(self.milieu('A', 'C'))
        self.points['L'].setXY(self.milieu('B', 'C'))

        self.updateRubberGeom()

    def canvasReleaseEvent(self, _):
        # réinitialisation des positions initiales
        for p in self.ALL_POINTS:
            self.initpos[p] = QgsPointXY(self.points[p].pxy)

        self.mode = self.MODE_NONE

        # traces
        # self.widget.log("az:[{}..{}]".format(self.azimuthLeft, self.azimuthRight))

    def activate(self):
        pass

    def deactivate(self):
        self.hide()

    def isZoomTool(self):
        return False

    def isTransient(self):
        return False

    def isEditTool(self):
        return True
