# -*- coding: utf-8 -*-
"""
/***************************************************************************
 DepthReaderOCRDialog
                                 A QGIS plugin
 Automatically extracts depth values from nautical charts using OCR.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-06-21
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Elivaldo Rocha
        email                : carvalhovaldo09@gmail.com
 ***************************************************************************/

/***************************************************************************
 * *
 * This program is free software; you can redistribute it and/or modify  *
 * it under the terms of the GNU General Public License as published by  *
 * the Free Software Foundation; either version 2 of the License, or     *
 * (at your option) any later version.                                   *
 * *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QFileDialog, QCheckBox, QVBoxLayout
from qgis.PyQt.QtCore import Qt

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'deep_reader_ocr_dialog_base.ui'))


class DepthReaderOCRDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(DepthReaderOCRDialog, self).__init__(parent)
        self.setupUi(self)

        
        # Adiciona o checkbox de modo de operação
        self.setup_ocr_checkbox()

        # Conectar botões de procurar
        if hasattr(self, 'btnBrowseDebugDir'):
            self.btnBrowseDebugDir.clicked.connect(self.browse_debug_directory)
        if hasattr(self, 'btnBrowseCSV'):
            self.btnBrowseCSV.clicked.connect(self.browse_csv_file)

        # Configurar informações e valores padrão
        self._set_default_values()
        
        # Força a atualização da visibilidade da aba 'Avançado' no início
        self.on_ocr_mode_changed(self.chkUseOCR.isChecked())
        
        # Define a aba "Geral" como a aba inicial padrão
        self.tabWidget.setCurrentIndex(1)

    def _set_default_values(self):
        """Configura os valores padrão para todos os campos da UI."""
        default_user_dir = os.path.expanduser("~")

        # Aba Geral
        if hasattr(self, 'leCSVPath'):
            self.leCSVPath.setText(os.path.join(default_user_dir, "batimetria.csv"))

        # Aba Avançado
        if hasattr(self, 'leDebugDir'):
            self.leDebugDir.setText(os.path.join(default_user_dir, "depth_reader_debug"))
        if hasattr(self, 'cbClipSize'):
            self.cbClipSize.clear()
            self.cbClipSize.addItems(["16", "32", "48", "64", "80", "96"])
            self.cbClipSize.setCurrentText("96")
            self.cbClipSize.setEnabled(True)
        
        # Define os valores padrão para os novos campos de configuração do OCR
        if hasattr(self, 'leRotations'):
            self.leRotations.setText("-90, -45, 0, 45, 90, 180, 270")
        if hasattr(self, 'chkClahe'):
            self.chkClahe.setChecked(True)
        if hasattr(self, 'chkGaussian'):
            self.chkGaussian.setChecked(True)
        if hasattr(self, 'chkMean'):
            self.chkMean.setChecked(True)
        
        # Aba Sobre
        if hasattr(self, 'tbInfo'):
            self.tbInfo.setText(self.get_info_text())
            self.tbInfo.setOpenExternalLinks(True)

    def setup_ocr_checkbox(self):
        """
        Adiciona o checkbox de OCR à interface de forma robusta.
        Procura por um layout chamado 'ocr_checkbox_placeholder_layout' no arquivo .ui.
        """
        try:
            # Cria o checkbox
            self.chkUseOCR = QCheckBox("🤖 Usar visão computacional (OCR)")
            self.chkUseOCR.setChecked(True)  # Marcado por padrão
            self.chkUseOCR.setToolTip(
                "✅ Marcado: Usa OCR automático para detectar profundidades\n"
                "❌ Desmarcado: Permite entrada manual de valores"
            )
            
            # Conecta o sinal 'toggled' para atualizar a interface
            self.chkUseOCR.toggled.connect(self.on_ocr_mode_changed)
            
            # Tenta encontrar o layout placeholder (Método 1)
            placeholder_layout = self.findChild(QVBoxLayout, "ocr_checkbox_placeholder_layout")
            
            if placeholder_layout:
                # Se encontrou o layout, adiciona o checkbox a ele. Esta é a forma ideal.
                placeholder_layout.addWidget(self.chkUseOCR)
                print("✅ Checkbox OCR adicionado ao layout de placeholder.")
            else:
                # --- INÍCIO DA CORREÇÃO (MÉTODO 2) ---
                # Se não encontrou o placeholder, usa o Plano B corrigido.
                print("⚠️  Aviso: Layout 'ocr_checkbox_placeholder_layout' não foi encontrado.")
                print("   Usando o Método 2 para posicionar o checkbox na aba 'Geral'.")
                
                geral_tab_widget = None
                # Procura a aba pelo nome "Geral" para ser mais robusto
                if hasattr(self, 'tabWidget'):
                    for i in range(self.tabWidget.count()):
                        if self.tabWidget.tabText(i) == "Geral":
                            geral_tab_widget = self.tabWidget.widget(i)
                            break

                if geral_tab_widget:
                    # Verifica se a aba "Geral" já tem um layout. Se não, cria um.
                    if not geral_tab_widget.layout():
                        print("   Criando um novo layout vertical para a aba 'Geral'.")
                        new_layout = QVBoxLayout()
                        geral_tab_widget.setLayout(new_layout)
                    
                    # Adiciona o checkbox ao layout da aba "Geral"
                    geral_tab_widget.layout().insertWidget(0, self.chkUseOCR) # Insere no topo
                    print("✅ Checkbox OCR adicionado com sucesso à aba 'Geral'.")
                else:
                    print("❌ ERRO CRÍTICO: Não foi possível encontrar a aba 'Geral'.")
                # --- FIM DA CORREÇÃO ---

        except Exception as e:
            print(f"⚠️ Erro crítico ao criar checkbox OCR: {e}")
            # Cria um atributo vazio para evitar que o resto do plugin falhe
            self.chkUseOCR = QCheckBox()
            self.chkUseOCR.setVisible(False)
            self.chkUseOCR.setParent(self)

    def on_ocr_mode_changed(self, checked):
        """Callback chamado quando o modo OCR é alterado."""
        # Habilita/desabilita a aba "Avançado" inteira
        try:
            # O índice 1 corresponde à segunda aba ("Avançado") na ordem Sobre, Geral, Avançado.
            # Uma forma mais robusta seria procurar pelo nome, mas o índice funciona.
            # Assumindo a ordem: 0=Sobre, 1=Geral, 2=Avançado
            advanced_tab_index = -1
            if hasattr(self, 'tabWidget'):
                for i in range(self.tabWidget.count()):
                    if self.tabWidget.tabText(i) == "Avançado":
                        advanced_tab_index = i
                        break

            if advanced_tab_index != -1:
                self.tabWidget.setTabEnabled(advanced_tab_index, checked)
                if checked:
                    print("✅ Modo OCR ativado. Aba de configurações avançadas habilitada.")
                else:
                    print("✋ Modo manual ativado. Aba de configurações avançadas desabilitada.")
        except Exception as e:
            print(f"⚠️ Erro ao alterar visibilidade da aba: {e}")

    def get_info_text(self):
        """Retorna o texto de informações da caixa de diálogo."""
        return (
            "<h3>Depth Reader OCR</h3>"
            "Este plugin permite extrair manual e automaticamente valores de profundidade de cartas náuticas raster (GeoTIFF) da Marinha do Brasil usando OCR.<br><br>"
            
            "<b>MODOS DE OPERAÇÃO:</b><br>"
            "<ul><li><b>Visão Computacional (OCR):</b> Detecta automaticamente as profundidades.</li>"
            "<li><b>Entrada Manual:</b> Permite que você digite os valores manualmente.</li></ul>"
            
            "<b>Onde Baixar Cartas Náuticas Raster?</b><br>"
            "Você pode baixar as Cartas Raster (GeoTIFF) gratuitamente no site oficial da Marinha do Brasil:<br>"
            "<a href='https://www.marinha.mil.br/chm/dados-do-segnav/cartas-raster'>www.marinha.mil.br/chm</a><br><br>"
            
            "<b>Requisitos do Sistema - Tesseract OCR:</b><br>"
            "Para funcionamento ideal, é necessário instalar o Tesseract OCR em seu sistema:<br>"
            "&bull; <a href='https://github.com/UB-Mannheim/tesseract/wiki'>Download Tesseract OCR (Windows)</a><br>"
            "&bull; <a href='https://tesseract-ocr.github.io/tessdoc/Installation.html'>Download Tesseract OCR (Linux/Mac)</a><br>"
            "<br><br>"
            "⚠️ <b>IMPORTANTE:</b> Na instalação, marque a opção 'Add Tesseract to PATH'. Se não aparecer a opção, você pode adicioná-la manualmente nas variáveis de ambiente do sistema."
            "<br><br>"
            "<br>📖 <b>Mais Informações:</b><br>"
            "Para detalhes sobre o uso avançado, instalações de dependências, solução de problemas e contribuições, consulte o "
            "<a href='https://github.com/elivaldorocha/depth-reader-ocr'>README no GitHub</a>.<br><br>"
        )

    def browse_debug_directory(self):
        directory = QFileDialog.getExistingDirectory(self, "Selecionar Diretório para Imagens de Debug", self.leDebugDir.text())
        if directory:
            self.leDebugDir.setText(directory)

    def browse_csv_file(self):
        file_path, _ = QFileDialog.getSaveFileName(self, "Salvar Arquivo CSV da Batimetria", self.leCSVPath.text(), "Arquivos CSV (*.csv)")
        if file_path:
            self.leCSVPath.setText(file_path)

    def get_debug_directory(self):
        return self.leDebugDir.text()

    def get_csv_path(self):
        return self.leCSVPath.text()

    def get_clip_size(self):
        try:
            return int(self.cbClipSize.currentText())
        except (AttributeError, ValueError):
            return 96 # Valor padrão seguro

    def get_use_ocr_mode(self):
        try:
            return self.chkUseOCR.isChecked()
        except AttributeError:
            return True

    def get_rotations(self):
        """Lê a string de rotações, limpa e converte para uma lista de inteiros."""
        try:
            raw_text = self.leRotations.text()
            rotations = [int(angle.strip()) for angle in raw_text.split(',') if angle.strip()]
            return rotations
        except (ValueError, AttributeError) as e:
            print(f"⚠️ Erro ao ler ângulos de rotação: {e}. Usando valores padrão.")
            return [-90, -45, 0, 45, 90]

    def get_preprocess_methods_config(self):
        """Verifica quais checkboxes de filtros estão marcados e retorna um dicionário de configuração."""
        try:
            config = {
                "clahe": self.chkClahe.isChecked(),
                "gaussian": self.chkGaussian.isChecked(),
                "mean": self.chkMean.isChecked()
            }
            return config
        except AttributeError as e:
            print(f"⚠️ Erro ao ler configuração de filtros: {e}. Usando valores padrão.")
            return {"clahe": True, "gaussian": True, "mean": True}