# -*- coding: utf-8 -*-
"""
cmd = "gdalwarp -crop_to_cutline -cutline C:\\tmp_augmentation\\teste_1.shp C:\\Users\\Gabriel\\Desktop\\CBERS_4_PAN5M_20150712_154_126_L4_BAND1_crop.tif C:\\Users\\Gabriel\\Desktop\\teste000.tif"

from osgeo import ogr
from osgeo import osr

# https://pcjericks.github.io/py-gdalogr-cookbook/projection.html

driver = ogr.GetDriverByName('ESRI Shapefile')
source = osr.SpatialReference()


target = osr.SpatialReference()
target.ImportFromEPSG(4326)

v = ogr.Open("/content/teste.shp")

/***************************************************************************
 DataAugmentation
                                 A QGIS plugin
 This plugin generate basic data augmentation.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-06-07
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Gabriel K. Alves, Sabrina G. Marques, Igor Kolesnikov, Rafael Marinho
        email                : gabriel.alves@inpe.br, rafael.andrade@inpe.br, sabrina.marques@inpe.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .DataAugmentation_dialog import DataAugmentationDialog
import os.path

from qgis import processing
from qgis.gui import QgsMapToolPan
from qgis.core import QgsProject, Qgis, QgsRasterLayer
from PyQt5 import QtWidgets, QtGui, QtCore
from PyQt5.QtWidgets import QTreeWidget, QTreeWidgetItem, QApplication, QWidget

import os
import glob
import shutil
from sys import platform
from . import check_dependences
from .libs.methods import *

from osgeo import gdal, ogr, osr

class DataAugmentation:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'DataAugmentation_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Data Augmentation')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Variables

        # Selected Layer
        self.selectedName = None
        self.selectedNameVector = None
        self.selected = dict()

        self.type = dict()
        self.type['layer']  = 'Layer'
        self.type['vector'] = 'Vector'

        # treeview
        self.methodItem = dict()

        self.topAugmentateTree = None
        self.rotate            = None
        self.blur              = None
        self.reshape           = None
        self.color             = None
        self.remote_sensing    = None
        self.flip              = None

        # Shape formats
        self.sf = ['.shp', '.geojson']

        # All methods
        self.methods = dict()

        # Split vector
        self.vectorsPath = []
        self.vectorsBufferPath = []
        self.imagesArray = []

        self.child = None

        self.round = 0
        self.roundPath = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('DataAugmentation', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/DataAugmentation/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Data Augmentation'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Data Augmentation'),
                action)
            self.iface.removeToolBarIcon(action)

    def selectLayerFile(self):
        selectedLayerPath, _ = QFileDialog.getOpenFileName(self.dlg, "Select input layer file", "", "*.tif")
        
        if(selectedLayerPath != ""):
            nameLayer = os.path.split(selectedLayerPath)[-1]
            nameLayer = nameLayer if os.path.splitext(nameLayer)[1] == "" else (os.path.splitext(nameLayer)[0]) # Remove extension

            self.dlg.layerCombo.addItem(nameLayer)
            self.dlg.layerCombo.setCurrentText(nameLayer)

            self.selectedName = nameLayer

            self.selected[nameLayer] = dict()
            self.selected[nameLayer]['path'] = selectedLayerPath
            self.selected[nameLayer]['type'] = self.type['layer']

    def selectVectorFile(self):
        selectedVectorPath, _ = QFileDialog.getOpenFileName(self.dlg, "Select input vector file", "", "*.shp;;*.geojson;;*.zip")

        if(selectedVectorPath != ""):

            nameVector   = os.path.split(selectedVectorPath)[-1]
            nameVector = nameVector if os.path.splitext(nameVector)[1] == "" else (os.path.splitext(nameVector)[0])

            self.dlg.vectorCombo.addItem(nameVector)
            self.dlg.vectorCombo.setCurrentText(nameVector)

            self.selectedNameVector = nameVector
            
            self.selected[nameVector] = dict()
            self.selected[nameVector]['name'] = nameVector
            self.selected[nameVector]['path'] = selectedVectorPath
            self.selected[nameVector]['type'] = self.type['vector']

    def getGeometryType(self, src=None):
        if not(src): return -1
        d = ogr.Open(src)
        l = d.GetLayer()
        feature = l.GetNextFeature()
        geometry = feature.GetGeometryRef()
        return geometry.GetGeometryName()

    def getProjectionVector(self, src=None):
        if not(src): return -1
        d = ogr.Open(src)
        layer = d.GetLayer()
        spatialRef = layer.GetSpatialRef()
        return spatialRef.GetAttrValue('AUTHORITY', 1)
    
    def getProjectionLayer(self, src=None):
        if not(src): return -1
        d = gdal.Open(src)
        proj = osr.SpatialReference(wkt=d.GetProjection())
        return proj.GetAttrValue('AUTHORITY',1)

    def reprojectionVector(self, src=None, dst=None, CRS=4326, layer=None):
        # Help
        # processing.algorithmHelp("native:assignprojection")
        if(src == None or dst == None): return -1
        p = dict()
        p['INPUT'] = src
        p['CRS'] = f"EPSG:{CRS}"
        p['OUTPUT'] = dst
        processing.run("native:assignprojection", p)
        return p['OUTPUT']

    def readShape(self, src=None):
        if(src == None): src = self.selected[self.selectedNameVector]['path']

        shapefile = ogr.Open(src)
        layer = shapefile.GetLayer(0)

        return shapefile, layer

    def splitPoint(self, dst=None):

        if(dst == None): dst = os.path.join(self.roundPath, "tmp_point.shp")

        shapefileRef, layerRef = self.readShape()

        # set up the shapefile driver
        driver = ogr.GetDriverByName("ESRI Shapefile")

        # create the spatial reference
        srs = layerRef.GetSpatialRef()

        for idx, featureRef in enumerate(layerRef):

            # name output
            dst_name = dst.replace(".shp", "_" + str(idx) + ".shp")

            # geometry 
            geometry = featureRef.GetGeometryRef()

            # create the data source
            data_source = driver.CreateDataSource(dst_name)

            # create the layer
            layer = data_source.CreateLayer("point", srs, ogr.wkbPoint)

            # create the feature
            feature = ogr.Feature(layer.GetLayerDefn())

            feature.SetGeometry(geometry)
            layer.CreateFeature(feature)

            self.vectorsPath.append(dst_name)

        feature = None
        data_source = None

    def readLayer(self, src=None):

        dataset = dict()

        # if src not provided read path selected layer
        if(src == None): 
            self.dataset = gdal.Open(self.selected[self.selectedName]['path'])
        else:
            self.dataset = gdal.Open(src)

        # Get first band -- considering only one band tif as input
        self.band      = self.dataset.GetRasterBand(1)
        self.bandArr   = self.band.ReadAsArray()

        # Get properties to save later
        self.geoTransform = self.dataset.GetGeoTransform()
        self.projection = self.dataset.GetProjection()

        # Mount dataset
        dataset["dataset"]      = self.dataset
        dataset["bandArr"]      = self.bandArr
        dataset["geoTransform"] = self.geoTransform
        dataset["projection"]   = self.projection

        # Validation
        # if image too big return
        # if(self.bandArr.shape()[0] > 5000 and self.bandArr.shape()[0] > 5000):
        # if(True): return "ok" #"err"
        # else: return "ok"

        return dataset

    def saveLayer(self, dst, arr):

        try:

            # check dst name extension
            dst = dst if os.path.splitext(dst)[1] != "" else (dst+".tif")

            [rows, cols] = arr.shape

            driver = gdal.GetDriverByName("GTiff")
            outdata = driver.Create(dst, cols, rows, 1, gdal.GDT_UInt16)

            outdata.SetGeoTransform(self.geoTransform)
            outdata.SetProjection(self.projection) 
            outdata.GetRasterBand(1).WriteArray(arr)
            outdata.GetRasterBand(1).SetNoDataValue(10000)

            outdata.FlushCache()

            outdata = None
        except Exception as err:
            print(err)
            return -1

        return dst
    
    def getBufferPoint(self, src, dst=None, buffer=0.0025):
        if(dst == None): return -1
        # https://docs.qgis.org/3.16/en/docs/user_manual/processing/console.html
        parameters = {'INPUT': src,
               'DISTANCE': buffer,
               'SEGMENTS': 4,
               'DISSOLVE': True,
               'END_CAP_STYLE': 2,
               'JOIN_STYLE': 1,
               'MITER_LIMIT': 20,
               'OUTPUT': dst}
        processing.run("native:buffer", parameters)
        return parameters['OUTPUT']

    def cropLayerByShape(self, src, shape, dst=None, dstNodata=0):
        if(dst == None): return -1 

        try:
            OutTile = gdal.Warp(dst,
                                src,
                                cutlineDSName=shape,
                                cropToCutline=True,
                                dstNodata = dstNodata)
            OutTile = None
        except Exception as err:
            self.iface.messageBar().pushMessage("Error", f"Error cropLayerByShape {err}", level=Qgis.Critical)
            return -1

        return dst
    
    def onChangeLayer(self, value):
        
        # Already has object
        if(value in self.selected.keys()):
            self.selectedName = value
        else:

            layer = [ layer for layer in self.layers if layer.name() == self.dlg.layerCombo.currentText() ]

            if(len(layer) != 0):
                layer = layer[0].layer()
    
                self.selected[self.dlg.layerCombo.currentText()] = dict()
                self.selected[self.dlg.layerCombo.currentText()]['path'] = layer.dataProvider().dataSourceUri()
                self.selected[self.dlg.layerCombo.currentText()]['type'] = self.type['layer']

                self.selectedName = self.dlg.layerCombo.currentText()

    def onChangeVector(self, value):
        if(value in self.selected.keys()):
            self.selectedNameVector = value
        else:
            vector = [ layer for layer in self.layers if layer.name() == self.dlg.vectorCombo.currentText() ]

            if(len(vector) != 0):
                vector = vector[0].layer()
    
                self.selected[self.dlg.vectorCombo.currentText()] = dict()
                self.selected[self.dlg.vectorCombo.currentText()]['path'] = vector.dataProvider().dataSourceUri()
                self.selected[self.dlg.vectorCombo.currentText()]['type'] = self.type['layer']

                self.selectedNameVector = self.dlg.vectorCombo.currentText()

    def chosenTreeItem(self, item, index):
        _item = item.clone()
        itemName = _item.text(index)

        # if last child level
        if(len(_item.takeChildren()) == 0):

            dst = self.saveLayer(os.path.join(self.roundPath, itemName), self.methodItem[itemName])
            if(dst == -1): return -1
            layer = QgsRasterLayer(dst, itemName)
            self.plotCanvas(layer)

    def options(self, imagesArr=[]):

        if(len(imagesArr) == 0): return
        else:

            # Rotate 90
            if self.dlg.rotate90.isChecked():

                if not('rotate' in self.methods.keys()): self.methods["rotate"] = dict() # If not exists create 
                self.methods["rotate"]["rotate90"] = dict()

                for image in imagesArr:

                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_ROTATE90.tif" # Example: /tmp/tmp_point_0_BUFFER_LC08_L1TP_220068_20210101_CROP_CROP_ROTATE90.tif
                    _outName = os.path.splitext(os.path.split(_out)[1])[0] # Example: tmp_point_0_BUFFER_LC08_L1TP_220068_20210101_CROP_CROP_ROTATE90
                    _img = rotate(dataset["bandArr"], 90)

                    # self.methods["rotate"]["rotate90"] = dict()
                    self.methods["rotate"]["rotate90"][_outName] = dict()
                    self.methods["rotate"]["rotate90"][_outName]["image"] = _img
                    self.methods["rotate"]["rotate90"][_outName]["name"]  = _outName
                    self.methods["rotate"]["rotate90"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Rotate180
            if self.dlg.rotate180.isChecked():

                if not('rotate' in self.methods.keys()): self.methods["rotate"] = dict() # If not exists create 
                self.methods["rotate"]["rotate180"] = dict()

                for image in imagesArr:

                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_ROTATE180.tif" 
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = rotate(dataset["bandArr"], 180)

                    # self.methods["rotate"]["rotate90"] = dict()
                    self.methods["rotate"]["rotate180"][_outName] = dict()
                    self.methods["rotate"]["rotate180"][_outName]["image"] = _img
                    self.methods["rotate"]["rotate180"][_outName]["name"]  = _outName
                    self.methods["rotate"]["rotate180"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img
            
            # Flip Hor
            if self.dlg.flipHor.isChecked():
                
                if not('flip' in self.methods.keys()): self.methods["flip"] = dict() # If not exists create 
                self.methods["flip"]["horizontal"] = dict()

                for image in imagesArr:

                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_FLIP_HOR.tif" 
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = flip(dataset["bandArr"], 'horizontal')

                    # self.methods["rotate"]["rotate90"] = dict()
                    self.methods["flip"]["horizontal"][_outName] = dict()
                    self.methods["flip"]["horizontal"][_outName]["image"] = _img
                    self.methods["flip"]["horizontal"][_outName]["name"]  = _outName
                    self.methods["flip"]["horizontal"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Flip Ver
            if self.dlg.flipVer.isChecked():

                if not('flip' in self.methods.keys()): self.methods["flip"] = dict() # If not exists create 
                self.methods["flip"]["vertical"] = dict()

                for image in imagesArr:

                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)
                        
                    _out = root + "_FLIP_VER.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = flip(dataset["bandArr"], "vertical")

                    self.methods["flip"]["vertical"][_outName] = dict()
                    self.methods["flip"]["vertical"][_outName]["image"] = _img
                    self.methods["flip"]["vertical"][_outName]["name"]  = _outName
                    self.methods["flip"]["vertical"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Rescale
            if self.dlg.rescale.isChecked():

                if not('reshape' in self.methods.keys()): self.methods["reshape"] = dict() # If not exists create 
                self.methods["reshape"]["rescale"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)
                        
                    _out = root + "_RESCALE.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = rescale(dataset["bandArr"], 0.5)

                    self.methods["reshape"]["rescale"][_outName] = dict()
                    self.methods["reshape"]["rescale"][_outName]["image"] = _img
                    self.methods["reshape"]["rescale"][_outName]["name"]  = _outName
                    self.methods["reshape"]["rescale"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Crop
            if self.dlg.crop.isChecked():
                
                if not('reshape' in self.methods.keys()): self.methods["reshape"] = dict() # If not exists create 
                self.methods["reshape"]["crop"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_CROP.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = crop(dataset["bandArr"], 100, 100, 200, 200)

                    self.methods["reshape"]["crop"][_outName] = dict()
                    self.methods["reshape"]["crop"][_outName]["image"] = _img
                    self.methods["reshape"]["crop"][_outName]["name"]  = _outName
                    self.methods["reshape"]["crop"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Truncate
            if self.dlg.truncate.isChecked():

                if not('reshape' in self.methods.keys()): self.methods["reshape"] = dict() # If not exists create 
                self.methods["reshape"]["truncate"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_TRUNC.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = trunc(dataset["bandArr"], 100, 200)

                    self.methods["reshape"]["truncate"][_outName] = dict()
                    self.methods["reshape"]["truncate"][_outName]["image"] = _img
                    self.methods["reshape"]["truncate"][_outName]["name"]  = _outName
                    self.methods["reshape"]["truncate"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Trim
            if self.dlg.trim.isChecked():

                if not('reshape' in self.methods.keys()): self.methods["reshape"] = dict() # If not exists create 
                self.methods["reshape"]["trim"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_TRIM.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = trim(dataset["bandArr"], 50, 50, 20, 20)
                    self.methods["reshape"]["trim"][_outName] = dict()
                    self.methods["reshape"]["trim"][_outName]["image"] = _img
                    self.methods["reshape"]["trim"][_outName]["name"]  = _outName
                    self.methods["reshape"]["trim"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Binary
            if self.dlg.binary.isChecked():

                if not('color' in self.methods.keys()): self.methods["color"] = dict() # If not exists create 
                self.methods["color"]["binary"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_BINARY.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = binary(dataset["bandArr"], bw=False, threshold=50, inv=True)

                    self.methods["color"]["binary"][_outName] = dict()
                    self.methods["color"]["binary"][_outName]["image"] = _img
                    self.methods["color"]["binary"][_outName]["name"]  = _outName
                    self.methods["color"]["binary"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Cloud
            if self.dlg.checkBox_15.isChecked():

                if not('remote_sensing' in self.methods.keys()): self.methods["remote_sensing"] = dict() # If not exists create 
                self.methods["remote_sensing"]["cloud"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_CLOUD.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = cloud(dataset["bandArr"])

                    self.methods["remote_sensing"]["cloud"][_outName] = dict()
                    self.methods["remote_sensing"]["cloud"][_outName]["image"] = _img
                    self.methods["remote_sensing"]["cloud"][_outName]["name"]  = _outName
                    self.methods["remote_sensing"]["cloud"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Degradation
            if self.dlg.checkBox_17.isChecked():

                if not('remote_sensing' in self.methods.keys()): self.methods["remote_sensing"] = dict() # If not exists create 
                self.methods["remote_sensing"]["degradation"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)
                    
                    _out = root + "_DEGRADATION.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = gdalDecreaseResolution(dataset["dataset"], image, dst=_out)

                    self.methods["remote_sensing"]["degradation"][_outName] = dict()
                    self.methods["remote_sensing"]["degradation"][_outName]["image"] = _img
                    self.methods["remote_sensing"]["degradation"][_outName]["name"]  = _outName
                    self.methods["remote_sensing"]["degradation"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Haze
            if self.dlg.checkBox_16.isChecked():

                if not('remote_sensing' in self.methods.keys()): self.methods["remote_sensing"] = dict() # If not exists create 
                self.methods["remote_sensing"]["haze"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_BLUR.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = blur(dataset["bandArr"], sigma=2)

                    self.methods["remote_sensing"]["haze"][_outName] = dict()
                    self.methods["remote_sensing"]["haze"][_outName]["image"] = _img
                    self.methods["remote_sensing"]["haze"][_outName]["name"]  = _outName
                    self.methods["remote_sensing"]["haze"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img

            # Edgy
            if self.dlg.checkBox_18.isChecked():

                if not('remote_sensing' in self.methods.keys()): self.methods["remote_sensing"] = dict() 
                self.methods["remote_sensing"]["sharpen_edges"] = dict()

                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)

                    _out = root + "_SHARPENEDGES.tif"
                    _outName = os.path.splitext(os.path.split(_out)[1])[0]
                    _img = edgy(dataset["bandArr"])

                    self.methods["remote_sensing"]["sharpen_edges"][_outName] = dict()
                    self.methods["remote_sensing"]["sharpen_edges"][_outName]["image"] = _img
                    self.methods["remote_sensing"]["sharpen_edges"][_outName]["name"]  = _outName
                    self.methods["remote_sensing"]["sharpen_edges"][_outName]["path"]  = _out
                    
                    self.methodItem[_outName] = _img
            
            # Augmentate
            if self.dlg.augmentate.isChecked():

                if not('augmentate' in self.methods.keys()): self.methods["augmentate"] = dict() 
                
                for image in imagesArr:
                    
                    dataset = self.readLayer(src=image)
                    root, ext = os.path.splitext(image)
                    _outName = os.path.splitext(os.path.split(image)[1])[0]

                    images, labelImage, q = augmentate(dataset["bandArr"], name=_outName, chunks=50)

                    # Merge to dict used to save tmp file {key: image array}
                    self.methodItem.update(labelImage)

                    if(len(self.methods['augmentate'].keys()) == 0):
                        self.methods["augmentate"] = images
                    else:
                        self.methods["augmentate"].update(images)

    def plotCanvas(self, layer):

        QgsProject.instance().addMapLayer(layer, False)

        self.dlg.canvas.clearCache()
        self.dlg.canvas.refresh()

        self.dlg.canvas.setExtent(layer.extent())
        self.dlg.canvas.setLayers([layer])
        self.dlg.canvas.refresh()

    def treeView(self):
        
        self.dlg.treeWidget.clear()

        for key in self.methods.keys():
            self.child = QTreeWidgetItem([key])

            for key2 in self.methods[key].keys():
                child2 = QTreeWidgetItem([key2])

                for key3 in self.methods[key][key2].keys():
                    child3 = QTreeWidgetItem([self.methods[key][key2][key3]['name']])
                    child2.addChild(child3)

                self.child.addChild(child2)
            
            self.dlg.treeWidget.addTopLevelItem(self.child)

    def main(self):

        # reset variable
        self.vectorsPath = []
        self.imagesArray = []
        self.vectorsBufferPath = []

        # Get selected layer and vector index
        self.selectedLayerIndex  = self.dlg.layerCombo.currentIndex()
        self.selectedVectorIndex = self.dlg.vectorCombo.currentIndex()

        # Required Layer
        if(self.selectedLayerIndex == -1):
            self.iface.messageBar().pushMessage("Error", "Required select layer", level=Qgis.Critical)
            return -1

        # Required Vector
        elif(self.selectedVectorIndex == -1):
            
            # If not vector do with all image if is not too big
            self.iface.messageBar().pushMessage("Error", "Vector not provided, trying with the whole image", level=Qgis.Info, duration=3)
            return -1
            # self.iface.messageBar().pushMessage("Error", "Required select vector", level=Qgis.Critical)
            # return

        # Get selected layer and vector
        # If not chosen file from button
        if(len(self.layers) != 0):
            if(self.selectedName == None):
                # Must be on layers
                layer = [ layer for layer in self.layers if layer.name() == self.dlg.layerCombo.currentText() ][0].layer()

                self.selected[self.dlg.layerCombo.currentText()] = dict()
                self.selected[self.dlg.layerCombo.currentText()]['path'] = layer.dataProvider().dataSourceUri()
                self.selected[self.dlg.layerCombo.currentText()]['type'] = self.type['layer']

                self.selectedName = self.dlg.layerCombo.currentText()
            if(self.selectedNameVector == None):
                vector = [ layer for layer in self.layers if layer.name() == self.dlg.vectorCombo.currentText() ]
                if(len(vector) > 0):
                    vector = vector[0].layer()

                    self.selected[self.dlg.vectorCombo.currentText()] = dict()
                    self.selected[self.dlg.vectorCombo.currentText()]['path'] = vector.dataProvider().dataSourceUri()
                    self.selected[self.dlg.vectorCombo.currentText()]['type'] = self.type['vector']
                    
                    self.selectedNameVector = self.dlg.vectorCombo.currentText()
        
        # check round (folders)
        if not(os.path.isdir(os.path.join(self.tmpFolder, str(self.round)))):
            self.roundPath = os.path.join(self.tmpFolder, str(self.round))
            os.mkdir(self.roundPath)
            self.round += 1
        else:
            # if exists, update round and create
            try:

                # max number round 
                newRound = max([int(x) for x in os.listdir(self.tmpFolder)])
                newRound += 1
                self.round = newRound

                # creating again
                if not(os.path.isdir(os.path.join(self.tmpFolder, str(self.round)))):
                    self.roundPath = os.path.join(self.tmpFolder, str(self.round))
                    os.mkdir(self.roundPath)
                    self.round += 1

            except Exception  as err:
                print(err)
                self.iface.messageBar().pushMessage("Error", f"{err}", level=Qgis.Critical)
                return -1

        # Clip by vector
        if(self.selectedNameVector and self.selectedName):
            
            # validade points
            try:
                # Check projection and feature type
                projectionVector = self.getProjectionVector(self.selected[self.selectedNameVector]['path'])

                # check path
                # _aux = self.selected[self.selectedNameVector]['path']
                # if(os.path.split(_aux)[0] == ""):
                #     self.iface.messageBar().pushMessage("Error", "Vector path is not defined", level=Qgis.Critical)    

                if(projectionVector != "4326"):
                    self.iface.messageBar().pushMessage("Error", "Vector must be in EPSG:4326", level=Qgis.Critical)
                    return 

                # Check geometry type
                geometryType = self.getGeometryType(self.selected[self.selectedNameVector]['path'])
                if(geometryType != "POINT"):
                    self.iface.messageBar().pushMessage("Error", f"Geometry type must be a POINT not {geometryType}", level=Qgis.Critical)
                    return 

            except Exception as err:
                self.iface.messageBar().pushMessage("Error", f"Error validating vector: {err}", level=Qgis.Critical)
                return 

            dst = self.selected[self.selectedNameVector]['path']
            dst = os.path.split(dst)[-1]
            dst = os.path.join(self.roundPath, dst)
            
            # Save each layer separated
            self.splitPoint(dst=dst) # feed self.vectorsPath array

            # Save Buffer array 
            for vector in self.vectorsPath:
                dst = vector.replace(".shp", "_BUFFER.shp")
                dst = os.path.split(dst)[-1]
                dst = os.path.join(self.roundPath, dst)
                buffer = self.getBufferPoint(vector, dst)
                self.vectorsBufferPath.append(buffer)

            # Clip image by buffer vectors array
            for buffer in self.vectorsBufferPath:
                dst = buffer.replace(".shp", "_" + self.selectedName + "_CROP.tif")
                dst = os.path.split(dst)[-1]
                dst = os.path.join(self.roundPath, dst)
                image = self.cropLayerByShape(self.selected[self.selectedName]['path'], buffer, dst)
                self.imagesArray.append(image)
        
        # validating arrays
        for i in self.imagesArray:
            if not(os.path.isfile(i)):
                self.iface.messageBar().pushMessage("Error", "Error saving image", level=Qgis.Critical)
                return -1
            
        # Get array and info from selected layer
        # responseRL = self.readLayer()
        responseRL = "ok"
        if(responseRL == "err"): 
            self.iface.messageBar().pushMessage("Error", "Image too large, please provide de vector file", level=Qgis.Critical)
            return -1
        else:
            # check chosen methods
            self.options(imagesArr=self.imagesArray)

            # Create Tree View
            self.treeView()

            # message success
            self.iface.messageBar().pushMessage( "Success!", level=Qgis.Success, duration=3)
    
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:

            # Creating and cleaning folder
            if(platform.startswith("win")): self.tmpFolder = "C:\\tmp_augmentation"
            else:                           self.tmpFolder = "/tmp"
            
            if (os.path.isdir(self.tmpFolder)): 
                for f in [os.path.join(self.tmpFolder, x) for x in os.listdir(self.tmpFolder)]:
                    try:
                        # remove dirs
                        if(os.path.isdir(f)):  shutil.rmtree(f)
                        # remove files
                        if(os.path.isfile(f)): os.remove(f)

                    except Exception as e: print(e)
            else: os.mkdir(self.tmpFolder)
            
            self.first_start = False
            self.dlg = DataAugmentationDialog()

            # On click button select output file
            self.dlg.layerButton.clicked.connect(self.selectLayerFile)
            self.dlg.vectorButton.clicked.connect(self.selectVectorFile)
            
            # On click tree view
            self.dlg.treeWidget.itemDoubleClicked.connect(self.chosenTreeItem)

            # On click run
            self.dlg.run.clicked.connect(self.main)

            # Change combobox
            self.dlg.layerCombo.currentTextChanged.connect(self.onChangeLayer)
            self.dlg.vectorCombo.currentTextChanged.connect(self.onChangeVector)
        
        
        tool = QgsMapToolPan(self.dlg.canvas)
        self.dlg.canvas.setMapTool(tool)

        # Get all active layers on the screen
        self.layers = QgsProject.instance().layerTreeRoot().children()

        # Clear the contents of the layerCombo from previous runs
        self.dlg.layerCombo.clear()
        self.dlg.vectorCombo.clear()

        # Populate the layerCombo with names of all the loaded layers
        # Check if there is layers
        if (len(self.layers) != 0): 
            self.dlg.layerCombo.addItems([layer.name() for layer in self.layers if not(os.path.splitext(layer.layer().dataProvider().dataSourceUri().lower())[1] in self.sf) ])
            self.dlg.vectorCombo.addItems([layer.name() for layer in self.layers if(os.path.splitext(layer.layer().dataProvider().dataSourceUri().lower())[1] in self.sf) ])

        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()
        
        if result:
            pass

# criar as subpastas