# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ValidateData
                                 A QGIS plugin
 prototype for data validation
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-02-20
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Michal Pilarski
        email                : michpil@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QPushButton, QMessageBox
from qgis.core import Qgis, QgsVectorLayer

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .data_validation_dialog import ValidateDataDialog
import os.path
import json
from qgis.PyQt.QtCore import QDir


class ValidateData:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ValidateData_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Data Validator')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ValidateData', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/data_validation/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Data Validator'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Data Validator'),
                action)
            self.iface.removeToolBarIcon(action)

    def select_expected_file(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select expected filename", "", '*.json')
        self.dlg.lineEdit_expected.setText(filename)

    def select_input_file(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select input filename", "", '*.*')
        self.dlg.lineEdit_input.setText(filename)

    def get_json(self, file):
        with open(file) as f:
            data = json.load(f)
        return data

    def show_errors(self):
        errors_ = "\n".join(log_errors)
        QMessageBox.critical(None, 'Validation Failures', count + '\n\n' + errors_)

    # def show_errors(self):
    #     errors = 'Hello my friend it is my first plugin\naaaa'
    #     QMessageBox.critical(None, 'Validation Errors', errors)


    def get_vector_metadata(self, filename):
        vlayer_act = QgsVectorLayer(filename, "vector layer", "ogr")
        encoding = vlayer_act.dataProvider().encoding()
        crs = vlayer_act.crs().authid()
        layer_type = str(vlayer_act.type())
        layer_name = vlayer_act.name()
        extent = {'MIN_LON': vlayer_act.extent().xMinimum(), 'MIN_LAT': vlayer_act.extent().yMinimum(),
                  'MAX_LON': vlayer_act.extent().xMaximum(), 'MAX_LAT': vlayer_act.extent().yMaximum()}
        geometry_type = str(vlayer_act.geometryType())
        storage_type = vlayer_act.storageType()
        fields_number = len(vlayer_act.fields())
        fields_name = {field.name(): field.name() for field in vlayer_act.fields()}
        fields_type_name = {field.name(): field.typeName() for field in vlayer_act.fields()}
        fields_type = {field.name(): field.type() for field in vlayer_act.fields()}
        fields_length = {field.name(): field.length() for field in vlayer_act.fields()}
        fields_precision = {field.name(): field.precision() for field in vlayer_act.fields()}
        fields_isreadonly = {field.name(): field.isReadOnly() for field in vlayer_act.fields()}
        fields_comment = {field.name(): field.comment() for field in vlayer_act.fields()}
        rows_number = vlayer_act.featureCount()
        return {'encoding': encoding, 'crs': crs, 'layer_type': layer_type, 'layer_name': layer_name,
                'extent': extent, 'geometry_type': geometry_type, 'storage_type': storage_type, 'fields_number': fields_number,
                'fields_name': fields_name, 'fields_type_name': fields_type_name, 'fields_type': fields_type, 'fields_length': fields_length,
                'fields_precision': fields_precision, 'fields_isreadonly': fields_isreadonly, 'fields_comment': fields_comment,
                'rows_number': rows_number}

    def check_actual_expected(self, actual, expected):
        log_errors_ = []
        attributes_to_check = ['encoding', 'crs', 'layer_type', 'layer_name', 'extent', 'geometry_type', 'storage_type', 'fields_number', 'rows_number']
        # attributes_to_check = ['encoding', 'crs']
        for attrib in attributes_to_check:
            exp_value = expected.get('METADATA').get(attrib)
            act_value = actual.get(attrib)
            if act_value != exp_value:
                log_errors_.append(f'{attrib} | ACTUAL: {str(act_value)} | EXPECTED: {str(exp_value)}')

        count = f'{len(attributes_to_check) - len(log_errors_)} PASSED | {len(log_errors_)} FAILED'
        return count, log_errors_

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = ValidateDataDialog()
            self.dlg.toolButton_input.clicked.connect(self.select_input_file)
            self.dlg.toolButton_expected.clicked.connect(self.select_expected_file)
            # self.dlg.toolButton_report.clicked.connect(self.select_expected_file)

        # clear the contents
        # self.dlg.lineEdit_input.clear()
        # self.dlg.lineEdit_expected.clear()
        # self.dlg.lineEdit_report.clear()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            filename_in = self.dlg.lineEdit_input.text()
            filename_exp = self.dlg.lineEdit_expected.text()
            if filename_in == '' or filename_exp == '':
                QMessageBox.critical(None, 'Error', 'Field(s) is empty')
            else:
                # Get metadata
                vector_metadata = self.get_vector_metadata(filename_in)
                expected_metadata = self.get_json(filename_exp)

                global log_errors
                global count
                count, log_errors = self.check_actual_expected(vector_metadata, expected_metadata)

                # Generate a message after running the plugin
                if len(log_errors) == 0:
                    self.iface.messageBar().pushMessage(
                        "REPORT", "VALIDATION STATUS - OKAY",
                        level=Qgis.Success, duration=10)

                elif len(log_errors) != 0:
                    widget = self.iface.messageBar().createMessage("REPORT", "VALIDATION STATUS - FAILED")
                    button = QPushButton(widget)
                    button.setText("Validation Failures")
                    button.pressed.connect(self.show_errors)
                    widget.layout().addWidget(button)
                    self.iface.messageBar().pushWidget(widget, Qgis.Critical)
