"""
API Client for Needle Digital Mining Data Service

This module provides a comprehensive HTTP client for communicating with the
Needle Digital mining data API, handling all aspects of authentication,
token management, and data requests.

Key Features:
    - Firebase Authentication integration
    - Automatic token refresh and management
    - Secure credential storage using QGIS settings
    - Robust error handling and retry logic
    - Signal-based asynchronous operations
    - Request/response logging for debugging
    - Network timeout and error recovery

Authentication Flow:
    1. User provides email/password credentials
    2. Firebase authentication via REST API
    3. JWT tokens stored securely in QGIS settings
    4. Automatic token refresh before expiration
    5. Silent re-authentication on startup

Security Features:
    - Secure token storage using QGIS encrypted settings
    - No plaintext password storage
    - Token validation and automatic refresh
    - Request signing and authorization headers
    - SSL/TLS encryption for all communications

API Endpoints:
    - Authentication: Firebase Auth API
    - Data fetching: Needle Digital mining database
    - Company search: Company directory API
    - Count queries: Data availability checks

Author: Needle Digital
Contact: divyansh@needle-digital.com
"""

import json
import time
import re
import zlib
from typing import Dict, Any, Optional, Callable
from qgis.PyQt.QtCore import QObject, pyqtSignal, QTimer, QUrl, QByteArray
from qgis.PyQt.QtNetwork import QNetworkAccessManager, QNetworkRequest, QNetworkReply
from qgis.core import QgsSettings

# Internal configuration and utilities
from ..config.settings import config  # API configuration and settings
from ..utils.logging import log_error, log_warning, log_info  # Centralized logging system
from ..utils.validation import get_user_role_from_token  # JWT token utilities


class ApiClient(QObject):
    """HTTP API Client for Needle Digital Mining Data Services.

    A comprehensive API client that handles all communication with the Needle Digital
    mining database, including authentication, token management, and data requests.

    Features:
        - Firebase Authentication with JWT tokens
        - Automatic token refresh and session management
        - Secure credential storage using QGIS settings
        - Asynchronous operations with Qt signals
        - Robust error handling and retry logic
        - Request/response logging for debugging
        - Network timeout and connection management
        - Server-Sent Events (SSE) streaming support

    Signals:
        login_success(): Emitted when authentication succeeds
        login_failed(str): Emitted when authentication fails with error message
        api_response_received(str, dict): Emitted with endpoint and response data
        api_error_occurred(str, str): Emitted with endpoint and error message

    Thread Safety:
        This class is designed to be used from the main Qt thread and uses
        Qt's signal/slot mechanism for asynchronous operations.
    """

    # Qt Signals for asynchronous operation feedback
    login_success = pyqtSignal()  # Authentication successful
    login_failed = pyqtSignal(str)  # Authentication failed with error message
    api_response_received = pyqtSignal(str, dict)  # endpoint, response_data
    api_error_occurred = pyqtSignal(str, str)  # endpoint, error_message

    def __init__(self):
        super().__init__()
        self.network_manager = QNetworkAccessManager()
        self.auth_token: Optional[str] = None
        self.refresh_token: Optional[str] = None
        self.token_expires_at: float = 0
        self.user_role: Optional[str] = None  # User role from token (tier_1, tier_2, admin)
        self.last_login_email: str = ""  # Store last successful login email for autofill
        self._initialization_complete = False
        self._active_replies = []  # Track active network requests for cancellation
        self._streaming_buffers = {}  # Track SSE parsing buffers by reply
        self._streaming_decompressors = {}  # Track gzip decompressor objects by reply
        self._streaming_text_buffers = {}  # Track text buffers for incomplete SSE events

        # Token refresh timer
        self.token_refresh_timer = QTimer()
        self.token_refresh_timer.timeout.connect(lambda: self.refresh_auth_token(silent=True))
        
        # Load saved tokens and expiration time but don't refresh immediately
        settings = QgsSettings()
        self.refresh_token = settings.value("needle/refreshToken", None)
        self.auth_token = settings.value("needle/authToken", None)

        # Extract role from stored token if available
        if self.auth_token:
            self.user_role = get_user_role_from_token(self.auth_token)

        # Safely load token expiration time
        try:
            expires_value = settings.value("needle/tokenExpiresAt", 0)
            self.token_expires_at = float(expires_value) if expires_value else 0
        except (ValueError, TypeError):
            self.token_expires_at = 0
    
    def complete_initialization(self):
        """Complete initialization and attempt silent token refresh if needed."""
        if self._initialization_complete:
            return

        self._initialization_complete = True

        # Check if we have a refresh token
        if self.refresh_token:
            # Check if access token is expired or will expire soon
            time_until_expiry = self.token_expires_at - time.time()

            if time_until_expiry < 60:  # Expired or expiring in <1 minute
                log_info(f"Token expired/expiring soon, attempting silent refresh...")
                self.refresh_auth_token(silent=True)
            else:
                log_info(f"Token still valid for {time_until_expiry/60:.1f} minutes")
                # Schedule next refresh at 50% of remaining lifetime
                refresh_delay_ms = max(0, int((time_until_expiry / 2) * 1000))
                self.token_refresh_timer.start(refresh_delay_ms)
                log_info(f"Next token refresh scheduled in {refresh_delay_ms/1000:.0f} seconds")
        else:
            log_warning("No refresh token found - user must login")
    
    def is_authenticated(self) -> bool:
        """Check if user is currently authenticated."""
        return bool(self.auth_token and time.time() < self.token_expires_at)

    def get_user_role(self) -> Optional[str]:
        """Get the current user's role."""
        return self.user_role

    def get_last_login_email(self) -> str:
        """Get the last successfully logged in email for autofill."""
        settings = QgsSettings()
        return settings.value("needle/lastLoginEmail", "")

    def ensure_token_valid(self) -> bool:
        """Ensure token is valid, refresh if needed. Returns True if valid/refreshed."""
        # Check if token will expire in next 5 minutes
        time_until_expiry = self.token_expires_at - time.time()

        if time_until_expiry < 300:  # Less than 5 minutes remaining
            log_info(f"Token expiring soon ({time_until_expiry:.0f}s), triggering proactive refresh...")
            self.refresh_auth_token(silent=True)
            return True

        return self.is_authenticated()

    def login(self, email: str, password: str) -> None:
        """Authenticate user with email and password."""
        if not email or not password or not re.match(r"[^@]+@[^@]+\.[^@]+", email):
            self.login_failed.emit("A valid email and password are required.")
            return

        # Store email for autofill on successful login
        self.last_login_email = email

        payload = {
            "email": email,
            "password": password,
            "returnSecureToken": True
        }

        self._make_request(
            url=config.firebase_auth_url,
            method="POST",
            data=payload,
            callback=self._handle_login_response,
            error_callback=lambda error: self.login_failed.emit(f"Login failed: {error}")
        )
    
    def logout(self) -> None:
        """Logout user and clear stored tokens."""
        self.auth_token = None
        self.refresh_token = None
        self.token_expires_at = 0
        self.user_role = None
        self.last_login_email = ""
        self.token_refresh_timer.stop()

        # Clear all stored tokens, expiration time, and email
        settings = QgsSettings()
        settings.remove("needle/refreshToken")
        settings.remove("needle/authToken")
        settings.remove("needle/tokenExpiresAt")
        settings.remove("needle/lastLoginEmail")

        # Cancel any ongoing requests
        self.cancel_all_requests()
    
    def cancel_all_requests(self) -> None:
        """Cancel all active network requests."""
        for reply in self._active_replies:
            if reply and not reply.isFinished():
                reply.abort()
        self._active_replies.clear()
    
    def refresh_auth_token(self, silent: bool = False) -> None:
        """Refresh the authentication token using refresh token."""
        if not self.refresh_token:
            if not silent:
                log_warning("Token refresh aborted: No refresh token available.")
            return
        
        
        payload = {
            "grant_type": "refresh_token",
            "refresh_token": self.refresh_token
        }
        
        self._make_request(
            url=config.firebase_refresh_url,
            method="POST",
            data=payload,
            callback=self._handle_refresh_response,
            error_callback=lambda error: log_error(f"Token refresh failed: {error}")
        )
    
    def make_api_request(self, endpoint: str, params: Dict[str, Any], callback: Optional[Callable] = None) -> None:
        """Make authenticated API request to Needle Digital service."""
        if not self.is_authenticated():
            self.api_error_occurred.emit(endpoint, "Authentication required")
            return

        url = f"{config.BASE_API_URL}/{endpoint}"
        headers = {"Authorization": f"Bearer {self.auth_token}"}

        self._make_request(
            url=url,
            method="GET",
            params=params,
            headers=headers,
            callback=lambda data: self._handle_api_response(endpoint, data, callback),
            error_callback=lambda error: self.api_error_occurred.emit(endpoint, error)
        )

    def make_streaming_request(self, endpoint: str, params: Dict[str, Any],
                               data_callback: Callable[[dict], None],
                               progress_callback: Callable[[dict], None],
                               complete_callback: Callable[[dict], None],
                               error_callback: Callable[[dict], None]) -> QNetworkReply:
        """
        Make authenticated streaming API request using Server-Sent Events (SSE).

        Args:
            endpoint: API endpoint path
            params: Query parameters dictionary
            data_callback: Called when 'data' event is received with batch of records
            progress_callback: Called when 'progress' event is received with progress info
            complete_callback: Called when 'complete' event is received with final summary
            error_callback: Called when 'error' event is received or connection fails

        Returns:
            QNetworkReply: The active network reply (for cancellation support)
        """
        if not self.is_authenticated():
            error_callback({"error": "Authentication required"})
            return None

        url = f"{config.BASE_API_URL}/{endpoint}"

        # Build URL with query parameters
        request_url = QUrl(url)
        if params:
            query_parts = []
            for key, value in params.items():
                if value is not None:
                    # Special handling for polygon_coords - add multiple coords parameters
                    if key == 'polygon_coords' and isinstance(value, list):
                        for lat, lon in value:
                            query_parts.append(f"coords={lat},{lon}")
                    else:
                        query_parts.append(f"{key}={str(value)}")
            if query_parts:
                query_string = "&".join(query_parts)
                request_url = QUrl(f"{url}?{query_string}")

        request = QNetworkRequest(request_url)

        # Set SSE-specific headers
        request.setRawHeader(b'Authorization', f"Bearer {self.auth_token}".encode())
        request.setRawHeader(b'Accept', b'text/event-stream')
        request.setRawHeader(b'Accept-Encoding', b'gzip, deflate')
        request.setRawHeader(b'Cache-Control', b'no-cache')

        # Make GET request
        reply = self.network_manager.get(request)

        # Track reply for cancellation
        self._active_replies.append(reply)

        # Initialize buffers for this reply
        reply_id = id(reply)
        self._streaming_buffers[reply_id] = b""  # Byte buffer for incoming data
        self._streaming_text_buffers[reply_id] = ""  # Text buffer for incomplete SSE events
        self._streaming_decompressors[reply_id] = None  # Will create decompressor if gzip detected

        # Store callbacks on reply object for access in handlers
        reply.data_callback = data_callback
        reply.progress_callback = progress_callback
        reply.complete_callback = complete_callback
        reply.error_callback = error_callback

        # Connect streaming event handlers
        reply.readyRead.connect(lambda: self._handle_streaming_data(reply))
        reply.finished.connect(lambda: self._handle_streaming_finished(reply))
        reply.errorOccurred.connect(lambda error_code: self._handle_streaming_error(reply, error_code))

        return reply

    def _handle_streaming_data(self, reply: QNetworkReply) -> None:
        """Handle incoming SSE data chunks with incremental gzip decompression support."""
        try:
            reply_id = id(reply)

            # Read available bytes
            chunk_bytes = bytes(reply.readAll())
            if not chunk_bytes:
                return

            # Check for gzip encoding on first chunk using rawHeader
            if self._streaming_decompressors.get(reply_id) is None:
                # Use rawHeader to get Content-Encoding
                content_encoding_bytes = reply.rawHeader(b'Content-Encoding')
                content_encoding = bytes(content_encoding_bytes).decode('utf-8', errors='ignore').lower() if content_encoding_bytes else ''

                if 'gzip' in content_encoding:
                    # Create incremental decompressor using zlib
                    import zlib
                    # wbits=MAX_WBITS | 16 enables gzip format detection
                    self._streaming_decompressors[reply_id] = zlib.decompressobj(wbits=zlib.MAX_WBITS | 16)
                    log_info("Streaming response is gzip-compressed, using incremental decompression")
                else:
                    # Not gzip, mark as plain text
                    self._streaming_decompressors[reply_id] = False
                    log_info("Streaming response is plain text (no gzip)")

            # Decompress if gzip
            decompressor = self._streaming_decompressors.get(reply_id)
            if decompressor and decompressor is not False:
                try:
                    # Incrementally decompress
                    decompressed_chunk = decompressor.decompress(chunk_bytes)
                    if not decompressed_chunk:
                        # No data yet, might need more chunks
                        return
                    text_chunk = decompressed_chunk.decode('utf-8')
                except Exception as e:
                    log_error(f"Decompression error: {e}")
                    import traceback
                    log_error(traceback.format_exc())
                    # Don't crash - call error callback and abort
                    if hasattr(reply, 'error_callback'):
                        reply.error_callback({"error": f"Decompression failed: {str(e)}"})
                    reply.abort()
                    return
            else:
                # Plain text - decode directly
                try:
                    text_chunk = chunk_bytes.decode('utf-8')
                except UnicodeDecodeError as e:
                    log_error(f"UTF-8 decode error: {e}")
                    import traceback
                    log_error(traceback.format_exc())
                    # Don't crash - call error callback and abort
                    if hasattr(reply, 'error_callback'):
                        reply.error_callback({"error": f"UTF-8 decode failed: {str(e)}"})
                    reply.abort()
                    return

            # Append to text buffer
            text_buffer = self._streaming_text_buffers.get(reply_id, "")
            text_buffer += text_chunk

            # Parse SSE events (separated by double newline)
            events = text_buffer.split('\n\n')

            # Keep the last incomplete event in buffer
            self._streaming_text_buffers[reply_id] = events[-1]

            # Process complete events
            for event_block in events[:-1]:
                if not event_block.strip():
                    continue

                # Parse event format: event: type\ndata: json
                lines = event_block.strip().split('\n')
                event_type = None
                event_data = None

                for line in lines:
                    if line.startswith('event:'):
                        event_type = line[6:].strip()
                    elif line.startswith('data:'):
                        try:
                            event_data = json.loads(line[5:].strip())
                        except json.JSONDecodeError as e:
                            log_error(f"Failed to parse SSE data: {e}")
                            log_error(f"Problematic line: {line[:200]}")
                            continue

                # Route event to appropriate callback with error handling
                if event_type and event_data:
                    try:
                        if event_type == 'data' and hasattr(reply, 'data_callback'):
                            reply.data_callback(event_data)
                        elif event_type == 'progress' and hasattr(reply, 'progress_callback'):
                            reply.progress_callback(event_data)
                        elif event_type == 'complete' and hasattr(reply, 'complete_callback'):
                            reply.complete_callback(event_data)
                        elif event_type == 'error' and hasattr(reply, 'error_callback'):
                            reply.error_callback(event_data)
                    except Exception as callback_error:
                        log_error(f"Error in {event_type} callback: {callback_error}")
                        import traceback
                        log_error(traceback.format_exc())

        except Exception as e:
            log_error(f"Critical error handling streaming data: {e}")
            import traceback
            log_error(traceback.format_exc())
            # Safely call error callback if it exists
            try:
                if hasattr(reply, 'error_callback'):
                    reply.error_callback({"error": f"Streaming handler error: {str(e)}"})
            except:
                pass
            # Abort to prevent further crashes
            try:
                reply.abort()
            except:
                pass

    def _handle_streaming_finished(self, reply: QNetworkReply) -> None:
        """Handle completion of streaming connection."""
        reply_id = id(reply)

        # Cleanup all tracking structures
        if reply in self._active_replies:
            self._active_replies.remove(reply)
        if reply_id in self._streaming_buffers:
            del self._streaming_buffers[reply_id]
        if reply_id in self._streaming_text_buffers:
            del self._streaming_text_buffers[reply_id]
        if reply_id in self._streaming_decompressors:
            del self._streaming_decompressors[reply_id]

        reply.deleteLater()

    def _handle_streaming_error(self, reply: QNetworkReply, error_code) -> None:
        """Handle streaming connection errors."""
        if reply.error() != QNetworkReply.NoError:
            error_msg = reply.errorString()
            log_error(f"Streaming network error: {error_msg}")
            reply.error_callback({"error": error_msg})

    def cancel_streaming_request(self, reply: Optional[QNetworkReply]) -> None:
        """Cancel an active streaming request."""
        if reply and reply in self._active_replies:
            reply.abort()
            self._active_replies.remove(reply)

            # Cleanup buffers and tracking
            reply_id = id(reply)
            if reply_id in self._streaming_buffers:
                del self._streaming_buffers[reply_id]
            if reply_id in self._streaming_text_buffers:
                del self._streaming_text_buffers[reply_id]
            if reply_id in self._streaming_decompressors:
                del self._streaming_decompressors[reply_id]

            log_warning("Streaming request cancelled by user")
    
    def _make_request(self, url: str, method: str = "GET", data: Optional[Dict] = None, 
                     params: Optional[Dict] = None, headers: Optional[Dict] = None,
                     callback: Optional[Callable] = None, error_callback: Optional[Callable] = None) -> None:
        """Make HTTP request with proper error handling."""
        request_url = QUrl(url)
        
        # Add query parameters for GET requests
        if method == "GET" and params:
            # Build query string manually to avoid Qt version compatibility issues
            query_parts = []
            for key, value in params.items():
                if value is not None:
                    query_parts.append(f"{key}={str(value)}")
            if query_parts:
                query_string = "&".join(query_parts)
                if "?" in url:
                    request_url = QUrl(f"{url}&{query_string}")
                else:
                    request_url = QUrl(f"{url}?{query_string}")
        
        request = QNetworkRequest(request_url)
        
        # Set headers
        if headers:
            for key, value in headers.items():
                request.setRawHeader(key.encode(), value.encode())
        
        # Make request based on method
        if method == "POST":
            request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
            json_data = QByteArray(json.dumps(data or {}).encode('utf-8'))
            reply = self.network_manager.post(request, json_data)
        else:
            reply = self.network_manager.get(request)
        
        # Track the reply for cancellation
        self._active_replies.append(reply)
        
        # Connect response handler
        reply.finished.connect(lambda: self._handle_network_reply(reply, callback, error_callback))
    
    def _handle_network_reply(self, reply: QNetworkReply, 
                             callback: Optional[Callable] = None,
                             error_callback: Optional[Callable] = None) -> None:
        """Handle network reply with proper error checking."""
        # Remove from active replies list
        if reply in self._active_replies:
            self._active_replies.remove(reply)
        
        try:
            if reply.error() != QNetworkReply.NoError:
                error_msg = reply.errorString()
                try:
                    error_data = json.loads(bytes(reply.readAll()).decode('utf-8'))
                    if 'error' in error_data:
                        error_msg = error_data['error'].get('message', error_msg)
                except:
                    pass
                
                log_error(f"Network error: {error_msg}")
                if error_callback:
                    error_callback(error_msg)
            else:
                response_data = json.loads(bytes(reply.readAll()).decode('utf-8'))
                if callback:
                    callback(response_data)
        except json.JSONDecodeError as e:
            error_msg = f"Invalid JSON response: {e}"
            log_error(error_msg)
            if error_callback:
                error_callback(error_msg)
        except Exception as e:
            error_msg = f"Unexpected error: {e}"
            log_error(error_msg)
            if error_callback:
                error_callback(error_msg)
        finally:
            reply.deleteLater()
    
    def _handle_login_response(self, response_data: Dict[str, Any]) -> None:
        """Handle login response from Firebase."""
        try:
            self.auth_token = response_data.get("idToken")
            self.refresh_token = response_data.get("refreshToken")

            if not self.auth_token or not self.refresh_token:
                self.login_failed.emit("Could not retrieve authentication credentials.")
                return

            # Extract user role from token
            self.user_role = get_user_role_from_token(self.auth_token)

            # Calculate token expiration
            expires_in = int(response_data.get("expiresIn", 3600))
            self.token_expires_at = time.time() + expires_in

            # Save tokens, expiration time, and email
            settings = QgsSettings()
            settings.setValue("needle/refreshToken", self.refresh_token)
            settings.setValue("needle/authToken", self.auth_token)
            settings.setValue("needle/tokenExpiresAt", str(self.token_expires_at))
            settings.setValue("needle/lastLoginEmail", self.last_login_email)

            # Setup token refresh timer - refresh at 50% of token lifetime for better persistence
            refresh_delay_ms = max(0, (expires_in // 2) * 1000)  # Refresh at halfway point (e.g., 30 min for 1hr token)
            self.token_refresh_timer.start(refresh_delay_ms)
            log_info(f"Token refresh scheduled in {refresh_delay_ms/1000:.0f} seconds ({refresh_delay_ms/60000:.1f} minutes)")

            self.login_success.emit()

        except Exception as e:
            error_msg = f"Login processing error: {e}"
            log_error(error_msg)
            self.login_failed.emit(error_msg)
    
    def _handle_refresh_response(self, response_data: Dict[str, Any]) -> None:
        """Handle token refresh response."""
        try:
            self.auth_token = response_data.get("access_token") or response_data.get("id_token")
            if response_data.get("refresh_token"):
                self.refresh_token = response_data.get("refresh_token")

            if self.auth_token:
                # Extract user role from refreshed token
                self.user_role = get_user_role_from_token(self.auth_token)

                expires_in = int(response_data.get("expires_in", 3600))
                self.token_expires_at = time.time() + expires_in

                log_info(f"Token refreshed successfully. New expiry: {expires_in}s from now")

                # Save updated tokens and expiration time
                settings = QgsSettings()
                settings.setValue("needle/refreshToken", self.refresh_token)
                settings.setValue("needle/authToken", self.auth_token)
                settings.setValue("needle/tokenExpiresAt", str(self.token_expires_at))

                # Schedule next refresh at 50% of lifetime for better persistence
                refresh_delay_ms = max(0, (expires_in // 2) * 1000)
                self.token_refresh_timer.start(refresh_delay_ms)
                log_info(f"Next token refresh scheduled in {refresh_delay_ms/1000:.0f} seconds")


                # Emit login_success signal to update UI
                self.login_success.emit()
            else:
                log_error("Token refresh failed: No token in response")
                # Emit login_failed signal to update UI
                self.login_failed.emit("Token refresh failed")

        except Exception as e:
            log_error(f"Token refresh processing error: {e}")
            # Emit login_failed signal to update UI
            self.login_failed.emit(f"Token refresh error: {e}")
    
    def _handle_api_response(self, endpoint: str, response_data, 
                           callback: Optional[Callable] = None) -> None:
        """Handle API response from Needle Digital service."""
        
        if callback:
            callback(response_data)
        
        # Only emit the signal if response_data is a dictionary
        # Some endpoints (like companies search) return lists directly
        if isinstance(response_data, dict):
            self.api_response_received.emit(endpoint, response_data)
