from qgis.core import (
    Qgis,
    QgsDataItemProvider,
    QgsDataCollectionItem,
    QgsDataItem,
    QgsDataProvider,
    QgsProject,
    QgsLayerTreeGroup,
    QgsApplication,
    QgsIconUtils,
    QgsVectorLayer, 
    QgsField, 
    QgsFields, 
    QgsGeometry,
    QgsFeature, 
    QgsProject, 
    Qgis, 
    QgsMessageLog
)
from .plugin_globals import PluginGlobals
from .plugin_icons import PluginIcons
from ..d4c_api_dialog_geoparams import Ui_GeoParams

from qgis.utils import iface
from qgis.gui import QgisInterface
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.Qt import QWidget
from qgis.PyQt.QtCore import QVariant
from PyQt5 import QtWidgets
# from .remote_platforms import RemotePlatforms
from qgis.PyQt.QtWidgets import QAction, QMenu
import os 
import json
import csv
import pathlib
import requests
from cryptography.fernet import Fernet

windowGeoParams = None
uiGeoParams = None

class d4cProvider(QgsDataItemProvider):

    def __init__(self, iface=iface):
        self.iface = iface
        QgsDataItemProvider.__init__(self)

    def name(self):
        return "D4C Provider"

    def capabilities(self):
        return QgsDataProvider.Net

    def createDataItem(self, path, parentItem):
        self.root = RootCollection(self.iface, parent = parentItem)
        return self.root


class RootCollection(QgsDataCollectionItem):
    def __init__(self, iface: QgisInterface, parent):
        self.iface = iface
        QgsDataCollectionItem.__init__(self, parent, "Data4Citizen", "/Data4Citizen")
        self.setIcon(
            QIcon(
                PluginGlobals.instance().plugin_path
                + "/icon.png"
            )
        )

    
    def createChildren(self):
        children = []
        for file in [GroupItem(parent=self,name= "Dernières ressources ajoutées", group =QgsLayerTreeGroup("Dernières ressources ajoutées")), GroupItem(parent=self,name= "Dernières connexions", group =QgsLayerTreeGroup("Dernières connexions"))]:
            children.append(file)
        return children
    
class GroupItem(QgsDataCollectionItem):
    def __init__(self, parent, name, group):
        self.path = "/Data4Citizen/{}".format(name)
        self.group = group
        QgsDataCollectionItem.__init__(self, parent, name, self.path)
        self.setIcon(QIcon(QgsApplication.iconPath("mIconFolder.svg")))

    def createChildren(self):
        
        if self.group.name() == "Dernières ressources ajoutées":
            return self.createLastFiles()
        elif self.group.name() == "Dernières connexions":
            
            return self.createLastConnections()
        else:
            return self.createConnexionChildrens()
            
    def createLastFiles(self):
        children = []
        config_file_path = os.path.join(PluginGlobals.instance().plugin_path, "dock_config","config.json") 
        with open(config_file_path, "r") as file:
            config = json.load(file)
        for file in config["children"][0]["children"]:
            children.append(FileItem(parent=self, name=file["title"],path=file["params"]["uri"],format=file["params"]["format"])) 
        return children
    
    def createLastConnections(self):
        children = []
        config_file_path = os.path.join(PluginGlobals.instance().plugin_path, "dock_config","config.json") 
        with open(config_file_path, "r") as file:
            config = json.load(file)
        for session in config["children"][1]["children"]:
            children.append(GroupItem(parent=self,name= session["title"], group =QgsLayerTreeGroup(session["title"]))) 
        return children

    def createConnexionChildrens(self):
        children = []
        config_file_path = os.path.join(PluginGlobals.instance().plugin_path, "dock_config","config.json") 
        with open(config_file_path, "r") as file:
            config = json.load(file)
        for session in config["children"][1]["children"]:
            if session["title"] == self.name():
                for dataset in session["children"]:
                    children.append(LayerItem(parent=self, name=dataset["title"],url=dataset["params"]["uri"], parentName = self.name())) 
                break
        return children

class FileItem(QgsDataItem):
    def __init__(self, parent, name, path, format):
        self.name = name    
        self.path = path
        self.format = format
        QgsDataItem.__init__(self, QgsDataItem.Custom, parent, name, self.path)
        self.setState(QgsDataItem.Populated)
        plugin_icons = PluginIcons.instance()
        if format == "csv":
            self.setIcon(plugin_icons.csv_icon)
        elif format == "json":
            self.setIcon(plugin_icons.json_icon)
        elif format == "geojson":
            self.setIcon(plugin_icons.geojson_icon)
        else :
            self.setIcon(plugin_icons.add_icon)
            

    def hasDragEnabled(self):
        return False

    def handleDoubleClick(self):
        self.addLayer()
        return True

    def hasChildren(self):
        return False

    def actions(self, parent):
        actions = list()
        add_d4c_action = QAction(QIcon(), self.tr("Choisir les colonnes géo"), parent)
        # Insérer la fonction ici
        add_d4c_action.triggered.connect(
            self.openGeoParams
            )
        # disable actions
        if self.format != "csv" :
            add_d4c_action.setDisabled(True)
            
        actions.append(add_d4c_action)
        return actions


    def addLayer(self):
        if self.format == "csv":
            importCSV(self.path, self.name)
        elif self.format == "json":
            importJSON(self.path, self.name)
        elif self.format == "geojson":
            importGEOJSON(self.path, self.name)


    def openGeoParams(self):
        self.windowGeoParams = QtWidgets.QDialog()
        self.uiGeoParams = Ui_GeoParams()
        self.uiGeoParams.setupUi(self.windowGeoParams)

        self.uiGeoParams.lineGeo2.setHidden(True)
        self.uiGeoParams.label_lon.setHidden(True)
        self.uiGeoParams.pushDeleteWord.setHidden(True)
        
        # Récupérer les headers 
        # Ouvrir le fichier et récupérer les headers : 
        with open(self.path, 'r') as file:
            reader = csv.reader(file)
            headers = next(reader)
            for header in headers:
                self.uiGeoParams.lineGeo1.addItem(header)
                self.uiGeoParams.lineGeo2.addItem(header)
        
        self.uiGeoParams.pushAddGeoParam.clicked.connect(self.addGeoParam)
        self.uiGeoParams.comboColumnType.currentIndexChanged.connect(self.geoParamsColumnTypeChanged)
        self.uiGeoParams.pushCancel.clicked.connect(self.windowGeoParams.close)
        
        if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt'):
                with open(str(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt'), 'w') as fichier:
                    fichier.write('')    

        self.windowGeoParams.exec_()


    def addGeoParam(self):
        if self.uiGeoParams.comboColumnType.currentIndex() == 0:
            
            if self.uiGeoParams.lineGeo1.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                # self.show_info_message(self.tr('Veuillez entrer un nom de colonne'))
                return
            
            # Mot ou tuple à ajouter
            geolabel = self.uiGeoParams.lineGeo1.currentText() 

            # Chemin du fichier d'énumération
            enum_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_geo.txt')
            
            # Lire les données actuelles du fichier
            with open(enum_file_path, 'r') as fichier:
                lignes = fichier.readlines()
            
            # Vérifier si le mot ou le tuple existe déjà
            if geolabel + '\n' not in lignes:
                # Ajouter le mot ou le tuple à la liste
                lignes.append(geolabel + '\n')
            

        if self.uiGeoParams.comboColumnType.currentIndex() == 1:

            if self.uiGeoParams.lineGeo1.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne pour la latitude !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
                
            if self.uiGeoParams.lineGeo2.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne pour la longitude !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
            
            # Mot ou tuple à ajouter
            if self.uiGeoParams.lineGeo1.currentText() == self.uiGeoParams.lineGeo2.currentText():
                short_message = "Veuillez entrer des noms de colonnes différents !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
            geolabel = (self.uiGeoParams.lineGeo1.currentText(), self.uiGeoParams.lineGeo2.currentText())
            enum_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_tuple_geo.txt')

            # Lire les données actuelles du fichier
            with open(enum_file_path, 'r') as fichier:
                lignes = fichier.readlines()

            # Convertir les chaînes en tuples
            tuples_existants = [tuple(ligne.strip().split(',')) for ligne in lignes]
            if geolabel not in tuples_existants:
                # Ajouter le tuple à la liste
                lignes.append(','.join(geolabel) + '\n')
            
        # Écrire les nouvelles données dans le fichier
        with open(enum_file_path, 'w') as fichier:
            fichier.writelines(lignes)
        short_message = "Colonne(s) ajoutée(s) !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        
    
    def geoParamsColumnTypeChanged(self):
        
        if self.uiGeoParams.comboColumnType.currentIndex() == 0:
            self.uiGeoParams.lineGeo2.setHidden(True)
            self.uiGeoParams.label_lon.setHidden(True)
            self.uiGeoParams.label_lat.setText("Nom de la colonne :")
        
        if self.uiGeoParams.comboColumnType.currentIndex() == 1:
            self.uiGeoParams.lineGeo2.setHidden(False)
            self.uiGeoParams.label_lon.setHidden(False)
            self.uiGeoParams.label_lat.setText("Nom de la colonne pour la latitude :")
            self.uiGeoParams.label_lon.setText("Nom de la colonne pour la longitude :")

class LayerItem(QgsDataItem):
    def __init__(self, parent, name, url, parentName):
        self.dataset_name = name
        self.url = url
        self.parentName = parentName
        
        QgsDataItem.__init__(self, QgsDataItem.Custom, parent, name, None)
        self.setState(QgsDataItem.Populated)  # no children
        plugin_icons = PluginIcons.instance()
        self.setIcon(plugin_icons.add_icon)


    def hasDragEnabled(self):
        return False

    def handleDoubleClick(self):
        self.addLayer()
        return True

    def hasChildren(self):
        return False
    
    def actions(self, parent):
        actions = list()
        add_d4c_action = QAction(QIcon(), self.tr("Choisir les colonnes géo"), parent)
        # Insérer la fonction ici
        add_d4c_action.triggered.connect(
            self.openGeoParams
        )
        actions.append(add_d4c_action)
        return actions

    def addLayer(self):
        self.download_dataset(self.url, self.dataset_name)
    
    def download_dataset(self, site_url, dataset_name):
        """ Get API call """
        key = read(pathlib.Path(__file__).resolve().parent.parent)
        user = None
        pwd = None
        sessions_file = str(pathlib.Path.home() / '.d4cplugin' / 'history.json')
        with open(sessions_file, 'r') as file:
            sessions = json.load(file)
        for session in sessions['last_session']['sessions']:
            if session['site_url'] == site_url and session['datasets'] == dataset_name:
                pwd = session['password']
                user = session['username']
                break

        if pwd.startswith('b\''):
            pwd = key.decrypt(pwd[1:-1])
            pwd = pwd.decode('utf-8')
        else:
            if pwd.endswith(' '):
                pwd = pwd[:-1]
            if pwd.startswith(' '):
                pwd = pwd[1:]

         

        if pwd == "":
            url = site_url + '/d4c/api/datasets/2.0/DATASETID/id='
        else:
            url = site_url + '/d4c/api/v1/dataset/find'

        headers = {
        'Content-Type': 'application/x-www-form-urlencoded',  # Specify the data format (optional)
        }
        try:
            if pwd == "":
                response = requests.get(url, headers=headers)
            else:
                response = requests.post(url, headers=headers, auth=(user, pwd), data={'dataset_id': dataset_name})
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return

        if response.status_code == 200:
            dataset = response.json()
            if pwd == "":
                if not "success" in dataset:
                    short_message = "Erreur de connexion vers l'API D4C !".format(
                        PluginGlobals.instance().PLUGIN_TAG
                    )
                    PluginGlobals.instance().iface.messageBar().pushMessage(
                        "Erreur", short_message, level=Qgis.Critical
                    )

                    long_message = "{0}\n".format(short_message)
                    QgsMessageLog.logMessage(
                        long_message,
                        tag=PluginGlobals.instance().PLUGIN_TAG,
                        level=Qgis.Critical,
                    )
                    return
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Impossible de trouver le fichier !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return
                url_file_csv = None
                url_file_json = None
                url_file_geojson = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            
                            if url_file_csv is None:
                                url_file_csv = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        elif resource['format'] == 'GeoJSON' or resource['name'].endswith('.geojson') or resource['mimetype'] == 'application/geo+json':
                            
                            if url_file_json is None:
                                url_file_json = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        elif resource['name'].endswith('.json') or resource['mimetype'] == 'application/json' or resource['format'] == 'JSON' or resource['format'] == 'json':
                            
                            if url_file_geojson is None:
                                url_file_geojson = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        else:
                            
                            short_message = "Aucune ressource trouvée !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
            else:
                if not dataset['status'] == 'success':
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Erreur de connexion vers l'API D4C !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return
                # check if key 'state' exists 
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        if dataset['result']['state'] == 'deleted': 
                            short_message = "Aucun jeu de donnée trouvé !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
                url_file_csv = None
                url_file_json = None
                url_file_geojson = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            if url_file_csv is None:
                                url_file_csv = resource['url']
                        elif resource['format'] == 'GeoJSON' or resource['name'].endswith('.geojson') or resource['mimetype'] == 'application/geo+json':
                            if url_file_geojson is None:
                                url_file_geojson = resource['url']
                        elif resource['name'].endswith('.json') or resource['mimetype'] == 'application/json' or resource['format'] == 'JSON' or resource['format'] == 'json':
                            if url_file_json is None:
                                url_file_json = resource['url']
                        else:
                            short_message = "Aucune ressource trouvée !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
                        
            # Import the resource (download the file and import it in QGIS)
            if url_file_csv is not None:
                self.importResource(url_file_csv, dataset_name, user, pwd)
            elif url_file_geojson is not None:
                self.importResource(url_file_geojson, dataset_name, user, pwd)
            elif url_file_json is not None:
                self.importResource(url_file_json, dataset_name, user, pwd)
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return
        
    def importResource(self, url_file, file_name, usr, pwd):
        if url_file.endswith('.json') and not(file_name.endswith('.json')):
            file_name = file_name + '.json'
        elif url_file.endswith('.csv') and not(file_name.endswith('.csv')):
            file_name = file_name + '.csv'
        elif url_file.endswith('.geojson') and not(file_name.endswith('.geojson')):
            file_name = file_name + '.geojson'    

        destination_path = str(pathlib.Path(__file__).resolve().parent.parent / 'resources' / file_name)
        try: 
            response = requests.get(url_file, auth=(usr, pwd))
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return 
                        
        if response.status_code == 200:
                    #write file content
                    with open(destination_path, 'wb') as f:
                        f.write(response.content)
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return

        #CSV import 
        if file_name.endswith('.csv'):
            importCSV(destination_path, file_name)
            
        #JSON import
        if file_name.endswith('.json'):
            importJSON(destination_path, file_name)

        #GeoJSON import
        if file_name.endswith('.geojson'):
            importGEOJSON(destination_path, file_name)

    def openGeoParams(self):
        self.windowGeoParams = QtWidgets.QDialog()
        self.uiGeoParams = Ui_GeoParams()
        self.uiGeoParams.setupUi(self.windowGeoParams)

        self.uiGeoParams.lineGeo2.setHidden(True)
        self.uiGeoParams.label_lon.setHidden(True)
        self.uiGeoParams.pushDeleteWord.setHidden(True)
        

        # Récupérer les headers 
        headers = self.call_api_to_get_headers(self.dataset_name, self.url)
        for header in headers: 
            self.uiGeoParams.lineGeo1.addItem(header)
            self.uiGeoParams.lineGeo2.addItem(header)
        
        
        self.uiGeoParams.pushAddGeoParam.clicked.connect(self.addGeoParam)
        self.uiGeoParams.comboColumnType.currentIndexChanged.connect(self.geoParamsColumnTypeChanged)
        self.uiGeoParams.pushCancel.clicked.connect(self.windowGeoParams.close)
        # self.uiGeoParams.pushDeleteWord.clicked.connect(openDeleteGeoParam)
        
        if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt'):
                with open(str(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt'), 'w') as fichier:
                    fichier.write('')    

        self.windowGeoParams.exec_()


    def addGeoParam(self):
        if self.uiGeoParams.comboColumnType.currentIndex() == 0:
            
            if self.uiGeoParams.lineGeo1.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
            
            # Mot ou tuple à ajouter
            geolabel = self.uiGeoParams.lineGeo1.currentText() 

            # Chemin du fichier d'énumération
            enum_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_geo.txt')
            
            # Lire les données actuelles du fichier
            with open(enum_file_path, 'r') as fichier:
                lignes = fichier.readlines()
            
            # Vérifier si le mot ou le tuple existe déjà
            if geolabel + '\n' not in lignes:
                # Ajouter le mot ou le tuple à la liste
                lignes.append(geolabel + '\n')
            
            # Ajouter dans le cache
            cache_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt')

            # Lire les données actuelles du fichier
            with open(cache_file_path, 'r') as fichier:
                lignes_cache = fichier.readlines()

            
            # Former le cache 
            cache = ('0' + '!!' 
                    + self.dataset_name + '!!'
                    +  self.dataset_name + ".csv" + '!!'
                    + self.url + '!!'
                    + geolabel + '\n')
            
        
            # Écrire les nouvelles données dans le fichier
            with open(cache_file_path, 'w') as fichier:
                if lignes_cache:
                    for ligne in lignes_cache:

                        if cache.split('!!')[1] == ligne.split('!!')[1] and cache.split('!!')[2] == ligne.split('!!')[2] and cache.split('!!')[3] == ligne.split('!!')[3]:
                            lignes_cache.remove(ligne)
                    if lignes_cache:
                        fichier.writelines(lignes_cache)
                        fichier.writelines(cache)
                    else:
                        fichier.writelines(cache)
                else:
                    fichier.writelines(cache)
                    


        if self.uiGeoParams.comboColumnType.currentIndex() == 1:

            if self.uiGeoParams.lineGeo1.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne pour la latitude !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
                
            if self.uiGeoParams.lineGeo2.currentText() == '':
                short_message = "Veuillez entrer un nom de colonne pour la longitude !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
            
            # Mot ou tuple à ajouter
            if self.uiGeoParams.lineGeo1.currentText() == self.uiGeoParams.lineGeo2.currentText():
                short_message = "Veuillez entrer des noms de colonnes différents !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return
            geolabel = (self.uiGeoParams.lineGeo1.currentText(), self.uiGeoParams.lineGeo2.currentText())
            enum_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_tuple_geo.txt')

            # Lire les données actuelles du fichier
            with open(enum_file_path, 'r') as fichier:
                lignes = fichier.readlines()

            # Convertir les chaînes en tuples
            tuples_existants = [tuple(ligne.strip().split(',')) for ligne in lignes]
            if geolabel not in tuples_existants:
                # Ajouter le tuple à la liste
                lignes.append(','.join(geolabel) + '\n')
            
            # Ajouter dans le cache
            cache_file_path = str(pathlib.Path.home() / '.d4cplugin' /  'enums_cache_jdd.txt')

            # Lire les données actuelles du fichier
            with open(cache_file_path, 'r') as fichier:
                lignes_cache = fichier.readlines()
                
            # Former le cache 
            cache = ('0' + '!!' 
                    + self.dataset_name + '!!'
                    +  self.dataset_name + ".csv" + '!!'
                    + self.url + '!!'
                    + geolabel[0] + ',' + geolabel[1] + '\n')
            
            # Écrire les nouvelles données dans le fichier
            with open(cache_file_path, 'w') as fichier:
                if lignes_cache:
                    for ligne in lignes_cache:
                        if cache.split('!!')[1] == ligne.split('!!')[1] and cache.split('!!')[2] == ligne.split('!!')[2] and cache.split('!!')[3] == ligne.split('!!')[3]:
                            lignes_cache.remove(ligne)
                    if lignes_cache:
                        fichier.writelines(lignes_cache)
                        fichier.writelines(cache)
                    else:
                        fichier.writelines(cache)
                else:
                    fichier.writelines(cache)
            
        # Écrire les nouvelles données dans le fichier
        with open(enum_file_path, 'w') as fichier:
            fichier.writelines(lignes)
        short_message = "Colonne(s) ajoutée(s) !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        
    
    def geoParamsColumnTypeChanged(self):
        
        if self.uiGeoParams.comboColumnType.currentIndex() == 0:
            self.uiGeoParams.lineGeo2.setHidden(True)
            self.uiGeoParams.label_lon.setHidden(True)
            self.uiGeoParams.label_lat.setText("Nom de la colonne :")
        
        if self.uiGeoParams.comboColumnType.currentIndex() == 1:
            self.uiGeoParams.lineGeo2.setHidden(False)
            self.uiGeoParams.label_lon.setHidden(False)
            self.uiGeoParams.label_lat.setText("Nom de la colonne pour la latitude :")
            self.uiGeoParams.label_lon.setText("Nom de la colonne pour la longitude :")

    def call_api_to_get_headers(self, dataset_name, site_url):
        
        key = read(pathlib.Path(__file__).resolve().parent.parent)
        user = None
        pwd = None
        sessions_file = str(pathlib.Path.home() / '.d4cplugin' / 'history.json')
        with open(sessions_file, 'r') as file:
            sessions = json.load(file)
        for session in sessions['last_session']['sessions']:
            if session['site_url'] == site_url and session['datasets'] == dataset_name:
                pwd = session['password']
                user = session['username']
                break

        if pwd.startswith('b\''):
            pwd = key.decrypt(pwd[1:-1])
            pwd = pwd.decode('utf-8')
        else:
            if pwd.endswith(' '):
                pwd = pwd[:-1]
            if pwd.startswith(' '):
                pwd = pwd[1:]

         

        if pwd == "":
            url = site_url + '/d4c/api/datasets/2.0/DATASETID/id='
        else:
            url = site_url + '/d4c/api/v1/dataset/find'

        headers = {
        'Content-Type': 'application/x-www-form-urlencoded',  # Specify the data format (optional)
        }
        try:
            if pwd == "":
                response = requests.get(url, headers=headers)
            else:
                response = requests.post(url, headers=headers, auth=(user, pwd), data={'dataset_id': dataset_name})
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return []

        if response.status_code == 200:
            dataset = response.json()
            if pwd == "":
                if not "success" in dataset:
                    short_message = "Erreur de connexion vers l'API D4C !".format(
                        PluginGlobals.instance().PLUGIN_TAG
                    )
                    PluginGlobals.instance().iface.messageBar().pushMessage(
                        "Erreur", short_message, level=Qgis.Critical
                    )

                    long_message = "{0}\n".format(short_message)
                    QgsMessageLog.logMessage(
                        long_message,
                        tag=PluginGlobals.instance().PLUGIN_TAG,
                        level=Qgis.Critical,
                    )
                    return []
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Impossible de trouver le fichier !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return []
                url_file_csv = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            
                            if url_file_csv is None:
                                url_file_csv = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']

            else:
                if not dataset['status'] == 'success':
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Erreur de connexion vers l'API D4C !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return []
                # check if key 'state' exists 
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        if dataset['result']['state'] == 'deleted': 
                            short_message = "Aucun jeu de donnée trouvé !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return []
                url_file_csv = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            if url_file_csv is None:
                                url_file_csv = resource['url']
                        
            # Import the resource (download the file and import it in QGIS)
            if url_file_csv is not None:
                return self.get_api_csv_headers(url_file_csv, user, pwd)
            else:
                short_message = "Aucune ressource CSV trouvée !".format(
                    PluginGlobals.instance().PLUGIN_TAG
                )
                PluginGlobals.instance().iface.messageBar().pushMessage(
                    "Erreur", short_message, level=Qgis.Critical
                )

                long_message = "{0}\n".format(short_message)
                QgsMessageLog.logMessage(
                    long_message,
                    tag=PluginGlobals.instance().PLUGIN_TAG,
                    level=Qgis.Critical,
                )
                return []
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return []
    def get_api_csv_headers(self, url, usr,pwd):
        try: 
            response = requests.get(url, auth=(usr, pwd))
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return []
                        
        if response.status_code == 200:
            #get headers
            csv_file = response.content
            csv_file = csv_file.decode('utf-8')
            csv_file = csv_file.split('\n')
            headers = csv_file[0].split(detect_delimiter(csv_file[0]))
            return headers
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return []

def importCSV(destination_path, file_name):
    """" Import a CSV file into QGIS as a vector layer 
        destination_path : the path of the file to import
    """
    layer_name = file_name
    vl = None
    csv_layer = QgsVectorLayer(destination_path, layer_name, "ogr")
    # Lire les énumérations de strings et créer les fichiers s'il n'existent pas
    if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'):
        with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'), 'w') as file:
            file.write('')

    if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'):
        with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'), 'w') as file:
            file.write('')

    with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'), 'r') as file:
        enum_strings = [line.strip() for line in file]

    # Lire les énumérations de tuples
    with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'), 'r') as file:
        enum_tuples = [tuple(line.strip().split(',')) for line in file]

    if not csv_layer.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(csv_layer)
        
        csv_data = []  # Créez une liste pour stocker les données CSV

        # Ouvrez le fichier CSV et lisez ses données
        try:
            with open(destination_path, 'r', encoding='utf-8', errors='replace') as csv_file:
                first_line = csv_file.readline()
                # go back to the beginning of the file
                csv_file.seek(0)
                csv_reader = csv.DictReader(csv_file , delimiter=detect_delimiter(first_line))
                for row in csv_reader:
                    csv_data.append(row)
            # Examinez les données du CSV et vérifiez si une colonne "geoshape" existe
            # Si oui, extrayez les données géospatiales et créez la couche
            if 'geo_shape' in csv_data[0] or 'geojson' in csv_data[0]:
                if 'geo_shape' in csv_data[0]:
                    colname = 'geo_shape'
                else:
                    colname = 'geojson'
                
                # Créez une couche vectorielle vide
                fields = QgsFields()
                for key, value in csv_data[0].items():
                    #if key != colname:
                    fields.append(QgsField(key, QVariant.String))
                if json.loads(row[colname]).get("type") == "Polygon":
                    vl = QgsVectorLayer("Polygon?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "Point":
                    vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "LineString":
                    vl = QgsVectorLayer("LineString?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiPolygon":
                    vl = QgsVectorLayer("MultiPolygon?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiLineString":
                    vl = QgsVectorLayer("MultiLineString?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiPoint":
                    vl = QgsVectorLayer("MultiPoint?crs=EPSG:4326", f"{layer_name}", "memory")
                vl.dataProvider().addAttributes(fields)
                vl.updateFields()
                vl.startEditing()
                # Parcourez les données du CSV, extrayez les données WKT de la colonne "geoshape"
                # Créez les entités géométriques et ajoutez-les à la couche
                i = 1
                for row in csv_data:
                    
                    wkt_data = row[colname]
                    if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                        wkt_data = geojson_to_wkt(json.loads(wkt_data))
                    
                        geometry = QgsGeometry.fromWkt(wkt_data)
                        if not geometry.isEmpty():
                            feature = QgsFeature()
                            feature.setGeometry(geometry)   
                            vl.dataProvider().addFeature(feature)
                            feature = vl.getFeature(i)
                            # Ajoutez les valeurs des autres colonnes à l'entité
                            for key, value in row.items():
                                #if key != colname:
                                    # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                    # Définissez la valeur du champ
                                vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                    i += 1

                vl.commitChanges()   

            
            else:
                # Variable qui permet de savoir si une colonne pour les coordonnées existe
                has_geo = False
                pair_geo = None
                columnName = None


                if not has_geo: 
                    # On cherche si une colonne pour les coordonnées existe dans l'énum
                    for columnGeoName in enum_strings:
                        if columnGeoName in csv_data[0]:
                            columnName = columnGeoName
                            has_geo = True
                        # Créez une couche vectorielle vide
                        
                        if has_geo:
                            fields = QgsFields()
                            for key, value in csv_data[0].items():
                                #if key != columnName:
                                fields.append(QgsField(key, QVariant.String))
                            vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                            vl.dataProvider().addAttributes(fields)
                            vl.updateFields()
                            vl.startEditing()

                            i = 1
                            for row in csv_data:

                                wkt_data = row[columnName]
                                if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                    wkt_data = geojson_to_wkt(list(eval(wkt_data)))
                                    
                                    geometry = QgsGeometry.fromWkt(wkt_data)
                                    if not geometry.isEmpty():
                                        feature = QgsFeature()
                                        feature.setGeometry(geometry)   
                                        vl.dataProvider().addFeature(feature)
                                        feature = vl.getFeature(i)
                                        # Ajoutez les valeurs des autres colonnes à l'entité
                                        for key, value in row.items():
                                            #if key != columnName:
                                                # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                                # Définissez la valeur du champ
                                            vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                                i += 1

                            vl.commitChanges() 
                            break
                else:
                    if columnName is not None:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            #if key != columnName:
                            fields.append(QgsField(key, QVariant.String))
                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:

                            wkt_data = row[columnName]
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(list(eval(wkt_data)))
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        #if key != columnName:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()
                    else:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            # if key != pair_geo[0] and key != pair_geo[1]:
                            fields.append(QgsField(key, QVariant.String))


                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:
                            wkt_data = [row[pair_geo[0]].replace(',','.'), row[pair_geo[1]].replace(',','.')]
                            
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(wkt_data)
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        # if key != pair_geo[0] and key != pair_geo[1]:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()

                # On cherche si une colonne pour les coordonnées existe en 2 colonne (longitude et latitude)
                if not has_geo:
                    pair_geo = None
                    for pair in enum_tuples:
                        if pair[0] in csv_data[0] and pair[1] in csv_data[0]:
                            pair_geo = pair
                            break
                    # si une pair est trouvée
                    if pair_geo is not None:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            #if key != pair_geo[0] and key != pair_geo[1]:
                            fields.append(QgsField(key, QVariant.String))


                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:
                            wkt_data = [row[pair_geo[0]].replace(',','.'), row[pair_geo[1]].replace(',','.')]
                            
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(wkt_data)
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        #if key != pair_geo[0] and key != pair_geo[1]:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()
                        
        except Exception as e:
            # Si une erreur se produit lors de l'ouverture du fichier, imprimer l'erreur
            short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )

        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        
    if vl is not None:
        # Ajoutez la couche au projet QGIS
        QgsProject.instance().addMapLayer(vl)
        
        # Enlever le fichier csv qui n'a pas de géométrie
        for layer in QgsProject.instance().mapLayers().values():
            if layer.name() == layer_name and str(layer.geometryType()) == 'GeometryType.Null':
                QgsProject.instance().removeMapLayer(layer)
                break
    
def geojson_to_wkt( geojson):
    if type(geojson) is list:
        coordinates = geojson
        if coordinates:
            # Format WKT en utilisant les coordonnées GeoJSON
            wkt = f"POINT({coordinates[1]} {coordinates[0]})"
            return wkt
    else:
        # Vérifie si le type est "Polygon"
        if geojson.get("type") == "Polygon":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "POLYGON(("
                for ring in coordinates:
                    for coord in ring:
                        wkt += f"{coord[0]} {coord[1]}, "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + "))"
                return wkt
        # Vérifie si le type est "Point"
        elif geojson.get("type") == "Point":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = f"POINT({coordinates[0]} {coordinates[1]})"
                return wkt  
        # Vérifie si le type est "LineString"
        elif geojson.get("type") == "LineString":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "LINESTRING("
                for coord in coordinates:
                    wkt += f"{coord[0]} {coord[1]}, "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + ")"
                return wkt
        # Vérifie si le type est "MultiPolygon"
        elif geojson.get("type") == "MultiPolygon":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "MULTIPOLYGON("
                for polygon in coordinates:
                    wkt += "(("
                    for ring in polygon:
                        for coord in ring:
                            wkt += f"{coord[0]} {coord[1]}, "
                    # Supprime la virgule finale et ajoute la parenthèse fermante
                    wkt = wkt[:-2] + ")), "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + ")"
                return wkt
            
        return None
    

# Détecter le délimiteur d'un fichier CSV (, ou ;)
def detect_delimiter(text):
    if ',' in text:
        return ','
    elif ';' in text:
        return ';'
    else:
        return ','

def importJSON(destination_path,file_name):

    layer_name = file_name
        
    json_layer = QgsVectorLayer(destination_path, layer_name, "ogr")

    if not json_layer.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(json_layer)
        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        


def importGEOJSON(destination_path,file_name):
    layer_name = file_name

    v1 = QgsVectorLayer(
        f"{destination_path}", 
        layer_name,
        "ogr"
    ) 

    if not v1.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(v1)
        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )

def read(plugin_dir):

    with open(str(plugin_dir / 'help'/'source'/'_static'/'const'/'temp'/'k_e_y_p_w_d.txt'), 'rb') as file:
        for line in file:
            key = line
        
    fernet = Fernet(key[2:-1])
    return fernet    


