# -*- coding: utf-8 -*-

import os
import pathlib
import csv
import json
import requests
from cryptography.fernet import Fernet
from qgis.PyQt.QtCore import QVariant
from qgis.core import QgsVectorLayer, QgsField, QgsFields, QgsGeometry, QgsFeature, QgsProject, Qgis, QgsMessageLog


from d4c_api.utils.plugin_globals import PluginGlobals
from d4c_api.utils.plugin_icons import PluginIcons


class FavoritesTreeNode:
    """ """

    def __init__(
        self,
        title,
        node_type=PluginGlobals.instance().NODE_TYPE_FOLDER,
        description=None,
        status=None,
        metadata_url=None,
        ident=None,
        params=None,
        bounding_boxes=None,
        parent_node=None,
    ):
        """ """

        self.parent_node = parent_node
        self.node_type = node_type
        self.title = title
        self.description = description
        self.status = status
        self.metadata_url = metadata_url
        self.ident = ident
        self.bounding_boxes = bounding_boxes
        self.children = []
        self.can_be_added_to_map = False
        self.icon = None

    def get_file_title_and_uri(self):
        """
        Return the mime data used by the drag and drop process
        and needed by QGIS to add the right layer to the map
        """

        # qgis_layer_details = self.get_qgis_layer_details()
        # mime_data = ":".join(
        #     [
        #         qgis_layer_details["type"],
        #         qgis_layer_details["provider"],
        #         qgis_layer_details["title"].replace(":", "\\:"),
        #         qgis_layer_details["uri"].replace(":", "\\:"),
        #     ]
        # )

        return self.get_file_title_and_uri()

    def run_add_to_map_action(self):
        """ """

        pass

    def run_show_metadata_action(self):
        """
        Opens in the default user web browser the web page displaying the resource metadata
        """

        import webbrowser

        if self.metadata_url:
            webbrowser.open_new_tab(self.metadata_url)

    def run_report_issue_action(self):
        """
        Opens the default mail client to let the user send an issue report by email
        """

        # import webbrowser
        # webbrowser.open('mailto:')
        pass


# Subclasses of FavoritesTreeNode


class FolderTreeNode(FavoritesTreeNode):
    """
    Tree node for a folder containing other nodes
    """

    def __init__(
        self,
        title,
        node_type=PluginGlobals.instance().NODE_TYPE_FOLDER,
        description=None,
        status=None,
        metadata_url=None,
        ident=None,
        params=None,
        bounding_boxes=None,
        parent_node=None,
    ):
        """ """

        FavoritesTreeNode.__init__(
            self,
            title,
            node_type,
            description,
            status,
            metadata_url,
            ident,
            params,
            bounding_boxes,
            parent_node,
        )

        # Icon
        plugin_icons = PluginIcons.instance()
        self.icon = plugin_icons.folder_icon
        if self.status == PluginGlobals.instance().NODE_STATUS_WARN:
            self.icon = plugin_icons.warn_icon

class FileTreeNode(FavoritesTreeNode):
    """
    Tree node for a file
    """
    def __init__(
        self,
        title,
        node_type=PluginGlobals.instance().NODE_TYPE_FILE,
        description=None,
        status=None,
        metadata_url=None,
        ident=None,
        params=None,
        bounding_boxes=None,
        parent_node=None,
    ):
        """ """
        FavoritesTreeNode.__init__(
            self,
            title,
            node_type,
            description,
            status,
            metadata_url,
            ident,
            params,
            bounding_boxes,
            parent_node,
        )

        self.uri = params.get("uri")
        self.layer_name = params.get("name")
        self.layer_format = params.get("format")
        self.layer_srs = params.get("srs")
        self.layer_style_name = params.get("style", "")
        self.can_be_added_to_map = True

        # Icon
        plugin_icons = PluginIcons.instance()
        if self.layer_format == "csv":
            self.icon = plugin_icons.csv_icon
        elif self.layer_format == "json":
            self.icon = plugin_icons.json_icon
        elif self.layer_format == "geojson":
            self.icon = plugin_icons.geojson_icon
        else: 
            self.icon = plugin_icons.add_icon

        if self.status == PluginGlobals.instance().NODE_STATUS_WARN:
            self.icon = plugin_icons.warn_icon

    def get_file_title_and_uri(self):
        """
        Return the details of the layer used by QGIS to add the layer to the map.
        This dictionary is used by the run_add_to_map_action and layerMimeData methods.
        """
        # qgis_layer_uri_details = {
        #     "type": "layer",
        #     "provider": "ogr",
        #     "title": self.title,
        #     "uri": u"crs={}&featureCount=10&format={}&layers={}&maxHeight=256&maxWidth=256&styles={}&url={}".format(
        #         self.layer_srs,
        #         self.layer_format,
        #         self.layer_name,
        #         self.layer_style_name,
        #         self.service_url,
        #     ),
        # }

        return self.title, self.uri

    def run_add_to_map_action(self):
        """
        Add the file to the layers panel
        """
        if self.layer_format == "csv":

            importCSV(self.uri, self.layer_name)
        elif self.layer_format == "json":
            importJSON(self.uri, self.layer_name)
        elif self.layer_format == "geojson":
            importGEOJSON(self.uri, self.layer_name)
        # qgis_layer_details = self.get_qgis_layer_details()
        # PluginGlobals.instance().iface.addRasterLayer(
        #     qgis_layer_details["uri"],
        #     qgis_layer_details["title"],
        #     qgis_layer_details["provider"],
        # )


class DatasetTreeNode(FavoritesTreeNode):
    """
    Tree node for a dataset
    """
    def __init__(
        self,
        title,
        node_type=PluginGlobals.instance().NODE_TYPE_DATASET,
        description=None,
        status=None,
        metadata_url=None,
        ident=None,
        params=None,
        bounding_boxes=None,
        parent_node=None,
    ):
        """ """
        FavoritesTreeNode.__init__(
            self,
            title,
            node_type,
            description,
            status,
            metadata_url,
            ident,
            params,
            bounding_boxes,
            parent_node,
        )

        self.uri = params.get("uri")
        self.layer_name = params.get("name")
        self.layer_format = params.get("format")
        self.layer_srs = params.get("srs")
        self.layer_style_name = params.get("style", "")
        self.can_be_added_to_map = True

        # Icon
        plugin_icons = PluginIcons.instance()
        self.icon = plugin_icons.add_icon

        if self.status == PluginGlobals.instance().NODE_STATUS_WARN:
            self.icon = plugin_icons.warn_icon

    def get_file_title_and_uri(self):
        """
        Return the details of the layer used by QGIS to add the layer to the map.
        This dictionary is used by the run_add_to_map_action and layerMimeData methods.
        """
        # qgis_layer_uri_details = {
        #     "type": "layer",
        #     "provider": "ogr",
        #     "title": self.title,
        #     "uri": u"crs={}&featureCount=10&format={}&layers={}&maxHeight=256&maxWidth=256&styles={}&url={}".format(
        #         self.layer_srs,
        #         self.layer_format,
        #         self.layer_name,
        #         self.layer_style_name,
        #         self.service_url,
        #     ),
        # }

        return self.title, self.uri

    def run_add_to_map_action(self):
        """
        call API and get the dataset resource
        """
        self.download_dataset(self.uri, self.layer_name)

    def download_dataset(self, site_url, dataset_name):
        """ Get API call """
        key = read(pathlib.Path(__file__).resolve().parent.parent)
        user = None
        pwd = None
        sessions_file = str(pathlib.Path.home() / '.d4cplugin' / 'history.json')
        with open(sessions_file, 'r') as file:
            sessions = json.load(file)
        for session in sessions['last_session']['sessions']:
            if session['site_url'] == site_url and session['datasets'] == dataset_name:
                pwd = session['password']
                user = session['username']
                break

        if pwd.startswith('b\''):
            pwd = key.decrypt(pwd[1:-1])
            pwd = pwd.decode('utf-8')
        else:
            if pwd.endswith(' '):
                pwd = pwd[:-1]
            if pwd.startswith(' '):
                pwd = pwd[1:]

         

        if pwd == "":
            url = site_url + '/d4c/api/datasets/2.0/DATASETID/id='
        else:
            url = site_url + '/d4c/api/v1/dataset/find'

        headers = {
        'Content-Type': 'application/x-www-form-urlencoded',  # Specify the data format (optional)
        }
        try:
            if pwd == "":
                response = requests.get(url, headers=headers)
            else:
                response = requests.post(url, headers=headers, auth=(user, pwd), data={'dataset_id': dataset_name})
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return

        if response.status_code == 200:
            dataset = response.json()
            if pwd == "":
                if not "success" in dataset:
                    short_message = "Erreur de connexion vers l'API D4C !".format(
                        PluginGlobals.instance().PLUGIN_TAG
                    )
                    PluginGlobals.instance().iface.messageBar().pushMessage(
                        "Erreur", short_message, level=Qgis.Critical
                    )

                    long_message = "{0}\n".format(short_message)
                    QgsMessageLog.logMessage(
                        long_message,
                        tag=PluginGlobals.instance().PLUGIN_TAG,
                        level=Qgis.Critical,
                    )
                    return
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Impossible de trouver le fichier !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return
                url_file_csv = None
                url_file_json = None
                url_file_geojson = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            
                            if url_file_csv is None:
                                url_file_csv = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        elif resource['format'] == 'GeoJSON' or resource['name'].endswith('.geojson') or resource['mimetype'] == 'application/geo+json':
                            
                            if url_file_json is None:
                                url_file_json = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        elif resource['name'].endswith('.json') or resource['mimetype'] == 'application/json' or resource['format'] == 'JSON' or resource['format'] == 'json':
                            
                            if url_file_geojson is None:
                                url_file_geojson = dataset['result']['url'] + '/resource/' + resource['id'] + '/download/' + resource['name']
                        else:
                            
                            short_message = "Aucune ressource trouvée !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
            else:
                if not dataset['status'] == 'success':
                    if dataset['result']['state'] == 'deleted': 
                        short_message = "Erreur de connexion vers l'API D4C !".format(
                            PluginGlobals.instance().PLUGIN_TAG
                        )
                        PluginGlobals.instance().iface.messageBar().pushMessage(
                            "Erreur", short_message, level=Qgis.Critical
                        )

                        long_message = "{0}\n".format(short_message)
                        QgsMessageLog.logMessage(
                            long_message,
                            tag=PluginGlobals.instance().PLUGIN_TAG,
                            level=Qgis.Critical,
                        )
                        return
                # check if key 'state' exists 
                if 'state' in dataset['result'].keys():
                    if dataset['result']['state'] == 'deleted': 
                        if dataset['result']['state'] == 'deleted': 
                            short_message = "Aucun jeu de donnée trouvé !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
                url_file_csv = None
                url_file_json = None
                url_file_geojson = None
                for resource in dataset['result']['resources']:
                    if resource['mimetype'] != "":
                        if resource['format'] == 'CSV' or resource['mimetype'] == 'text/csv' or resource['name'].endswith('.csv'):
                            if url_file_csv is None:
                                url_file_csv = resource['url']
                        elif resource['format'] == 'GeoJSON' or resource['name'].endswith('.geojson') or resource['mimetype'] == 'application/geo+json':
                            if url_file_geojson is None:
                                url_file_geojson = resource['url']
                        elif resource['name'].endswith('.json') or resource['mimetype'] == 'application/json' or resource['format'] == 'JSON' or resource['format'] == 'json':
                            if url_file_json is None:
                                url_file_json = resource['url']
                        else:
                            short_message = "Aucune ressource trouvée !".format(
                                PluginGlobals.instance().PLUGIN_TAG
                            )
                            PluginGlobals.instance().iface.messageBar().pushMessage(
                                "Erreur", short_message, level=Qgis.Critical
                            )

                            long_message = "{0}\n".format(short_message)
                            QgsMessageLog.logMessage(
                                long_message,
                                tag=PluginGlobals.instance().PLUGIN_TAG,
                                level=Qgis.Critical,
                            )
                            return
                        
            # Import the resource (download the file and import it in QGIS)
            if url_file_csv is not None:
                self.importResource(url_file_csv, dataset_name, user, pwd)
            elif url_file_geojson is not None:
                self.importResource(url_file_geojson, dataset_name, user, pwd)
            elif url_file_json is not None:
                self.importResource(url_file_json, dataset_name, user, pwd)
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return

                        
    def importResource(self, url_file, file_name, usr, pwd):
        if url_file.endswith('.json') and not(file_name.endswith('.json')):
            file_name = file_name + '.json'
        elif url_file.endswith('.csv') and not(file_name.endswith('.csv')):
            file_name = file_name + '.csv'
        elif url_file.endswith('.geojson') and not(file_name.endswith('.geojson')):
            file_name = file_name + '.geojson'    

        destination_path = str(pathlib.Path(__file__).resolve().parent.parent / 'resources' / file_name)
        try: 
            response = requests.get(url_file, auth=(usr, pwd))
        except requests.exceptions.RequestException as e:
            short_message = "Erreur de connexion !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return 
                        
        if response.status_code == 200:
                    #write file content
                    with open(destination_path, 'wb') as f:
                        f.write(response.content)
        else:
            short_message = "Erreur de connexion".format(
                PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n".format(short_message, response.status_code)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )
            return

        #CSV import 
        if file_name.endswith('.csv'):
            importCSV(destination_path, file_name)
            
        #JSON import
        if file_name.endswith('.json'):
            importJSON(destination_path, file_name)

        #GeoJSON import
        if file_name.endswith('.geojson'):
            importGEOJSON(destination_path, file_name)



def importCSV(destination_path, file_name):
    """" Import a CSV file into QGIS as a vector layer 
        destination_path : the path of the file to import
    """
    layer_name = file_name
    vl = None
    csv_layer = QgsVectorLayer(destination_path, layer_name, "ogr")
    # Lire les énumérations de strings et créer les fichiers s'il n'existent pas
    if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'):
        with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'), 'w') as file:
            file.write('')

    if not pathlib.Path.exists(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'):
        with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'), 'w') as file:
            file.write('')

    with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_geo.txt'), 'r') as file:
        enum_strings = [line.strip() for line in file]

    # Lire les énumérations de tuples
    with open(str(pathlib.Path.home() / '.d4cplugin' / 'enums_tuple_geo.txt'), 'r') as file:
        enum_tuples = [tuple(line.strip().split(',')) for line in file]

    if not csv_layer.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(csv_layer)
        
        csv_data = []  # Créez une liste pour stocker les données CSV

        # Ouvrez le fichier CSV et lisez ses données
        try:
            with open(destination_path, 'r', encoding='utf-8', errors='replace') as csv_file:
                first_line = csv_file.readline()
                # go back to the beginning of the file
                csv_file.seek(0)
                csv_reader = csv.DictReader(csv_file , delimiter=detect_delimiter(first_line))
                for row in csv_reader:
                    csv_data.append(row)
            # Examinez les données du CSV et vérifiez si une colonne "geoshape" existe
            # Si oui, extrayez les données géospatiales et créez la couche
            if 'geo_shape' in csv_data[0] or 'geojson' in csv_data[0]:
                if 'geo_shape' in csv_data[0]:
                    colname = 'geo_shape'
                else:
                    colname = 'geojson'
                
                # Créez une couche vectorielle vide
                fields = QgsFields()
                for key, value in csv_data[0].items():
                    #if key != colname:
                    fields.append(QgsField(key, QVariant.String))
                if json.loads(row[colname]).get("type") == "Polygon":
                    vl = QgsVectorLayer("Polygon?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "Point":
                    vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "LineString":
                    vl = QgsVectorLayer("LineString?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiPolygon":
                    vl = QgsVectorLayer("MultiPolygon?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiLineString":
                    vl = QgsVectorLayer("MultiLineString?crs=EPSG:4326", f"{layer_name}", "memory")
                elif json.loads(row[colname]).get("type") == "MultiPoint":
                    vl = QgsVectorLayer("MultiPoint?crs=EPSG:4326", f"{layer_name}", "memory")
                vl.dataProvider().addAttributes(fields)
                vl.updateFields()
                vl.startEditing()
                # Parcourez les données du CSV, extrayez les données WKT de la colonne "geoshape"
                # Créez les entités géométriques et ajoutez-les à la couche
                i = 1
                for row in csv_data:
                    
                    wkt_data = row[colname]
                    if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                        wkt_data = geojson_to_wkt(json.loads(wkt_data))
                    
                        geometry = QgsGeometry.fromWkt(wkt_data)
                        if not geometry.isEmpty():
                            feature = QgsFeature()
                            feature.setGeometry(geometry)   
                            vl.dataProvider().addFeature(feature)
                            feature = vl.getFeature(i)
                            # Ajoutez les valeurs des autres colonnes à l'entité
                            for key, value in row.items():
                                #if key != colname:
                                    # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                    # Définissez la valeur du champ
                                vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                    i += 1

                vl.commitChanges()   

            
            else:
                # Variable qui permet de savoir si une colonne pour les coordonnées existe
                has_geo = False
                pair_geo = None
                columnName = None


                if not has_geo: 
                    # On cherche si une colonne pour les coordonnées existe dans l'énum
                    for columnGeoName in enum_strings:
                        if columnGeoName in csv_data[0]:
                            columnName = columnGeoName
                            has_geo = True
                        # Créez une couche vectorielle vide
                        
                        if has_geo:
                            fields = QgsFields()
                            for key, value in csv_data[0].items():
                                #if key != columnName:
                                fields.append(QgsField(key, QVariant.String))
                            vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                            vl.dataProvider().addAttributes(fields)
                            vl.updateFields()
                            vl.startEditing()

                            i = 1
                            for row in csv_data:

                                wkt_data = row[columnName]
                                if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                    wkt_data = geojson_to_wkt(list(eval(wkt_data)))
                                    
                                    geometry = QgsGeometry.fromWkt(wkt_data)
                                    if not geometry.isEmpty():
                                        feature = QgsFeature()
                                        feature.setGeometry(geometry)   
                                        vl.dataProvider().addFeature(feature)
                                        feature = vl.getFeature(i)
                                        # Ajoutez les valeurs des autres colonnes à l'entité
                                        for key, value in row.items():
                                            #if key != columnName:
                                                # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                                # Définissez la valeur du champ
                                            vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                                i += 1

                            vl.commitChanges() 
                            break
                else:
                    if columnName is not None:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            #if key != columnName:
                            fields.append(QgsField(key, QVariant.String))
                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:

                            wkt_data = row[columnName]
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(list(eval(wkt_data)))
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        #if key != columnName:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()
                    else:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            # if key != pair_geo[0] and key != pair_geo[1]:
                            fields.append(QgsField(key, QVariant.String))


                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:
                            wkt_data = [row[pair_geo[0]].replace(',','.'), row[pair_geo[1]].replace(',','.')]
                            
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(wkt_data)
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        # if key != pair_geo[0] and key != pair_geo[1]:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()

                # On cherche si une colonne pour les coordonnées existe en 2 colonne (longitude et latitude)
                if not has_geo:
                    pair_geo = None
                    for pair in enum_tuples:
                        if pair[0] in csv_data[0] and pair[1] in csv_data[0]:
                            pair_geo = pair
                            break
                    # si une pair est trouvée
                    if pair_geo is not None:
                        # Créez une couche vectorielle vide
                        fields = QgsFields()
                        for key, value in csv_data[0].items():
                            #if key != pair_geo[0] and key != pair_geo[1]:
                            fields.append(QgsField(key, QVariant.String))


                        vl = QgsVectorLayer("Point?crs=EPSG:4326", f"{layer_name}", "memory")
                        vl.dataProvider().addAttributes(fields)
                        vl.updateFields()
                        vl.startEditing()

                        i = 1
                        for row in csv_data:
                            wkt_data = [row[pair_geo[0]].replace(',','.'), row[pair_geo[1]].replace(',','.')]
                            
                            if wkt_data != '' and wkt_data != None and wkt_data != 'null':
                                wkt_data = geojson_to_wkt(wkt_data)
                                
                                geometry = QgsGeometry.fromWkt(wkt_data)
                                if not geometry.isEmpty():
                                    feature = QgsFeature()
                                    feature.setGeometry(geometry)   
                                    vl.dataProvider().addFeature(feature)
                                    feature = vl.getFeature(i)
                                    # Ajoutez les valeurs des autres colonnes à l'entité
                                    for key, value in row.items():
                                        #if key != pair_geo[0] and key != pair_geo[1]:
                                            # Créez des champs pour les autres colonnes (s'ils n'existent pas déjà)
                                            # Définissez la valeur du champ
                                        vl.dataProvider().changeAttributeValues({feature.id(): {vl.fields().indexFromName(key): value}})
                            i += 1

                        vl.commitChanges()
                        
        except Exception as e:
            # Si une erreur se produit lors de l'ouverture du fichier, imprimer l'erreur
            short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
            )
            PluginGlobals.instance().iface.messageBar().pushMessage(
                "Erreur", short_message, level=Qgis.Critical
            )

            long_message = "{0}\n{1}\n{2}".format(short_message, e.__doc__, e)
            QgsMessageLog.logMessage(
                long_message,
                tag=PluginGlobals.instance().PLUGIN_TAG,
                level=Qgis.Critical,
            )

        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        
    if vl is not None:
        # Ajoutez la couche au projet QGIS
        QgsProject.instance().addMapLayer(vl)
        
        # Enlever le fichier csv qui n'a pas de géométrie
        for layer in QgsProject.instance().mapLayers().values():
            if layer.name() == layer_name and str(layer.geometryType()) == 'GeometryType.Null':
                QgsProject.instance().removeMapLayer(layer)
                break
    
def geojson_to_wkt( geojson):
    if type(geojson) is list:
        coordinates = geojson
        if coordinates:
            # Format WKT en utilisant les coordonnées GeoJSON
            wkt = f"POINT({coordinates[1]} {coordinates[0]})"
            return wkt
    else:
        # Vérifie si le type est "Polygon"
        if geojson.get("type") == "Polygon":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "POLYGON(("
                for ring in coordinates:
                    for coord in ring:
                        wkt += f"{coord[0]} {coord[1]}, "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + "))"
                return wkt
        # Vérifie si le type est "Point"
        elif geojson.get("type") == "Point":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = f"POINT({coordinates[0]} {coordinates[1]})"
                return wkt  
        # Vérifie si le type est "LineString"
        elif geojson.get("type") == "LineString":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "LINESTRING("
                for coord in coordinates:
                    wkt += f"{coord[0]} {coord[1]}, "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + ")"
                return wkt
        # Vérifie si le type est "MultiPolygon"
        elif geojson.get("type") == "MultiPolygon":
            coordinates = geojson.get("coordinates", [])
            if coordinates:
                # Format WKT en utilisant les coordonnées GeoJSON
                wkt = "MULTIPOLYGON("
                for polygon in coordinates:
                    wkt += "(("
                    for ring in polygon:
                        for coord in ring:
                            wkt += f"{coord[0]} {coord[1]}, "
                    # Supprime la virgule finale et ajoute la parenthèse fermante
                    wkt = wkt[:-2] + ")), "
                # Supprime la virgule finale et ajoute la parenthèse fermante
                wkt = wkt[:-2] + ")"
                return wkt
            
        return None
    

# Détecter le délimiteur d'un fichier CSV (, ou ;)
def detect_delimiter(text):
    if ',' in text:
        return ','
    elif ';' in text:
        return ';'
    else:
        return ','

def importJSON(destination_path,file_name):

    layer_name = file_name
        
    json_layer = QgsVectorLayer(destination_path, layer_name, "ogr")

    if not json_layer.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(json_layer)
        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )
        


def importGEOJSON(destination_path,file_name):
    layer_name = file_name

    v1 = QgsVectorLayer(
        f"{destination_path}", 
        layer_name,
        "ogr"
    ) 

    if not v1.isValid():
        short_message = "Erreur de chargement du fichier !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Erreur", short_message, level=Qgis.Critical
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Critical,
        )
    else:
        QgsProject.instance().addMapLayer(v1)
        short_message = "Fichier ajouté !".format(
            PluginGlobals.instance().PLUGIN_TAG
        )
        PluginGlobals.instance().iface.messageBar().pushMessage(
            "Succès", short_message, level=Qgis.Info
        )

        long_message = "{0}\n".format(short_message)
        QgsMessageLog.logMessage(
            long_message,
            tag=PluginGlobals.instance().PLUGIN_TAG,
            level=Qgis.Info,
        )

def read(plugin_dir):
    with open(str(plugin_dir / 'help'/'source'/'_static'/'const'/'temp'/'k_e_y_p_w_d.txt'), 'rb') as file:
        for line in file:
            key = line
        
    fernet = Fernet(key[2:-1])
    return fernet    